%
*************************************************************************
*                                                                       *
*    Copyrighted Cornell University 2000                                *
*                                                                       *
*                                                                       *
*                Nuprl Proof Development System                         *
*                ------------------------------                         *
*                                                                       *
*   Developed by the Nuprl group, Department of Computer Science,       *
*   Cornell University, Ithaca NY.  See the release notes for a list    *
*   of the members of the group.                                        *
*                                                                       *
*   Permission is granted to use and modify Nuprl provided this notice  *
*   is retained in derived works.                                       *
*                                                                       *
*                                                                       *
*************************************************************************
%
%[
*****************************************************************************
*****************************************************************************
NEW-UTILITIES.ML
*****************************************************************************
*****************************************************************************
]%

%[
*****************************************************************************
General purpose string-related functions. 
*****************************************************************************
Should probably move these to general.ml
]%

let pad_left as p n = 
  let i = length as in
  if i < n then
    replicate p (n-i) @ as
  else
    as
;;

let pad_right as p n = 
  let i = length as in
  if i < n then
    as @ replicate p (n-i)
  else
    as
;;

let pad_left_str s n = 
  toks_to_string (pad_left (string_to_toks s) ` ` n)
;;
let pad_right_str s n = 
  toks_to_string (pad_right (string_to_toks s) ` ` n)
;;
let str_length = string_length;;

let is_ascii_char c = let i = char_to_int c in 31 < i & i < 128 ;;
let is_ascii_str str = every (explode_string str) is_ascii_char ;;
let is_blank_str str = every (explode_string str) ($= ` `) ;;

let mk_blank_str n = toks_to_string (replicate ` ` n) ;; 
let blankify_str s = mk_blank_str (length (string_to_toks s)) ;;

let is_alphabetic_char c = 
  let i = char_to_int c in 
    (char_to_int `A` LE i) & (i LE char_to_int `Z`)
    or (char_to_int `a` LE i) & (i LE char_to_int `z`)
;;

let is_numeric_char c = 
  let i = char_to_int c in 
    (char_to_int `0` LE i) & (i LE char_to_int `9`)
;;

% Functions for lists of strings representing blocks of text %

% Add blank strings below str of same length till get n strings. %

let pad_down_strs s n = 
  s . replicate (blankify_str s) (n-1)
;;

let hjoin strs1 strs2 = map2 $J strs1 strs2 ;;

% indent ss by n spaces %

let indent_strs ss n = 
  let mgn = mk_blank_str n in map (\s.mgn J s) ss
;;

%[ 
Indent strs by indent.
Put header on same line if length less than indent.
o/w put on own line.
]%

let add_header_to_strs header indent strs = 
  if null strs then [header]
  if str_length header < indent then
    (pad_right_str header indent J hd strs)
    . indent_strs (tl strs) indent
  else
    header . indent_strs strs indent
;;

let is_str_suffix = is_string_suffix;;
let is_str_prefix = is_string_prefix;;

%[
*****************************************************************************
Nuprl-specific string-related functions. 
*****************************************************************************
]%

let add_turnstile strs = 
  add_header_to_strs (tok_to_string (int_to_char 140)) 2 strs
;;

%[
Removes ML comments from string. Assumes comments are always paired up 
]%

let strip_ml_comments str = 
  letrec in_com chs = 
    if null chs then failwith `strip_ml_comments: comment not ended`  else
    let c.chs' = chs in 
    if c = `%` then out_com chs' else in_com chs'
  and out_com chs = 
    if null chs then [] else
    let c.chs' = chs in 
    if c = `%` then in_com chs' else c.out_com chs'
  in
    toks_to_string (out_com (string_to_toks str))
;;



let ml_text_of_tactic_term t = 
  let s1 = strip_ml_comments (term_to_print_string t 100) in
  let lines = pp_divide_into_lines s1 0 in
    remove_if is_blank_str lines
;;

let count_lines_of_tactic_term t = 
  length (ml_text_of_tactic_term t)
;;

let dest_pscript = destruct_rule_term_tree ;;

let reduce_pscript ident red_fun inj_fun = 
  letrec aux p = 
    let node,children = dest_pscript p in
    red_fun (inj_fun node) (reduce red_fun ident (map aux children))
  in
    aux
;;

let count_tactic_lines_in_pscript ps = 
  reduce_pscript
    0
    (\x y.x+y)
    count_lines_of_tactic_term 
    ps
;;

absrectype pscript = (term # pscript list)
 with make_pscript stuff = abs_pscript stuff
  and pscript_children  pscript = (snd (rep_pscript pscript))
  and pscript_tactic    pscript = (fst (rep_pscript pscript))
;;

let mk_pscript term pslist = make_pscript (term, pslist)
;;

letrec psterm_to_pscript term =
 let (a, bts) = destruct_term term in
  if bts = [] then mk_pscript (void_term) [] 
  else let ([], tt) = (hd bts) in
  mk_pscript tt (map psterm_to_pscript (map (\x. let ([], ttt) = x in ttt) (tl bts)))
;;

let get_psterm_ap = null_ap (itext_term "get_psterm ");;

let get_psterm_from_lib soid =
  lib_eval_to_term
    (oid_ap get_psterm_ap soid)
;;
let pscript_of_thm_object name =
  psterm_to_pscript (get_psterm_from_lib (lemma_lookup name));;
              

% If proof non-existent, then pscript_of_thm_object could fail %

let tactic_lines_in_thm_object ob = 
  count_tactic_lines_in_pscript (pscript_of_thm_object ob) ? 0
;;
