
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************

#+cltl2(in-system-package)
#-cltl2(in-package *system-package-name*)

;;;;	
;;;;	
;;;;	

(defconstant imouse-left 'mouse-left)
(defconstant imouse-middle 'mouse-middle)
(defconstant imouse-right 'mouse-right)

;; backspace : do delete/backspace mods.


  

(defvar *fociitoks* (let ((a (make-array 8)))
		      (dotimes (i 8)
			(setf (aref a i)
			      (intern (concatenate 'string "FOCUS" (princ-to-string (1+ i))))))
		      a))

(defun focustok-p (a)
  (or (eql 'focus a)
      ;;(some #'(lambda (b) (eql a b)) *fociitoks*)
      (eql 'focus1 a)
      (eql 'focus2 a)
      (eql 'focus3 a)
      (eql 'focus4 a)
      (eql 'focus5 a)
      (eql 'focus6 a)
      (eql 'focus7 a)
      (eql 'focus8 a)
      ))

;; note that expect a to not have side effects or cost. (ie should be a variable)
(defmacro focusich-p (a)
  ;;  (and (consp a)
  ;;(null (cdr a))
  ;;(focustok-p (car a)))

  `(and (not (integerp ,a))
    (focustok-p ,a)))


(defun focustok (i)
  (aref *fociitoks* i))

;;;;	
;;;;	focus<i>
;;;;	  - input buffer FOCUS
;;;;	      * The input buffer consists of a list of input characters and a focus-stack.
;;;;		A character is an integer, a token, or cons of a list of modifiers and a character.
;;;;	          - Could hide window in modifier list of focus token rather than managing focus stack.
;;;;		  - Seems less complex but more milling.  Leave as is fttb.
;;;;	     * if the input event came from a different window then the last input event
;;;;	       then a focus token is inserted in the input buffer prior to the character
;;;;	       of the event. 
;;;;	     * there are never two adjacent focus tokens in the input buffer.
;;;;	  - macro model  (FOCUS<i>)
;;;;	     * there may be more focus tokens in the input buffer then the macro model.
;;;;	       those not used in matching the model are ignored. 
;;;;	  - macro body   (m-x)focus<i>
;;;;		focus on widow corresponding to ith matchded focus.

;;;;	
;;;;	autofocus : if focus at head of buffer but no matching in macro token
;;;;	 then autofocus's on buffer head.
;;;;
;;;;	Want Autofocus to happen prior to echo of input buffer.
;;;;	Auto focus with text changes.
;;;;	

(defstruct (echo (:print-function echo-print-function))
  (buffer nil)
  (view nil)
  (focii nil)
  (ifocii nil)
  )


;; this should do the matching, or be called by one that does?
(defun window-of-focus-index (j e)

  (nth j (ifocii-of-echo e))

  ;;(let ((focii (echo-focii e)))
  ;;    (nth (- (length focii)
  ;;(1+ j))
  ;;focii))
  )


(defun ifocii-of-echo (e)
  (echo-ifocii e))


(defun echo-set-ifocii (e token)
  (setf (echo-ifocii e)
	;; match
	(let ((focii (reverse (echo-focii e))))
	  (let ((acc (list (car focii))))
	    (setf focii (cdr focii))
	    (do ((c token (cdr c))
		 (b (echo-buffer e) (cdr b)))
		((null b))
	      (when (focusich-p (car b))
		(if (focusich-p (car c))
		    (setf acc (cons focii acc)
			  focii (cdr focii))
		    (setf focii (cdr focii)
			  b (cdr b)))
		    
		(unless (or (focusich-p (car b))
			    (equal (car b) (car c)))
		  ;;(setf -b b -c c -token token -buffer (echo-buffer e))
		  ;;(break "ioe")
		  )))
	    (nreverse acc)))))

(defun echo-print-function (echo stream depth)
  (declare (ignore depth))
  (format stream "EchoBuffer: ~a : (~a ~a)~%"
	  (estring-to-string (echo-buffer echo))
	  (length (echo-focii echo))
	  (length (echo-ifocii echo))))

	   

;; could do similar for multiple clicks.
;; could have click1-8 stack as well then mouse-point(1-8)
;; or just have stack accessible via ml.


(defun echo-clear (e)
  (setf (echo-buffer e) nil
	(echo-focii e) nil))

(defun echo-null-p (e) (null (echo-buffer e)))


;;;	
;;;	current focus
;;;	event window
;;;	focii
;;;	
;;;	
;;;	null focii -> init.
;;;	  - push current and event if diff from current.
;;;	if 
;;;	

(defun update-edit-buffer-focus (e w)
  ;;(setf -w w -e e) (break "ei")
  (when (null (echo-focii e))
    (setf (echo-focii e) (list *oed-focus*))
    (unless (eql *oed-focus* w)
      ;; auto-focus : only if echo-focii nil.
      (oed-focus-on w)))

  (unless (eql (car (echo-focii e)) w)
    (push w (echo-focii e))
    (setf (echo-buffer e) (nconc (echo-buffer e)
				 (list 'focus)))
    ))

(defun set-edit-buffer-focus (e)
  (let ((b (echo-buffer e)))
    (if (null b)
	(setf (echo-focii e) nil)
	(setf (echo-focii e)
	      (cons *oed-focus*
		    (let ((remainder (subseq (echo-focii e) 0
					     (let ((count 0))
					       (dolist (ich b)
						 (when (focustok-p ich) (incf count)))
					       count))))
		      ;; if last of remainder is not focus then focus on last, ie simulate init
		      (when remainder
			(unless (eql *oed-focus* (car (last remainder)))
			  (oed-focus-on (car (last remainder)))))
		      remainder))))))


(defun echo-insert (e c)
  (let ((w (xwin-event-window)))

    (when w
      (update-edit-buffer-focus e w))
    
    (setf (echo-buffer e)
	  (nconc (echo-buffer e) (list c)))))


(defun echo-rubout (e)
  (setf (echo-buffer e) (butlast (echo-buffer e)))
  (when (focustok-p (car (last (echo-buffer e))))
    (setf (echo-buffer e) (butlast (echo-buffer e))
	  (echo-focii e) (cdr (echo-focii e)))))


;; auto-focus???
;; should check subseq end.
(defun set-echo-buffer (e b)
  (setf (echo-buffer e) b)
  (set-edit-buffer-focus e))

(defun set-echo-view (e v) (setf (echo-view e) v))

(defun view-of-echo (e) (echo-view e))
(defun buffer-of-echo (e) (echo-buffer e))




(defstruct (edit-state (:include view-state))
  ;;window

  restore-stack				; backup for abort.
  
  lines
  zoom					; stack of dtree/point

  ;; modes
  (left-right (left-mode))
  (text-term (text-mode))
  (insert-replace (insert-mode))	; only applicable if term-mode-p
  (dtree-concrete (dtree-mode))

  (mark nil)

  (cursor nil)
  (echo nil)

  ;;(height 0)
  ;;(width 0)
  (offset 0)

  (output-p nil)

  (point-suppression nil)

  (mouse-row 1)
  (mouse-col 1)

  (verified-p nil)
  (evaled-p nil)
  (walked-p nil)

  (buffers nil)
  )


(eval-when (compile)
  (proclaim '(inline
	      window-of-edit-state object-of-edit-state
	      reset-term-of-edit-state restore-stack-of-edit-state
	      dtree-of-edit-state 
	      left-right-of-edit-state
	      mark-of-edit-state cursor-of-edit-state
	      ;;height-of-edit-state width-of-edit-state
	      offset-of-edit-state
	      suppressed-dforms-of-edit-state
	      ;;modified-of-edit-state
	      mouse-row-of-edit-state mouse-col-of-edit-state
	      output-edit-state-p output-cursor-edit-state-p
	      history-of-edit-state)))

(defun window-of-edit-state (state) (edit-state-window state))
(defun object-of-edit-state (state) (edit-state-object state))
(defun restore-stack-of-edit-state (state) (edit-state-restore-stack state))
;;(defun modified-of-edit-state (state) (edit-state-modified state))
(defun dtree-of-edit-state (state) (edit-state-dtree state))
(defun suppressed-dforms-of-edit-state (state) (edit-state-suppressed-dforms state))
(defun mouse-row-of-edit-state (state) (edit-state-mouse-row state))
(defun mouse-col-of-edit-state (state) (edit-state-mouse-col state))
(defun history-of-edit-state (state) (edit-state-history state))
(defun edit-state-point-suppression-p (state) (edit-state-point-suppression state))

(defun set-edit-state-mouse-position (state row col)
  (setf (edit-state-mouse-row state) row
	(edit-state-mouse-col state) col))



(defun edit-walk-f (dir stop-ce stopf view)
  (setf (edit-state-walked-p view) nil)
  (when (edit-walk nil			; permuted order
		   stop-ce	
		   stopf		; no stop hook
		   (false-ce)	
		   (false-ce)		; no floor
		   ;;*not-instantiated-ce* ; floor 
		   'point		; move point 
		   dir			
		   view)
    (setf (edit-state-walked-p view) t)
    (point-rehash view 'point))
  view)



(defun edit-search-aux (searchp termp dirp view)
  (format t "~%Searching~%")
  (edit-walk-f dirp
	       *true-ce*
	       #'(lambda (dtree view)
		   (declare (ignore view))
		   (let ((foundp nil))
		     (if (and termp (not (dtree-leaf-p dtree)))
			 (when (funcall searchp (term-of-dtree dtree))
			   (setf foundp t))
			 (when (not termp)
			   (map-dtree-children #'(lambda (dtree)
						   (when (dtree-leaf-p dtree)
						     (let ((s (implode-to-string (text-of-dtree-leaf dtree))))
						       ;;(format t "S: ~a~%" s)
						       (when (funcall searchp s)
							 ;;(format t "FOUND~%") 
							 (setf foundp t)))))
					       nil
					       t
					       dtree)))
		     foundp))
	       view))

(defun edit-search-text (pattern dirp view)
  (edit-search-aux (string-pattern-search #'identity pattern nil)
			nil
			dirp
			view)
  #|(let ((searchf (string-pattern-search #'identity pattern nil)))
    (format t "~%Searching~%")
    (edit-walk-f dirp
		 *true-ce*
		 #'(lambda (dtree view)
		     (declare (ignore view))
		     (let ((foundp nil))
		       (map-dtree-children #'(lambda (dtree)
					       (when (dtree-leaf-p dtree)
						 (let ((s (implode-to-string (text-of-dtree-leaf dtree))))
						   (format t "S: ~a~%" s)
						   (when (funcall searchf s)
						     (format t "FOUND~%") 
						     (setf foundp t)))))
					   nil
					   t
					   dtree)
		       foundp))
		 view))|#
)

(defunml (|make_edit_search| (pattern))
    (string -> (bool -> (view -> unit)))

  (make-closure #'(lambda (v dirp) (edit-search-text pattern dirp v))
		2))

(defunml (|make_edit_term_search| (f))
    ((term -> bool) -> (bool -> (view -> unit)))

  (make-closure #'(lambda (v dirp)
		    (edit-search-aux #'(lambda (term) (funmlcall f term))
				     t dirp v))
		2))


;;;;
;;;;	desire editor to be able to object modified since last save/load.
;;;;	
;;;;	load : read from lib.
;;;;	  - twould be nice to attempt to preserve marks if terms match to some extent.
;;;;	save : write to lib.
;;;;	 diff saves for diff attrs.
;;;;	  geo 
;;;;	  implicit
;;;;	  term
;;;;	 
;;;;	have vobject which caches lib term, implicit, and geo?
;;;;	
;;;;	previously kept a reset stack. That seems inappropriate, however we have
;;;;	maintained the stack mechanism but limited it to length 1, which should match
;;;;	the vobject state.
;;;;	
;;;;	new features should be added by calling ml.
;;;;	
;;;;	
;;;;	
;;;;	


(defun edit-state-save (state term)
  (setf  (edit-state-restore-stack state) (list term)))
  ;;(unless (compare-terms-p term (car (restore-stack-of-edit-state state)))
  ;;  (push term (edit-state-restore-stack state)))

(defun reset-term-of-edit-state (state)
  (car (restore-stack-of-edit-state state)))
    
(defun edit-state-pop (state term)
  (when (and (cdr (restore-stack-of-edit-state state))
	     (compare-terms-p term (car (restore-stack-of-edit-state state))))
    (setf (edit-state-restore-stack state) (cdr (restore-stack-of-edit-state state)))))

(defun edit-state-abort (state)
  (setf (edit-state-restore-stack state)
	(last (restore-stack-of-edit-state state)))
  (car (restore-stack-of-edit-state state)))


;;;;
;;;;  modes
;;;; 

(eval-when (compile)
  (proclaim '(inline
	      mode mode-p toggle-mode
	      right-mode left-mode
	      right-mode-p left-mode-p
	      term-mode term-mode
	      text-mode-p text-mode-p
	      insert-mode replace-mode
	      insert-mode-p replace-mode-p
	      )))

(defun mode () t)
(defun mode-p (mode) mode)
(defun toggle-mode (mode) (not mode))


;;;
;;; point-suppression-p
;;;

(defun edit-state-point-suppression-mode (state)
  (setf (edit-state-point-suppression state)
	(toggle-mode (edit-state-point-suppression-p state)))
  state)

(defun edit-state-point-suppression-mode-on (state)
  (setf (edit-state-point-suppression state) (mode))
  state)

(defun edit-state-point-suppression-mode-off (state)
  (setf (edit-state-point-suppression state) (not (mode)))
  state)


;;;
;;; left/right
;;;

(defun right-mode () (not (mode)))
(defun left-mode () (mode))
(defun right-mode-p (m) (not (mode-p m)))
(defun left-mode-p (m)  (mode-p m))
       
(defun left-right-of-edit-state (state)
  (edit-state-left-right state))

(defun right-edit-state-p (state)
  (right-mode-p (left-right-of-edit-state state)))

(defun left-edit-state-p (state)
  (left-mode-p (left-right-of-edit-state state)))

(defun toggle-edit-state-left-right (state)
  (setf (edit-state-left-right state)
	(toggle-mode (left-right-of-edit-state state)))
  state)

(defun right-edit-state (state)
  ;;(break)
  (setf (edit-state-left-right state) (right-mode))
  state)

(defun left-edit-state (state)
  ;;(break)
  (setf (edit-state-left-right state) (left-mode))
  state)


;;;
;;; text/term
;;;

(defun term-mode () (mode))
(defun text-mode () (not (mode)))
(defun term-mode-p (m) (mode-p m))
(defun text-mode-p (m) (not (mode-p m)))

(defun text-term-of-edit-state (state)
  (edit-state-text-term state))

(defun term-edit-state-p (state)
  (term-mode-p (text-term-of-edit-state state)))

(defun toggle-edit-state-text-term (state)
  (setf (edit-state-text-term state)
	(toggle-mode (text-term-of-edit-state state)))
  state)

(defun text-edit-state-p (state)
  (text-mode-p (text-term-of-edit-state state)))

(defun term-edit-state (state)
  (setf (edit-state-text-term state) (term-mode))
  state)

(defun text-edit-state (state)
  (setf (edit-state-text-term state) (text-mode))
  state)


;;;
;;; insert/replace
;;; 

(defun insert-mode () (mode))
(defun replace-mode () (not (mode)))
(defun insert-mode-p (m) (mode-p m))
(defun replace-mode-p (m) (not (mode-p m)))

(defun insert-replace-of-edit-state (state)
  (edit-state-insert-replace state))

(defun replace-edit-state-p (state)
  (replace-mode-p (insert-replace-of-edit-state state)))

(defun toggle-edit-state-insert-replace (state)
  (setf (edit-state-insert-replace state)
	(toggle-mode (insert-replace-of-edit-state state)))
  state)

(defun insert-edit-state-p (state)
  (insert-mode-p (insert-replace-of-edit-state state)))

(defun replace-edit-state (state)
  (setf (edit-state-insert-replace state) (replace-mode))
  state)

(defun insert-edit-state (state)
  (setf (edit-state-insert-replace state) (insert-mode))
  state)



;;;
;;; dtree/concrete(term) mode
;;;

(defun concrete-mode () (mode))
(defun dtree-mode () (not (mode)))

(defun concrete-mode-p (m) (mode-p m))
(defun dtree-mode-p (m) (not (mode-p m)))

(defun dtree-concrete-of-edit-state (state)
  (edit-state-dtree-concrete state))

(defun concrete-edit-state-p (state)
  (concrete-mode-p (dtree-concrete-of-edit-state state)))

(defun toggle-edit-state-dtree-concrete (state)
  (setf (edit-state-dtree-concrete state)
	(toggle-mode (dtree-concrete-of-edit-state state)))
  state)

(defun dtree-edit-state-p (state)
  (dtree-mode-p (dtree-concrete-of-edit-state state)))

(defun concrete-edit-state (state)
  (setf (edit-state-dtree-concrete state) (concrete-mode))
  state)

(defun dtree-edit-state (state)
  (setf (edit-state-dtree-concrete state) (dtree-mode))
  state)





(defun mark-of-edit-state (state)
  (dtree-at-label 'mark (dtree-of-view state)))

(defun cursor-of-edit-state (state)
  (when (edit-state-p state)
    (edit-state-cursor state)))

(defun echo-of-edit-state (state)
  (let ((e (edit-state-echo state)))
    (when (and e (not (echo-null-p e)))
      e)))


(defun set-view-echo (v e)
  (let ((ov (when e (view-of-echo e))))

    ;;(when (or t (null v)) (setf -v v -ov ov -e e) (break "sve"))
    (cond
      ((or (null v) (not (edit-state-p v)))
       (when ov
	 (set-echo-view e nil)
	 (set-view-echo ov nil)))
      
      ((null e)
       (setf (edit-state-echo v) nil)
       (when (not (view-flag-cursor-visible-p v))
	 (view-flag-set-display-required v t)
	 (view-flag-set-cursor-present-required v t)
	 ;;(view-flag-set-cursor-visible v t)
	 ))

      ((eql v ov)
       (if (echo-null-p e)
	   (when (not (view-flag-cursor-visible-p v))
	     (view-flag-set-display-required v t)
	     (view-flag-set-cursor-present-required v t)
	     (view-flag-set-cursor-visible v t)
	     )
	   (progn
	     (when (view-flag-cursor-visible-p v)
	       (view-flag-set-cursor-present-required v t)
	       (view-flag-set-cursor-visible v nil))
	     (view-flag-set-echo-display-required v t))))

      (t
       (when ov
	 (set-view-echo ov nil))
       (unless (echo-null-p e)
	 (when (view-flag-cursor-visible-p v)
	   (view-flag-set-cursor-present-required v t))
	 (view-flag-set-cursor-visible v nil)
	 (view-flag-set-echo-display-required v t))
       (setf (edit-state-echo v) e)
       (set-echo-view e v)
       ))))


;;(defun height-of-edit-state (state) (1- (edit-state-height state))) ;; 1- for status line.
;;(defun width-of-edit-state (state) (edit-state-width state))
(defun offset-of-edit-state (state) (edit-state-offset state))

(defun output-edit-state-p (state) (edit-state-output-p state))
(defun output-cursor-edit-state-p (state) (edit-state-cursor-output-p state))

;; defunct ?
#|
(defun edit-state-new-dtree (state dtree)
  (break "esnd")
  (setf (edit-state-dtree state) dtree)
  (view-flag-set-layout-required state t)
  state)
|#


(defun new-edit-state ()
  (make-edit-state :output-p t
		   ))


;;;
;;; cursor
;;;

(defstruct (cursor (:print-function print-cursor))
  (state nil)

  layout

  ;; cursor-layout
  ;;(row 0)				; relative to screen
  ;;(col 0)				; ditto
  ;;(indentation 0)			; ditto
  ;;(lines nil)
  )

(eval-when (compile)  (proclaim '(inline state-of-cursor)))

(defun state-of-cursor (cursor) (cursor-state cursor))

(defun layout-of-cursor (c) (cursor-layout c))

(defun row-of-cursor (cursor)
  (let ((l (layout-of-cursor cursor)))
    (and l (row-of-xcursor l))))

(defun col-of-cursor (cursor)
  (let ((l (layout-of-cursor cursor)))
    (and l (col-of-xcursor l))))


(defstruct (point-cursor (:include cursor))
  (point nil))

(defun point-cursor (state point)
  (when (null point) (break))
  (make-point-cursor :state state :point point))

(defun point-of-point-cursor (cursor) (point-cursor-point cursor))

(defstruct (screen-cursor (:include cursor))
  (point nil))

(defun screen-cursor (state row col)
  (when (or (zerop col) (zerop row)) (break))
  (make-screen-cursor :state state :layout (new-blob-cursor row col)))

(defun point-of-screen-cursor (cursor)
  (or (screen-cursor-point cursor)
      (setf (screen-cursor-point cursor)
 	    (row-col-to-point (state-of-cursor cursor)
			      (row-of-cursor cursor)
			      (col-of-cursor cursor)))))


; assumes blob always next to last character in lines.
(defstruct (echo-cursor (:include cursor))
  (shadowed-cursor nil)
  (buffer nil)
  (flash-p nil))


(defun update-echo-cursor (cursor buffer)
  (setf (echo-cursor-buffer cursor) buffer)
  (setf (cursor-lines cursor) nil)
  cursor)

(defun echo-cursor (state row col buffer)
  (make-echo-cursor :state state
		    :shadowed-cursor (cursor-of-edit-state state)
		    :layout (new-spot-cursor row col nil)
		    :buffer buffer))

(defun buffer-of-echo-cursor (cursor) (echo-cursor-buffer cursor))
(defun shadowed-cursor-of-echo-cursor (cursor) (echo-cursor-shadowed-cursor cursor))


(defun set-edit-state-cursor (new-cursor state)
  ;;(xwin-undraw-cursor (window-of-edit-state state) (buffers-of-edit-state state))
  (view-flag-set-cursor-layout-required state t)

  (setf (edit-state-cursor state) new-cursor)

  state)
  

(defun point-state (state point)
   (when (null point)
     (process-err
      "Point not set. Probably caused by unexpected term structure for last command"))
   (set-edit-state-cursor (point-cursor state point) state))

;; maybe should just find label in dtree. relying on label cache for efficiency.
(defun point-of-edit-state (state)
  ;;(setf -v state) (break "pes")
  (point-of-cursor (cursor-of-edit-state state)))

(defun print-cursor (cursor stream depth)
  (declare (ignore depth))
  (format stream "~aCursor~%"
	  (cond ((screen-cursor-p cursor) "Screen")
		((point-cursor-p cursor) "Point")
		((echo-cursor-p cursor) "Echo")
		(t "?"))))

(defun layout-row-of-cursor (cursor state)
  (+ (offset-of-edit-state state)
     (row-of-cursor cursor)))

(defun layout-col-of-cursor (cursor state)
  (declare (ignore state))
  (col-of-cursor cursor))



;;; point
;;;  if screen cursor
;;;    then  point derived from cursor.
;;;  if echo cursor
;;;    then point derived from echo-shadowed-cursor
;;;  otherwise point is unchanged
;;;    then eq (point-cursor (state-point state))
(defun point-of-cursor (cursor)
  (cond
    ((echo-cursor-p cursor)
     (point-of-cursor (shadowed-cursor-of-echo-cursor cursor)))
    ((screen-cursor-p cursor)
     (point-of-screen-cursor cursor))
    ((point-cursor-p cursor)
     (point-of-point-cursor cursor))))


