
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************

#+cltl2(in-system-package)
#-cltl2(in-package *system-package-name*)

;;;;	inform-fonts (<string{font name}) list
;;;;	  * internally fonts will be referenced by position of font name in list.
;;;;
;;;;	 ** NB: presentation and edit must be supplied the same list (in same order).
;;;;
;;;;	ascii-to-glyphs (<string> <token{font modifier}> list)	: <glyphs>
;;;;
;;;;	
;;;;	Implementation : for nuprl fonts
;;;;	 labels can migrate to the ptree from the dtree. The only way for local dform labels
;;;;	 to migrate is through the modifier map. These are the text labels, 
;;;;	
;;;;	<modifier-map>	: <token{modifier}> list . <string{font name}>
;;;;	
;;;;
;;;;	inform-font-modifier-map(<modifier-map>) 
;;;;
;;;;	map-font-modifiers (<token{modifier}> list)	: INTEGER{font}
;;;;	
;;;;	 
;;;;	  * might expect character to map to base-font and font-index 
;;;;	  * and base-font and modifiers to map to font. However, we
;;;;	  * haven't committed to that here.
;;;;	



;; RLE ??? shared by all components in process.
(defvar *fonts* nil)			; possibly shared by win.

(defun get-xfont (i) (aref *fonts* i 0))
(defun get-font-name (i) (aref *fonts* i 1))

(defun set-xfont (i xfont) (setf (aref *fonts* i 0) xfont))
(defun set-font-name (i name) (setf (aref *fonts* i 1) name))

(defun descent-of-xfont (i)
  (let ((xfont (get-xfont i)))
    (if xfont
	(xlib:font-descent xfont)
	2)))

(defun ascent-of-xfont (i)
  (let ((xfont (get-xfont i)))
    (if xfont
	(xlib:font-ascent xfont)
	7)))

(defun width-of-xfont (i index) 
  (let ((xfont (get-xfont i)))
    (if xfont
	(xlib:char-width xfont index)
	13)))



(defun refresh-fonts (open-f)
  (dotimes (i (array-dimension *fonts* 0))
    (set-xfont i (funcall open-f (get-font-name i)))))


(defun inform-fonts (font-names)
  (let ((l (length font-names)))
    ;; first field is for pointer to open font, second is font name.
    (setf *fonts* (make-array (list l 2)))
    (dotimeslist (i font font-names)
		 (set-font-name i font))))

(defun find-font-by-name (name)
  (dotimes (i (array-dimension *fonts* 0))
    (when (string= name (get-font-name i))
      (return-from find-font-by-name i)))
  (raise-error (error-message '(font find name unknown) name)))





;; RLE ??? shared by all components in process ?
(defvar *font-modifier-map* nil)
(defvar *font-modifier-map-default* nil)

(defun inform-font-modifier-map (map default)
  (setf *font-modifier-map* (mapcar #'(lambda (entry)
					(cons (car entry) (find-font-by-name  (cdr entry))))
				    map)
	*font-modifier-map-default* (find-font-by-name default)))
  

(defun font-modifiers-to-font (modifiers)
  (or (cdr (assoc modifiers *font-modifier-map* :test #'equal))
      *font-modifier-map-default*))

;; this will change when character-glyph-font schema changes.


;;;;	
;;;;	Character glyph map : probably better somewhere else as prs has no need.
;;;;	
;;;;	
;;;;	



;; in input order grouped by labels. Within label group ordered by character range.
;; overlaps


;;;;	Use a-priori knowledge of supported character ranges to allocate
;;;;	lookup arrays.
;;;;	

(defvar *character-ranges* (let ((l '((#x0000 . #x007f) ; ascii
				      (#x00a0 . #x00ff) ; latin1
				      (#x0390 . #x03cf) ; greek
				      (#x2070 . #x208f) ; supers and subs
				      (#x2190 . #x21ff) ; arrows
				      (#x2200 . #x22ff) ; math
				      (#x2300 . #x232f) ; misc tech
				      (#x2500 . #x257f) ; form and chart
				      (#x25a0 . #x25ef) ; geom shapes
				      )))
			     (let ((a (make-array (list (length l) 2))))
			       (dotimeslist (i r l)
					    (setf (aref a i 0) (car r)
						  (aref a i 1) (cdr r)))
			       a)))
			       

(defun allocate-character-map-table ()
  (let ((len (array-dimension *character-ranges* 0)))
    (let ((char-table (make-array len)))
      (dotimes (i len)
	(setf (aref char-table i) (make-array (1+ (- (aref *character-ranges* i 1)
						     (aref *character-ranges* i 0)))
					      :initial-element nil)))
      char-table)))

;;;
;;;	(<labels> . ((<char-start> . <char-end>) . (<font> . <font-index-start>)))
;;;
;;;
;;; Order by range start unless range is x0000 -xffff
;;;

(defvar *character-glyph-map* (allocate-character-map-table))

(defvar *character-glyph-map-defaults* nil)

(defun reallocate-glyph-map ()
  (setf *character-glyph-map* (allocate-character-map-table))
  (setf *character-glyph-map-defaults* nil))


(defun character-map-array (ch)
  (dotimes (i (array-dimension *character-ranges* 0))
    (when (<= ch (aref *character-ranges* i 1))
      (let ((start-index (aref *character-ranges* i 0)))
	(when (>= ch start-index)
	  (return-from character-map-array
	    (values (aref *character-glyph-map* i) (- ch start-index))))))))


(defun inform-character-glyph-map (maps)
  (let ((bad-chars nil))

    ;; reverse any old entries.
    (setf *character-glyph-map-defaults* (reverse *character-glyph-map-defaults*))
    (dotimes (i (array-dimension *character-ranges* 0))
      (let ((start (aref  *character-ranges* i 0))
	    (end  (aref  *character-ranges* i 1))
	    (array (aref *character-glyph-map* i)))
	;;(setf -e start -f end -g array) 
	(dotimes (j (1+ (- end start)))
	  ;;(setf -h j) 
	  (let ((val (aref array j)))
	      (when (consp val) 
		(setf (aref array j) (reverse val)))))))

    ;; push new
    (dolist (map maps)
      ;;(setf a map)
      (if (and (= #x0000 (car map)) (= #xffff (cadr map)))
	  (push (cons (delete '!default (third map))
		      (cons (find-font-by-name (fourth map)) (fifth map)))
		*character-glyph-map-defaults*)
	  (let ((entry (when (member '!default (third map))
			 (cons (delete '!default (third map))
			       (cons (find-font-by-name (fourth map)) (fifth map))))))
	    (do ((i (car map) (incf i)))
		((> i (cadr map)))
	      ;;(setf b i)
	      (mlet* (((array index) (character-map-array i)))
		     ;;(setf c array d index)
		     (if (null array)
			 (pushnew i bad-chars)
			 (push (or entry
				   (cons (third map)
					 (cons (find-font-by-name (fourth map)) (+ (fifth map) (- i (car map))))))
			       (aref array index))))))))

    ;; nreverse
    (setf *character-glyph-map-defaults* (nreverse *character-glyph-map-defaults*))
    (dotimes (i (array-dimension *character-ranges* 0))
      (let ((start (aref  *character-ranges* i 0))
	    (end  (aref  *character-ranges* i 1))
	    (array (aref *character-glyph-map* i)))
	(dotimes (j (1+ (- end start)))
	  (setf (aref array j)
		(nreverse  (aref array j))))))

    (when bad-chars
      (message-emit (warn-message '(character-glyph-map characters out-of-range) bad-chars)))
    ))


;; returns first mapping where all map labels are contained in arg labels.
;; returns font and font index.
(defun map-character-to-glyph (ch labels)
  (mlet* (((array index) (character-map-array ch)))

	 (or (when array
	       (find-first #'(lambda (entry)
			       (let ((elabels (car entry)))
				 (when (forall-p #'(lambda (label) (member label labels))
						 elabels)
				   (cdr entry))))
			   (aref array index)))
	     (find-first #'(lambda (entry)
			     (let ((elabels (car entry)))
			       (when (forall-p #'(lambda (label) (member label labels))
					       elabels)
				 (cdr entry))))
			 *character-glyph-map-defaults*)
	     (cdr (last  *character-glyph-map-defaults*))

	     ;; there should always be a (nil . (<font> . <font-index>)) default.
	     (progn (setf a array b index c labels) (break) nil)
	     (raise-error (error-message '(character-glyph-map character no-map) ch)))))

  
(defun ascii-to-glyphs (string labels &optional len)

  ;; Though true for nuprl fonts it is not true for unicode that the same 
  ;; font will be used for each all characteres in a string.
  (let* ((l (or len (length string)))
	 (font-buffer (new-glyph-index-array 255))
	 (run-acc nil)
	 (font nil)
	 (font-run-i 0)
	 )

    (labels
	((end-run ()
	   (when (> font-run-i 0)
	     (let* ((font-run (new-glyph-run font font-run-i))
		    (array (array-of-glyph-run font-run)))
	       (dotimes (j font-run-i)
		 (setf (aref array j) (aref font-buffer j)))
	       (push font-run run-acc)))
	   (setf font nil
		 font-run-i 0))

	 (add-run (index)
	   (setf (aref font-buffer font-run-i) index)
	   (incf font-run-i)
	   (when (= font-run-i 255)
	     (end-run)))
	 )

      (do ((i 0))
	  ((>= i l)
	   (end-run)
	   (let* ((num-runs (length run-acc))
		  (glyphs (new-glyphs num-runs labels))
		  (runs (runs-of-glyphs glyphs)))
	     (dotimeslist (i run (nreverse run-acc))
			  (setf (aref runs i) run))
	     glyphs))

	;;(setf a i b string c l)
	(let ((ch (char string i)))
	  (incf i)
	  (let ((glyph (map-character-to-glyph (if (eql ch #\escape)
						   (prog1
						       ;; RLE PERF: there is probably a more efficient method
						       (+ (* 256 (char-code (char string i)))
							  (char-code (char string (1+ i))))
						     (incf i 2))
						   (char-code ch))
					       labels)))
	    (if (eql font (car glyph))
		(add-run (cdr glyph))
		(progn (end-run)
		       (setf font (car glyph))
		       (add-run (cdr glyph))))))))))





