
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************

#+cltl2(in-system-package)
#-cltl2(in-package *system-package-name*)

;;- no meta variable bindings, ie $x.t not ok.
;;- no first or second-order variable may occur free on rhs
;;-   which does not occur on lhs.
;;- no meta variables in variable terms: variable{$x:v} not ok.
;; on lhs:
;;-   all variables in a binding list of a bound-term must be distinct.
;;-   no duplicate parameter variables on lhs, ie foo{$s:s, $s:t} not ok.
;;-   no duplicate first or second-order variable ids in subterms
;;-     ie foo(x.A[x];y.A[y]) not ok.
;;-   all variables occuring in binding of second-order variable 
;;-      must occur in same order as subterms of second order variable.

;;;; RLE TODO strip marks out of lhs/rhs parameters/terms.
;; RLE TODO MILL matrix-of-abstraction -> rhs-of-abstraction  lhs-term -> lhs



;;;;
;;;;	LHS:
;;;;
;;;;	<lhs>			: <lhs-op>(<lhs-bound-term> list)
;;;;	<lhs-bound-term>	: <binding> list.variable{<target>:v}
;;;;							 ([variable{<binding>:v}] list)
;;;;	
;;;;	* The value of <binding> in the binding list and the variable injection should
;;;;	  be the same in the instance. Ie a,b.variable{A:v}(a; b) is proper, but
;;;;	  a,b.variable{A:v}(b; a) and a,b.variable{A:v}(c; d) are not.
;;;;	  There should be no duplicates in the binding list.
;;;;	* Parameters must be real or meta valued.
;;;;	* <target> must be real valued (ie not meta).
;;;;	* No meta variables may be repeated in the parameter list.	
;;;;
;;;;	RHS:
;;;;	
;;;;	* Parameters must be real or meta valued. No parameters of type variable
;;;;	  may contain meta values.
;;;;	* Any variable occuring free must occur on lhs with same arity.
;;;;	* No second order variable occurence may be in the scope a similar first order
;;;;	  binding. Note that there is no notion of second order binding in our syntax.
;;;;	* All subterms of a second order variable must have nil binding lists.
;;;;   


;;; this works in tandem with abstraction match-p, any liberalization here
;;; will likely require a change there.
;;; also import-abstraction.

;; allow slots on rhs.

(defun abstraction-translate-source-r (term)
  ;;(break "atsr")

  (with-tag 'abstraction-translate-source-r

    (unless (iabstraction-term-p term)
      (raise-error (error-message '(abs syntax term) term)))

    (let* ((lhs (lhs-of-iabstraction-term term))
	   (rhs (rhs-of-iabstraction-term term))
	   (lhs-term-sig (term-sig-of-term lhs))

	   (num-slots 0)
	   (slot-types nil)
	   
	   (error-p nil)

	   (lhs-parameter-variables nil)
	   (lhs-level-variables nil)
	   (lhs-variables nil)

	   (rhs-variables nil))


      ;; Check conditions.
      (with-handle-error (('(abs)) (setf error-p t))
	(icondition-sexpr-r (conditions-of-iabstraction-term term)))

	  
      ;;
      ;; Check lhs.
      ;; 

      ;; No meta variables may be repeated in the parameter list.	
      (dolist (parameter (parameters-of-term lhs))
	;; Parameters must be real or meta valued.
	(let ((value (value-of-parameter-m parameter)))
	  (if (level-expression-parameter-p parameter)
	      (if (level-variable-p value)
		  (if (member value lhs-level-variables)
		      (raise-error (error-message '(abs syntax lhs level-variable duplicate) value))
		      (push value lhs-level-variables))
	      	  (when (block vp
			  (level-expression-visit (value-of-parameter parameter)
						  #'(lambda (x) (declare (ignore x)) (return-from vp t)))
			  nil)
		    (raise-error
		      (error-message '(abs syntax lhs level-expression complex)
				     value
				     "Level Expressions on lhs must only be simple variables or constant expressions"))))

	      (if (abstraction-meta-variable-id-p value)
		  (progn
		    (when (member value lhs-parameter-variables)
		      (raise-error (error-message '(abs syntax lhs meta-variable duplicate) value)))
		    (push value lhs-parameter-variables))
		  (when (and (or (symbolp value) (stringp value)) (eql (char (string value) 0) #\$))
		    (format t " $ as first char of parameter in abs def??? : ~a " value)
		    (break "ding-ding-ding")
		    nil))
	  )))

      (do ((bound-terms (bound-terms-of-term lhs) (cdr bound-terms))
	   (i 0 (1+ i)))
	  ((null bound-terms))
      
	;;	<lhs-bound-term>	: <binding> list.variable{<target>:v}
	;;							 ([variable{<binding>:v}] list)
	(let ((term  (term-of-bound-term (car bound-terms))))
	  (unless (variable-term-p term)
	    (raise-error (error-message '(abs syntax lhs subterm variable not)
					i term)))

	  ;; <target> must be real valued (ie not meta).
	  (when (meta-variable-id-p (id-of-variable-term term))
	    (raise-error (error-message '(abs syntax lhs variable meta)
					i term))))

	;; The value of <binding> in the binding list and the variable injection should
	;;   be the same in the instance. Ie a,b.variable{A:v}(a; b) is proper, but
	;;   a,b.variable{A:v}(b; a) and a,b.variable{A:v}(c; d) are not.
	;;  There should be no duplicates in the binding list.
	(with-variable-minor-invocation
	    (do ((bindings (bindings-of-bound-term-r (car bound-terms))
			   (cdr bindings))
		 (var-subs (bound-terms-of-term
			    (term-of-bound-term (car bound-terms)))
			   (cdr var-subs))
		 (j 0 (1+ j)))
		((and (null bindings) (null var-subs)))
	  
	      (cond
		;; more subterms then bindings of lhs second order var.
		((null bindings)
		 (raise-error (error-message '(abs syntax lhs subterm arity) i)))

		;; more bindings than subterms of lhs second order var.
		((null var-subs)
		 (raise-error (error-message '(abs syntax lhs subterm arity) i)))

		;; subterm of lhs second order var not a first order var.
		((not (variable-p (term-of-bound-term (car var-subs))))
		 (raise-error (error-message '(abs syntax lhs subterm subterm variable not)
					     i j
					     (term-of-bound-term (car bound-terms)))))

		;; id of variable subterm of lhs second order variable does
		;; not match binding of second order variable.
		((not (eql (car bindings) 
			   (id-of-variable-term
			    (term-of-bound-term (car var-subs)))))
		 (raise-error (error-message '(abs syntax lhs subterm binding subterm match not)
					     i j
					     (term-of-bound-term (car bound-terms)))))

		;; binding occurs more than once in same lhs binding list.
		((set-variable-minor-use (car bindings))
		 (raise-error (error-message '(abs syntax lhs subterm binding duplicate)
					     i j
					     (term-of-bound-term (car bound-terms)))))

		;; looks good.
		(t nil))))
      
	;; not lhs variable.
	(let ((so-var (cons (id-of-variable-term (term-of-bound-term (car bound-terms)))
			    (length (bindings-of-bound-term (car bound-terms))))))
	  (when (member so-var lhs-variables :key #'car)
	    (raise-error (error-message '(abs syntax lhs second variable duplicate)
					i (car so-var))))

	  (push so-var lhs-variables)))


      ;;
      ;;  Check RHS.
      ;;

      (when (inull-abstraction-term-p rhs)
	(return-from abstraction-translate-source-r))

      (labels
	  ((note-error (msg)
	     (setf error-p t)
	     (message-emit msg))
	 
	   (rhs-variable-usage (v arity)
	     (when (meta-variable-id-p v)
	       (note-error
		(error-message '(abs syntax rhs variable meta)
			       (string v))))

	     (let ((item (cons v arity)))
	       (unless (member item lhs-variables :test #'equal)
		 (note-error (error-message '(abs syntax rhs second variable unbound))))
	       (unless (member item rhs-variables :test #'equal)
		 (push item rhs-variables))))
	 
	   (visit (term)
	     (when (iplaceholder-term-p term)
	       (incf num-slots)
	       (pushnew 'term slot-types))
	     (when (and (equal-term-sigs-p lhs-term-sig (term-sig-of-term term))
			(abstraction-match-p lhs term))
	       ;;(setf -term term -lhs-term-sig lhs-term-sig) (break "ssr")
	       (note-error (error-message '(abs syntax self-reference))))
	     (when (variable-operator-p (operator-of-term term))
	       (cond
		 ;; second order variable.
		 ((not (null (bound-terms-of-term term)))

		  ;;  No second order variable occurence may be in the scope a similar first order
		  ;;  binding. Note that there is no notion of second order binding in our syntax.
		  (when (bound-var-p (id-of-variable-term term))
		    (note-error
		     (error-message '(abs syntax rhs second bound)
				    (id-of-variable-term term))))

		  ;;  All subterms of a second order variable must have nil binding lists.
		  (when (exists-p #'(lambda (bt)
				      (not (null (bindings-of-bound-term-n bt))))
				  (bound-terms-of-term term))
		    (note-error
		     (error-message '(abs syntax rhs second subterm bound)
				    term)))
		
		  (rhs-variable-usage (id-of-variable-term term)
				      (length (bound-terms-of-term term))))

		 ;; free first-order variable.
		 ((free-var-p (id-of-variable-term term))
		  (rhs-variable-usage (id-of-variable-term term) 0))))

	     ;;	* Any variable occuring free must occur on lhs with same arity.

	     (dolist (parameter (parameters-of-term term))
	       ;; Parameters must be real, meta, or slots.
	       (let ((value (value-of-parameter parameter)))
		 (cond
		   ((slot-parameter-value-p value)
		    (incf num-slots)
		    (pushnew (type-id-of-parameter parameter) slot-types))

		   ((and (level-expression-parameter-p parameter)
			 (real-parameter-value-p value (type-of-parameter parameter)))
		    (walk-parameter-value
		     value
		     (type-of-parameter parameter)
		     #'(lambda (v)
			 (when (not (member v lhs-level-variables))
			   (note-error
			    (error-message '(abs syntax rhs parameter level variable unbound)
					   v)))
			 v)))

		   ;; No parameters of type variable may contain meta values.
		   ((variable-parameter-p parameter)
		    (unless (real-parameter-value-p value (type-of-parameter parameter))
		      (note-error (error-message '(abs syntax rhs parameter not-real variable variable)
						 (parameter-to-pretty-string parameter)))))
		
		   ((abstraction-meta-variable-id-p value)
		    (when (not (member value lhs-parameter-variables))
		      (note-error (error-message '(abs syntax rhs parameter meta variable unbound)
						 (string value)))))

		   (t (when (not (real-parameter-value-p value (type-of-parameter parameter)))
			(note-error (error-message '(abs syntax rhs parameter not-real)
						   (parameter-to-pretty-string parameter))))))))

	     (mapc #'(lambda (bound-term)
		       (let ((bindings (bindings-of-bound-term-n bound-term)))
			 (when (exists-p #'slot-parameter-value-p bindings)
			   (incf num-slots)
			   (pushnew 'binding slot-types))
			 (enter-bindings bindings)
			 (visit (term-of-bound-term bound-term))
			 (exit-bindings bindings)))
		   (bound-terms-of-term term))))

	;; traverse rhs 
	(with-variable-invocation (visit rhs))

	;; check if all lhs variables occur on rhs.
	(dolist (so-var rhs-variables)
	  (unless (member so-var lhs-variables :test #'equal)
	    (message-emit (warn-message '(abs syntax lhs second variable unused) so-var))))

	;;(setf -error-p error-p) (break "lib-abs")

	(when (not (zerop num-slots))
	  (message-emit (warn-message '(abs syntax slots) (list num-slots slot-types))))
	(when error-p
	  (raise-error (error-message '(abs syntax))))
    
	t))))
