
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************

#+cltl2(in-system-package)
#-cltl2(in-package *system-package-name*)

;;;;	
;;;;	Importing MetaPRL (MP) dforms : 
;;;;
;;;;    MP dforms dynamically expand during display. We can not support dynamic expansion
;;;;	however we can statically expand a subset at dform definition time. IE, Expandable
;;;;	subterms of the dform format list will be expanded at definition time.
;;;;	
;;;;	  - MODEL : dforms whose model(pattern) term contains a subterm which is not a variable
;;;;	    but which contains free-vars is disallowed.
;;;;	
;;;;	  - SLOT : any dform containing a slot which contains more than one subterm or whose subterm is not
;;;;	    a variable is disallowed.
;;;;	
;;;;	  - FORMAT : any dform containing an unrecognized subterm is disallowed, presuming that the unrecoginzed
;;;;	    term is an instance of a disallowed dform expansion.
;;;;	
;;;;	
;;;;	
;;;;	All allowed dforms define an abstraction as well as a dform. The abstractions are used
;;;;	to expand formats in later imported dforms.
;;;;	
;;;;	  - normally the dform formats are expanded dynamically at display time.
;;;;	      * expansion is parameterized by the modes.
;;;;	
;;;;	  - each dform is tagged with a list of modes.
;;;;	      * assume no modes means unconditional.
;;;;	      * assume if modes but during display if current mode parameter is not
;;;;		a member of modes then ignore.
;;;;	
;;;;	  * each MP dform will define an abstraction and a dform. 
;;;;	      - MP abstractions will be expanded away when checking dforms.
;;;;	      - If a format is an instance of an MP abs with modes then
;;;;		one dform for each mode should be defined so that format
;;;;		can be expanded. The dforms will inherit the mode.
;;;;	      - If all instances are unconditional or have equivalent modes
;;;;		or all are a subset of the modes of the dform being defined 
;;;;		then only one dform need be defined with those modes.
;;;;	
;;;;	  * each MP abstraction will have only simple formats in expansion since
;;;;	    it is derived from a pre-expanded dform.
;;;;	
;;;;	
;;;;	
;;;;	
;;;;	Import MP dform:
;;;;	 
;;;;	  - (source-reduce tterm '(MP_import)) 
;;;;	    expands primitive MP formats to Nuprl equivalents, ie zones, etc.
;;;;	  - (source-reduce lhs (cons 'METAPRL modes))
;;;;	    expand defined MP formats
;;;;	      * If expansion result contains MP terms then DISALLOW SLOT or FORMAT.
;;;;	  - fixup-lhs : converts operators not easily converted via expansion.
;;;;	    eg, cons -> lhs_cons; 
;;;;	      * at the moment, it is not convenient to define an expansion for cons
;;;;		which we can be sure will not affect other inappriate occurences.
;;;;	  - fixup-rhs : converts MP-vars to display meta forms.
;;;;	    DISALLOW inappropriate MODEL term
;;;;	
;;;;	
;;;;	
;;;;	Modes -> conditions, Each MetaPrl dform and abs should get extra METAPRL condition.
;;;;	
;;;;	For each METAPRL abs instance format
;;;;	  accumulate conditions
;;;;	
;;;;	  If no abs instance formats or all conditions equal or single METAPRL
;;;;	    then build single abs and dform with those conditions.
;;;;	  else union accumulated conditions, plus modes of dform itself.
;;;;	    foreach condition build dform and abs with formats reduced in scope of condition.
;;;;	
;;;;	

;;;;	
;;;;	Build a psuedo lattice from nl declarations.
;;;;	
;;;;	assume limited depth. 
;;;;	
;;;;	list of unrelated
;;;;	each unrelated is ordered,
;;;;	each ordered is list of unrelated equivalence classes.
;;;;	

;; all lists have are consed with a tok to facilitate destructive updates.


;; when given set of relations, heuristic should be to find
;; maximimal ordered chains. Or to do all pairs with same lhs first.
;; maybe this find elements not occuring on lhs.
;; add pairs containg those elements,
;; then find pairs whose lhs are rhs of previous set.

;;;;	tlt : destructive tagged list tree. Tag is primarily excuse for initial cons
;;;;		to facilitate in place destructive updates.
;;;;		 
;;;;	<tlt>		: (<tag> . <tlt-element> list)
;;;;	<tlt-element>	: <tlt> | *
;;;;	
;;;;	
;;;;	tlt-make   (<tag> <tlt-element>)	: <tlt>
;;;;	
;;;;	tlt-push   (<tlt> <tlt-element>)	: NULL
;;;;	
;;;;	tlt-append (<tlt> <tlt-element>)	: NULL
;;;;	  * adds to end of list.
;;;;	
;;;;	tag-of-tlt  (<tlt>)			: <tag>
;;;;	list-of-tlt (<tlt>)			: <tlt-element> list
;;;;	
;;;;	tlt-lookup  (<tlt> INT)			: <tlt-element>
;;;;	
;;;;	tlt-insert (<tlt> INT <tlt-element>)	: NULL
;;;;	tlt-add    (<tlt> INT <tlt-element>)	: NULL
;;;;	
;;;;
;;;;	tlt-tail   (<tlt> INT)			: <tlt-element> list
;;;;	tlt-rplacd (<tlt> INT <tlt-element> list)
;;;;	  : NULL
;;;;	
;;;;	tlt-split  (<tlt> INT{n})		: (<tlt-element> list . <tlt-element> list)
;;;;	  * last elemnt of first result list is nth element
;;;;	
;;;;	


(defmacro list-of-tlt (tlt) `(cdr ,tlt))
(defmacro tag-of-tlt (tlt) `(car ,tlt))
(defmacro tlt-set-list (tlt l) `(setf (cdr ,tlt) ,l))
(defmacro tlt-push (tlt i) `(setf (cdr ,tlt) (cons ,i (cdr ,tlt))))
(defmacro tlt-delete (tlt i) `(setf (cdr ,tlt) (delete ,i (cdr ,tlt))))
(defmacro tlt-make (tag i) `(cons ,tag (list ,i)))
(defmacro tlt-make-l (tag l) `(cons ,tag ,l))
(defmacro tlt-lookup (tlt n) `(nth ,n (list-of-tlt ,tlt)))
(defmacro tlt-tail (tlt n) `(nthcdr ,n (list-of-tlt ,tlt)))
(defun tlt-split (tlt n)
  (if (zerop n)
      (cons nil (list-of-tlt tlt))
      (let* ((l (list-of-tlt tlt))
	     (r (nthcdr (1- n) l))
	     (rr (cdr r)))
	(setf (cdr r) nil)
	(cons l rr))))

(defmacro tlt-replacd (tlt n l)
  (let ((point (gensym)))
  `(if (zerop ,n)
    (setf (cdr ,tlt) ,l)
    (let ((,point (nthcdr (1- ,n) (list-of-tlt ,tlt))))
      (setf (cdr ,point) ,l)))))
	  
(defmacro tlt-insert (tlt n i)
  `(tlt-replacd ,tlt ,n (cons ,i (tlt-tail ,tlt ,n))))

(defmacro tlt-add (tlt n i)
  `(tlt-replacd ,tlt (1+ ,n) (cons ,i (tlt-tail ,tlt (1+ ,n)))))
  
(defmacro tlt-nconc (tlta tltb)
  (let ((tltaa (gensym)))
    `(let ((,tltaa ,tlta)) (tlt-replacd ,tltaa (length (list-of-tlt ,tltaa)) (list-of-tlt ,tltb)))))

(defmacro tlt-maybe-p (e)
  `(and (consp ,e) (symbolp (car ,e)) (listp (cdr ,e))))


(defun make-bound-term (term &optional bindings)
  (construct-bound-term term bindings))

;; depth first.
(defun tlt-search (tlt target &key (test #'eql))
  (labels ((mysearch (i l)
	     ;;(setf -i i -l l -t target) (break "tltf")
	     (if (null l)
		 nil
		 (let ((e (car l)))
		   (if (funcall test target e)
		       (list i)
		       (or (when (tlt-maybe-p e)
			     (let ((paddr (mysearch 0 (list-of-tlt e))))
			       (when paddr (cons i paddr))))
			   (mysearch (1+ i) (cdr l))))))))
    
    (mysearch 0 (list-of-tlt tlt))))

(defun tlt-find (tlt addr)
  (if (null addr)
      tlt
      (tlt-find (tlt-lookup tlt (car addr)) (cdr addr))))
  

;; next    
;; set-next
;;(defmacro tlt-next (tlt n) `(tlt-lookup ,tlt (1+ ,n)))


(defvar *mp-lattice* '(top)) ;; (tlt-make-l 'top nil)

#|
(defun mp-prec-address (a)
  (labels ((find-unr (i unr)
	     (if (null unr)
		 nil
		 (let ((paddr (find-ord 0 (cdar unr))))
		   (if paddr
		       (cons i paddr)
		       (find-unr (1+ i) (cdr unr))))))

	   (find-ord (i ord)
	     (if (null ord)
		 nil
		 (let ((paddr (find-ord 0 (cdar ord))))
		   (if paddr
		       (cons i paddr)
		       (find-unre (1+ i) (cdr ord)))))
	   
	   (find-unre (i unr)
	     (if (null unr)
		 nil
		 (if (member a (cdar unr))
		     (list i)
		     (find-unre (1+ i) (cdr unr))))))

    (find-unr 0 *mp-lattice*))))
|#

(defun mp-declare-lt (mpl a b)

  (let ((aa (tlt-search mpl a))
	(ab (tlt-search mpl b)))

       ;; (butlast aa 1) or (subseq 0 3) - gives equ tlt. 
       ;; (butlast aa 2) or (subseq 0 2) - gives unr tlt. 
       ;; (butlast aa 3) or (subseq 0 1) or lookup (car aa) - gives ord tlt. 
       ;; (butlast aa 4) or nil - gives top tlt. 

    (setf -aa aa -ab ab -a a -b b) ;; (break "mpdlt")

    (cond
      ((and (null aa) (null ab))
       (tlt-push mpl (tlt-make-l 'ord
				 (list (tlt-make 'unr (tlt-make 'equ a))
				       (tlt-make 'unr (tlt-make 'equ b))))))

      ((null aa)
       ;; put a in unrelated list prior to ab
       (let ((ord (tlt-lookup mpl (car ab)));; find ord tlt of b.
	     (abo (cdr ab)))

	 ;; '(top ... ('ord ... ('unre ...('equ ... 'b) )))
	 (cond
	   ((zerop (car ab))		; first in ord
	    (tlt-push ord (tlt-make 'unr (tlt-make 'equ a))))

	   ((onep (car ab))		; one unrelated tlt prior.
	    (tlt-push (tlt-lookup ord 0)
		      (list 'equ a)))

	   (t
	    (format t "Warn: Ambiguous: a < b, H < [b] < J, ((length H) > 1).~%")
	    ;; stick it in last of H anyways.
	    (tlt-push (tlt-lookup ord (1- (car abo)))
		      (list 'equ a))))))
		     
      ((null ab)
       ;; put b in unrelated list following aa
       (let ((ord (tlt-lookup mpl (car aa)))
	     (aao (cdr aa)))
	 
	 (let ((rest-ord (tlt-tail ord (car aao))))
	   ;;(setf -ro rest-ord -ord ord -aao aao) (break "yo")
	   (cond
	     ((null (cdr rest-ord))
	      (tlt-add ord (car aao) (tlt-make 'unr (tlt-make 'equ b))))
	     ((null (cddr rest-ord))
	      (tlt-push (cadr rest-ord) (tlt-make 'equ b)))
	     (t
	      ;; stick it in first of J anyways.
	      (format t "Warn: Ambiguous: a < b, H < [a] < J and length J is > 1.~%")
	      )))))
      
      ;; in same ord list.
      ;;  ... [a] < ... < [b]  	-> noop.
      ;;  H < [a,b ...] < J		-> H < [a] < [b ...] < J  or H < [a ...] < [b] < J
      ((eql (car ab) (car aa))
       (let ((ord (tlt-lookup mpl (car aa))))

	 (cond
	   ((< (cadr aa) (cadr ab)))	; correctly ordered already!

	   ((> (cadr aa) (cadr ab))	; </> contradiction.
	    (format t "Warn: Contradiction: a<b, H < [b] < J < [a] < K.~%"))

	   ((equal aa ab)		; </= contradiction.
	    (format t "Warn: Contradiction: a<b, H < [(a=b)] < J.~%"))

	   (t				; in unordered subset of same ordered element.
	    ;; need to partition the unrelated list.
	    (let* ((unr (tlt-lookup ord (cadr aa)))
		   (unrl (list-of-tlt unr))
		   (equa (tlt-lookup unr (caddr aa))))

	      (when (not (= 2 (length unrl)))
		(format t "Warn: Ambiguous: a < b, H < [a,b,c] < J and c is not empty, ie does c go with a or stay with b?.~%"))

	      (tlt-delete unr equa)
	      (tlt-insert ord (cadr aa) (tlt-make 'unr equa))
	      )))))
      (t
       ;; [a] < [b] < [c] and [n] < [m] < [o]  and b < m
       ;; [a] < [b, n] < [m,c] < [o]  ;; ng since c could be > o
       ;; [a] < [b, n] < [m] < [o,c]  ;; ng since c could be < m
       ;; ie do not know c's relation to o and m.

       ;; cases we can handle :
       ;;  H < [a] and [b] < J		-> H < [a] < [b] < J
       ;;  [a] and [c] < [b] < J	-> [a,c] < [b] < J
       ;;  H < [a] < [c] and [b] 	-> H < [a] < [c, b]
       (let ((orda (tlt-lookup mpl (car aa)))
	     (ordb (tlt-lookup mpl (car ab))))

	 (cond
	   ;;  H < [a] and [b] < J	-> H < [a] < [b] < J
	   ;;  H < [a] and [b]	 	-> H < [a] < [b]
	   ;;  [a] and [b] < J		-> [a] < [b] < J
	   ((and (zerop (cadr ab))
		 (= (car aa) (1- (length (list-of-tlt orda)))))
	    (tlt-delete mpl ordb)
	    (tlt-replacd orda (car aa) (list-of-tlt ordb)))
	   
	   ;; [a] and [c] < [b] < J		-> [a,c] < [b] < J
	   ;; [a] and H < [c] < [b] < J		-> H < [a,c] < [b] < J but H?a
	   ;; note that if (cadr ab) is zero previous clause applies since (length orda) = 1 -> (= (car aa) (1- (length (list-of-tlt orda))))
	   ((onep (length (list-of-tlt orda)))
	    (tlt-delete mpl orda)

	    ;; if b not hd then warn.
	    (unless (onep (cadr ab))
	      (format t "Warn: Ambiguous: a < b, [a] and [c] < H < [b] < J  -> H < [a,c] < [b] < J but H?a.~%"))
	    ;; insert a into unr of head of ordb.
	    (let ((unrb (tlt-lookup ordb (1- (cadr ab)))))
	      (tlt-nconc unrb (car (list-of-tlt orda)))))
			   

	   ;;  H < [a] < [c] and [b]	 	-> H < [a] < [c,b]
	   ;;  H < [a] < [c] < J and [b] 	-> H < [a] < [c,b] < J	but b?J
	   ((onep (length (list-of-tlt ordb)))
	    (tlt-delete mpl ordb)

	    ;; if a one less than last then warn.
	    (unless (= (cadr aa) (- (length (list-of-tlt orda)) 2))
	      ;;  H < [a] < J and [b] 		-> H < [a] < [b] < J
	      (format t "Warn: Ambiguous: a < b, H < [a] < [c] < J and [b] -> H < [a] < [c,b] < J, but b?J.~%"))
	    (let ((unra (tlt-lookup orda (1+ (cadr aa))))
		  )
	      (tlt-nconc unra (car (list-of-tlt ordb)))))


	   ;;  H < [a] < K < [c]  and [d] < L < [b] < J
	   ;;	-> H < [a] < K < [c,d] < L < [b] < J
	   ;;   + unless K and L null, warn ambiguous K,c?L,b,d,j d,L?H,K,a,c.
	   ;; subsumes all following:
	   ;;  H < [a] < [c] and [d] < [b] < J	-> H < [a] < [c,d] < [b] < J, but c?b a?d.
	   ;;   + warn ambiguous c?b a?d.
	   ;; H < [a] < [c] and [b] < J		-> H < [a] < [b,c] < J, but c?J
	   ;; H < [a] and [c] < [b] < J		-> H < [a,c] < [b] < J,	but H?c

	   (t (tlt-delete mpl ordb)
	      (let ((ordb-l (list-of-tlt ordb))
		    (orda-last (tlt-lookup orda (1- (length orda)))))
		(tlt-nconc orda-last (car ordb-l))
		(tlt-replacd orda (length (list-of-tlt orda)) (cdr ordb-l))
		(format t "Warn: Ambiguous: a < b, H < [a] < K < [c]  and [d] < L < [b] < J -> H < [a] < K < [c,d] < L < [b] < J, but K,c?L,b,d,j d,L?H,K,a,c.")
		))))))

    ;;(setf -mpl mpl -aa aa -ab ab -a a -b b) (break "mpdlt")
    ))


;; assume  do all equ first then only need to produce list of singleton equiv classes.

(defun mp-declare-equal (mpl a b)

  ;; one or both not in mpl -> add to others equ tlt.
  ;; [a]  and [b]	-> [(a=b)]
  ;; buy preceding assumption then this are th only possible cases.

  
  (let ((aa (tlt-search mpl a))
	(ab (tlt-search mpl b)))

    ;; (butlast aa 1) or (subseq 0 3) - gives equ tlt. 
    ;; (butlast aa 2) or (subseq 0 2) - gives unr tlt. 
    ;; (butlast aa 3) or (subseq 0 1) or lookup (car aa) - gives ord tlt. 
    ;; (butlast aa 4) or nil - gives top tlt. 

    ;;(setf -mpl mpl -aa aa -ab ab -a a -b b) (break "mpdeq")

    (cond
      ((and (null aa) (null ab))
       (tlt-push mpl (tlt-make 'ord (tlt-make 'unr (tlt-make-l 'equ (list a b))))))
						
      ((null aa)
       ;; put a in equ list of b.
       (let ((equ (tlt-find (butlast ab) mpl)))
	 (tlt-insert equ 0 a)))

      ((null ab)
       ;; put b in equ list of a.
       (let ((equ (tlt-find mpl (butlast aa))))
	 (tlt-insert equ 0 b)))

      ((equal (butlast aa) (butlast ab)))
	       
      (t (when (eql (car aa) (car ab))
	   ;; shouldn't be related unless equal. since doing all equal before any inequality.
	   (break "mpde"))

	 ;; ord index and unr index should be zero for both.
	 (unless (and (zerop (cadr aa))
		      (zerop (caddr aa))
		      (zerop (cadr ab))
		      (zerop (caddr ab)))
	   (break "mpde"))

	 ;;merge
	 (let ((equ (tlt-find mpl (butlast ab))))
	   (tlt-nconc equ (tlt-find mpl (butlast aa)))
	   (tlt-delete mpl (tlt-lookup mpl (car aa))))))
    
    ;; J < [a] < K < [b] < l	-> Contradiction.
    ;; J < [a,b] < K	-> J < [(b=a)] < K

    ;; [a] and J < [b] < K	-> J < [(b=a)] < K
    ;; J < [a] and [b] < K	-> J < [(b=a)] < K

    ;; J < [a] < K and L < [b] < M	-> J < L <  [a=b] < K < M, but J?L K?M

    ))


;;;;	
;;;;	Mp_prec_pair (labels relations)
;;;;
;;;;	labels can be mapped to tokens
;;;;	
;;;;	label l -> !dform_address{:t, l:s} in prec object l.
;;;;	relation : op name; name
;;;;	
;;;;	name needs to name a prec object. ie if name is l
;;;;	name -> !precedence_object{l:t}
;;;;
;;;;	
;;;;	ie map over labels and produce prec objects with !dform_address{:t, l:s} contents.
;;;;	 then map over relations and produce prec objects with 
;;;;	 !precedence_<rel>(!precedence_object{l1};!precedence_object{l2}) contents.
;;;;	

(define-primitive |!mp_prec_pair| () (labels relations))
(define-primitive |!mp_prec_cons| () (car cdr))
(define-primitive |!mp_prec_rel_cons| () (car cdr))
(define-primitive |!mp_prec_rel| ((token . rel)) (fst snd))
 

;; map over labels of prec pair and produce !dform_address labels.
(defun mp-prec-string (mplabel)
  (let ((ls (mapcar #'value-of-parameter-f
		    (value-of-parameter
		     (car (parameters-of-term mplabel))))))
    (apply #'concatenate 'string (cons "Modules_prec"
				       (mapcan #'(lambda (x)
						   (list "_" x))
					       ls)))))

(defun mp-prec-labels (mplabels)
  (mapcar #'(lambda (mplabel)
	      (let ((mpstring (mp-prec-string mplabel)))
		(cons (intern mpstring) (idform-address-term mpstring))))
	  mplabels))

(defun mp-prec-relation (term)
  (let ((r (rel-of-imp-prec-rel-term term)))
    
    (cond

      ((eql r '|lt|)
       (list r
	     (intern (mp-prec-string (snd-of-imp-prec-rel-term term)))
	     (intern (mp-prec-string (fst-of-imp-prec-rel-term term)))))

      ((eql r '|eq|)
       (list r
	     (intern (mp-prec-string (snd-of-imp-prec-rel-term term)))
	     (intern (mp-prec-string (fst-of-imp-prec-rel-term term)))))

      (t (setf -r r -term term) (break "mppr")))))

(defun mp-import-prec (place name term)
  (unless (equal '|Modules_prec_itt_theory_prec_apply| name)
    (let ((labels (when nil (mp-prec-labels (ilist-to-list (labels-of-imp-prec-pair-term term)
							   (imp-prec-cons-op)))))
	  (relations (when nil (mapcar #'mp-prec-relation
				       (ilist-to-list (relations-of-imp-prec-pair-term term)
						      (imp-prec-rel-cons-op))))))
  
      ;;(setf -a labels -r relations -term term) (break "nip")

      (when (not (is-lib-member name))
	(lib-create name 'com (list 'before place)))

      ;; set and check dform.
      (let ((o (library-object name)))
	(setf (object-term o) term))

      (when nil
	(dolist (label labels)

	  (when (not (is-lib-member (car label)))
	    (lib-create (car label) 'com (list 'before place)))

	  ;; set and check dform.
	  (let ((o (library-object (car label))))
	    (setf (object-term o) term)
	    )))

      (when nil
	(dotimeslist (i r relations)
		 
		     (let ((pname (intern (concatenate 'string (string name) "_precrel_" (princ-to-string i)))))

		       (when (not (is-lib-member pname))
			 (lib-create pname 'lat (list 'before place)))

		       ;; set and check dform.
		       (let ((o (library-object pname)))
			 (setf (object-term o) r)
			 (check-object o))
		       )))
      )))

(defun separate-list (l f)
  (let ((acc nil)
	(acco nil))
    (dolist (m l)
      (if (funcall f m)
	  (push m acc)
	  (push m acco)))
	
    (values acc acco)))

(defun mpl-to-iprec (mpl)
  (if (symbolp mpl)
      (idform-address-term mpl))
      (list-to-ilist-by-op (mapcar #'(lambda (m)
				   (mpl-to-iprec m))
			       (list-of-tlt mpl))
			   (case (tag-of-tlt mpl)
			     ((top unr)	(iprecedence-unrelated-op))
			     (ord	(iprecedence-ordered-op))
			     (equ	(iprecedence-equal-op))))))

(defun mp-prec-rehash (place)
  (let ((acc-eq nil)
	(acc-lt nil))

    (dolist (n library$)
      (let ((o (library-object n)))
	(when (eql (object-kind o) 'com)
	  (let ((term (object-term o)))
	    (when (imp-prec-pair-term-p term)
	      (mapcar #'(lambda (rterm)
			  (let ((r (mp-prec-relation rterm)))
			    (cond
			      ((eql (car r) '|lt|)
			       (push r acc-lt))
			      ((eql (car r) '|eq|)
			       (push r acc-eq))
			      (t (setf -rterm rterm -r r) (break "nlpr")))))
		      (ilist-to-list (relations-of-imp-prec-pair-term term) (imp-prec-rel-cons-op)))
	      )))))

    (let ((mpl (tlt-make-l 'top nil)))
      (dolist (e acc-eq)
	(mp-declare-equal mpl (cadr e) (caddr e)))

      ;; sort acc-lt
      ;; acc all names, then find names not occuring on any rhs.
      ;; find all rel's containing one of those names on lhs.
      ;;  allnames <- allnames - names.
      ;;  allrels <- allrels - rels
      ;; then find names not occuring on any rhs.
      ;; find all rels  containing one of those names on lhs.
      ;; etc.
      (let ((names nil))
	(dolist (r acc-lt)
	  (pushnew (cadr r) names)
	  (pushnew (caddr r) names))
	
	(let ((classes nil))

	  (do ((names names)
	       (rels acc-lt))
	      ((null rels))
	    (mlet* (((yrhs nrhs) (separate-list names #'(lambda (n) (member n rels :key #'caddr))))
		    ((acc acco) (separate-list rels
					       #'(lambda (r)
						   (member (cadr r) nrhs)))))
		   (push acc classes)
		   (setf  -acc-lt acc-lt -classes classes)
		   (setf rels acco
			 names yrhs)
		   
		   (when acco
		     ;;(setf -acco acco -acc acc) (break  "afk")
		     (let ((nnames nil))
		       (dolist (r acc)
			 (pushnew (caddr r) nnames))
		       (setf names nnames)))))

	  (mapc #'(lambda (class)
		    ;;(setf -mpl mpl -class class) (break "class")
		    (mapc #'(lambda (r) (mp-declare-lt mpl (cadr r) (caddr r))) class))
		(reverse classes))))

      ;; convert mpl to lattice object
      (setf -mpl mpl) (break "mpl")
      ;; convert mpl to precedence and install in object.
      (let ((pname '|MetaPrlPrecedence|))
	(when (not (is-lib-member pname))
	  (lib-create pname 'lat (list 'before place)))

	;; set and check prec
	(let ((o (library-object pname)))
	  (setf (object-term o) (mpl-to-iprec mpl))
	  (check-object o))
	))))


;; MP parens
;;  slot[p:s]{t} : parenthesis control:
;;          if p = "le" then print parens around t if its precedence is
;;             no more than the current precedence.
;;          if p = "lt" then print parens around t if its precedence is
;;             less than the current precedence.
;;          otherwise, do not print parens around t
;;      
;;      le == EQUAL
;;      lt == LESS
;;      otherwise == ALL
;;      


(defun mp-import-abort (tag)
  ;;(break "mp-import-abort")
  (throw 'mp-import-abort tag))

(defun idform-child-parens-term (r ld rd precinj)
  (construct-term (instantiate-operator '|!dform_child_parens|
					(list (instantiate-parameter-r r *token-type*)
					      (instantiate-parameter-r ld *string-type*)
					      (instantiate-parameter-r rd *string-type*)))
		  (list (make-bound-term precinj))))

(defun abs-instance-p (term name)
  (let ((abs (abstraction-of-term term)))
    (and abs (eql (name-of-definition abs) name))))

(defun mp-term-with-op-p (term opname)
  (let ((p (car (parameters-of-term term))))
    (when (and p (parameter-list-parameter-p p))
      (equal opname (mapcar #'value-of-parameter-f (value-of-parameter p))))))

(define-primitive |!mode_cons| nil ())

(defun imode-nil-term-p (term)
  (and (equal-operators-p (imode-cons-op)
			  (operator-of-term term))
       (null (bound-terms-of-term term))))


(define-primitive |mp_comment_disp| nil (comment))

(defun mp-map-op (term)
  (let* ((abs (abstraction-of-term term))
	 (oname (when abs
		  (name-of-definition abs))))

    (cond
      ((and oname (eql oname `|mp_slot|)
	    (variable-term-p (term-of-bound-term (car (bound-terms-of-term term)))))
       
       (let ((vname (string (id-of-variable-term
			     (term-of-bound-term (car (bound-terms-of-term term)))))))
	 (idform-variable-child-term vname vname (idform-child-parens-term 'all "(" ")" (void-term)))))

      ((and oname (eql oname `|mp_slot_le|)
	    (variable-term-p (term-of-bound-term (car (bound-terms-of-term term)))))
       
       (let ((vname (string (id-of-variable-term
			     (term-of-bound-term (car (bound-terms-of-term term)))))))
	 (idform-variable-child-term vname vname (idform-child-parens-term 'equal "(" ")" (void-term)))))

      ((and oname (eql oname `|mp_slot_lt|)
	    (variable-term-p (term-of-bound-term (car (bound-terms-of-term term)))))
       
       (let ((vname (string (id-of-variable-term
			     (term-of-bound-term (car (bound-terms-of-term term)))))))
	 (idform-variable-child-term vname vname (idform-child-parens-term 'less "(" ")" (void-term)))))
    
      ((and oname (eql oname `|mp_parm_slot|))
       ;;(setf -a term) (break "nps")
       (let ((vname (string (value-of-parameter (cadr (parameters-of-term term))))))
	 (idform-variable-child-term vname vname (idform-child-attr-nil-term))))

      ((and oname (eql oname `|mp_le_parm_slot|))
       ;;(setf -a term) (break "nps")
       (let ((vname (string (value-of-parameter (cadr (parameters-of-term term))))))
	 (idform-variable-child-term vname vname (idform-child-attr-nil-term))))

      ((and oname (eql oname `|mp_prec|))
       ;;(let ((pterm (term-of-bound-term (car (bound-terms-of-term term)))))
 	 ;;(idform-precedence-injection-term
	  ;;(idform-address-term (mp-prec-string pterm)))))
       (idform-attr-nil-term))

      ((imode-nil-term-p term)
       (idform-attr-nil-term))
    
      ((imode-cons-term-p term)
       ;;(setf -a term) (break "mode")
       (let ((a (icar term))
	     (d (icdr term)))
	 (idform-attr-cons-term
	  (if (itoken-term-p a)
	      (icondition-term (string (token-of-itoken-term a)))
	      a)
	  (if (itoken-term-p d)
	      (icondition-term (string (token-of-itoken-term d)))
	      d))))

      ((idform-attr-cons-term-p term)
       term)

      ((idform-attr-nil-term-p term)
       term)

      ((itext-term-p term)
       term)

      ((and oname (eql oname `|mp_perv_nil|))
       term)

      ((and oname (eql oname `|mp_perv_cons|))
       term)

      ((and oname (eql oname `|mp_pushfont|))
       (mp-comment-disp-term term))

      ((and oname (eql oname `|mp_popfont|))
       (mp-comment-disp-term term))

      (t (setf -term term) ;;(break "hello")
	 term)
      )))

(defun mp-map-term (term &optional f)
  (labels ((visit (term)
	     (if (mp-comment-disp-term-p term)
		 term
		 (construct-term (operator-of-term term)
				 (mapcar #'(lambda (bt)
					     (make-bound-term (funcall (or f #'mp-map-op)
								       (visit
									(term-of-bound-term bt)))
							      (bindings-of-bound-term bt)))
					 (bound-terms-of-term term))))))
    (visit term)))

(defun mp-variable-term-p (term)
  (abs-instance-p term '|mp_var_ab|))

(defun mp-variable-term-p (term)
  (or (abs-instance-p term '|mp_var_ab|)
      (abs-instance-p term '|mp_var_ab3|)))

(defun mp-var-term-p (term)
  (mp-term-with-op-p term (list "var")))

(defun variable-of-mp-variable-term (term)
  (value-of-parameter (cadr (parameters-of-term term))))

(defun parameter-variable-id-p (id)
  (and (symbolp id)
       (or (variable-id-p id) (meta-variable-id-p id))))

(defun mp-fixup-dform-rhs (term)
  ;;(setf -term term)(break "fixup-rhs")
  (construct-term (instantiate-operator (id-of-term term)
					(mapcar #'(lambda (p)
						    (let ((v (value-of-parameter p)))
						      (if (parameter-variable-id-p v)
							  (instantiate-parameter
							   (get-template-variable-id (string v))
							   (type-of-parameter p))
							  p)))
						(parameters-of-term term)))

		  (mapcar #'(lambda (bt)
			      (let ((tt (term-of-bound-term bt)))
				(make-bound-term (if (or (mp-variable-term-p tt) (mp-var-term-p tt))
						     ;;(template-term (variable-of-mp-variable-term tt))
						     (itemplate-term (string (variable-of-mp-variable-term tt)))
						     (progn
						       (term-walk tt
								  #'mp-variable-term-p
								  #'(lambda (wt f) (mp-import-abort 'MODEL)))
						       tt))
						 (mapcar #'(lambda (b)
							     (when (parameter-variable-id-p b)
							       (format t "~%~%parameter binding variable~%~%")
							       ;;(break "pvidp")
							       )
							     (if (or (parameter-variable-id-p b) (variable-id-p b))
								 (get-template-variable-id (string b))
								 b))
							 (bindings-of-bound-term bt)))))
			  (bound-terms-of-term term))))


(defun fixup-dform-child (term)
  (mp-map-term term #'(lambda (term)
			(if (idform-variable-child-term-p term)
			    (let ((attrs (attributes-of-idform-variable-child-term term)))
			      (if (and (idform-attr-cons-term-p attrs)
				       (variable-term-p (icar attrs)))
				  (idform-variable-child-term (string (id-of-variable-term (icar attrs)))
						     (descriptor-of-idform-variable-child-term term)
						     (icdr attrs))
				  (idform-variable-child-term (string (id-of-idform-variable-child-term term))
						     (descriptor-of-idform-variable-child-term term)
						     attrs)))
			    term))))
			

;; a filter to find dforms whose model terms may have more depth then expected
;; dform validiation will not object to these as they do not have embedded templates terms.
(defun find-non-simple-dforms (b e)
  (mapcan #'(lambda (n)
	      (let ((o (library-object n)))
		(when (eql 'disp (object-kind o))
		  (when (exists-p #'(lambda (dform)
				      (let ((tterm (term-of-dform dform)))
					(format t "~a"(length (bound-terms-of-term tterm)) )
					(exists-p #'(lambda (bt)
						      (format t "t")						      
						      (not (template-term-p (term-of-bound-term bt))))
						  (bound-terms-of-term tterm))))
				  (let ((dforms (display-forms-of-object o)))
				    (when dforms 
				      (list-of-dforms dforms))))
		    (list n)
		    ))))
	  (lib-segment b e)))


(defvar *hex-chars* (let ((acc nil))
		    (let ((s "01234567890abcdefABCDEF"))
		      (dotimes (i (length s))
			(push (char s i) acc)))
		    (nreverse acc)))
			
(defun escaped-hex-string-p (s)
  (and (>= (length s) 5)
       (eql #\\ (char s 0))
       (member (char s 1) *hex-chars*)
       (member (char s 2) *hex-chars*)
       (member (char s 3) *hex-chars*)
       (member (char s 4) *hex-chars*)
       )
  )

(defun hex-char-to-int (ch)
  (let ((code (char-code ch)))
    (cond
      ;; 0-9
      ((and (>= code 48) (<= code 57))
       (- code 48))

      ;; a-f
      ((and (>= code 97) (<= code 102))
       (+ 10 (- code 97)))

      ;; A-F
      ((and (>= code 65) (<= code 70))
       (+ 10 (- code 65)))

      (t (process-err  "hex-char-to-int")))))

(defvar *mp-font-map* '((125 . 125)
			(127 . 126)
			(128 . 127)
			(143 . 128)
			(145 . 129)
			(141 . 130)
			(131 . 131)
			(144 . 132)
			(153 . 133)
			(148 . 134)
			(156 . 135)
			(157 . 136)
			(158 . 137)
			(154 . 138)
			(155 . 139)
			(159 . 140)
			(160 . 141)
			(204 . 142)
			(205 . 143)
			(161 . 144)
			(162 . 145)
			(207 . 146)
			(191 . 147)
			(209 . 148)
			(176 . 149)
			(171 . 150)
			(163 . 151)
			(164 . 152)
			(206 . 153)
			(210 . 154)
			(135 . 155)
			(211 . 157)
			(212 . 158)
			(213 . 159)
			(214 . 160)
			(215 . 161)
			(216 . 162)
			(217 . 163)
			(224 . 165)
			(220 . 166)
			(222 . 167)
			(225 . 168)
			(234 . 169)
			(218 . 170)
			(219 . 171)
			(226 . 172)
			(208 . 173)
			(223 . 174)
			(221 . 175)
			(199 . 177)
			(194 . 178)
			(198 . 179)
			(227 . 180)
			(228 . 181)
			(229 . 182)
			(230 . 183)
			(231 . 184)
			(232 . 185)
			(233 . 186)
			(240 . 187)
			(241 . 188)
			(242 . 189)
			(243 . 190)
			(251 . 191)
			(192 . 192)
			(177 . 193)
			(237 . 194)
			(236 . 196)
			(235 . 197)
			(238 . 198)
			(254 . 199)
			(255 . 200)
			(253 . 201)
			(252 . 202)
			(127 . 204)
			(129 . 205)
			(130 . 207)
			(132 . 209)
			(133 . 206)
			(134 . 210)
			(136 . 234)
			(137 . 211)
			(138 . 212)
			(139 . 213)
			(140 . 214)
			(142 . 215)
			(146 . 216)
			(147 . 217)
			(149 . 164)
			(150 . 224)
			(151 . 220)
			(152 . 222)
			(165 . 225)
			(166 . 156)
			(167 . 218)
			(168 . 219)
			(169 . 226)
			(170 . 208)
			(172 . 223)
			(173 . 221)
			(174 . 176)
			(175 . 227)
			(178 . 228)
			(179 . 229)
			(180 . 230)
			(181 . 231)
			(182 . 232)
			(183 . 233)
			(184 . 240)
			(185 . 241)
			(186 . 242)
			(187 . 243)
			(188 . 251)
			(189 . 237)
			(190 . 999)
			(193 . 236)
			(196 . 238)
			(197 . 254)
			(200 . 255)
			(201 . 253)
			(202 . 252)))

(defun metaprl-char-map (i)
  (or (cdr (assoc i  *mp-font-map*)) i))

(defun convert-escaped-hex-string (s)
    
  (labels
      ((escaped-hex-string-p (i)
	 (and (>= (length s) 5)
	      (eql #\\ (char s i))
	      (member (char s (+ 1 i)) *hex-chars*)
	      (member (char s (+ 2 i)) *hex-chars*)
	      (member (char s (+ 3 i)) *hex-chars*)
	      (member (char s (+ 4 i)) *hex-chars*)
	      )))


    (let ((chars nil)
	  (i 0))
      (do ()
	  ((= i (length s)))
	(if (escaped-hex-string-p i)
	    (progn
	      (push (code-char (metaprl-char-map (+ (hex-char-to-int (char s (+ i 4)))
						    (* 16 (hex-char-to-int (char s (+ i 3))))
						    (* 256 (hex-char-to-int (char s (+ i 2))))
						    (* 4096 (hex-char-to-int (char s (+ i 1)))))))
		    chars)
	      (setf i (+ i 5)))
	    
	    (progn (push (code-char (metaprl-char-map (char-code (char s i)))) chars)
		   (incf i))))
      
      (let* ((l (length chars))
	     (news (make-string l))
	     )
	
	(dotimeslist (i ch chars)
		     (setf (char news (- l i 1)) ch))
	news))))

(defun variable-opid-p (id)
  (equal-opids-p *variable* id))

(defvar *metaprl-implementation-id* `|!metaprl_implementation|)
(defvar *metaprl-slot-opid-parm* (parameter-list-parameter (list (string-parameter "slot"))))
(defun metaprl-font-term-p (term)
  (equal-parameters-p (car (value-of-parameter (car (parameters-of-term term))))
		      (string-parameter "Nuprl_font")))
;; map perv{cons}(0;0)
;; map perv{nil}()   to !dform_lhs_cons
(defun mp-fixup-dform-lhs-old (term)
  (setf -term term)(break "fixup")
  (if;;(abs-instance-p term `|mp_perv_cons|)  ;; may not have abs in lib, better to check directly
   (mp-term-with-op-p term (list "Perv" "cons"));; opname is string list
   (let ((car (icar term)))
     (if (equal-parameters-p (car (parameters-of-term car)) *metaprl-slot-opid-parm*)
	 (idform-format-cons-term (mp-fixup-dform-lhs car)
				  (mp-fixup-dform-lhs (icdr term)))
	 (idform-lhs-cons-term (mp-fixup-dform-lhs car)
			       (mp-fixup-dform-lhs (icdr term)))))
   (if;;(abs-instance-p term `|mp_perv_nil|)
    (mp-term-with-op-p term (list "Perv" "nil"))       
    (construct-term (idform-lhs-cons-op))
    (if (eql (id-of-term term) *metaprl-implementation-id*)
	(if (equal-parameters-p (car (parameters-of-term term)) *metaprl-slot-opid-parm*)
	    (let ((l (length (bound-terms-of-term term))))
	      (if (or (zerop l) (onep l))
		  (progn (setf -a term)
			 (format t "slot should be ok")
			 ;;(break "slot should be ok")
			 (mp-import-abort 'SLOT))
		  (mp-import-abort 'SLOT)))
	    (progn
	      (format t "~%MetaPrl format opid : ~a~%" (parameter-to-string (car (parameters-of-term term))))
	      (mp-import-abort 'FORMAT)))
	(if (variable-term-p term)
	    (let ((vname (string (id-of-variable-term term))))
	      (idform-variable-child-term vname vname (idform-child-parens-term 'all "(" ")" (void-term))))
	    (if (and (variable-opid-p (id-of-term term))
		     (null (bound-terms-of-term term)))
		(progn (setf -term term)
		       (let ((vname (string (id-of-variable-term term))))
			 (idform-variable-child-term vname vname (idform-child-attr-nil-term)))
		       )
		(if (itext-term-p term)
		    (let ((s (string-of-itext-term term)))
		      (if (escaped-hex-string-p s)
			  (itext-term (convert-escaped-hex-string s))
			  term))
		    term)))))))

(defun mp-fixup-dform-lhs (term)
  ;;(setf -term term)(break "fixup")
  (if;;(abs-instance-p term `|mp_perv_cons|)  ;; may not have abs in lib, better to check directly
   (mp-term-with-op-p term (list "Perv" "cons"));; opname is string list
   (let ((car (icar term)))
     (if t;;(equal-parameters-p (car (parameters-of-term car)) *metaprl-slot-opid-parm*)
	 (idform-format-cons-term (mp-fixup-dform-lhs car)
				  (mp-fixup-dform-lhs (icdr term)))
	 (idform-lhs-cons-term (mp-fixup-dform-lhs car)
			       (mp-fixup-dform-lhs (icdr term)))))
   (if;;(abs-instance-p term `|mp_perv_nil|)
    (mp-term-with-op-p term (list "Perv" "nil"))       
    (construct-term (idform-lhs-cons-op))
    (if (eql (id-of-term term) *metaprl-implementation-id*)
	(if (equal-parameters-p (car (parameters-of-term term)) *metaprl-slot-opid-parm*)
	    (let ((l (length (bound-terms-of-term term))))
	      (if (zerop l)
		  ;; then has extra param, meta-level-exp?
		  (my-import-slot-parm (cadr (parameters-of-term term)))
		  (if (onep l)
		      (progn (setf -a term)
			     (format t "slot should be ok")
			     ;;(break "slot should be ok")
			     ;;(mp-import-abort 'SLOT))
			
			     (my-import-slot (icar term)))
		      (mp-import-abort 'SLOT))))
	    (if (metaprl-font-term-p term)
		term
		(progn (format t "~%MetaPrl format opid : ~a~%" (parameter-to-string (car (parameters-of-term term))))
		       ;;(setf -f term) (break)
		       (mp-import-abort 'FORMAT))))
	(if (variable-term-p term)
	    (let ((vname (string (id-of-variable-term term))))
	      (idform-variable-child-term vname vname (idform-child-parens-term 'all "(" ")" (void-term))))
	    (if (and (variable-opid-p (id-of-term term))
		     (null (bound-terms-of-term term)))
		(progn (setf -term term)
		       (let ((vname (string (id-of-variable-term term))))
			 (idform-variable-child-term vname vname (idform-child-attr-nil-term)))
		       )
		(if (itext-term-p term)
		    (let ((s (string-of-itext-term term)))
		      (if (escaped-hex-string-p s)
			  (itext-term (convert-escaped-hex-string s))
			  term))
		    term)))))))

(defun my-import-slot (term)
   (if (mp-term-with-op-p term (list "var"))
       (my-import-slot-parm (cadr (parameters-of-term term)))
       (progn (setf -term term ) ;;(break "import slot?")
	      (mp-import-abort 'FORMAT))))

(defun my-import-slot-parm (p)
  ;;(when (and p (variable-parameter-p p)))
  (let ((vname (string (value-of-parameter p))))
    (idform-variable-child-term vname vname (idform-child-attr-nil-term))))
       

(defun extract-conditions-from-attrs (attrs)

  (let ((alist (ilist-to-list attrs (idform-attr-cons-op))))
    (let ((conditions nil))
      (dolist (term alist)
	(cond

	  ((icondition-term-p term)
	   (push (term-to-conditional term) conditions))
	
	  ((idform-conditions-term-p term)
	   (setf conditions
		 (nconc conditions
			(mapcar #'term-to-conditional
				(ilist-to-list
				 (list-of-idform-conditions-term term)
				 (icond-cons-op))))))))
      conditions)))

(defun matrix-of-abstraction (abstraction &optional nil-ok-p)
  (unless nil-ok-p
	(let ((am (abstraction-matrix abstraction)))
    (unless am 
      (process-err "Abstraction has no matrix."))
    am)))


(defun extract-instance-conditions-from-formats (fterm)
  (let ((acc nil))
    (labels ((visit (term)
	       (let* ((abs (abstraction-of-term term))
		      (m (when abs (matrix-of-abstraction abs t)))
		      (conditionals (when abs (expansion-conditionals-of-abstraction abs))))
		 (if (and m abs conditionals)
		     (push conditionals acc)
		     (mapc #'(lambda (bt) (visit (term-of-bound-term bt)))
			   (bound-terms-of-term term))))))
      (visit fterm))

    ;;(when acc (setf -acc acc) (break "eicff")) ;; just curious.

    acc))

(defun mp-import-dform-term (name term)
  (let* ((dterm term)
	 
	 (tag (catch 'mp-import-abort
		(setf dterm (mp-import-dform-term-aux name term)))))
    ;;(setf dt term) (break "dt")
			  
    (if tag
	(progn
	  (when (eql tag 'slot) (terpri))
	  (format t "MP dform import aborted  [~a] : ~a~%~%" name tag)
	  dterm)
	dterm)))


(defun mp-import-dform-term-aux (name term)
  (let ((tterm (idform-term (attributes-of-idform-term term)
			    (rhs-of-idform-term term)
			    (lhs-of-idform-term term))))
    (let* ((rterm (source-reduce term '(MP_import)))) ;; conversions
      (setf -rterm rterm)
      (let ((modes (extract-conditions-from-attrs (attributes-of-idform-term rterm))))

	(let ((instance-modes (extract-instance-conditions-from-formats (lhs-of-idform-term rterm))))

	  (let ((unscoped-modes nil))
	    (dolist (imodes instance-modes)
	      (dolist (m imodes)
		(when (and (not (member m modes)) (not (eql 'metaprl m)))
		    (pushnew m unscoped-modes))))

	    (if unscoped-modes
		(progn
		  (setf -unscoped-modes unscoped-modes) (break "nid-un")
		  )
	
		(let* ((mterm (mp-map-term rterm))
		       (fterm (idform-term (idform-attr-cons-term (icondition-term "METAPRL")
								  (attributes-of-idform-term mterm))
					   (let ((dlhs (lhs-of-idform-term mterm)))
					     (mp-fixup-dform-lhs
					      (if instance-modes
						  (fixup-dform-child (source-reduce dlhs (cons 'METAPRL modes)))
						  dlhs)))
					   (mp-fixup-dform-rhs (rhs-of-idform-term mterm)))))
    
		  ;;(setf -n name -a term -b tterm -c rterm -d mterm -f fterm) (break "mp-id")
		  fterm ))))))))

(defvar *dform-names* nil)
(defvar *dform-prefix* "/")

(defun alias-of-name (name)
  (let* ((s1 (subseq (string name) 8));; take off modules prefix
	 (s2 (do ((s s1 (subseq s 1)))
		 ((string= "_" s :end2 1) (subseq s 1)))))
    (concatenate 'string *dform-prefix* s2)))

(defun mp-import-dform-term-alias (name term tok)
  (let* ((alias (concatenate 'string *dform-prefix* (string tok)));;"_df";;(alias-of-name name)
	 (tterm (idform-term (attributes-of-idform-term term)
			     (rhs-of-idform-term term)
			     (lhs-of-idform-term term))))
    (let* ((rterm (source-reduce term '(MP_import))));; conversions
      ;;(setf -rterm rterm) ;;(break "d")
      (let ((modes (extract-conditions-from-attrs (attributes-of-idform-term rterm))))

	(let ((instance-modes (extract-instance-conditions-from-formats (lhs-of-idform-term rterm))))

	  (let ((unscoped-modes nil))
	    (dolist (imodes instance-modes)
	      (dolist (m imodes)
		(when (and (not (member m modes)) (not (eql 'metaprl m)) (not (eql 'prl m)));;lal prl
		  (pushnew m unscoped-modes))))

	    (if unscoped-modes
		(progn
		  (setf -unscoped-modes unscoped-modes) (break "nid-un")
		  )
	
		(let* ((mterm (mp-map-term rterm))
		       (fterm (idform-term (idform-attr-cons-term (idform-edit-macro-term alias)
								  (idform-attr-cons-term (icondition-term "METAPRL")
											 (attributes-of-idform-term mterm)))
					   (let ((dlhs (lhs-of-idform-term mterm)))
					     (mp-fixup-dform-lhs
					      (if instance-modes
						  (fixup-dform-child (source-reduce dlhs (cons 'METAPRL modes)))
						  dlhs)))
					   (mp-fixup-dform-rhs (rhs-of-idform-term mterm)))))
    
		  ;; (setf -n name -a term -b tterm -c rterm -d mterm -f fterm) (break "mp-id")
		  fterm ))))))))


(defun idform-to-iabstraction (term)
  (let ((usages nil))
    
    ;; template{<t>} -> variable{t:v}
    ;; <t> -> $t
    ;; {<t>:t} -> {$t:t}
    (labels ((dform-rhs-to-abs-lhs-subterm (term)
	       ;; must be template. Could be fixed by allowing abs lhs to have constant subterms.
	       ;; or by some seperate dform expansion mechanism - probably best as there may be other expansions
	       ;; not fitting the abstraction form.
	       (if (template-term-p term)
		   (let ((v (id-of-template-term term)))
		     (push (cons (intern (string v)) 'term) usages)
		     (variable-term v))
		   (progn
		     (format t "abstraction lhs not suitable. ~%~%")
		     (return-from idform-to-iabstraction nil))))

	     (dform-rhs-to-abs-lhs (term)
	       (construct-term (instantiate-operator (id-of-term term)
						     (mapcar #'(lambda (p)
								 (if (template-parameter-p p)
								     (let ((s (string (value-of-parameter p))))
								       (push (cons (intern s) (type-id-of-parameter p)) usages)
								       (instantiate-parameter (get-parameter-variable-id s)
											      (type-of-parameter p)))
								     p))
							     (parameters-of-term term)))
			       (mapcar #'(lambda (bt)
					   (make-bound-term (dform-rhs-to-abs-lhs-subterm (term-of-bound-term bt))
								 (mapcar #'(lambda (b)
									     ;; need second order vars. not likely to be useful.
									     (return-from idform-to-iabstraction nil)
									     (let ((s (string b)))
									       (push (cons (intern s) 'variable) usages)
									       (if (template-variable-id-p b)
										   (get-parameter-variable-id s)
										   b)))
									 (bindings-of-bound-term bt))))
				       (bound-terms-of-term term))))
	     ;; very kludgey : 
	     ;;   briefly: 
	     ;;     - need dform abstractions to capture MP dform expansion
	     ;;     - our slots have no term positions for abstraction matrix variable to occur in.
	     ;;     - push into dform-child attrs and then pull out after expansion in dform being defined.
	     ;;
	     (dform-formats-to-abs-matrix (term)
	       (if (idform-variable-child-term-p term)
		   (let ((usage (cdr (assoc (intern (id-of-idform-variable-child-term term)) usages))))
		     (if (eql usage 'term)
			 (idform-variable-child-term "fu"
					    (string usage)
					    (idform-attr-cons-term (variable-term (get-variable-id (id-of-idform-variable-child-term term)))
								   (attributes-of-idform-variable-child-term term)))
			 (idform-variable-child-term (get-parameter-variable-id (id-of-idform-variable-child-term term))
					    (string usage)
					    (attributes-of-idform-variable-child-term term))
			 ))

		   (construct-term (operator-of-term term)
				   (mapcar #'(lambda (bt)
					       (make-bound-term (dform-formats-to-abs-matrix (term-of-bound-term bt))
								     (bindings-of-bound-term bt)))
					   (bound-terms-of-term term))))))

      (let ((drhs (rhs-of-idform-term term)))
	(let ((dabs (abstraction-of-term drhs)))
	  (when dabs
	    (when (member 'metaprl (expansion-conditionals-of-abstraction dabs))
	      (format t "duplicate abs already defined ~%~%")
	      )
	    (return-from idform-to-iabstraction nil)))

	(iabstraction-term (list-to-ilist-by-op (mapcar #'(lambda (cond)
							    (icondition-term (string cond)))
							(extract-conditions-from-attrs
							 (attributes-of-idform-term term)))
						(icond-cons-op))
			   (dform-rhs-to-abs-lhs drhs)
			   (dform-formats-to-abs-matrix (lhs-of-idform-term term)))))))

(defun mp-import-dform (name term place)
  (let ((tag (catch 'mp-import-abort
	       (let ((dterm (mp-import-dform-term name term)))

		 (let ((*suppress-lib-update* t))

		   ;; create obj if not present.
		   (when (not (is-lib-member name))
		     (lib-create name 'disp (list 'before place)))

		   ;; set and check dform.
		   (let ((o (library-object name)))
		     (setf (object-term o) dterm)
		     (check-object o))

		   (let ((absname (intern (concatenate 'string (string name) "_abs"))))

		     (when (is-lib-member absname)
		       (lib-delete (list absname absname)))
		     (let ((absterm (idform-to-iabstraction dterm)))
		   
		       ;;(setf -dterm dterm -absterm absterm) (break "nid")
		   
		       (when absterm

			 (when (not (is-lib-member absname))
			   (lib-create absname 'abs (list 'before place)))

			 (let ((o (library-object absname)))
			   (setf (object-term o) absterm)
			   (check-object o))
			 ))))))))
    (if tag
	(progn
	  (when (eql tag 'slot) (terpri))
	  (format t "MP dform import aborted  [~a] : ~a~%~%" name tag)
	  nil)
	t)))

(defun mp-import-dform2 (name term place alias)
  (let ((tag (catch 'mp-import-abort
	       (let ((dterm (mp-import-dform-term-alias name term alias)))

		 (let ((*suppress-lib-update* t))

		   ;; create obj if not present.
		   (when (not (is-lib-member name))
		     (lib-create name 'disp (list 'before place)))

		   ;; set and check dform.
		   (let ((o (library-object name)))
		     (setf (object-term o) dterm)
		     (check-object o))

		   (let ((absname (intern (concatenate 'string (string name) "_abs"))))

		     (when (is-lib-member absname)
		       (lib-delete (list absname absname)))
		     (let ((absterm (idform-to-iabstraction dterm)))
		   
		       ;;(setf -dterm dterm -absterm absterm) (break "nid")
		   
		       (when absterm

			 (when (not (is-lib-member absname))
			   (lib-create absname 'abs (list 'before place)))

			 (let ((o (library-object absname)))
			   (setf (object-term o) absterm)
			   (check-object o))
			 ))))))))
    (if tag
	(progn
	  (when (eql tag 'slot) (terpri))
	  (format t "MP dform import aborted  [~a] : ~a~%~%" name tag)
	  nil)
	t)))

(defun maybe-mp-import-dform2 (name term place alias)
  (let ((tag (catch 'mp-import-abort
	       (when (not (is-lib-member name))
		     (let ((dterm (mp-import-dform-term-alias name term alias)))

		 (let ((*suppress-lib-update* t))

		   ;; create obj if not present.
		   (when (not (is-lib-member name))
		     (lib-create name 'disp (list 'before place)))

		   ;; set and check dform.
		   (let ((o (library-object name)))
		     (setf (object-term o) dterm)
		     (check-object o))

		   (let ((absname (intern (concatenate 'string (string name) "_abs"))))

		     (when (is-lib-member absname)
		       (lib-delete (list absname absname)))
		     (let ((absterm (idform-to-iabstraction dterm)))
		   
		       ;;(setf -dterm dterm -absterm absterm) (break "nid")
		   
		       (when absterm

			 (when (not (is-lib-member absname))
			   (lib-create absname 'abs (list 'before place)))

			 (let ((o (library-object absname)))
			   (setf (object-term o) absterm)
			   (check-object o))
			 )))))))))
    (if tag
	(progn
	  (when (eql tag 'slot) (terpri))
	  (format t "MP dform import aborted  [~a] : ~a~%~%" name tag)
	  nil)
	t)))

(defunml (|mp_import_dform_term| (name term))
  (tok -> (term -> term))
  (format t "importing ~s ~%" name)
  (let ((ret (mp-import-dform-term name term)))
    (format t "imported ~s ~%" name)
    ret))
