
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************

#+cltl2(in-system-package)
#-cltl2(in-package *system-package-name*)


;;;;  MathBus term representation:
;;;;
;;;;	 All io across mathbus channels is in the form of mathbus terms (<mbterms>).  The
;;;;	 mbterm structure is shared across differing platforms and implementations, making
;;;;	 term communication between nuprl and metaprl (where term representation differs)
;;;;	 possible. Mbterms should be thought of as trees, and the syntax below gives a parent
;;;;	 node followed by a list of its children.  In lisp, these trees are implemented as
;;;;	 arrays.

;;;;
;;;;	 <mbterm>	: (<mbparameter> 1,... <mbparameter> n,
;;;;			   <mbterm> 1, <mbbindings> , <mbterm> 2,... <mbterm> k)
;;;;			| (TermIndex)
;;;;
;;;;	    * if a subterm node (represented as <mbterm>) has binding variables, it is
;;;;	      preceded by a <mbbindings> node, else it simply follows a <mbterm> or
;;;;	      <mbparameter> node 
;;;;
;;;;
;;;;	 <mbbindings>	: Stringid list
;;;;	    * in the near future, nuprl bindings will be represented as terms, and
;;;;		 this will become <mbterm> list
;;;;			 
;;;;
;;;;	 <mbparameter>	:  String | Variable | LongInteger | Token | LevelExpression
;;;;			| MString | MVariable | MLongInteger | MToken | MLevelExpression
;;;;			| ObjectId | ParmList
;;;;
;;;;	 Global mathbus registry labels:
;;;;
;;;;	 Variable, Token, M*	: string
;;;;	 ObjectId		: integer 
;;;;	 ParmList		: <mbparameter> list
;;;;	 LevelExpression	: (LongInteger > 0, LevelVariable 0,... LevelVariable p)
;;;;	 LevelVariable		: (String, LongInteger)
;;;;	 TermIndex		: integer

;;;;
;;;;	 Compression issues:
;;;;
;;;;	 mbterms may be tagged with mbs_Attributes describing desired compression task
;;;;	 ie.add term to compression level 1
;;;;	 this tag will be a 32 bit integer adhering to current nuprl compression implementation
;;;;

;;;;
;;;;	 Functions:
;;;;
;;;;
;;;;	 term-to-mbterm (term)				: <mbterm>
;;;;	 mbterm-to-term (mbterm)			: <term>
;;;;
;;;;
;;;;	 assumptions/limitations:
;;;;
;;;;	 1.  Bindings are sent as strings fttb (as implemented in metaprl), thus, nuprl5
;;;;	 can only send variable-ids, not meta or extended       
;;;;
;;;;	 2.  The first parameter, p0, determines whether the term is a nuprl5 term
;;;;	 or a metaprl term.  If it's a param list of strings, it is metaprl and p1 is
;;;;	 the opname, else nuprl5
;;;;
;;;;	 3.  The opname ["!nuprl5-implementation!"] is reserved to represent nuprl5 terms
;;;;	 in metaprl.  On nuprl5 side, |!metaprl_implementation|:t becomes opid,
;;;;	 (first parameter)
;;;;
;;;;	 4.  Nuprl parameters that have types not represented in metaprl, ie quote,
;;;;	 time, bool, are embedded in a parameter of type *parameter-list* and
;;;;	 value (nuprl5-type:t, value:n | t), depending on special type. if nuprl uses
;;;;	 param list this way, quote mechanism solves this.
;;;;	 Alternatively, could wrap term with map id, but former is easier to convert and
;;;;	 retrieve, no need to maintain indices, etc.  
;;;;
;;;;	 5.  Subterms and bindings are not collected independently, will turn out to be
;;;;	 inefficient if new term representation is adopted (but easy to change)
;;;;	 io-mbs-new-trms contains other version
;;;;
;;;;	 6.level expressions are not affected/normalized when communicating between lisp <-> lisp
;;;;	 level expressions read in ocaml from lisp are normalized and sorted.

;;;;
;;;; mathbus <-> nuprl term conversion
;;;;


;;;;    handles metaprl opname

(defvar *metaprl-token* (intern "!metaprl_implementation"))

(defun leaves-to-operator (leaves)
  (let ((param (car leaves)))
    (if (and (equal (type-of-parameter param) *parameter-list-type*)
	     (forall-p #'(lambda (p) (equal (type-of-parameter p) *string-type*))
		       (value-of-parameter param)))
	(instantiate-operator *metaprl-token* leaves)
      (instantiate-operator (intern (real-parameter-value-to-string
					   (value-of-parameter param) *token-type*))
			    (cdr leaves)))))


(defun mbterm-to-term (mbterm)
  ;;(unless (= (mbnode-label mbterm) MBS_Term)
	  ;;(raise-error (error-message '(mbterm-to-term label))))
  
  (let ((nsubterms (mbnode-nSubterms mbterm))
	(leaves nil)
	(bound-terms nil)
	(index 1)) 
    
    (do ((node (mbnode-subterm mbterm index)
	       (mbnode-subterm mbterm (min nsubterms index))))
	((let ((label (mbnode-label node)))
	   (or (= label MBS_Term) (= label MBS_Bindings) (> index nsubterms) 
	     ;;(= (mbnode-label node) MBS_TermIndex) ;; LAL term index later
	     )))
	(setf leaves (cons (mbparameter-to-parameter node) leaves))
	(setf index (1+ index)))
    ;;(format t " leaves:~s" leaves)
    
    (when (not (> index nsubterms))
	  (do ((node (mbnode-subterm mbterm index)
		     (mbnode-subterm mbterm (min nsubterms index))))
	      ((> index nsubterms))
	      (setf index (1+ index))
	      (let ((label (mbnode-label node)))
		(cond
		 ((= label MBS_Bindings)
		  (setf bound-terms
			(cons (cons (mbbindings-to-bindings node)
				    (mbterm-to-term (mbnode-subterm mbterm index)))
			      bound-terms))
		  (setf index (1+ index)))
		 ((= label MBS_Term)
		  (setf bound-terms (cons (cons nil (mbterm-to-term node));; LAL inst
					  bound-terms))))
		;;(format t " subterms:~s" bound-terms)
		)))
     
    ;;(format t " leaves:~s"  leaves)
    ;; (format t " subterms:~s"  subterms)
    ;; (format t " bindings:~s"  bindings)
    (instantiate-term (leaves-to-operator (reverse leaves)) bound-terms)))

#|  OR 
(loop while (not (> index nsubterms)) do
	  (let* ((label (mbnode-label node))
		 (node (mbnode-subterm mbterm index)))
		(setf index (1+ index))
		(cond
		 ((= label MBS_Bindings)
		  (setf bound-terms
			(cons (cons (mbbindings-to-bindings node)
				    (mbterm-to-term (mbnode-subterm mbterm index)))
			      bound-terms))
		  (setf index (1+ index)))
		 ((= label MBS_Term)
		  (setf bound-terms (cons (cons nil (mbterm-to-term node));; LAL inst
					  bound-terms))))
		;;(format t " subterms:~s" bound-terms)
		))
|#

(defun mbnode-to-binding (node)
  (let ((label (mbnode-label node)))
    (cond
     ((= label MBS_Variable)
      (maybe-string-to-parameter-value (string-value node) *variable-type*))
     ((= label MBS_MVariable)
      (get-abstraction-meta-variable-id
       (maybe-string-to-parameter-value (string-value node) *variable-type*)))
     ((= label MBS_ParmList)
      (let ((node1 (string-value (mbnode-subterm node 1)))
	    (node2 (string-value (mbnode-subterm node 2))))
	(cond
	 ((string= node1 "extended")
	  (slot-parameter-value
	   (if (= (mbnode-nSubterms node) 3)
	       (if (string= node2 "abstraction")
		   (get-abstraction-meta-variable-id (string-value (mbnode-subterm node 3)))
		 (get-display-meta-variable-id (string-value (mbnode-subterm node 3))))
	     node2)))
	 ((string= node1 "display")
	  (get-display-meta-variable-id
	   (maybe-string-to-parameter-value node2 *variable-type*)))
	 (t (break "foo")))))
     (t (break "fo")))))
					    
		   
	 

(defun mbbindings-to-bindings (mbterm)
  (unless (= (mbnode-label mbterm) MBS_Bindings)
	  (raise-error (error-message '(mbbindings-to-bindings label))))
  (let ((nsubterms (mbnode-nSubterms mbterm)))
    (do ((i 1 (+ 1 i))
	 (bindings nil;;(cons (variable-term
		   ;;     (get-variable-id (string-value
		   ;;		(mbnode-subterm mbterm i)))) bindings)
		   (cons (mbnode-to-binding (mbnode-subterm mbterm i))
			 bindings)))
	((> i nsubterms) (reverse bindings)))))

(defun params-of-mbs-pl (mbparameter nsubterms)
  (do ((i nsubterms (1- i))
       (parms nil (cons (mbparameter-to-parameter
			 (mbnode-subterm mbparameter i))
			parms)))
      ((= i 1) parms)))
  

(defun mbnode-to-level-parameter-q (mbparameter)
  (level-expression-parameter (string-to-level-expression (string-value (mbnode-subterm mbparameter 1)))))
(defun string-to-level-variable (str)
  (with-string-scanner (str) (scan-string *level-expression-escape-sbits*)))

(defun mbnode-to-level-expression-q (mbparameter)
  (string-to-level-expression (string-value mbparameter)))
   
(defun mbnode-to-level-parameter (mbparameter)
   (level-expression-parameter (mbnode-to-level-expression mbparameter)))

(defun mbnode-to-level-expression (mbparameter)
  (let* ((nsubterms (mbnode-nSubterms mbparameter))
	 (node1 (mbnode-subterm mbparameter 1))
	 (label (mbnode-label node1)))
    (cond
     ((= label MBS_String)
      (make-level-variable (string-value node1)))
     ((= label MBS_LongInteger)
      (make-level-constant (integer-value node1)))
     ((= label MBS_Level)
      (if (> 1 nsubterms)
	  (let ((node2 (mbnode-subterm mbparameter 2)))
	    (if (= (mbnode-label node2) MBS_LongInteger)
		(make-level-increment (mbnode-to-level-expression node1) (integer-value node2))
	      (do* ((i 2 (1+ i))
		    (node node2 (mbnode-subterm mbparameter i))
		    (exprs (list (mbnode-to-level-expression node1) (mbnode-to-level-expression node2))
			   (cons (mbnode-to-level-expression node) exprs)))
		   ((= i nsubterms) exprs))))
	(list (mbnode-to-level-expression node1))))
     (t (raise-error (error-message '(mbnode to level expression) label))))))

   
(defun mbnode-to-normalized-level-parameter (mbparameter)
   (let* ((nsubterms (mbnode-nSubterms mbparameter))
	     (constant (make-level-constant
			(1+ (integer-value (mbnode-subterm mbparameter 1)))));;lal 1+
	     (le-vars (do ((i 2 (+ 2 i))
			   (exprs nil
				  (cons (make-level-increment
					 (intern (string-value
							(mbnode-subterm mbparameter i)))
					 (integer-value (mbnode-subterm mbparameter (1+ i))))
					exprs)))
			  ((> i nsubterms) exprs))))
	(instantiate-parameter (cons constant le-vars) *level-expression-type*)))
   
(defun mbnode-to-normalized-level-expression (mbparameter)
  (let* ((nsubterms (mbnode-nSubterms mbparameter))
	 (constant (make-level-constant
		     (1+ (integer-value (mbnode-subterm mbparameter 1)))))
	 (le-vars (do ((i 2 (+ 2 i))
		       (exprs nil
			      (cons (make-level-increment
				     (intern (string-to-level-variable (string-value
									      (mbnode-subterm mbparameter i))))
				     (integer-value (mbnode-subterm mbparameter (1+ i))))
				    exprs)))
		      ((> i nsubterms) exprs))))
    (cons constant le-vars)))

;; all mappings are parm lists of 2 elements, first is special type, second is value
(defun mbparameter-to-parameter (mbparameter &optional quote-p nuprl5-p)
  (let ((label (mbnode-label mbparameter)))
    (cond
      ((= label MBS_String)
       (string-parameter
	(maybe-string-to-parameter-value (string-value mbparameter) *string-type*))) 
      ((= label MBS_Token)
       (token-parameter
	(maybe-string-to-parameter-value (string-value mbparameter) *token-type*)))
      ((= label MBS_TokenIndex)
       (token-parameter
	(maybe-string-to-parameter-value (gethash (integer-value mbparameter) mathbus::*token-table*) *token-type*)))
      ((= label MBS_StringIndex)
       (string-parameter
	(maybe-string-to-parameter-value (gethash (integer-value mbparameter) mathbus::*token-table*) *string-type*)))
      ((= label MBS_LongInteger)
       (natural-parameter (integer-value mbparameter)))
      ((= label MBS_Variable)
       (variable-parameter
	(maybe-string-to-parameter-value (string-value mbparameter) *variable-type*)))
      ((= label MBS_ObjectId)
       (let* ((process-id (value-of-parameter (mbparameter-to-parameter
					       (mbnode-subterm mbparameter 4))))
	      (transaction (value-of-parameter (mbparameter-to-parameter
						(mbnode-subterm mbparameter 3))))
	      (sequence (value-of-parameter (mbparameter-to-parameter
					     (mbnode-subterm mbparameter 1))))
	      (time (value-of-parameter (mbparameter-to-parameter
					 (mbnode-subterm mbparameter 2))))
	      (stamp (cons (cons transaction process-id)
			   (cons sequence time))))
	 (oid-parameter (new-oid stamp))))

      ((= label MBS_ParmList)
       (let* ((nsubterms (mbnode-nSubterms mbparameter))
	      (p1 (when (> nsubterms 0)
		    (mbparameter-to-parameter (mbnode-subterm mbparameter 1))))
	      (type1 (if p1 (type-id-of-parameter p1))))
	 (if (and (not quote-p) (token-typeid-p type1)
		  (> nsubterms 1) (< nsubterms 5))
	     (let ((value1 (value-of-parameter p1))
		   (node2 (mbnode-subterm mbparameter 2)))
	       ;; (setf v value1)(break)
	       (case value1			
		 (|quote|
		  (let ((label (mbnode-label node2)))
		    (if (= label MBS_ParmList)
			(mbparameter-to-parameter node2 t)
			(quote-parameter (intern (string-value node2))))))

		 (|extended|
		  (let ((value2
			 (case (intern (string-value node2))
			   (|slot| (if (= 4 nsubterms)
				       (let ((meta-type
					      (intern (string-value
							     (mbnode-subterm mbparameter 4)))))
					 (slot-parameter-value
					  (case meta-type
					    (|abstraction|
					     (get-abstraction-meta-variable-id "slot"))
					    (|a|
					     (get-abstraction-meta-variable-id "slot"))
					    (|display|
					     (get-display-meta-variable-id "slot"))
					    (|d|
					     (get-display-meta-variable-id "slot")))))
				       (slot-parameter-value "slot")))
			   (|error| (error-parameter-value
				     ""
				     (if (= 4 nsubterms)
					 (string-value (mbnode-subterm mbparameter 4))
					 nil)))
			   ;;   (|marks|)
			   (t (setf s (intern (string-value node2)))(break "mbs")
			      ;;(parameter-of-mbs-pl mbparameter nsubterms)
			      ))))
		    (instantiate-parameter value2
					   (type-id-to-type
					    (intern (string-value
							   (mbnode-subterm mbparameter 3)))))))
		 (|display|
		  (let ((param (mbparameter-to-parameter (mbnode-subterm mbparameter 2))))
		    (instantiate-parameter
		     (get-display-meta-variable-id (value-of-parameter param))
		     (type-of-parameter param))))
		 (|abstraction|
		  (let ((param (mbparameter-to-parameter (mbnode-subterm mbparameter 2))))
		    (instantiate-parameter
		     (get-abstraction-meta-variable-id (value-of-parameter param))
		     (type-of-parameter param))))
		
		 (|bool|
		  (if (equal (mbnode-label node2) MBS_LongInteger)
		      (bool-parameter (= 1 (integer-value node2)))
		      ;;meta here
		      (bool-parameter (string-value node2))))
		 (|time|
		  (time-parameter;;(logior (ash (integer-value node2) 16)
		   ;; (integer-value (mbnode-subterm mbparameter 3)))
		   (integer-value (mbnode-subterm mbparameter 2))))
		 (|nuprl5_level_expression|
		  (let* ((n1 (mbnode-subterm mbparameter 2))
			 (e1 (mbnode-to-normalized-level-expression n1))
			 (n2 (mbnode-subterm mbparameter 3))
			 (e2 (mbnode-to-level-expression-q n2)))
		    (if (equal-level-expressions-p e1 e2) 
			(level-expression-parameter e2) 
			(progn ;;(setf a e1 b e2);; (break "mb")
			       (level-expression-parameter e2)))))
	
		 (|abstract_parameter_type|
		  (string-to-parameter (string-value node2)))
	       
		 (t (parameter-list-parameter
		     (cons p1 (params-of-mbs-pl mbparameter nsubterms))))))

	     (parameter-list-parameter
	      (when p1 (cons p1 (params-of-mbs-pl mbparameter nsubterms)))))))
      ((= label MBS_Level)
       (let* (;;(n1 (mbnode-subterm mbparameter 1))
	      (e1 (mbnode-to-normalized-level-expression mbparameter)))
	
	 (meta-level-expression-parameter (level-expression-to-string e1))))
     
      ((= label MBS_MString)
       (string-parameter (get-abstraction-meta-variable-id (string-value mbparameter)))) 
      ((= label MBS_MVariable)
       (variable-parameter (get-abstraction-meta-variable-id (string-value mbparameter))))
      ((= label MBS_MToken)
       (token-parameter (get-abstraction-meta-variable-id (string-value mbparameter))))
      ((= label MBS_MLongInteger)
       (natural-parameter (get-abstraction-meta-variable-id (string-value mbparameter))))
      ((= label MBS_MLevel)
       (meta-level-expression-parameter (get-abstraction-meta-variable-id (string-value mbparameter))))
      ((= label MBS_TermIndex)
       (token-parameter
	(maybe-string-to-parameter-value (gethash (integer-value mbparameter) mathbus::*token-table*) *token-type*)))
      (t (raise-error (error-message '(mbparameter-to-parameter label)))))))
            


;;;; nuprl terms to mathbus terms

;;;; LAL bindings can be (or (extended-parameter-value-p b) (meta-variable-id-p b) (variable-id-p b)) but now only assuming variable
(defun term-to-mbterm (term)
  ;;(setf -term term) (break "mbt")
  (let* ((leaves (leaves-of-term term))
	 (nuprl5-p (not (and (token-typeid-p (type-id-of-parameter (car leaves)))
			     (string= (real-parameter-value-to-string
				       (value-of-parameter (car leaves)) *token-type*)
				    "!metaprl_implementation"))))
	 (params (mapcar #'(lambda (p) (parameter-to-mbparameter p nil nuprl5-p))
			 (if nuprl5-p leaves (cdr leaves))))
	 (bound-terms nil))
    (mapcar #'(lambda (x)
		(setf bound-terms (cons (term-to-mbterm (term-of-bound-term x))
					bound-terms))
		(when (bindings-of-bound-term x)
		      (setf bound-terms
			    (cons (mbnode MBS_Bindings
					  (mapcar #'binding-to-mbnode
						  (bindings-of-bound-term x)))
				   bound-terms))))
	    (bound-terms-of-term term))
    (mbnode MBS_Term (append params bound-terms)))) 

(defvar *extended-node* (mb-string "extended" MBS_Token))

;; given a parameter-list, returns true if it represents an embedding of a special nuprl5
;; parameter into one understandable by metaprl
(defun embedded-parameter-list-p (value)
  (or (and (= (length value) 2)
	   (let* ((first (car value))
		  (first-value (value-of-parameter first))
		  (second-typeid (type-id-of-parameter (cadr value))))
	     (and (token-typeid-p (type-id-of-parameter first))
		  (or (and (eql first-value '|quote|) (token-typeid-p second-typeid))
		      (and (eql first-value '|bool|) (natural-typeid-p second-typeid))
		      ))))
      (and (= (length value) 3)
	   (equal (car value) '|time|)
	   (natural-typeid-p (type-id-of-parameter (cadr value)))
	   (natural-typeid-p (type-id-of-parameter (caddr value))))))

(defun binding-to-mbnode (binding)
  (let ((value (value-of-parameter-value binding)))
    (cond
     ((extended-parameter-value-p value)
      (cond
       ((slot-parameter-value-p value)
	(let ((s (descriptor-of-slot-parameter-value value)))
	  (if (meta-variable-id-p s)
	      (mbnode MBS_ParmList (list *extended-node*
					 (mb-string (string (type-of-meta-variable-id s)) MBS_Token)
					 (mb-string (string s) MBS_Token)))
	    (mbnode MBS_ParmList (list *extended-node*
				       (mb-string (string s) MBS_Token))))))
       ((error-parameter-value-p value)
	(break "binding is error")
	(raise-error (error-message '(binding to mbnode))))))
    
     ((real-parameter-value-p value *variable-type*)
      (mb-string (variable-id-to-string value) MBS_Variable))
    
     ((meta-parameter-value-p value)
      (let ((node (mb-string (variable-id-to-string value) MBS_MVariable)))
	(cond ((abstraction-meta-variable-id-p value)
	       node)
	      ((display-meta-variable-id-p value)
	       (mbnode MBS_ParmList (list (mb-string "display" MBS_Token)
					  node)))))))))


;; LAL TODO:for ocaml cons normal bit to subterm list in each case
(defun level-expression-to-mbnode (expr)
  (cond
   ((symbolp expr)
    (when (display-meta-variable-id-p expr)
	  (raise-error (error-message'(level expression to mbnode display) expr)))
    (mbnode MBS_Level (list (mb-string (string expr)))))
    
   ((integerp expr)
    (unless (> expr 0)
	    (raise-error (error-message'(level expression to mbnode 0) expr)))
    (mbnode MBS_Level (list (mb-integer expr))))
    	    
   ((and (level-max-p expr)
	 (null (last expr 0)))
    (mbnode MBS_Level (mapcar #'level-expression-to-mbnode expr)))
   ((and (consp expr)
	 (integerp (cdr expr))
	 (>= (cdr expr) 0))
    (mbnode MBS_Level (list (level-expression-to-mbnode (car expr))
			    (mb-integer (cdr expr)))))
   (t (raise-error (error-message'(level expression to mbnode) expr)))))
   

(defun level-expression-to-normalized-mbnode (expr)
  (let* ((constant (car expr))
	 (rest (cdr expr)))
    (mbnode MBS_Level
	    (cons (mb-integer (1- constant))
		  (flatten
		   (mapcar #'(lambda (x)
			       (cons (mb-string (level-variable-to-string (car x)))
				     (mb-integer (cdr x)))) 
			   rest))))))


(defun parameter-to-mbparameter (parameter &optional quote-p nuprl5-p)
  (let* ((type (type-of-parameter parameter))
	 (typeid (type-id-of-type type))
	 (value (value-of-parameter parameter)))
    (cond
      ((extended-parameter-value-p value)
       (let ((type-string (type-to-short-string type)))
	 (cond
	   ((slot-parameter-value-p value)
	    (let* ((s (descriptor-of-slot-parameter-value value))
		   (meta-type (cond ((abstraction-meta-variable-id-p s)
				     'abstraction)
				    ((display-meta-variable-id-p s)
				     'display))))
	      (if meta-type;;(meta-variable-id-p s)
		  (mbnode MBS_ParmList (list *extended-node*
					     (mb-string (string meta-type;;(type-of-meta-variable-id s)
								) MBS_Token)
					     (mb-string type-string MBS_Token)
					     (mb-string  (variable-id-to-string s) MBS_Token)))
		  (mbnode MBS_ParmList (list *extended-node*
					     (mb-string "slot" MBS_Token)
					     (mb-string type-string MBS_Token))))))
	   ((error-parameter-value-p value)
	    (let ((error-string (car (rest-of-extended-parameter-value value))))
	      (if (stringp error-string)
		  (mbnode MBS_ParmList (list *extended-node*
					     (mb-string "error" MBS_Token)
					     (mb-string type-string MBS_Token)
					     (mb-string error-string MBS_Token)))
		  (mbnode MBS_ParmList (list *extended-node*
					     (mb-string "error" MBS_Token)
					     (mb-string type-string MBS_Token)))))))))
      ((meta-parameter-value-p value)
       (let ((node 
	      (cond
		((natural-typeid-p typeid)
		 (mb-string (variable-id-to-string value) MBS_MLongInteger))
		((variable-typeid-p typeid)
		 (mb-string (variable-id-to-string value) MBS_MVariable))
		((string-typeid-p typeid)
		 (mb-string (variable-id-to-string value) MBS_MString))
		((token-typeid-p typeid)
		 (mb-string (variable-id-to-string value) MBS_MToken))
		((meta-level-expression-typeid-p typeid)
		 (mb-string (variable-id-to-string value) MBS_MLevel))
		((level-expression-typeid-p typeid) ;; nv5 level should not be meta
		 (mb-string (variable-id-to-string value) MBS_MLevel))
		((bool-typeid-p typeid)
		 (mbnode MBS_ParmList (list (mb-string "bool" MBS_Token)
					    (mb-string (variable-id-to-string value)))))
		(t (break) (mbnode MBS_ParmList (list (mb-string "nuprl_parameter_type" MBS_Token)
						      (mb-string (parameter-to-string parameter))))))))
	 (cond ((abstraction-meta-variable-id-p value)
		node)
	       ((display-meta-variable-id-p value)
		(mbnode MBS_ParmList (list (mb-string "display" MBS_Token)
					   node))))))
      ((real-parameter-value-p value type)
       (cond
	 ((natural-typeid-p typeid) (mb-integer value t))
	 ((variable-typeid-p typeid) (mb-string (real-parameter-value-to-string value type)
						MBS_Variable))
	 ((string-typeid-p typeid) (mb-string (real-parameter-value-to-string value type)))
	 ((token-typeid-p typeid) (mb-string (real-parameter-value-to-string value type)
					     MBS_Token))
	 ((oid-typeid-p typeid)
	  (let ((stamp (stamp-of-oid value)))
	    (mbnode MBS_ObjectId
		    (list (mb-integer (sequence-of-stamp stamp) t)
			  (mbnode MBS_ParmList
				  ;;(list (mb-string "time" MBS_Token)
				  ;;  (mb-integer (ash (time-of-stamp stamp) -16) t)
				  ;;  (mb-integer (logand (time-of-stamp stamp) #xffff) t))
				  (list (mb-string "time" MBS_Token)
					(mb-integer (time-of-stamp stamp) nil)))
			  (mb-integer (transaction-of-stamp stamp) t)
			  (mb-string (string (process-id-of-stamp stamp)) MBS_Token)
			
				    

			  ;;  (mb-string (string-of-oid value) MBS_String)
			  ))))

	 ((time-typeid-p typeid)  ;;LAL need Jason to fix big nums in ocaml
	  (mbnode MBS_ParmList (list (mb-string "time" MBS_Token)
				     ;;(mb-integer (ash value -16) t) 
				     ;;(mb-integer (logand value #xffff) t)))
				     (mb-integer value nil))))
	 ((bool-typeid-p typeid)
	  (mbnode MBS_ParmList (list (mb-string "bool" MBS_Token)
				     (mb-integer (if value 1 0)))))
	 ((quote-typeid-p typeid)
	  (mbnode MBS_ParmList (list (mb-string "quote" MBS_Token)
				     (mb-string (real-parameter-value-to-string value type)
						MBS_Token))))
	 ((parameter-list-typeid-p typeid)
	  (if (null value)
	      (mbnode MBS_ParmList '())
	      (if (and nuprl5-p
		       (not quote-p)
		       (embedded-parameter-list-p value))
		  (mbnode MBS_ParmList
			  (list (mb-string "quote" MBS_Token)
				(mbnode MBS_ParmList
					(mapcar #'(lambda (x) (parameter-to-mbparameter x t t))
						value))))
		  (mbnode MBS_ParmList (mapcar #'parameter-to-mbparameter value)))))

	 ((level-expression-typeid-p typeid) 
	  (let* ((nexpr (normalize-level-expression value))
		 (node (level-expression-to-normalized-mbnode nexpr))
		 (level-string (level-expression-to-string value)))
	    (if nil;;(string= level-string 
		   ;;(level-expression-to-string (sort-level-expression nexpr)));;already normal
		node
		(mbnode MBS_ParmList (list (mb-string "nuprl5_level_expression" MBS_Token)
					   node
					   (mb-string level-string))))))

	 ;; representing mp levels as strings in nup fttb, conversion may not be reliable
	 ((meta-level-expression-typeid-p typeid) 
	  (let ((nexpr (normalize-level-expression (string-to-level-expression value)))) ;; should be normal already
	    (level-expression-to-normalized-mbnode nexpr)))
		 
	 (t (break) (mbnode MBS_ParmList (list (mb-string "abstract_parameter_type" MBS_Token)
					       (mb-string (parameter-to-string parameter)))))))
	
      (t (error "parameter-to-mbparameter type?")))))

(defun sort-level-expression (expr)
  (labels ((insert (le l)
		   (let ((e1 (string (expression-of-level-increment le))))
		     (do* ((y l (cdr y))
			   (le2 (car y) (car y))
			   (e2 (string (expression-of-level-increment le2))
			       (string (expression-of-level-increment le2)))
			   (z (if (string> e1 e2)
				  (prog1 (cons le y) (setf y nil))
				(list le2))
			      (if (string> e1 e2)
				  (prog1 (append (reverse z) (cons le y)) (setf y nil))
				(cons le2 z))))
			  ((null y) z))))
	   
	   (vsort (l1 l2)
		  (if (null l1)
		      l2
		    (vsort (cdr l1) (insert (car l1) l2)))))
       					  
		       
	  (let ((constant (car expr))
		(increments (cdr expr)))
	    (if increments
		(vsort (cdr increments) (list (car increments)))
	      expr))))

(defun test-server-mathbus (port &optional block-p)

  (let* ((data nil)
	 (in-sock (new-socket port nil t 5 block-p))
	 (fd (do ((f (accept-new-client in-sock) (accept-new-client in-sock)))
		 (f f) (sleep 1))))
    (format t "accept ~s ~%" fd)
	  
    (let* ((in-socket (make-socket :port (port-of-socket in-sock)
				  :stream-fd fd))
						    
			
	  (link (new-mathbus-link (list (cons in-socket nil)) #'write-node #'read-node)))
      (link-open link)
      ;; (loop)
      (setf data (link-recv link t))
      (format t "received ~s ~%" (string-value data))
      ;;  (when (= data 98) (return))
      (link-close link)
      (destroy-socket in-sock)
    
      )))

(defun  test-server-mathbus2 (port)

  (let* ((in-sock (new-socket-listen port 5))
	 (fd (do ((f (accept-new-client in-sock) (accept-new-client in-sock)))
		 (f f) (sleep 1))))
    
    (format t "accept ~s ~%" fd)
	  
    (let* ((in-socket (make-socket :port (port-of-socket in-sock)
				   :stream-fd fd))
						    
			
	   (link (new-mathbus-link (list (cons in-socket nil)) #'write-node #'read-node)))
      (link-open link)
      ;; (loop)
      (link-send link  ;;  (mb-string "Lori")
			 (mb-integer 57))
      (force-output (out-stream-of-stream-channel (car (channels-of-link link))))

      (destroy-socket in-sock)
      link   
      )))
#|
(defun  test-server-mathbus3 (port term)

  (let* ((in-sock (new-socket-listen port 5))
	 (fd (do ((f (accept-new-client in-sock) (accept-new-client in-sock)))
		 (f f) (sleep 1))))
    (format t "accept ~s ~%" fd)
	  
    (let* ((in-socket (make-socket :port (port-of-socket in-sock)
				   :stream-fd fd))
						    
			
	   (link (new-mathbus-link (list (cons in-socket nil)) #'write-node #'read-node)))
      (link-open link)
      ;; (loop)
      (link-send link (term-to-mbterm term) )
      (force-output (out-stream-of-stream-channel (car (channels-of-link link))))

      (destroy-socket in-sock)
      link   
      )))

(defun itest-term ( bool time string)
  (instantiate-term
   (instantiate-operator '|!test|
			 (list  (instantiate-parameter-r bool *bool-type*)
			       (instantiate-parameter-r (intern string) *quote-type*)
			         (instantiate-parameter-r time *time-type*)))
   nil))
(defun itest-quote-term (string val)
  (instantiate-term
   (instantiate-operator '|!test|
			 (list (instantiate-parameter-r (list (instantiate-parameter-r (intern string) *token-type*) (instantiate-parameter-r (intern val) *token-type*)) *parameter-list-type*)))
   nil))

(defun itest-bindings-term ()
  (instantiate-term
   (instantiate-operator '|!test|
			 (list (instantiate-parameter-r 7 *natural-type*)))
   (list (instantiate-bound-term (iack-term)
				 (list (maybe-string-to-parameter-value "hello"									*variable-type*)(maybe-string-to-parameter-value "hi"
									*variable-type*)))
	 (instantiate-bound-term (itest-quote-term "quote" "also")
				 (list (maybe-string-to-parameter-value " blue"
									*variable-type*)))
	  (instantiate-bound-term (itest-quote-term "quote" "a")
				 (list (maybe-string-to-parameter-value " be"
									*variable-type*)
				       (maybe-string-to-parameter-value " flower"
									*variable-type*)
									(maybe-string-to-parameter-value " purple"
									*variable-type*))))))
 
 ;;(allocate-term (list (instantiate-parameter-r '|!test| *token-type*) (instantiate-parameter-r  7 *natural-type*)) (list (iack-term)) (list (list (maybe-string-to-parameter-value "hello"
	 ;;								*variable-type*))))

;;(test-server-mathbus3 2284)
;;(link-send link (term-to-mbterm term))

;;(defunml (|test-server-mathbus3| (port termlist))
;;   (test-server-mathbus3 port term))

;;(defunml (|link-close| (link))
;;  (link-close link))

;; benchmark test
;; send buf-size bytes num times
(defun  test-client-mathbus (num buf-size port)
 (declare (ignore num buf-size))
  (let* ((data nil)
	 (out-sock (new-socket-primary-connect port "alfheim"))
	 (link (new-mathbus-link (list (cons nil out-sock)) #'write-node #'read-node)))
    (link-open link) 
      
    (setf data (link-recv link t))
    (format t "received ~s ~%" (mbterm-to-term data))
    (link-close link)
    
    ;;nil
    ))


(defun  test-client (num buf-size port)
  (let* ((d nil)
	 (out-sock (new-socket-primary-connect port "alfheim"))
	 (link (new-stream-link (list (cons nil out-sock)) #'write-byte #'read-byte)))
    (link-open link) 
      
    (setf d (char-code #\d))
    (do ((i 1 (+ i 1)))
	((> i num))
	(do ((p 1 (+ p 1)))
	    ((> p buf-size))
	    (link-send link d))
	(force-output (out-stream-of-stream-channel (car (channels-of-link link)))))

    (link-send link (char-code #\b))
    (force-output (out-stream-of-stream-channel (car (channels-of-link link))))

    ;;(link-close link)
    link 
    ;;nil
    ))
|#

(defun write-node-to-file (node filename)
  (with-open-file (stream filename
		  :direction :output)
		  
		  (write-node node stream))
  nil)
(defun read-node-from-file (filename)
  (with-open-file (stream filename)
		  
		  (read-node stream))
  )


;;(defunml (|assign_term| (term))
;;    (term -> term)
;;  (setf myterm term))
