
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************

#+cltl2(in-system-package)
#-cltl2(in-package *system-package-name*)

;;;;  -page-

;;;; TODO RLE inline trm stuff.
;;;;
;;;;
;;;; -docs- (mod trm data)
;;;;
;;;;	Term :
;;;;
;;;;	 Terms are the predominate data structure. The structure
;;;;	 editor is tailored to display and edit terms.  All source code and
;;;;	 definitions are terms. Proofs are can be represented and
;;;;	 are stored as terms. The ML evaluator accepts terms as source. IO to
;;;;	 files is terms.
;;;;	
;;;; -doct- (mod trm)
;;;;	
;;;;	 Terms can be represented three primary ways:
;;;;	    1) as an abstract data type.
;;;;	    2) as an s-expr.
;;;;	    3) as an ascii string.
;;;;
;;;;
;;;;	<term>			: #term[<marks> <operator> <bound-term> list]
;;;;	<operator>		: (<id> . <parameter> list)
;;;;	<bound-term>		: (<binding> list . <term>) list
;;;;	<parameter>		: #parameter[<parameter-value> . <parameter-type>]
;;;;	<parameter-value>	: <real-value>
;;;;				| <meta-parameter-value>
;;;;				| <extended-value>
;;;;	<binding>		: <variable-id>
;;;;				| <meta-variable-id>
;;;;				| <extended-value>
;;;;	 ** Note <binding> is a subtype of <parameter-value>
;;;;	<extended-value>	: #parameter-slot[<marks> <string>]
;;;;				| #parameter-error[<marks> <string> <message>]
;;;;				| #parameter-marks[<marks> <real-value>]
;;;;				| #parameter-marks[<marks> <meta-parameter-value>]
;;;;	<descriptor>		: <string>
;;;;	<parameter-type>	: #parameter-type[<type-id>
;;;;						  <closure> <closure> <closure>
;;;;						  <closure> <closure> <closure>]
;;;;	<meta-parameter-value>	: <meta-variable-id>
;;;;	<meta-variable-id>	: <disp-meta-variable-id>
;;;;				| <abs-meta-variable-id>
;;;;	<disp-meta-variable-id>	: #display-meta-variable[<id>]
;;;;	<abs-meta-variable-id>	: #abstraction-meta-variable[<id>]
;;;;
;;;;	<variable-id>		: #var[<id>]
;;;;	<type-id>		: token | variable | natural | string | level-expression
;;;;				| ifname | opname | time | object-id
;;;;	 ** the list of recognized types is extendable. 
;;;;
;;;;
;;;;	A notational Syntax for terms:
;;;;
;;;;	<note-term>		: <id>{<note-parameters>}(<note-bound-terms>)
;;;;				| <id>(<note-bound-terms>)
;;;;				| <id>{<note-parameters>}
;;;;				| <id>()
;;;;	<note-parameters>	: <note-parm> | <note-parm>, <note-parameters>
;;;;	<note-bound-terms>	: <note-bound-term> | <note-bound-term>, <note-bound-terms>
;;;;	<note-parameter>	: <text>:<text>
;;;;	<note-bound-term>	: <note-bindings>.<note-term>
;;;;				| <note-term>
;;;;	<note-bindings>		| <text> | <text>, <note-bindings>
;;;;	
;;;;
;;;;	This notational syntax will be used interchangebly with the abstract
;;;;	 syntax described above.
;;;;
;;;;
;;;;  -page-
;;;;
;;;;	Term as S-EXPR :
;;;;
;;;;	<term-sexpr> 		: ((<string> . <parm-sexpr> list) . <bound-term-sexpr> list)
;;;;	<parm-sexpr>		: ((<parm-value-sexpr> . <string>)
;;;;	<bound-term-sexpr>	: ((<parm-value-sexpr> list . <term-sexpr>)
;;;;	<parm-value-sexpr>	: <string>
;;;;				| (<slot-char> . nil)
;;;;				| (<meta-char> . (<slot-char> . nil))
;;;;				| (<meta-char> . <string>)
;;;;	<meta-char>		: a | d
;;;;	<slot-char>		: s
;;;;
;;;;
;;;;	Access/Projections
;;;;
;;;;
;;;;	Variables:
;;;;
;;;;	variable-id-p (<*>)			: <bool>
;;;;	dummy-variable-id-p (<*>)		: <bool>
;;;;	dummy-display-meta-variable-id-p (<*>)	: <bool>
;;;;	display-meta-variable-id-p (<*>)	: <bool>
;;;;	abstraction-meta-variable-id-p (<*>)	: <bool>
;;;;	meta-variable-id-p (<*>)		: <bool>
;;;;
;;;;	
;;;;	Operator:
;;;;
;;;;	id-of-operator (<term>)			: <id>
;;;;	parameters-of-operator (<term>)		: <parameter> list.
;;;;	equal-operators-p (<op> <op>)		: <bool>
;;;;	operator-to-pretty-string (<op>)	: <string>
;;;;
;;;;
;;;;  -page-
;;;;
;;;;	Parameters: 
;;;;
;;;;	Parameters manifest themselves as true parameters in operators as well as
;;;;	parameter values with implicit type "variable" in bindings.
;;;;
;;;;	The types of parameter values may be classed as follows:
;;;;	  ** edit-type : `SLOT` `ERROR` or `NIL`.
;;;;	  ** meta-type : `ABSTRACTION` `DISPLAY` or `NIL`
;;;;
;;;;	 If a parameter value has null edit type, then it must
;;;;	 be a meta-parameter value and/or a real-value. It is possible for a value be
;;;;	 both a real value and a meta value. Currently, this is only possible with
;;;;	 level expressions. Also, it is possible for a slot to have a non-null meta type.
;;;;
;;;;	For some purposes it is convienient to be able to mark a parameter. Marks
;;;;	are expected to be quite rare on parameters. The implementation of the
;;;;	parameters has made a trade-off	between abstractness and efficiency. 
;;;;
;;;;	Most of the time, we would like to hide the marks. Two alternatives: marks are
;;;;	omni-present and functions manipulating parameter values are cognizant of the marks;
;;;;	marks are stripped and functions manipulating parameter values are not cognizant of
;;;;	the marks. Both these alternatives are supported, with methods for moving between them.
;;;;
;;;;	Of course the most attractive option would have been to have marks omni-
;;;;	present but insulated functions. This is not feasable as we would
;;;;	be obligated to support another layer of indirection. Ie a parameter value
;;;;	would not be an actual value but instead a structure containing possible marks
;;;;	and the parameter value. Note that the parameter structure itself may not contain
;;;;	the marks as the bindings have no parameter structure.
;;;;
;;;;	It is possible to write code by using a subset of the following functions
;;;;	which need not be aware of the existence of marks. Note that such code is
;;;;	unlikely to preserve marks.
;;;;
;;;;	Then we can divide the parameter related functions into four classes:
;;;;	  - those that ignore but preserve the marks.
;;;;	  - those that strip marks from parameters.
;;;;	  - those that require marks be stripped from parameter values.
;;;;	  - those that access or add marks.
;;;;	 
;;;;  -page-
;;;;
;;;;	Following ignore but preserve marks:
;;;;
;;;;	parameter-p (<*>)				: <bool>
;;;;	type-of-parameter (<parameter>)			: <parameter-type>
;;;;	type-id-of-parameter (<parameter>)		: <id>
;;;;	type-id-of-type (<parm-type>)			: <id>
;;;;	type-to-short-string (<parm-type>)		: <string>
;;;;	equal-types-p (<parm-type> <parm-type>)		: <bool>
;;;;
;;;;	parameter-to-sexpr (<parameter>)		: <parm-value-sexpr>
;;;;	parameter-to-pretty-string (<parm> &optional <bool>)	: <string>
;;;;	 ** if bool is t then result will include ":<type-id>"
;;;;	equal-parameters-p (<parm> <parm>)		: <bool>.
;;;;	hash-parameter(<parm>)				: INTEGER
;;;;
;;;;	meta-parameter-p (<parm>)			: <bool>
;;;;	abstraction-meta-parameter-p (<parm>)		: <bool>
;;;;	display-meta-parameter-p (<parm>)		: <bool>
;;;;
;;;;
;;;;	Following strip marks when parameter values accessed:
;;;;
;;;;	value-of-parameter-r (<parameter>)		: <parameter-value>
;;;;	 ** fails if not a real-valued parameter. 
;;;;	value-of-parameter-m (<parameter>)		: <parameter-value>
;;;;	 ** fails if not a real or meta valued parameter.
;;;;	value-of-parameter (<parameter>)		: <parameter-value>
;;;;	 ** always succeeds. 
;;;;	value-of-parameter-value (<parm-value>)		: <parameter-value>
;;;;
;;;;
;;;;	Following preserves marks when parameter values accessed:
;;;;
;;;;	value-of-parameter-n (<parameter>)		: <parameter-value>
;;;;	 ** always succeeds. Leaves marks.
;;;;	
;;;;	The following functions require that marks have been stripped from parameter values.
;;;;
;;;;	meta-type-of-parameter-value(<parm-value>)	: SYMBOL
;;;;	 ** result is one of 'display, 'abstaction, or nil.
;;;;	edit-type-of-parameter-value(<parm-value>)	: SYMBOL
;;;;	 ** one of slot, error, or nil.
;;;;	 ** note that is is possible for a slot to have a non-null meta type.
;;;;
;;;;	edit-parameter-value-p (<parameter-value>)	: <bool>
;;;;	slot-parameter-value-p (<parameter-value>)	: <bool>
;;;;	error-parameter-value-p (<parameter-value>)	: <bool>
;;;;	meta-parameter-value-p (<parameter-value>)	: <bool>
;;;;	 ** returns nil if parameter value is a slot.
;;;;	real-parameter-value-p (<parm-value> <parm-type>)	: <bool>
;;;;
;;;;	descriptor-of-slot-parameter-value (<parm-value>)
;;;;	 : <string> | <meta-variable-id>
;;;;	sexpr-of-error-parameter-value (<parm-value>)	: <string>
;;;;	message-of-error-parameter-value (<parm-value>)	: <string>
;;;;
;;;;	equal-real-parameter-values (<parm-value> <parm-value> <parm-type>)
;;;;	 : <bool>
;;;;	real-parameter-value-to-string (<parm-value> <parm-type>)	: <string>
;;;;	hash-real-parameter-value (<parm-value> <parm-type>)		: INTEGER
;;;;
;;;;
;;;;  -page-
;;;;
;;;;	The following functions preserve but ignore marks.
;;;;
;;;;	parameter-value-p (value type)
;;;;	equal-parameter-values-p (<parm-value> <parm-value> <parm-type>)
;;;;	 : <bool>
;;;;	 ** all slots are considered to be equal (ie descriptors are not compared).
;;;;	 ** in error values the strings are compared but not the mesages.
;;;;	parameter-value-to-pretty-string (<parm-value> <parm-type>)	: <string>
;;;;	parameter-value-to-sexpr (<parm-value> <parm-type>)		: <parm-value-sexpr>
;;;;
;;;;
;;;;    The following functions access or add marks.
;;;;
;;;;	marks-of-parameter-value (<parameter-value>)	: <prop> list
;;;;
;;;;	extended-parameter-value-p (<parameter-value>)	: <bool>.
;;;;	 ** the extended parameter values currently consist of the non null edit-types,
;;;;	 ** marked real, and marked meta values.
;;;;	 ** If a parameter value is an extended parameter value than it is derived
;;;;	 ** from marks and the usual mark functions are applicable.
;;;;	 ** The mark edit type is used to make real and meta values markable.
;;;;
;;;;	type-of-extended-parameter-value (<parm-value>)	: SYMBOL
;;;;	 ** 'one of slot, 'error, 'mark, or nil.
;;;;	mark-parameter-value (<parameter-value>)	: <extended-value>
;;;;	mark-parameter-value-p (<parameter-value>)	: <bool>
;;;;	value-of-mark-parameter-value (<parm-value>)	: <parameter-value>
;;;;	 ** value will be either a meta-value or a real-value.
;;;;	 ** arg must be an extended parameter value with type 'mark.
;;;;
;;;;	All other functions related to parameters will preserve but ignore marks.
;;;;
;;;;	id-of-term (<term>)				: <id>
;;;;	parameters-of-term (<term>)			: <parameter> list
;;;;	operator-of-term (<term>)			: <operator>
;;;;	arities-of-term (<term>)			: INTEGER list
;;;;	 ** conses up a new list each call so use sparingly.
;;;;
;;;;	bound-terms-of-term (<term>)			: <bound-term> list
;;;;	term-of-bound-term (<bound-term>)		: <term>
;;;;
;;;;	Following strip marks:
;;;;
;;;;	bindings-of-bound-term-r (<term>)		: <variable-id> list
;;;;	 ** fails if all bindings are not real valued.
;;;;	bindings-of-bound-term-m (<term>)
;;;;	 : (<variable-id> | <meta-variable-id>) list
;;;;	 ** fails if all bindings are not real valued or meta valued.
;;;;	bindings-of-bound-term (<term>)			: <parameter-value> list
;;;;
;;;;
;;;;	Following preserves but ignores marks.
;;;;
;;;;	bindings-of-bound-term-n (<term>)		: <parameter-value> list
;;;;
;;;;	term-p (<*>)					: <bool>
;;;;	bound-term-p (<*>)				: <bool>
;;;;
;;;;
;;;;  -page-
;;;;
;;;;
;;;;	Instantiation:
;;;;
;;;;
;;;;	get-variable-id (<string>)			: <variable-id>
;;;;	get-dummy-variable-id ()			: <variable-id>
;;;;	get-display-meta-variable-id (<string>)		: <disp-meta-variable-id>
;;;;	get-abstraction-meta-variable-id (<string>)	: <abs-meta-variable-id>
;;;;	variable-id-to-string (<variable-id>)		: <string>
;;;;	 ** v can be a meta or real variable id. Result has no syntactic text, ie $ or <>.
;;;;	variable-id-to-pretty-string <variable-id>)	: <string>
;;;;	 ** v can be a meta or real variable id. Result has syntactic text, ie $ or <>.
;;;;
;;;;	In order to be able to treat the different types of parameters abstractly, the
;;;;	parameter types must be defined before they can be used:
;;;;
;;;;	define-typeid ((<id> <id{aliases}> list <id{global}>)
;;;;			<closure{verify-f}>
;;;;			<closure{eq-f}>
;;;;			<closure{unparse-f}>
;;;;			<closure{parse-f}>
;;;;			<closure{hash-f}>
;;;;			&optional <closure{walk-f}>)
;;;;	 : NULL
;;;;	 ** verify-f (<parm-value>)		: <bool>
;;;;	 ** eq-f (<parm-value> <parm-value>)	: <bool>	
;;;;	 ** unpars-f (<parm-value>)		: <string>
;;;;	 ** parse-f (<string>)			: <parm-value>
;;;;	     * result will be verified. Ie, parse may return bad values.
;;;;	 ** hash-f (<parm-value>)		: INTEGER
;;;;	 ** walk-f (<parm-value> <closure{f}>)	: <parm-value>
;;;;	     * f (<meta-parm-value>)		: <parm-value>
;;;;	     * Intent is to allow substitutions in expressions with complex parameter values.
;;;;	 ** eventually may need match func
;;;;	 **  : match-f (<meta-parm-value> <parm-value> <closure{meta-p-f}> <closure{env-f}>)
;;;;	 ** global is name used to define global variable containing id. This allows for
;;;;		dependent code to refer to the type id indirectly. Ie, you can change the
;;;;		id without changing the code if global remains the same. The global name
;;;;		will be defined within asterisks, eg. *global*).
;;;;
;;;	Defines : 
;;;;	  - *<global>*				: <tok>
;;;;	     * global containing canonical id.
;;;;	  - *<ID>-TYPE*				: <parameter-type>
;;;;	     * global containing type structure.
;;;;	  - <ID>-TYPE-P (<parameter-type>)	: BOOL
;;;;	  - <ID>-TYPEID-P (<tok>)		: BOOL
;;;;	  - <ID>-PARAMETER-P (<parameter>)	: BOOL
;;;;
;;;;  -page-
;;;;
;;;;	type-id-to-type (<id>)				 : <parameter-type>
;;;;
;;;;	string-to-real-parameter-valuea (<string> <parm-type>)	: <parameter-value>
;;;;	maybe-string-to-parameter-value (<string> <parm-type>)	: <parameter-value>
;;;;	 ** never fails.
;;;;	sexpr-to-parameter-value (<parm-sexpr> <parm-type>)	: <parameter-value>
;;;;	coerce-parameter-value (<parm-type> <parm-type> <parm-value>)
;;;;	 : <parameter-value>
;;;;	 ** first type is source type and second type is destination type.
;;;;
;;;;	meta-parameter-value (<string> <meta-type-id>)		: <parameter-value>
;;;;	 ** <meta-type-id>	: 'DISPLAY | 'ABSTRACTION
;;;;	slot-parameter-value (&optional <string>|<meta-variable-id>)
;;;;	 : <parameter-value>
;;;;
;;;;	walk-real-parameter-value (<parm-value> <parm-type> <closure>)
;;;;	 : <parameter-value>
;;;;	walk-parameter-value (parm-value> <parm-type> <closure>)
;;;;	 : <parameter-value>
;;;;
;;;;	instantiate-parameter-s (<string> <parm-type>)		: <parameter>
;;;;	instantiate-parameter-m (<string>|<meta-variable-id> <parm-type>)
;;;;	 : <parameter>
;;;;	instantiate-parameter-r (<*> <parm-type>)		: <parameter>
;;;;	 ** checks value to make sure valid parameter value.
;;;;	instantiate-parameter (<parm-value> <parm-type>)	: <parameter>
;;;;
;;;;	string-to-parameter (<string> <id>)		: <parameter>
;;;;	sexpr-to-parameter (<parm-sexpr>)		: <parameter>
;;;;	walk-parameter (<parameter> <closure{f}>)	: <parameter>
;;;;	 ** f (<meta-variable-id>)	: <parameter-value>
;;;;	substitute-parameter (<parameter> <sub> list)
;;;;	 : parameter
;;;;	 ** <sub>	: (<meta-variable-id> . <parm-value>)
;;;;
;;;;	instantiate-operator (<id> <parameter> list)		: <operator>
;;;;	maybe-instantiate-operator (<op> <id> <parameter> list)	: <operator>
;;;;	 ** if result would be lisp:equal to <op> arg then returns arg.
;;;;	 ** to op, op is returned.
;;;;
;;;;	nreplace-operator (<term> <op>)			: <term>
;;;;	 ** destructively modifies term. Use with extreme caution.
;;;;	 ** ***VERY IMPORTANT*** : Should only be used within dynamic extent
;;;;	 ** 			   of creation of term. 
;;;;
;;;;	with-allow-dummy-variable-operator (&body body)	: macro
;;;;
;;;;	substitute-in-operator (<op> <sub> list) 	: <operator>
;;;;	 ** <sub>	: (<variable-id> . <parm-value>)
;;;;	 ** <value>	: <string> | <parameter> | <meta-variable-id>
;;;;
;;;;	substitute-in-bindings (<binding> list <sub> list)	: <binding-value> list
;;;;	 ** <sub>	: (<variable-id> . <value>)
;;;;	 ** <value>	: <string> | <binding>
;;;;
;;;;
;;;;	instantiate-term (<op> &optional <bound-term> list)	: <term>
;;;;	maybe-instantiate-term (<term> <op> <bound-term> list)	: <term>
;;;;	 ** if result would be lisp:equal to <term> arg, then returns arg.
;;;;
;;;;	instantiate-bound-term (<term> &optional <binding> list)		: <bound-term>
;;;;	maybe-instantiate-bound-term (<bound-term> <binding> list <term>)	: <bound-term>
;;;;	 ** if result would be lisp:equal to <bound-term> arg, then returns arg.
;;;;
;;;;  -page-
;;;;
;;;;	OpQuotes:
;;;;
;;;;	<quoted-term>	: <id>{<q>:quote, <parm list>}(<bound-term> list)
;;;;
;;;;	opquote-term (<term> <id>) 				: <quoted-term>
;;;;
;;;;	<conditions>	: <id> list | <id>
;;;;
;;;;	opquoted-term-p (<term> <conditions)			: <bool>
;;;;	 ** t when <term> is a <quoted-term> and (member <q> <conditions>) is t.
;;;;	un-opquote-term (<term> <conditions>)			: <term>
;;;;	un-opquote-term-r (<opquoted-term> <conditions>)	: <term>
;;;;	 ** fails when first quote, if quoted, is not one of the conditions.
;;;;
;;;;
;;;;
;;;;	Variables as terms:
;;;;
;;;;	<variable-term>		: variable{<binding>:variable}(<note-bound-terms>)
;;;;
;;;;	variable-term-p (<term>)			: <bool>
;;;;	 ** t if <term> is <variable-term>
;;;;	variable-p (<term>)				: <bool>
;;;;	 ** <first-order-variable-term>	: variable{<binding>:variable}()
;;;;	 ** t if <term> is <first-order-variable-term>
;;;;
;;;;	id-of-variable-term (<variable-term>)		: <binding> 
;;;;
;;;;	nreplace-binding (<variable-id> <bound-term> INTEGER)	: <binding> list
;;;;	 ** replaces binding at position specified by INTEGER.
;;;;	 ** destructively modifies bound-term. Use with extreme caution.
;;;;	 ** ***VERY IMPORTANT*** : Should only be used within dynamic extent
;;;;	 ** 			   of creation of bound-term. 
;;;;	nreplace-variable-term-id (<variable-term> <id>)	: <variable-term>
;;;;	 ** destructively modifies <variable-term>. Use with extreme caution.
;;;;	 ** ***VERY IMPORTANT*** : Should only be used within dynamic extent
;;;;	 ** 			   of creation of variable term. 
;;;;
;;;;	variable-term (<id> &optional <bound-term> list)	: <variable-term>
;;;;	modifiable-variable-term (<id>)				: <variable-term>
;;;;	 ** must not be used if is any possibility of nreplace-operator being
;;;;	 ** called on result.
;;;;	canonical-variable-term ()				: <variable-term>
;;;;	 ** variable-p (canonical-variable-term) == t.
;;;;
;;;;
;;;;	Term Sig:
;;;;
;;;;	<term-sig>		: (<symbol> . (<type-id> list . <arity> list))
;;;;	<arity>			: INTEGER
;;;;
;;;;
;;;;	id-of-term-sig (<term-sig>) 			: <id>
;;;;	parameters-of-term-sig (<term-sig>)		: <type-id> list
;;;;	arities-of-term-sig (<term-sig>)		: INTEGER list
;;;;	equal-term-sigs-p (<term-sig> <term-sig>)	: <bool>
;;;;
;;;;	term-sig-of-term (<term>)			: <term-sig>
;;;;	 ** term-sig is id, parameter type list and arities.
;;;;	term-sig-of-term-p (<term-sig> <term>)		: <bool>
;;;;
;;;;
;;;;  -page-
;;;;
;;;;
;;;;	Sexpr:
;;;;
;;;;	term-to-sexpr (<term>)				: <term-sexpr>
;;;;	sexpr-to-term (<term-sexpr>)			: <term>
;;;;
;;;;
;;;;	File IO: 
;;;;
;;;;	TODO ??? <file-spec> should be a logical pathname ala the portable utilities ???
;;;;
;;;;	term-reader (<file-spec> <closure{f}>)		: NULL
;;;;	 ** f (<term>)	: NULL
;;;;	term-writer (<file-spec> <closure{f}>)		: NULL
;;;;	 ** f ()	: <term> | NIL
;;;;
;;;;	
;;;;	Due to the vagaries of the character codes, term reading and writing
;;;;	might not work across platforms or lisp vendors. 
;;;;	TODO : This could be fixed by using the ascii or compressed ascii io.
;;;;
;;;;
;;;;  -page-
;;;;
;;;;
;;;;	Certain terms are manipulated explicitly in code. The following
;;;;	macro allows one to easily define the obvious functions.
;;;; 
;;;;	define-primitive ((<id{opid}>
;;;;			  &optional <parameter-spec> list <bound-term-spec> list)
;;;;	 : NULL
;;;;	 ** <parameter-spec> 	: (<type-id> . <name>)
;;;;	 ** <bound-term-spec> 	: <name> | (<arity> . <name>)
;;;;	 **  * default arity is 0.
;;;;	 ** 		 	
;;;;	 **  let <f-root> be upcased opid with underscores converted to hyphens
;;;;	 ** 	         and ! converted to I.
;;;;	 ** 		 	
;;;;	 **  Functions produced :
;;;;	 **  	<f-root>-term-p (<term>)	: <bool>
;;;;	 **	 * t when term-sig of term matches term-sig of canonical term.
;;;;	 **	canonical-<f-root>-term ()	: <term>
;;;;	 **	 * parameters are meta abstraction parameters with parameter spec name ids.
;;;;	 **	 * bindings are meta abstraction variable ids such as "$bi".
;;;;	 **	<f-root>-op ()			: <operator>
;;;;	 **	 * parameters are meta abstraction parameters with parameter spec name ids.
;;;;	 **	<f-root>-term (<instances>)	: <term>
;;;;	 **	 * <instances>		: <parameter-args> <bound-term-args>
;;;;	 **	 * <parameter-args>	: EPSILON 
;;;;	 **	 * 			| <parameter-arg>
;;;;	 **	 * 			| <parameter-arg> <parameter-args> 
;;;;	 **	 * <bound-term-args>	: EPSILON 
;;;;	 **	 * 			| <bound-term-arg>
;;;;	 **	 * 			| <bound-term-arg> <bound-term-args> 
;;;;	 **	 * <parameter-arg>	: <parameter-value>
;;;;	 **	 * <bound-term-arg>	: <binding> list <term-arg>
;;;;	 **	 * 			| <binding> <term-arg>
;;;;	 **	 * 			| <term-arg>
;;;;	 **	ForEach <parameter-spec>
;;;;	 **  	 <name>-of-<f-root>-term (<term>)		: <parameter-value>
;;;;	 **	ForEach <bound-term-spec>
;;;;	 **  	 <name>-of-<f-root>-term (<term>)		: <term>
;;;;	 **	 When <arity> = 1 :
;;;;	 **  	  binding-of-<name>-of-<f-root>-term (<term>)	: <binding>
;;;;	 **	 When <arity> > 1 :
;;;;	 **	  bindings-of-<name>-of-<f-root>-term (<term>)	: <binding> list
;;;;	 **
;;;;	 **  globals:
;;;;	 **	*<f-root>*			: <id>
;;;;	 **	*<f-root>-term-sig*		: <term-sig>
;;;;	 **	*<f-root>-operator*		: <operator>
;;;;	 **
;;;;	 **  When <f-root> ends in "CONS"
;;;;	 **	  and <parameter-spec> list = NIL
;;;;	 **	  and <arity> list = (0 0)
;;;;	 **    let <f-root-nil> be <f-root> - "CONS" + "NIL"
;;;;	 **	<f-root-nil>-term-p()		: <bool>
;;;;	 **	<f-root-nil>-term()		: <bool>
;;;;	 **	<f-root-nil>-op()		: <bool>
;;;;
;;;;
;;;; -doct- (mod trm data)
;;;;
;;;;	The currently recognized parameter types and aliases are:
;;;;	  - token 		: NATURAL, nat, NAT, n, N.
;;;;	  - variable 		: TOKEN, tok, TOK, t, T.
;;;;	  - natural 		: VARIABLE, var, VAR, v, V.
;;;;	  - string 		: STRING, s, S.
;;;;	  - level-expression	: LEVEL-EXPRESSION, l, L.
;;;;	  - quote		: QUOTE, q, Q.
;;;;	  - time		: TIME.
;;;;	  - ifname 		: IFNAME
;;;;	  - opname 		: OPNAME
;;;;	  - object-id		: OBJECT-ID, oid, OID, o, O.
;;;;
;;;;	 ** the list of recognized types is extendable. 
;;;;
;;;;  -page-
;;;;
;;;;	Meta/Extended parameter values.
;;;;
;;;;	 In some contexts extended parameter values and some meta values
;;;;	are nonsensical. For example, t[s/$a], ie substitute term s for
;;;;	abstraction meta-variable $a in term t. In other contexts, such
;;;;	values can be treated as constants. EG, in foo{}(<x>.x)[y/x] it
;;;;	is feasible to return foo{}(<x>.y). Wherever possible such values
;;;;	will be treated as constants.
;;;;
;;;;	 Error raised when :
;;;;	  - meta variables are used as targets for term substitution, ie t[s/$a].
;;;;	  - a meta abstraction variable in a variable operator, ie variable{$v:v}, in a
;;;;	    term in which parameters maybe  substituted into.
;;;;	  - pattern variable for match is meta.
;;;;	  
;;;;	 The rule interpreter expects to see only real values in rule instances.
;;;;	 If a non real value is encountered, the rule intepreter may raise an error
;;;;	 or it may produce unpredictable results. The implementation attempts to
;;;;     check all new input (ie existing proof nodes are not checked) to the rule
;;;;	 intepreter for non real values.
;;;;
;;;;
;;;;	Parameter Coercion : 
;;;;
;;;;	coercion may occur during abstraction expansion, bi-modal second order
;;;;	substition, evaluation, and primitive rule refinement.
;;;;
;;;;	- values may not be coerced into variable or level-expression types.
;;;;	  an error is signalled if such a coercion is attempted.
;;;;	- slots remain slots and coercion succeeds.
;;;;	- meta variables remain meta variables and coercion succeeds.
;;;;	- otherwise
;;;;	   the value is converted into a string in the source type; (error values ok)
;;;;	   then the string is converted to a value in the destination type;
;;;;	   then the new value is converted back into a string.
;;;;	  An error is signalled :
;;;;	   - if the conversion to the destination type resulted in an error;
;;;;	   - if the original string and the converted string are not identical.
;;;;	  Otherwise, the new value is returned.
;;;;
;;;;
;;;;  -page-
;;;;
;;;;	Time : a parameter type for universal time.
;;;;	 An integer value which is the number of seconds since Midnight 1/1/1900 GMT.
;;;;
;;;;  -page-
;;;;
;;;;	Level Expressions : (adapted, from doc written by Doug Howe with
;;;;	                        changes by Paul Jackson wrt le constants).
;;;;
;;;;	 i' ==> i 1
;;;;
;;;;	  Let v be a "level assignment" --- an assignment of numbers to level
;;;;	  variables.  Extend v to level expressions by v(e n) = v(e)+n and v([e1
;;;;	  ... ek]) = max(v(e1),...,v(ek)).
;;;;
;;;;	  For level expressions e1 and e2, define
;;;;	    1) e1 < e2 if for all v, v(e1) < v(e2),
;;;;	    2) e1 ~ e2 if for all v, v(e1) = v(e2).
;;;;
;;;;	  A level expression is in normal form if it is [k | i1 n1 | i2 n2 | ... | ik nk]
;;;;	  for some distinct level variables ij and some numbers nj.  Every
;;;;	  expression e has a normal form, i.e. there is an e' in normal form
;;;;	  such that e~e'.
;;;;	
;;;;	  We can decide the relations ~ and < by computing normal forms and
;;;;	  applying the following facts about normal forms.
;;;;	
;;;;	  Assume normal form of
;;;;	
;;;;	   1)  le is  [k | v1 i1 | ... | vn in]
;;;;	   2)  le' is  [k' | v'1 i'1 | ... | v'n' i'n']
;;;;	
;;;;	   (putting k = 1 if no explicit constant)
;;;;	
;;;;	   let floor(le) = max(k,1 + i1, ... , 1 + in)
;;;;	
;;;;	   floor(le) is the least value a level exp can take on under any
;;;;	   substitution.
;;;;	
;;;;	   Definitions of relns are:
;;;;	
;;;;	   le less le' == for all j (in range 1 to n)
;;;;	                    there must be an j' (in range 1...n')
;;;;	                      such that vj = v'j' & ij < i'j'
;;;;	                  and
;;;;	                     floor(le) < floor(le')
;;;;	
;;;;	   le less_equal le'
;;;;	               == for all j (in range 1 to n)
;;;;	                    there must be an j' (in range 1...n')
;;;;	                         such that vj = v'j' & ij =< i'j'
;;;;	                  and
;;;;	                     floor(le') =< floor(le')
;;;;	
;;;;	   le equal le' == le less_equal le' & le' less_equal le
;;;;
;;;; -doct- (mod trm)
;;;;
;;;;	normalize-level-expression (expr) : level-expression
;;;;	less-level-expression-p (expr1 expr2) : <bool>
;;;;	 ** t if expr 1 is less then expr 2.
;;;;	equal-less-level-expression-p (expr1 expr2) : <bool>
;;;;	 ** t if expr 1 is less than or equal to expr 2.
;;;;	equal-level-expressions-p (expr1 expr2) : <bool>
;;;;
;;;; -doce-

;;;;	
;;;;	Level expressions and meta variables:
;;;;	  
;;;;	  As level expressions have thier own variables, there is no
;;;;	need to allow meta-variables in level-expression parameters.
;;;;	In fact, it should be impossible to create such a parameter
;;;;	through normal channels. 
;;;;	
;;;;	 In the implementation then we can not always treat parameters
;;;;	where we allow meta-values uniformly. Ie, we need a case split
;;;;	to detect level-expressions and look for a level-variable.
;;;;	Cases where this arises:
;;;;	  - matching models with instances.
;;;;	     * abstractions
;;;;	     * rules
;;;;	     * termofs
;;;;	
;;;;	Nuprl-light :
;;;;	  
;;;;	Nuprl light does have a Meta level variable which we can not
;;;;	represent in the level expression parameter. So 
;;;;	make a diff type of parameter if the value is meta.
;;;;	



;;;; RLE NAP font modifiers in ichars ?

;;;; RLE TODO add ascii term stuff.

;;;; RLE TODO MILL fixup tt-compress and file compress to use standard sexpr rep.

;;;; TODO RLE
;;;;
;;;;	Some things to include in the term module:
;;;;
;;;;	 some way of finding and replacing as many terms as possible.
;;;;	  - replacement may be easy by (setf (cddr term) (cddr new-term))
;;;;	    or the like which leaves some structure intact but replaces large part.
;;;;	  - must not collects pointers to terms as that would preclude garbage-collecting terms.
;;;;	 - Thus collect closures which know where to find active terms.
;;;;
;;;;	 some way of searching terms for a pattern.
;;;;	 abstract ways of referring to class of terms.
;;;;	 return hits as addresses, term-address and library addresses or whatever other way of locating term
;;;;	 ie global ml variable.
;;;;
;;;;	RLE NAP might save some space to not derive term structure from marks but instead have a marked-term
;;;;	structure derived from the term structure. Then when marking convert to derived structure. Thus
;;;;	not wasting the word in every term just those marked. Access is same whether marked or no!
;;;;	If marked term has marks after values, no over head when looking up values, only overhead when looking
;;;;	for marks.
;;;;	
;;;;	have term be unnamed structure of three or more elements
;;;;	 : termtype parameters subterms [bindings, marks]
;;;;	??? need in place modification to do marks??? if so then make termtype extensible.
;;;;	 ie 'term is default but ('term . <marks>) is possible.
;;;;	(defun term-p (term ) (or (eql 'term termtype) (eql 'term (car termtype))))


#-dontinline
(eval-when (compile)
  (proclaim '(inline
	      variable-id-p display-meta-variable-id-p abstraction-meta-variable-id-p
	      meta-variable-id-p type-of-meta-variable-id
	      equal-types-p type-id-of-type lookup-typeid type-id-p
	      type-id-to-type unalias-typeid type-to-short-string
	      real-parameter-value-p equal-real-parameter-values real-parameter-value-to-string
	      string-to-real-parameter-value hash-real-parameter-value walk-real-parameter-value
	      type-of-extended-parameter-value rest-of-extended-parameter-value unmark-parameter-value
	      mark-of-parameter-value mark-parameter-value-p value-of-mark-parameter-value
	      slot-parameter-value-p descriptor-of-slot-parameter-value
	      error-parameter-value-p sexpr-of-error-parameter-value message-of-error-parameter-value
	      value-of-parameter-value marks-of-parameter-value
	      meta-type-id-char
	      type-of-parameter type-id-of-parameter value-of-parameter-n value-of-parameter
	      instantiate-parameter
	      edit-type-of-parameter-value meta-type-of-parameter-value
	      abstraction-meta-parameter-p display-meta-parameter-p
	      meta-parameter-p equal-parameters-p hash-parameter real-parameter-p
	      make-level-max level-max-p expressions-of-level-max
	      make-level-constant int-of-level-constant level-constant-p
	      make-level-variable level-variable-to-string level-variable-p
	      make-level-increment level-increment-p expression-of-level-increment increment-of-level-increment
	      parse-level-expression level-expression-to-string level-expression-visit
	      free-var-val-p left-free-var-val-p right-free-var-val-p
	      bound-var-val-p left-bound-var-val-p right-bound-var-val-p
	      variable-id-to-string
	      get-dummy-variable-id dummy-variable-id-p
	      get-dummy-display-meta-variable-id dummy-display-meta-variable-id-p
	      id-of-operator parameters-of-operator
	      parameter-of-operator parameter-type-of-operator parameter-value-of-operator
	      instantiate-operator equal-operators-p substitute-in-operator
	      id-of-term parameters-of-term operator-of-term
	      bound-terms-of-term bindings-of-bound-term-n term-of-bound-term
	      arities-of-term instantiate-term instantiate-bound-term term-to-pretty-string
	      meta-parameter-value-p
	      instantiate-variable-operator id-of-variable-term
	      id-of-term-sig parameters-of-term-sig arities-of-term-sig
	      equal-term-sigs-p term-sig-of-term
	      )))


;;PERF might be able to do without symbolp in many circumstances
(defmacro variable-id-p (id)
  (let ((v (gensym)))
    `(let ((,v ,id))
      (and
       (symbolp ,v)
       (eq (symbol-package ,v) *system-variable-package*)))))

(defun display-meta-variable-id-p (id)
  (and
    (symbolp id)
    (eq (symbol-package id) *system-display-meta-variable-package*)))

(defun abstraction-meta-variable-id-p (id)
  (and
    (symbolp id)
    (eq (symbol-package id) *system-abstraction-meta-variable-package*)))


(defun meta-variable-id-p (v)
  (or (display-meta-variable-id-p v)
      (abstraction-meta-variable-id-p v)))


(defun type-of-meta-variable-id (v)
  (cond
    ((abstraction-meta-variable-id-p v)
     'abstraction)
    ((display-meta-variable-id-p v)
     'display)
    (t (system-error (error-message '(type-of-meta-variable-id))))))


(defmacro variable-id-to-string (v)
  `(string ,v))

(defun variable-id-to-pretty-string (v)
  (cond
    ((display-meta-variable-id-p v)
     (concatenate 'string "<" (string v) ">"))
    ((abstraction-meta-variable-id-p v)
     (concatenate 'string "$" (string v)))
    (t (string v))))
       

;;;; ---------------------------------------------------------------------
;;;; -----------------    parameters     ----------------------------------
;;;; ---------------------------------------------------------------------
;;;
;;;	Parameter Type
;;;


(defstruct parameter-type
  (id nil)
  (verify-function #'stringp)
  (equality-function #'eql)
  (string-function #'(lambda (x) (princ-to-string x)))
  (parse-function #'(lambda(x) (read-from-string x)))
  (hash-function #'sxhash)
  (visit-function #'(lambda (x f) (declare (ignore f)) x))
  (shortest-alias nil)
  (upcase-id nil)
  )


(defun equal-types-p (type-a type-b)
  (eql (parameter-type-id type-a) (parameter-type-id type-b)))

(defun type-id-of-type (type)
  (parameter-type-id type))



;;;;
;;;;	Defining parameter types:
;;;;


(defvar *typeid-table* (make-hash-table :size 32))

(defun lookup-typeid (id)
  (gethash id *typeid-table*))

(defun typeid-add (id type)
  (when (lookup-typeid id)
    (warn "Redefining typeid ~a" id))
  (setf (gethash id *typeid-table*) type))

(defun type-id-p (id)
  (and (lookup-typeid id) t))

(defun type-id-to-type (type-id &optional errp)
  (or (lookup-typeid type-id)
      (if errp
	  (raise-error (error-message '(parameter type-id) (list type-id)))
	  (lookup-unknown-type type-id))))

(defun define-typeid-aliases (id aliases)
  (let ((type (type-id-to-type id)))
    (mapc #'(lambda (alias)
	      (when (< (length (string alias))
		       (length (string (parameter-type-shortest-alias type))))
		(setf (parameter-type-shortest-alias type) alias))
	      (unless (eql id alias)
		(typeid-add alias type)))
	  aliases)))


(defun unalias-typeid (type-id)
  (type-id-of-type (type-id-to-type type-id)))

(defun type-to-short-string (type)
  (string (parameter-type-shortest-alias type)))

(defun type-id-to-short-string (type-id)
  (string (parameter-type-shortest-alias (type-id-to-type type-id))))


(defun define-type-id (id f g h i j visit)
  (typeid-add id
	      (make-parameter-type :id id 
				   :verify-function f
				   :equality-function g
				   :string-function h
				   :parse-function i
				   :hash-function j
				   :visit-function visit
				   :shortest-alias id
				   :upcase-id (intern-system (string-upcase (string id))))))

(defvar *unknown-typeid-table* (make-hash-table :size 32))
(defun new-unknown-type-id (n)
  (make-parameter-type :id n
		       :verify-function #'stringp
		       :equality-function #'string=
		       :string-function #'(lambda (s) s)
		       :parse-function #'(lambda (s) s)
		       :hash-function #'sxhash))

(defun lookup-unknown-type (typeid)
  (or (gethash typeid *unknown-typeid-table*)
      (setf (gethash typeid *unknown-typeid-table*) (new-unknown-type-id typeid))))


;;; TODO : instances of (instantiate-parameter x *foo*) should be changed to (foo-parameter x)
(defmacro define-typeid ((id aliases global) f g h i j 
			 &optional visit)
  `(progn
    #-dontinline
    (eval-when (compile)
      (proclaim '(inline
		  ,(intern (concatenate 'string (string-upcase (string id)) "-PARAMETER-P"))
		  ,(intern (concatenate 'string (string-upcase (string id)) "-PARAMETER"))
		  ,(intern (concatenate 'string (string-upcase (string id)) "-TYPE-P"))
		  ,(intern (concatenate 'string (string-upcase (string id)) "-TYPEID-P"))
		  )))
    (define-type-id ',id ,f ,g ,h ,i ,j
		    ,(if visit
			 visit
			 `#'(lambda (x f)
			      (if (meta-variable-id-p x)
				  (funcall f x)
				  x))))
    (defconstant ,(intern (concatenate 'string "*" (string global) "*"))
      ',id)
    (defvar ,(intern (concatenate 'string "*" (string-upcase (string id)) "-TYPE*")))
    (setf ,(intern (concatenate 'string "*" (string-upcase (string id)) "-TYPE*"))
     (type-id-to-type ',id))
    (define-typeid-aliases ',id ',aliases)
    (defun ,(intern (concatenate 'string (string-upcase (string id)) "-TYPEID-P")) (typeid)
      (eql typeid 
	   ,(intern (concatenate 'string "*" (string global) "*"))))
    (defun ,(intern (concatenate 'string (string-upcase (string id)) "-TYPE-P")) (type)
      (eql (parameter-type-id type) 
	   ,(intern (concatenate 'string "*" (string global) "*"))))
    (defun ,(intern (concatenate 'string (string-upcase (string id)) "-PARAMETER-P"))
	(parameter)
      (and (parameter-p parameter)
	   (eql (type-id-of-parameter parameter)
		,(intern (concatenate 'string "*" (string global) "*")))))
    (defun ,(intern (concatenate 'string (string-upcase (string id)) "-PARAMETER"))
	(value)
      (instantiate-parameter value
       ,(intern (concatenate 'string "*" (string-upcase (string id)) "-TYPE*"))))))



;;;
;;;	Parameter Values:
;;;	
;;;	Functions expect real values and types.
;;; 	These are not functions on parameters, but on parameter componenets.
;;;

(defun real-parameter-value-p (value type)
  (funcall (parameter-type-verify-function type) value))

(defun equal-real-parameter-values (value-a value-b type)
  (funcall (parameter-type-equality-function type) value-a value-b))
    
(defun real-parameter-value-to-string (value type)
  (funcall (parameter-type-string-function type) value))

(defun string-to-real-parameter-value (string type)
  (let ((value (funcall (parameter-type-parse-function type) string)))
    (if (real-parameter-value-p value type)
	value
	(raise-error (error-message '(parameter parse) 
				    (list string (type-id-of-type type)))))))

(defun hash-real-parameter-value (value type)
  (funcall (parameter-type-hash-function type) value))

(defun walk-real-parameter-value (value type f)
  (funcall (parameter-type-visit-function type) value f))



;;;
;;;	Edit parameter values, including marks.
;;;

(defstruct (extended-parameter-value (:include marks))
  (type nil)				; one of '(slot error mark)
  (rest nil))

(defun type-of-extended-parameter-value (value)
  (if (extended-parameter-value-p value)
      (extended-parameter-value-type value)
      nil))

(defun rest-of-extended-parameter-value (value)
  (if (extended-parameter-value-p value)
      (extended-parameter-value-rest value)
      value))


;;;
;;;	Mark
;;;

(defun copy-extended-parameter-value (e)
  (make-extended-parameter-value
   :alist (marks-alist e)	;; maybe bad to copy, but expect caller to handle it.
   :type (extended-parameter-value-type e)
   :rest (extended-parameter-value-rest e)))
  
(defun mark-parameter-value (value &optional label mark)
  (let ((new-value (if (extended-parameter-value-p value)
		       value
		       (make-extended-parameter-value :type 'mark :rest value))))
    (when label (mark new-value label mark))
    new-value))
  
(defun unmark-parameter-value (value label)
  (when (extended-parameter-value-p value)
    (unmark value label)))
  
(defun mark-of-parameter-value (value label)
  (when (extended-parameter-value-p value)
    (mark-value value label)))

(defmacro mark-parameter-value-p (value)
  (let ((v (gensym)))
    `(let ((,v ,value))
      (and (extended-parameter-value-p ,v)
       (eql 'mark (type-of-extended-parameter-value ,v))))))

(defun value-of-mark-parameter-value (value)
  (rest-of-extended-parameter-value value))


;;;
;;;	Slot
;;;

(defvar *default-slot* (make-extended-parameter-value :type 'slot :rest "slot"))

(defun slot-parameter-value (&optional descriptor)
  (if descriptor
      (make-extended-parameter-value :type 'slot :rest (or descriptor "slot"))
      *default-slot*))

(defun slot-parameter-value-p (value)
  (and (extended-parameter-value-p value)
       (eql 'slot (type-of-extended-parameter-value value))))

;; may be meta var or string.
(defun descriptor-of-slot-parameter-value (value)
  (rest-of-extended-parameter-value value))


;;;
;;;	Error
;;;



;;;	Currently error sexpr is expected to be :
;;;
;;;	<parm-value-sexpr>	: <string>
;;;				| (<meta-char> . <string>)
;;;
;;;	RLE TODO should be extended to include slots. Callers should be
;;;	able to treat cleaner than current???
;;;

(defun error-parameter-value (sexpr msg)
  (make-extended-parameter-value :type 'error :rest (cons sexpr msg)))

(defun error-parameter-value-p (value)
  (and (extended-parameter-value-p value)
       (eql 'error (type-of-extended-parameter-value value))))

(defun message-of-rest-of-error-value (rest)
  (cdr rest))

(defun sexpr-of-error-parameter-value (value)
  (car (rest-of-extended-parameter-value value)))

(defun message-of-error-parameter-value (value)
  (message-of-rest-of-error-value (rest-of-extended-parameter-value value)))


;;rle todo inline new stuff above.


;;;
;;;	Extended parameter values
;;;

(defmacro parameter-error-message (tags value type &rest rest)
  (let ((gentype (gensym)))
    `(let ((,gentype ,type))
      (error-message '(parameter ,@tags)
       (list* (parameter-value-to-pretty-string ,value ,gentype)
	(type-id-of-type ,gentype)
	,@rest)))))

(defmacro value-of-parameter-value (value)
  (let ((v (gensym)))
    `(let ((,v ,value))
      (if (mark-parameter-value-p ,v)
	  (value-of-mark-parameter-value ,v)
	  ,v))))

(defun value-of-parameter-value-f (value)
  (value-of-parameter-value value))

(defun marks-of-parameter-value (value)
  (if (extended-parameter-value-p value)
      (mark-values value)
      nil))

(defun marks-of-parameter (p)
  (let ((value (value-of-parameter-n p)))
    (if (extended-parameter-value-p value)
	(mark-values value)
	nil)))


(defun copy-parameter-value (value)
  (if (mark-parameter-value-p value)
      (let ((marks (mark-values value))
	    (nvalue (value-of-parameter-value value)))
	(dolist (mark marks)
	  (setf nvalue (mark-parameter-value nvalue (car mark) (cdr mark))))
	nvalue)
      value))

	
	
	    

;;;
;;;	Parameter meta variables.
;;;
;;;	 distinquishable by examining value. It is possible for a value
;;;	to be both a meta value and a real value. Eg, an abstraction meta
;;;	variable is also a level expression value.

(defun meta-parameter-value-p (value)
  (meta-variable-id-p (value-of-parameter-value value)))
  
(defun meta-type-of-parameter-value (value)
  (cond
    ((and (extended-parameter-value-p value)
	  (slot-parameter-value-p value))
     (type-of-meta-variable-id  (descriptor-of-slot-parameter-value value)))
    (t (type-of-meta-variable-id value))))



(defun parameter-value-p (value type)
  (let ((value (value-of-parameter-value value)))
    (or (extended-parameter-value-p value)
	(meta-parameter-value-p value)
	(real-parameter-value-p value type))))



(defun equal-parameter-values-p (a b type)
  (let ((x (value-of-parameter-value a))
	(y (value-of-parameter-value b)))
    (cond
      
      ((extended-parameter-value-p x)
       (and (extended-parameter-value-p y)
	    (eql (type-of-extended-parameter-value x)
		 (type-of-extended-parameter-value y))
	    (case (type-of-extended-parameter-value y)
	      (slot (let ((descr-x (descriptor-of-slot-parameter-value x))
			  (descr-y (descriptor-of-slot-parameter-value y)))
		      ;; equal if meta types of values are eql, ie ignore descriptor values.
		      (if (meta-variable-id-p descr-x)
			  (when (meta-variable-id-p descr-y)
			    (eql (type-of-meta-variable-id descr-x) (type-of-meta-variable-id descr-y)))
			  (not (meta-variable-id-p descr-y)))))
	      (error (equal (sexpr-of-error-parameter-value x)
			    (sexpr-of-error-parameter-value y)))
	      (otherwise (system-error (error-message '(equal-parameter-values-p)))))))
      ((extended-parameter-value-p y)
       nil)
      
      ((meta-parameter-value-p x)
       ;; rle perf should be better way of handling following
       ;; but beware (meta-type-of-parameter-value y) fails if (meta-parameter-value-p y) is removed
       (and (meta-parameter-value-p y) (eql (meta-type-of-parameter-value x) (meta-type-of-parameter-value y))
	    (eql x y)))
      ((meta-parameter-value-p y) nil)
      
      (t (equal-real-parameter-values x y type)))))
     
;; suitable to pass back through string-to-parameter functions. ?? slot ??
;; however not suitable for ascii rep due to lack of escapes.
(defun parameter-value-to-pretty-string (a type)
  (let ((value (value-of-parameter-value a)))
    (cond
      ((extended-parameter-value-p value)
       (cond
	 ((slot-parameter-value-p value)
	  ;; apply string in case descriptor is a meta-variable-id.
	  (let ((descr (descriptor-of-slot-parameter-value value)))
	    (if (meta-variable-id-p descr)
		(variable-id-to-pretty-string descr)
		descr)))
	 ((error-parameter-value-p value)
	  (format-string (format-string "Error[>~a<]: ~a"
					(message-to-string (message-of-error-parameter-value value))
					(princ-to-string (sexpr-of-error-parameter-value value)))))
	 (t (system-error (error-message '(parameter-value-to-pretty-string))))))
      ((meta-parameter-value-p value)
       (variable-id-to-pretty-string value))
      (t (real-parameter-value-to-string value type)))))


(defun hash-parameter-value (a type)
  (let ((value (value-of-parameter-value a)))
    (cond
      ((extended-parameter-value-p value)
       (cond
	 ((slot-parameter-value-p value)
	  (let ((s (descriptor-of-slot-parameter-value value)))
	    (if (meta-variable-id-p s)
		(sxhash '(s . slot))
		(sxhash 'slot))))
	 ((error-parameter-value-p value)
	  (sxhash (sexpr-of-error-parameter-value value)))
	 (t (system-error (error-message '(hash-parameter-value))))))
      ((meta-parameter-value-p value)
       (sxhash value))
      (t (hash-real-parameter-value value type)))))

(defun walk-parameter-value (a type f)
  (let ((value (value-of-parameter-value a)))
    (cond
      ((extended-parameter-value-p value) a)
      ((abstraction-meta-variable-id-p value)
       (let ((new-value (funcall f value)))
	 (if (eql new-value value)
	     a
	     new-value)))
      (t (walk-real-parameter-value value type f)))))


(defun meta-type-id-char (meta-type-id)
  (case meta-type-id
    (abstraction 'a)
    (display 'd)
    (otherwise (system-error (error-message '(meta-type-id-char) meta-type-id)))))
    
(defconstant *ia* (char-code #\a))
(defconstant *islota* (char-code #\A))
(defconstant *id* (char-code #\d))
(defconstant *islotd* (char-code #\D))
(defconstant *islot* (char-code #\S))

(defun meta-type-id-code (meta-type-id &optional slotp)

  (case meta-type-id

    (abstraction	(if slotp *islota* *ia*))
    (display		(if slotp *islotd* *id*))

    (otherwise (system-error (error-message '(meta-type-id-char) meta-type-id)))))
    

(defun parameter-value-to-sexpr (a type)
  (let ((value (value-of-parameter-value a)))
    ;;(setf v value b type)  ;;(break)
    (cond
      ((extended-parameter-value-p value)
       (cond
	 ((slot-parameter-value-p value)
	  (let ((s (descriptor-of-slot-parameter-value value)))
	    (if (meta-variable-id-p s)
		(cons (meta-type-id-char (type-of-meta-variable-id s)) (cons 's nil))
		(cons 's nil))))
	 ((error-parameter-value-p value)
	  (sexpr-of-error-parameter-value value))
	 (t (system-error (error-message '(parameter-value-to-sexpr))))))
      ((meta-parameter-value-p value)
       (cons (meta-type-id-char (type-of-meta-variable-id value))
	     (variable-id-to-string value)))
      ((real-parameter-value-p value type)
       (real-parameter-value-to-string value type))
      (t (break "pvts") ;;
         (system-error (error-message '(parameter-value-to-sexpr value?)))
	 ))))


(defun maybe-string-to-parameter-value (string type)
  (with-handle-error (()
		      (error-parameter-value string
					     (error-message '(parameter string value)
							    (list string
								  (type-id-of-type type)))))
    (string-to-real-parameter-value string type)))


(defun sexpr-to-parameter-value (sexpr type)
  (cond
    ((stringp sexpr)
     (maybe-string-to-parameter-value sexpr type))

    ((consp sexpr)
     (case (car sexpr)

       (a (cond
	    ((stringp (cdr sexpr))
	     (get-abstraction-meta-variable-id (cdr sexpr)))
	    ((and (consp (cdr sexpr)) (eql 's (cadr sexpr)))
	     (slot-parameter-value (get-abstraction-meta-variable-id "slot")))
	    (t (system-error (error-message '(sexpr-to-parameter-value-a)
					   (list (princ-to-string sexpr)
						 (type-id-of-type type)))))))

       (d (cond
	    ((stringp (cdr sexpr))
	     (get-display-meta-variable-id (cdr sexpr)))
	    ((and (consp (cdr sexpr)) (eql 's (cadr sexpr)))
	     (slot-parameter-value (get-display-meta-variable-id "slot")))
	    (t (system-error (error-message '(sexpr-to-parameter-value-d)
					   (list (princ-to-string sexpr)
						 (type-id-of-type type)))))))

       (s (slot-parameter-value "slot"))

       (otherwise (system-error (error-message '(sexpr-to-parameter-value-other)
					      (list (princ-to-string sexpr)
						    (type-id-of-type type)))))))
    (t (system-error (error-message '(sexpr-to-parameter-value) (list (princ-to-string sexpr)
								     (type-id-of-type type)))))))


(defun coerce-parameter-value (src-type dest-type value)
  (cond
    ((equal-types-p src-type dest-type)
     value)

    ((variable-type-p dest-type)
     (raise-error (parameter-error-message (coerce variable) value src-type)))

    ((level-expression-type-p dest-type)
     (raise-error (parameter-error-message (coerce level-expression) value src-type)))

    (t (let* ((src-string nil)		; save string if produced.
	      (dest-value (cond
			    ((extended-parameter-value-p value)
			     (cond
			       ((slot-parameter-value-p value)
				value)

			       ;; NB: this allows an error to be coerced. Desirable as one may want
			       ;; to use coercion to fix an incorrectly specifed type.
			       ((error-parameter-value-p value)
				(let ((sexpr (sexpr-of-error-parameter-value value)))
				  (cond
				    ((stringp sexpr)
				     (setf src-string sexpr)
				     (maybe-string-to-parameter-value src-string dest-type))
				    ((and (consp sexpr)
					  (or (eql 'a (car sexpr))
					      (eql 'd (car sexpr)))
					  (stringp (cdr sexpr)))
				     (case (car sexpr)
				       (a (get-abstraction-meta-variable-id (cdr sexpr)))
				       (d (get-display-meta-variable-id (cdr sexpr)))))
				    (t (system-error (error-message '(coerce-parameter-value error)))))))

			       (t (system-error (error-message '(coerce-parameter-value))))))

			    ((meta-parameter-value-p value)
			     value)
			  
			    (t (setf src-string (real-parameter-value-to-string value src-type))
			       (maybe-string-to-parameter-value src-string dest-type)))))

	 (if (error-parameter-value-p dest-value)
	     (raise-error (parameter-error-message (coerce error)
						   dest-value dest-type
						   (type-id-of-type src-type)))

	     ;; check that string projections of real values are identical.
	     ;; if coercion changes strings then user needs to know (even for error values).
	     (when src-string
	       ;; rle todo Error if dest value is meta variable (and it may be).
	       (let ((dest-string (real-parameter-value-to-string dest-value dest-type)))
		 (when (not (string= src-string dest-string))
		   (raise-error (parameter-error-message (coerce diverged)
							 value src-type
							 dest-string (type-id-of-type dest-type)))))))
				    
	 ;; ok
	 dest-value))))

;;; This is a weak form of coercion, no guarauntees. 
(defun parameter-value-to-parameter-value (old-value type)
  (let* ((marks (when (extended-parameter-value-p old-value)
		  (mark-values old-value)))
	 (value (value-of-parameter-value old-value)))
    (cond
      ((extended-parameter-value-p value)
       old-value)
      ((meta-variable-id-p value)
       old-value)
      ((stringp value)
       (if marks
	   (marks (mark-parameter-value (string-to-real-parameter-value value type)) marks)
	   (string-to-real-parameter-value value type)))
      (t (unless (real-parameter-value-p value type)
	   (raise-error (error-message '(parameter value-to-value) (list (princ-to-string value)
									 (type-id-of-type type)))))
	 (if marks
	     (marks (mark-parameter-value value) marks)
	     value)))))


;;;;
;;;;	Parameters
;;;;

;;;;	
;;;;	
;;;;	
;;;;	
;;;;	all real parameter values are coercible to strings.
;;;;	But what is a string, suprisingly there is more than one answer.
;;;;	  - standard string
;;;;	      * all chars of string are standard chars with other chars represented
;;;;		as embedded unicode strings. Ie, if a\0080b corresponds to a
;;;;		string with codes (97 128 98)
;;;;	  - string
;;;;	  - escaped string
;;;;	
;;;;	
;;;;	
;;;;	


(defstruct (parameter (:print-function terminal-print-parameter))
  (type nil)
  (value nil)
  ;;  fbb-cache representation of parameter value in compressed ascii form.
  ;;  if present, then when writing parameter we can just copy the cache.
  (fbb-cache nil)
  )

(defun parameter-set-fbb-cache (p fbb)
  (setf (parameter-fbb-cache p) fbb))

(defun fbb-cache-of-parameter (p)
  (parameter-fbb-cache p))

(defun terminal-print-parameter (parameter stream depth)
  (declare (ignore depth))
  (format stream "#<~s>" (parameter-to-pretty-string parameter :include-type-id t))
  nil)


(defmacro type-of-parameter (parameter)
  `(parameter-type ,parameter))

(defun type-id-of-parameter (parameter)
  (parameter-type-id (type-of-parameter parameter)))

(defun type-upcase-id-of-parameter (parameter)
  (parameter-type-upcase-id (type-of-parameter parameter)))


(defun value-of-parameter-n (parameter)
  (parameter-value parameter))

(defmacro value-of-parameter (parameter)
  `(value-of-parameter-value (parameter-value ,parameter)))

(defun value-of-parameter-f (p) (value-of-parameter p))

(defun value-of-parameter-m (parameter)
  (let ((value (value-of-parameter parameter)))
    (cond
      ((extended-parameter-value-p value)
       (raise-error (parameter-error-message (value-m)
					     value (type-of-parameter parameter))))
      ((meta-parameter-value-p value)
       value)
      
      (t value))))

(defun value-of-parameter-r (parameter)
  (let ((value (value-of-parameter parameter)))
    
    ;; need to check if real in case a meta value is also a real value.
    (if (not (real-parameter-value-p value (type-of-parameter parameter)))
	(progn ;;(break "vpr")
	       (raise-error (parameter-error-message (value-r)
						     value (type-of-parameter parameter))))
	value)))


(defun real-parameter-p (parameter)
  (real-parameter-value-p (value-of-parameter parameter)
			  (type-of-parameter parameter)))

(defun check-too-long (s)
  (if (and (stringp s) (> (length s) 4000))
      (progn
	(setf -s s)
	(break "string too long")
	"fubar")
      s))

(defun instantiate-parameter (value type)
  ;; (and (oid-type-p type) (null value)) (break "ip"))
  (when (basic-message-p value) (break "ip"))
  (make-parameter :type type :value (check-too-long value)))

(defun instantiate-parameter-r (v type)
  (let ((value (value-of-parameter-value v)))
    (cond
      ((meta-variable-id-p value)
       (if (and (level-expression-type-p type)
		(abstraction-meta-variable-id-p value))
	   (raise-error (error-message '(parameter instantiate-r le-abs-meta)
				       (list (princ-to-string value)
					     (type-id-of-type type))))
	   (make-parameter :type type :value v)))
      
      ((slot-parameter-value-p value)
       (make-parameter :type type :value v))

      ((real-parameter-value-p value type)
       (make-parameter :type type :value (check-too-long v)))

      ;; if error then value is sexpr.
      (t ;;(break "ip") 
	 (raise-error (error-message '(parameter instantiate-r)
				     (list (princ-to-string value)
					   (type-id-of-type type))))))))


;; when unsure if input is meta value or string
(defun instantiate-parameter-m (meta-string type)
  (cond
    ((meta-variable-id-p meta-string)
     (if (and (level-expression-type-p type)
	      (abstraction-meta-variable-id-p meta-string))
	 (raise-error (error-message '(parameter instantiate-m le-abs-meta)
				     (list (princ-to-string meta-string)
					   (type-id-of-type type))))
	 (make-parameter :type type :value meta-string)))

    ((stringp meta-string)
     (make-parameter :value (let ((value (string-to-real-parameter-value meta-string type)
				    ))
			      (when (and (oid-type-p type) (null value))
				(break "ipm"))
			      (check-too-long value))

		     :type type))
    (t ;;(setf -meta-string meta-string -type type) (break "ipm")
       (raise-error (error-message '(parameter instantiate-m)
				   (list (princ-to-string meta-string)
					 (type-id-of-type type)))))))

(defun instantiate-parameter-s (string type)
  (cond
    ((stringp string)
     (instantiate-parameter (string-to-real-parameter-value string type)
			    type))
    (t (raise-error (error-message '(parameter instantiate-s)
				   (list (princ-to-string string)
					 (type-id-of-type type)))))))



(defun edit-type-of-parameter-value (value)
  (when (extended-parameter-value-p value)
    (type-of-extended-parameter-value value)))

(defun abstraction-meta-parameter-p (parameter)
  (abstraction-meta-variable-id-p (value-of-parameter parameter)))

(defun display-meta-parameter-p (p)
  (display-meta-variable-id-p (value-of-parameter p)))

(defun parameter-to-pretty-string (parameter &key suffix prefix include-type-id)
  (concatenate 'string
	       (or prefix "")
	       (parameter-value-to-pretty-string (value-of-parameter-n parameter)
						 (type-of-parameter parameter))
	       (if include-type-id
		   (concatenate 'string
				":"
				(string (type-id-of-parameter parameter)))
		   "")
	       (or suffix "")))



(defun meta-parameter-p (p)
  (meta-variable-id-p (value-of-parameter p)))

(defun slot-parameter-p (p)
  (slot-parameter-value-p (value-of-parameter p)))


(defun equal-parameters-p (parm-a parm-b)
  (and (equal-types-p (type-of-parameter parm-a) (type-of-parameter parm-b))
       (equal-parameter-values-p (value-of-parameter-n parm-a)
				 (value-of-parameter-n parm-b)
				 (type-of-parameter parm-a))))


(defun hash-parameter (parameter)
  (hash-parameter-value (value-of-parameter-n parameter)
			(type-of-parameter parameter)))

(defun walk-parameter (parameter f)
  (let* ((old-value (value-of-parameter-n parameter))
	 (type (type-of-parameter parameter))
	 (new-value (walk-parameter-value old-value type f)))
    (if (eql old-value new-value)
	parameter
	(instantiate-parameter new-value type))))



;; sub : (v . value)
;; where v is either template or parameter variable and
;;   and value is a template or parameter variable or a string or a parameter. 

(defun substitute-parameter (parameter subs)
  (let ((value (value-of-parameter parameter))
	(type (type-of-parameter parameter)))
    (if (display-meta-variable-id-p value)
	(let ((sub (cdr (assoc value subs))))
	  (cond
	    ((null sub)
	     parameter)
	    ((parameter-p sub)
	     (instantiate-parameter (coerce-parameter-value (type-of-parameter sub)
							    type
							    (value-of-parameter sub))
				    type))
	    ((meta-variable-id-p sub)
	     (instantiate-parameter sub type))
	    ((stringp sub)
	     (instantiate-parameter-s sub type))
	    ((slot-parameter-value-p sub)
	     (instantiate-parameter-r sub type))
	    
	    (t (system-error (error-message '(substitute-parameter meta-parameter))))))

	(with-tag '(substitute)
	  (walk-parameter
	   parameter
	   #'(lambda (v)
	       (let* ((p (cdr (assoc v subs))))
		 ;;(setf -v v -p p -parameter parameter -subs subs -type type) (break "sp")
		 (cond
		   ((null p) v)
		   ((parameter-p p)
		    (coerce-parameter-value (type-of-parameter p)
					    type
					    (value-of-parameter-m p)))
		   ((meta-variable-id-p p) p)
		   ((stringp p) (string-to-real-parameter-value p type))
		   (t ;;(setf e v f subs g p) (break)
		      (system-error (error-message '(substitute-parameter visit))))))))))))


;; RLE NAP PERF could try to avoid consing new list in case where no subs applicable
(defun substitute-parameters (parameters subs)
  (if (null subs)
      parameters
      (mapcar #'(lambda (parameter)
		  (substitute-parameter parameter subs))
		  parameters)))


(defun decimal-string-to-integer (str)
  (when (or (null str)
	    (string= "" str))
    (raise-error (error-message '(integer parse)
				str)))
  (with-string-scanner (str)
    (scan-decimal-num)))


;;;;
;;;; Standard parameter types:
;;;;


(define-typeid (|natural| (natural nat |nat| |n| n) natural-typeid)
	       #'(lambda (x) (and (integerp x) (>= x 0)))
	       #'eql
	       #'princ-to-string
	       #'decimal-string-to-integer
	       #'sxhash)


(define-typeid (|time| (time) time-typeid)
	       #'(lambda (x) (and (integerp x) (>= x 0)))
	       #'eql
	       #'princ-to-string
	       #'decimal-string-to-integer
	       #'sxhash)

(define-typeid (|token| (token tok |tok| |t| t) token-typeid)
    #'(lambda (x)
	(and (symbolp x)
	     ;;(when (variable-id-p x) (break "var tok"))
	     (not (meta-variable-id-p x))))
  #'eql
  #'string
  #'intern-system
  #'sxhash)

;; kludge 
(define-typeid (|OPID| () opid-typeid)
    #'(lambda (x)
	(and (symbolp x)
	     ;;(when (variable-id-p x) (break "var tok"))
	     (not (meta-variable-id-p x))))
  #'eql
  #'string
  #'intern-system
  #'sxhash)


(define-typeid (|string| (string |s| s) string-typeid)
    #'stringp
  #'string=
  #'(lambda (s) s)
  #'(lambda (s) s)
  #'sxhash)

(define-typeid (|quote| (quote |q| q) quote-typeid)
    #'(lambda (x)
	(and (symbolp x)
	     (not (meta-variable-id-p x))))
  #'eql
  #'string
  #'intern
  #'sxhash)

(define-typeid (|bool| (bool |b| b) bool-typeid)
    #'(lambda (x)
	(or (eql nil x) (eql t x)))
  #'eql
  #'(lambda (b) (if b "T" "F"))
  #'(lambda (x)
      (let ((ups (string-upcase x)))
	(unless (or (string= ups "NIL")
		  (string= ups "F")
		  (string= ups "FALSE")
		  (string= ups "0"))
	  t)))
  #'sxhash)


(define-typeid (|variable| (|variable-id| variable-id variable |v| v)
			   variable-id-typeid)
		#'(lambda (id) 
		    (variable-id-p id))
		#'eql
		#'string
		#'(lambda (x) (get-variable-id x))
		#'sxhash)


;;;;
;;;; level-expressions
;;;;

(defun make-level-max (exprs)
  (unless exprs
    (raise-error (error-message '(level-expression max) "No sub-expressions")))
  exprs)

(defun level-max-p (expr)
  (and (consp expr)
       (not (integerp (cdr expr)))))

(defun expressions-of-level-max (expr)
  expr)


(defun make-level-constant (i)
  (if (and (integerp i) (>= i 1))
      i
      (raise-error (error-message '(level-expression constant)
				  (princ-to-string i)))))


(defun int-of-level-constant (e) e)

(defun level-constant-p (i)
  (integerp i))

(defvar *level-variable-break-p* t)

(defun make-level-variable (string)
  (intern-system string))

;; we should not allow abstraction meta-variables.
;; this is sufficient if we know we have a level expression already though.
(defun level-variable-p (expr)
  (symbolp expr))

(defmacro level-variable-equal-p (a b) `(eql ,a ,b))

(defun make-level-increment (expr incr)
  (unless (>= incr 0)
    (raise-error (error-message '(level-expression increment)
				(princ-to-string incr))))
  (cons expr incr))

(defun level-increment-p (expr)
  (and (consp expr) (not (level-max-p expr))))

(defun expression-of-level-increment (expr)
  (car expr))

(defun increment-of-level-increment (expr)
  (cdr expr))


(defun meta-level-expression-p (expr)
  (cond
    ((symbolp expr) (not (or (display-meta-variable-id-p expr)
			     (abstraction-meta-variable-id-p expr))))
    ((integerp expr) (> expr 0))
    ((and (consp expr)
	  (not (integerp (cdr expr)))
	  (null (last expr 0))
	  (forall-p-optimized (e expr) (level-expression-p e))))
    ((consp expr)
     (and (integerp (cdr expr))
	  (>= (cdr expr) 0)
	  (meta-level-expression-p (car expr))))
    (t nil)))


(defun level-expression-p (expr)
  (cond
   ((symbolp expr) (not (display-meta-variable-id-p expr)))
   ((integerp expr) (> expr 0))
   ((and (consp expr)
	 (not (integerp (cdr expr)))
	 (null (last expr 0))
	 (forall-p-optimized (e expr) (level-expression-p e))))
   ((consp expr)
    (and (integerp (cdr expr))
	 (>= (cdr expr) 0)
	 (level-expression-p (car expr))))
   (t nil)))



;; in expr not null then normalized expressions first value is floor 
(defun normalize-level-expression (expr)
  (let ((normal-expression nil))
    (labels
	((le-floor (expr)
	   (cond
	     ((level-constant-p (car expr))
	      (let ((increments (mapcar #'increment-of-level-increment
					(cdr expr))))
		(if (null increments)
		    expr
		    (cons (make-level-constant
			   (max (int-of-level-constant (car expr))
				(1+ (apply #'max increments))))
			  (cdr expr)))))
	     (t (cons (1+ (apply #'max
				 (mapcar #'increment-of-level-increment expr)))
		      expr))))

	 (update-normal-expression (le nexpr)
	   (cond
	     ((null nexpr)
	      (setf normal-expression
		    (cond
		      ((null normal-expression) (list le))
		      ((level-constant-p (car normal-expression))
		       (cons (car normal-expression)
			     (cons le (cdr normal-expression))))
		      (t (cons le normal-expression)))))
	     ((level-constant-p le)
	      (if (level-constant-p (car normal-expression))
		  (when (> (int-of-level-constant le)
			   (int-of-level-constant (car normal-expression)))
		    (setf (car normal-expression) le))
		  (setf normal-expression (cons le normal-expression))))
	     ((and (level-increment-p (car nexpr))
		   (eql (expression-of-level-increment le)
			(expression-of-level-increment (car nexpr))))
	      (when (> (increment-of-level-increment le)
		       (increment-of-level-increment (car nexpr)))
		(setf (car nexpr) le)))
	     (t
	      #+cmu(let* ((n (cdr nexpr))
			  (r (update-normal-expression le n)))
		     r)
	      #-cmu(update-normal-expression le (cdr nexpr)))))

	 (le-reduce (expr i)
	   (cond
	     ((level-increment-p expr)
	      (le-reduce (expression-of-level-increment expr)
			 (+ (increment-of-level-increment expr) i)))
	     ((level-max-p expr)
	      (mapc #'(lambda (e) (le-reduce e i))
		    (expressions-of-level-max expr)))
	     (t (update-normal-expression (if (level-constant-p expr)
					      (+ (int-of-level-constant expr) i)
					      (make-level-increment expr i))
					  normal-expression)))))

      (le-reduce expr 0)
      (make-level-max (le-floor normal-expression)))))


(defun equal-level-expressions-p (e1 e2)
  ;;(break "e")
  (labels ((equal-normalized-level-expressions (expr1 expr2)
	     (and (= (length expr1) (length expr2))
		  (forall-p-optimized
		   (elt1 (expressions-of-level-max expr1))
		   (let ((elt2 (assoc (expression-of-level-increment elt1)
				      (expressions-of-level-max expr2))))
		     (and elt2
			  (= (increment-of-level-increment elt1)
			     (increment-of-level-increment elt2))))) )))
		  
    (let ((expr1 (normalize-level-expression e1))
	  (expr2 (normalize-level-expression e2)))

      (and (= (int-of-level-constant (car expr1))
	      (int-of-level-constant (car expr2)))
	   (equal-normalized-level-expressions (cdr expr1) (cdr expr2))))))


(defun less-level-expression-p (lesser other)
  (labels ((less-normalized-level-expression-p (nlesser nother)
	     (forall-p-optimized
	      (elt-l (expressions-of-level-max nlesser))
	      (let ((elt-o (assoc (expression-of-level-increment elt-l)
				  (expressions-of-level-max nother))))
		(and elt-o
		     (< (increment-of-level-increment elt-l)
			(increment-of-level-increment elt-o)))))))
    
  (let ((nlesser (normalize-level-expression lesser))
	(nother  (normalize-level-expression other)))

    (and (< (int-of-level-constant (car nlesser))
	    (int-of-level-constant (car nother)))
	 (less-normalized-level-expression-p (cdr nlesser)
					     (cdr nother))))))
  

;; [a1 m1 ... ak mk] =< [b1 n1 ... bl nl] iff for each 
;; i, 1<=i<=k, there is a j such that ai=bj and mi=<nj.

(defun equal-less-level-expression-p (l o)
   ;;(break "el")
  (labels ((equal-less-normalized-level-expression-p (nol noo)
	     (forall-p-optimized (elt-l (expressions-of-level-max nol))
				 (let ((elt-o (assoc (expression-of-level-increment elt-l)
						     (expressions-of-level-max noo))))
				   (and elt-o
					(<= (increment-of-level-increment elt-l)
					    (increment-of-level-increment elt-o)))))))

    (let ((nol (normalize-level-expression l))
	  (noo (normalize-level-expression o)))
      
      (and (<= (int-of-level-constant (car nol))
	       (int-of-level-constant (car noo)))
	   (equal-less-normalized-level-expression-p (cdr nol)
						     (cdr noo))))))



;;;
;;; Level expression parse/unparse.
;;;

(defvar *level-expression-escape-sbits*
  (standard-character-sbits (list iescape ispace inewline itab iquote ilsquare irsquare)))


(defun level-variable-to-string (v)
  (escape-string (string v)
		 *level-expression-escape-sbits*))

(defun string-to-level-expression (str)
  (when (or (null str)
	    (string= "" str))
    (raise-error (error-message '(level-expression parse)
				str)))
  (with-string-scanner (str)
    (scan-level-expression)))

(defun string-to-meta-level-expression (str)
  (when (or (null str)
	    (string= "" str))
    (raise-error (error-message '(level-expression parse)
				str)))
  (with-string-scanner (str)
    (scan-level-expression)))

(defun scan-level-expression ()
  (labels 
      ((scan-atom ()
	 (if (scan-at-byte-p ilsquare)
	     (prog1 (make-level-max (scan-delimited-list #'scan-expression
							 ilsquare irsquare
							 #'(lambda () (scan-byte ibar))))
	       (scan-whitespace))
	     (if (numeric-digit-code-p (scan-cur-byte))
		 (make-level-constant (scan-decimal-num))
		 (prog1 (make-level-variable (scan-string *level-expression-escape-sbits*))
		   (scan-whitespace)))))    
  
       (scan-numbers (expr)
	 (cond
	   ((scan-at-byte-p iquote)
	    (scan-next)
	    (scan-numbers (make-level-increment expr 1)))
	   ((numeric-digit-code-p (scan-cur-byte))
	    #+cmu(let ((r (scan-numbers (make-level-increment expr (scan-decimal-num)))))
		   r)
	    #-cmu(scan-numbers (make-level-increment expr (scan-decimal-num)))
	    )
	   (t expr)))

       (scan-expression ()
	 (scan-numbers (scan-atom))))

    (prog1
	(scan-expression)
      (unless (scan-eof-p)
	(scan-error '(level-expression) "unparseable text follows level expression")))))


;; rle nap : do some kind of accumulator or sexpr thing rather than nconc.
(defun level-expression-to-string (expr)
  (labels
      ((level-expression-to-sexpr (expr)
	 (cond
	   ((level-constant-p expr)
	    (let ((s (princ-to-string (int-of-level-constant expr))))
	      (values s (length s))))
	   ((level-max-p expr)
	    (max-to-sexpr expr))
	   ((level-variable-p expr)
	    (let ((s (level-variable-to-string expr)))
	      (values s (length s))))
	   (t (level-increment-to-sexpr expr nil 0))))
       
       (max-to-sexpr (expr)
	 (list-to-delimited-string-sexpr (expressions-of-level-max expr)
					 #'level-expression-to-sexpr
					 " | "
					 "["
					 "]"))

       (level-increment-to-sexpr (expr sexpr l)
	 (cond
	   ((level-increment-p expr)
	    (if (onep (increment-of-level-increment expr))
		(level-increment-to-sexpr (expression-of-level-increment expr)
					  (cons "'" sexpr)
					  (1+ l))
		(let ((incr-s (princ-to-string (increment-of-level-increment expr))))
		  (level-increment-to-sexpr (expression-of-level-increment expr)
					    (cons (cons " " incr-s)
						  sexpr)
					    (+ 1 l (length incr-s))))))
	   (t (mlet* (((le-sexpr le-l) (level-expression-to-sexpr expr)))
		     (values (cons le-sexpr sexpr) (+ l le-l)))))))

    (multiple-value-call #'string-sexpr-to-string (level-expression-to-sexpr expr))))
	     

(defun level-expression-visit (e f)
  (level-expression-visit-aux e f))
	       
(define-typeid (|level-expression| (|l| l level-expression) level-expression-typeid)
	       #'level-expression-p
	       #'equal-level-expressions-p
	       #'level-expression-to-string
	       #'string-to-level-expression
	       #'sxhash
	       #'level-expression-visit)

(defun level-expression-visit-aux (e f)
  (labels
      ((visit (e)
	 (cond
	   ((level-constant-p e) e)
	   ((level-variable-p e)
	    (funcall f e))
	   ((level-max-p e)
	    (make-level-max (mapcar #'visit
				    (expressions-of-level-max e))))
	   ((level-increment-p e)
	    (make-level-increment (visit (expression-of-level-increment e))
				  (increment-of-level-increment e)))
	   (t (system-error (parameter-error-message '(level-expression visit)
						    e *level-expression-type*))))))

    (visit e)))


(define-typeid (|meta-level-expression| (meta-level-expression |m| m) meta-level-expression-typeid)
  #'stringp ;;meta-level-expression-p
  #'string=
  #'(lambda (s) s)
  #'(lambda (s) s)
  ;;#'(lambda (s) (if (stringp s) s (level-expression-to-string s)))
  ;;#'(lambda (s) (string-to-meta-level-expression s))
  #'sxhash)

#|
(define-typeid (|meta-level-expression| (meta-level-expression |m| m) meta-level-expression-typeid)
  #'stringp
  #'string=
  #'(lambda (s) s)
  #'(lambda (s) s)
  #'sxhash)
|#

(define-typeid (|ifname| (ifname) ifname-typeid)
		#'type-id-p
		#'eql
 
		#'princ-to-string
		#'intern-system
		#'sxhash)

(define-typeid (|opname| (opname) opname-typeid)
		#'(lambda (id) (symbolp id))
		#'eql
		#'princ-to-string
		#'intern-system
		#'sxhash)

(define-typeid (|parameter-list| (parameter-list |pl| pl)
				 parameter-list-typeid)
		#'(lambda (pl)
		     (or (null pl)
			 (and (consp pl)
			      (forall-p #'parameter-p pl))))
		#'(lambda (pla plb)
		    (apply-predicate-to-list-pair pla plb
						  #'equal-parameters-p))
		#'(lambda (pl)
		    (with-byte-accumulator ('string)
		      (walk-list-delimited pl
					 #'(lambda (p)
					     (let ((type (type-of-parameter p)))
					       (walk-parameter-sexpr-ascii
						(parameter-value-to-sexpr (value-of-parameter-n p)
									  type)
						#'accumulate-byte)
					       (accumulate-byte icolon)
					       (string-to-byte-accumulator (type-to-short-string type)
									   *ascii-escape-sbits*
									   #'accumulate-byte)))
					 #'accumulate-byte
					 icomma ilcurly ircurly)))

		#'(lambda (s)
		    (with-string-scanner (s)
		      (scan-delimited-list
		       #'(lambda ()
			   (let ((sexpr (scan-parameter-value-sexpr (scan-ascii-string)))
				 (type (progn (scan-byte icolon)
					      (type-id-to-type
					       (intern-system (scan-ascii-string))))))
			     (instantiate-parameter (sexpr-to-parameter-value sexpr type)
						    type)))
		       ilcurly ircurly
		       #'(lambda ()
			   (scan-byte icomma)))))

		#'(lambda (pl)
		    (hash-parameters pl)))



;;;; ---------------------------------------------------------------------
;;;; -----------------    variables    ----------------------------------
;;;; ---------------------------------------------------------------------

(defstruct prl-var

  (template-symbol nil)
  (op-injection nil)	;garbage
  (term-injection nil)	;garbage

  (current-use 0)		; invocation count of last use.
  (current-minor-use 0)		; invocation count of last minor use.

  (vmarks (make-marks))		; marks within var-invoc.

  ;; rest could be on marks
  (bindings nil)		; deBruijn Index stack.

  (left-indices nil)		; for comparision of vars during
  (right-indices nil)		;  ...  recursive descent through two terms.

  (evcb nil)			; Expansion Variable Control Block
  )


(defstruct binding
  (index 0)			; deBruijn index.
  (value nil)			; user supplied value.
  )



;;;; ---------------------------------------------------------------------
;;;; ----------------- internal functions --------------------------------
;;;; ---------------------------------------------------------------------

(defun construct-prl-var (var-id)
  (let* ((id-str (string var-id))
	 (prl-var (make-prl-var ;;:name (intern id-str)
				)))
    (setf (prl-var-template-symbol prl-var)
	  (if (eq (symbol-package var-id) *system-display-meta-variable-package*)
	      var-id
	      (get-display-meta-variable-id id-str)))
    prl-var))

(defun check-variable-id-binding (id)
  (when (or (eq (symbol-package id) *system-variable-package*)
	    (eq (symbol-package id) *system-display-meta-variable-package*)
	    (eq (symbol-package id) *system-abstraction-meta-variable-package*))
    (unless (boundp id)
      (setf (symbol-value id) (construct-prl-var id))))
  id)

;; desires real value but will fail gracefully if not.
(defun variable-prl-var (var-sym)
  ;;(let ((value (value-of-parameter-value var-sym))) <body> )
    (unless (symbolp var-sym)
      (raise-error (parameter-error-message '(variable value) var-sym *variable-type*)))
    (symbol-value var-sym))



(defun free-var-val-p (var-val)
  (or (null var-val)
      (null (prl-var-bindings var-val))))

(defun left-free-var-val-p (var-val)
  (or (null var-val)
      (null (prl-var-left-indices var-val))))

(defun right-free-var-val-p (var-val)
  (or (null var-val)
      (null (prl-var-right-indices var-val))))

(defun bound-var-val-p (var-val)
  (and var-val
       (not (null (prl-var-bindings var-val)))))

(defun left-bound-var-val-p (var-val)
  (and var-val
       (not (null (prl-var-left-indices var-val)))))

(defun right-bound-var-val-p (var-val)
  (and var-val
      (not (null (prl-var-right-indices var-val)))))



;;;;
;;;;	variable-ids
;;;;


(defun get-variable-id (print-name)
  (let ((str (string print-name)))
    (when (string= "~" str)
      (setf str ""))
    (let ((sym (intern str *system-variable-package*)))
      
      (unless (boundp sym)
	(setf (symbol-value sym)
	      (construct-prl-var sym)))
      sym)))

(defun get-display-meta-variable-id (print-name)
  (let ((sym (intern (string print-name) *system-display-meta-variable-package*)))
    (if (boundp sym) 
	(unless (typep (symbol-value sym) 'prl-var)
	  (error "value of symbol ~a is not of type prl-var in system-display-meta-variable package" 
		 print-name))
	(setf (symbol-value sym) (construct-prl-var sym)))
    sym))

(defun get-abstraction-meta-variable-id (print-name)
  (let ((sym (intern (string print-name) *system-abstraction-meta-variable-package*)))
    (if (boundp sym) 
	(unless (typep (symbol-value sym) 'prl-var)
	  (error "value of symbol ~a is not of type prl-var in system-abstraction-meta-variable package" 
		 print-name))
	(setf (symbol-value sym) (construct-prl-var sym)))
    sym))


(defparameter *dummy-var* (get-variable-id '||))

(defun get-dummy-variable-id ()
  *dummy-var*)

(defun dummy-variable-id-p (id)
  (eq id *dummy-var*))


(defun get-dummy-display-meta-variable-id ()
  (get-display-meta-variable-id ""))

(defun dummy-display-meta-variable-id-p (id)
  (eql id (get-display-meta-variable-id "")))




;;;; ---------------------------------------------------------------------
;;;; -----------------    operators     ----------------------------------
;;;; ---------------------------------------------------------------------

(defmacro id-of-operator (op)
  `(car ,op))

(defmacro parameters-of-operator (op)
  `(cdr ,op))

(defun parameter-of-operator (op index)
  (nth index (parameters-of-operator op)))

(defun parameter-type-of-operator (op index)
  (type-of-parameter (nth index (parameters-of-operator op))))

(defun parameter-value-of-operator (op index)
  (value-of-parameter (nth index (parameters-of-operator op))))


(defconstant *variable* '|variable|) ;; variable opid.

(defvar *allow-dummy-variable-operator* nil)

(defvar *dummy-variable-error-operator*
  (cons *variable*
	(list (instantiate-parameter
	       (error-parameter-value "" (error-message '(parameter dummy variable)))
	       *variable-type*))))

(defmacro with-allow-dummy-variable-operator (&body body)
  `(let ((*allow-dummy-variable-operator* t))
    (declare (special *allow-dummy-variable-operator*))
    ,@body))


(defun dummy-variable-error-operator-r ()
  (declare (special *allow-dummy-variable-operator*))

  (if *allow-dummy-variable-operator*
      (message-emit (warn-message '(operator variable dummy)))
      (raise-error (error-message '(operator variable dummy))))
  *dummy-variable-error-operator*)
  

(defun instantiate-operator (id &optional parameters)
  (if (and (eql *variable* id)
	   (dummy-variable-id-p (value-of-parameter (car parameters))))
      (dummy-variable-error-operator-r)
      (cons id parameters)))

(defun maybe-instantiate-operator (op id parameters)
  (if (and (eql id (id-of-operator op))
	   (or (eql parameters (parameters-of-operator op))
	       (apply-predicate-to-list-pair-optimized 
		      parameters
		      (parameters-of-operator op)
		      eql)))
      op
      (instantiate-operator id parameters)))


(defmacro equal-opids-p (a b) `(eql ,a ,b))

(defun equal-operators-p (op-l op-r)
  (and (equal-opids-p (id-of-operator op-l) (id-of-operator op-r))
       (apply-predicate-to-list-pair-optimized ;; LAL did this for cmu, no optimized
	(parameters-of-operator op-l) 
	(parameters-of-operator op-r)
	equal-parameters-p)))


(defun operator-to-pretty-string (op)
  (list-to-string (parameters-of-operator op)
		  #'(lambda (parameter)
		      (parameter-to-pretty-string parameter :include-type-id t))
		  ", "
		  (concatenate 'string
			       (string (id-of-operator op))
			       "{")
		  "}"))


(defun substitute-in-operator (op subs)
  ;;(setf -op op -subs subs) (break (format-string "sio ~a" -op))
  (maybe-instantiate-operator
   op
   (id-of-operator op)
   (substitute-parameters (parameters-of-operator op) subs)))





;;;;
;;;;	term
;;;;

(defvar *term-to-pretty-string-f* nil)

(defun term-to-pretty-string (term)
  (funcall *term-to-pretty-string-f* term))


(defun term-print-function (term stream depth)
  (terpri stream)
  (print-term term (or depth 0) stream))

(defstruct (term (:include marks)
		 (:print-function term-print-function)
		 ;; (:type list)
		 ;;(:named)
		 )
  (values nil))


(defmacro id-of-term (term)
  `(caar (term-values ,term)))

(defmacro parameters-of-term (term)
  `(cdar (term-values ,term)))

;; to be used with caution.
(defun nreplace-operator (term new-op)
  (setf (car (term-values term)) new-op))

(defmacro operator-of-term (term)
  `(car (term-values ,term)))

(defmacro bound-terms-of-term (term)
  `(cdr (term-values ,term)))


(defun bindings-of-bound-term-r (bt)
  (let ((bindings (car bt)))
    (when bindings
      (if (exists-p #'(lambda (b)
			(when (not (real-parameter-value-p (value-of-parameter-value b) *variable-type*))
			  (raise-error (parameter-error-message (binding value-r)
								b *variable-type*)))
			(mark-parameter-value-p b))
		    bindings)
	  (mapcar #'value-of-parameter-value-f bindings)
	  bindings))))

(defun bindings-of-bound-term-m (bt)
  (let ((bindings (car bt)))
    (when bindings
      (if (exists-p #'(lambda (b)
			(when (extended-parameter-value-p (value-of-parameter-value b))
			  (raise-error (parameter-error-message (binding value-m)
								b *variable-type*)))
			(mark-parameter-value-p b))
		    bindings)
	  (mapcar #'value-of-parameter-value-f bindings)
	  bindings))))

(defmacro bindings-of-bound-term (bt)
  `(let ((bindings (car ,bt)))
    (when bindings
      (if (exists-p-optimized (b bindings) (mark-parameter-value-p b))
	  (mapcar #'value-of-parameter-value-f bindings)
	  bindings))))

(defmacro bindings-of-bound-term-n (bt)
  `(car ,bt))

(defmacro term-of-bound-term (bt)
  `(cdr ,bt))

(defun term-of-bound-term-f (bt) (cdr bt))

(defun bound-term-p (bound-term)
  (and (consp bound-term)
       (listp (car bound-term))
       (forall-p #'(lambda (b) (or (extended-parameter-value-p b)
				   (meta-variable-id-p b)
				   (variable-id-p b)))
		 (bindings-of-bound-term-n bound-term))
       (term-p (cdr bound-term))))



(defun substitute-in-bindings (bindings subs)
  (if (forall-p #'(lambda (b) (not (meta-variable-id-p (value-of-parameter-value b)))) bindings)
      bindings
      (mapcar #'(lambda (b)
		  (let* ((sub (cdr (assoc (value-of-parameter-value b) subs)))
			(value (value-of-parameter-value sub)))
		    (cond
		      ((null sub) b)
		      ((stringp value) (get-variable-id value))
		      ((or (variable-id-p value)
			   (meta-variable-id-p value)
			   (extended-parameter-value-p value)) ; slot or error
		       sub)
		      (t (system-error (error-message '(substitute-in-bindings)))))))
	      bindings)))


(defun arities-of-term (term)
  (mapcar #'(lambda (bound-term) (length (bindings-of-bound-term-n bound-term)))
	  (bound-terms-of-term term)))


(defun instantiate-term (op &optional bound-terms)
  ;;(when (and (eql `|!label| (car op))  (null bound-terms) ) (break  "aojf"))
  (make-term :values (cons op bound-terms)))

(defun maybe-instantiate-term (term op bound-terms)
  (if (and (eql op (operator-of-term term))
	   (or (eql bound-terms (bound-terms-of-term term))
	       (forall-p #'eql bound-terms (bound-terms-of-term term))))
      term
      (instantiate-term op bound-terms)))


(defun instantiate-bound-term (term &optional bindings)
  ;;(dolist (v bindings)
  ;;(unless (or (variable-id-p v) (meta-variable-id-p v)) (break "ibtnv")))

  (cons bindings term))

(defun maybe-instantiate-bound-term (bound-term bindings term)
  (let ((bt-bindings (bindings-of-bound-term-n bound-term)))
    (if (and (or (eql bindings bt-bindings)
		 (forall-p #'eql bindings bt-bindings))
	     (eql term (term-of-bound-term bound-term)))
	bound-term
	(instantiate-bound-term term bindings))))

;; address is two element list where first is index into bindings and second is bound-term
(defun nreplace-binding (new-id bound-term binding-index)
  (setf (car bound-term)
	(replace (copy-list (bindings-of-bound-term-n bound-term))
		 (list new-id)
		 :start1 binding-index :end1 (1+ binding-index))))


;;;
;;;  Following used in building define-primitive functions.
;;;

(defun get-term-in-term (term address)
  (if (endp address)
      term
      (get-term-in-term (term-of-bound-term (nth (car address) 
						 (bound-terms-of-term term)))
			(cdr address))))

(defun get-binding-in-term (term address)
  (cond 
    ((endp (cdr address)) nil)
    ((endp (cddr address))
     (value-of-parameter-value (nth (cadr address) 
				    (bindings-of-bound-term-n (nth (car address)
								   (bound-terms-of-term term))))))
    (t (get-binding-in-term (term-of-bound-term (nth (car address)
						     (bound-terms-of-term term)))
			    (cdr address)))))

(defun get-bound-term-in-term (term address)
  (cond
    ((endp address) nil)
    ((endp (cdr address)) (nth (car address) 
			       (bound-terms-of-term term)))
    (t (get-bound-term-in-term (term-of-bound-term (nth (car address)
							(bound-terms-of-term term)))
			       (cdr address)))))



;;;;  
;;;;	Variable terms.
;;;;


(defun instantiate-variable-operator (id)
  (op-injection-of-prl-var id))


(defun display-meta-variable-term-p (term)
  (and (null (bound-terms-of-term term))
       (let ((op (operator-of-term term)))
	 (and (eql *variable* (id-of-operator op))
	      (let ((parameters (parameters-of-operator op)))
		(and parameters (null (cdr parameters))
		     (let ((parameter (car parameters)))
		       (variable-type-p (type-of-parameter parameter))
		       (display-meta-variable-id-p (value-of-parameter parameter)))))))))  

(defmacro variable-operator-p (op)
  `(and (eql *variable* (id-of-operator ,op))
       (let ((parameters (parameters-of-operator ,op)))
	 (and parameters (null (cdr parameters))
	      (let ((parameter (car parameters)))
		(variable-type-p (type-of-parameter parameter))
		(variable-id-p (value-of-parameter parameter)))))))

(defun variable-term (name &optional bound-terms)
  (unless (or (variable-id-p name)
	      (meta-variable-id-p name))
    (system-error (error-message '(variable-term))))
  (if bound-terms
      (instantiate-term (instantiate-variable-operator name)
		      bound-terms)
      (term-injection-of-prl-var name)))

(defun canonical-variable-term ()
  (canonical-term
    (instantiate-operator *variable*
			  (list (instantiate-parameter (get-abstraction-meta-variable-id 'v) 
						       *variable-type*)))
    nil))

(defun modifiable-variable-term (name)
  (if (or (variable-id-p name)
	  (display-meta-variable-id-p name)
	  (abstraction-meta-variable-id-p name))
      (instantiate-term (op-injection-of-prl-var name))
      (system-error (error-message '(modifiable-variable-term)))))


(defmacro variable-p (term)
  (let ((v (gensym)))
    `(let ((,v ,term))
      (and (variable-operator-p (operator-of-term ,v))
       (null (bound-terms-of-term ,v))))))

(defun variable-term-p (term)
  (and (variable-operator-p (operator-of-term term))
       (forall-p-optimized (bound-term (bound-terms-of-term term))
			   (null (bindings-of-bound-term-n bound-term))
			   )
       ))

(defun id-of-variable-term (var-term)
  (value-of-parameter (car (parameters-of-term var-term))))

(defun nreplace-variable-term-id (term new-id)
  (nreplace-operator term
		     (op-injection-of-prl-var new-id)))


;; in some contexts, ie substitution, it is an error to have
;; an abs meta varible as the variable parameter value.
(defun abstraction-meta-variable-term-r (term)
  (when (let ((op (operator-of-term term)))
	  (and (eql *variable* (id-of-operator op))
	       (let ((parameters (parameters-of-operator op)))
		 (and parameters (null (cdr parameters))
		      (let ((parameter (car parameters)))
			(variable-type-p (type-of-parameter parameter))
			(abstraction-meta-variable-id-p (value-of-parameter parameter)))))))
    (raise-error (error-message '(variable parameter meta)
				(variable-id-to-pretty-string 
				 (value-of-parameter (car (parameters-of-term term))))))))



;;;
;;; Beware abbreviations in term-to-sexpr such as 1
;;; for natural_number{1:n} and u for variable{u:v}
;;;

(defun term-to-sexpr-stream (term stream)
  (write (term-to-sexpr term) :stream stream :level nil :length nil))
		   
(defun term-to-sexpr (term)
  (if (and (variable-p term)
	   nil)
      (string (id-of-variable-term term))
      (cons (cons (string (id-of-term term))
		  (mapcar #'(lambda (p)
			      (cons (parameter-value-to-sexpr (value-of-parameter-n p)
							      (type-of-parameter p))
				    (type-to-short-string (type-of-parameter p))))
			  (parameters-of-term term)))
	    (mapcar #'(lambda (bt)
			(cons (mapcar #'(lambda (b)
					  (parameter-value-to-sexpr b *variable-type*))
				      (bindings-of-bound-term-n bt))
			      (term-to-sexpr (term-of-bound-term bt))))
		    (bound-terms-of-term term)))))

;;;;	RLE TODO : sexpr files may contain non-standard strings. Convert here.
;;;;	RLE TODO : sexpr files with non-standard strings are vestigial from original migration.
;;;;	RLE TODO : once dynamic migration is available they should be deleted.
;;;;	RLE TODO : questionable whether to support sexpr files at all as they are space consuming.
;;;;	RLE TODO : they have been more robust then the compressed ascii though.

(defun parameter-sexpr-to-standard-string-sexpr (psexpr)
  ;;(setf a psexpr)( break)
  (when psexpr
    (if (stringp psexpr)
	(string-to-standard-character-string psexpr)
	(if (consp psexpr)
	    (cons (car psexpr)
		  (parameter-sexpr-to-standard-string-sexpr (cdr psexpr)))))))

(defun sexpr-to-term (sexpr)
  (if (stringp sexpr)
      (variable-term (get-variable-id (string-to-standard-character-string sexpr)))
      (instantiate-term
       (instantiate-operator (intern-system (string-to-standard-character-string (caar sexpr)))
			     (mapcar #'(lambda (sexpr)
					 (let ((type (type-id-to-type
						      (intern-system (cdr sexpr)))))
					   (instantiate-parameter
					    (sexpr-to-parameter-value
					     (parameter-sexpr-to-standard-string-sexpr (car sexpr)) type)
					    type)))
				     (cdar sexpr)))
       (mapcar #'(lambda (sexpr)
		   (instantiate-bound-term (sexpr-to-term (cdr sexpr))
					 (mapcar #'(lambda (sexpr)
						     (sexpr-to-parameter-value
						      (parameter-sexpr-to-standard-string-sexpr sexpr)
						      *variable-type*))
						 (car sexpr))))
	       (cdr sexpr)))))





(defun term-injection-of-prl-var (var)
    (or (prl-var-term-injection (symbol-value var))
	(setf (prl-var-term-injection (symbol-value var))
	      (instantiate-term (op-injection-of-prl-var var)))))

(defun op-injection-of-prl-var (var)
    (or (prl-var-op-injection (symbol-value var))
	(setf (prl-var-op-injection (symbol-value var))
	      (instantiate-operator *variable* (list (instantiate-parameter var *variable-type*))))))


;;;;
;;;;	Term Sigs:
;;;;


(defmacro id-of-term-sig (sig) `(car ,sig))
(defmacro parameters-of-term-sig (sig) `(cadr ,sig))
(defmacro arities-of-term-sig (sig) `(cddr ,sig))

(defmacro equal-term-sigs-p (a b) `(equal ,a ,b))


(defun term-to-term-sig (term)
  (list* (id-of-operator (operator-of-term term))
	 (mapcar #'type-id-of-parameter
		 (parameters-of-operator (operator-of-term term)))
	 (mapcar #'(lambda (bound-term)
		     (length (bindings-of-bound-term-n bound-term)))
		 (bound-terms-of-term term))))

(defun term-sig-of-term (term)
  (mark-f term 'term-sig #'term-to-term-sig))


(defun term-sig-of-term-p (sig term)
  (and (eql (id-of-term term) (id-of-term-sig sig))
       (apply-predicate-to-dotted-lists (parameters-of-term term)
					(parameters-of-term-sig sig)
					#'(lambda (p typeid)
					    (eql (type-id-of-parameter p) typeid)))
       (apply-predicate-to-dotted-lists (bound-terms-of-term term)
					(arities-of-term-sig sig)
					#'(lambda (bound-term arity)
					    (= (length (bindings-of-bound-term-n bound-term)) arity)))))

(defun term-sig-op-of-term-p (sig term)
  (and (eql (id-of-term term) (id-of-term-sig sig))
       (apply-predicate-to-dotted-lists (parameters-of-term term)
					(parameters-of-term-sig sig)
					#'(lambda (p typeid)
					    (eql (type-id-of-parameter p) typeid)))
       ))

(defun term-sig-arities-of-term-p (sig term)
  (and (eql (id-of-term term) (id-of-term-sig sig))
       (apply-predicate-to-dotted-lists (bound-terms-of-term term)
					(arities-of-term-sig sig)
					#'(lambda (bound-term arity)
					    (= (length (bindings-of-bound-term-n bound-term)) arity)))
       ))
		 

;;;  --------------------------------------------------------------
;;;  ----------------- Primitives ---------------------------------
;;;  --------------------------------------------------------------

(defun canonical-term (op arities)
  (labels
    ((build-bound-term-list (arities i)
       (if (null arities)
	   nil
	   (cons (instantiate-bound-term (variable-term 
					 (get-variable-id
					   (concatenate 'string "t" (prin1-to-string i))))
				       (build-binding-list (car arities) 1))
		 (build-bound-term-list (cdr arities) (1+ i)))))

     (build-binding-list (arity i)
       (if (> i arity)
	   nil
	   (cons (get-variable-id (concatenate 'string "b" (prin1-to-string i)))
		 (build-binding-list arity (1+ i))))))
	
  (instantiate-term op
		  (build-bound-term-list arities 1))))


;; RLE TODO canonical-x-term is wrong when there are sub terms.
;; defconstant of canonical op seems wasteful.

(defun _->- (in)
  (let ((s (string-upcase (lisp:copy-seq (string in)))))
    (dotimes (i (length s))
      (when (eql (aref s i) #\_)
	(setf (aref s i) #\-)))
    s))
    
;;;;	define-primitive-x (pon function-root ...)
;;;;	 would allow alternate function set
;;;;	
;;;;	then when two possibilities ie one is extenstion of first
;;;;	then define two alternates and handcode usual


(defmacro define-primitive (pon
			    &optional dotted-parameters 
			    bt-specs
			    xfuncs-p
			    )
  (labels ((convert-opid-to-f-root (opid)
	     (let ((s (string-upcase (lisp:copy-seq (string opid)))))
	       (dotimes (i (length s))
		 (cond
		   ((eql (aref s i) #\_)
		    (setf (aref s i) #\-))
		   ((eql (aref s i) #\!)
		    (setf (aref s i) #\I))))
	       s))
	
	   (f-root-to-nil-alias (f-root)
	     (let* ((s (string f-root))
		    (l (length s))
		    (sc "CONS")
		    (lc (length sc)))

	       (when (and (> l lc) 
			  (string= sc f-root :start2 (- l lc)))
		 (concatenate 'string (subseq s 0 (- l lc)) "NIL")))))
    
    (let* ((parameter-names (mapcar #'cdr dotted-parameters))
	   (type-ids (mapcar #'(lambda (dp) (unalias-typeid (car dp))) dotted-parameters))
	   (types (mapcar #'(lambda (typeid)
			      (intern (concatenate 'string
						   "*"
						   (string-upcase (string typeid))
						   "-TYPE*")))
			  type-ids))
	   (function-alias (convert-opid-to-f-root (string pon)))
	   (arities (mapcar #'(lambda (bt-spec)
				(if (consp bt-spec)
				    (car bt-spec)
				    0))
			    bt-specs))
	   (subterm-names (mapcar #'(lambda (bt-spec)
				      (if (consp bt-spec)
					  (cdr bt-spec)
					  bt-spec))
				  bt-specs))
	   (opid-global (intern (concatenate 'string "*" function-alias "*")))
	   (op-global (intern (concatenate 'string "*" function-alias "-OPERATOR*")))
	   (term-sig-global (intern (concatenate 'string "*" function-alias "-TERM-SIG" "*")))
	   (nil-f-root (when (and (null parameter-names) (equal arities '(0 0)))
			 (f-root-to-nil-alias function-alias)))
	   )

      `(progn
	#-dontinline
	(eval-when (compile)
	  (proclaim '(inline
		      ,(intern (concatenate 'string function-alias "-OP"))
		      ,(intern (concatenate 'string function-alias "-TERM"))
		      ,(intern (concatenate 'string function-alias "-TERM-P"))
		      ,@(when xfuncs-p
			      (list
			       (intern (concatenate 'string function-alias "-TERM-OP-P"))
			       (intern (concatenate 'string function-alias "-TERM-ARITIES-P")))))))

	;; *<f-root>*
	(defconstant ,opid-global ',pon)

	;; *<f-root>-operator*
	(defparameter ,op-global
	  (instantiate-operator ,opid-global
				(list ,@(mapcar #'(lambda (name type)
						    `(instantiate-parameter
						      (get-abstraction-meta-variable-id ,(string name))
						      ,type))
						parameter-names
						types))))

	;; *<f-root>-term-sig*
	(defparameter ,term-sig-global
	  (cons ,opid-global
		(cons '(,@type-ids)
		      '(,@arities))))
      
	;; <f-root>-term-p
	(defun ,(intern (concatenate 'string function-alias "-TERM-P")) (term)
	  (term-sig-of-term-p ,term-sig-global term))


	,@(when xfuncs-p
		(list
		 `(defun ,(intern (concatenate 'string function-alias "-TERM-OP-P")) (term)
		   (term-sig-op-of-term-p ,term-sig-global term))
	
		 `(defun ,(intern (concatenate 'string function-alias "-TERM-ARITIES-P")) (term)
		   (term-sig-arities-of-term-p ,term-sig-global term))))

	;; <f-root[nil/cons]>-term-p
	,@(when nil-f-root
	    (list
	     `(defun ,(intern (concatenate 'string nil-f-root "-TERM-P")) (term)
	       (and (eql ,opid-global (id-of-term term))
		(null (parameters-of-term term))
		(null (bound-terms-of-term term))))
	   
	     ;;  <f-root[nil/cons]>-op
	     ;;`(defun ,(intern (concatenate 'string nil-f-root "-OP")) ()
	     ;;,op-global)

	     ;;  <f-root[nil/cons]>-op
	     `(defparameter ,(intern (concatenate 'string "*" nil-f-root "-TERM*"))
	       (instantiate-term ,op-global nil))

	     ;; <f-root[nil/cons]>-term
	     `(defun ,(intern (concatenate 'string nil-f-root "-TERM")) ()
	       ,(intern (concatenate 'string "*" nil-f-root "-TERM*")))))

	;; <f-root>-canonical-term
	(defun ,(intern (concatenate 'string "CANONICAL-" function-alias  "-TERM")) ()
	  (canonical-term ,op-global ',arities))

	;;  <f-root>-op
	(defun ,(intern (concatenate 'string function-alias "-OP")) (,@parameter-names)
	  ,(if (zerop (length parameter-names))
	       op-global
	       `(instantiate-operator ,opid-global
		 (list ,@(mapcar #'(lambda (name type)
				     `(instantiate-parameter-r ,name ,type))				     
			  parameter-names
			  types)))))

	,@(if (and (null parameter-names) (null arities))
	      `(
		(defparameter ,(intern (concatenate 'string "*" function-alias "-TERM*"))
		  (instantiate-term ,op-global nil))

	       ;; <f-root>-term
		(defun ,(intern (concatenate 'string function-alias "-TERM")) ()
		  ,(intern (concatenate 'string "*" function-alias "-TERM*"))))

	  
	      (let ((args nil)
		    (constructs nil))
		(do ((arits arities (cdr arits))
		     (i (length arities) (1- i)))
		    ((endp arits))
		  (if (= 0 (car arits))
		      (progn 
			(push (gensym) args)
			(push `(instantiate-bound-term ,(car args)) constructs))
		      (progn
			(push (gensym) args)
			(push (gensym) args)
			(push `(instantiate-bound-term ,(car args)
				,(if (= 1 (car arits))
				     `(list (get-variable-id ,(cadr args)))
				     `(mapcar #'get-variable-id ,(cadr args))))
			      constructs))))

		(setf constructs (nreverse constructs))


 
		`(
		  ;; <f-root>-term
		  (defun ,(intern (concatenate 'string function-alias "-TERM"))
		   (,@parameter-names ,@(nreverse args))
		   (instantiate-term 
		    ,(if (zerop (length parameter-names))
			 op-global
			 `(instantiate-operator ,opid-global
			   (list
			    ,@(mapcar #'(lambda (name type)
					  `(instantiate-parameter-r ,name ,type))
			       parameter-names types))))
		    ,(if arities
			 `(list ,@constructs)
			 nil)))

		 ;; <parameter-spec-name>-of-<f-root>-term
		 ,@(do ((pnames parameter-names (cdr pnames))
			(i 0 (1+ i))
			(funcs nil))
		       ((endp pnames) funcs)
		     (push `(defun ,(intern (concatenate 'string
					     (string-upcase (string (car pnames)))
					     "-OF-"
					     function-alias
					     "-TERM"))
			     (term)
			     (parameter-value-of-operator (operator-of-term term) ,i))
			   funcs))
		   

		 ;; <bt-spec-name>-of-<f-root>-term
		 ;; binding-of-<bt-spec-name>-of-<f-root>-term (term) : binding
		 ;; bindings-of-<bt-spec-name>-of-<f-root>-term (term) : binding list
		 ,@(do ((snames subterm-names (cdr snames))
			(arits arities (cdr arits))
			(i 0 (1+ i))
			(funcs nil))
		       ((endp snames) funcs)
		     (push `(defun ,(intern (concatenate 'string 
					     (string-upcase (string (car snames)))
					     "-OF-" 
					     function-alias
					     "-TERM")) 
			     (term)
			     (get-term-in-term term '(,i)))
			   funcs)
		     (unless (= (car arits) 0)
		       (if (= (car arits) 1)
			   (push `(defun ,(intern (concatenate 'string 
						   "BINDING-OF-"
						   (string-upcase (string (car snames)))
						   "-OF-"
						   function-alias
						   "-TERM"))
				   (term)
				   (get-binding-in-term term '(,i 0)))
				 funcs)
			   (push `(defun ,(intern (concatenate 'string 
						   "BINDINGS-OF-"
						   (string-upcase (string (car snames)))
						   "-OF-"
						   function-alias
						   "-TERM"))
				   (term)
				   (bindings-of-bound-term (get-bound-term-in-term term '(,i))))
				 funcs)))))))))))

;;;;
;;;;
;;;;	Term as mixed S-EXPR :
;;;;
;;;;	<term-msexpr> 		: ((<mstring> . <parm-msexpr> list) . <bound-term-msexpr> list)
;;;;				| <term-index>
;;;;				| ('add <term-index>
;;;;				       ((<mstring> . <parm-msexpr> list) . <bound-term-msexpr> list))
;;;;	<parm-msexpr>		: (<parm-value-msexpr> . <mstring>)
;;;;				| <parm-index>
;;;;				| (add <parm-index> (<parm-value-msexpr> . <mstring>))
;;;;	<bound-term-msexpr>	: (<parm-value-msexpr> list . <term-msexpr>)
;;;;				| <bound-term-index> 
;;;;				| (add <bound-term-index> (<parm-value-msexpr> list . <term-msexpr>))
;;;;	<parm-value-msexpr>	: <mstring>
;;;;				| (<slot-char> . nil)
;;;;				| (<meta-char> . (<slot-char> . nil))
;;;;				| (<meta-char> . <mstring>)
;;;;	<meta-char>		: a | d
;;;;	<slot-char>		: s
;;;;
;;;;	<mstring>		: <string> | <string-index> | (add <string-index> <string>)
;;;;
;;;;	*** might want to include hash numbers in add forms to save some work for receiver.
;;;;	*** requires us to ensure that hash numbers depend only on sexpr information.

;;;;	compression-table :
;;;;
;;;;	<compr-entry>		: #compr-entry[<term> <tag> <alist{links}>]
;;;;	<compr-table>		: <compr-entry> hash-table
;;;;
;;;;
;;;;
;;;;	  - for each link, consider one endpoint to be odd and the other even.
;;;;	    each endpoint maintains duplex lookup arrays (one odd one even) for each link.
;;;;	    The even endpoint may add even indices to the even array.
;;;;	    The odd endpoint may add odd indices to the odd array.
;;;;	    These duplex arrays should be mapped so that they are dense.
;;;;
;;;;	  - an index taken off a link, it should be one of the duplex arrays.
;;;;	  - a term to be sent through a link, should be expressable as a mixed-sexpr
;;;;	    where the indices are indices in the links duplex arrays.
;;;;
;;;;	 Each endpoint should have a local hash table.
;;;;	  - hash table is used to find to alist of link indices.
;;;;	  - hash table should provide for some method of tagging terms in the equivalence
;;;;		class of terms.
;;;;
;;;;	Lexicographic equality (ie compare-terms-p) can be as an equivalence relation to
;;;;	divide the terms into equivalence classes. Members of such an equivalence classes
;;;;	will be considered identical. Lexicographic equality is a function on the sexpr
;;;;	representation of terms. Thus if we can assign unique tags to members of an
;;;;	equivalence class, we could check equality by comparing the tags.
;;;;	
;;;;	 Different terms will may have the same hash number.
;;;;	 - track stats to check up on hash function.
;;;; 	 - an entry in the hash table will have a list of terms for the hash key.
;;;;	 - if just one term, it must still be compared for match.
;;;;	 Term : should be marked with hash number, and could be marked with mixed sexpr.
;;;;	  when comparing terms if both terms have mixed sexprs then mixed sexprs could be compared.
;;;;
;;;;	Delete Protocol :
;;;;	 request-delete :
;;;;	  - after producing request-delete, endpoint may no longer send deleted indices
;;;;	    but must still be prepared to recieve them.
;;;;	  - after receiveing request-delete, endpoint may not longer send and need not expect
;;;;	    to receive them.
;;;;     ack-delete : after receiveing ack-delete endpoint need not expect to receive deleted indices.
;;;;	*** NB : must ensure synchonization, ie don't want delete message to get ahead of messages
;;;;	 using deleted indices.


;;; would be neat to have local compression in a proof that compresses all assum types and concls
;;; within proof step. May make some huge primitive proofs tractable.


;;;;
;;;; RLE TODO : Need many more of these
;;;;
#-cmu 
(eval-when (compile load)
  (proclaim '(function bound-terms-of-term (term) list))
  (proclaim '(function bindings-of-bound-term (cons) list))
  (proclaim '(function term-of-bound-term (cons) term))
  )





;;;
;;;	oid and stamps
;;;

(defstruct oid
  stamp
  string ;; string rep of stamp
  )

(defun new-oid (stamp) (make-oid :stamp stamp))
(defun new-oid-s (s)  (make-oid :string s))
(defun new-oid-ss (string stamp)  (make-oid :string string :stamp stamp))

(defun stamp-of-oid (o)
  (or (oid-stamp o)
      (setf (oid-stamp o) (term-to-stamp (standard-character-string-to-term (oid-string o))))))

(defun string-of-oid (o)
  (or (oid-string o)
      (setf (oid-string o) (term-to-standard-character-string (stamp-to-term (oid-stamp o))))))

(defun string-to-oid (str)
  (new-oid-ss str (term-to-stamp (standard-character-string-to-term str))))

;; <stamp> : ((n{tcount} . string) . (n{seq} . n{time}))
;; internal rep to be stamp or stamp/string combo. do not want to 
(define-typeid (|oid| (oid |o|) oid-typeid)
    #'oid-p
  #'(lambda (a b)
      (equal-stamps-p (stamp-of-oid a) (stamp-of-oid b)))
  #'string-of-oid
  #'(lambda (str)
      (new-oid-ss str (term-to-stamp (standard-character-string-to-term str))))
  #'(lambda (o) (sxhash (stamp-of-oid o))))





(defun edit-parameter-parts (parameter)
  (let* ((type (type-of-parameter parameter))
	 (value (value-of-parameter-n parameter))
	 (sexpr (parameter-value-to-sexpr value type)))
    ;;(setf c value b sexpr)
    (list* (cond ((error-parameter-value-p value)
		  'error)
		 ((and (consp sexpr)
		       (or (eql (car sexpr) 's)
			   (and (consp (cdr sexpr))
				(eql (cadr sexpr) 's))))
		  'slot)
		 (t nil))
	   (when (and (consp sexpr)
		      (not (eql (car sexpr) 's)))
	     (case (car sexpr)
	       (a 'abstraction)
	       (d 'display)
	       (otherwise nil)))
	   (type-id-of-type type)
	   (cond ((stringp sexpr) sexpr)
		 ((and (consp sexpr)
		       (stringp (cdr sexpr)))
		  (cdr sexpr))
		 (t "")))))


(defun make-edit-parameter (edit-type meta-type type string &optional marks)
  (let ((value-sexpr (case edit-type
		       (slot (case meta-type
			       (display (cons 'd (cons 's string)))
			       (abstraction (cons 'a (cons 's string)))
			       ((nil) (cons 's nil))
			       (otherwise (raise-error
					   (error-message '(make_edit_parameter slot meta-type))))))
		       ((error nil) (case meta-type
				      (display (cons 'd string))
				      (abstraction (cons 'a string))
				      ((nil) string)
				      (otherwise (raise-error
						  (error-message (list 'make_edit_parameter edit-type))))))
		       (t  (raise-error (error-message '(make_edit_parameter edit-type)))))))

    (with-handle-error (()
			;;(setf a value-sexpr) (break)
			(raise-error (error-message '(make_edit_parameter value-sexpr)
						    (list edit-type meta-type type string))))
      (let ((type (type-id-to-type type)))
	(instantiate-parameter (if marks
				   (marks-c (mark-parameter-value
					     (sexpr-to-parameter-value value-sexpr type))
					    marks)
				   (sexpr-to-parameter-value value-sexpr type))
			       type)))))


(defun edit-type-of-parameter-parts (pparts) (car pparts))
(defun meta-type-of-parameter-parts (pparts) (cadr pparts))
(defun type-id-of-parameter-parts (pparts) (caddr pparts))
(defun string-of-parameter-parts (pparts) (cdddr pparts))

(defun parameter-marks-union (p marks)
  ;; marks union-c ensures no sharing of marks with original parameter.
  (instantiate-parameter (marks-union-c
			  (copy-extended-parameter-value
			   (mark-parameter-value (value-of-parameter-n p))) ; idiom to get marks of parameter:
			  marks)
			 (type-of-parameter p)))
