
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************

#+cltl2(in-system-package)
#-cltl2(in-package *system-package-name*)

;;;;
;;;; -docs- (mod com)
;;;;
;;;;
;;;;	Abstraction: 
;;;;
;;;;	<abstraction>		: abstraction[<definition>
;;;;					      <term{lhs}>
;;;;					      <term{rhs}>
;;;;					      <id{condition}> list
;;;;					      <expansion>]
;;;;
;;;;
;;;;	lhs-of-abstraction(<abstraction>)			: <term>
;;;;	rhs-of-abstraction(<abstraction>)			: <term>
;;;;	conditions-of-abstraction(<abstraction>)		: <id> list
;;;;	expansion-of-abstraction(<abstraction>)			: <expansion>
;;;;
;;;;
;;;;	LHS:
;;;;
;;;;	<lhs>			: <lhs-op>(<lhs-bound-term> list)
;;;;	<lhs-bound-term>	: <binding> list.variable{<target>:v}
;;;;							 ([variable{<binding>:v}] list)
;;;;	
;;;;	* The value of <binding> in the binding list and the variable injection should
;;;;	  be the same in the instance. Ie a,b.variable{A:v}(a; b) is proper, but
;;;;	  a,b.variable{A:v}(b; a) and a,b.variable{A:v}(c; d) are not.
;;;;	* Parameters must be real or meta valued.
;;;;	* No meta variables may be repeated in the parameter list.	
;;;;
;;;;	RHS:
;;;;	
;;;;	* Parameters must be real or meta valued. No parameters of type variable
;;;;	  may contain meta values.
;;;;	* Any variable occuring free must occur on lhs with same arity.
;;;;	* No second order variable occurence may be in the scope a similar first order
;;;;	  binding. Note that we have no notion of second order bindng in our syntax.
;;;;	* All subterms of a second order variable must have nil binding lists.
;;;;   
;;;;  -page- 
;;;;
;;;;	Selections :
;;;;	
;;;;	It may be that a user will not be pleased with an automatic choice
;;;;	of bound variables on expansion.  Suppose, for example, that an
;;;;	operator ALL of arity 0 1 is given the abstraction with body pi(A[];x.B[x]).
;;;;	The obvious way to expand ALL(N;y.y>0) would give pi(N;x.x>0);
;;;;	but it seems that many would demand that pi(N;y.y>0) result.  That is,
;;;;	they often want the binding variable of the expansion to be copied
;;;;	from the binding variables of the term to be expanded.
;;;; 
;;;;	Any binding variable occuring on the LHS in a binding position which
;;;;	also occurs on the RHS is selected.  Upon expansion, the occurences
;;;;	of a selected variable on the RHS should inherit the name of the 
;;;;	variable of the abstraction instance which occupies the binding
;;;;	position of the selected variable on the LHS.
;;;;	
;;;;	Eg, consider the all definition above.  x is a selected variable
;;;;	as it occurs in a binding position on the LHS and occurs on the RHS.
;;;;	Thus, all(Int; z.z) will expand to z:Int -> z as z occupies the 
;;;;	binding position of x on the LHS.
;;;;
;;;;	Incidentally, any variable occurring in a binding position on the LHS
;;;;	and occuring on the RHS, can not occur free on the RHS, and thus
;;;;	must also occurin a binding position on the RHS.
;;;;
;;;;  -page-
;;;;
;;;;	Clash :
;;;;
;;;;	We do not allow the definition of abstractions such that a term
;;;;	could be an instance of more than one abstraction def. We define
;;;;	a predicate which detects such ambiguity between two abstraction
;;;;	definitions.
;;;;
;;;;	Recall a term-sig of a term consist of the opid, an ordered list of
;;;;	parameter types, and an ordered list of arities. Any two abstractions
;;;;	whose terms sigs are not identical are trivially unambiguous.
;;;;
;;;;	Thus the test reduces to comparing the values of the parameter lists of
;;;;	terms with identical terms sigs. Assume that all parameters will be real
;;;;	valued or meta valued, ie no error or slot values. We claim if there
;;;;	exists a pair of real values where the values differ then ambiguity
;;;;	cannot arise. It is self evident that no instantiation can make
;;;;	different constants similar. Thus two abstractions with similar
;;;;	term-sigs are considered to clash unless when the parameter value lists
;;;;	are examined pairwise, there exists a pair of constant values which
;;;;	differ.
;;;;
;;;;	Can there exist a pair of abstractions which do clash but for which
;;;;	there exists no ambiguous instantiations?  Either, all meta variables of
;;;;	the value list for each abstraction are distinct, or one or both value
;;;;	lists contain repeated meta variables.
;;;;
;;;;	In the first case, examine the value lists pairwise, each pair then is
;;;;	either a pair of equal constants, a pair of meta variables, or a pair of
;;;;	a meta variable and a constant. Then for the meta variable pairs assign
;;;;	equivalent valuations and for the meta variable - constant pair assign
;;;;	the constant as valuation to produce identical instantiations.
;;;;
;;;;	In the case where meta variables may be repeated in a value list, it is
;;;;	possible to have clashes when no ambiguity is possible. For example, if
;;;;	one value list is ($a $a) and the other is (a b) where a does not equal
;;;;	b, then clash would be detected. However, no ambiguity is possible since
;;;;	in the first list in any instance both values will be the same, but in
;;;;	the second list the values differ.
;;;;
;;;;	Currently, we do not allow abstraction definitions with repeated meta
;;;;	variables.  Thus, no generality is lost. However, if at some point
;;;;	repeated meta variables are allowed than this algorithm needs to be
;;;;	reviewed.
;;;;
;;;;
;;;;	abstraction-clash-p(<term{lhs}> <term{key}>)			: <bool>
;;;;
;;;;  -page-
;;;;
;;;;	Match: detects when a term is an instance of an abstraction.
;;;;
;;;;	abstraction-match-p(<term{lhs}> <term{instance}>)		: <bool>
;;;;
;;;; -doct- (mod com data)
;;;;  -page-
;;;;
;;;;	Import/Export :
;;;;
;;;;	<abs-export-term>	: !abstraction{}(<term{lhs}>;
;;;;						 <term{rhs}>;
;;;;						 <condition> !condition_cons ilist)
;;;;	<condition>		: !condition{<id>:t}
;;;;
;;;; -doct- (mod com)
;;;;	export-abstraction(<abstraction>)			: <abs-export-term>
;;;;	import-abstraction(<abs-export-term>)			: <abstraction>
;;;;
;;;;  -page-
;;;;
;;;;	Abstraction Table : Definiton table of abstractions.
;;;;
;;;;	*** It is assumed that all definitions in a table originate in one library,
;;;;	*** and there is only one library per environment.
;;;;
;;;;	Abstraction tables are a resource within an environment.
;;;;
;;;;	abstraction-table (<process-id> <tag>)		: <definition-table{abstraction}>
;;;;
;;;;	abstraction-of-term(<term>)				: <abstraction> | nil
;;;;	abstraction-lookup(<oa>)				: <abstraction> | nil
;;;;
;;;;	expand-term(<term>)					: <term>
;;;;	conditions-of-abstraction-term(<term>)			: <id{condition}> list
;;;;	conditionally-expand-term(<term> <id{condition}> list)	: <term>
;;;;	 ** repeatedly expands top operator until no longer expandable under conditions.
;;;;	 ** an abstraction instance is conditionally expandable when it's condition
;;;;	 ** list contains a member of the expand conditions.
;;;;
;;;;	*** expanding an abstraction instance adds dependency of abstraction to
;;;;	*** the 'EXPAND dependency store.
;;;;	
;;;;  -page- hard
;;;;
;;;;	Quotes/Literals/Source Reduction :
;;;;
;;;;	Terms are used as data to the system, eg, terms can be used as source
;;;;	for ml.  We will refer to such terms as source terms. It is natural for
;;;;	source terms to contain abstraction instances meant to be expanded
;;;;	before the data is intepreted by the system.  This expansion phase will
;;;;	be called source reduction. Source reduction is parameterized by a list
;;;;	of conditions.  Source reduction expands all abstraction instances where
;;;;	the intesection of the abstraction's condition list with the source
;;;;	reduction's condition list is not null.
;;;;
;;;;	Source reduction satisfies the following invariant. If a term t is
;;;;	reduced to term t' through conditional expansion of abstraction
;;;;	instances under conditions C, then the source reduction of t under C is
;;;;	alpha equal to the source reduction of t' under C.
;;;;	
;;;;	In some contexts, we need to suppress the expansion of certain
;;;;	instances, ie one might wish a normally expandable operator of a source
;;;;	term to be data. Expansion can be averted by opquoting. An operator is
;;;;	quoted by adding a quote parameter to the front of the parameter list.
;;;;	A quote parameter is similar to the token parameter. The values of the
;;;;	parameters are ids. These ids will be interpreted as conditions.
;;;;
;;;;	Opquoting changes one operator into another thereby averting expansion
;;;;	of the former.  There is no restriction against defining abstractions
;;;;	with quote parameters.
;;;;
;;;;	opquote-term(<term> <id{quote}>)			: <term>
;;;;	opquoted-term-p (<term> <id{condition}> list)		: <bool>
;;;;	un-opquote-term(<term> <id-sexpr>)			: <term>
;;;;	 * removes quote from top operator.
;;;;	erase-opquotes(<term> <id-sexpr>)			: <term>
;;;;	 * recursively un-opquotes all operators of term.
;;;;	 * <id-sexpr>	: <id> | <id> list
;;;;
;;;;	source-reduce(<term> &optional (<id{condition}> list))	: <term>
;;;;	
;;;;  -page-
;;;;
;;;;	ML:
;;;; -doct- (mod com ml)
;;;;	 ** condition list, lhs, rhs, expansion defined.
;;;;
;;;;	<abstraction>	: (tok{condition} list # term {lhs} # term {rhs} # bool{expandable})
;;;;
;;;;	abstraction_lookup_by_name	: tok -> object_id
;;;;	name_of_abstraction		: object_id -> tok
;;;;
;;;;	obid_of_abstraction_term	: term -> object_id
;;;;
;;;;	abstraction_lookup		: object_id ->	(tok list # term # term # bool)
;;;;	abstraction_of_term		: term ->	(tok list # term # term # bool)
;;;;	 ** fails if term is not an abstraction instance.
;;;;
;;;;	expandable_abstraction_instance_p 	: term -> bool
;;;;
;;;;	source_reduce				: term -> tok{condition} list -> term
;;;;	  * recursively descends conditionally expanding abstractions.
;;;;
;;;;	Following applies to both abstractions and termofs.
;;;;
;;;;	expandable_instance_p			: term -> bool
;;;;
;;;;	expand_term				: term{def} -> term
;;;;
;;;;	<termof>	: (term {lhs} # term {rhs} # bool{expandable})
;;;;
;;;;	<definition>	: (tok{condition} list # term {lhs} # term {rhs} # bool{expandable})
;;;; -doce-

;;;;	
;;;;	Abstraction and termofs are similar and are subclasses of a more
;;;;	general expandable class. In general abstraction means abs object, 
;;;;	otherwise it can be abs or prf and might be called a definition.
;;;;	Note that this definition is a subclass of the implemention definition
;;;;	class which covers all data in a distributed table.
;;;;	
;;;;	
;;;;	
;;;;	
;;;;	
;;;;	
;;;;	




;;;;	RLE ??? TODO following is not correct, it is the interpreter looking at
;;;;	RLE ??? TODOreduced term which must unquote. In the case of ml quotes the quotes
;;;;	RLE ??? TODOmust be removed at term-> text time not at source reduce.
;;;;	RLE ??? TODOWe add a second pass to the source-reduction algorithm to remove one layer
;;;;	RLE ??? TODOof opquotes if the value of the quote is a condition of the source reduction.

;;;	RLE NAP2) Implement a term-quote simulation facility in the editor.
;;;	RLE NAP   The term-quote operator, by which I mean an operator that quotes a subterm,
;;;	RLE NAP   is very natural to use in simple cases, and so we expect many
;;;	RLE NAP   users would appreciate having it.
;;;	RLE NAP   But a term-quote operator is incompatible with the op-expansion invariant 
;;;	RLE NAP   above, and it would have to be introduced as a new concept into the
;;;	RLE NAP   basic syntax in order to define substitution, and yet it is not
;;;	RLE NAP   naturally extensible to handle mixtures of quoted and non-quoted operators.
;;;	RLE NAP   (Quote escape conventions are contrivances that are inferior to 
;;;	RLE NAP    simple op-quotation (says me).)
;;;	RLE NAP   So, I think the consequences of adding a term-quote operator to our
;;;	RLE NAP   syntax are worse than the benefit.
;;;	RLE NAP
;;;	RLE NAP   But your idea for simulating a term-quote operator via the editor seems
;;;	RLE NAP   managable to me. 
;;;	RLE NAP   This should be an optional editor feature set by the user.
;;;	RLE NAP   As I understand your idea, the user would not directly edit the
;;;	RLE NAP   term to be stored to the library, or passed by the editor to ML
;;;	RLE NAP   for evaluation. Instead, if this feature is used for the session,
;;;	RLE NAP   the term he is directly editing would be preprocessed into a term
;;;	RLE NAP   handled by the system as if the feature were not being used.
;;;	RLE NAP   And aptly, any term imported into the editor by being loaded from
;;;	RLE NAP   an object or returned by ML, would be converted to the desired form.
;;;	RLE NAP
;;;	RLE NAP   A term-quote operator !term-quote{?:s}(?) would be designated,
;;;	RLE NAP   and whenever a term is exported from the editor, each instance of the 
;;;	RLE NAP   !term-quote{fu:s}(A) which is not itself a subterm to another instance of 
;;;	RLE NAP   !term-quote{fu:s}(?) would be replaced by the term gotten by opquoting 
;;;	RLE NAP   every operator of its subterm with fu:quote.
;;;	RLE NAP   Whenever a term is imported into the editor, any term occurring within it
;;;	RLE NAP   all of whose operators are quoted with fu:quote, and not having as
;;;	RLE NAP   parent operator an operator staring with fu:quote, 
;;;	RLE NAP   would be replaced by !term-quote{fu:s}(A), where A is the result of
;;;	RLE NAP   removing the quotation from all the operators.
;;;	RLE NAP
;;;	RLE NAP   Remarks: 
;;;	RLE NAP   1) If we consider the composition of this editor preprocessing
;;;	RLE NAP      with the canoncial explanation of various system uses,
;;;	RLE NAP      then this new relation really does violate the op-expansion
;;;	RLE NAP      invariant for the canonical explanation.
;;;	RLE NAP      But it is still possible for users to share system objects
;;;	RLE NAP      since the system objects are always interpreted in the
;;;	RLE NAP      canonical way. This is possible since the user who chooses
;;;	RLE NAP      the term-quote feature of the editor doesn't directly edit these objects.
;;;	RLE NAP   2) This could be done by the user without system help by simply
;;;	RLE NAP      making his macros for saving and calling ML do the preprocessing
;;;	RLE NAP      before the save or call, and them restore the original;
;;;	RLE NAP      and his object-init hook and call-to-ml macros could be set
;;;	RLE NAP      up so that they stick in the term-quote operators and unquote
;;;	RLE NAP      the ops of the terms that are to be term-quoted.
;;;	RLE NAP      But this is probably too slow, so we implement the features directly.
;;;	RLE NAP   3) The main defect is that the connection between what is edited
;;;	RLE NAP      and what is stored is non-trivial, and may eventually confuse someone.
;;;	RLE NAP      But for people whose quoting needs are simple, and who will
;;;	RLE NAP      probably never need to know about the difference between what
;;;	RLE NAP      they edit and what they store, it's no problem.


;; RLE NAP  might be useful to have a version of source reduce which does not
;; RLE NAP implicitly do unconditionals. Then have edit condition which blows away
;; RLE NAP things like marks and user defined edit tools like cover ala sfa.
;; RLE NAP Also would be nice to change many unconditional instances to more specific conditions.

;; RLE NAP add warning if abs defined with quote parameter.

;;;
;;;	 Abstraction terms.
;;;


;; RLE TODO might be better to write match clash abstractly using vist func???


;;   Assumes:
;;	- args are proper. Ie parms are abstraction meta or constant values (and no marks!).
;;	- term-sigs are equal.
;;   assumes basis is similar, ie same id and parameters and arities.
;;     - basis does not imply same opids.
(defun abstraction-clash-p (lhs key)
  (and (eql (id-of-term lhs) (id-of-term key))
       (not (exists-p #'(lambda (lhs-parameter key-parameter)
			  (let ((lhs-value (value-of-parameter-n lhs-parameter))
				(key-value (value-of-parameter-n key-parameter)))

			    (cond
			      ((level-expression-parameter-p lhs-parameter)
			       (and (not (level-variable-p lhs-value))
				    (not (level-variable-p key-value))
				    (not (equal-parameter-values-p lhs-value key-value
								   (type-of-parameter lhs-parameter)))))
		    
			      (t (and (not (abstraction-meta-variable-id-p lhs-value))
				      (not (abstraction-meta-variable-id-p key-value))
				      (not (equal-parameter-values-p lhs-value key-value
								     (type-of-parameter lhs-parameter))))))))
		      (parameters-of-term lhs)
		      (parameters-of-term key)))))


(defun abstraction-match-parameter-p (key-parameter ist-parameter)
  (let ((key-value (value-of-parameter-n key-parameter))
	(ist-value (value-of-parameter ist-parameter))
	(type (type-of-parameter key-parameter)))
    (cond
      ((level-expression-type-p type)
       (or (level-variable-p key-value)
	   (and (real-parameter-value-p ist-value type)
		(equal-parameter-values-p key-value ist-value type))))
      (t (or (abstraction-meta-variable-id-p key-value)
	     (and (real-parameter-value-p ist-value type)
		  (equal-parameter-values-p key-value ist-value type)))))))

;;   Assumes
;;	- key arg is proper.
(defun abstraction-match-p (key instance)
  ;; tixt-table treats opid as parm so we need to check for eql opids
  ;; as table may not distinquish (ie clash).
  (and (eql (id-of-term key) (id-of-term instance))
       (forall-p #'abstraction-match-parameter-p
		 (parameters-of-term key)
		 (parameters-of-term instance))))



;;;
;;;	Abstraction defs.
;;;

;; expansion-conditionals -> conditions.

(defstruct (abstraction-super (:include definition))
  (lhs nil)			; a "most-general" term instance of the abstraction.
  (rhs nil)			; second-order substitution matrix.
  (expansion nil)
  (conditions nil)
  (other-models nil)
  (properties nil))


(defun conditions-of-abstraction (abs) (abstraction-super-conditions abs))
(defun lhs-of-abstraction (abs) (abstraction-super-lhs abs))
(defun rhs-of-abstraction (abs) (abstraction-super-rhs abs))
(defun expansion-of-abstraction (abs) (abstraction-super-expansion abs))
(defun other-models-of-abstraction (abs) (abstraction-super-other-models abs))

(defstruct (abstraction (:include abstraction-super)))
  
(defun abstraction-property-get (name def)
  (assoc name (abstraction-properties def)))

(defun abstraction-property-set (name prop def)
  (setf (abstraction-properties def)
	(acons name prop
	       (delete name (abstraction-properties def) :key #'car)))
  prop)

(defun abstraction-property-get-or-set (p def)
  (let ((a (abstraction-property-get (car p) def)))
    (if a
	(cdr a)
	(abstraction-property-set (car a) (funcall (cdr p) def) def))))

(defun abstraction-property-delete (name def)
  (setf (abstraction-properties def)
	(delete name (abstraction-properties def) :key #'car)))



;; RLE NAP may need to define null abstraction for itext-term with conjoin condition.


;;;;	<expansion>	: expansion['trivial]
;;;;			| expansion['fos]
;;;;			| expansion['sos <selection> list]
;;;;			| expansion['simple <term{replace}>]
;;;;			| expansion['variable-subterms]


(defstruct expansion
  dependency-tag
  etype
  other)


(defun new-expansion (type &optional other tag)
  (make-expansion :etype type :other other :dependency-tag (or tag 'abstraction)))

(defun dependency-tag-of-expansion (exp) (expansion-dependency-tag exp))
(defun type-of-expansion (exp) (expansion-etype exp))

(defun selections-of-sos-expansion (exp) (expansion-other exp))
(defun replace-of-replace-expansion (exp) (expansion-other exp))
(defun tags-of-error-expansion (exp) (expansion-other exp))

;; Assumes lhs, rhs proper wrt abstraction definition.
(defun find-selections (lhs rhs)
  (let ((selections nil))
    (with-variable-minor-invocation
	(labels
	    ((mark-bindings (term)
	       (mapc #'(lambda (bound-term)
			 (mapc #'set-variable-minor-use
			       (bindings-of-bound-term-n bound-term))
			 (mark-bindings (term-of-bound-term bound-term)))
		     (bound-terms-of-term term))))

	  (mark-bindings rhs)
	  
	  (dotimeslist (i bt (bound-terms-of-term lhs))
	    (dotimeslist (j binding (bindings-of-bound-term bt))
	      (when (variable-minor-use-p binding)
		(push (make-variable-selection :id binding
					       :term-index i
					       :binding-index j)
		      selections))))))

    (nreverse selections)))

(define-primitive |!null_abstraction|)

(defun make-abstraction-expansion (lhs rhs)
  (cond
    ((inull-abstraction-term-p rhs) nil)
    ;;((variable-subterms-abstraction-p lhs rhs)
    ;; (make-expansion 'variable-subterms))
    ((trivial-abstraction-p lhs rhs)
     (new-expansion 'trivial))
    ((parameter-abstraction-p lhs rhs)
     (new-expansion 'parameter))
    ((simple-abstraction-p lhs rhs)
     (new-expansion 'replace (make-replacement-matrix lhs rhs)))
    ((first-order-abstraction-p lhs rhs)
     (new-expansion 'fos))
    (t (new-expansion 'sos (find-selections lhs rhs)))))



(define-primitive |!abstraction| () (conditions lhs rhs))



(defun provide-abstraction-definition (def)
  (let* ((substance (ephemeral-substance-of-definition def 'substance))
	(aterm (term-of-substance substance)))

    (unless (iabstraction-term-p aterm)
      (raise-error (error-message '(abstraction import) aterm)))

    (let ((lhs (lhs-of-iabstraction-term aterm))
	  (rhs (rhs-of-iabstraction-term aterm)))
      (setf (definition-name def) (name-property-of-substance substance)
	    (abstraction-keys def) (list lhs)
	    (abstraction-lhs def) lhs
	    (abstraction-rhs def) rhs
	    (abstraction-conditions def) (term-to-condition-list
					  (conditions-of-iabstraction-term aterm))
	    (abstraction-expansion def) (make-abstraction-expansion lhs rhs)))
    def))
	
;; It might be a win to wrap lhs,rhs,conditions, and expansion in another struct 
;; to be demand loaded, then unused abstractions will not occupy space.
;; also those used near in time may end up on same swap pages.

(defun import-abstraction (term)
  (provide-abstraction-definition
   (make-abstraction :substance (term-to-data term))))

#|
(defun import-abstraction-super (term)
  (let ((data (term-to-data term)))
    (if (eql (type-of-data data) 'proof-substance)
	(import-proof data)
	(import-abstraction data))))
|#

(defun import-statement-termof (term statement)
  (declare (ignore term))

  (let ((substance (substance-of-definition statement 'statement-substance)))
    (let ((ext (extract-of-statement-substance substance)))
      (import-termof statement
		     (lemma-of-statement statement)
		     ext
		     (statement-name statement))
  
      statement)))




;;;
;;;	Abstraction tables.
;;;

;;;; abstraction table will contain abstractions and termofs.
;;;; abstraction-super is super class of abstraction and termof.

;;;;	could have seperate oid tables and a common term table where 
;;;;	termtable updates driven by insert/delete/commit/undo hooks
;;;;	of oid tables, but fttb single abs/prf table.
;;;;	

(defun abstraction-table (stamp tag)
  (name-table stamp (list 'abstraction tag) t
	      :meta-parameter-p #'(lambda (p)
				    (if (level-expression-parameter-p p)
					(level-variable-p (value-of-parameter p))
					(abstraction-meta-parameter-p p)))
	      :key-match-f #'abstraction-match-p
	      :keys-f #'(lambda (abs) (cons (lhs-of-abstraction abs) (other-models-of-abstraction abs)))
	      :key-clash-f #'abstraction-clash-p

	      :import-f #'import-abstraction
	      :import-aux-f #'import-statement-termof

	      :visibility (new-table-visibility `|visible_abstractions| "get_visible_abstractions")
	      ))

(defun allocate-abstractions (stamp tag)
  (abstraction-table stamp tag))


#+dontdoit
(defun abstraction-table (stamp tag)
  (define-definition-table
      stamp
      (list 'abstraction tag)
    t ;; nil ;; ordered-p
    :meta-parameter-p #'abstraction-meta-parameter-p
    :key-match-f #'abstraction-match-p
    :keys-f #'(lambda (abs) (cons (lhs-of-abstraction abs) (other-models-of-abstraction abs)))
    :key-clash-f #'abstraction-clash-p

    ;; todo needs to import both abstractions and termofs?
    :import-f #'import-abstraction-super
    ))



;;;;	
;;;;	
;;;;	
;;;;	
(defvar *multi-arity-abs-match-compatibility* nil)

(defun abstraction-of-term (term &optional dont-note)

  (let (;;(abs (mark-value term 'abstraction-definition))
	(abstractions (resource 'abstractions)))

    ;;(or (definition-valid-p abs abstractions)

    (let ((abs (definition-lookup abstractions term dont-note #'abstraction-visible-p)))

      ;;(setf -abs abs -term term) (break "aot")
      ;; following hack finds special multi-arity abstraction.
      ;; I recall allowing trivial abstractions to be defined with variable number of 0 arity bound terms
      ;; however I cannot find use at the moment.
      (unless (or (not *multi-arity-abs-match-compatibility*) abs)
	(when (forall-p #'(lambda (bt) (null (bindings-of-bound-term bt))) (bound-terms-of-term term))
	  (setf abs (definition-lookup abstractions
			(instantiate-term (operator-of-term term)
					  (list (instantiate-bound-term (ivoid-term))))))))

      ;; (mark term 'abstraction-definition abs)

      (when (typep abs 'abstraction)

	;;(setf -abs abs) (break "aot")

	abs))

    ;;)

    ))


(defun require-termof (oid)
  (funmlcall (ml-text "require_termof ") oid))

(defvar *check-term-lookup-visibilility-p* t)

(defun abstraction-super-of-term (term &optional dont-note forcep) 
  (let ((abstractions (resource 'abstractions)))
    (let ((def (definition-lookup abstractions term dont-note
				  (when *check-term-lookup-visibilility-p*
				    #'abstraction-visible-p))))
      (or def
	  (when forcep
	    (when (is-termof-id term)
	      (require-termof (oid-of-termof-term term))
	      (definition-lookup abstractions term dont-note) ))))))


(defun abstraction-lookup (oid)
  (let ((def (definition-lookup-by-oid (resource 'abstractions) oid)))
    (if (typep def 'abstraction)
	def
	(raise-error (oid-error-message (list oid) '(abstraction lookup oid not))))))


(defun conditions-of-abstraction-term (term)
  (let ((abstraction (abstraction-super-of-term term)))
    (when abstraction
      (conditions-of-abstraction abstraction))))


(defun abstraction-intertable-clash-tables () (list (resource 'proofs)))



;;;
;;; expand
;;;

(defun expand-term (term)

  ;;(format t "ExpandTerm ~a~%" (id-of-term term))

  (let* ((abs (abstraction-super-of-term term t))
	 (exp (when abs (expansion-of-abstraction abs))))

    (if exp
	(expand-term-aux term abs exp)
	term)))


(defun compute-expand-p (oid dir obids)
  (or (and (null dir) (null obids))
      (let ((p (member oid obids :test #'equal-oids-p)))
	(if dir
	    p
	    (not p))))))

(defun maybe-expand-term (dir obids p term)

  ;;(format t "ExpandTerm ~a~%" (id-of-term term))
  ;;(setf -term term -dir dir -obids obids -p p) (break "mep")
  (let ((abs (abstraction-super-of-term term t)))
    (if (and abs (compute-expand-p (oid-of-definition abs) dir obids) )
	(let ((exp (when abs (expansion-of-abstraction abs))))

	  (if exp
	      (if (and p (abstraction-property-get-or-set p abs))
		  term
		  (expand-term-aux term abs exp))
	      term))
	term)))


(defun ref-environment-abstractions-index ()
  (if (eql 'not-initialized *ref-environment-abstractions-index*)
      (setf *ref-environment-abstractions-index*
	    (update-definition-table-visibility (resource 'abstractions)))
      *ref-environment-abstractions-index*))

(defun abstraction-visible-p (def)
  (if (typep def 'statement)
      (statement-visible-p def (get-definition-table-visibility (resource 'statements)))
      (definition-visible-p def (ref-environment-abstractions-index))))

(defun abstraction-reference (def &optional exp)

  ;; TODO : instead of failing. Abs lookup should not return abs if abs not visible.
  ;;  also, allow clashing abstractions as long as not simultaneously viewable.
  ;;  ie, lookup returns list of abstractions, then choose one visible.
  ;;   define function to test for clashing abs in list to allow clash to be checked among visible.
  ;;   possibly require check whenever abs ref_state updated.
  (unless (abstraction-visible-p def)
    ;;(break "vn")
    (raise-error (oid-error-message (list (oid-of-definition def)) '(abstraction reference visible not))))

  (dependency-note-reference (if exp
				 (dependency-tag-of-expansion exp)
				 'expand-term)
			     (dependency-of-definition def)))

(defunml (|expand_an_abstraction| (oid term) :error-wrap-p nil)
    (object_id -> (term -> term))
  (let ((abs (abstraction-lookup oid)))
    (if (and abs (abstraction-match-p (lhs-of-abstraction abs) term))
	(expand-term-aux term abs (expansion-of-abstraction abs))
	term)))


(defun expand-abstraction (term)
  (let* ((abs (abstraction-of-term term t))
	 (exp (when abs (expansion-of-abstraction abs))))

    (if exp
	(expand-term-aux term abs exp)
	term)))

(defun make-abstraction-parameter-subs (model-parameters instance-parameters)
  (mapcan #'(lambda (mparm iparm)
	      (let ((value (value-of-parameter mparm)))
		(cond
		  ((level-expression-parameter-p mparm)
		   (when (level-variable-p value)
		     (list (cons value iparm))))
		  (t (when (abstraction-meta-variable-id-p value)
		       (list (cons value iparm)))))))
	  model-parameters
	  instance-parameters))


(defun expand-term-aux (term abs exp)

  ;;(setf -term term -abs abs -exp exp) (break "eta")
  (if (and nil (typep abs 'statement))
      (when (typep abs 'statement);; kludge alert. seems some more general inclusive method is possible.
	(check-ref-validity (ref-dag-of-proof abs))
	(dependency-note-reference (dependency-tag-of-expansion exp)
				   (dependency-of-definition abs)))
      (abstraction-reference abs exp))


  (case (type-of-expansion exp)

    (trivial (instantiate-term (operator-of-term (rhs-of-abstraction abs))
			       (bound-terms-of-term term)))

    (parameter (substitute-parameters-in-term
		(rhs-of-abstraction abs)
		(make-abstraction-parameter-subs (parameters-of-term (lhs-of-abstraction abs))
						 (parameters-of-term term))))

    (replace (expansion-replace term
				(replace-of-replace-expansion exp))) 

    (fos (expansion-fos term
			(lhs-of-abstraction abs)
			(rhs-of-abstraction abs)))

    (sos (expansion-sos term
			(lhs-of-abstraction abs)
			(rhs-of-abstraction abs)
			(selections-of-sos-expansion exp)))

    ;;(variable-subterms (expansion-variable-subterms term
    ;;						    (lhs-of-abstraction abs)
    ;;						    (rhs-of-abstraction abs)))
	
    (error (raise-error (error-message (tags-of-error-expansion exp) term)))

    (otherwise (system-error (error-message '(expand-term))))))


;; RLE NAP need to prevent definition of abstraction for !text_term_literal 
;; RLE NAP or at least warn of consequences.
(define-primitive |!call_ml| () (expression))

(defun conditionally-expand-term (term conditions)
  ;; this seems like a dangerous idea
  (if (and (iexpression-term-p term)
	   (transaction-expression-implicit-eval-ok term))
      (let ((r (environment-eval term)))
	;;(setf -r r) (break "cett")
	(or (interpret-result r)
	    (progn (setf -r r -term term)
		   (break "cet"))))
      (let* ((abs (abstraction-super-of-term term t))
	     (exp (when abs (expansion-of-abstraction abs))))
    
	;;(when (and abs (not (eql '|inewline| (abstraction-name abs))))
	;;  (setf -abs abs -cterm term -conditions conditions) (break "cet"))
	(if (and exp
		 (or (exists-p #'(lambda (condition) (member condition conditions))
			       (conditions-of-abstraction abs))
		     (member 'always conditions)))
	    (conditionally-expand-term (expand-term-aux term abs exp) conditions)
	    term))))

(defun conditionally-expand-term-once (term conditions)
  (let* ((abs (abstraction-super-of-term term t))
	 (exp (when abs (expansion-of-abstraction abs))))

    (if (and exp
	     (or (exists-p #'(lambda (condition) (member condition conditions))
			   (conditions-of-abstraction abs))
		 (member 'always conditions)))
	(expand-term-aux term abs exp)
	term)))
	

;;;;	
;;;;	!label abstractions need to be expanded during label abstraction definition
;;;;	this would be true for any ephemeral edit abstractions. Thus they must be  
;;;;	hardcoded. 
;;;;	
;;;;

(define-primitive |!comment| () (comment term))

(defun reduce-hard-coded (term)
  (case (id-of-term term)
    (|!label| (if (real-ilabel-term-p term)
		  (progn ;;(setf -term term) (break "rdc")
			 (term-of-bound-term (car (bound-terms-of-term term))))
		    term))
    (|!comment| (if (icomment-term-p term)
		    (term-of-icomment-term term)
		    term))

    (otherwise term)))
		
;; conservative-reduce
;;  - repeat expand top 

(defun source-reduce (term &optional (conditions nil) (consertive-p nil))
  ;;(setf a term b conditions) (break "sr")
  ;;(format t "SourceReduce ~a~%" conditions)
  (let ((conditions (if consertive-p
			conditions
			(cons 'unconditional conditions))))
    ;;(setf a term b conditions) (break "sr")
    (labels
	;; want to remove marks from parameters to prevent contamination
	;; of terms derived from source terms.
	((visit-op (op)
	   (let ((parms (parameters-of-operator op)))
	     (if (exists-p #'(lambda (p)
			       (mark-parameter-value-p (value-of-parameter p)))
			   parms)
		 (instantiate-operator (id-of-operator op)
				       (mapcar #'(lambda (p)
						   (let ((val (value-of-parameter p)))
						     (if (mark-parameter-value-p val)
							 (instantiate-parameter
							  (value-of-parameter-value val)
							  (type-of-parameter p))
							 p)))
					       parms))
		 op)))
		       
	 (visit (term)

	   ;;(when (iobject-address-term-p term)
	   ;;(with-ignore
	   ;;(note-iobject-address-reference (term-to-object-address term))))
		  
	   (let ((new-term (conditionally-expand-term (reduce-hard-coded term) conditions)))
	     ;;(setf a new-term b term) (break "sr")
	     (cond

	       ;; not really sure, if we avoid term literals we
	       ;; at least need to expand or filter labels.
	       ;; to avoid ml-text being expanded/recognized use scan:q.

	       ((and t (itext-term-literal-term-p new-term))
		;;(setf a new-term b term) (break "sr")
		new-term)

	       ;; RLE TODO : shouldn't be here. Only protects at top level??
	       ;;((or (iml-woargs-term-p term) (iml-term-p term))
	       ;;(instantiate-term (operator-of-term new-term)
	       ;;(cons (instantiate-bound-term (visit (text-of-iml-term new-term)))
	       ;;(cdr (bound-terms-of-term new-term)))))

	       (t;;(setf -nt new-term)  (break)
		(maybe-instantiate-term
		 new-term
		 (visit-op (un-opquote-operator (operator-of-term new-term) conditions))
		 (mapcar #'(lambda (bound-term)
			     (maybe-instantiate-bound-term
			      bound-term
			      (bindings-of-bound-term-n bound-term)
			      (visit (term-of-bound-term bound-term))))
			 (bound-terms-of-term new-term))))))))

      (if *ref-environment*
	  (without-ref-environment 
	   (if (resource-p 'abstractions)
	       (with-dependencies
		   (visit term))
	       term))
	  (if (resource-p 'abstractions)
	      (with-dependencies
		  (visit term))
	      term)))))


;; collects static references to abstraction occuring in a term.
(defun reference-static-abstractions (term)
  (with-dependency-environment
    (term-walk term
	       #'(lambda (term)
		   ;; force abs reference:
		   (abstraction-of-term term)
		   nil))
    (environment-dependencies-collected-term)))


(defunml (|abs_reference_static| (reterm term))
  (term -> (term -> term))
  (if (ivoid-term-p reterm)
      (reference-static-abstractions term)
      (with-make-ref-environment (#'make-explicit-ref-environment reterm)
	(with-reference-environment-term reterm
	  (reference-static-abstractions term)))))

#|
(defunml (|abs_reference_static| (term))
  (term -> term)
  (reference-static-abstractions term))
|#

;;;
;;;	Expansion optimizations:
;;;	


;;;; RLE NAP document and rationalize sos optimizations.

;; abstraction categories.

;; replace-op : lhs - rhs equivalent except op
;; replace-op+ : ditto but rhs has extra dummy variables. add dummy binding
;; subst-op : subst into and then replace op.
;; first-order-subst : no second-order variables and not replace-op.
;; replace : all rhs second order occurences are simple replacements with same variable.
;;   more general case of replace-op.

;; trivial : replace op - substitution into op not possible.
;; replace : replace second-order-occurrences(implicitly first order as well)
;;   - no capture possible - may add dummy bindings - substititutes into op.
;; first-order : no second order variables, substition into op, capture possible.
;; simple-second-order : capture possible within matrix but not instances.
;; second-order : everything.

;; trival : replace operator
(defun trivial-abstraction-p (lhs rhs)
  (and (null (parameters-of-term lhs))
       (= (length (bound-terms-of-term lhs))
	  (length (bound-terms-of-term rhs)))
       (every #'(lambda (bound-term-a bound-term-b)
		  (and (= (length (bindings-of-bound-term-n bound-term-a))
			  (length (bindings-of-bound-term-n bound-term-b)))
		       (every #'(lambda (binding-a binding-b)
				  (eql binding-a binding-b))
			      (bindings-of-bound-term-r bound-term-a)
			      (bindings-of-bound-term-r bound-term-b))
		       (equal-terms-p (term-of-bound-term bound-term-a)
				      (term-of-bound-term bound-term-b))))
	      (bound-terms-of-term lhs)
	      (bound-terms-of-term rhs))))


(defun parameter-abstraction-p (lhs rhs)
  (declare (ignore rhs))
  (and (null (bound-terms-of-term lhs))
       (not (some #'(lambda (parm)
		      (and (variable-parameter-p parm)
			   (abstraction-meta-parameter-p parm)))
		  (parameters-of-operator (operator-of-term lhs))))))


;; no possiblility-of capture
;;  - first note that variables may not occur free on rhs.
;;    bound-terms which are first-order variables may occur free on rhs.
;;    Thus we reject any def where a first order variable occurs as a binding.
;;  - Every second-order occurence on rhs is identical to occurence on lhs.
;;    ie B[x] is B[x] and not B[y] or B[foo(x)].
;;    as a consequence, all variables are selected.
;;  - there are no bindings of soo on rhs except for bindings of the selected
;;   variables. ie x.B[x] ok, but b.B[x] not ok.
;;  - no two selected variables are the same ie foo(x.A[x];x.B[x]) is not ok.


(defun simple-abstraction-p (lhs rhs)
  (let ((collection nil)
	(lsoos (mapcar #'term-of-bound-term-f (bound-terms-of-term lhs))))
    (labels
	((collect (term bindings)
	   (if (variable-term-p term)
	       (push (cons bindings term) collection)
	       (mapc #'(lambda (bound-term)
			 (collect (term-of-bound-term bound-term)
				  (union-vars (bindings-of-bound-term-r bound-term)
					      bindings)))
		     (bound-terms-of-term term))))

	 (duplicate-selections-p ()
	   (let ((selections nil))
	     (some #'(lambda (bound-term)
		       (some #'(lambda (binding)
				 (if (member binding selections)
				     t
				     (progn (push binding selections) nil)))
			     (bindings-of-bound-term-r bound-term)))
		   (bound-terms-of-term lhs))))
				     
	 (simple-occurence-p (soo)
	   ;;(setf e lsoos) (break)
	   (let ((lsoo (some #'(lambda (lsoo)
				 (compare-terms-p (cdr soo) lsoo))
			     lsoos)))
	     (and lsoo
		  (every #'dummy-variable-id-p
			 (diff-vars (car soo) (free-vars (cdr soo))))))))

      (and (not (duplicate-selections-p))
	   (not (some #'(lambda (parm)
			  (and (variable-parameter-p parm)
			       (abstraction-meta-parameter-p parm)))
		      (parameters-of-operator (operator-of-term lhs))))
	   (let ((first-order-vars (mapcan #'(lambda (bt)
					       (when (variable-p (term-of-bound-term bt))
						 (list (id-of-variable-term (term-of-bound-term bt)))))
					   (bound-terms-of-term lhs)))
		 (bindings (mapcan #'(lambda (bt) (copy-list (bindings-of-bound-term-r bt)))
				   (bound-terms-of-term lhs))))
	     (null (intersect-vars first-order-vars bindings)))
	   (every #'simple-occurence-p (progn (collect rhs nil)
					      collection))))))

			

(define-primitive |!replace| ((token . id)))

(defun make-replacement-matrix (lhs rhs)
  (let ((replacements (mapcar #'(lambda (bound-term)
				 (cons
				  (ireplace-term (gentemp))
				  (term-of-bound-term bound-term)))
			     (bound-terms-of-term lhs)	)))
    (labels ((visit (term)
	       (let ((replacement (when (variable-term-p term)
				     (some #'(lambda (replacement)
					       (when (compare-terms-p (cdr replacement)
								      term)
						 (car replacement)))
					   replacements))))
		 (or replacement
		     (instantiate-term
		      (operator-of-term term)
		      (mapcar #'(lambda (bound-term)
				  (instantiate-bound-term
				   (visit (term-of-bound-term bound-term))
				   (bindings-of-bound-term-r bound-term)))
			      (bound-terms-of-term term)))))))

      (cons (parameters-of-term lhs)
	    (cons (mapcan #'(lambda (bound-term)
			      (copy-list (bindings-of-bound-term-r bound-term)))
			  (bound-terms-of-term lhs))
		  (cons (mapcar #'(lambda (replacement)
				    (id-of-ireplace-term (car replacement)))
				replacements)
			(visit rhs)))))))

;; need to add op subst.
(defun expansion-replace (instance replacement-matrix)
  (let ((parameter-substs (make-abstraction-parameter-subs (car replacement-matrix)
							   (parameters-of-term instance)))
	(binding-replacements (pairlis (cadr replacement-matrix)
				       (mapcan #'(lambda (bound-term)
						   (copy-list
						    (bindings-of-bound-term-n bound-term)))
					       (bound-terms-of-term instance))))
	(replacements (mapcar #'(lambda (replace-id bound-term)
				  (cons
				   replace-id
				   (term-of-bound-term bound-term)))
			      (caddr replacement-matrix)
			      (bound-terms-of-term instance))))

    ;;(setf -a parameter-substs -b replacements -instance instance -replacement-matrix replacement-matrix) (break "er")
    (labels ((visit (term)
	       (if (ireplace-term-p term)
		   (find-first #'(lambda (replacement)
				   (when (eql (car replacement)
					      (id-of-ireplace-term term))
				     (cdr replacement)))
			       replacements)
		   (progn
		     (abstraction-meta-variable-term-r term)
		     (maybe-instantiate-term
		      term
		      (if parameter-substs
			  (substitute-in-operator (operator-of-term term)
						  parameter-substs)
			  (operator-of-term term))
		      (mapcar #'visit-bound-term
			      (bound-terms-of-term term))))))
	     (visit-bound-term (bound-term)
	       (maybe-instantiate-bound-term
		bound-term
		(mapcar #'(lambda (binding)
			    (let* ((binding (value-of-parameter-value binding))
				   (replacement (when (variable-id-p binding)
						  (cdr (assoc binding binding-replacements)))))
			      (or replacement binding)))
			(bindings-of-bound-term-n bound-term))
		(visit (term-of-bound-term bound-term)))))
      (visit (cdddr replacement-matrix)))))



;; all first order and possiblility-of capture
(defun first-order-abstraction-p (lhs rhs)
  (declare (ignore rhs))
  (every #'(lambda (bound-term)
	     (null (bindings-of-bound-term-n bound-term)))
	 (bound-terms-of-term lhs)))


(defun expansion-fos (instance lhs rhs)
  ;;(format t "sos-first ~a~%" (id-of-term instance))
  (let ((parameter-substs  (make-abstraction-parameter-subs (parameters-of-term lhs)
							    (parameters-of-term instance)))
	(subs (mapcar #'(lambda (ibound-term lbound-term)
			  (cons (id-of-variable-term (term-of-bound-term lbound-term))
				(term-of-bound-term ibound-term)))
		      (bound-terms-of-term instance)
		      (bound-terms-of-term lhs))))
    (substitute (substitute-parameters-in-term rhs parameter-substs)
		subs)))


;; expects lhs to be proper,
;;;;  ie no marks, slots, errors, etc. Satisfies all bound terms
;;  Expects (abstraction-match-p lhs instance) == t.
(defun expansion-sos (instance lhs rhs selections)
  (second-order-substitution-with-maintenance
   rhs
   (make-abstraction-parameter-subs (parameters-of-term lhs)
				    (parameters-of-term instance))
   (mapcar #'(lambda (lhs-bt instance-bt)
	       (cons (id-of-variable-term (term-of-bound-term lhs-bt))
		     instance-bt))
	   (bound-terms-of-term lhs)
	   (bound-terms-of-term instance))
   selections))



;;;;	A kludge to allow abstraction expansion with a variable number of subterms.
;;;;	
;;;;	foo{}(*) == goo{}(*)
;;;;	
;;;;	foo{}(*) will match and foo{}(a1; ...; an) and will produce goo{}(a1; ...; an)
;;;;
;;;;    For match to succeed a1 - an must have no bindings and abs model must have no
;;;;	abstraction meta variable parameters. foo must have a single subterm and it
;;;;	must have * as variable. goo can have other subterms and the * sequence will be
;;;;	cut in where it appears.

;;;;	this is being disabled as kludge not presently needed.
;;;;	If reactivated must add to abstraction clash to avoid in abs defined which could clash with variable
;;;;	version.

#+dontdoit
(defun variable-subterms-variable-term-p (term)
  (and (variable-term-p term)
       (eql (get-variable-id '|!*|) (id-of-variable-term term))))

#+dontdoit
(defun variable-subterms-abstraction-p (model matrix)
  (declare (ignore matrix))
  (and (forall-p #'(lambda (p) (not (abstraction-meta-parameter-p p)))
		 (parameters-of-term model))
       (let ((bound-terms (bound-terms-of-term model)))
	 (and bound-terms
	      (null (cdr bound-terms))
	      (null (bindings-of-bound-term (car bound-terms)))
	      (variable-subterms-variable-term-p (term-of-bound-term (car bound-terms)))))))
		 
#+dontdoit
(defun expansion-variable-subterms (instance model matrix)
  (declare (ignore model))
  (instantiate-term (operator-of-term matrix)
		    (let ((matrix-bound-terms (bound-terms-of-term matrix)))
		      (cond
			((null matrix-bound-terms) nil)
			((null (cdr matrix-bound-terms))
			 (bound-terms-of-term instance))
			(t (apply #'append (mapcar #'(lambda (bt)
						       (if (variable-subterms-variable-term-p
							    (term-of-bound-term bt))
							   (bound-terms-of-term instance)
							   (list bt)))
						   matrix-bound-terms)))))))
							     

;;;
;;; Opquotes:
;;;

;; this is not the term version of opquote for display.
;;(define-primitive |!quote| ((quote . quote)) (term))

;; this is the term version of opquote for display.
;;(define-primitive |!opquote| ((quote . quote)) (term))

(defun opquote-term (term tok)
  (let ((op (operator-of-term term)))
    (instantiate-term (instantiate-operator (id-of-operator op)
					    (cons (instantiate-parameter-r tok *quote-type*)
						  (parameters-of-operator op)))
		      (bound-terms-of-term term))))


(defun opquoted-term-p (term conditions)
  (let ((parameter (car (parameters-of-term term))))
    (and parameter
	 (quote-parameter-p parameter)
	 (if (symbolp conditions)
	     (eql (value-of-parameter-m parameter) conditions)
	     (member (value-of-parameter-m parameter) conditions)))))


(defun un-opquote-term (term conditions)
  (let* ((op (operator-of-term term))
	 (parameter (car (parameters-of-operator op))))
    (if (and parameter
	     (quote-parameter-p parameter)
	     (if (symbolp conditions)
		 (eql (value-of-parameter-m parameter) conditions)
		 (member (value-of-parameter-m parameter) conditions)))
	(instantiate-term (instantiate-operator (id-of-operator op)
						(cdr (parameters-of-operator op)))
			  (bound-terms-of-term term))
	term)))

(defun un-opquote-operator (op conditions)
  ;;(format t "UnOpquoteOp ~a ~a ~%" conditions op)
  (let* ((parameter (car (parameters-of-operator op))))
    (if (and parameter
	     (quote-parameter-p parameter)
	     (if (symbolp conditions)
		 (eql (value-of-parameter-m parameter) conditions)
		 (member (value-of-parameter-m parameter) conditions)))
	(instantiate-operator (id-of-operator op)
			      (cdr (parameters-of-operator op)))
	op)))


;; RLE NAP is this of any value?? un-opquote-term-r
(defun un-opquote-term-r (term conditions)
  (let* ((op (operator-of-term term))
	 (parameters (parameters-of-operator op))
	 (parameter (car parameters)))
    (if (and parameter
	     (quote-parameter-p parameter)
	     (if (symbolp conditions)
		 (eql (value-of-parameter-m parameter) conditions)
		 (member (value-of-parameter-m parameter) conditions)))
	(instantiate-term (instantiate-operator (id-of-operator op)
					      (cdr parameters))
			(bound-terms-of-term term))
	(raise-error (error-message '(un-opquote-r))))))
					  

(defun erase-quotes (term conditions)
  (labels
      ((list-visit (term)
	 (maybe-instantiate-term
	  term
	  (let* ((op (operator-of-term term))
		 (parameter (car (parameters-of-operator op))))
	    (if (and parameter
		     (quote-parameter-p parameter)
		     (member (value-of-parameter-m parameter) conditions))
		(instantiate-operator (id-of-operator op)
				      (cdr (parameters-of-operator op)))
		op))
	  (mapcar #'(lambda (bt)
		      (maybe-instantiate-bound-term bt
						    (bindings-of-bound-term bt)
						    (list-visit (term-of-bound-term bt))))
		  (bound-terms-of-term term))))
       
       (visit (term)
	 (maybe-instantiate-term
	  term
	  (let* ((op (operator-of-term term))
		 (parameter (car (parameters-of-operator op))))
	    (if (and parameter
		     (quote-parameter-p parameter)
		     (eql (value-of-parameter-m parameter) conditions))
		(instantiate-operator (id-of-operator op)
				      (cdr (parameters-of-operator op)))
		op))
	  (mapcar #'(lambda (bt)
		      (maybe-instantiate-bound-term bt
						    (bindings-of-bound-term bt)
						    (visit (term-of-bound-term bt))))
		  (bound-terms-of-term term)))))

    (unless (null term)
      (if (null conditions)
	  term
	  (if (consp conditions)
	      (list-visit term)
	      (visit term))))))




;;;
;;; ML
;;;



(defunml (|abstraction_of_term| (term) :error-wrap-p nil)
    (term -> ((tok list) |#| (term |#| (term |#| bool))))
  (let ((abs (abstraction-of-term term)))
    ;;(setf a abs) (break "aa")
    (if abs
	(list* (conditions-of-abstraction abs)
	       (lhs-of-abstraction abs)
	       (rhs-of-abstraction abs)
	       (and (expansion-of-abstraction abs) t))
	(breakout evaluation "abstraction_of_term"))))


(defunml (|obid_of_abstraction_term| (term) :error-wrap-p nil)
    (term -> object_id)

  (let ((abs (abstraction-of-term term)))
    ;;(setf a abs) (break "aa")
    (if abs
	(oid-of-definition abs)
	(breakout evaluation "obid_of_abstraction_term"))))


(defun abstraction-lookup-oid (name)
  (let ((abs nil))

    (let ((abss (name-table-lookup (resource 'abstractions)
				   name
				   (current-transaction-stamp))))
      (dolist (tabs abss)

	(when (and (typep tabs 'abstraction) (abstraction-visible-p tabs))
	  (when abs
	    (setf -abss abss)
	    (raise-error '(abstraction lookup name duplicate)))
	  (setf abs tabs)))
	      
      (unless abs
	(raise-error '(abstraction lookup name none)))

      (oid-of-definition abs))))

(defun abstractions-lookup-oid (name)
    (mapcar #'oid-of-definition
	    (name-table-lookup (resource 'abstractions)
			       name
			       (current-transaction-stamp))))

  
(defunml (|abstraction_lookup_by_name| (name))
    (tok ->  object_id)
  (abstraction-lookup-oid name))


(defunml (|abstraction_lookup| (oid))
    (object_id -> ((tok list) |#| (term |#| (term |#| bool))))

  (let ((abs (abstraction-lookup oid)))
    (list* (conditions-of-abstraction abs)
	   (lhs-of-abstraction abs)
	   (rhs-of-abstraction abs)
	   (and (expansion-of-abstraction abs) t))))


(defunml (|abstraction_match_p| (instance obid) :error-wrap-p nil)
    (term -> (object_id -> bool))

  (let ((r nil))
    (with-ignore ()
      (when (abstraction-match-p (lhs-of-abstraction (abstraction-lookup obid))
				 instance)
	(setf r t)))
    r))

(defunml (|name_of_abstraction| (oid))
    (object_id -> tok)

  (name-of-definition (abstraction-lookup oid)))


(defunml (|map_abstractions| (f))
    ((object_id -> unit) -> unit)

  (without-dependencies
   (definition-table-map (resource 'abstractions)
       (current-transaction-stamp)
     #'(lambda (oid def)
	 (declare (ignore oid))
	 (unless (typep def 'proof)
	   (ap f (oid-of-definition def))))))
    nil)




(defunml (|expandable_instance_p| (term) :error-wrap-p nil)
    (term -> bool)
  (let ((abs (abstraction-super-of-term term)))
    (and abs (expansion-of-abstraction abs) t)))

(defunml (|expandable_abstraction_instance_p| (term) :error-wrap-p nil)
    (term -> bool)


  (let ((abs (abstraction-of-term term)))
    (and abs
	 (abstraction-visible-p abs)
	 (expansion-of-abstraction abs)
	 t)))


(defunml (|expand_term| (term) :error-wrap-p nil)
    (term -> term)
  (expand-term term))


(defunml (|source_reduce_with_dependencies| (conditions term) :error-wrap-p nil)
    ((tok list) -> (term -> (term |#| term)))
  (with-dependency-environment
      (let ((rterm (source-reduce term conditions)))
	(cons rterm (environment-dependencies-collected-term)))))


(defunml (|source_reduce| (term conditions) :error-wrap-p nil)
    (term -> ((tok list) -> term))
  (source-reduce term conditions))



(defun abs-list ()
  (let ((acc nil))
    (definition-table-map (resource 'abstractions)
	(current-transaction-stamp)
      #'(lambda (key value) (push (cons key value) acc)))
    acc))
