%
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************
%

let equal_operators_p op1 op2 = 
 let (nid, nparms) = op1 in
  let  (oid, oparms) = op2 in
  (nid = oid) & (equal_lists_p equal_parameters nparms oparms)
;;

let split_list p l =
 letrec aux l =
  (if (null l) then (nil, nil)
  else if p (hd l) then (nil, l)
  else let (prefix, suffix) = aux (tl l) in
	(((hd l) . prefix), suffix)
   ) in

   aux l
;;     

let id_of_term t = let ((opid, parms), bts) = destruct_term t in opid;;

let false_bool_parameter = make_bool_parameter false;;
let true_bool_parameter = make_bool_parameter true;;

let object_id_parameter_p p = `oid` = type_of_parameter p;;

let null_token = string_to_tok "";;

let ivoid_term = make_term (`!void`, nil) nil;;
let ivoid_term_p t = 
 (let (id, []),[] = destruct_term t in (id = `!void`)) ? false
;;


letref debug_term = ivoid_term;;
letref debug_tok = `void`;;


let map_isexpr_to_list op f ilist =
 letrec aux ilist acc = 
  if (equal_operators_p op (operator_of_term ilist)) then
     (let (op, bts) = (destruct_term ilist) in
       if bts = [] then acc
       else let [([],car); ([],cdr)] = bts in
	(aux car (aux cdr acc)))
     else ((f ilist) . acc) in
 aux ilist []
;;

let map_isexpr_to_term_list op ilist =
 map_isexpr_to_list op (\p.p) ilist
;;

let map_to_ilist f op list =
 letrec aux l =
   if (l = nil)
      then (make_term op nil)
      else (make_term op [([],(f (hd l))); ([], aux (tl l))])
 in aux list
;;


let icons_op = (`!cons`, nil);;
let icons_term a b = make_term icons_op [([],a); ([],b)];;
let inil_term = make_term icons_op [];;

let ihead t = let ((id, []), [([],a); b]) = destruct_term t in a;;
let itail t = let ((id, []), [a; ([],b)]) = destruct_term t in b;;

let make_icons_term op a b = make_term op [([],a); ([],b)];;

let itok_cons_op = (`!tok_cons`, nil);;
let tok_icons_term a b = make_term itok_cons_op [([],a); ([],b)];;
let itok_nil_term = make_term itok_cons_op [];;

let inat_cons_op = (`!nat_cons`, nil);;
let nat_icons_term a b = make_term inat_cons_op [([],a); ([],b)];;
let inat_nil_term = make_term inat_cons_op [];;

let itext_term s = make_term (`!text`, [make_string_parameter s]) [];;
let itext_term_p t = `!text` = (id_of_term t);;
let itext_cons_op = (`!text_cons`, nil);;

let string_of_itext_term t = destruct_string_parameter (hd (parameters_of_term t));;						 
						 
let itext_term_literal t = make_term (`!text_term_literal`, []) [([],t)];;

let inewline_term = make_term (`!newline`,nil) nil;;						 
let strings_to_itext ss =
 map_to_ilist (\s. make_icons_term itext_cons_op (itext_term s) inewline_term)
  (itext_cons_op)
  ss
;;
						 
let tactic_tree_cons_op = (`tactic_tree`, nil);;

let inatural_term i = make_term (`!natural`, [(make_natural_parameter i)]) nil;;
let number_of_inatural_term t = destruct_natural_parameter (hd (parameters_of_term t));;

let itoken_term t = make_term (`!token`, [(make_token_parameter t)]) nil;;
let token_of_itoken_term t = destruct_token_parameter (hd (parameters_of_term t));;

let itokens_term toks = make_term (`!token`, map make_token_parameter toks) nil;;
let tokens_of_itokens_term t = map destruct_token_parameter (parameters_of_term t);;

let istring_term s = make_term (`!string`, [(make_string_parameter s)]) nil;;
let string_of_istring_term t = destruct_string_parameter (hd (parameters_of_term t));;

let istrings_term l = make_term (`!string`, map make_string_parameter l) nil;;
let strings_of_istring_term t = map destruct_string_parameter (parameters_of_term t);;

let ioid_term oid = make_term (`!oid`, [(make_object_id_parameter oid)]) nil;;
let oid_of_ioid_term t = destruct_object_id_parameter (hd (parameters_of_term t));;

let ivariable_term variable = make_term (`!variable`, [(make_variable_parameter variable)]) nil;;
let variable_of_ivariable_term t = destruct_variable_parameter (hd (parameters_of_term t));;

let ioids_term oids = make_term (`!oid`, map make_object_id_parameter oids) [];;
let oids_of_ioid_term t = map destruct_object_id_parameter (parameters_of_term t);;

let iterms_term = map_to_ilist (\x.x) icons_op;;

let ioid_cons_op = `!oid_cons`, nil;;
let ioid_cons_term = make_icons_term ioid_cons_op;;

let ibool_term b = make_term (`!bool`, [(make_bool_parameter b)]) nil;;
let bool_of_ibool_term t = destruct_bool_parameter (hd (parameters_of_term t));;

let ibool_true_term = ibool_term true;;
let ibool_false_term = ibool_term false;;

let isome_term a = make_term (`!some`, nil) [([],a)];;
let isome_term_p t = let ((id, []), bts) = destruct_term t in id = `!some`;;
let term_of_isome_term t = let ((id, []), [[], a]) = (destruct_term t) in a;;
let some_of_isome_term t = 
  ( inr (let ((id, []), [[],s]) = destruct_term t in s))
  ? inl ()
;;

let inone = make_term (`!none`,[]) [];; 

let mk_isome_term a = 
 if (isl a) then inone
 else isome_term (outr a)
;;


let isome_oid oid = 
 if isl oid then inone
 else make_term (`!some`, [make_object_id_parameter (outr oid)]) [];;

let isome_oid_p t = 
 let ((opid, [oparm]), []) = 
  destruct_term t in opid = `!some`
  & can destruct_object_id_parameter oparm;;

let some_of_isome_oid t = 
  ( inr (let ((id, [oparm]), []) = (destruct_term t) in
          destruct_object_id_parameter oparm))
  ? inl ()
;;


let iml_cons_op = (`!ml_text_cons`, []);;
let iml_cons_term a b = make_term iml_cons_op [([], a); ([], b)];;

let ilist_op = (`!list`, []);;
let ipair_op = (`!pair`, []);;
let ileft ipair =
  let (op, [([], a);([], b)]) = (destruct_term ipair) in a
;;
let iright ipair =
  let (op, [([], a);([], b)]) = (destruct_term ipair) in b
;;
let ipair_term l r = make_term ipair_op [[],l;[],r];;


let map_ilist_to_list op f ilist =
 letrec aux ilist acc = 
  if (equal_operators_p op (operator_of_term ilist)) then
     (let (op, bts) = (destruct_term ilist) in
       if bts = [] then acc
       else let [([],car); ([],cdr)] = bts in
	     ((f car) . (aux cdr acc)))
     else ((f ilist) . acc) in
 aux ilist []
;;


let imp_implementation_op parms = (`!metaprl_implementation`, parms);;
					       
let iannotation_cons_op = (`!annotation_cons`, nil);;

let iextract_not_term = make_term (`!extract`,nil) nil;;

let iproperty_term name value = make_term (`!property`, [make_token_parameter name]) [([],value)];;

let name_of_iproperty_term iprop =
 let ((id, parms), []) = destruct_term iprop in
  destruct_token_parameter (hd parms)
;;

let value_of_iproperty_term iprop =
  let (op, [[], a]) = (destruct_term iprop) in a
;;

let iplaceholder_term =
  make_term (`!placeholder`, []) []
;;

let default_placeholder_term = iplaceholder_term;;

let iplaceholder_term_p t = lex_equal_terms t iplaceholder_term;;	    

% index starting at 1 %
let replace_parameter i p term = 
 let ((opid, parms), bts) = destruct_term term in
   let (pre, post) = split (i - 1) parms in
     make_term (opid, (append pre (p . (if (nil = post) then nil else (tl post)))))
       bts
;;

% index starting at 1 %
let replace_subterm i subterm term = 
 let (op, bts) = destruct_term term in
   let (pre, post) = split (i - 1) bts in
     make_term op
       (append pre ((nil,subterm) . (if (nil = post) then nil else (tl post))))
;;

%
;;;;
;;;;	Wrapped terms : certain ops are invisible in some circumstances.
;;;;	
;;;;	
%					       

let wrapped_term_p t =
 
 (let ((id, parms), (([],sub) . bts)) = destruct_term t in
   if (parms = nil & id = `!condition_cut`)
      then true
      else (let p = hd parms in
             (id = `!wrapper` or id = `!label` or id = `!tag` or id = `!MARK`)))
 ? false
;;

letrec term_of_wrapped_term t =
 (if (wrapped_term_p t)
     then (let ((id, parms), (([],sub) . bts)) = destruct_term t in
	    (term_of_wrapped_term sub))
     else t)
 ? t
;;

let destruct_wrapped_term t = destruct_term (term_of_wrapped_term t);;

let first_oid_of_term term =
 let (id, parms), bts = destruct_wrapped_term term in
  destruct_object_id_parameter (find (\p. `oid` = type_of_parameter p) parms)
;;
let last_oid_of_term term =
 let (id, parms), bts = destruct_wrapped_term term in
  destruct_object_id_parameter (find (\p. `oid` = type_of_parameter p) (rev parms))
;;

let first_tok_of_term term =
 let (id, parms), bts = destruct_wrapped_term term in
  destruct_token_parameter (find (\p. `token` = type_of_parameter p) parms)
;;

let second_oid_of_term term =
 let (id, parms), bts = destruct_wrapped_term term in
  destruct_object_id_parameter (nth 2 (filter (\p. `oid` = type_of_parameter p) parms))
;;


let second_tok_of_term term =
 let (id, parms), bts = destruct_wrapped_term term in
  destruct_token_parameter (nth 2 (filter (\p. `token` = type_of_parameter p) parms))
;;

let first_string_of_term term =
 let (id, parms), bts = destruct_wrapped_term term in
  destruct_string_parameter (hd (filter (\p. `string` = type_of_parameter p) parms))
;;

let second_oid = second_oid_of_term;;	    
let first_oid = first_oid_of_term;;
let first_tok = first_tok_of_term;;
let first_string = first_string_of_term;;
let second_tok = second_tok_of_term;;
let last_oid = last_oid_of_term;;


let normalize_isexpr op isexpr =
 map_to_ilist (\x.x) op (map_isexpr_to_term_list op isexpr)
;;

let term_to_property_list t =
  map_isexpr_to_list icons_op 
		     (\iprop. let ((a, [type]), [([],prop)]) = destruct_term iprop in
			       ((destruct_token_parameter type), prop))
		     t
;;

let property_list_to_term l =
 map_to_ilist
	(\p. let (name, prop) = p in 
	 	iproperty_term name prop)
	icons_op
	l
;;

let search_isexpr op f g isexpr =
 letrec aux isexpr acc = 
  if (equal_operators_p op (operator_of_term isexpr)) then
     (let (op, bts) = (destruct_term isexpr) in
       if bts = [] then acc
       else let [([],car); ([],cdr)] = bts in
	(aux car (aux cdr acc)))
     else if (f isexpr) then ((g isexpr) . acc) else acc in
 aux isexpr []
;;

let visit_isexpr op f term =
  search_isexpr op (\t. f t; false) (\t.t) term; ()
;;
 
let inil_term_p cons_op term = 
  let (op, bts) = (destruct_term term) in
    if (equal_operators_p op icons_op)
	then if null bts
		then true
		else false
	else false
;;
     
let icons_term_p op term =
  (let (op, [([],car); ([],cdr)]) = (destruct_term term) in
     if (equal_operators_p op (operator_of_term term))
	then true
	else false)
   ? false
;;


let map_isexpr op f isexpr =
 letrec aux isexpr = 
  if (equal_operators_p op (operator_of_term isexpr)) then
     (let (op, bts) = (destruct_term isexpr) in
       (if bts = [] 
	then isexpr
        else (let [([],car); ([],cdr)] = bts in
		make_term op [([],(aux car)); ([],(aux cdr))])))
     else (f isexpr) in
 aux isexpr 
;;

% cons_tree : term . cons_tree list
%
let id x = x;;
			      
let term_at_tree_address op address term =
 letrec aux addr t =
   if null addr then (if (icons_term_p op t) then ihead t else t) 
   else aux (tl addr) (nth (hd addr) (map_ilist_to_list op id (itail t)))
 
 in aux address term
;;			      


let reverse_rotate_ilist op ilist =
 let l = map_isexpr_to_list op id ilist in
  let pre,suf = split ((length l) - 1) l in
   map_to_ilist id op (append suf pre)
;;


let rotate_ilist op ilist =
 let l = map_isexpr_to_list op id ilist in
  let pre,suf = split 1 l in
   map_to_ilist id op (append suf pre)
;;


let ddg_to_term ddg =
 map_to_ilist
   (\e. ipair_term
         (ioid_term (fst e))
         (ioids_term (snd e)))
   (icons_op)
    ddg
;;					       
let term_to_ddg =
 map_ilist_to_list icons_op
   (\t. oid_of_ioid_term (ileft t), oids_of_ioid_term (iright t))
;;
			      
%
;;;;	Remote closures:
;;;; -docs- (mod orb ml)
;;;;	
;;;;	To ease the development of inter-process functionality, the ability
;;;;	to dynamically marshall/unmarshall cross process calls is supported.
;;;;	
;;;;	
;;;;	<term-expression> 	: <text>
;;;;	  * term expands to an evaluable text expression.
;;;;
;;;;	<term-closure>		 : <text>, <term{arg}> list
;;;;	  * the arg list when unmarshalled will be args to the expression represented by the text.
;;;;	
;;;;	begin_ap		: string -> <term-closure>
;;;;	  
;;;;	In the remote process the expr represented by the <term-expression> will be evaluated.
;;;;    For example : begin_ap "\t s. [t; s]" evals to - : (* -> * -> * list)
;;;;
;;;;	There is a suite of functions defined to apply term-closures to arguments:
;;;;	  
;;;;	string_ap	: <term-closure> -> string -> <term-closure>
;;;;	token_ap	: <term-closure> -> tok -> <term-closure>
;;;;	nat_ap		: <term-closure> -> int -> <term-closure>
;;;;	oid_ap		: <term-closure> -> object_id -> <term-closure>
;;;;	oids_ap		: <term-closure> -> object_id list -> <term-closure>
;;;;	term_ap		: <term-closure> -> term -> <term-closure>
;;;;	
;;;;	Then token_ap (begin_ap "\t s. [t; s]") `t`			evals to - : tok -> tok list
;;;;	and  token_ap (token_ap (begin_ap "\t s. [t; s]") `t`) `s`)	evals to [`t`; `s`] : tok list
;;;;	
;;;;	Note that earlier args in the evaluable expression are nested deeper 
;;;;	in the marshalling expression.
;;;;
;;;;	
;;;;	Once comprehended, the task of defining new marshalling/unmarshalling functions is
;;;;	straightforward :
;;;;
;;;;	make_ap 	: string {unmarshall} -> (* -> term){unmarshall}
;;;;				-> <term-closure> -> * -> <term-closure>)
;;;;	  
;;;;	  unmarshall : text of remote unmarshalling function.
;;;;	  marshall : local marshalling function.
;;;;	
;;;;	The remote unmarshalling function for a type T must have type: ((term list -> T -> *) -> term list -> *)
;;;;	
;;;;	The remote Unmarshall_ap function can be used to construct an unmarshalling function from a
;;;;	function with type : term -> T.
;;;;
;;;;	Unmarshall_ap : ((term -> *) -> (term list -> * -> **) -> term list -> **)
;;;;	
;;;;	For example: 
;;;;
;;;;	let bool_ap = make_ap "Unmarshall_ap bool_of_ibool_term" ibool_term 
;;;;	Then bool_ap : <term-closure> -> bool -> <term-closure>
;;;;	
;;;;	The Unmarshall_ap function is not hidden as it is expected that its application will
;;;;	be compiled in at the remote process, ie :
;;;;	let Bool_ap = Unmarshall_ap bool_of_ibool_term;;
;;;;
;;;;	And then in the local process :
;;;;	let bool_ap = make_ap "Bool_ap" ibool_term;;
;;;;
;;;;  -page-
;;;;
;;;;	Remote evaluation:
;;;;
;;;;	Let's assume that we have a function to perform remote evaluation on a term
;;;;	closure which returns a term:
;;;;
;;;;	<remote-eval>	: <term-closure> -> term 
;;;;	
;;;;	There is a suite of functions for marshalling the remote value to a term
;;;;	and locally unmarshalling :
;;;;	
;;;;	make_oid_return		: <remote-eval> -> <term-closure> -> object_id
;;;;	make_string_return	: <remote-eval> -> <term-closure> -> string
;;;;	make_token_return	: <remote-eval> -> <term-closure> -> token
;;;;	make_bool_return	: <remote-eval> -> <term-closure> -> bool
;;;;	make_terms_return	: <remote-eval> -> <term-closure> -> term list
;;;;	
;;;;	
;;;;	make_return_eval : string -> (term -> T) -> <remote-eval> -> <term-closure> -> T
;;;;	
;;;;	For example
;;;;	let make_nat_return = make_return_eval "inatural_term" number_of_inatural_term
;;;;	  : <remote-eval> -> <term-closure> -> int
;;;;	
;;;;	This then could be combined with arg marshalling:
;;;;	
;;;;	remote_add' i j  = \re. make_nat_return re (nat_ap (nat_ap (begin_ap "\i j. i + j") i) j)
;;;;	  : <remote-eval> -> int -> int -> int
;;;;	
;;;;	Usually <remote-eval> is applied first:
;;;;	assume remote_eval : <term-closure> -> term 
;;;;
;;;;	make_eval_to_nat = make_nat_return remote_eval
;;;;	  : <term-closure> -> int
;;;;	
;;;;	remote_add i j  = make_eval_to_nat (nat_ap (nat_ap (begin_ap "\i j. i + j") i) j)
;;;;	  : int -> int -> int
;;;;	
;;;;  -page-
;;;;
;;;;	Compose : marshalling functions can be composed.
;;;;	
;;;;	X_ap <term-closure> : X -> <term-closure>
;;;;
;;;;	Y_ap o (X_ap (begin_ap "foo")) : X -> Y -> <term-closure>
;;;;	
;;;;	Beware the arg order.
;;;;
;;;;	
;;;;	There are utilities to build remote eval functions :
;;;;	
;;;;	orb_eval_args_to_term : tok{fan} -> term{description} -> <term-closure> -> term
;;;;	  - description : each connected environment supplies a description.
;;;;		only envirionments whose descriptions match the arg are applicable.
;;;;	  - fan : chooses among applicable connected environments.
;;;;	      * one : must be only one applicable.
;;;;	      * any : first found.
;;;;	      * all 	
;;;;
;;;;	let lib_eval_to_term = orb_eval_args_to_term `ONE` 'LIB';;
;;;;	
;;;;	Then,
;;;;	lib_eval_add i j = make_nat_return lib_eval_to_term (nat_ap (nat_ap (begin_ap "\i j. i + j") i) j)
;;;;	  : int -> int -> int
;;;;	
;;;;	
;;;;	Occasionally, you do not want a return value:
;;;;	
;;;;	orb_eval_args : tok{fan} -> term{description} -> <term-closure> -> unit
;;;;	
;;;;	lib_eval = orb_eval_args `ONE` 'LIB';;
;;;;
;;;;	lib_set_flag b = lib_eval (ibool_ap (begin_ap "\b. flag := b") b)
;;;;	  : unit
;;;;	
;;;;	The print message from the remote evaluator is returned and printed locally.
;;;;	
;;;;  -page-
;;;;
;;;;	Example :	
;;;;	
;;;;	Thus in summary, a truly brave soul could at a toploop prompt, perform a remote
;;;;	evaluation of an expression with no predefined marshalling of args or return values:
;;;;	
;;;;	Assume lib_eval_to_term,
;;;;	assume lib has fu : (tok + unit) -> natural # natural
;;;;	and some marshalling primitives to marshall (tok + unit) to !some{}(0) or !none().
;;;;	
;;;;	make_return_eval
;;;;	  "\\p. icons_term (inatural_term (fst p)) (inatural_term (snd p)) "	
;;;;	  (\t. (number_of_inatural_term (ihead t), number_of_inatural_term (itail t)))
;;;;	  lib_eval_to_term
;;;;	  (make_ap "Unmarshall_ap (\\t. if not (isome_term_p t)
;;;;					  then inr ()
;;;;					  else inl (token_of_itoken_term (term_of_isome_term t)))"
;;;;	  	   (\t. (isome_term (itoken_term t))) (begin_ap "fu") `t`)
;;;;	
;;;;	Of course, one might argue that it is not a matter or courage but sanity.
;;;;	However, with some editor support this could be made to look quite sane.	
;;;;
;;;;	This could be a very useful feature when debugging a remote environment
;;;;	such as the refiner. This allows evaluation from the editor of expressions
;;;;	whose components are defined in the refiner on data from the editor. So
;;;;	you can pick apart your proof in the editor and then apply snippets of refiner
;;;;	code to the pieces. 
;;;;	
;;;;	These features are also useful for prototyping the functionality 
;;;;	to be defined in a static interface.
;;;;
;;;;  -page-
;;;;
;;;;	Posure : Portable term closure.
;;;;
;;;;    As a term closure consists of combinations of terms it is a simple
;;;;	matter to marshall a closure itself. This allows a remote environment
;;;;	to be passed a closure to be used to call another environment.
;;;;	For example, assume you have a sequent <S> in the editor and you want to 
;;;;	apply some refiner predicate <P> to it and a goal <G> from the library.
;;;;
;;;;	The term closure of the predicate and the edit sequent can be produced :
;;;;	(term_ap begin_ap "P" S)
;;;;
;;;;	Then call lib with a term-closure which picks up the lib goal and completes
;;;;	the call to the refiner:
;;;;
;;;;	lib_eval_to_bool (posure_ap (begin_ap "\po. ref_eval_to_bool (term_ap po <G>)")
;;;;			            (term_ap (begin_ap "<P>") <S>))
;;;;	
;;;;	
;;;;  -page-
;;;;
;;;;	caveats :
;;;;	  - \ needs to be escaped in strings in edit objects.
;;;;	  - no type-checking of expression until eval time
;;;;	     * recommend unit test prior to deployment.
;;;;	  - 4.2 some of these functions are reimplemented natively in 4.2 and some
;;;;	    are inherited (source) from 5.
;;;;
;;;;	There are some variants which expect terms instead of string to allow
;;;;	avoiding repeatedly instantiating a term for the same constant string.
;;;;
;;;;	make_ap_aux
;;;;	  : term -> (* -> term) -> <term-closure> -> * -> <term-closure>
;;;;	make_return_eval_aux
;;;;	  : term -> (term -> *) -> <remote-eval> -> <term-closure> -> *
;;;;	null_ap 		: <term> -> <term-closure>
;;;;	
;;;;	So you could define a global to hold the text term of a the remote call
;;;;	and use that as arg to these variants.
;;;;	
;;;;
;;;;	RMI/RPC ?
;;;;
;;;;	How does this differ from rmi/rpc?
;;;;	Evaled not compiled. Can compile marshalling functions.
;;;;	In rmi must compile stubs for all possible functions called.
;;;;
;;;;	Suitable for top loop use.
;;;;	But expensive and thus must be used coarsely, ie suitable for for top loop use.
;;;;	
;;;; -doce-
;;;;	
%

let lparen_itext = itext_term "(";;
let rparen_itext = itext_term ")";;
let wrap_parens t = iml_cons_term lparen_itext (iml_cons_term t rparen_itext);;

let null_ap_itext = itext_term "Null_ap ";;
let string_ap_itext = itext_term "String_ap ";;
let nat_ap_itext = itext_term "Nat_ap ";;
let bool_ap_itext = itext_term "Bool_ap ";;
let token_ap_itext = itext_term "Token_ap ";;
let oid_ap_itext = itext_term "Oid_ap ";;
let some_oid_ap_itext = itext_term "SomeOid_ap ";;
let some_term_ap_itext = itext_term "SomeTerm_ap ";;
let oids_ap_itext = itext_term "Oids_ap ";;
let tokens_ap_itext = itext_term "Tokens_ap ";;
let strings_ap_itext = itext_term "Strings_ap ";;
let term_ap_itext = itext_term "Term_ap ";;
let terms_ap_itext = itext_term "Terms_ap ";;
let term_list_ap_itext = itext_term "TermList_ap ";;
let unit_ap_itext = itext_term "Unit_ap ";;
let ddg_ap_itext = itext_term "DDG_ap ";;
	       
let make_ap_aux remote_termto_itext toterm =
  \m v. (wrap_parens (iml_cons_term remote_termto_itext (fst m)), ((toterm v) . snd m))
;;
	       
let make_ap remote_termto toterm =
 make_ap_aux (itext_term remote_termto) toterm
;;


let null_ap term = (wrap_parens (iml_cons_term null_ap_itext (wrap_parens term)), []);;
let terms_ap term ts = (wrap_parens (iml_cons_term terms_ap_itext term), ts);;

let string_ap m s =  (wrap_parens (iml_cons_term string_ap_itext (fst m)), ((istring_term s) . snd m));;
let oid_ap m oid  =  (wrap_parens (iml_cons_term oid_ap_itext (fst m)), ((ioid_term oid) . snd m));;
let some_oid_ap m oid  =  (wrap_parens (iml_cons_term some_oid_ap_itext (fst m)), ((isome_oid oid) . snd m));;
let some_term_ap m t  =  (wrap_parens (iml_cons_term some_term_ap_itext (fst m)), ((mk_isome_term t) . snd m));;
let oidt_ap m oidt  =  (wrap_parens (iml_cons_term oid_ap_itext (fst m)), (oidt . snd m));;
let oids_ap m oids  =  (wrap_parens (iml_cons_term oids_ap_itext (fst m)), ((ioids_term oids) . snd m));;
let token_ap m s =  (wrap_parens (iml_cons_term token_ap_itext (fst m)), ((itoken_term s) . snd m));;
let tokens_ap m toks  =  (wrap_parens (iml_cons_term tokens_ap_itext (fst m)), ((itokens_term toks) . snd m));;
let strings_ap m toks  =  (wrap_parens (iml_cons_term strings_ap_itext (fst m)), ((istrings_term toks) . snd m));;
let nat_ap m i =   (wrap_parens (iml_cons_term nat_ap_itext (fst m)), ((inatural_term i) . snd m));;
let bool_ap m i =  (wrap_parens (iml_cons_term bool_ap_itext (fst m)), ((ibool_term i) . snd m));;
let term_ap m t =  (wrap_parens (iml_cons_term term_ap_itext (fst m)), (t . snd m));;
let term_list_ap m ts =  (wrap_parens (iml_cons_term term_list_ap_itext (fst m)), ((iterms_term ts) . snd m));;
let unit_ap m =   (wrap_parens (iml_cons_term unit_ap_itext (fst m)), (ivoid_term . snd m));;			      
let ddg_ap m ddg =   (wrap_parens (iml_cons_term ddg_ap_itext (fst m)), (ddg_to_term ddg . snd m));;

let posure_ap_itext = itext_term "Posure_ap ";;

let posure_ap m cl =
  (wrap_parens (iml_cons_term posure_ap_itext (fst m))
  , ((map_to_ilist (\x.x) icons_op ((fst cl) . (snd cl)))
    . snd m))
;;

let begin_ap s = null_ap (itext_term s);;

let abstract_prefix_itext = (itext_term "\\l. ");;
let abstract_suffix_itext = (itext_term " l");;

% \l. f(g l) %			      
let with_abstract f g =
  wrap_parens (iml_cons_term abstract_prefix_itext
			     (iml_cons_term f 
					    (wrap_parens (iml_cons_term g abstract_suffix_itext))))
;;
% (\l.(f l))%			      
let with_abstract_wo_marshall f g =
  wrap_parens (iml_cons_term abstract_prefix_itext
			(wrap_parens (iml_cons_term g abstract_suffix_itext)))
;;

let make_return_eval_aux toterm_itext termto evalf m =
  termto (evalf (with_abstract toterm_itext (fst m), (snd m)))
;;  

let make_return_eval_marshalled toterm_itext evalf m =
  (evalf (with_abstract toterm_itext (fst m), (snd m)))
;;  
let make_return_eval_unmarshall termto evalf m =
  termto (evalf (with_abstract_wo_marshall id (fst m), (snd m)))
;;  

let make_return_eval to_term_string = 
  make_return_eval_aux (wrap_parens (itext_term to_term_string))
;;


let oid_return_itext = itext_term "ioid_term ";;
let oids_return_itext = itext_term "ioids_term ";;
let string_return_itext = itext_term "istring_term ";;
let strings_return_itext = itext_term "istrings_term ";;
let token_return_itext = itext_term "itoken_term ";;
let bool_return_itext = itext_term "ibool_term ";;
let nat_return_itext = itext_term "inatural_term ";;
let terms_return_itext = itext_term "map_to_ilist (\\x.x) icons_op ";;
let some_term_return_itext = itext_term "mk_isome_term ";;

% old 
let oid_return m = (with_abstract oid_return_itext (fst m), (snd m));;
let string_return m = (with_abstract string_return_itext (fst m), (snd m));;
let token_return m = (with_abstract token_return_itext (fst m), (snd m));;
let bool_return m = (with_abstract bool_return_itext (fst m), (snd m));;
let terms_return m = (with_abstract terms_return_itext (fst m), (snd m));;
%

let make_oid_return = make_return_eval_aux oid_return_itext oid_of_ioid_term;;
let make_oids_return =  make_return_eval_aux oids_return_itext oids_of_ioid_term;;
let make_string_return = make_return_eval_aux string_return_itext string_of_istring_term;;
let make_strings_return = make_return_eval_aux strings_return_itext strings_of_istring_term;;
let make_token_return = make_return_eval_aux token_return_itext token_of_itoken_term;;
let make_bool_return = make_return_eval_aux bool_return_itext bool_of_ibool_term;;
let make_nat_return = make_return_eval_aux nat_return_itext number_of_inatural_term;;
let make_terms_return = make_return_eval_aux terms_return_itext (map_isexpr_to_term_list icons_op);;
let make_some_term_return =  make_return_eval_aux some_term_return_itext some_of_isome_term;;

let make_string_return_marshalled = make_return_eval_marshalled string_return_itext;;
let make_string_return_unmarshall = make_return_eval_unmarshall string_of_istring_term;;

let make_oids_return_marshalled = make_return_eval_marshalled oids_return_itext;;
let make_oids_return_unmarshall = make_return_eval_unmarshall oids_of_ioid_term;;

let eval_args = orb_eval_args `ONE`;;			      
let eval_args_to_term = orb_eval_args_to_term `ONE`;;			      

let eval_args_to_oids desc = make_oids_return (eval_args_to_term desc);;
let eval_args_to_oid desc = make_oid_return (eval_args_to_term desc);;
let eval_args_to_tok desc = make_token_return (eval_args_to_term desc);;
let eval_args_to_string desc = make_string_return (eval_args_to_term desc);;
let eval_args_to_bool desc = make_bool_return (eval_args_to_term desc);;
let eval_args_to_terms desc = make_terms_return (eval_args_to_term desc);;

			      
let nuprl5_refiner_description_term =
  make_term (`!description`, [(make_token_parameter `NUPRL`)])
            [(nil, icons_term (inatural_term 5) (inatural_term 0));
             (nil, itoken_term `REFINE`)]
;;

% if snd cl contains a term with icons_op then other side gets confused.
  could fail here. %
let posure_to_term cl =
  make_term (`!posure`, [])
    [ [], fst cl
    ; [], (map_to_ilist (\x. x) icons_op (snd cl))
    ]
;;

let term_to_posure t = (subtermn 1 t), map_ilist_to_list icons_op (\x.x) (subtermn 2 t);;

let make_eval_args_term desc posure =
  make_term (`!eval_args`, [])
    [ [], desc
    ; [], posure_to_term posure
    ]
;;
 
let term_to_eval_args t = 
  eval_args (subtermn 1 t) (term_to_posure (subtermn 2 t))
;;

let nuprl5_library_description_term =
  make_term (`!description`, [(make_token_parameter `NUPRL`)])
            [(nil, icons_term (inatural_term 5) (inatural_term 0));
             (nil, itoken_term `LIBRARY`)]
;;

let nuprl5_edit_description_term =
  make_term (`!description`, [(make_token_parameter `NUPRL`)])
            [(nil, icons_term (inatural_term 5) (inatural_term 0));
             (nil, itoken_term `EDIT`)]
;;

let metaprl_refiner_description_term =
  make_term (`!description`, [(make_token_parameter `metaprl`)])
            [(nil, (inatural_term 0));
             (nil, itoken_term `REFINER`)]
;;
			      
let nuprl5_rdb_description_term =
  make_term (`!description`, [(make_token_parameter `NUPRL`)])
            [(nil, icons_term (inatural_term 5) (inatural_term 0));
             (nil, itoken_term `RDB`)]
;;

let nuprl5_lib_and_edit_description_term =
  make_term (`!description`, [(make_token_parameter `NUPRL`)])
            [(nil, icons_term (inatural_term 5) (inatural_term 0));
             (nil, icons_term (itoken_term `LIBRARY`) (itoken_term `EDIT`))]
;;

let nuprl5_ref_lib_and_edit_description_term =
  make_term (`!description`, [(make_token_parameter `NUPRL`)])
            [(nil, icons_term (inatural_term 5) (inatural_term 0));
             (nil, icons_term (itoken_term `REFINE`) (icons_term (itoken_term `LIBRARY`) (itoken_term `EDIT`)))]
;;

let object_id_dag_description_term =
  make_term (`!description`, [(make_token_parameter `!ANY`)])
            [(nil, inil_term);
             (nil, itoken_term `ObjectIdDAG`)]
;;

let derived_object_id_dag_description_term =
  make_term (`!description`, [(make_token_parameter `!ANY`)])
            [(nil, inil_term);
	     (nil, icons_term (itoken_term `ObjectIdDAG`) (itoken_term `Derived`))]
;;

let system_of_description_term term =
 destruct_token_parameter (hd (parameters_of_term term))
;;

let purposes_of_description_term term =
 let (op, [[],version; [],purposes]) = destruct_term term in
  map_isexpr_to_list icons_op first_tok purposes
;;

%
;;;;
;;;;	DAG primitives.
;;;;	
;;;;	
%

let idag_cons_op = (`!dag_cons`, nil);;
let idag_cons_term a b = make_term idag_cons_op [([],a); ([],b)];;
let idag_nil_term = make_term idag_cons_op [];;

let idag_child_id = `!dag_child`;;

let idag_child name oid = 
  make_term (idag_child_id, 
	     [ (make_token_parameter name)
	     ; (make_object_id_parameter oid)
	     ])
	    [];;

let name_of_idag_child_term t =
  destruct_token_parameter (hd (parameters_of_term t))
;;

let oid_of_idag_child_term t =
  destruct_object_id_parameter (hd (tl (parameters_of_term t)))
;;


let term_to_tokens =  map_isexpr_to_list icons_op token_of_itoken_term ;;
let tokens_to_term = map_to_ilist itoken_term icons_op;;
						 

letrec descendent_of_term t addr =
  if addr = [] then t
  else descendent_of_term (subterm_of_term t (hd addr)) (tl addr)
;;




let iabstraction_term conds lhs rhs =
   make_term (`!abstraction`, nil)
    [[], conds; [], lhs; [],rhs]
;;

let icondition_cons_op = `!condition_cons`, nil;;
let icondition_nil_term = make_term icondition_cons_op nil;;

let abstraction_template = 
  iabstraction_term
    (make_term icondition_cons_op nil)
    iplaceholder_term iplaceholder_term
;;

let group_by_prefix m l =

  letrec aux prefix l lol r =
     if (null r) then ((string_to_tok prefix,l) . lol)
     else let (p,n) = hd r in
           if p = prefix then aux prefix (n . l) lol (tl r)
	   else aux p [n] ((string_to_tok prefix,l) . lol) (tl r)
  in

   let l = rev
             (quicksort (\a b. (string_lt (fst a) (fst b)))
               (map (\n. (fst (string_split true m (tok_to_string n))),n) l)) in
    
    if (null l) then nil
    else let (p, n) = hd l in
	   aux p [n] [] (tl l)
;;

let opid_alist_cons = `opid_alist_cons`, nil;;
let opid_alist_term opid l = make_term (`opid_alist`, [make_token_parameter opid]) [[],l];;

let opid_of_opid_alist oa = destruct_token_parameter (parameter_of_term oa 1);;
let list_of_opid_alist oa = subterm_of_term oa 1;;

let opid_alist_to_term oa =
  map_to_ilist (\(opid, l). opid_alist_term opid (map_to_ilist itoken_term itok_cons_op l))
	       opid_alist_cons
	       oa
;;


let term_to_opid_alist t = 
  map_isexpr_to_list opid_alist_cons 
     (\oaterm. ( opid_of_opid_alist oaterm
	       , (map_isexpr_to_list itok_cons_op token_of_itoken_term
		    (list_of_opid_alist oaterm))))
     t
;;


%
;;;;	
;;;;	prefer oaa 
;;;;	  append any tokens from oab not in oaa in order to end.
;;;;	
%


let merge_opid_alist oaa oab = 
 if (null oab) then oaa
    else append oaa (filter (\b. not (member b oaa)) oab)
;;


let merge_opid_alists oaa oab = 
 map (\oa . (fst oa, (merge_opid_alist (snd oa) (snd (assoc (fst oa) oab)))) ? oa) 
     oaa
;;

let map_term_p p t = 
 letref r = false in
 
  map_term p (\t f. r:= true; t) t

 ; r
;;

let map_term_op p f =
 map_term p (\t contf. 
               let (op, bts) = destruct_term t in
                f (make_term op (map (\bt. (fst bt, contf (snd bt))) bts)))
;;


let term_string_search s = string_match_op (string_match_f false s);;

let term_accumulate contp accf t = 
 letrec aux t acc =
  (if not (contp t) then acc
   else let nacc = btsaux (bound_terms_of_term t) acc in
        (((accf t) . nacc) ? nacc))
 and btsaux bts acc =
  if null bts then acc
  else let nacc = btsaux (tl bts) acc in
        aux (snd (hd bts)) nacc

 in aux t nil
;;

let static_oids_of_term term =
 letref acc = nil in
  map_term_p (\t. map (\p. if (object_id_parameter_p p) 
                              then ((acc := (destruct_object_id_parameter p) . acc; ()) ? ()))
		      (parameters_of_term t)
	          ; false)
             term
  ; (fast_remove_duplicate_oids acc)
;;

let include_properties_term props term = 
  make_term (`!include_properties`, []) [[], property_list_to_term props; [], term]
;;

let add_props_to_odata props t =
 replace_subterm 1 (include_properties_term props (subtermn 1 t)) t;;

letrec remove_labels t =
 if (`!label` = id_of_term t) then remove_labels (subtermn 1 t)
 else let (op, bts) = destruct_term t in
       make_term op (map (\b,bt. (b, remove_labels bt)) bts)
;;  					 


let funny_tok_list_to_toks t = 
 letrec aux t = 
   ((if ((id_of_term t) = `!tok_cons`) then ((first_tok t) . (aux (subtermn 1 t)))
   else nil) ? nil)
 in (aux (subtermn 1 t) ? nil)
;;

%let gcvcn () = 
  map funny_tok_list_to_toks (map_ilist_to_list (`!log_cons`,nil) id (view_collect ()));;
let gcvrn () = 
  map funny_tok_list_to_toks (map_ilist_to_list (`!log_cons`,nil) id (view_remaining ()));;
let gcvpn () = 
  map funny_tok_list_to_toks (map_ilist_to_list (`!log_cons`,nil) id (permanent_logs ()));;
%
let testtree =
  (icons_term (itoken_term `a`) 
              (map_to_ilist id icons_op
               ( (icons_term (itoken_term `b`) (map_to_ilist id icons_op
                                                 ( (icons_term (itoken_term `d`) inil_term) 
                                                 . (icons_term (itoken_term `e`) inil_term)
                                                 . nil)))
               . (icons_term (itoken_term `c`) inil_term)
               . nil)))  
;;


letref temp_oid = dummy_object_id();;

let set_temp_oid oid = temp_oid := oid;;
let set_temp_oidt oidt = temp_oid := (oid_of_ioid_term oidt);;
					 
%	
;;;;	map_term : (term -> bool) -> (term -> (term -> term) -> term) -> term -> term 
%	

let map_term_top_down p f =
 map_term p (\t contf. 
               let (op, bts) = destruct_term (f t) in
                (make_term op (map (\bt. (fst bt, contf (snd bt))) bts)))
;;

let make_map_term_bottom_up (f: term -> term) t contf = 
   let (op, bts) = destruct_term t in
       f (make_term op (map (\bt. (fst bt, contf (snd bt))) bts))
;;

let map_term_bottom_up p f = map_term p (make_map_term_bottom_up f);;


	     

absrectype * tree = * # ((* tree) list)
 with mk_tree n c = abs_tree (n, c)
 and dest_tree t = rep_tree t
;;

let tree_to_term consop treeop nodef t =
 letrec aux t = 
  let n,c = dest_tree t in
    make_term treeop [[], nodef n; [], listf c]  
 and listf l = map_to_ilist aux consop l in

 aux t
;;

		      
let term_to_tree consop treeop nodef t =
 letrec aux t = 
   let op, [[], n; [],c] = destruct_term t in
    if not (equal_operators_p op treeop) then fail	     
    else mk_tree (nodef n) (listf c) 

 and listf t = map_ilist_to_list consop aux t in
 
 aux t
;;

let map_tree tree f =
 letrec aux t =
  let n,c = dest_tree t in
   mk_tree (f n) (map aux c)
 in aux tree
;;

let itree_op = `!tree`, nil;;
let i_term_to_tree = term_to_tree icons_op itree_op;;
let i_tree_to_term = tree_to_term icons_op itree_op;;


let with_spool = with_spool_aux false;;
let with_spool_overwrite = with_spool_aux true;;	    

let make_tmp_filename dirs fname ftype =  
  make_system_filename (["library"; "tmp"] @ dirs) fname ftype
;;			
	    
