%
*************************************************************************
*                                                                       *
*    Copyrighted Cornell University 2000                                *
*                                                                       *
*                                                                       *
*                Nuprl Proof Development System                         *
*                ------------------------------                         *
*                                                                       *
*   Developed by the Nuprl group, Department of Computer Science,       *
*   Cornell University, Ithaca NY.  See the release notes for a list    *
*   of the members of the group.                                        *
*                                                                       *
*   Permission is granted to use and modify Nuprl provided this notice  *
*   is retained in derived works.                                       *
*                                                                       *
*                                                                       *
*************************************************************************
%

%[
*****************************************************************************
*****************************************************************************
UTILITIES.ML
*****************************************************************************
*****************************************************************************
Functions for assisting in adding objects associated with:

1. Untyped definitions
2. Typed definitions
3. Recursive definitions
4. Set definitions
]%


let mk_typed_def_n n opid = 
  mlbreak `create`; fail
;;

let mk_typed_def opid = mk_typed_def_n 1 opid ;;


%[
*****************************************************************************
Formatting Blocks of text
*****************************************************************************
A "string block" is a string with imbedded newlines, so it forms a block
of text.
]%

let newline_term = mk_simple_term `!newline` [] ;;

let newline_char = int_to_char 10;;
let newline_p c = c = newline_char;;

let find_newline_position l = 
 let c = find newline_p l
 in position c l
;;

let strip_last_newline l = 
  if null l then l
  if newline_p (last l) then
    remove_last l
  else 
    l
;;

let string_divide_into_lines s n =
  let toks = strip_last_newline (explode_string s) in
  let split_at_ret l = 
    let a,b = split (find_newline_position l - 1) l in a, (tl b) 
  in
  let natural_tok_lines =
    unreduce1 split_at_ret toks 
  in
  let clipped_tok_lines = 
  ( if n > 0 then
      flatten
        (map
          (unreduce1 (\l.if length l > n then split n l else fail))
          natural_tok_lines
        )
    else
      natural_tok_lines
  )
  in
    map toks_to_string clipped_tok_lines
;;

let string_block_to_term sb = 
  let lines = string_divide_into_lines sb 100
  in 
  let tm_list = 
    flatten
      (map
        (\lin.[mk_text_term lin;newline_term])
        lines
      )
  in
    mk_text_seq `!text_cons` tm_list
;;



%[
*****************************************************************************
Adding new display form definitions.
*****************************************************************************
Functions for building display form definitions:

]%
% 
-----------------------------------------------------------------------------
List making functions   
-----------------------------------------------------------------------------
%


let mk_df_def_list defs = 
  mk_term_seq `!dform_cons` defs ;;
let mk_df_slot_option_list opts = 
  mk_term_seq `!dform_child_attr_cons` opts ;;
let mk_df_attr_list attrs = 
  mk_term_seq `!dform_attr_cons` attrs ;;
let mk_condition_list conds = 
  mk_term_seq `!condition_cons` conds ;;


% 
-----------------------------------------------------------------------------
Display form Attribute building
-----------------------------------------------------------------------------
%

let mk_dform_macro name =
 mk_term (`!dform_macro_name`,[mk_string_parm name]) nil
;;



% 
-----------------------------------------------------------------------------
Display form LHS building
-----------------------------------------------------------------------------
%

let mk_df_format_list formats = 
  mk_text_seq `!dform_format_cons` formats ;;


let mk_df_slot_parens option = 
  let mk_tm tok = 
     mk_term (`!dform_child_parens`
                  ,[mk_token_parm tok
                   ;mk_string_parm "("
                   ;mk_string_parm ")"
                   ]
                  )
                  [[],ivoid_term]
  in 
      option = "L" => mk_tm `LESS`
    | option = "E" => mk_tm `EQUAL`
    | option = "*" => mk_tm `ALL`
    | failwith `mk_df_slot_parens: unrecognized option`
;;



% parens is "L" "E" "*", or "none" %

let mk_df_slot_format id desc parens= 
  mk_term 
    (`!dform_variable_child`,[mk_string_parm id;mk_string_parm desc])
    [[],mk_df_slot_option_list 
          (parens = "none" => [] | [mk_df_slot_parens parens])
    ]
;;


% takes list of terms and interleaves appropriate delimiters %

let mk_format_seq null_str pre_str in_str post_str args = 
  letrec join ts = 
    if null ts then 
      [mk_text_term post_str]
    else 
    ( let t.ts' = ts
      in
        mk_text_term in_str . t . join ts'
    )
  in
  if null args then 
    [mk_text_term null_str]
  else
    mk_text_term pre_str . (hd args) . join (tl args)
;;
    


% takes list of term lists and interleaves appropriate delimiters %

let mk_format_seq_seq null_str pre_str in_str post_str seqs = 
  letrec join ts = 
    if null ts then 
      [[mk_text_term post_str]]
    else 
    ( let t.ts' = ts
      in
        [mk_text_term in_str] . t . join ts'
    )
  in
  if null seqs then 
    [mk_text_term null_str]
  else
    mk_text_term pre_str . (flatten (hd seqs . join (tl seqs)))
;;
    




let mk_df_lhs_bvar_seq = 
  mk_format_seq "" "" "," "."  
;;

let mk_df_lhs_bterm_seq = 
  mk_format_seq_seq "()" "(" ";" ")"  
;;

let mk_df_lhs_parm_seq = 
  mk_format_seq "" "{" "," "}"  
;;


% if parm is abstraction meta, then create a slot format. Otherwise, create
  a text format.
%
letref parm_type_strings = 
  [`token`,"tok"
  ;`natural`,"nat"
  ;`string`,"string"
  ;`variable`,"var"
  ;`bool`,"bool"
  ;`level-expression`,"level"
  ]
;;

let is_obid_parm p = `oid` = type_of_parameter p;;

let mk_df_fmt_for_parm p = 
  if is_ab_mparm p then
  ( let pstr = tok_to_string (id_of_mparm p)
    in let ptype = apply_alist parm_type_strings (type_of_parm p)
    in
      mk_df_slot_format pstr ptype "none"
  )
  else
    mk_text_term (parameter_to_string p)
;;

% This creates a rather dull but usable def %

let mk_default_df_lhs_for_term t = 
  let bvs_to_fmts vs = 
    mk_df_lhs_bvar_seq 
        (map 
          (\v.let vstr = tok_to_string (var_to_tok v)
              in
                mk_df_slot_format vstr "var" "none"
          )
          vs
        )
  in
  let bterm_to_fmts (vs,t') = 
  ( let tstr = (tok_to_string o var_to_tok o fst o dest_so_var) t'
    in
      bvs_to_fmts vs @ [mk_df_slot_format tstr tstr "*"]
  )
  in let (opid,parms),bterms = dest_term t
  in
  (mk_df_format_list o flatten)
        [[mk_text_term (tok_to_string opid)]
        ;mk_df_lhs_parm_seq (map mk_df_fmt_for_parm
	                     % take out obid constants. %
	                     (filter (\p. (is_ab_mparm p) or not (is_obid_parm p)) parms))
        ;mk_df_lhs_bterm_seq (map bterm_to_fmts bterms)
        ]
;;

% 
-----------------------------------------------------------------------------
Display form RHS building
-----------------------------------------------------------------------------
%

% 
1. convert all (poss so) variables into display meta terms
2. convert all binding vars into display meta vars
3. convert all ab-meta parameters into display-meta parms
%


let mk_df_rhs_for_term t = 
  let conv_parm p = 
    is_ab_mparm p 
    => 
      mk_disp_mparm (type_of_parm p) (id_of_mparm p) 
    | 
      p
  in
  let conv_var v = mk_disp_mvar (var_to_tok v)
  in
  let conv_term t = mk_disp_meta_term (var_to_tok (fst (dest_so_var t)))
  in
  let conv_bterm (vs,t) = map conv_var vs , conv_term t
  in
  let (opid,ps),bts = destruct_term t  
  in
    make_term
      (opid, map conv_parm ps)
      (map conv_bterm bts)
;;

% 
-----------------------------------------------------------------------------
Display Object Building
-----------------------------------------------------------------------------
%

let mk_dform_def_term name lhs rhs = 
  mk_simple_term 
    `!dform`
    [mk_df_attr_list [mk_dform_macro name]
    ;lhs
    ;rhs
    ]
;;

let mk_disp_object_term_for_term t name = 
  mk_df_def_list 
    [mk_dform_def_term
      name
      (mk_default_df_lhs_for_term t)
      (mk_df_rhs_for_term t)
    ]
;;


% 
-----------------------------------------------------------------------------
Abstraction Object Building
-----------------------------------------------------------------------------
%

let mk_ab_object_term lhs rhs = 
  mk_simple_term
    `!abstraction`
      [mk_condition_list []
      ;lhs
      ;rhs
      ]
;;

% use this on rhs if you don't want ab to be unfolded. %

let null_ab = mk_simple_term `!null_abstraction` [];;

% 
-----------------------------------------------------------------------------
Library Object Creation
-----------------------------------------------------------------------------
%

%
;;;;	
;;;;	Following accumulate object content needed to create objects in lib.
;;;;	  Intention is that content for a group of objects can be created then
;;;;	    data passed onto lib for creation. 
;;;;	  Trying to localize content creation from lib modification.
;;;;	
;;;;	tok{kind} # tok{name} # term{content}
;;;;	
;;;;	THM content is proof{inf-tree} from which stm and prf can be extracted.
;;;;	
%

let create_disp_obj_data t obname = (`DISP`, obname, t);;
let create_abs_obj_data l r obname = (`ABS`, obname, (mk_ab_object_term l r));; 
let create_ab_obj_data ab obname = (`ABS`, obname, ab);; 
let create_ml_obj_data code obname = (`ML`, obname, code);; 
let create_df_obj_data df obname = (`DISP`, obname, df);;
let create_stm_obj_data l obname = (`STM`, obname, l);; 
let create_thm_obj_data goal tac obname =
  (`THM`, obname, (mk_term (`!cons`, nil) [([],goal); ([],tac)]));; 

let create_disp_content_for_new_def t name mname = 
   create_disp_obj_data (mk_disp_object_term_for_term t mname) name
;;

let create_ab_content_for_new_def lhs_term rhs_term obname = 
  create_abs_obj_data lhs_term rhs_term obname
;;



