
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2000                                *
;;;                                                                       *
;;;                                                                       *
;;;                Nuprl Proof Development System                         *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the Nuprl group, Department of Computer Science,       *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify Nuprl provided this notice  *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;************************************************************************

#+cltl2(in-system-package)
#-cltl2(in-package *system-package-name*)

(defun arith-reflexive-equal-term (type term)
  (instantiate-term (instantiate-operator *equal* nil)
		    (list (instantiate-bound-term type)
			  (instantiate-bound-term term))))


(defun not-term$ (term)
  (or (not-equal-term-p term) (not-less-than-term-p term)))

(defun not-term-body$ (term)
  (lefttype-of-function-term term))


;--
;-- arith (assns:declaration-list, concl:term, rule:??)
;--           
;--     This returns either the atom GOOD if the deduction succeeds or an
;--     atom whose print name is an error message if the deduction fails or
;--     its inputs are not of the form specified above.
;--

(defun do-arith (assns concl rule)
  (declare (ignore rule))
  (catch 
    'arith-result 
    (doarith$  (bust-conjunctions (ok-arith-assums$ assns)) concl)))


;--
;-- ok-arith-assums (assums:list-of-declarations)
;--
;--     Return a list of terms where each element of the list is a term of the 
;--     sort that arith can handle.
;--

(defun ok-arith-assums-old$ (assums);;lal
  (let ((result nil))

    (dolist (decl assums)
      (setf result
	    (cons (type-of-assumption decl)
		  (cons (arith-reflexive-equal-term (type-of-assumption decl)
						    (variable-term (id-of-assumption decl)))
			result))))

    result))

 

(defun ok-arith-assums$ (assums);;lal new, more like arith.lisp
  (let ((result nil))

    (dolist (decl assums)
      (setf result
	    (cons (type-of-assumption decl)
		  (if (id-of-assumption decl)
		      (cons (arith-reflexive-equal-term (type-of-assumption decl)
						    (variable-term (id-of-assumption decl)))
			result)
		    result))))

    result))

 

; Given a list of nuprl terms, return a list of terms such
;  that none of the terms is a conjunction (indep. product).
;  Each independent-product term in input list is destructed into
;  their two constituent terms. 
; This allows Arith Rule to "reach inside" conjunctions, where it
;  used to ignore conjunctions as not being of valid Arith format.

(defun bust-conjunctions (tmlist)
  (cond
    
    ((null tmlist)  nil)
    
    (t
      (let* ((head  (car tmlist)))
	(cond 
	  ((independent-product-term-p head)
	   (append (bust-conjunctions (list (lefttype-of-product-term head)))
                   (append (bust-conjunctions (list (righttype-of-product-term head)))
			   (bust-conjunctions (cdr tmlist)))))

	  (t  (cons
		head
		(bust-conjunctions (cdr tmlist)))))))))




;;;--
;;;-- doarith$ (assums: term-list, concl:term)
;;;--
;;;--     Check if the conclusion (which may be a literal or a disjunction
;;;--     of literals) follows from the assumptions by considerations of
;;;--     arithmetic, simplification, and equality/list knowledge.
;;;--     Throw (with tag 'arith-result) the atom 'GOOD if the conclusion
;;;--     follows, an atom whose print name is an error message, otherwise.
;;;--
;;;--     Non-literals are ignored.
;;;--
;;;--     A "literal" here is an EQUAL or LESS term wrapped in any number of
;;;--     negations (-> VOIDs).  This allows users to define relational operators,
;;;--     such as a>=b as NOT(a<b), and then have NOT(...) be considered a literal.
;;;--

(defvar A-graph-literals$ nil)       ;-- literals from assums and concl that give
                                     ;--   rise to edges in the A-graph -- this
                                     ;--   means < and ~< relations.  Each entry
                                     ;--   in the list is a triple: the relation
                                     ;--   (LESS or NOT-LESS), the simplified form
                                     ;--   of the left term, the simplified form of
                                     ;--   the right term.

(defvar A-disequalities$ nil)            ;-- strict disequality literals contained
                                     ;--   in assums and concl between arithmetic
                                     ;--   terms -- this is a list of two element
                                     ;--   lists, each of which is: the simplified
                                     ;--   form of the left operand, the simplified
                                     ;--   form of the right operand.

(defvar A-nodes$ nil)                    ;-- representatives in E-graph for the rest
                                     ;--   part of normal form terms entered
                                     ;--   into A-graph.  That is, each node in
                                     ;--   the A-graph is "named" by the (rep of
                                     ;--   the) class in the E-graph to which its
                                     ;--   normal form term (without constant part)
                                     ;--   belongs.

(defvar A-edges$ nil)                    ;-- a list of 4 element lists, (si,ti,c,lit),
                                     ;--   where si and ti are A-node indices
                                     ;--   (1,2,...) indexing into A-nodes$,
                                     ;--   c is the weight of the edge from si
                                     ;--   to ti, and lit is the literal that
                                     ;--   gave rise to the edge.

(defvar A-matrix$ nil)                   ;-- the weighted connectivity matrix over
                                     ;--   A-nodes$xA-nodes$ with entries from
                                     ;--   A-edges$.  (see build-matrices$)

(defvar L-matrix$ nil)                   ;-- the literal matrix corresponding to
                                     ;--   A-matrix$.  (see build-matrices$)

(defvar A*-matrix$ nil)                  ;-- the transitive closure of A-matrix$,
                                     ;--   containing max path weights in the
                                     ;--   A-graph.  (see build-matrices$)

(defvar A-size$ nil)                     ;-- the size of the square matrices A-matrix$,
                                     ;--   L-matrix$, and A*-matrix$.

(defvar type-int$ (int-term))


;--
;-- process-assumptions$ (assums:term-list)
;--
;--     process each literal in assums
;--

(defun process-assumptions$ (assums)
  (dolist (a assums)
    (process-literal$ a)))


;;;--
;;;-- assert-E-graph-disequality$(termlist:list of prlterms)
;;;--
;;;--     Add the knowledge ~(t=..=t' in T) to the E-box.
;;;--

(defun assert-E-graph-disequality$ (termlist)
    (do-disequality termlist))


;--
;-- E-graph-class$(a:term)
;--
;--     Return the representative for a (with type T) in the E-graph.  If a does 
;--     not occur in the E-graph, add it and return its representative.
;--

(defun E-graph-class$ (a)
    (eq-insert a))


;--
;-- do-E-disequalities$()
;--
;--     Check whether the E-graph has any conflict with the disequalities
;--     that have been asserted into it.  If so, the literal set is not
;--     satisfiable, so throw 'GOOD to 'arith-result.

(defun do-E-disequalities$ ()

  (dolist (diseq A-disequalities$)
    (do-disequality diseq))
 
  (when (unsatisfiable-graph)
    (throw 'arith-result 'GOOD)))




(defun doarith$ (assums concl)
  
    (setf A-disequalities$ nil)
    (setf A-graph-literals$ nil)
   
    (init-E-graph$)

    (process-assumptions$ assums)
    (process-conclusion$ concl)
    
    (propogate-A-E-equalities$)

    (do-A-disequalities$)

    (do-E-disequalities$)

    ;-- There is a choice of values for variables that makes the literals
    ;-- satisfiable.  Thus, the conclusion does not follow from the assumptions.
      (progn ;;(break "ar")
	(throw 'arith-result
               "Conclusion does not follow from assumptions by ARITH." )))



;--
;-- process-conclusion$ (concl:term)
;--
;--     negate the conclusion and process the resulting (conjunction of) literals
;--

(defun get-disjuncts$ (union-term)
  (if (union-term-p (righttype-of-union-term union-term))
      (cons (lefttype-of-union-term union-term) 
	    (get-disjuncts$ (righttype-of-union-term union-term)))
      (list (lefttype-of-union-term union-term) 
	    (righttype-of-union-term union-term))))

(defun process-conclusion$ (concl)
  (cond
    ((union-term-p concl)
     (dolist (disjunct (get-disjuncts$ concl))
       (if (not-term$ disjunct)
	   (process-literal$ (not-term-body$ disjunct))
	   (process-literal$ (not-term disjunct)))))

    ((not-term$ concl)
     (process-literal$ (not-term-body$ concl)))

    (t (process-literal$ (not-term concl)))))



;--
;-- process-literal$ (literal:term)
;--
;--     Unwrap multiple levels of negation from literal and put it in
;--     A-graph, E-graph, or disequality list, as appropriate.
;--

(defun process-literal$ (literal)

  (let ((poslit literal) ;;; the positive form of literal (literal with all
	                  ;;;   layers of negation stripped off)
	(negated nil))	;;; non-nil if literal is the negation of poslit

    (do ((term literal (term-of-not-term term)))
	((not (not-term-p term))
	 (if (or (less-than-term-p term)
		 (equal-term-p term))
	     (setf poslit term)
	     (setf negated nil)))
      (setf negated (not negated)))
	      
    (cond
      ((less-than-term-p poslit)
       (let ((left (simplify (leftterm-of-less-than-term poslit)))
	     (right (simplify (rightterm-of-less-than-term poslit))))

	 (assert-E-graph-equality$  
	  (list (leftterm-of-less-than-term poslit) left))
	 (assert-E-graph-equality$  
	  (list (rightterm-of-less-than-term poslit) right))

	 (if negated
	     (setf A-graph-literals$
		   (cons (list 'NOT-LESS left right) A-graph-literals$))
	     (setf A-graph-literals$
		   (cons (list 'LESS left right) A-graph-literals$)))))

       ((and (equal-term-p poslit)
	     (equal-terms-p (type-of-equal-term poslit) type-int$))
	
	(let ((poslit (funky-equal-term-for-arith poslit))
	      (sterms (mapcar #'simplify
			      (terms-of-equal-term poslit))))

	  (mapc #'(lambda (term sterm)
		    (assert-E-graph-equality$ (list term sterm)))
		(terms-of-equal-term poslit)
		sterms)

	  ;;- We have an arithmetic equality or disequality. For an equality,
	  ;;- since the nodes of the A-graph will not necessarily be left
	  ;;- or right, but will be the rest parts of their normal forms,
	  ;;- it is not adequate to simply assert this into the E-graph --
	  ;;- we must treat it as left<=right & right <= left.  For a disequality,
	  ;;- we save it and later see if left<right or right<left is satisfiable.
	  (if (not negated)
	      (do ((terms sterms (cdr terms)))
		  ((null (cdr terms)))
		(push (list 'not-less (car terms) (cadr terms))
		      A-graph-literals$)
		(push (list 'not-less (cadr terms) (car terms))
		      A-graph-literals$))
	      (when (= (length sterms) 2)
		(push (list (car sterms) (cadr sterms)) A-disequalities$))))))))


(defun funky-equal-term-for-arith-old (term)
  (if (and (equal-term-p term)
	   (reflexive-equal-term-p term))
      (let ((bound-terms (bound-terms-of-term term)))
	(instantiate-term (instantiate-operator *equal* nil)
			  (list (car bound-terms) (cadr bound-terms))))
      term))


(defun funky-equal-term-for-arith (term) ;;lal
  (if (and (equal-term-p term)
	   (reflexive-equal-term-p term))
      (let ((bound-terms (bound-terms-of-term term)))
	(instantiate-term (instantiate-operator *equal* nil)
			  (list (cadr bound-terms))))
      term))





;;;-- 
;;;-- propogate-A-E-equalities$ ()
;;;--
;;;--     Propogate equalities between the A-graph and the E-graph.
;;;--     If the A-graph ever becomes unsatisfiable, then throw the
;;;--     atom GOOD to tag arith-result.
;;;--

(defun propogate-A-E-equalities$ ()

    (let ((new-equalities-generated t))    ;;;-- t iff E-graph has been changed since
                                             ;;;--   A-graph was built and processed
      (do ()
	  ((null new-equalities-generated))
	(build-A-graph$)
	(build-matrices$ A-edges$)
	(setf new-equalities-generated (propogate-A-equalities-to-E-graph$)))))




;;;--
;;;--
;;;-- A-node-index$(q:term)
;;;--
;;;--     Return the index of (the class of) q in A-nodes$ (1,2,...).
;;;--     If q does not occur in A-nodes$, then add it and return its index.
;;;--

(defun A-node-index$ (q)

  (let ((index 1)             ;-- index of (car nodes) in A-nodes$
	(class (E-graph-class$ q)))          ;-- the class in the E-graph into which q fal ls

    (do ((nodes A-nodes$ (cdr nodes)))
	((or (null nodes) (eql class (car nodes)))
	 (if (null nodes)
	     (progn
	       (setf A-nodes$ (append A-nodes$ (list class)))
	       (length A-nodes$))
	     index))
      (setf index (1+ index)))))


;;;--
;;;-- add-LE-to-A-graph$(q:norm-arith-term, r:norm-arith-term, lit:literal)
;;;--
;;;--     Add the relationship q<=r to the arithmetic graph.
;;;--
;;;--     This is done by entering an edge from the rest part of q to the rest
;;;--     part of r with weight (const part of q)-(const part of r).  The edge
;;;--     is annotated with lit, the literal that gave rise to the edge.
;;;--
 

(defun add-LE-to-A-graph$ (q r lit)

  (let ((qi (A-node-index$ (cadr q)))   ;-- index of rest part of q in A-nodes$
	(ri (A-node-index$ (cadr r))))   ;-- index of rest part of r in A-nodes$
    
    (setf A-edges$
	  (cons (list qi ri (- (car q) (car r)) lit)
		A-edges$))))


;--
;-- build-A-graph$()
;--
;--     Process A-graph-literals and build A-nodes$ and A-edges$
;--     using knowledge in the E-graph on what terms are equal.
;--

(defun build-A-graph$ ()

    (setf A-nodes$ nil)
    (setf A-edges$ nil)

    (dolist (lit A-graph-literals$)
      (let ((op      (car lit))		;-- the literal's  operator (LESS/NOT-LESS)
	    (a       (cadr lit))	;-- the left operand of the literal
	    (b       (caddr lit))	;-- the right operand of the literal
	    (a-norm  nil)		;-- the normal form of a: (const rest)
	    (b-norm  nil))		;-- the normal form of b: (const rest)
	      
	(setf a-norm (norm-simplified-term$ a))
	(setf b-norm (norm-simplified-term$ b))

	(if (eql op 'LESS)
	    (add-LE-to-A-graph$ (list (1+ (car a-norm)) (cadr a-norm))
				b-norm
				lit)
	    (add-LE-to-A-graph$ b-norm a-norm lit)))))



;;;--
;;;-- build-matrices$(edges)
;;;--
;;;--     Process A-nodes$ and edges and build the weighted connectivity
;;;--     matrix A, where A[i,j] = max weight edge from i to j and is
;;;--     minus-infinity if there is no edge from i to j.  Diagonal entries
;;;--     of A areis made to be 0 if they otherwise would be less than 0.
;;;--
;;;--     Build an associated matrix L, where for each non-diagonal entry
;;;--     of A, say A[i,j], that isn't minus-infinity, L[i,j] is the literal
;;;--     that gave rise to the entry of A.
;;;--
;;;--     Finally, construct A*, the transitive closure of A, where A*[i,j]
;;;--     is the weight of the maximum weight path in A from i to j.
;;;--
;;;--     If any diagonal entry of A* has weight greater than 0, there is a
;;;--     positive weight cycle in A so the literal set is unsatisfiable and
;;;--     we throw the atom 'GOOD to tag arith-result.  Otherwise we return nil.
;;;--

(defun build-matrices$ (edges)

  ;;-- set A-matrix$ to an "identity matrix" under (add,times)=(max,+)
  (setf A-size$ (length A-nodes$))
  (setf A-matrix$
	(make-array (list A-size$ A-size$) :initial-element 'minus-infinity))

  (dotimes (i A-size$)
    (setf (aref A-matrix$ i i) 0))

  (setf L-matrix$ (make-array (list A-size$ A-size$)))

  ;;-- add the edges into the identity matrix already formed,
  ;;-- recording the generating literals in L-matrix$.
  (dolist (edge edges)
    (let ((A-val (aref A-matrix$ (1- (car edge))
		       (1- (cadr edge))
		       ))
	  (edge-val  (caddr edge))
	  (edge-lit  (cadddr edge)))
	    
      (when (or (eql A-val 'minus-infinity)
		(> edge-val A-val))
		    
	(setf (aref A-matrix$ (1- (car edge)) (1- (cadr edge)))
	      edge-val)
	(setf (aref L-matrix$ (1- (car edge)) (1- (cadr edge)))
	      edge-lit))))

  ;;-- set A*-matrix$ to the transitive closure of A-matrix$
  ;;-- under (add,times)=(max,+) throwing 'GOOD to 'arith-result
  ;;-- if any diagonal entry becomes positive
  (setf A*-matrix$ (make-array (list A-size$ A-size$)))
    
  (dotimes (i A-size$)
    (dotimes (j A-size$)
      (setf (aref A*-matrix$ i j)
	    (aref A-matrix$ i j))))

  (do ((matrix-changed t))
      ((null matrix-changed))

    (setf matrix-changed (matrix-square$ A*-matrix$ A-size$))
    (dotimes (i A-size$)
      (when (> (aref A*-matrix$ i i) 0)
	(throw 'arith-result 'GOOD))))

  nil)



;--
;-- max-edge-weight$(x:weight,y;weight)
;--
;--     Return the maximum of x and y as edge weights.  Edge weights are either numbers
;--     or the symbol 'minus-infinity, which is less than any number.
;--

(defun max-edge-weight$ (x y)
  (cond
    ((eql x 'minus-infinity) y)
    ((eql y 'minus-infinity) x)
    (t (max x y))))


;--
;-- sum-edge-weight$(x:weight,y:weight)
;--
;--    Return the sum of x and y as edge weights.  This is their integer sum,
;--    unless one is minus-infinity, in which case the sum is minus-infinity.
;--

(defun sum-edge-weight$ (x y)
  (cond
    ((eql x 'minus-infinity) 'minus-infinity)
    ((eql y 'minus-infinity) 'minus-infinity)
    (t (+ x y))))



;--
;-- matrix-square$ (A:array,size:integer)
;--
;--     Set A, a size*size 2 dimensional array, to A*A where (add,times)=(max,+).
;--     Return t if A*A differs from A, else nil.


(defun matrix-square$ (A size)

  (let ((B (make-array (list size size)))     ;-- to hold A*A temporarily
	(matrix-changed nil))
    
    (dotimes (i size)
      (dotimes (j size)
	(let ((x 'minus-infinity))
	  (dotimes (k size)
	    (setf x
		  (max-edge-weight$ x
				    (sum-edge-weight$ (aref A i k) (aref A k j)))))
	  (setf (aref B i j) x)
	  (when (not (equal (aref A i j) x))
	    (setf matrix-changed t)))))

    (dotimes (i size)
      (dotimes (j size)
	(setf (aref A i j) (aref B i j))))

    matrix-changed))


                              
;;;--
;;;-- propogate-A-equalities-to-E-graph$()
;;;--
;;;--     Examine the A, L, and A* matrices and propogate the knowledge produced by
;;;--     zero weight cycles to the E-graph.  If the E-graph is changed as a result
;;;--     of sending it this knowlegde (equalities) then yield t, otherwise yield nil.
;;;--
;;;--     Given a zero weight cycle in A, say: 
;;;--
;;;--              w1     w2     wn-1
;;;--           t1---->t2---->...---->tn = t1
;;;--
;;;--     we deduce that t1+w1=t2, t2+w2=t3, ..., tn-1+wn-1=t1.  In order that the
;;;--     E-graph get some equalities between terms it already knows about, we take
;;;--     each of these equalities and find the literal that generated its edge.
;;;--     Such a literal is either a<b or ~a<b.  In the second case, ~a<b generated
;;;--     the relation b<=a, b and a were normalized to br+bc<=ar+ac (xr means rest
;;;--     part of x, xc means const part of x), and an edge br--->ar with weight bc-ac
;;;--     was generated.  Thus, br+(bc-ac)=ar means br+bc=ar+ac, which in turn means
;;;--     that b=a.  Similarly, the first form of literal, a<b, generates the equality
;;;--     a=b-1.
;;;--
;;;--     Notes:
;;;--       - we we have chosen what equalities (among the many possible) to propogate
;;;--         to the E-graph, and some other choice might be better.
;;;--
;;;--       - since every node is equal to itself, and the E-graph already knows this,
;;;--         we do not propogate such equalities.
;;;--
;;;--       - the cycle in which an edge participates has no bearing on what equality
;;;--         we generate for it, thus below we look for the edges that participate in
;;;--         zero weight cycles, but don't really find the cycles.  Now, an edge (i,j)
;;;--         participates in such a cycle iff it can be viewed as the last edge in the
;;;--         cycle.  This means that j must be a node that occurs in a zero weight cycle.
;;;--         Given that the graph has no positive weight cycles, a node j occurs in a
;;;--         zero weight cycle iff A*[j,j]=0.  So we consider those j for which this is
;;;--         the case.  Now, the edge (i,j) of weight w is the last edge in a path from
;;;--         j to j iff the max weight path from j to i has weight -w (ie, A*[j,i]=-w),
;;;--         because if it were greater than -w then there would be a positive weight
;;;--         path from j to j (which we have forbidden), and if it were less than -w then
;;;--         the max weight of a path j-->...-->i-->j would be less than zero.  Finally,
;;;--         if A[i,j]=w and A*[j,i]=-w then A*[j,j]=0, so we needn't check that. Thus,
;;;--         an edge (i,j) participates in a zero weight cycle iff A[i,j]=-A*[j,i].
;;;--

(defun propogate-A-equalities-to-E-graph$ ()

  (let ((new-equalities-generated nil))

    (dotimes (i A-size$)
      (dotimes (j A-size$)
	(when (and (not (equal i j))
		   (not (eql (aref A*-matrix$ j i) 'minus-infinity))
		   (equal (aref A-matrix$ i j)
			  (- (aref A*-matrix$ j i))))

	  (let (( lit (aref L-matrix$ i j)))
	    (unless (eql (car lit) 'LESS)

	      ;;-- at this point we would like to assert (cadr lit)+1=(caddr lit)
	      ;;-- or (cadr lit)=(caddr lit)-1.  For now, we do nothing.
	      (when (assert-E-graph-equality$ (list (cadr lit) (caddr lit)))
		(setf new-equalities-generated t)))))))

    new-equalities-generated))


;;;-- 
;;;-- norm-simplified-term$(q:term)
;;;--
;;;--     Given a simplified term q, return a normal form for the term.
;;;--     The normal form is a list of two elements (const rest), where
;;;--     const is an integer and rest is a prl term.
;;;--
;;;--     Simplified terms are assumed to be prlterms which together with
;;;--     this function have the property that
;;;--
;;;--         for any q,s:term . 
;;;--             q=s under the ring axioms,
;;;--             qn=norm-simplified-term$(simplify(q)),
;;;--             sn=norm-simplified-term$(simplify(s))  ==> 
;;;--                    const-part(qn)=const-part(sn) &
;;;--                    equal-term(rest-part(qn),rest-part(sn)).
;;;--
;;;--     Also, the normal form has the property that when a term is changed
;;;--     by adding/subtracting an integer the normal form changes only in
;;;--     the constant part.  That is,
;;;--
;;;--         for any q:term, c:integer .
;;;--             equal-term(rest-part(norm-simplified-term$(simplify(q))),
;;;--                        rest-part(norm-simplified-term$(simplify(ADD q c)))
;;;--                       )
;;;--

(defun norm-simplified-term-old$ (q)

    (cond
      ((integer-term-p q)
       (list (integer-of-integer-term q)
	     (natural-number-term 0)))
      
      ((and (add-term-p q)
	    (integer-term-p (leftterm-of-binary-term q)))
       (list (integer-of-integer-term (leftterm-of-binary-term q))
	     (rightterm-of-binary-term q)))
      
      (t (list 0 q))))


(defun norm-simplified-term$ (q);;lal

  (cond
   ((integer-term-p q)
    (list (integer-of-integer-term q)
	  (natural-number-term 0)))
      
   ((and (minus-term-p q)
	 (integer-term-p (term-of-minus-term q)))
    (list (- (integer-of-integer-term (term-of-minus-term q)))
	  (natural-number-term 0)))
      
   ((and (add-term-p q)
	 (integer-term-p (leftterm-of-binary-term q)))
    (list (integer-of-integer-term (leftterm-of-binary-term q))
	  (rightterm-of-binary-term q)))
      
   ((and (add-term-p q)
	 (minus-term-p (leftterm-of-binary-term q))
	 (integer-term-p (term-of-minus-term (leftterm-of-binary-term q))))
    (list (- (integer-of-integer-term (term-of-minus-term (leftterm-of-binary-term q))))
	  (rightterm-of-binary-term q)))
      
   (t (list 0 q))))



;--
;-- do-A-disequalities$()
;--
;--     Check whether the disequalities together with the A-graph are satisfiable.
;--     If not, throw 'GOOD to 'arith-result.
;--

(defun do-A-disequalities$ ()

  (let ((d-edges nil))		;-- the disequality edges built from A-disequalities$

    ;;-- build d-edges by turning each member of A-disqualities$, a pair (a,b) of
    ;;-- simplified terms, into a edge (ai bi c nil) where ai and bi are the rest
    ;;-- parts of the normal forms of a and b, and c is (const a)-(const b).

    (dolist (diseq A-disequalities$)	;-- a member of A-disequalities$, so a two element list (a,b)
      (let ((an (norm-simplified-term$ (car diseq)))	;-- the normal form of a
	    (bn (norm-simplified-term$ (cadr diseq))))		;-- the normal form of b
      (push (list (A-node-index$ (cadr an))
		  (A-node-index$ (cadr bn))
		  (- (car an) (car bn))
		  nil)			;-- the literal doesn't matter now
	    d-edges)))

    (when (not (test-disequality-satisfiability$ d-edges A-edges$))
      ;;-- literal set is not satisfiable, so deduction is valid
      (throw 'arith-result 'GOOD))))


;;;--
;;;-- test-disequality-satisfiability$(disequality-edges:list, edges:list)
;;;--
;;;--     Build a graph of weighted edges from the elements of edges.   Each element
;;;--     of disequality-edges represents a pair of nodes that are disequal.  For each of
;;;--     the (exponential number of) possible orderings of these nodes (x<y or y<x)
;;;--     test if the augmented A-graph is satisfiable and return t if some ordering
;;;--     Is satisfiable, otherwise return nil.  A graph is satisfiable exactly when
;;;--     it contains no cycles of weight > 0.
;;;--

(defun test-disequality-satisfiability$ (disequality-edges edges)

  (if (not (null disequality-edges)) 
      ;;-- try each ordering of the first disequality and test satisfiability of
      ;;-- the remainder under that ordering.
      (let ((first-dis (car disequality-edges))
	    (rest-dis  (cdr disequality-edges)))
	      
	(or (test-disequality-satisfiability$
	     rest-dis
	     (cons (list (car first-dis)
			 (cadr first-dis)
			 (1+ (caddr first-dis))
			 nil)		;-- the literal is irrelevant now
		   edges))
	    (test-disequality-satisfiability$
	     rest-dis
	     (cons (list (cadr first-dis)
			 (car first-dis)
			 (- (1- (caddr first-dis)))
			 nil)		;-- the literal is irrelevant now
		   edges))))

      ;;-- there are no disequalities -- build the graph and check it for positive cycles
      ;;-- by building the transitive closure of the graph in a matrix and checking if the
      ;;-- diagonal ever has an entry > 0.  If this happens, return nil, else t.
      (not (eql (catch 'arith-result (build-matrices$ edges)) 'GOOD))))




;;;--------------------------------;
;;;                                ;
;;;      E-graph Interface         ;
;;;                                ;
;;;--------------------------------;

;;;--
;;;-- init-E-graph$()
;;;--
;;;--     Initialize the equality box.
;;;--

(defun init-E-graph$ ()
    (init-equality))


;--
;-- assert-E-graph-equality$(termlist:list of prlterms)
;--
;--     Assert the relation t=..=t' in T in the E-graph.  Return t if this was 
;--     not previously known to the E-graph, nil otherwise.
;--

(defun assert-E-graph-equality$ (termlist)

  (let ((rep (E-graph-class$ (car termlist))))

    (do ((terms (cdr termlist) (cdr terms))
	 (term (car termlist) (car terms)))
      
	((null terms) nil)
    
      (when (not (eql rep (E-graph-class$ (car terms))))
	(do-equality (cons term terms))
	(return t)))))




;;;
;;;
;;;

(defun make-n-lambda-lambda-term$ (term n id)
  (if (zerop n)
      (axiom-term)
      (lambda-term id
		   (make-n-lambda-lambda-term$ term (1- n) id))))


(defun ext-arith (pt)
  (let ((concl (conclusion-of-proof-node pt))
	(id (get-dummy-variable-id)))
    (if (union-term-p concl)
	(ext-arith-union$ concl id)
	(let ((term-not-num-list (processable-arith-term$ concl 0)))
	  (if term-not-num-list
	      (make-n-lambda-lambda-term$ (car term-not-num-list)
					  (cadr term-not-num-list)
					  id)
	      ;; assumptions were contadictory, so can extract anything
	      (axiom-term))))))



(defun ext-arith-union$ (term id)
  (if (union-term-p term)
      (make-arith-ext-term$ (processable-arith-term$ (lefttype-of-union-term term) 0)
			    (ext-arith-union$ (righttype-of-union-term term) id)
			    id)
      (let ((term-not-num-list (processable-arith-term$ term 0)))
	(when term-not-num-list
	  (make-n-lambda-lambda-term$ (car term-not-num-list)
				      (cadr term-not-num-list)
				      id)))))



(defun make-arith-ext-term$ (term-not-num-list ext-term id)
  (if (null term-not-num-list)
      (if (null ext-term)
	  (axiom-term)
	  (inr-term ext-term))
      (cond
	((null ext-term)
	 (inl-term (make-n-lambda-lambda-term$ (car term-not-num-list)
					       (cadr term-not-num-list)
					       id)))
	((equal-term-p (car term-not-num-list))
	 (if (evenp (cadr term-not-num-list))
	     (int-eq-term (car (terms-of-equal-term (car term-not-num-list)))
			  (cadr (terms-of-equal-term (car term-not-num-list)))
			  (inl-term (make-n-lambda-lambda-term$ (car term-not-num-list)
								(cadr term-not-num-list)
								id))
			  (inr-term ext-term))
	     (make-diseq-arith-ext$ (car term-not-num-list)
				    (terms-of-equal-term (car term-not-num-list))
				    (cadr term-not-num-list)
				    ext-term
				    id)))
	(t (if (evenp (cadr term-not-num-list))
	       (less-term (leftterm-of-less-than-term (car term-not-num-list))
			  (rightterm-of-less-than-term (car term-not-num-list))
			  (inl-term (make-n-lambda-lambda-term$ (car term-not-num-list)
								(cadr term-not-num-list)
								id))
			  (inr-term ext-term))
	       (less-term (leftterm-of-less-than-term (car term-not-num-list))
			  (rightterm-of-less-than-term (car term-not-num-list))
			  (inr-term ext-term)
			  (inl-term (make-n-lambda-lambda-term$ (car term-not-num-list)
								(cadr term-not-num-list)
								id))))))))



(defun processable-arith-term$ (term curr-num-nots)
  (declare (special type-int$))

  (cond
    ((not-term-p term) 
     (processable-arith-term$ (not-term-body$ term) (1+ curr-num-nots)))

    ((less-than-term-p term)
     (list term curr-num-nots))

    ((and (equal-term-p term)
	  (equal-terms-p (type-of-equal-term term) type-int$)
	  (or (evenp curr-num-nots)
	      (oddp curr-num-nots)))
     (list term curr-num-nots))

    (t nil)))



(defun make-diseq-arith-ext$ (orig-term diseq-terms num-nots ext-term id)
  (if (= (length diseq-terms) 2)
      (int-eq-term (car diseq-terms)
		   (cadr diseq-terms)
		   (inr-term ext-term)
		   (inl-term (make-n-lambda-lambda-term$ orig-term num-nots id)))
      (int-eq-term (car diseq-terms)
		   (cadr diseq-terms)
		   (make-diseq-arith-ext$ orig-term
					  (cdr diseq-terms)
					  num-nots
					  ext-term
					  id)
		   (inl-term (make-n-lambda-lambda-term$ orig-term num-nots id)))))


