%
*************************************************************************
*                                                                       *
*    Copyrighted Cornell University 2000                                *
*                                                                       *
*                                                                       *
*                Nuprl Proof Development System                         *
*                ------------------------------                         *
*                                                                       *
*   Developed by the Nuprl group, Department of Computer Science,       *
*   Cornell University, Ithaca NY.  See the release notes for a list    *
*   of the members of the group.                                        *
*                                                                       *
*   Permission is granted to use and modify Nuprl provided this notice  *
*   is retained in derived works.                                       *
*                                                                       *
*                                                                       *
*************************************************************************
%

%[
****************************************************************************
****************************************************************************
ALGEBRA-CONV.ML
****************************************************************************
****************************************************************************
Conversions useful for algebra.
]%


%[
***************************************************************************
Right associating an operator.
***************************************************************************
Strategy:
  Assume have

    AssocC : (a + b) + c ==> a + (b + c)

  for some binary operator +.

  Work top down. At each level repeat applying AssocC. 
  This should put all nested occurrences of + in right associated form.
]%

let RAssocC AssocC = TryC (SweepDnC (RepeatC AssocC)) ;;


%[
***************************************************************************
Sorting elements in assoc-comm list.
***************************************************************************

Strategy:
  Take as input:
   1. two conversions
    EndSwapC : (a + b) ==>  (b + a)
    InsideSwapC : a + (b + c) ==> b + (a + c)

   2. destructor function for + which fails if not applicable
    dest_op : term -> (term # term)

   3. term ordering function
    tm_lt : term -> term -> bool

  Assume all nested occurrences of + are right associated.
  Use bubblesort to order elements in right associated lists least to 
  greatest.


Usage summary:

  BubbleSortC
    (a + b) ==>  (b + a)
    a + (b + c) ==> b + (a + c)
    dest_op
    tm_lt

]%


let BubbleSortC EndSwapC InsideSwapC dest_op tm_lt = 
  let is_op t = not (fails_p dest_op t)
  in

  % Try swap. Only succeeds if swap applicable and OK by tm_lt %

  let SwapC e t = 
    let ta,tbc = dest_op t
    in
    if is_op tbc then
    ( let tb,tc = dest_op tbc
      in
      if tm_lt tb ta then
        InsideSwapC e t      
      else 
        FailC e t
    )
    if tm_lt tbc ta then
      EndSwapC e t
    else
      FailC e t
  in
    RepeatC (SweepDnC SwapC)
;;

%[
***************************************************************************
Eliminating Identities
***************************************************************************
Assume Trees are right associated.

Strategy:

  Input 2 conversions
  1. IdLeftC id x a = a
  2. IdRightC a x id = a


  Use 1st on single traverse down, then 2nd.


Usage summary:

  MonIdentElimC
    id x a = a
     a x id = a
]%

let MonIdentElimC IdLeftC IdRightC = 
  TryC (SweepDnC (RepeatC IdLeftC))
  ANDTHENC TryC (SweepDnC IdRightC)
;;


%[
***************************************************************************
Monoid Normalization Functions:
***************************************************************************
Usage Summary:


MonoidNormC
    (a + b) + c = a + (b + c)
    id x a = a
    a x id = a

AbMonoidNormC
    (a + b) + c = a + (b + c)
    id x a = a
    a x id = a
    (a + b) =  (b + a)
    (a + b) + c = (b + a) + c
    dest_op
    tm_lt
]%

let MonoidNormC AssocC IdLeftC IdRightC =
  RAssocC AssocC 
  ANDTHENC MonIdentElimC IdLeftC IdRightC
;;

let AbMonoidNormC AssocC IdLeftC IdRightC EndSwapC InsideSwapC dest_op tm_lt = 
  RAssocC AssocC 
  ANDTHENC MonIdentElimC IdLeftC IdRightC
  ANDTHENC BubbleSortC EndSwapC InsideSwapC dest_op tm_lt 
;;


%[
***************************************************************************
Eliminating Inverses and Identities
***************************************************************************
Precond:

Trees are right associated, with 
  a) all invs below all ops
  b) multiple invs cancelled
  c) inv(id)'s reduced to ids.


Strategy:

  Input ? conversions
  1. IdLeftC id x a = a
     IdRightC a x id = a

  2. InvLeftC          i(a) x a = id
     InsideInvLeftC    i(a) x (a x b)  = b
     InvRightC           a  x i(a) = id
     InsideInvRightC     a  x (i(a) x b) = b


  Do:

  1. SweepDnC RepeatC with 1a&b
  2. 
]%

let GrpInvIdentElimC 
  InvLeftC InvRightC InsideILeftC InsideIRightC 
  IdLeftC IdRightC
  = 
  let AuxC = 
    TryC (SweepUpC (FirstC [InsideILeftC;InsideIRightC;InvLeftC;InvRightC]))
  in
    RepeatC (AuxC ANDTHENC MonIdentElimC IdLeftC IdRightC)
;;

%[
***************************************************************************
Group Normalization Conversion
***************************************************************************
Strategy:

1. SweepDn with -(a x b) = (-b x -a)
2. SweepDn with --a = a and -1 = 1
3. right associate x 
4. elim inverse pairs  and identity els.

Usage summary:

  GroupNormC
    -(a x b) = (-b x -a)
    --a = a
     -e = e
    (a + b) + c = a + (b + c)
    i(a) x a = id
    a  x i(a) = id
    i(a) x (a x b)  = b
    a  x (i(a) x b) = b
    id x a = a
    a x id = a
]%

let GroupNormC 
  InvThruOpC  InvInvC InvIdC
  AssocC
  InvLeftC InvRightC InsideInvLeftC InsideInvRightC
  IdLeftC IdRightC
  =
  TryC (SweepDnC InvThruOpC)
  ANDTHENC TryC (SweepDnC (InvInvC ORELSEC InvIdC))
  ANDTHENC RAssocC AssocC
  ANDTHENC GrpInvIdentElimC 
             InvLeftC InvRightC InsideInvLeftC InsideInvRightC 
             IdLeftC IdRightC
;;


%[
***************************************************************************
Abelian Group Normalization Conversion
***************************************************************************

Usage summary:

  AbGroupNormC
    -(a + b) = (-b + -a)
    --a = a
     -e = e
    (a + b) + c = a + (b + c)
    (a + b) =  (b + a)
    (a + (b + c) = b + (a + c)
    dest_op
    dest_inv
    tm_lt
    -a + a = e
    a  + -a = e
    -a + (a + b)  = b
    a  + (-a + b) = b
    e + a = a
    a + e = a
]%

% dest_inv destructs the inverse constructor %

% makes `a' immediately less than `-a' %

let term_lt_for_grps tm_lt dest_inv t1 t2 = 
  let analyze_tm t = (true,dest_inv t ? false, t)
  in let is_inv1,t1' = analyze_tm t1
  in let is_inv2,t2' = analyze_tm t2
  in if alpha_equal_terms t1' t2' then
    not is_inv1 & is_inv2
  else
   tm_lt t1' t2'
;;



let AbGroupNormC 
  InvThruOpC  InvInvC InvIdC
  AssocC
  EndSwapC InsideSwapC dest_op dest_inv tm_lt 
  InvLeftC InvRightC InsideInvLeftC InsideInvRightC
  IdLeftC IdRightC
  =
  TryC (SweepDnC InvThruOpC)
  ANDTHENC TryC (SweepDnC (InvInvC ORELSEC InvIdC))
  ANDTHENC RAssocC AssocC
  ANDTHENC BubbleSortC EndSwapC InsideSwapC dest_op 
   (term_lt_for_grps tm_lt dest_inv)
  ANDTHENC GrpInvIdentElimC 
             InvLeftC InvRightC InsideInvLeftC InsideInvRightC 
             IdLeftC IdRightC
;;


%[
***************************************************************************
Ring Normalization Functions:
***************************************************************************
3 Parts:

1. Push all x's below all +'s and -'s,
   (noting that 0 is an annihilator)
2. Normalize (abelian?) monoid <*,1>
3. Normalize abelian group <+,0,->

Lemmas for part 1:

TimesThruPlusL (a + b) x c = a x c + b x c
TimesThruPlusR a x (b + c) = a x b + a x c

TimesThruMinusL (-a) x b = -(a x b)
TimesThruMinusR a x (-b) = -(a x b)

TimesZeroL 0 x a = 0
TimesZeroR a x 0 = 0

Usage summary:

  RngTimesPushdownC  
    (a + b) x c = a x c + b x c
    a x (b + c) = a x b + a x c
    (-a) x b = -(a x b)
    a x (-b) = -(a x b)
    0 x a = 0
    a x 0 = 0
]%

let RngTimesPushdownC  
  TimesThruPlusLC
  TimesThruPlusRC
  TimesThruMinusLC 
  TimesThruMinusRC
  TimesZeroLC
  TimesZeroRC
  = 
  SweepDnC 
  (FirstC 
     [TimesThruPlusLC
     ;TimesThruPlusRC
     ;TimesThruMinusLC 
     ;TimesThruMinusRC
     ]
  )
  ORTHENC
  SweepUpC 
  (FirstC 
     [TimesZeroLC
     ;TimesZeroRC
     ]
  )
;;

% use abelian monoid norm conversion,
  if ring is commutative.
%

let RingNormC 
  TimesPushdownC
  TimesMonoidNormC
  PlusAbGroupNormC
  = 
  TryC
   (SomeC
     [TimesPushdownC
     ;TimesMonoidNormC
     ;PlusAbGroupNormC
     ]
   )
;;

%[
***************************************************************************
Monoid Normalization Functions:
***************************************************************************
]%
