%
*************************************************************************
*                                                                       *
*    Copyrighted Cornell University 2000                                *
*                                                                       *
*                                                                       *
*                Nuprl Proof Development System                         *
*                ------------------------------                         *
*                                                                       *
*   Developed by the Nuprl group, Department of Computer Science,       *
*   Cornell University, Ithaca NY.  See the release notes for a list    *
*   of the members of the group.                                        *
*                                                                       *
*   Permission is granted to use and modify Nuprl provided this notice  *
*   is retained in derived works.                                       *
*                                                                       *
*                                                                       *
*************************************************************************
%

%[
*****************************************************************************
*****************************************************************************
environments 
*****************************************************************************
*****************************************************************************
Environments are lists of both bindings for variables and assumptions. 
They are used principally by the get_type routines, and by the rewrite package.
Since in rewriting we need to regenerate hyp lists from environments, we
include info about hidden assumptions. For efficiency, the bindings are
maintained in reverse order.
]%

abstype env = (var # term # bool) list
  with form_env x = abs_env x 
   and open_env x = rep_env x
;;

let env_to_term e = 
 map_to_ilist 
  (\v,t,b. make_term (`env`, [make_variable_parameter v; make_bool_parameter b])
        [[],t])
  (`env_cons`,nil) 
  (open_env e)
;;

let null_env = 
  form_env [] 
;;

let extend_env e (v,t,b) = 
  form_env ((v,t,b). open_env e)
;;

% vtbs should have leftmost bindings first %

let list_extend_env e vtbs = 
  form_env (rev vtbs @ open_env e) ;;

let decls_to_env ds =
  form_env (rev (map dest_declaration ds)) 
;;

let stdize_rev_bindings bs = 

  letrec change_var old_v new_v_tm ds =
    if null ds then []
    else
    let (v,t,b).ds' = ds 
    in 
    if v = old_v then
      (v,subst [old_v,new_v_tm] t,b).ds'
    else
      (v,subst [old_v,new_v_tm] t,b).change_var old_v new_v_tm ds'
  in

  % typical call:
     fix_vs [di;...;dn] [di-1';...;d1']
 
    Where d1;...;dn is conventional (outside in) order for declarations.
  %

  letrec fix_vs old_ds new_ds =
    if null old_ds then new_ds
    else
    let (v,t,b).old_ds' = old_ds 
    in let new_vs = map fst new_ds 
    in 
    if is_null_var v then
    ( let v' = new_var basic_invisible_var new_vs 
      in fix_vs old_ds' ((v',t,b).new_ds)
    )
    if member v new_vs then
    ( let v' = new_var v new_vs 
      in let old_ds'' = change_var v (mk_var_term v') old_ds'
      in fix_vs old_ds'' ((v',t,b).new_ds)
    )
    else
      fix_vs old_ds' ((v,t,b).new_ds) 
  in
    fix_vs (rev bs) []
;;

let env_to_decls e = 
  map
    mk_declaration 
    (rev (stdize_rev_bindings (open_env e)))
;;


let mk_sequent_using_env e conc =
  make_proof_node
    (env_to_decls e)
    conc
;;

let env_of_sequent p = decls_to_env (hyps p)
;;




%
Allow easy user extension of environment update function.
%

let mk_sub_envs_for_typed_binder_term e t =
  let [[],A;vs,B] = bterms_of_term t in
       [e; list_extend_env e (map (\x.x,A,false) vs)]
;;


%
NB. new_invisible_var might be a rather costly. Does lisp check distinct
variable naming whenever proof node constructed?
%

let mk_implies_sub_envs e t =
  let [[],A;[],B] = bterms_of_term t 
  in
    [e;extend_env e (null_var,A,false)]
;;

let initial_sub_env_cache =
  [`product`,mk_sub_envs_for_typed_binder_term
  ;`function`,mk_sub_envs_for_typed_binder_term
  ;`exists`,mk_sub_envs_for_typed_binder_term
  ;`all`,mk_sub_envs_for_typed_binder_term
  ;`implies`,mk_implies_sub_envs
  ]
;;

%
letref sub_env_cache = initial_sub_env_cache
;;

let add_env_update_fun opid fun =
  sub_env_cache := update_alist sub_env_cache opid fun ; ()
;;
%
 
letref sub_env_assoc =
  new_ref_state `sub_env`
    flatten
    (\data gedata. append gedata data)
    (nil : (object_id # (tok # (env -> term -> env list)) list) list)
    (nil : (tok # (env -> term -> env list)) list)
;;

let ref_add_sub_env_aux index edges items =
 sub_env_assoc 
   := declare_ref_state_data_indirect `sub_env` sub_env_assoc index items edges
;;

let ref_add_sub_env index edges items =
 declare_ref_state_index index `sub_env` index;
 sub_env_assoc 
   := declare_ref_state_data_indirect `sub_env` sub_env_assoc index items edges
;;

let ref_add_sub_env_additions items index edges =
 sub_env_assoc
   := ref_state_modify_state_aux (\data gedata. append gedata data)
         sub_env_assoc items index edges
 ; ()
;;

let sub_env_add_data oid data =
 sub_env_assoc := ref_state_set_data sub_env_assoc [oid, data]
;;

let SubEnv_add oid data = 
 reset_ref_environment_data oid;
 add_ref_environment_data oid `sub_env` sub_env_add_data data
;;

let sub_env_do_updates oid edges oids = 
 sub_env_assoc := ref_state_do_updates sub_env_assoc oid oids edges
; ()
;;
let undeclare_sub_env oid = 
 (sub_env_assoc := ref_state_remove sub_env_assoc oid; ())
 ? ()
;;
	 
update_ref_state_view
 (\(). sub_env_assoc)
 (ref_state_view_list_entry (itoken_term o fst))
;;     

let lookup_sub_env_alist () = 
  ref_state_get sub_env_assoc (current_ref_environment_index `sub_env`)
;;

letref find_sub_env = (\t. fail) : (tok -> (env -> term -> env list));;

let build_find_sub_env uoid =
  find_sub_env := if (isl uoid) then (\t. fail)
                  else apply_alist (ref_state_get sub_env_assoc (outr uoid))
  ; ()
;;

update_ref_environment_cache_hook `sub_env` build_find_sub_env ;;
       
let lookup_sub_env id  = apply_alist (lookup_sub_env_alist ()) id;;



let list_env_update_funs () = map fst (lookup_sub_env_alist ()) ;;
%let reset_env_update_funs () = sub_env_cache := initial_sub_env_cache ; () ;;
%
  
let get_sub_envs e t = 
    (find_sub_env (opid_of_term t)) e t
  ?
    replicate e (length (bterms_of_term t))
;;


% copes with -ve indices... %

let get_pos_env_num i e = 
  if i > 0 then i else length (open_env e) + i + 1
;;


let nth_assum_in_env n e = 
  let as = rev (open_env e) in 
( if not n = 0 then 
    nth (get_pos_env_num n e) as
  else
    fail
) ? failwith `nth_assum_in_env: index out of range`
;;

let lookup_var_type_in_env e v = 
  let xAhs = open_env e 
  in 
    fst (apply_alist xAhs v) ? failwith `lookup_var_type_in_env`
;;


%
consider:

#i-1: y:T
#i: All xAs. a[xs,y] R b[xs,y]

...
|- All y:T. a[??,y]

And using hyp i as a rewrite rule.
Variables free in hyp i but treated as constants for rewriting purposes
(such as y above) can creep into bad contexts where the declaration
of these vars is shadowed (as if one did (RWH (HypC i) 0)).

This function is used to avoid this situation.

Assume vs are free vars in rewrite relation (excluding those in context of
reln)
Returns true just when some part of env beyond position n
shadows one or more vars in vs.

%

let env_shadows_vars e n vs = 
  let n' = get_pos_env_num n e in
  let vs' = (map fst o nthtl n' o rev o open_env) e 
  in
    not null (intersection vs vs')
;;


%
Sometimes environments of terms when justifications are run are
different from when justifications are generated.

This function makes a plausible guess at how a term must be fixed
up to correct for these differences. For it to work, rewrite 
environments must be updated when descending under binders in same
way that they are updated by functionality lemmas, since
it is these binders that might change.

Could try stripping out null_vars from e,
invisible_vars from e' to allow for minor differences in environments.

substitution is reversed so that when vars are shadowed in e, the innermost
var is picked from e'. (when multiple bindings for a variable occur in a sub list, 
subst gives precedence to the latest binding.)

%

let term_in_env_fixup e t e' = 
  let get_vars ee = map fst (open_env ee) in
  let vs = get_vars e in
  let vs' = get_vars e' in
  if not length vs = length vs' or vs = vs' then t else 
  let sub = zip vs (map mvt vs') 
  in
    subst (rev sub)  t 
;;
