%[
***************************************************************************
***************************************************************************
rel_rst.ml
***************************************************************************
***************************************************************************
Functions here used for equivalence and order relation reasoning.
]%

%[
***************************************************************************
Graph-related functions
***************************************************************************
]%
%
v : int (vertex)

(from,to),(ename,kind) : (edge)

A (directed) Graph g is a list of edges

edges come in 2 kinds: `lt` and `le` 

The `lt` edges are sometimes called strict edges.
%

%
find_next_edges_*** g k' v

input: v:vertex
output: list of triples of form t,(e,k')

  the t is a vertex reachable by edge e of kind ***
  the edge e is paired with a new kind k' that indicates
  the kind of the whole path from the start vertex.
%

let find_next_edges_lt_le g k' v =
  mapfilter (\(f:*,t:*),(e,k:tok).if f = v then t,(e,k':tok) else fail) g
;;
let find_next_edges_lt g k' v =
  mapfilter 
    (\(f:*,t:*),(e,k).if f = v & k = `lt` then t,(e,k':tok) else fail) g
;;
let find_next_edges_le g k' v =
  mapfilter 
    (\(f:*,t:*),(e,k).if f = v & k = `le` then t,(e,k':tok) else fail) g
;;

let find_edge_vertices g e = 
  fst (find (\(():*,():*),(e',k).e' = e) g) ;;
let find_from_vertex g e = fst (find_edge_vertices g e) ;;

%
an `edge_set' is a list of vertex,e-name,kind trips

input: edge_set
       old_vs  : list of already visited vertices 

output: edge_set

  max one entry per vertex
%

letrec prune_to_edges old_vs edge_set = 
  if null edge_set then []
  else 
  let (v,ek).edge_set' = edge_set in
  if member v old_vs then
    prune_to_edges old_vs edge_set'
  else
    (v,ek) . prune_to_edges (v.old_vs) edge_set'
;;


%
The idea here is to build a list of pairs of edge_sets:

  es_le1  es_le2 ...

  es_lt1  es_lt2 ...

  where a triple v,e,k in es_*i indicates that one of the shortest paths of 
  kind * to vertex v from the start vertex passed last through edge e
  and that the path just before this edge is of kind k.

form_new_edge_sets 
  constructs a pair

    es_lei+1

    es_lti+1

  given input

    old_vs_le   start_vs_le

    old_vs_lt   start_vs_lt

  where the start_vs_le (start_vs_lt) give the vertices in es_lei (es_lti)
  and the old_vs_le (old_vs_lt) give the vertices in es_le1...eslei
  (es_lt1...eslti) 

  (so start_vs_* are always a subset of the old_vs_*)

%

let form_new_edge_sets g (old_vs_le,old_vs_lt) (start_vs_le,start_vs_lt) = 
  let new_e_v_prs_le = flatten (map (find_next_edges_le g `le`) start_vs_le)
  in let new_e_v_prs_lt = 
      flatten (map (find_next_edges_lt g `le`) start_vs_le)
      @ flatten (map (find_next_edges_lt_le g `lt`) start_vs_lt)
  in
    prune_to_edges old_vs_le new_e_v_prs_le
    ,prune_to_edges old_vs_lt new_e_v_prs_lt
;;


% constructs list of edge_sets in reverse order %



let build_paths g start_v end_v end_kinds =

  letrec aux 
           edge_set_le_lt_list 
           (old_vs_le,old_vs_lt)
           (start_vs_le,start_vs_lt)
    =
    if member `le` end_kinds & member end_v start_vs_le 
       or member `lt` end_kinds & member end_v start_vs_lt 
    then 
      edge_set_le_lt_list 
    else

    let new_edge_set_le,new_edge_set_lt = 
      form_new_edge_sets g (old_vs_le,old_vs_lt) (start_vs_le,start_vs_lt)
    in
    if null new_edge_set_le & null new_edge_set_lt then
      failwith `find_shortest_path: no path found`
    else
      aux  
        ((new_edge_set_le,new_edge_set_lt).edge_set_le_lt_list) 
        (start_vs_le @ old_vs_le ,start_vs_lt @ old_vs_lt) 
        (map fst new_edge_set_le ,map fst new_edge_set_lt)
  in
    aux [] ([start_v],[]) ([start_v],[]) 
;;

let extract_path g edge_set_le_lt_list end_v end_kinds = 
  letrec aux edge_sets_list v kinds edges = 
    if null edge_sets_list then edges
    else
    let (es_le,es_lt).edge_sets_list' = edge_sets_list
    in let edge,prev_kind = 
      (if member `le` kinds & member `lt` kinds then
         (apply_alist es_le v ? apply_alist es_lt v)
       if member `lt` kinds then
         apply_alist es_lt v
       if member `le` kinds then
         apply_alist es_le v
       else
         failwith `extract_path: bad kinds`
      )
    in let edge_from_v = find_from_vertex g edge
    in 
      aux edge_sets_list' edge_from_v [prev_kind] (edge.edges)
  in
    aux edge_set_le_lt_list end_v end_kinds []
;;
  
let find_shortest_path g start_v end_v kinds = 
  let edge_set_list = build_paths g start_v end_v kinds
  in 
    extract_path g edge_set_list end_v kinds
;;

%
let test_graph = 
 [(1,2),`a`,`le`
 ;(2,3),`b`,`lt`
 ;(3,4),`c`,`le`
 ;(4,7),`d`,`le`
 ;(2,5),`e`,`le`
 ;(5,6),`f`,`le`
 ;(6,4),`g`,`le`
 ;(6,6),`h`,`lt`
 ]
;;
%
%[
***************************************************************************
Extract equivalence and order relation info from Sequent.
***************************************************************************

dest_known_rel_term
~~~~~~~~~~~~~~~~~~~~
Destructs a known relation R or negation of known relation.

Input:  t  : term        (clause of sequent of form ta R tb or not(ta R tb))
        e : env          (env of sequent that t comes from.
                              Needed to find if R lin.) 
        concl_p : bool   (true iff destructing conclusion)

Output: (iproperties @ rproperties @ oproperties)
        ,rootname
        ,(t1,t2)

  where:

    t1:term 
    t2:term 
        t1 is left-hand-arg when R is put in standard sense 
        t2 is right-hand-arg when R is put in standard sense 
        (a < b is in standard sense , b > a is in reverse sense) 
    rootname:tok
        name of root equivalence relation for R.
          (root relation is always an equiv reln. 
           e.g. root of => is <=>, root of <z and =<z is =z)

    iproperties: tok list 
      list of `instance-properties' of R. These describe ways in which the
      relation has been normalized as it has been destructed.

      possible elements:
      `rev`  - The reversing operation has been done on relation
               in reverse sense. (only used for non-symmetric rels)
               e.g. the relation 2 > 1 is the reversed sense of 1 < 2
      `comp` - complement operation has been done on relation. 
               (only valid for non-symmetric linear rels)
               Complement understood as being done *after* elimination of
               reverse sense relations. 

    rproperties: tok list
      list of `relation-properties' of R. These describe intrinsic properties
      of the normalized relation.
      
      `neg`  - negated with respect to hyps
               (an un-negated concl is considered to be a negated hyp)
      `refl` - reflexive
      `sym`  - symmetric

    oproperties: tok list
      list of `order properties' whether the order is 
      a linear order or not.

      `lin`  - linear         

Function fails if relation not recognized.

Complement operation is done on non-symmetric negated linear hyps 
]%
  
let dest_known_rel_term e concl_p t =
  let tr,neg_p = 
  (if is_term `not` t then sub_term 1 t,true
   if is_term `nequal` t then mk_simple_term `equal` (subterms t),true
   else t,false
  )
  in
  let r,ta,tb = dest_rel_term_without_check tr in
  let kind,(),() = identify_rel r in
  let sym_p = kind = 0 in
  let refl_p = -2 < kind & kind < 2 in
  let rev_p = kind < 0 in
  let lin_p = is_linear_rel e r in 
  let comp_p = lin_p & neg_p & not sym_p & not concl_p in

  % below, if comp_p true, several things are flipped around %

  let root_r = get_root_rel r in
  let t1,t2 = rev_p xor comp_p => (tb,ta) | (ta,tb) in
  let properties = 
   (refl_p xor comp_p => [`refl`] | [])
   @
   (sym_p => [`sym`] | [])
   @
   (rev_p => [`rev`] | [])
   @
   (neg_p xor concl_p xor comp_p => [`neg`] | [])
   @ 
   (lin_p => [`lin`] | [])
   @
   (comp_p => [`comp`] | [])
  in
    properties,name_of_rel root_r,t1,t2
;;



% 
rel_rst_proc_sequent
~~~~~~~~~~~~~~~~~~~~~
Input:   p:proof
Output: 
  list of info on relations in sequent.
  Each element of form:

   ps,r,i,v1,v2 
  where 
    ps: tok list - list of properties. see dest_known_rel_term 
     r: tok =  root relation name
     i:int  -  # of clause containing relation
     v1 v2:int - are numbers identifying the relation arguments
% 

let rel_rst_proc_sequent p = 
  let e = env_of_sequent p in
  let cls =  ([0,dest_known_rel_term e true (concl p)] ? [])
                @ mapfilter (\i,h.i,dest_known_rel_term e false (type_of_declaration h))
                            (number (hyps p))
  in let tms = flatten (map (\(),(),(),a,b.[a;b]) cls)
  in let convert_tms (i,ps,rnam,a,b) = ps,rnam,i,term_position a tms,term_position b tms
  in 
    map convert_tms cls
;;

%[
***************************************************************************
Process relational problem.
***************************************************************************
A problem takes the form of
  1. r: a relation to focus on
& 2. rs: a list of relations

Relations in format as output by rel_rst_proc_sequent

rs are all `pos' relations. They provide a context. They are the set of the
relations being assumed true.
 r is a relation which determines problem kind and hence method of 
attempted solution. If r is also a `pos' relation, then solution sought
by contradiction. If r is a negated relation, then solution sought by 
trying to derive r from relations in context by chaining them together
using transitivity rules.

Output takes the form of a list of relation `names', where name is
pair of number and direction.  Number is clause #. Direction is token:
either `same' or `rev'). Direction indicates whether clause in sequent
needs reversing to get it in right sense for chain.  

Currently, the options for `r' are as follows:

1. r is un-negated and irrefl.

   r := a < b

   method searches for `le' or `lt' chain from b to a, 
   when graph composed from all rels no weaker than r.

   returns list of names of relations in chain, including relation r
   to start.

2. r is negated

  a) r := a == b (related by symmetric reflexive relation)

   method searches for `le' chain from a to b, when graph composed from
   all symmetric rels no weaker than r.
   returns list of names of relations in chain. If a is identical to b,
   then returns empty chain.

  b) r := a =< b (related by non-symmetric reflexive relation)

   method searches for `le' or `lt' chain from a to b, when graph composed 
   from all rels no weaker than r.
   returns list of names of relations in chain. If a is identical to b,
   then returns empty chain.
  
  c) r := a < b (related by non-symmetric irreflexive relation)

   method searches for `lt' chain from a to b, when graph composed from
   all rels no weaker than r.
   returns list of names of relations in chain. Chain always must be 
   at least one element long.


Known obvious missing options are:

2 d) r := a == b 

   r can be split into two relations: a =< b and b =< a
   (doesn't work for quasi-orders, since anti-symmetry missing)

2 e) order r is over is linear. In this case can exploit conversions:

    not(a < b)  -->  b =< a
    not(a =< b) -->  b < a


edge names:

  are pair i,op : int # tok

  where 
   i is the # of the clause containing the edge
   op indicates a preparatory operation to do on clause 

   current ops are:
   `rev`  reverse the sense of the relation 
   `comp` complement the relation 
   `revcomp` reverse and complement
   `none` 
]%


let rel_rst_mk_graph
  c_rel  % select only edges with stronger or equal root strength %
  hyp_i_rel_prs
  refl_p % refl_p = true: select only reflexive edges %
  sym_p  % sym_p =  true: select only symmetric edges %
  =
  let extract_edge_info (h_props,h_str,hnam,ht1,ht2) = 
    let edge_kind = if member `refl` h_props then `le` else `lt`    
    in
    let opa,opb = if member `comp` h_props then 
                    `comp`,`revcomp`
                  else
                    `none`,`rev`
    in let enam1,enam2 = if member `rev` h_props then 
                           (hnam,opb),(hnam,opa)
                          else 
                           (hnam,opa),(hnam,opb)
    in
    if not rel_less_or_eq (form_untyped_rel h_str) (form_untyped_rel c_rel) 
       or refl_p & not member `refl` h_props
       or sym_p & not member `sym` h_props
    then
      fail
    if member `sym` h_props then
      [(ht1,ht2),enam1,edge_kind ;(ht2,ht1),enam2,edge_kind]
    else
      [(ht1,ht2),enam1,edge_kind]
  in
    flatten (mapfilter extract_edge_info hyp_i_rel_prs)
;;



let rel_rst_solve_problem env_rs focus_r = 
  let f_props,f_str,f_name,v1,v2 = focus_r in
  let mk_graph = rel_rst_mk_graph f_str env_rs in

  if not member `neg` f_props & not member `refl` f_props then
  % case 1: v1 < v2 %

    (f_name, member `rev` f_props => `rev` | `same`) 
    . find_shortest_path (mk_graph false false) v2 v1 ``le lt``

  if not member `neg` f_props then 
    failwith `rel_rst_solve_problem: unrecognised problem 1`
  if subset ``sym refl`` f_props then
  % case 2a: not(v1 == v2) %

    find_shortest_path (mk_graph true true) v1 v2 ``le``

  if member `refl` f_props then
  % case 2b: not(v1 =< v2) %

    find_shortest_path (mk_graph false false) v1 v2 ``le lt``

  else
  % case 2c: not(v1 < v2) %

    find_shortest_path (mk_graph false false) v1 v2 ``lt``
;;

% assume rs as output by proc sequent. Returns relation focussed on
and list of names of edges from environment to chain together %

let rel_rst_find_solution rs = 
  let neg_rs,env_rs = divide_list (member `neg` o fst) rs in
  let strict_env_rs = remove_if (member `refl` o fst) env_rs in
  let split_sym_rel_rs (props,root_r,nam,t1,t2) =
    if member `sym` props then
      (remove `sym` props,root_r,nam,t1,t2)
      ,(remove `sym` props,root_r,nam,t2,t1)
    else
      fail
  in
  let sym_neg_rs_a,sym_neg_rs_b =
    unzip (mapfilter split_sym_rel_rs neg_rs)
  in
   [first_value 
       (\r.r,rel_rst_solve_problem env_rs r) 
       (neg_rs @ strict_env_rs)
   ] ?
   [first_value 
       (\r.r,rel_rst_solve_problem env_rs r) 
       sym_neg_rs_a 
   ;first_value 
       (\r.r,rel_rst_solve_problem env_rs r) 
       sym_neg_rs_b 
   ]
;;

%
***************************************************************************
Tactics
***************************************************************************
%

% 
Yet a few more kinds of property lemmas needed.


assume lemmas in library of form:

1 <opid of R>_irreflexivity

  All xs:As, All y:T Bs => y R y => False

  for R a strict relation.

2. <opid of LER>_antisymmetry

All xs:As, All y,y':T Bs => y LER y' => y' LER y => y EqR y'
  for LER a `less than or equal' relation

3a. <opid of LER>_complement
3b. <opid of LTR>_complement

All xs:As, All y,y':T Bs => not(y LER y') => y' LTR y 
All xs:As, All y,y':T Bs => not(y LTR y') => y' LER y 

  for LER a `less than or equal' relation, and LTR a `less than' relation
  in a linear order.



R can be underneath assert.
(if R is application, should grab opid of head...)
%

let get_rel_opid R = 
  if is_term `assert` R then opid_of_term (sub_term 1 R) else opid_of_term R
;;

let IrreflHyp i p = 
 (FLemma (get_rel_opid (h i p) ^ `_irreflexivity`) [i] THENM FalseHD (-1)) p
;;

let AntiSymConcl p = 
  let r,(),() = dest_rel_term (concl p) in
  let r' = get_order_rel r 
  in
    BLemma (name_of_rel r' ^ `_antisymmetry`) p
;;

let ComplementHyp i p = 
  let i' = get_pos_hyp_num i p 
  in
  SeqOnM
   [FLemma (get_rel_opid (sub_term 1 (h i' p)) ^ `_complement`) [i'] 
   ;MoveToHyp i' (-1)
   ;Thin (i'+1) 
   ] p
;;


%
dir = `rev` means relation needs reversing.
dir = `comp` means relation needs complementing
dir = `revcomp` means relation needs reversing and complementing

let soft matching do reversing for revcomp option. (knowing 
this option only applied to order relations and that they 
always end up in standard sense)

Assume at least one element in i_dir_prs

Leaves Join as new last hyp.


%

let JoinRelChain i_dir_prs p = 
  let hs_to_invert = 
    mapfilter (\i,dir.if dir = `rev` then i else fail) i_dir_prs  
  in
  let hs_to_complement = 
    mapfilter 
     (\i,dir.if member dir ``revcomp comp`` then i else fail) i_dir_prs  
  in let hd_h.tl_hs = map fst i_dir_prs
  in let Step i = 
    JoinRels (-1) i THENM Thin (-2)
  in
    SeqOnM 
      (map InvertRel hs_to_invert 
       @ map ComplementHyp hs_to_complement
       @ [CopyToEnd hd_h] @ map Step tl_hs
      ) p
;;



%
To remember:
1. if concl is in rchain, then must have been negated. Need to get it
into hyp list and make sure RSTJoinRels finds it.
2. If negated hyp is focus, then must be made conclusion (by ID to prevent
hyp #'s getting messed up) (however concl is already concl)
3. any reversed focus term must be put in regular sense before 
   doing strengthening.
4. After any chaining together of env, the solution technique is:

  Case 1 (a < b in env)
     chain cannot be empty (in a < a case, chain contains 1 element).
     Use irreflexivity lemma on result of chaining.
  Case 2
   Assume focus has been made concl.
   Focus is:
   a) (neg(a == b))
     if empty chain, then Strengthen concl to equality.
     o/w strengthen concl to result of chaining.
   b) (neg(a =< b))
     if empty chain, then Strengthen concl to equality.
     o/w strengthen concl to result of chaining.
   c) (neg(a < b))
     chain cannot be empty
     strengthen concl to result of chaining.

  Note 2a)b)c) all require compatible actions, so no need to distinguish
  between these cases.

  chaining always leaves result as last hyp
%

let RelReflSymTransAux2 focus_r echain p = 
  let f_props,f_str,f_num,() = focus_r in
  let nconc = num_hyps p in
  let fix_edge (i,dir) = if i = 0 then nconc,dir else i,dir in

  let MaybeChain = 
    if null echain then Id else JoinRelChain (map fix_edge echain)
  in
  let PostTac = 
     if not member `neg` f_props then
       IrreflHyp (-1)
     if null echain then
       StrengthenRel THENM (FoldTop `member` 0 THEN AddHiddenLabel `wf`)
     else
       StrengthenRelToHyp (-1) THENM NthHyp (-1)
  in
   (MaybeChain THENM PostTac) p
;;

let RelReflSymTransAux1 sols p = 
  let focus_r = fst (hd sols) in
  let echains = map snd sols in
  let f_props,f_str,f_num,() = focus_r in
  let nconc = num_hyps p + 1 in
  let fix_edge (i,dir) = if i = 0 then nconc,dir else i,dir in

  let MaybeNegConclToHyp = 
    if member 0 (map fst (flatten echains)) then D 0 else AddHiddenLabel `main`
  in
  let MaybeNegHypToConcl = 
    if member `neg` f_props & not f_num = 0 then ID f_num else Id
  in
  let MaybeInvertFocusR = 
    if subset ``rev neg`` f_props then InvertRel 0 else Id
  in
  let MaybeSplitR = 
    if length echains = 2 then AntiSymConcl else Id
  in
    SeqOnM
     [MaybeNegConclToHyp
     ;MaybeNegHypToConcl
     ;MaybeInvertFocusR
     ;MaybeSplitR THENML (map (RelReflSymTransAux2 focus_r) echains)
     ]
     p
;;

let RelReflSymTrans p = 
  let rs = rel_rst_proc_sequent p in
  let sols = rel_rst_find_solution rs 
  in
    RelReflSymTransAux1 sols p
;;



let RelRST p = RelReflSymTrans p ;;


%
  just-manips using conversions

%

% R is a r b and T is a tactic such that H |- a r b  By T.
  We return R',T' where R' is b r a and H |- b r a By T'
%
let GeneralInvertRel (R, T) =
 let r,a,b = dest_rel_term_without_check R in
 if is_order_rel r then
   let addr = if is_term `assert` R then [1] else [] in
   let cnv = 
     if is_std_order_rel r then
       AddrC addr (FoldTopC (name_of_rel (invert_rel r))) 
     else
       AddrC addr UnfoldTopAbC in
   (apply_conv cnv R, Assert R THENL [T; RW cnv 0])

 else
 let inv_rtm = mk_similar_rel_term R b a in
 let sub_j = form_tactic_just (DebugTry T) in
 let r' = %if i = 0 then invert_rel r else% r in
 let T' = open_tactic_just (get_inversion_just r' sub_j) in

 ( inv_rtm, T')
;;

% R is not (a r b) and T is a tactic such that H |- not (a r b)  By T.
  We return R',T' where R' is b r' a and H |- b r' a By T'
%
let GeneralComplementRel (R, T) =
 let lemma = (get_rel_opid (subtermn 1 R)) ^ `_complement` in
 let T' = Assert R THENL [T; BackThruLemma lemma THENM T ] in
 let R' = complement_rel_term R  in
 ( R', T')
;;


%
dir = `rev` means relation needs reversing.
dir = `comp` means relation needs complementing
dir = `revcomp` means relation needs reversing and complementing

let soft matching do reversing for revcomp option. (knowing
this option only applied to order relations and that they
always end up in standard sense)

Assume at least one element in i_dir_prs

Returns pair (R,T) where R is the joined relation term and T is a tactic
to prove it.

%

let GeneralJoinRelChain e f  i_dir_prs =
  let f' (i,dir) =
       if dir = `rev` then GeneralInvertRel (f i)
       else if member dir ``revcomp comp`` then GeneralComplementRel (f i)
       else f i in
  accumulate1 (GeneralJoinRels e) (map f' i_dir_prs)  
;;

  

