
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************

#+cltl2(in-system-package)
#-cltl2(in-package *system-package-name*)


;;;;	
;;;;	Pasting can be implemented in terms of other primitives?
;;;;	
;;;;	  - cut target.
;;;;	  - then walk over target and extract fragments.
;;;;	  - then walk over source and paste fragments
;;;;	  - then paste source to target slot.
;;;;	
;;;;	
;;;;	Ie, it seems cut and paste to slot is sufficient so 


;;;;	
;;;;	label cache : an assoc list of labels and dtrees
;;;;	  - flush (or update) cache after modification 
;;;;	  
;;;;	Must be prepared to move labels to/from lazy terms.
;;;;    labels to lazy terms.

;;; maybe some heterogeneous dtree/term tag/label manipulation walk function.
;;; ie abstractly add/remove/collect labels from dtree.
;;; can use test-condition-expression to locate labels. however
;;; if modifying need diff func to replace changed terms.
;;; moving a label prob ought not to expand dtree.


    
;;;;	expand when moving labels?
;;;;	
;;;;	If searching then modifing when found then might require expanding
;;;;	bottom up. (maybe with null dfparms and refresh bit on).
;;;;	
;;;;	maybe better to search, then use address to modify.
;;;;	
;;;;	Looks like its expanding the path to the label.
;;;;	Seems sensible as it is only the path. 
;;;;	
;;;;	If you do some search by pushing a label around a tree it
;;;;	could end up expanding whole term.  May need to implement
;;;;	move label to term/dtree hybrid directly or expect
;;;;	users to avoid pushing a label around a tree by using cond-expr search.




;;;;	
;;;;	Edit History.
;;;;	
;;;;	
;;;;	<zoom>	: <dtree>, <zoom-stack>
;;;;	
;;;;	
;;;;	history is update after each edit cycle.
;;;;	
;;;;	
;;;;	
;;;;	

(defun view-update-history (v)

  ;;(unless (or (null -edtree) (eql (aref (dtree-children (dtree-parent  (dtree-parent -edtree))) 0) (dtree-parent -edtree)))
  ;;(break "vhu"))

  (when (view-flag-history-required-p v)
    (view-flag-set-history-required v nil)
    (when (view-state-p v)
      (let ((h (history-of-view v))
	    (d (dtree-of-view v))
	    (z (zoom-stack-of-view v)))
	    
	(let ((first (first-of-history h)))
	  (unless (and (eql (car first) d)
		       (eql (cdr first) z))
	
	    (history-update h (cons d z))))))))


(defun zoom-to-term (zoom)
  (unless (null (cdr zoom))
    (raise-error (error-message '(edit history zoom zoomed)
				"Haven't implemented support in history for logging zoomed views")))

  ;; don't lift text edits.
  (cons (dtree-to-term (car zoom))
	;; need to do more for zoom.
	nil))

		     
;; every 8 update the 64th most recent is moved to second generation.
;; every 16th move the 32th most recently moved is moved to third generation.
;; every 16 2ngen move results in move/loss of 16th most recent third generation entry.
;; and so on
;; oldest available was made ~ 8*16*16*16 = 32K updates ago.

(defun new-edit-history ()
  (new-history
   (cons (list* 64 7 #'zoom-to-term)
	 '((32 15) (16 15) (16)))))



(defun edit-history-restore (v w)
  (let ((z (history-walk-peek w)))
    (let ((a (if (consp z) (car z) z))) ; temp check for cons to avoid bug, but maybe sensible to have direct structure if no zoom.

      (unless (or (not (consp z)) (null (cdr z)))
	(raise-error
	 (error-message '(edit history undo zoomed)
			"Haven't implemented support in history for logging zoomed views")))
	
      (set-view-dtree v
		      (if (dtree-p a)
			  (progn (dtree-clean a) a)
			  (new-dtree a (implicit-of-view-object v)))
		      t))))

;; need to assume clean text edits from dtree in history
;;  is ok. This is ok as long as when we start history walk
;;  we first raise text edits.
(defun edit-undo (v)
  (let ((w (or (history-walk-of-view v)
	       (view-begin-history-walk v))))
    
    (when w
      ;;(setf -w w -v v) (break "eu")
      (when (history-walk-forward w)
	(edit-history-restore v w)
	t))))


(defun edit-redo (v)
  (let ((w (history-walk-of-view v)
	  ))
    
    (when w
      (when (history-walk-backward w)
	(edit-history-restore v w)
	t))))

(defun edit-abort-undo (v)
  (let ((w (history-walk-of-view v)))
    
    (when w
      (when (history-walk-jump-back w)
	(edit-history-restore v w)
	(view-abort-history-walk v)
	t))))


(defun edit-commit-undo (v)
  (let ((w (history-walk-of-view v)))

    (when w
      (when (history-walk-jump-back w)
	(edit-history-restore v w)
	(view-end-history-walk v)
	t))))


;; if require dtree p is true then f can expect dtree arg
;; otherwise it may be term or dtree.

;;  Guess this is incomplete version of general purpose
;; function to operate on term/dtree hybrid.
#|
(defun operate (dtree address require-dtree-p f)

  (labels ((visit-dtree (dtree addr)
	     ??(dtree-instantiated-p dtree t)
	     (cond
	       ((null addr)
		dtree)
	       ((dtree-leaf-p dtree)
		(raise-error (error-message '(dtree address leaf) address "left:" addr)))
	       (t (let* ((children (children-of-dtree dtree))
			 (l (length children))
			 (i (car addr)))
		    (cond
		      ((zerop i)
		       (raise-error (error-message '(dtree address index "address:" address "left:" addr))))
		      ((< i 0)
		       (let ((j (+ l i 1)))
			 (when (< j 0)
			   (raise-error (error-message '(dtree address out-of-range)
						       l "address:" address "left:" addr)))
			 (visit (aref (children-of-dtree dtree) j) (cdr addr))))
		      (t;; (> i 0)
		       (when (> i l)
			 (raise-error (error-message '(dtree address out-of-range)
						     l "address:" address "left:" addr)))
		       (visit (aref (children-of-dtree dtree) (1- i)) (cdr addr)))))))))

    (visit dtree address))
      )

  )
|#



;;;;	
;;;;	Move label in dtree.
;;;;	
;;;;	Take care to tag modify path appropriately.
;;;;	

;;;;	
;;;;	addresses start at 1 not 0.
;;;;	
;;;;	direction == true means left to right.
;;;;	 which is opposite of left-right-of-edit-state = t (although that would be easy to change).
;;;;	


;; will also want addresses of condition.

;; could this be an instance of a more abstract apply ce to tree function.
;; then we might be able to do functions like dtree-at-label more directly.
(defun address-of-condition (dtree ce)
  (let ((max 0))

    (labels
	((visit (item mode)

	   (let ((terminate-p nil)
		 (result  nil))

	     ;; check quick termination
	     (if (eql mode 'term)
		 (when (not (parameter-p item))
		   (if (icondition-cut-term-p item)
		       (setf terminate-p t
			     result nil)
		       (let ((s (summary-labels-tags-of-term item)))
			 (when s
			   (let ((r (test-condition-expression-summary ce 'term item
								       (labels-of-summary s)
								       (tags-of-summary s))))
			     (unless (eql r 'maybe)
			       (setf terminate-p t
				     result r)) )))))
	   
		 (when (and (eql mode 'dtree)
			    (not (dtree-leaf-p item)))
		   (when (icondition-cut-term-p (dtree-term item))
		     (setf terminate-p t
			   result nil) )))

	     (if terminate-p
		 result
		 ;;(setf -a dtree -b ce -c item) (break "aoc")
		 (progn
		   (when (and nil
			      (eql mode 'term)
			      (not (parameter-p item)))
		     (let ((count (term-op-count item)))
		       (when (> count max)
			 (setf max count)
			 (format t "term op count ~a~%" max))))
	       
		   (if (test-condition-expression ce mode item)
		       t
		       (let ((i 0)
			     (addr nil))
			 (dtree-structure-children-apply
			  item mode
			  #'(lambda (child)
			      (let ((c-addr (cond
					      ((or (eql mode 'term)
						   (or (dtree-leaf-p child) (dtree-flag-instantiated-p child)))
					       (visit child mode))
					      ;; ie not term (ie 'dtree) and not instantiated.
					      (t (or (and (test-condition-expression ce mode child) t)
						     (visit (term-of-dtree child) 'term))))))
				(cond
				  ((null c-addr)
				   (incf i)
				   nil)
				  ((eql t c-addr)
				   (setf addr (list (1+ i)))
				   t)
				  (t (setf addr (cons (1+ i) c-addr))
				     t)))))
			 addr)))))))

      (let ((c-addr (if (dtree-flag-instantiated-p dtree)
			(visit dtree 'dtree)
			(visit (summarize-term-tags-and-labels (term-of-dtree dtree)) 'term))))
	(if (eql c-addr t)
	    nil
	    c-addr)))))


(defun address-of-label (dtree label)
  (address-of-condition dtree (new-token-ce 'label label)))


(defun dtree-at-label (tag view)
  (or (cdr (assoc tag (label-cache-of-view view)))
      (let ((dtree (dtree-of-view view)))
	;;(setf -dtree dtree -tag tag -view view) (break "ahello")
	;; PERF : this is kind of the long way around:
	(dtree-at-address dtree (address-of-label dtree tag)))))


;;;;	
;;;;	general problem
;;;;	  term-modified 'tag
;;;;	    means the tag of the encoded term may have changed.
;;;;	  
;;;;	  dtree-to-term d 'tag
;;;;	    means give us a term without worrying about the tags.
;;;;	    does not mean term will not have tags. 
;;;;	    
;;;;	  moving tags around dtrees in general does change tagging.
;;;;	  however if we move a tag in then out we would like to think we
;;;;	  were never there. Although may not be worth the complexity.
;;;;	



(defun maybe-eager-dtree-layout (dtree)
  (when dtree
    (if (dtree-leaf-p dtree)
	(maybe-eager-dtree-layout (parent-of-dtree dtree))
	(let ((dform (dform-of-dtree dtree)))
	  (cond
	    ((null dform) nil)
	    ((member 'ChooseDFormEagerly (conditions-of-dform dform))
	     (dtree-path-layout-required dtree))
	    (t (maybe-eager-dtree-layout (parent-of-dtree dtree))))))))


(defun edit-move-label-to-dtree (view tag source target &optional silent)

  ;;(break "emltd")

  (when source
    (untag-dtree source tag t)
    (unless silent
      (maybe-eager-dtree-layout source)
      (dtree-path-layout-modified source 'tag)
      (dtree-path-term-modified source 'tag)))

  ;; we are making the term and dtree dissonant wrt to tags so why not have structure-modified updated.
  (when target
    (tag-dtree target tag t)

    (unless silent
      ;; (setf -tag tag -target target -silent silent) (break "emltd")
      (maybe-eager-dtree-layout target)
      ;;(dtree-path-layout-modified target 'text)  ; at cvs merge these differed not sure which is better.
      (dtree-path-layout-modified target 'tag)
      (dtree-path-term-modified target 'tag)))

  ;; update-cache
  (set-view-label-cache view
			(let ((dcache (delete tag (label-cache-of-view view) :key #'car)))
			  (if target
			      (acons tag target dcache)
			      dcache))))

(defun edit-remove-label (view label)
  (edit-move-label-to-dtree view label (dtree-at-label label view) nil))

(defun edit-move-label (view label target &optional silent)
  (edit-move-label-to-dtree view label (dtree-at-label label view) target silent))

(defun edit-move-label-to-address (view tag addr)
  (edit-move-label view tag (dtree-at-address (dtree-of-view view) addr)))

(defun edit-move-label-to-label (view tag target)
  (let ((point (dtree-at-label target view)))
    (edit-move-label view tag point)
    (when (dtree-leaf-p point)
      (set-dtree-leaf-index point tag (or (index-of-dtree-leaf point target) 0)))))


;; permuted?
;; f : <int>{child index} {child}
(defun map-dtree-children (f perp dir dtree)
  
  (unless (or (null dtree) (dtree-leaf-p dtree))
    (let* ((children (children-of-dtree-c dtree))
	   (dform (dform-of-dtree-c dtree))
	   (indices (unless perp (preorder-indices-of-dform-model (model-of-dform dform))))
	   (formats (when perp (children-of-dform-formats (formats-of-dform dform))))
	   (l (if perp
		  (length formats)
		  (length indices)))
	   (seen nil)
	   )
      

      (labels ((do-call (i)
		 ;;(setf -a i -b l -c formats -d indices -e children) (break "emld")
		 ;; if perp but index occurs more than once ignore later occurences.
		 ;; avoids loop in walk, there may be other case where needed then need
		 ;; another parameter to map.
		 (if perp
		     (let ((perpi (let ((dindex (dtree-index-of-dform-child (aref formats i))))
				    (unless (member dindex seen)
				      (push dindex seen)
				      dindex))))
		       (when perpi
			 (funcall f (aref (children-of-dtree dtree) perpi))))
		     (funcall f (aref children (aref indices i))))))

	  
	(if dir
	    ;; lower to higher
	    (do ((i 0 (1+ i)))
		((or (= i l)
		     (do-call i)) ))
	    ;; higher to lower.
	    (do ((i (1- l) (1- i)))
		((or (< i 0)
		     (do-call i)) )))
	))))

(defun dtree-child-position (child parent)
  (let ((i 1))
    (map-dtree-children #'(lambda (c)
			    (if (eq c child)
				(return-from dtree-child-position i)
				(incf i))
			    nil)
			t
			t
			parent) ))




;; if child is t then starts from end.
;; abort-f sometimes caller wants to know who we're avoiding.
(defun dtree-find-sibling (perp dir ce child parent &optional abort-f)

  (let ((found-p (when (eql t child) t))
	(sibling nil))

    (map-dtree-children
     #'(lambda (dtree)
	 (cond
	   (found-p
	    (cond
	      ((test-condition-expression ce 'dtree dtree)
	       (setf sibling dtree)
	       t)

	      (t (when abort-f
		   (when (funcall abort-f dtree)
		     t)))))

	   ((eql child dtree)
	    (setf found-p t)
	    nil)

	   (nil nil)))
     perp dir parent)

    sibling))


;; abort f is called only when ce does not eval to true.
(defun dtree-child (perp dir ce dtree &optional abort-f)
  (unless (dtree-leaf-p dtree)
    (dtree-find-sibling perp dir ce t dtree  abort-f)))



(defun edit-move-label-to-sibling (perp dir expr tag view)

  (let* ((dtree (dtree-at-label tag view))
	 (parent (parent-of-dtree dtree)))

    (when parent
      (let ((sibling (dtree-find-sibling perp dir expr dtree parent)))

	;;(setf -sibling sibling -dtree dtree -parent parent) (break "emlts")
	(when sibling
	  (edit-move-label-to-dtree view tag dtree sibling)
	  ;;(setf -a dtree -b parent -c expr -d sibling) (break "emlts")
	  t)))))


;; bool indicates if move happened.
(defun edit-move-label-up (tag view)
  (let* ((dtree (dtree-at-label tag view))
	 (parent (parent-of-dtree dtree)))
    
    (if parent
	(progn
	  (edit-move-label-to-dtree view tag dtree parent)
	  t)
	(progn
	  (message-emit (warn-message '(edit label up top) tag)))
	)))

;; returns nil if label was already at top t otherwise.
(defun edit-move-label-top (tag view)
  (let* ((top (dtree-of-view view))
	 (dtree (dtree-at-label tag view)))

    (unless (eql top dtree)
      (edit-move-label-to-dtree view tag dtree top)
      t)))



;; looks only at visible direct children.
;; bool indicates if move happened.
(defun edit-move-label-down (perp dir ce tag view)
  (let ((dtree (dtree-at-label tag view)))
    (when (and dtree (not (dtree-leaf-p dtree)))
      (let ((child (dtree-child perp dir ce dtree)))
	(when child
	  (edit-move-label-to-dtree view tag dtree child)
	  t)))))


(defunml (|edit_sibling| (perp dir ce tag view))
    (bool -> (bool -> (cond_expr -> (tag -> (view -> bool)))))

  (edit-move-label-to-sibling perp dir ce tag view))
    

(defunml (|edit_down| (perp dir ce tag view))
    (bool -> (bool -> (cond_expr -> (tag -> (view -> bool)))))

  (edit-move-label-down perp dir ce tag view)
  )

(defunml (|edit_up| (tag view))
    (tag -> (view -> bool))

  (edit-move-label-up tag view)
  )

(defvar *placeholder* (iplaceholder-term))

(defun edit-cut (tag view)
  (let ((dtree (dtree-at-label tag view))
	(slot (new-dtree (iplaceholder-term) nil)))
    ;;(setf -dtree dtree) (break "ec")
    (unless (let ((dform (dform-of-dtree dtree)))
	      ;; might be better to find a dform? but this will work 99.9999%
	      (and dform (member 'nocut (conditions-of-dform dform))))
      (when (and dtree
		 (not (dtree-leaf-p dtree))
		 ;; is it possible for literal term to be dtree when encoded term is not?
		 ;; I don't think so but if so then followin nfg.
		 (not (compare-terms-p *placeholder* (term-of-dtree dtree)))
		 )
	(tag-dtree slot tag t)
	(set-view-dtree view (dtree-replace dtree slot))
	;;(setf -slot slot -v view) (break "ec")

	t))))


;; simple paste to slot.
(defun edit-paste-term (tag term view)
  (let ((dtree (dtree-at-label tag view)))

    (when (dtree-leaf-p dtree)
      (raise-error (error-message '(edit paste term target parameter))))
    
    ;;(unless (compare-terms-p *placeholder* (term-of-dtree dtree))
    ;;(raise-error (error-message '(edit paste target slot not))))
    
    (let ((ndtree (new-dtree term nil)))
      ;;(setf -ndtree ndtree -dtree dtree) (break "ept")
      (set-view-dtree view (dtree-replace dtree ndtree))
      (tag-dtree ndtree tag t)
      t)))



(defun edit-replace (view dtree term)
  (let ((ndtree (new-dtree term nil)))
    (set-view-dtree view (dtree-replace dtree ndtree))
    ;;(when tag (tag-dtree ndtree tag t))
    t))



(defunml (|edit_cut| (tag view))
    (tag -> (view -> bool))

  (edit-cut tag view))
	  
(defunml (|edit_term_at_label| (tag tags-p view))
    (tag -> (bool -> (view -> term)))

  ;;(setf -tag tag -view view) (break "etal")

  (let ((term (dtree-to-term (or (let ((dtree (dtree-at-label tag view)))
				   (when dtree
				     (if (dtree-leaf-p dtree)
					 (parent-of-dtree dtree)
					 dtree)))
				 (failwith `notfound))
			     (if tags-p nil 'tag))))
    ;;(setf -term term)
    (if tags-p
	(if (real-ilabel-term-p term)
	    (subterm-of-term term '(0))
	    term)
	(filter-term-tags term *filter-all*))))


(defunml (|edit_paste_term| (tag term view))
    (tag -> (term -> (view -> bool)))

  ;;(setf -term term) (break "ept")

  (edit-paste-term tag term view))


(defunml (|edit_undo| (view))
    (view -> bool)

  (edit-undo view))

(defunml (|edit_redo| (view))
    (view -> bool)

  (edit-redo view))

(defunml (|edit_abort_undo| (view))
    (view -> bool)

  (edit-abort-undo view))

(defunml (|edit_commit_undo| (view))
    (view -> bool)

  (edit-commit-undo view))



;;;;	
;;;;	ML funcs.
;;;;	
;;;;	
;;;;	

;;;
;;;  views
;;;

(defunml (|view_discard| (v))
    (view -> unit)

  (check-not-lightweight v "view_discard")

  (when (or (object-of-view v)
	    (view-window-open-p v))

    (raise-error (error-message '(edit view discard inuse)
				   (list (when (object-of-view v) t)
					 (when (history-of-view v) t)))))
  (when (eql v (current-view))
    ;;(break "hello")
    (set-current-view nil))
  
  (setf *views* (delete v *views*)) )

    
  

(defunml (|new_view| (bool) :error-wrap-p nil)
    (bool -> view)
  (new-view bool))

(defunml (|view_lookup| (oid) :error-wrap-p nil)
    (object_id -> (view list))

  (mapcan #'(lambda (v)
	      (when (and (view-object-p v)
			 (equal-oids-p oid (oid-of-view v)))
		(list v)))
	  *views*))

(defunml (|views| (unit) :error-wrap-p nil :declare ((declare (ignore unit))))
    (unit -> (view list))

  *views*)


(defunml (|set_current_view| (v) :error-wrap-p nil)
    (view -> unit)

  (set-current-view v)
  nil)

(defunml (|reset_set_current_view| (unit) :error-wrap-p nil :declare ((declare (ignore unit))))
    (unit -> unit)

  (set-current-view nil)
  nil)

(defunml (|tok_to_tag| (tt) :error-wrap-p nil)
    (tok -> tag)
  tt)

(defunml (|new_tag| (unit) :error-wrap-p nil :declare ((declare (ignore unit))))
    (unit -> tag)
  (gensym))


(defunml (|edit_remove_label| (label v))
    (tag -> (view -> unit))

  (edit-remove-label v label))


(defunml (|edit_tag_address| (tag addr label-p v ))
    (tag -> ((int list) -> (bool -> (view -> unit))))
  ;;(setf -addr addr) (break "tag addr")
  (let ((dtree (dtree-at-address (dtree-of-view v) addr)))
    (if label-p
	(edit-move-label-to-address v tag addr)
	(tag-dtree dtree tag label-p))
    (values)))

(defunml (|edit_tag_label| (label tag label-p v))
    (tag -> (tag -> (bool -> (view -> unit))))
  
  (let ((dtree (dtree-at-label label v)))
    ;;(setf a label b tag c dtree d v) (break "etl")
    (if label-p
	(edit-move-label v tag dtree)
	(tag-dtree dtree tag label-p))
    (values)))



;;;;	edit_ce_string_to_expression	: string -> cond_expr
(defunml (|edit_ce_string_to_expression| (s))
    (string -> cond_expr)
  (string-to-cond-expr s))



(defunml (|view_test_p|  (ce label v))
    (cond_expr -> (tag -> (view -> bool)))

  ;;(setf a ce b label c v) (break "vtp")
  (and (test-condition-expression ce 'dtree (dtree-at-label label v) :extend-p t)
       t)
  )


(defun view-search-up (dtree ce)
  (unless (null dtree)
    (if (test-condition-expression ce 'dtree dtree)
	dtree
	(view-search-up (parent-of-dtree dtree) ce))))

;; place search label at first ancestor of point which satisfies condition.
(defunml (|view_search_up|  (ce point search v))
    (cond_expr -> (tag -> (tag -> (view -> bool))))

  ;;(setf -ce ce -point point -search search -v v) (break "vsu")
  (let ((dtree (or (view-search-up (dtree-at-label point v) ce)
		   (failwith `notfound))))
    (edit-move-label v search dtree)))


(defun view-search-down (dtree ce)
  (unless (null dtree)
    (if (test-condition-expression ce 'dtree dtree)
	dtree
	(let* ((ch (children-of-dtree dtree))
	       (l (length ch))
	       (found nil))
	  ;;(break)
	  (do ((i 0 (1+ i)))
	      ((or found (>= i l)) found)
	    (let ((dc (aref ch i)))
	      (setf found (when (test-condition-expression ce 'dtree dc)
			    dc))))))))


;; place search label at first ancestor of point which satisfies condition.
(defunml (|view_search_down|  (ce point search v))
    (cond_expr -> (tag -> (tag -> (view -> bool))))

  ;;(setf -ce ce -point point -search search -v v) (break "vsu")
  (let ((dtree (or (view-search-down (dtree-at-label point v) ce)
		   (failwith `notfound))))
    (edit-move-label v search dtree)))






;;;
;;;  associate object.
;;;


;; should fail if already associated?
;; pros easier for UI coder to program.
;; cons : failure would serve as sanity check.
;; what about history? ditto.

(defun set-view-object (v vobj)

  (check-not-lightweight v "set-object-view")

  ;;(setf -source source)  (break "vao2")
  ;; set modified state.
  (when vobj
    (if (compare-terms-p (term-of-vobject vobj) (term-of-view v))
	(view-flag-set-modified v 'nil)
	(let ((view-mod-q (view-flag-modified-q v)))
	  (unless (member view-mod-q '(remote both))
	    (if (eql 'local view-mod-q)
		(view-flag-set-modified v 'both)
		(view-flag-set-modified v 'remote)
		)))))
	     
  (setf (view-state-object v) vobj))


(defun view-refresh-object (v)

  (let ((oid (oid-of-view-r v)))
    (let ((vobj (instantiate-vobject oid)))

      (set-view-object v vobj))))


(defunml (|view_associate_object_lite| (oid v))
    (object_id -> (view -> unit))

  ;;(setf -v v -oid oid)  (break "vao")
  (view-assign-oid v oid t)

  (values))

(defunml (|view_associate_object| (oid v))
    (object_id -> (view -> unit))

  (check-not-lightweight v "view_associate_object")
  
  (when (or (object-of-view v)
	    (history-of-view v))
    (raise-error (oid-error-message (cons oid (when (object-of-view v)
						(let ((o (oid-of-view v)))
						  (when o (list o)))))
				    '(edit view associate inuse)
				    (list (when (object-of-view v) t)
					  (when (history-of-view v) t)))))

  ;; (setf -v v -oid oid)  (break "vao")
  (view-assign-oid v oid)

  (values))

(defunml (|view_associate_prf_object| (soid poid v))
    (object_id -> (object_id -> (view -> unit)))

  (check-not-lightweight v "view_associate_prf_object")
  
  (when (or (object-of-view v)
	    (history-of-view v))
    (raise-error (oid-error-message (cons oid (when (object-of-view v)
						(let ((o (oid-of-view v)))
						  (when o (list o)))))
				    '(edit view associate inuse)
				    (list (when (object-of-view v) t)
					  (when (history-of-view v) t)))))

  ;; (setf -v v -oid oid)  (break "vapo")
  (view-assign-prf-oid v poid soid)

  (values))


;; resets dtree to represent term of object.


(defunml (|view_set_term| (term force_p v))
    (term -> (bool -> (view -> unit)))

  ;; TODO : need other criteria for failing if not lightweight.
  (unless (or (lightweight-view-p v) force_p)
    ;;(setf a v) (break)
    (raise-error (error-message '(view_set_term force))))
  
  (view-dtree-init v term nil)
  (values))


(defunml (|view_disassociate_object| (force v))
    (bool -> (view -> unit))
  
  (check-not-lightweight v "view_disassociate_object")
  
  (when  nil ;; RLE TODO: unsaved data
    (if force
	(message-emit (warn-message '(edit view disassociate unsaved)))
	(raise-error (error-message '(edit view disassociate unsaved)))))


  ;; TODO twould be nice to save the history (on disk) so that it could be recovered
  ;; but that should be done as it is modified (in case of crash) and not just at quit.

  (when (view-object-p v)
    (edit-oid-assoc-delete (oid-of-view v)))

  (set-view-object v nil)
  (view-history-reset v)
  (view-dtree-reset v)
  
  (values))






;;;
;;; window open/close
;;;

(defun view-open-window (v)
  (unless (windows-p)
    (raise-error (error-message '(view open windows not))))
  
  (let ((ewin (if (view-window-p v)
		  (window-of-view v)
		  (let ((geo (when (view-object-p v) (geometry-of-view-object v))))
		    (set-view-window v
				     (if geo
					 (new-ewin (first geo) (second geo) (third geo) (fourth geo))
					 (new-ewin 10 10 50 20)))))))


    ;;(setf v -v )(break "vow")
    (view-flag-set-open v t)
    (view-set-window-title v)

    (if (window-open-p ewin)
	(message-emit (warn-message '(edit view window open open)))
	(edit-open-win v))))

(defunml (|view_open_window| (v))
    (view -> unit)

  (view-open-window v)

  (values))


(defunml (|view_close_window| (v))
    (view -> unit)
 
  (let ((ewin (when (view-window-p v)
		(window-of-view v))))

    (view-flag-set-open v nil)
    (if (and ewin (window-open-p ewin))
	(edit-close-win v)
	(message-emit (warn-message '(edit view window close open not)))))

  (values))

(defunml (|view_window_open_p| (v))
    (view -> bool)

  (view-window-open-p v))


(defvar *print-view* (new-view t))

(defunml (|print_ascii| (ce width term fname))
    (cond_expr -> (int -> (term -> (string -> unit))))

  ;;(setf -c ce -f fname -w width) (break "pai")
  (with-dform-ce (ce)

    (let ((v *print-view*))

      (view-dtree-reset v)
      (view-dtree-init v term nil)

      (let ((ptree (layout (dtree-of-view v) width)))

	(let ((wbuff (write-ptree-to-buffer ptree 0 0)))

	  ;;(setf -ptree ptree -wbuff wbuff) (break "pa")
		
	  ;; dump wbuff to file.
	  (with-prl-open-file (s fname out)
	    (write-wbuff-to-ascii-stream wbuff s)))))))


(defun oed-print-ascii (ce width term fname)
  ;;(setf -term term -w width -ce ce) (break "opa1")
  (with-dform-ce (ce)

    (let ((v *print-view*))

      (view-dtree-reset v)
      (view-dtree-init v term nil)

      (let ((dtree (dtree-of-view v)))
	(let ((ttree (layout-of-dtree-c dtree width)))
	  (let ((lines (layout-visit ttree
				     width
				     100000
				     0
				     dtree)))

	  
	    ;;(setf -term term -ttree ttree -dtree dtree -lines lines) (break "opa")

	    ;; dump wbuff to file.
	    (with-open-file (s fname :direction :output :if-exists :rename)
	      ;;(setf -s s) (break "opas")
	      (dolist (li lines)
		(let ((l (delete ireturn (delete ipage li))))
		
		  (let ((str (if (eql (car l) 'continue)
				 (progn (write-char #\> s)
					(implode-to-string (cdr l)))
				 (implode-to-string l))))
		  
		    (write-string str s)
		    (write-char #\Return s)
		    (write-char #\newline s)
		    ))))
	    
	    ;;(break "opad")

	    ))))))


(defunml (|oed_print_ascii| (ce width term fname))
    (cond_expr -> (int -> (term -> (string -> unit))))

  ;;(setf -term term -w width -ce ce) (break "mopa")
  (oed-print-ascii ce width term fname)

  nil)


(defml |view_quit| (v))
  


(defunml (|view_dform_q| (tag v))
    (tag -> (view -> (object_id |#| int)))

  (let ((dtree (dtree-at-label tag v)))
    (if (and (not (dtree-leaf-p dtree)) (dform-of-dtree dtree))
	(let* ((dform (dform-of-dtree dtree))
	       (oid (oid-of-dform dform))
	       (dforms (dforms-lookup oid))
	       (index  (position dform (list-of-dforms dforms))))
	  (cons oid (or index -1)))
	(view-error-message v '(view dform query not)))))


(defunml (|view_object_id_q| (tag v))
    (tag -> (view -> object_id))

  ;;(setf -tag tag -v v) (break "voiq")
  (let ((dtree (dtree-at-label tag v)))
    (if (dtree-leaf-p dtree)
	;; dtree-to-parameter not required since oid parameter is not modifiable.
	(let ((p (parameter-of-dtree dtree)))
	  (if (oid-parameter-p p)
	      (value-of-parameter-r p)
	      (view-error-message v (list 'view 'q tag 'object_id 'not))))
	;; find first oid parameter of term of dtree.
	(let ((oid (or (find-first #'(lambda (p)
			    (when (and (oid-parameter-p p) (real-parameter-p p))
			      (value-of-parameter-r p)))
			(parameters-of-term (term-of-dtree dtree)))
	    (view-error-message v (list 'view 'object_id 'q tag 'not)))))
	    ;;(setf -oo oid)(break)
	    (if (eql 'stm (kind-of-ostate oid))
		(progn (break)  oid)
		oid)))))


(defunml (|view_parameter_q| (tag v))
    (tag -> (view -> parameter))

  ;; could do for carrier, or if term use first parameter.
  (let ((dtree (dtree-at-label tag v)))
    (if (dtree-leaf-p dtree)
	(dtree-to-parameter dtree 'tag)	  
	(view-error-message v (list 'view 'parameter 'q tag 'not)))))



