
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************

#+cltl2(in-system-package)
#-cltl2(in-package *system-package-name*)

(defun traversable-dtree-p (dtree)
  (and (not (dtree-flag-non-modifiable-p dtree))
       t))
  

(defun first-non-irrelevant-dtree (dtree direction &optional intarget)
  (let ((target intarget)
	(seen nil)
	(found nil))
    (map-dtree-children #'(lambda (d)
			    (cond
			      ((equal-dtree-p target d)
			       (setf target nil
				     seen (cons d seen))
			       nil)
			       
			      ((and (null target)
				    (not (dtree-flag-irrelevant-p d))
				    (not (member d seen)))
			       (setf found d)
			       t)
			      (t (setf seen (cons d seen))
				 nil)))

			t direction
			dtree)
    found))

(defun child-of-dtree (dtree mode)
  (when (and (traversable-dtree-p dtree)
	     (not (dtree-leaf-p dtree)))
    (first-non-irrelevant-dtree dtree mode)))


(defun only-child-of-dtree (dtree)
  (child-of-dtree dtree (left-mode)))


(defun text-carrier-p (term &optional other-term &key (meta-ok nil))
  (let ((sig (term-sig-of-term term)))
    (and (null (arities-of-term-sig sig))
	 (= 1 (length (parameters-of-term-sig sig)))
	 (or meta-ok (not (meta-parameter-p (car (parameters-of-term term)))))
	 (if other-term
	     (equal-term-sigs-p sig (term-sig-of-term other-term))
	     (member (car (parameters-of-term-sig sig))
		     *itext-parameter-typeids*)))))


(defun parameter-of-text-carrier (term)
  (car (parameters-of-term term)))

(defun string-of-text-carrier (term)
  (string (value-of-parameter-r (parameter-of-text-carrier term))))
			       
    

(defun conjoinable-p (term &optional meta-ok)
  (or (and (itext-term-p term)
	   (not (meta-parameter-p (car (parameters-of-term term)))))
      (and (text-carrier-p term nil :meta-ok meta-ok)
	   (abstraction-condition-p term 'conjoin))))


(defun edit-text-carrier-p (point &optional (meta-ok nil))
  (and
   (conjoinable-p (term-of-dtree point) meta-ok)
   (dtree-leaf-p (only-child-of-dtree point))
   ))






;;;;
;;;; ILIST, term-text mixed.
;;;;


;;
;; car, up, cdr, next, prev, insert, delete
;;
;; (a b c)  (cons a (cons b (cons c nil)))
;;
;; list condition
;;    - up,next,prev,insert,add,normalize: parent has list in conditionals or is icons.
;;    - up,down,normalize : term is list. 
;;  nil is list abstraction with no null arities.

(defun icons-operator-p (term)
  (and (icons-term-p term)
       (null (parameters-of-term term))))

(defvar *check-list-conditions* nil)

(defun weak-ilist-cons-p (term op)
  (if op
      (ilist-cons-p term op)
      (and (or (icons-operator-p term)
	       (and (or (not *check-list-conditions*)
			(abstraction-condition-p term 'cons))
		    (null (parameters-of-term term))))
	   (equal '(0 0) (arities-of-term term)))))

(defun weak-ilist-nil-p (term &optional op)
  (if op
      (ilist-nil-p term op)
      (and (or (icons-operator-p term)
	       (and (or (not *check-list-conditions*)
			(abstraction-condition-p term 'nil))
		    (null (parameters-of-term term))))
	   (null (arities-of-term term)))))

(defun weak-ilist-list-p (term &optional op)
  (or (weak-ilist-cons-p term op)
      (weak-ilist-nil-p term op)))

(defun edit-ilist-nil-p (point &optional op)
  (and point (not (dtree-leaf-p point))
       (weak-ilist-nil-p (term-of-dtree point) op)))

(defun edit-ilist-cons-p (point &optional op)
  (and point (not (dtree-leaf-p point))
       (weak-ilist-cons-p (term-of-dtree point) op)))

(defun edit-ilist-list-p (point &optional op)
  (and point
       (or (edit-ilist-cons-p point op)
	   (edit-ilist-nil-p point op))))

(defun edit-ilist-element-p (point)
  (let ((parent (parent-of-dtree point)))
    (and (edit-ilist-cons-p parent)
	 (not (edit-ilist-cons-p point (operator-of-term (term-of-dtree parent))))
	 )))

(defun edit-ilist-nil-element-p (point)
  ;;(setf -point point) (break "einep")
  (and (edit-ilist-element-p point)
       (edit-ilist-nil-p point (operator-of-term (term-of-dtree (parent-of-dtree point))))))


(defun edit-ilist-top (point op)
  (let* ((parent (parent-of-dtree point)))
    (if (edit-ilist-cons-p parent op)
	(edit-ilist-top parent op)
	point)))


;; assumes point is an element of a list.
(defun edit-ilist-normal-p (point)
  (let ((op (operator-of-term (term-of-dtree (parent-of-dtree point)))))
    (labels ((visit (dtree)
	       (cond
		 ((edit-ilist-nil-p dtree op)
		  t)
		 ((edit-ilist-cons-p dtree op)
		  (let ((left-child (child-of-dtree dtree (left-mode))))
		    (and (not (edit-ilist-cons-p left-child op))
			 (not (edit-ilist-nil-p left-child op))
			 (visit (child-of-dtree dtree (right-mode))))))
		 (t nil))))
    (visit (edit-ilist-top point op)))))
  

(defun set-point-state (state point)
  (edit-move-label state 'point point)
  (point-rehash state 'point))


(defun edit-ilist-local-top (point op)
  (let* ((parent (parent-of-dtree point)))
    (if (and (edit-ilist-cons-p parent op)
	     (eql point (child-of-dtree parent (right-mode))))
      (edit-ilist-top parent op)
      point)))

(defun edit-ilist-up (state)
  (let ((point (point-of-edit-state state)))
    (cond
      ((edit-ilist-cons-p (parent-of-dtree point))
       (let ((parent (parent-of-dtree point)))
	 (set-point-state state
		      (edit-ilist-local-top parent
				      (operator-of-term (term-of-dtree parent))))))

      ((edit-ilist-cons-p point)
       (set-point-state state
		    (edit-ilist-local-top point
				    (operator-of-term (term-of-dtree point)))))
      (t state))))

(defun edit-ilist-down (state)
  (labels ((iterate (point op)
	     (if (edit-ilist-cons-p point op)
		 (let ((child (child-of-dtree point (left-right-of-edit-state state))))
		   (if (edit-ilist-nil-p child op)
		       (child-of-dtree point (toggle-mode (left-right-of-edit-state state)))
		       (iterate child op)))
		 point)))

    (let* ((point (point-of-edit-state state)))
      (if (edit-ilist-cons-p point)
	  (set-point-state state (iterate point (operator-of-term
					     (term-of-dtree point))))
	  state))))

(defun edit-ilist-top-p (point)
  (let ((parent (parent-of-dtree point)))
    (or (null parent)
	(and (or (null (parent-of-dtree parent))
		 (not (edit-ilist-cons-p (parent-of-dtree parent)
					 (operator-of-term (term-of-dtree parent)))))
	     (eql point (child-of-dtree parent (left-mode)))))))

(defun edit-ilist-bottom-p (point)
  (or (null (parent-of-dtree point))
      (eql point (child-of-dtree (parent-of-dtree point) (right-mode)))))

(defun edit-ilist-bottom (point)
  (labels ((iterate (point op)
	     (if (edit-ilist-cons-p point op)
		 (let ((child (child-of-dtree point (right-mode))))
		   (if (edit-ilist-nil-p child op)
		       (child-of-dtree point (left-mode))
		       (iterate child op)))
		 point)))

    (if (edit-ilist-cons-p point)
	(iterate point (operator-of-term
			(term-of-dtree point)))
	point)))


(defun edit-ilist-last (state)
  (let* ((point (point-of-edit-state state)))
    ;;(setf -a point) (break "eil")
    (if (edit-ilist-cons-p point)
	(set-point-state state (edit-ilist-bottom point))
	state)))


(defun edit-ilist-right (state)
  (if (not (edit-ilist-element-p (point-of-edit-state state)))
      state
      (let* ((parent (parent-of-dtree (point-of-edit-state state)))
	     (term (when parent (term-of-dtree parent))))
	(let ((child (child-of-dtree parent (right-mode))))
	  (if (edit-ilist-cons-p child (operator-of-term term))
	      (set-point-state state (child-of-dtree child (left-mode)))
	      (set-point-state state child))))))

(defun edit-ilist-left (state)
  (let ((point (point-of-edit-state state)))
    (if (not (edit-ilist-element-p point))
	state  
	(let* ((parent (parent-of-dtree point))
	       (op (when parent (operator-of-term (term-of-dtree parent))))
	       (left-child (child-of-dtree parent (left-mode))))
	  (if (eql point left-child)
	      (let ((grand-parent (parent-of-dtree parent)))
		(if (or (null grand-parent)
			(not (edit-ilist-cons-p grand-parent op))
			(eql parent (child-of-dtree grand-parent (left-mode))))
		    state		; ie at start of list.
		    (set-point-state state (child-of-dtree grand-parent (left-mode)))))
	      (set-point-state state left-child))))))


(defun oed-edit-replace (state dtree term)
  (edit-replace state dtree term)

  ;;(setf -v state -dtree dtree -term term) (break "oer")
  (point-rehash state 'point))



;;  p(nil)		->	p(nil)
;;  p(a) . nil		->	p(nil)
;;  a . p(nil)		->	a . p(nil)
;;  p(a) . b		->	p(b)
;;  a . p(b)		->	a . p(b)
;;  p(a) . x . y	->	p(x) . y
;;  p(a) . b . nil	->	p(b) . nil
;;  x . p(a) . y . z	->	x . p(y) . z 
;;  x . p(a) . b	->	x . p(b)
;;  x . a . p(b)	->	x . a . p(b)
;;  x . p(a) . nil	->	x . p(a) . nil
;;  x . a . p(nil)	->	x . a . p(nil)

;; icdr no tags
(defun icdrnt (term)
  (term-from-dtree-tags-term (icdr (term-from-dtree-tags-term term))))


(defun edit-ilist-delete-right (state)
  (let ((point (point-of-edit-state state)))
    (cond

      ;;  p(nil)		->	p(nil)
      ((not (edit-ilist-element-p point))
	state)
      
      ;;  a . p(nil)		->	a . p(nil)
      ;;  x . a . p(nil)	->	x . a . p(nil)
      ((edit-ilist-nil-element-p point)
       state)

      ((edit-ilist-element-p point)
       (let* ((parent (parent-of-dtree point))
	      (term (dtree-to-term parent))
	      (terma (term-from-dtree-tags-term term))
	      (right-child (child-of-dtree parent (right-mode)))
	      (left-child (child-of-dtree parent (left-mode)))
	      (op (operator-of-term terma))
	      (icdr (icdrnt terma)))

	 ;;(setf -a parent -b right-child -c point -d state -f left-child) (break "eidr")
	 (cond
	   ;;  a . p(b)		->	a . p(b)
	   ;;  x . a . p(b)	->	x . a . p(b)
	   ((and (eql right-child point)
		 (not (ilist-nil-p icdr op)))
	    state)

	   ((and (eql left-child point) (ilist-nil-p icdr op))
	    (let ((grand-parent (parent-of-dtree parent)))
	      (if (and (edit-ilist-cons-p grand-parent op)
		       (not (ilist-nil-p (icdrnt (term-of-dtree grand-parent)) op))
		       (eql parent (child-of-dtree grand-parent (right-mode))))
		  ;;  x . p(a) . nil	->	x . p(a) . nil
		  state
		  ;;  p(a) . nil		->	p(nil)
		  (oed-edit-replace state parent (ilabel-term 'point icdr)))))

	   
	   ((eql left-child point)
	    (oed-edit-replace state parent
			 (cond
			   ;;  p(a) . x . y		->	p(x) . y
			   ;;  x . p(a) . y . z	->	x . p(y) . z 
			   ((ilist-cons-p icdr op)
			    (icons-term-of-op op
					      (ilabel-term 'point
							   (oed-filter-point-mark (dtree-to-term
										   (child-of-dtree right-child (left-mode)))))
					      (if (ilist-nil-p (icdrnt icdr) op)
						  ;;  p(a) . b . nil	->	p(b) . nil
						  (icdrnt icdr)	
						  (dtree-to-term
						   (child-of-dtree right-child (right-mode)) 
						    ))))
						     
			   ;;  p(a) . b	->	p(b)
			   ;;  x . p(a) . b	->	x . p(b)
			   (t (ilabel-term 'point (dtree-to-term right-child))))))

	   (t (break)))))

      (t (break)))))


;; edit-ilist-delete-left
;; cons(a;x) -> a
;; cons(x;nil) cons(x;a) cons(x;cons(a;b)) 
;;  p(nil)		->	p(nil)
;;  p(a) . nil		->	p(a) . nil | p(nil)
;;  a . p(nil)		->	p(a)
;;  p(a) . b		->	p(a) . b
;;  a . p(b)		->	p(a)
;;  p(a) . x . y	->	p(a) . x . y
;;  x . p(a) . y . z	->	p(x) . y . z 
;;  x . p(a) . b	->	p(x) . b
;;  x . a . p(b)	->	x . p(a)
;;  x . p(a) . nil	->	p(x) . nil
;;  x . a . p(nil)	->	x . p(a)

(defun edit-ilist-delete-left (state)
  (let ((point (point-of-edit-state state)))
    (cond
      ;;  p(nil)	->	p(nil)
      ((and (edit-ilist-nil-p point) (not (edit-ilist-element-p point)))
       state)

      ;;  a . p(nil)	->	p(a)
      ((edit-ilist-nil-element-p point)
       (let* ((parent (parent-of-dtree point))
	      (left-child (child-of-dtree parent (left-mode))))
	 
	 (if (eql left-child point)
	     ;;  p(nil)		->	p(*)
	     (oed-edit-replace state point (ilabel-term 'point (make-free-placeholder)))
	     ;;  a . p(nil)	->	p(a)
	     ;;  x . a . p(nil)	->	x . p(a)
	     (oed-edit-replace state parent
			       (ilabel-term 'point (dtree-to-term left-child)) ))))
       
      ((edit-ilist-element-p point)
       (let* ((parent (parent-of-dtree point))
	      (term (dtree-to-term parent))
	      (right-child (child-of-dtree parent (right-mode)))
	      (left-child (child-of-dtree parent (left-mode)))
	      (op (operator-of-term (term-from-dtree-tags-term term)))
	      (icdr (icdrnt term)))
 
	 (cond
	   ;;  a . p(b)		->	p(a)
	   ;;  x . a . p(b)	->	x . p(a)
	   ((and (eql right-child point)
		 (not (and (ilist-cons-p term op)
			   (ilist-nil-p icdr op))))
	    (oed-edit-replace state parent
			      (ilabel-term 'point
					   (dtree-to-term left-child))))
	   
	   ((and (eql right-child point)
		 (ilist-nil-p icdr op))
	    (let ((grand-parent (parent-of-dtree parent)))
	      (if (and (edit-ilist-cons-p grand-parent op)
		       (not (ilist-nil-p (icdrnt (term-of-dtree grand-parent)) op))
		       (not (eql parent (child-of-dtree grand-parent (left-mode)))))
		  ;;  x . p(a) . nil	->	p(x) . nil
		  (oed-edit-replace state grand-parent
				    (icons-term-of-op op
						      (ilabel-term 'point
								   (dtree-to-term
								    (child-of-dtree grand-parent
										    (left-mode))))
						      icdr))
		  ;;  p(a) . nil	->	p(a) . nil
		  state)))

	   ((or (eql right-child point) (eql left-child point))
	    (let ((grand-parent (parent-of-dtree parent)))
	      (if (and (edit-ilist-cons-p grand-parent op)
		       (not (eql parent (child-of-dtree grand-parent (left-mode)))))
		  ;;  x . p(a) . y . z	->	p(x) . y . z 
		  ;;  x . p(a) . b	->	p(x) . b
		  (oed-edit-replace state grand-parent
				    (icons-term-of-op op
						      (ilabel-term 'point
								   (dtree-to-term
								    (child-of-dtree grand-parent
										    (left-mode))))
						      (dtree-to-term right-child)))
		  ;;  p(a) . x . y	->	p(a) . x . y
		  state)))

	   (t (break) state))))
	   
	       ;;(let ((term (with-mark-maybe (nil (mark-of-edit-state state))
	       ;;(dtree-to-term grand-parent)))
	       ;;(edit-modify state grand-parent
	       ;;(icons-term-of-op op
	       ;;(ilabel-term 'point (icar term))
	       ;;(icdr (icdr term))))))

      (t state))))

;;;;	
;;;;	tags and labels messing up ops by hiding cons structure.
;;;;	
;;;;	

(defun oed-filter-temp-point-mark (term)
  (filter-term-tags term *temp-point-mark-filter*))

(defun oed-filter-point-mark (term)
  (filter-term-tags term *point-mark-filter*))

(defun oed-filter-point (term)
  (filter-term-tags term *point-filter*))

(defun oed-filter-mark (term)
  (filter-term-tags term *mark-filter*))


;;  p(nil)		->	p(*) . nil
;;  p(a) . nil		->	p(*) . a . nil
;;  a . p(nil)		->	a . p(*) . nil
;;  p(a) . b		->	p(*) . a . b
;;  a . p(b)		->	a . p(*) . b
;;  p(a) . x . y	->	p(*) . a . x . y
;;  x . p(a) . y . z	->	x . p(*) . a . y . z 
;;  x . p(a) . b	->	x . p(*) . a . b
;;  x . a . p(b)	->	x . a . p(*) . b
;;  x . p(a) . nil	->	x . p(*) . a . nil
;;  x . a . p(nil)	->	x . a . p(*) . nil

(defun edit-ilist-insert (state)
  (let* ((point (point-of-edit-state state))
	 (pterm (oed-filter-point (dtree-to-term point))))
    
    (cond
      ;;  a . p(nil)		->	a . p(*) . nil
      ;;  x . a . p(nil)	->	x . a . p(*) . nil
      ((edit-ilist-nil-element-p point)
       (oed-edit-replace state point
			 (icons-term-of-op (operator-of-term (term-of-dtree point))
					   (ilabel-term 'point (make-free-placeholder))
					   (oed-filter-point (dtree-to-term point)))
			 ))

      ((edit-ilist-element-p point)
       (let* ((parent (parent-of-dtree point))
	      (term   (oed-filter-point-mark (dtree-to-term parent)))
	      (op     (operator-of-term term))
	      (right-child (child-of-dtree parent (right-mode)))
	      (icdr   (term-from-dtree-tags-term (icdr term))))

	 ;;  p(a) . nil		->	p(*) . a . nil
	 ;;  p(a) . b		->	p(*) . a . b
	 ;;  a . p(b)		->	a . p(*) . b
	 ;;  p(a) . x . y	->	p(*) . a . x . y
	 ;;  x . p(a) . y . z	->	x . p(*) . a . y . z 
	 ;;  x . p(a) . b	->	x . p(*) . a . b
	 ;;  x . a . p(b)	->	x . a . p(*) . b
	 ;;  x . p(a) . nil	->	x . p(*) . a . nil
	 (cond
	   ((and (eql point right-child) (not (ilist-nil-p icdr op)))
	    (oed-edit-replace state point
			      (copy-term-labels
			       pterm
			       (icons-term-of-op op
						 (ilabel-term 'point (make-free-placeholder))
						 pterm))))
	   (t
	    (let ((pterm (oed-filter-point (dtree-to-term parent))))

	      ;;(setf -point point -pterm pterm) (break "eii")

	      (oed-edit-replace state parent
				(copy-term-labels
				 pterm
				 (icons-term-of-op op
						   (ilabel-term 'point (make-free-placeholder))
						   pterm) )))))))

      ;;  p(nil)		->	p(*) . nil
      ((edit-ilist-nil-p point)
       (oed-edit-replace state point
			 (copy-term-labels
			  pterm
			  (icons-term-of-op (operator-of-term (term-of-dtree point))
					    (ilabel-term 'point (make-free-placeholder))
					    pterm) )))

      (t state))))

(defun edit-ilist-transpose-aux (state)

  (let* ((point (point-of-edit-state state))
	 (parent (parent-of-dtree point))
	 )
    (setf -point point -parent parent) ;;(break "eita")
    (cond
      ((edit-ilist-cons-p point)
       (let* ((left-child (child-of-dtree point (left-mode)))
	      (right-child (child-of-dtree point (right-mode)))
	      (term (dtree-to-term point))
	      (terma  (term-from-dtree-tags-term term))
	      (op (operator-of-term terma)))
	
	 (if (or (not (edit-ilist-cons-p right-child op))
		 (edit-ilist-cons-p left-child op))

	     ;;  p(a . b)
	     ;;  p(a . nil)
	     state

	     ;;  p(a . b . z)	->	p(b . a . z)
	     (oed-edit-replace state point
			       (copy-term-labels
				term
				(ilabel-term 'point
					     (icons-term-of-op
					      op
					      (dtree-to-term (child-of-dtree right-child (left-mode)))
					      (icons-term-of-op
					       op
					       (dtree-to-term left-child)
					       (dtree-to-term (child-of-dtree right-child (right-mode))))))))	      
		   
	     )))
	    
      ;;  p(nil)
      ((edit-ilist-nil-p point)
       state)


      (t
       (let* ((left-child (child-of-dtree parent (left-mode)))
	      (right-child (child-of-dtree parent (right-mode)))
	      (term   (dtree-to-term parent))
	      (terma  (term-from-dtree-tags-term term))
	      (op     (operator-of-term terma))
	      (pterm  (oed-filter-point (dtree-to-term point))))

	 (setf -left-child left-child -right-child right-child -term term -terma terma -op op -pterm pterm)

	 (cond 
	   ;;  a . p(nil)		->	a . p(*) . nil
	   ;;  x . a . p(nil)	->	x . a . p(*) . nil
	   ((edit-ilist-nil-element-p point)
	    state)

	   ((edit-ilist-element-p point)
	    (let* ((grandparent (parent-of-dtree parent))
		   )

	      (cond
		;; point is first element of list.
		;;  p(a) . nil
		;;  p(a) . b		
		;;  p(a) . x . y	
		;;  a . p(b)	
		((or (null grandparent)
		     (not (edit-ilist-cons-p grandparent op)))
		 state)

		;;  x . a . p(b)
		((or (not (or (edit-ilist-cons-p right-child op)
			      (edit-ilist-nil-p right-child op)))
		     (not (eql left-child point)))
		 state)
	   
		;; point is last element of list.
		;;((ilist-nil-p icdr op)
		;;state)


		;;  x . p(a) . y . z	->	p(a) . x . y . z 
		;;  x . p(a) . b	->	p(a) . x . b
		;;  x . p(a) . nil	->	p(a) . x . nil

		(t (oed-edit-replace state grandparent
				     (copy-term-labels
				      (dtree-to-term grandparent)
				      (icons-term-of-op op
							(ilabel-term 'point pterm)
							(icons-term-of-op op
									  (dtree-to-term (child-of-dtree grandparent (left-mode)))
									  (dtree-to-term (child-of-dtree parent (right-mode)))))))
		   ))))

	   (t state)))))))

(defun edit-ilist-transpose (state)
  (edit-ilist-transpose-aux state))


;;  p(nil)		->	p(*) . nil
;;  p(a) . nil		->	a . p(*) . nil
;;  a . p(nil)		->	a . p(*) . nil
;;  p(a) . b		->	a . p(*) . b
;;  a . p(b)		->	a . b . p(*)
;;  p(a) . x . y	->	a . p(*) . x . y
;;  x . p(a) . y . z	->	x . a . p(*) . y . z
;;  x . p(a) . b	->	x . a . p(*) . b
;;  x . a . p(b)	->	x . a . b . p(*)
;;  x . p(a) . nil	->	x . a . p(*) . nil
;;  x . a . p(nil)	->	x . a . p(*) . nil

(defun edit-ilist-add (state)
  (let* ((point (point-of-edit-state state))
	 (pterm (oed-filter-point (dtree-to-term point))))
    (cond
      ;;  a . p(nil)		->	a . p(*) . nil
      ;;  x . a . p(nil)	->	x . a . p(*) . nil
      ((edit-ilist-nil-element-p point)
       (let* ((op (operator-of-term (term-of-dtree point))))
	 (oed-edit-replace
	  state point
	  (copy-term-labels
	   pterm
	   (icons-term-of-op op
			     (ilabel-term 'point (make-free-placeholder))
			     pterm))
	  )))
      
      ((edit-ilist-element-p point)
       (let* ((parent (parent-of-dtree point))
	      (term   (oed-filter-point-mark (dtree-to-term parent)))
	      (op     (operator-of-term term))
	      (right-child (child-of-dtree parent (right-mode)))
	      (icdr   (term-from-dtree-tags-term (icdr term))))

	 (cond
	   ;; x . a . p(b)	-> x . a . b . p(*)
	   ;; a . p(b)		-> a . b . p(*)
	   ((and (eql point right-child) (not (ilist-nil-p icdr op)))
	    (oed-edit-replace
	     state point
	     (copy-term-labels
	      pterm
	      (icons-term-of-op op
				pterm
				(ilabel-term 'point (make-free-placeholder))))
	     ))

	   ;;  p(a) . nil	->	a . p(*) . nil
	   ;;  p(a) . b		->	a . p(*) . b
	   ;;  p(a) . x . y	->	a . p(*) . x . y
	   ;;  x . p(a) . y . z	->	x . a . p(*) . y . z
	   ;;  x . p(a) . b	->	x . a . p(*) . b
	   ;;  x . p(a) . nil	->	x . a . p(*) . nil
	   (t (oed-edit-replace
	       state parent
	       (copy-term-labels
		pterm
		(icons-term-of-op op
				  pterm
				  (icons-term-of-op op
						    (ilabel-term 'point (make-free-placeholder))
						    icdr)))
	       )))))

      
      ;;  p(nil)		->	p(*) . nil
      ((edit-ilist-nil-p point)
       (let* ((op (operator-of-term (term-of-dtree point))))
	 (oed-edit-replace
	  state point
	  (copy-term-labels
	   pterm
	   (icons-term-of-op op
			    (ilabel-term 'point (make-free-placeholder))
			    pterm))
	  )))

      (t state))))


(defun edit-ilist-stack-push (l op)
  (term-stack-push (map-list-to-ilist l (instantiate-term op nil))))

(defun edit-ilist-stack-peek (op)
  (let ((term (term-stack-peek)))
    (when term (map-isexpr-to-list term op))))




;; does not implicitly nornalize ?
(defun edit-ilist-yank (state)
  (let ((point (point-of-edit-state state)))
    (if (or (dtree-leaf-p point)
	    (not (iplaceholder-term-p (term-of-dtree point))))
	(progn (message-emit (warn-message '(oed ilist yank placeholder not)))
	       state)
	(let ((term1 (term-stack-peek)))
	  (if (null term1)
	      state
	      (let* ((term (oed-filter-point-mark term1))
		     (new-state (oed-edit-replace state point (ilabel-term 'point term)))
		     (new-point (point-of-edit-state new-state)))
		
		;;(setf -a new-point -b new-state -c term) (break "iy")
		(cond
		  ((edit-ilist-cons-p new-point
				      (operator-of-term
				       (term-of-dtree (dtree-parent new-point))))
		   (let ((first (child-of-dtree new-point (left-mode)))
			 (last (edit-ilist-bottom new-point)))

		     (edit-move-label state 'mark
				      (if (edit-text-carrier-p last)
					  (only-child-of-dtree last)
					  last))
		     (when (edit-text-carrier-p last)
		       (edit-move-label-text-to-end (right-mode) 'mark state))
		     
		     (edit-move-label state 'point
				      (if (edit-text-carrier-p first)
					  (only-child-of-dtree first)
					  first))
		     (when (edit-text-carrier-p first)
		       (edit-move-label-text-to-end (left-mode) 'mark state))

		     (point-rehash state) )) 
		  
		  ((edit-text-carrier-p new-point)
		   (edit-move-label state 'mark
				    (only-child-of-dtree new-point))
		   (edit-move-label state 'point
				    (only-child-of-dtree new-point))
		   (edit-move-label-text-to-end (right-mode) 'mark state)		   
		   (edit-move-label-text-to-end (left-mode) 'point state)		   
		   
		   (point-rehash state))

		  (t
		   (edit-move-label state 'mark new-point)
		   state))))))))


;; --|-- /\ ----- /|\
(defun edit-ilist-push (state)
  (let ((mdtree (dtree-at-label 'mark state)))
    (when (null mdtree)
      (message-emit (warn-message '(oed ilist push mark not)))
      (return-from edit-ilist-push state))
    
    (let* ((pdtree (point-of-edit-state state))
	   (point-element (if (dtree-leaf-p pdtree)
			      (parent-of-dtree pdtree)
			      pdtree))
	   (point-term (dtree-to-term point-element))
	   (filtered-point-term (oed-filter-point-mark point-term))
	   (mark-element (if (dtree-leaf-p mdtree)
			     (parent-of-dtree mdtree)
			     mdtree))
	   (mark-term (dtree-to-term mark-element))
	   (filtered-mark-term (oed-filter-point-mark mark-term))
	   )

      ;;(setf -a pdtree -b point-element -c point-term -d mdtree -e mark-element -f mark-term)
      (labels
	  ((iterate (point op other)
	     (cond
	       ((eql point other) t)
	       ((edit-ilist-cons-p (parent-of-dtree point) op)
		(iterate (parent-of-dtree point) op other))
	       (t nil)))

	   (bottom-term (label bottom bottom-term)
	     (mapcar #'oed-filter-point-mark
	      (if (dtree-leaf-p bottom)
		  (list (car (text-carrier-split bottom-term
						 (index-of-dtree-leaf bottom label)
						 label)))
		  (list bottom-term))))

	   (top-term (label top top-term)
	     (oed-filter-point-mark
	      (if (dtree-leaf-p top)
		  (cdr (text-carrier-split top-term
					   (index-of-dtree-leaf top label)
					   label))
		  top-term)))

	   (extract (blabel bottom bottom-e bottom-term tlabel top top-e top-term)
	     (let ((nil-p (edit-ilist-nil-p bottom-e
					    (operator-of-term
					     (oed-filter-point-mark
					      (term-of-dtree (parent-of-dtree bottom-e)))))))
	       (do* ((point (if nil-p
				bottom-e
				(if (eql (child-of-dtree (parent-of-dtree bottom-e) (right-mode))
					 bottom-e)
				    bottom-e
				    (parent-of-dtree bottom-e)))
			    (parent-of-dtree point))
		     (acc (unless nil-p (bottom-term blabel bottom bottom-term))
			  (cons (dtree-to-term (child-of-dtree point
							       (left-mode)))
				acc)))
		    ((eql (parent-of-dtree point) (parent-of-dtree top-e))
		     (cons (top-term tlabel top top-term) acc))))))
	     
	(cond
	  ((and (dtree-leaf-p pdtree)
		(not (text-carrier-p filtered-point-term)))
	   (message-emit (warn-message '(oed ilist push point list not))))

	  ((and (dtree-leaf-p mdtree)
		(not (text-carrier-p filtered-mark-term)))
	   (message-emit (warn-message '(oed ilist push mark list not))))

	  ;; point mark on same dtree.
	  ;; not text sequence.
	  ((and (eql pdtree mdtree)
		(not (dtree-leaf-p pdtree))
		(not (dtree-leaf-p mdtree)))
	   (if (edit-ilist-nil-p pdtree)
	       (edit-ilist-stack-push nil
				      (operator-of-term (term-of-dtree point-element)))
	       (edit-ilist-stack-push (list point-term)
				      (let ((parent (parent-of-dtree point-element)))
					(when (and parent
						   (edit-ilist-cons-p parent))
					  (operator-of-term
					   (term-of-dtree parent)))))))

	  ;; point mark on same dtree.
	  ;; text sequence.
	  ((and (eql pdtree mdtree)
		(dtree-leaf-p pdtree)
		(dtree-leaf-p mdtree))
	   (edit-ilist-stack-push
	    (list (itext-term (implode-to-string
			       (isegment-of-dtree-leaf pdtree 'point 'mark))))
	    (let ((parent (parent-of-dtree point-element)))
	      (when (and (edit-ilist-cons-p parent) parent)
		(operator-of-term (term-of-dtree parent))))))
	  
	  ((eql point-element mark-element)
	   (message-emit (warn-message '(oed ilist push mark text term inconsistent))))
	   
	  ;; siblings. 
	  ((and (edit-ilist-cons-p (parent-of-dtree point-element))
		(eql (parent-of-dtree point-element)
		     (parent-of-dtree mark-element)))
	   (edit-ilist-stack-push (if (eql (child-of-dtree (parent-of-dtree point-element)
							   (right-mode))
					   point-element)
				      (extract 'point pdtree point-element point-term
					       'mark mdtree mark-element mark-term)
				      (extract 'mark mdtree mark-element mark-term
					       'point pdtree point-element point-term))
				  (operator-of-term
				   (term-of-dtree (parent-of-dtree point-element)))))
	  
	  ((and (edit-ilist-cons-p (parent-of-dtree point-element))
		(iterate (parent-of-dtree point-element) nil
			 (parent-of-dtree mark-element)))
	   (edit-ilist-stack-push (extract 'point pdtree point-element point-term
					   'mark mdtree mark-element mark-term)
				  (operator-of-term
				   (term-of-dtree (parent-of-dtree point-element)))))

	  ((and (edit-ilist-cons-p (parent-of-dtree mark-element))
		(iterate (parent-of-dtree mark-element) nil
			 (parent-of-dtree point-element)))
	   (edit-ilist-stack-push (extract 'mark mdtree mark-element mark-term
					   'point pdtree point-element point-term)
				  (operator-of-term
				   (term-of-dtree (parent-of-dtree point-element)))))
	  (t (message-emit (warn-message '(oed ilist push point mark lists differ)))))))
	
    state))




(defun edit-ilist-delete-segment (state)
  (let ((mdtree (dtree-at-label 'mark state)))
    (when (null mdtree)
      (message-emit (warn-message '(oed ilist segment mark not)))
      (return-from edit-ilist-delete-segment state))
    
    (let* ((pdtree (point-of-edit-state state))
	   (point-element (if (dtree-leaf-p pdtree)
			      (parent-of-dtree pdtree)
			      pdtree))
	   (point-term (dtree-to-term point-element))
	   (mark-element (if (dtree-leaf-p mdtree)
			     (parent-of-dtree mdtree)
			     mdtree))
	   (mark-term (dtree-to-term mark-element)))

      (labels
	  ((iterate (point op other)
	     (cond
	       ((null point) nil)
	       ((eql point other) t)
	       ((edit-ilist-cons-p (parent-of-dtree point) op)
		(iterate (parent-of-dtree point) op other))
	       (t nil)))

	   (bottom-term (label bottom bottom-term)
	     (oed-filter-point-mark
	      (if (dtree-leaf-p bottom)
		  (car (text-carrier-split bottom-term
					   (index-of-dtree-leaf bottom label)
					   label))
		  bottom-term)))
	   
	   (top-term (label top top-term)
	     (oed-filter-point-mark
	      (if (dtree-leaf-p top)
		  (cdr (text-carrier-split top-term
					   (index-of-dtree-leaf top label)
					   label))
		  top-term)))

	   (extract (blabel bottom bottom-e bottom-term tlabel top top-e top-term)
	     (let* ((bottom-parent (parent-of-dtree bottom-e))
		    (op (operator-of-term (term-of-dtree bottom-parent)))
		    (slot (make-free-placeholder))
		    (bottom-right (child-of-dtree bottom-parent (right-mode)))
		    (bottom-split (when (dtree-leaf-p bottom)
				    (cdr (text-carrier-split bottom-term
							     (index-of-dtree-leaf bottom blabel)
							     blabel))))
		    (rest (mapcar #'oed-filter-point-mark
				  (if bottom-split
				      (if (eql bottom-e bottom-right)
					  nil
					  (map-isexpr-to-list (icdr (dtree-to-term bottom-parent)) op))
				      (if (edit-ilist-nil-p bottom-e
							    (operator-of-term
							     (term-of-dtree bottom-parent)))
					  nil
					  (if (eql bottom-e bottom-right)
					      nil
					      (map-isexpr-to-list (dtree-to-term bottom-right) op))))))
		    (top-split (when (dtree-leaf-p top)
				 (let ((i (index-of-dtree-leaf top tlabel)))
				   (when (not (zerop i))
				     (car (text-carrier-split top-term i tlabel))))))
		    (terms (append (when top-split (list top-split))
				   (list (ilabel-term 'point slot))
				   (when bottom-split (list bottom-split))
				   rest))
		    (new-ilist (map-list-to-ilist terms
						  (instantiate-term   
						   (operator-of-term
						    (term-of-dtree bottom-parent))))))

	       (oed-edit-replace state
				 (if (eql top-e bottom-right)
				     top-e
				     (parent-of-dtree top-e))
				 new-ilist)
	       )))
			      
	(cond
	  ((or (and (dtree-leaf-p pdtree)
		    (not (text-carrier-p (oed-filter-point-mark point-term))))
	       (not (or (edit-ilist-element-p point-element)
			(edit-ilist-nil-p point-element))))
	   (message-emit (warn-message '(oed ilist segment point list not))))

	  ((or (and (dtree-leaf-p mdtree)
		    (not (text-carrier-p (oed-filter-point-mark mark-term))))
	       (not (or (edit-ilist-element-p point-element)
			(edit-ilist-nil-p point-element))))
	   (message-emit (warn-message '(oed ilist segment mark list not))))

	  ((and (eql pdtree mdtree)
		(or (and (dtree-leaf-p pdtree) (not (dtree-leaf-p mdtree)))
		    (and (dtree-leaf-p mdtree) (not (dtree-leaf-p pdtree)))))
	   (message-emit (warn-message '(oed ilist segment point mark same not))))
	
	  ((and (eql pdtree mdtree)
		(dtree-leaf-p pdtree)
		(dtree-leaf-p mdtree))
	   (edit-delete-text-segment-at-labels 'point 'mark state))
	  
	  ((eql (parent-of-dtree point-element) (parent-of-dtree mark-element))
	   (if (eql point-element (child-of-dtree (parent-of-dtree point-element)
						  (right-mode)))
	       (extract 'point pdtree point-element point-term 'mark mdtree mark-element mark-term)
	       (extract 'mark mdtree mark-element mark-term 'point pdtree point-element point-term)))

	  ((iterate (parent-of-dtree point-element) nil
		    (parent-of-dtree mark-element))
	   (extract 'point pdtree point-element point-term 'mark mdtree mark-element mark-term))
	  
	  ((iterate (parent-of-dtree mark-element) nil
		    (parent-of-dtree point-element))
	   (extract 'mark mdtree mark-element mark-term 'point pdtree point-element point-term))

	  (t (message-emit (warn-message '(oed ilist segment point mark lists differ))))))))

  state)


(defun edit-ilist-normalize (state)
  (let* ((point (point-of-edit-state state))
	 (point-dtree (if (dtree-leaf-p point)
			  (parent-of-dtree point)
			  point)))

    (if (edit-ilist-element-p point-dtree)
	(if (edit-ilist-nil-element-p point-dtree)
	    (progn
	      ;;(setf -point point -point-dtree -point-dtree) (break "ein")
	      (message-emit (warn-message '(oed conjoin point nil)))
	      state)
	    (if (edit-ilist-normal-p point-dtree)
		state
		(let* ((op (operator-of-term (term-of-dtree (parent-of-dtree point-dtree))))
		       (top (edit-ilist-top (parent-of-dtree point-dtree) op))	
		       (top-term (map-list-to-ilist (map-isexpr-to-list (dtree-to-term top) op)
						    (instantiate-term op))))
		  ;;(setf -top-term top-term b top) (break)
		  (oed-edit-replace state top top-term))))
	state)))


;; may desire to include other marks when merging, but on the other hand expect
;; to lose marks during destructive mods. But not labels.
(defun text-carrier-join (a b)
  (let* ((a-parm (parameter-of-text-carrier a))
	 (b-parm (parameter-of-text-carrier b))
	 (a-marks (marks-of-parameter a-parm))
	 (b-marks (marks-of-parameter b-parm)))
	 
    ;;(setf -a-parm a-parm -b-parm b-parm -a-marks a-marks -b-marks b-marks) (break "tcj")

    (let* ((a-str (string-of-itext-term a))
	   (b-str (string-of-itext-term b))
	   (s (cond
		((and a-str b-str) (concatenate 'string a-str b-str))
		(a-str a-str)
		(b-str b-str)
		(t "")))

	   (p (instantiate-parameter-s s (type-of-parameter a-parm))))
	   
      (let* ((b-indices (cdr (assoc 'edit-indices b-marks)))
	     (a-indices (cdr (assoc 'edit-indices a-marks)))
	     (indices a-indices))
		  
	(setf -b-indices b-indices -a-indices a-indices) 	    
	(dolist (b-index b-indices)
	  (unless (member (car b-index) a-indices :key #'car)
	    (push (cons (car b-index)
			(+ (cdr b-index) (length a-str)))
		  indices)))
		 
	(when indices
	  ;;(setf -indices indices) (break "text-carrier-join")
	  (mark-parameter p 'edit-indices indices))

	(let ((a-tags (cdr (assoc 'tags a-marks)))
	      (b-tags (cdr (assoc 'tags b-marks))))

	  
	  (when (or a-tags b-tags)
	    (mark-parameter p 'tags (new-tags-and-labels nil
							 (union (when a-tags (labels-of-tags a-tags))
								(when b-tags (labels-of-tags b-tags))))))
	  
	  (instantiate-term (instantiate-operator (id-of-term a)
						  (list p))
			    nil)
	  )))))

(defun mark-parameter (p mark value)
  (setf (parameter-value p)
	(mark-parameter-value (value-of-parameter-n p) mark value)))
			 

(defun conjoin (terms)
  ;;(setf -terms terms) (break "cj")		       
  (if (null terms)
      nil
      (let* ((cdr (conjoin (cdr terms)))
	     (car (car terms))
	     (next (unless (null cdr) (car cdr))))
	(mlet* (((a labels-a tags-a) (decode-dtree-tags car))
		((b labels-b tags-b) (when next (decode-dtree-tags next))))
	       ;;(when (let ((p (car (parameters-of-term a)))) (and p (meta-parameter-p p)))
	       ;;(setf -a a -b b -car car -cdr cdr -next next)
	       ;;(break "cjt"))
	       (if (and b
			(conjoinable-p a)
			(text-carrier-p b a)
			(conjoinable-p b))
		   (progn
		     ;;(setf -labels-a labels-a -labels-b labels-b) (break "cjo")
		     (cons (encode-dtree-tags (text-carrier-join a b)
					      (union labels-b labels-a)
					      (union tags-b tags-a))
			   (cdr cdr)))
		   (progn
		     ;;(setf a car b cdr c next e point) ;(break "cdring")
		     (if (and (text-carrier-p a)
			      (display-meta-parameter-p (car (parameters-of-term a))))
			 ;; convert display meta parameter to null string ?? with point??
			 (progn (break "wow")
			 (cons (encode-dtree-tags (instantiate-text-carrier car "" 'point 0)
						  labels-a tags-a)
			       cdr)
			 )
			 (cons car cdr))))))))



;; apparently ensures an alternating sequence of text and non-text terms.
(defun conjoin-normalize-nulls (l)
  (let* ((null-carrier (or (let ((carrier (find-first #'(lambda (x)
							  (let ((term (term-from-dtree-tags-term x)))
							    (when (conjoinable-p term)
							      term)))
						      l)))
			     (when carrier
			       (instantiate-text-carrier carrier "")))
			   (itext-term ""))))
    (if (null l)
	(list null-carrier)
	(do* ((terms l (cdr terms))
	      (acc (if (conjoinable-p (term-from-dtree-tags-term (car terms)))
		       (list (car terms))
		       (list (car terms) null-carrier))
		   (if (null terms)
		       acc
		       (cons (car terms) acc))))
	     ((null terms) (nreverse acc))
	  (unless (or (conjoinable-p (term-from-dtree-tags-term (car terms)))
		      (and (cdr terms)
			   (conjoinable-p (term-from-dtree-tags-term (cadr terms)))))
	    (push null-carrier acc))))))


(defun ancestors (dtree)
  (cons dtree
	(unless (null (dtree-parent dtree))
	  (ancestors (dtree-parent dtree))))
  )

(defun edit-conjoin-aux (state)
  (let* ((point-dtree (point-of-edit-state state))
	 (point-element (if (dtree-leaf-p point-dtree)
			    (parent-of-dtree point-dtree)
			    point-dtree))
	 )

    (unless (and (edit-ilist-element-p point-element)
		 (not (edit-ilist-nil-p point-element))
		 (or (not (dtree-leaf-p point-dtree))
		     (edit-text-carrier-p point-element)))
      
      ;;(setf -v state -point-dtree point-dtree -point-element point-element) (break "eja")
      (display-msg "Conjoining requires point at element or text of list.")
      ;;(setf a point-element) (break)
      (return-from edit-conjoin-aux state))

    ;; make sure point gets into term. but this is a problem for any label in dtree
    ;; but not in term.
    ;;(dtree-path-term-modified point-dtree 'tag)

    ;;(setf -pd point-dtree -pe point-element) (break "ecj0")
    (let* ((dtree-list (edit-ilist-top point-element
				       (if (edit-ilist-element-p point-element)
					   (operator-of-term
					    (term-of-dtree (parent-of-dtree point-element)))
					   (operator-of-term (term-of-dtree point-element)))))
	   ;; what if label on cons op.
	   (term-list (map-isexpr-to-list (setf -yo (dtree-to-term dtree-list nil))
					  (operator-of-term (term-of-dtree dtree-list)))))
      ;;(setf -tl term-list -dl dtree-list -pd point-dtree -pe point-element) (break "ecj")
      (let ((joined (conjoin (conjoin-normalize-nulls term-list))))

	;;(setf -tl term-list -j joined -dl dtree-list -pd point-dtree -pe point-element) (break "ecj2")
	(oed-edit-replace state dtree-list
			  (map-list-to-ilist joined
					     (instantiate-term (operator-of-term
								(term-of-dtree dtree-list)))))))))

(defun edit-conjoin (state)
  (edit-conjoin-aux state))


;;; beware : point/mark in text requires the dtree-leaf be tagged. Presence of point/mark in
;;; edit indices is not the same.


;; todo: initial position in text leaf??
;; labels ?
;; 1: (a . ( b . rest))	-> (ab . rest)
;; 2: (a . b)		-> ab
;; 
;;  ab gets labels from a and b.
;;  in 1 cons gets labels from conses.
;;  in 2 ab gets label from cons.
(defun edit-text-join (state)
  (let ((point (point-of-edit-state state)))
    (when (not (dtree-leaf-p point))
      (let* ((tterm (filter-term-tags (dtree-to-term point nil)
				      *point-filter*))
	     (term (term-from-dtree-tags-term tterm)))
	(when (weak-ilist-cons-p term nil)
	  (let* ((first (term-from-dtree-tags-term (icar term)))
		 (icdr (term-from-dtree-tags-term (icdr term)))
		 (tsecond (if (ilist-cons-p icdr (operator-of-term term))
			      (icar icdr)
			      (icdr term)))
		 (second (term-from-dtree-tags-term tsecond)))

	    (when (and (text-carrier-p first)
		       (text-carrier-p second first))

	      (let ((text (conjoin (list (icar term) tsecond))))

		(if (ilist-cons-p icdr (operator-of-term term))
		    (edit-down-left
		     (edit-down-left
		      (oed-edit-replace state point
					(copy-term-labels
					 tterm
					 (icons-term-of-op (operator-of-term term)
							   text
							   (icdr icdr))
					 (union '(point) (labels-of-term icdr))))))
		    (edit-down-left
		     (oed-edit-replace state point
				       (copy-term-labels tterm text '(point)))))))))))
    state))


(defun edit-text-split (state)
  (let ((point (point-of-edit-state state)))
    (when (dtree-leaf-p point)
      (let ((parent (parent-of-dtree point)))
	(when (and (edit-ilist-element-p parent)
		   (text-carrier-p (term-of-dtree parent)))
	  (let* ((grand-parent (parent-of-dtree parent))
		 (right-child (child-of-dtree grand-parent (right-mode)))

		 (tterm (dtree-to-term grand-parent nil))
		 (term (term-from-dtree-tags-term tterm))
		 (op (operator-of-term term))
		 (carriers (text-carrier-split (oed-filter-point-mark (dtree-to-term parent nil))
					       (index-of-dtree-leaf point 'point) 'point)))

	    ;; save labels but lose tags on parent
	    (if (eql parent right-child)

		(oed-edit-replace state parent
				  (encode-dtree-tags (icons-term-of-op op
								       (car carriers)
								       (cdr carriers))
						     (union (labels-of-dtree parent) '(point))
						     nil))

		(oed-edit-replace
		 state grand-parent
		 (copy-term-labels tterm
				   (icons-term-of-op op
						     (car carriers)
						     (icons-term-of-op op
								       (cdr carriers)
								       (icdr term)))
				   (union '(point) (labels-of-dtree parent))))))))))
  state)
