
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************

#+cltl2(in-system-package)
#-cltl2(in-package *system-package-name*)

;;;;	
;;;;	
;;;;	
;;;;	Proof is inf-tree. Inf tree is tree of object-ids of inf objects.
;;;;	  Proof-editor shows a view of some inf-tree. 
;;;;	  Proof-editor may encode multiple inf-trees. Particularly it would
;;;;	    be good to allow tree variants. 
;;;;	  proof in editor does not need to match proof in library.
;;;;	  proof-save should dump some inf-tree to lib. 
;;;;	
;;;;	lib may need some work to support this model easier.
;;;;	
;;;;	
;;;;	
;;;;	variants : generalize !proof_editor so that each !proof_editor term could be 
;;;;	 some variant list then have commands which can rotate through the list.
;;;;	 maybe !proof-variants(<proof-editor{top}>; <proof-editor> stack)
;;;;	 then children of proof-editor may(must) then be !proof-variants.
;;;;	a variant tree would in some view look like a simple tree but each subtree is actually
;;;;	a stack of different variations. The essential feature we need is to have previous
;;;;	children available after a re-refine. 
;;;;
;;;;	 save then walks over tree and pulls out oids of nodes of tops.
;;;;
;;;;	 variants need not be saved between edit sessions but could save by
;;;;	 having some variant inf-tree property on the proof. 
;;;;	
;;;;	  variants could be in top term but not in edit term, although
;;;;	  if in edit term then avail to user via dform suppression.
;;;;	
;;;;	stack should be hidden by display forms
;;;;	
;;;;	need edit commands to rotate-stack, pop top,
;;;;	result of refine should implicitly push onto stack. as well as replace top.
;;;;	then rotating means rotate stack then place copy in top. ie top and top-of stack
;;;;	identical after rotate and refine.
;;;;
;;;;	need to be able to push walk down to a subtree, push subtree then walk to another tree
;;;;	and yank pushed tree to replace point tree. Might check if goal matches.
;;;;	yank should be similar to refine and push yanked onto stack as well as replace top.
;;;;	push should be usual term push but need special purpose proof-tree-yank
;;;;
;;;;	all these edit commands should be written in ml with the current tree as arg.
;;;;	ie proof-edit-cmd : proof-editor{variants?} -> proof-editor{variants?}
;;;;	
;;;;	I re-implemented down, and added a left and right, see oed_proof_down|left|right for examples.
;;;;	left and right only work if parent is visible. Twould be nice to have it work
;;;;	when current is top visible. 
;;;;	This is a different down, which just goes to first child. I like your version too but maybe
;;;;	move one level down toward the point rather than jump to the point.
;;;;	
;;;;	I added properties to views so you could write most or not all of ref funcs as ml funcs
;;;;	where the proof-object data is stored on the property list.
;;;;	Redoing up and fixing left/right may be a worthwhile exercise. 
;;;;	
;;;;	if a primitive requires a previous act like the push before down then,
;;;;	down should check validity of arg and noop or fail if not correct. 
;;;;	doing primitives incorrectly should do as little damage as possible.
;;;;	
;;;;	
;;;; Proof Editor:
;;;;
;;;;
;;;; The proof editor window consists of a single term.  The term contains the main goal,
;;;; subgoals, tactics, and other useful information about the proof (ie. completeness status
;;;; and node addresses).  Proof annotations are supported, but hidden.  Information specific
;;;; to the refinement such as dependencies, extracts, primitives, can be made available.
;;;; 
;;;; Currently, the user has full capacity to edit any part of this term, however it seems
;;;; advantageous to disallow some edits (for example, assumptions of generated subgoals) in
;;;; the future.  Note, even though the user can thus create the appearence of a bogus proof
;;;; in the window, storage of such a "mal-editted" proof as will be refused in the library.
;;;;
;;;; The structure of the proof editor term is described below.
;;;; 
;;;; <proof-editor>	:!proof_editor{<view-mode>:t}(<proof-node>)
;;;; <view-mode>	: d = default, t = tactic tree, a = address tree, v = verbose
;;;; <proof-node>	:!proof_node{<inf>:o, depth:n, <view-mode>:t, count:n}
;;;;			  (<status>; <address>; <goal>; <refinement>; <subgoals>; <annotations>)
;;;;	<status>	   :!proof_status{complete:t | incomplete:t | unknown:t |failed:t}()
;;;;	<address>	   :!pui_addr_cons{}(...)  ;;address of node in proof
;;;;				ie. nil is top, '(2 1) is first subgoal of second subgoal of prf
;;;;	<goal>		   :can be any term
;;;;    <refinement>	   :!cons_left{}(<tactic> <ref-info>) ;;cons_left display form hides
;;;;			    ref-info (showing only "left" of cons term)
;;;;    <subgoals>	   :!proof_node_cons{}(...) cons of !proof_node terms
;;;;    <annotations>	   :!annotation_cons{}(...)
;;;;
;;;;
;;;; Movements of down to child, up to parent, and up to top of proof are supported.
;;;; Currently all display is in one window, but design supports easy adaptation to
;;;; multiple window mode (user may wish to view and work on multiple subgoals simultaneously).
;;;;
;;;; Window Object Parameters:
;;;; The main proof editor window maintains the following parameters:
;;;;	poid	 : object ID of proof (static)
;;;;	soid-list: object IDs of statements/lemmas that are not allowed during refinements
;;;;	top-proof: proof node term representation of entire proof at present (dynamic),
;;;;		   as stored in lib
;;;;	eddtop-proof: proof node term representation of entire proof at present (dynamic),
;;;;		   not necessarily same as lib
;;;;	history: walks of previous top-proofs (dynamic)
;;;;
;;;;
;;;; Editor/Refiner interaction:
;;;;
;;;; The proof editor expects an object id argument, which may be newly created.
;;;; Slots for entering the proof goal and tactic are made available to the user when opening
;;;; proof editor.  When the user requests "refine", the editor sends the proof node,
;;;; tactic, poid, and oid list to library.  The library acquires the description property of
;;;; the poid's object contents, which describes the refiner to use.  The inf object contents,
;;;; containing the goal and annotations of the proof node, as well as the tactic and oid list
;;;; are sent to this refiner, an inf tree is returned.  This inf tree is replaced in the proof
;;;; at the address of the proof node.  Goal matching is enforced at this step.  The inf tree is
;;;; then converted to a proof editor term, where subgoals are numbered, which is sent back 
;;;; to the editor for displaying and window object parameters are updated.
;;;;


(defvar *default-prf-geo* (list 8 2 84 58)) ;;default geometry of proof window
;;view modes are addresses only (a), default or normal (d), tactics (t), and verbose (v)
(defvar *view-modes*
   (acons `|a| `|a| (acons `|d| `|d| (acons `|t| `|t| (acons `|v| `|v| nil)))))
(defvar *default-view* (intern-system "d")) ;;view mode for proof is set to normal
(defvar *max-depth* 4) ;;max number of ancestor nodes to be shown in the window
(defvar *max-depth-complete* 4) ;; max number of complete ancestor nodes to be shown
(defvar *max-nhyps* 20) ;;max number of hyps above which we set the depth differently
(defvar *max-depth-whyps* 1) ;;max depth given nhyps exceeds limit above

(define-primitive |!refine_failure| () (term))
(define-primitive |!ttree| () (tac children))
(define-primitive |!tgtree| () (tac ioid subgoals children))
(define-primitive |!ttree_cons| () (car cdr))
(define-primitive |!tgtree_cons| () (car cdr))
(define-primitive |!hidden_subgoals| ((natural . count)))  
(define-primitive |!proof_lib| ((token . view) (natural . depth)) (address goal tactic rhyps))
(define-primitive |!pending_refinement| () (tactic))

(define-primitive |ped-THEN| () (left right))
(define-primitive |ped-THENL| () (tactic list))
(defvar *THEN-abs-oid*)
(defvar *THENL-abs-oid*)

(defun proof-node-template ()
  (iproof-node-term (slot-parameter-value "slot") ;; inf oid slot
		    0 ;; depth
		    *default-view* ;; proof view mode 
		    0 ;; count

		    (iproof-status-term (intern-system "incomplete"))
		    (ipui-addr-nil-term)
		    (ilabel-term 'point (iplaceholder-term)) ;; goal

		    (icons-left-term (itext-term "") ;; tactic slot
				     (ivoid-term)) ;; refinement information
		    (iproof-node-nil-term) ;; subgoals
		    (iannotation-nil-term)))

(defun modify-proof-node (node &key status goal subgoals refinement annotations address view)
  ;;(break "mod")
  (iproof-node-term (oid-of-iproof-node-term node)
		    (depth-of-iproof-node-term node)
		    (or view (view-of-iproof-node-term node))
		    (count-of-iproof-node-term node)
		    (or status (status-of-iproof-node-term node))
		    (or address (address-of-iproof-node-term node))
		    (or goal (goal-of-iproof-node-term node))      
		    (or refinement (refinement-of-iproof-node-term node))
		    (or subgoals (subgoals-of-iproof-node-term node))
		    (or annotations (annotations-of-iproof-node-term node))))

(defstruct (proof-object (:include vobject))
  (oid-list nil)			; statement oid; others to ignore when proving
  (top-proof (proof-node-template))	; proof-node-term of entire proof in lib
  (eddtop-proof (proof-node-template))	; proof-node-term of entire proof in edd
  (refiner '|nv5|)			; currently not used, refiner is a property of oid
  (refresh-term nil)
  (history-walk nil)
  (history (new-proof-history))
  (variants *view-modes*)		; variants of top-proof for different kinds of display,
					; assoc list of view-mode * term
  )

(defun oid-list-of-proof-object (o) (proof-object-oid-list o))
(defun oid-of-proof-object (o) (proof-object-oid o))
(defun top-proof-of-proof-object (o) (proof-object-top-proof o))
(defun eddtop-proof-of-proof-object (o) (proof-object-eddtop-proof o))
(defun name-of-proof-object (o) (name-property-of-ostate (oid-of-vobject o)))
(defun refiner-of-proof-object (o) (proof-object-refiner o))
(defun history-walk-of-proof-object (o) (proof-object-history-walk o))
(defun history-of-proof-object (o) (proof-object-history o))
(defun variants-of-proof-object (o) (proof-object-variants o))
(defun refresh-term-of-proof-object (o) (proof-object-refresh-term o))
(defun nl-refiner-p (o) (equal (refiner-of-proof-object o) '|nl|))

(defun state-of-proof-node-term (term)
  (state-of-iproof-status-term (status-of-iproof-node-term term)))

(defun proof-node-complete-p (node)
  (eql '|complete| (state-of-proof-node-term node)))

(defun proof-node-incomplete-p (node)
  (or (eql '|failed| (state-of-proof-node-term node))
      (eql '|incomplete| (state-of-proof-node-term node))))

(defun limit-depth (term)
  (let* ((label-p (real-ilabel-term-p term))
	 (proof-node (label-unwrap term))
	 (nhyps (nhyps-of-proof-node-term proof-node))
	 (base-depth (depth-of-iproof-node-term proof-node))
	 (max-depth (+ base-depth (if (> nhyps *max-nhyps*) *max-depth-whyps* *max-depth*)))
	 (max-depth-complete (+ base-depth (if (> nhyps *max-nhyps*)
					       *max-depth-whyps*
					       *max-depth-complete*))))

    (labels ((visit (node)
	       (let* ((subs (subgoals-of-iproof-node-term node))
		      (depth (depth-of-iproof-node-term node))
		      (sublist (map-isexpr-to-list subs (iproof-node-cons-op)
						   #'(lambda (x)
						       (if (real-ilabel-term-p x)
							   (label-wrap (visit (label-unwrap x)))
							   (visit x)))))
		      (count (length sublist)))
		 
		 (if (> count 0)
		     (if (>= depth max-depth-complete)
			 (let* ((incomplete-subgoals (filter #'proof-node-incomplete-p sublist))
				(i (length incomplete-subgoals))
				(j (- count i)))
		     
			   (if (>= depth max-depth)					
			       (modify-proof-node node
						  :subgoals (iproof-node-nil-term)
						  :annotations
						  (iannotation-cons-term
						   (ihidden-subgoals-term (+ i j)) ;; lal give more info
						   (annotations-of-iproof-node-term node)))

			       (let ((modified-subgoals
				      (map-list-to-isexpr incomplete-subgoals
							  (iproof-node-nil-term))))
				 (if (> j 0)
				     (modify-proof-node node
							:subgoals modified-subgoals
							:annotations
							(iannotation-cons-term
							 (ihidden-subgoals-term j)
							 (annotations-of-iproof-node-term node)))
				 
				     (modify-proof-node node :subgoals modified-subgoals)))))
		   
			 (if (>= (depth-of-iproof-node-term node) max-depth)
			     (modify-proof-node node
						:subgoals (iproof-node-nil-term)
						:annotations
						(iannotation-cons-term
						 (ihidden-subgoals-term count) ;; lal give more info
						 (annotations-of-iproof-node-term node)))
			     (modify-proof-node
			      node
			      :subgoals (map-list-to-isexpr sublist (iproof-node-nil-term)))))
		     node))))      
      
      (if label-p
	  (label-wrap (visit proof-node))
	  (visit proof-node)))))

(defun replace-hidden-subgoals (subs target proof) 
  (let ((l nil))
    (labels ((visit (s ta)
	       (if (null s)
		   (reverse l)
		   (let ((first (car s))
			 (node (car ta)))
		     (if (and node (compare-terms-p (address-of-iproof-node-term first)
						    (address-of-iproof-node-term node)))
			 (progn (push (unlimit-depth node proof) l)
				(visit (cdr s) (cdr ta)))
			 (progn (push first l)
				(visit (cdr s) ta)))))))      
      (visit subs target))))
		   					    

(defun unlimit-depth (term proof)  
  (let ((annos (filter #'ihidden-subgoals-term-p
		       (map-isexpr-to-list (annotations-of-iproof-node-term term)
					   (iannotation-cons-op)))))
    (if annos
	(let ((subs (subgoals-of-iproof-node-term
		     (find-node-at-address proof (address-of-proof-node-term term)))))
	  
	  (remove-hidden-subgoal-annotations
	   (modify-proof-node term
			      :subgoals (map-list-to-isexpr
					 (replace-hidden-subgoals
					  (map-isexpr-to-list subs (iproof-node-cons-op))
					  (map-isexpr-to-list (subgoals-of-iproof-node-term term)
							      (iproof-node-cons-op)) proof)
					 (iproof-node-nil-term)))))	
	(modify-proof-node term
			   :subgoals 
			   (map-list-to-isexpr
			    (map-isexpr-to-list (subgoals-of-iproof-node-term term)
						(iproof-node-cons-op)
						#'(lambda (x) (unlimit-depth x proof)))
			    (iproof-node-nil-term))))))

;;when ready, also put button wrap here
(defun proof-editor-wrap (node &optional view)
  (iproof-editor-term (or view *default-view*) node))
    
(defun proof-editor-wrap-with-depth (node &optional view)
  (iproof-editor-term (or view *default-view*) (limit-depth node)))

(defun proof-editor-unwrap (term)
  (node-of-iproof-editor-term term))

(defun proof-editor-unwrap-with-depth (term proof)
  (unlimit-depth (node-of-iproof-editor-term term) proof))
    
(defun new-proof-object (oid term geo implicit &optional oids)
  (make-proof-object :oid oid
		     :term (proof-editor-wrap-with-depth term)
		     :implicit implicit
		     :geo geo
		     :top-proof term
		     :eddtop-proof term
		     :oid-list (or oids (list oid))))

(defun edit-step-refine-term (term oid oids)
  (declare (ignore oids))
  (unless (oid-p oid)
    (raise-error (error-message '(orb get-inf-term oid not))))

  (funmlcall (ml-text "step_refinet_term") term oid))

(defun edit-refine-term (term oid oids)
  (declare (ignore oids))
  (unless (oid-p oid)
    (raise-error (error-message '(orb get-inf-term oid not))))
  (funmlcall (ml-text "refinet_term") term oid))

(defun edit-refine-tree-term (term oid oids)
  (unless (oid-p oid)
    (raise-error (error-message '(orb tree-term oid not))))
  (funmlcall (ml-text "refine_tree ") term oid oids))

(defun edit-refine-tttt-tree-term (term oid oids)
  (unless (oid-p oid)
    (raise-error (error-message '(orb tree-term oid not))))
  (funmlcall (ml-text "refine_tttt_tree ") term oid oids))

(defun edit-refine-edit-tree-term (term oid oids)  
  (unless (oid-p oid)
    (raise-error (error-message '(orb tree-term oid not))))
  (funmlcall (ml-text "edit_refine_tree ") term oid oids))

(defun get-inf-term-at-address (oid address)
  (unless (oid-p oid)
    (raise-error (error-message '(orb get-inf-term-at-address oid not))))
  (funmlcall (ml-text "get_inf_term_at_address ") oid address))

(defun get-inf-term (oid)
  (unless (oid-p oid)
    (raise-error (error-message '(orb get-inf-term oid not))))
  (or (with-ignore (funmlcall (ml-text "get_inf_term ") oid))
      (proof-node-template)))

(defun instantiate-proof-object (oid &optional oids node)	 
  (let ((o (new-proof-object oid
			     (or node (get-inf-term oid))
			     (or (edit-read-geo oid) *default-prf-geo*)
			     (edit-read-implicit oid)
			     oids)))      
    (proof-update-history o (top-proof-of-proof-object o))      	
    o))

;; during refine, proof oid gets saved in lib and notifies edd, which sets
;; touch flag, but we don't want to re-instantiate at this time since we
;; are getting result back from lib (or cur-obj (instant..) is a fix fttb
(defun edit-proof-refresh (v)  
  (let* ((oid (oid-of-view v))
	 (cur-vobj (object-of-view v))
	 (rterm (when cur-vobj (refresh-term-of-proof-object cur-vobj))))

    ;;(setf -v v -rterm rterm -c cur-vobj) (break "epr")
    ;;(format t "refresh: rterm ~s cur-obj ~s " (if rterm t nil)(if cur-vobj t nil))
  
    (let ((vobj (if rterm
		    (progn (setf (vobject-term cur-vobj) rterm)
			   cur-vobj)
		    (or cur-vobj
			(instantiate-proof-object oid
						  (when cur-vobj (oid-list-of-proof-object cur-vobj)))))))

      (set-view-object v vobj)
      (view-dtree-init v
		       (term-of-vobject vobj)
		       (implicit-of-vobject vobj) ))))

(defun set-proof-view-refresh-term (v term)
  (let ((vobj (object-of-view v)))
    (when (or (null vobj)
	      (not (proof-object-p vobj)))
      (break "spvrt"))
    (setf (proof-object-refresh-term vobj) term)
    (view-flag-set-touched v t)))

(defunml (|proof_view_add_oid| (v oid))
    (view -> (object_id -> unit))
  (let ((vobj (object-of-view v)))
    (push v *views*))
  nil)

(defunml (|prf_tac_addr| (unit) :declare ((declare (ignore unit))))
    (unit -> (int list))
  (list 1 8 1))  ;; could put this in oedm-edt, put here to ease debugging.

(defvar *multiple-windows-p* nil)
(defun toggle-window-mode ()
  (setf *multiple-windows-p* (not *multiple-windows-p*)))

(defun label-wrap (term)
  (ilabel-term 'point term))

(defun label-unwrap (term)
  (if (real-ilabel-term-p term)
      (subterm-of-term term '(0))
      term))

(defun find-node-at-address (proof address &optional (error-p t))
  (let ((top-address (address-of-proof-node-term (label-unwrap proof))))
    (labels ((visit (term addr)	  
	       (if (null addr)
		   term
		   (let* ((index (1- (car addr)))
			  (node (label-unwrap term))
			  (subs (map-isexpr-to-list (subgoals-of-iproof-node-term node)
						    (iproof-node-cons-op))))
		     (if (>= index (length subs))
			 (if error-p
			     (raise-error (error-message '(find node not)))
			     term)
			 (visit (nth index subs) (cdr addr)))))))

      (visit proof (nthcdr (length top-address) address)))))


(defun address-of-proof-node-term (term)
  (reverse (map-isexpr-to-list (address-of-iproof-node-term term)
			       (ipui-addr-cons-op)
			       #'natural-of-ipui-addr-term)))

(defun ipui-addr-term-to-list (term)
  (reverse (map-isexpr-to-list term (ipui-addr-cons-op)
			       #'natural-of-ipui-addr-term)))

;;repeat hyps for display
(defun rhyps-of-proof-node-term (term)
  (labels ((visit (x l)	   
	     (if (ipui-sequent-term-p x)
		 (let ((seq (sequent-of-ipui-sequent-term x)))
		   (if (iinf-sequent-term-p x)
		       (visit (sequent-of-iinf-sequent-term seq)
			      (if (repeat-p-of-ipui-sequent-term x)
				  (cons (numeral-of-ipui-sequent-term x) l)
				  l)))
		   (cons (numeral-of-ipui-sequent-term x) l))
		 l)))
    
    (map-list-to-isexpr
     (visit (sequent-of-iinf-goal-term (goal-of-iproof-node-term term)) nil)
     (inil-term)
     #'inatural-term)))
     	     	
;;number of hyps or proof-node-term
(defun nhyps-of-proof-node-term (term)
  (labels ((visit (x i)	   
	     (if (ipui-sequent-term-p x)
		 (let ((seq (sequent-of-ipui-sequent-term x)))
		   ;; rle: when concl sometimes no iinf?
		   (if (iinf-sequent-term-p seq)
		       (visit (sequent-of-iinf-sequent-term seq) (1+ i))
		       i))
		 i)))
		      
    (visit (sequent-of-iinf-goal-term (goal-of-iproof-node-term term)) 0)))
     
     	     	
;;for pasting nodes on top of nodes
(defun prf-yank-adjust-address (term top-address)
  (labels ((visit (node)	       
	     (modify-proof-node node
				:address (map-list-to-ilist
					  (reverse (append top-address
							   (address-of-proof-node-term node)
							   ))
					  (ipui-addr-nil-term)
					  #'ipui-addr-term)
				:status (iproof-status-term (intern-system "unknown"))
				:subgoals
				(map-list-to-isexpr
				 (map-isexpr-to-list (subgoals-of-iproof-node-term node)
						     (iproof-node-cons-op) #'visit)
				 (iproof-node-nil-term))))) 
    (visit term)))

;; for scratch proofs, where the top address is not necessarily null
;; so if the top address is really 2 1 1 then a node with address
;; 2 1 1 1 would have the pseudo address of 1
(defun pseudo-address-of-iproof-node-term (term top-address)
  (map-list-to-ilist (reverse (nthcdr (length top-address) (address-of-proof-node-term term)))
		     (ipui-addr-nil-term)
		     #'ipui-addr-term))
     
(defun unpseudo-address-of-iproof-node-term (term address)
  ;;(setf tt term aa address )(break)
  (map-list-to-ilist (reverse (append address (address-of-proof-node-term term)))
		     (ipui-addr-nil-term)
		     #'ipui-addr-term))
      
(defun adjust-address-of-iproof-node-term (term top-address &optional status-p)
  (labels ((visit (node)	       
	     (modify-proof-node node
				:address (unpseudo-address-of-iproof-node-term node top-address)
				:status (if status-p
					    (iproof-status-term (intern-system "incomplete"))
					    (status-of-iproof-node-term node))
				:subgoals
				(map-list-to-isexpr
				 (map-isexpr-to-list (subgoals-of-iproof-node-term node)
						     (iproof-node-cons-op) #'visit)
				 (iproof-node-nil-term)))))
  
    (visit term)))

(defun modify-address-of-iproof-node-term (term top-address)
  (labels ((visit (node)	       
	     (modify-proof-node node
				:address (pseudo-address-of-iproof-node-term node top-address)
				:subgoals
				(map-list-to-isexpr
				 (map-isexpr-to-list (subgoals-of-iproof-node-term node)
						     (iproof-node-cons-op) #'visit)
				 (iproof-node-nil-term)))))
  
    (if (null top-address) term (visit term))))

;;strips off hypothesis numbering used in pui
(defun unnumber-node (term top-address)
  (labels ((visit (seq)	     
	     (if (inum-sequent-term-p seq);;syntax for mp sequents
		 (let ((sequent (sequent-of-inum-sequent-term seq)))
		   (iinf-sequent-term (hidden-of-iinf-sequent-term sequent)
				      (type-of-iinf-sequent-term sequent)
				      (binding-of-sequent-of-iinf-sequent-term sequent)
				      (visit (sequent-of-iinf-sequent-term sequent))))

		 (if (ipui-sequent-term-p seq);;syntax for nv5
		     (let ((sequent (sequent-of-ipui-sequent-term seq))
			   (concl-p (zerop (numeral-of-ipui-sequent-term seq))))
		       (if concl-p
			   sequent
			   (iinf-sequent-term (hidden-of-iinf-sequent-term sequent)
					      (type-of-iinf-sequent-term sequent)
					      (binding-of-sequent-of-iinf-sequent-term sequent)
					      (visit (sequent-of-iinf-sequent-term sequent)))))
		     seq))))
    
    (let ((goal (goal-of-iproof-node-term term)))	         
      (modify-proof-node term
			 :address (if top-address (pseudo-address-of-iproof-node-term term top-address))
			 :goal (iinf-goal-term (visit (sequent-of-iinf-goal-term goal))
					       (annotations-of-iinf-goal-term goal))
			 :subgoals (map-list-to-isexpr
				    (map-isexpr-to-list (subgoals-of-iproof-node-term term)
							(iproof-node-cons-op)
							#'(lambda (x) (unnumber-node x top-address)))
				    (iproof-node-nil-term))))))
		      
(defun first-unrefined-node-of-proof (term)
  (labels ((visit (s)
	     (let ((subs (filter #'(lambda (x)
				     (not (eql '|complete| (state-of-proof-node-term x))))
				 
				 (map-isexpr-to-list (subgoals-of-iproof-node-term s)
						     (iproof-node-cons-op)))))
	       (if (null subs)
		   (return-from first-unrefined-node-of-proof s)
		   (mapcar #'visit subs)))))

    (if (eql '|complete| (state-of-proof-node-term term))
	term
	(visit term))))
	

(defun next-unrefined-node-of-proof (address node)
  ;;(setf aa address tt node) (break)
  (labels ((visit (a)
	     (let* ((number (car (last a)))
		    (parentaddr (butlast a))
		    (parent (find-node-at-address node parentaddr))
		    (subs (filter #'(lambda (x)
				      (not (eql '|complete| (state-of-proof-node-term x))))
				  (nthcdr number
					  (map-isexpr-to-list (subgoals-of-iproof-node-term parent)
							      (iproof-node-cons-op))))))
	       (if (null subs)
		   (next-unrefined-node-of-proof parentaddr node)
		   (first-unrefined-node-of-proof (car subs))))))
		       
    (if (null address)
	(first-unrefined-node-of-proof node)
	(if (eql '|complete| (state-of-proof-node-term node))
	    (find-node-at-address node address)
	    (visit address)))))

(defun upmost-unrefined-node-of-proof (address node)
  ;;(setf aa address tt node) (break)
  (if (equal address (address-of-proof-node-term node))
      node 
      (let ((current (find-node-at-address node address)))
	(if (not (eql '|complete| (state-of-proof-node-term current)))
	    current
	    (upmost-unrefined-node-of-proof (butlast address) node)))))
					  	            
;; do not check initial address
(defun next-node-of-proof (p address topproof)
  (labels ((children (a tree)
	     (declare (ignore a))
	     (map-isexpr-to-list (subgoals-of-iproof-node-term tree)
				 (iproof-node-cons-op)))
	   (parent (a tree)
	     (declare (ignore tree))
	     (find-node-at-address topproof (butlast a)))
	   
	   (down (a tree)
	     (let ((children (children a tree)))
	       (if (null children)
		   (visit a tree)
		   (down (nconc a (list 1)) (car children)))))
	   
	   (next (a tree)
	     (if (null a)
		 (down a tree)
		 (let* ((number (car (last a)))
			(parentaddr (butlast a))
			(parent (parent a tree))
			(sib (nth number (children a parent))))
		   (if sib
		       (down (nconc parentaddr (list (1+ number))) sib)
		       (visit parentaddr parent)))))
	   (visit (a tree)
	     (if (or (funcall p tree) (equal a address))
		 (cons a tree)
		 (next a tree))))

    (next address (find-node-at-address topproof address))))

;; note that address are by nature reversed.
(defun filter-trees-of-proof (p topproof)
  (let ((acc nil))
    (labels ((children (a tree)
	       (declare (ignore a))
	       (map-isexpr-to-list (subgoals-of-iproof-node-term tree)
				   (iproof-node-cons-op)))

	     (visit (a tree)
	       (when (funcall p tree)
		 ;;(setf -a a -tree tree) (break "hello")
		 (push (cons a tree) acc))
	       (dotimeslist (i tree (children a tree))
			    (visit (cons (1+ i) a) tree))))

      (visit nil topproof))

    ;;(setf -acc acc) (break "ftop")
    acc))


	

;; status only changes if calling after a refine so we use flag to determine which fn to call
(defun replace-proof-node (top node &optional refined-p)
  ;;accounts for allowance of top to to have non nil address
  (let ((addr (nthcdr (length (address-of-proof-node-term (label-unwrap top)))
		      (address-of-proof-node-term (label-unwrap node)))))
    (if refined-p
	(replace-refined-proof-node top node addr)
      (replace-edd-proof-node top node addr))))
  
(defun replace-refined-proof-node (top node addr)
  (labels ((visit (target address)	       
		  (unless target (raise-error (error-message '(replace node not))))
		  (if (null address)
		      node		      
		    (let* ((subgoals (map-isexpr-to-list (subgoals-of-iproof-node-term target)
							 (iproof-node-cons-op)))
			   (next (nth (1- (car address)) subgoals))
			   (new-node (visit next (cdr address)))
			   (state (state-of-proof-node-term target))
			   (incomplete-subgoals-p (filter #'(lambda (x)
							      (let ((s (if (compare-terms-p x next)
									   (state-of-proof-node-term new-node)
									 (state-of-proof-node-term x))))
								(eql '|incomplete| s)))
							  subgoals))
			   (new-state (if (and incomplete-subgoals-p (eql '|complete| state))
					  '|incomplete|
					(if (and (not incomplete-subgoals-p) (eql '|incomplete| state))
					    '|complete|
					  state))))
						   		     		  
		      (modify-proof-node target
					 :status (iproof-status-term new-state)
					 :subgoals (map-list-to-ilist
						    (replace-in-list new-node (car address) subgoals)
						    (iproof-node-nil-term)))))))
	           
	  (visit top addr)))
	     

;;does not adjust status of nodes					
(defun replace-edd-proof-node (top node addr)
  (labels ((visit (target address)
		  (unless target (raise-error (error-message '(replace node q not))))
		  (if (null address)		 
		      node
		    (let* ((subgoals (map-isexpr-to-list (subgoals-of-iproof-node-term target)
							 (iproof-node-cons-op)))
			   (new-node (visit (nth (1- (car address)) subgoals) (cdr address))))
		     
		      (modify-proof-node target
					 :subgoals
					 (map-list-to-ilist (replace-in-list new-node (car address) subgoals)
							    (iproof-node-nil-term)))))))
	     
	  (visit top addr)))
					
(defun set-proof-node-status (node status)
  (modify-proof-node node :status status))

(defun set-proof-node-status-to-fail (term)
  (set-proof-node-status term (iproof-status-term '|failed|)))
  
(defun set-proof-node-status-to-complete (term)
  (set-proof-node-status term (iproof-status-term '|complete|)))
  
(defun remove-message-annotations (term)
  (modify-proof-node term :annotations (iannotation-nil-term)))

(defun remove-hidden-subgoal-annotations (term)
  (let ((annos (map-list-to-isexpr
		(filter #'(lambda (x)
			    (not (ihidden-subgoals-term-p x)))
			(map-isexpr-to-list (annotations-of-iproof-node-term term)
					    (iannotation-cons-op)))
		(iannotation-nil-term))))
    (modify-proof-node term :annotations annos)))

(defvar *do-profiling* nil)

(defun iproof-node-term-to-ttt (term)
  (let ((subgoals (map-isexpr-to-list (subgoals-of-iproof-node-term term)
				      (iproof-node-cons-op))))
    (ittree-term (icar (refinement-of-iproof-node-term term))
		 (map-list-to-isexpr subgoals (ittree-nil-term)
				     #'iproof-node-term-to-ttt))))

(defun iproof-node-term-to-tgt (term)
  ;;(break "tgt")
  (let ((subgoals (map-isexpr-to-list (subgoals-of-iproof-node-term term)
				      (iproof-node-cons-op)))
	(refinement (refinement-of-iproof-node-term term)))
    (itgtree-term (icar refinement)
		  (icdr refinement)
		  (subgoals-of-iproof-node-term term)
		  (map-list-to-isexpr subgoals (itgtree-nil-term)
				      #'iproof-node-term-to-tgt))))

(defun tactic-tree-term-to-ttt (tterm)
  ;;(setf -term tterm) (break "ttt")
  (let ((term (label-unwrap tterm)))
    (unless (tactic-tree-term-p term)
      (raise-error (error-message '(tactic tree term not))))
    (let ((subgoals (map-ilist-to-list (children-of-tactic-tree-term term)
				       (tactic-tree-op))))
      (ittree-term (icar term)
		   (map-list-to-isexpr subgoals (ittree-nil-term)
				       #'tactic-tree-term-to-ttt)))))

(define-primitive |!thenl_cons| () (car cdr))
(define-primitive |!paren_wrap| () (tactic))

(defun tactics-to-text-list (tactics)
  ;;(break "ttl")
  (map-list-to-ilist tactics (ithenl-nil-term)))


(defun text-list-to-tactics (text)
  ;;(break "tltt")
  (map-isexpr-to-list text (ithenl-cons-op)))


(defun kreitzed-tactic-of-iproof-node-term (term &optional (in-list-p t))

  (let* ((tactic (icar (refinement-of-iproof-node-term term)))
	 (subgoals (map-isexpr-to-list (subgoals-of-iproof-node-term term)
				       (iproof-node-cons-op)))
	 (refined-p
	  (filter #'(lambda (x)
		      (let ((subs (map-isexpr-to-list (subgoals-of-iproof-node-term x)
						      (iproof-node-cons-op))))
			(or subs (proof-node-complete-p x))))
		  subgoals)))

    (if refined-p
	(if (= 1 (length subgoals))
	    (ped-then-term tactic
			   (kreitzed-tactic-of-iproof-node-term (car subgoals) nil))
	    (ped-thenl-term tactic
			    (tactics-to-text-list (mapcar #'kreitzed-tactic-of-iproof-node-term subgoals))))
		    
	(if in-list-p
	    (if (or (proof-node-complete-p term)
		    (map-isexpr-to-list (subgoals-of-iproof-node-term term)
					(iproof-node-cons-op)))
		(iparen-wrap-term tactic)
		(itext-term "Id"))
	    tactic))))

(defun dekreitzed-tactic-of-iproof-node-term (term)
  (labels ((visit (term)
	     (let ((tactic (if (iparen-wrap-term-p term)
			       (tactic-of-iparen-wrap-term term)
			       term)))
	       (if (ped-then-term-p tactic)
		   (ittree-term (left-of-ped-then-term tactic)		
				(visit (right-of-ped-then-term tactic)))
		   (if (ped-thenl-term-p tactic)
		       (ittree-term
			(tactic-of-ped-thenl-term tactic)
			(map-list-to-isexpr (text-list-to-tactics (list-of-ped-thenl-term tactic))
					    (ittree-nil-term)
					    #'(lambda (x) (visit x))))
		      
		       (ittree-term tactic (ittree-nil-term)))))))
	  
    (visit (icar (refinement-of-iproof-node-term term)))))
		       

(defun edit-proof-refine-aux (v term refinef tactic &optional tree-p contf)
  ;;(setf -v v -term term) (break "epra")
  (let* ((obj (object-of-view v))
	 (top (top-proof-of-proof-object obj))
	 (top-address (address-of-proof-node-term top))
	 (node (unnumber-node term top-address))
	 (cterm (iproof-lib-term (view-of-iproof-node-term term)
				 (depth-of-iproof-node-term term)
				 (address-of-iproof-node-term node)
				 (goal-of-iproof-node-term node)
				 tactic
				 (rhyps-of-proof-node-term term))))

    ;;(setf tt term rr node ta tactic) (break)       
    (with-handle-error-and-message
	(nil
	 #'(lambda (m)
	     (let* ((fterm (modify-proof-node term
					      :status (iproof-status-term '|failed|)
					      :annotations (iannotation-cons-term
							    (message-to-term m)
							    ;;(annotations-of-iproof-node-term term)
							    (iannotation-nil-term))))
		 
		    (newtop (replace-proof-node top fterm t)))
	       (setf (proof-object-eddtop-proof obj) newtop)		     
	       fterm)))
       
      ;;(setf -term term -cterm cterm -obj obj) (break "edrf")
      (let* ((result (if *do-profiling*
			 (progn (format t "~%profiling ~%")
				#-cmu (prof:start-profiler)
				(prog1 
				    (if contf 
					(funcall contf term (funcall refinef cterm (oid-of-vobject obj)
								     (oid-list-of-proof-object obj)))
					(funcall refinef cterm (oid-of-vobject obj) (oid-list-of-proof-object obj)))
				  #-cmu (prof:stop-profiler)))
			 
			 (if contf 
			     (funcall contf term (funcall refinef cterm (oid-of-vobject obj)
							  (oid-list-of-proof-object obj)))  
			     (funcall refinef cterm (oid-of-vobject obj) (oid-list-of-proof-object obj)))))
	     
	     (newnode (modify-proof-node (if top-address
					     (adjust-address-of-iproof-node-term result top-address)
					     result);;unless tree-p
					 :goal (unless nil (goal-of-iproof-node-term term))))
	     (newtop (replace-proof-node top newnode t)))
	
	(setf (proof-object-eddtop-proof obj) newtop
	      (proof-object-top-proof obj) newtop)

	(proof-update-history obj newtop)
	newnode))))


(defun merge-matching-sbgoals (new-proof old-proof)
  (labels ((visit (x)
	     (if (proof-node-complete-p x) 
		 x
		 (let ((subgoals (map-isexpr-to-list (subgoals-of-iproof-node-term x)
						     (iproof-node-cons-op))))
		   (if subgoals
		       (modify-proof-node x
					  :subgoals (map-list-to-isexpr subgoals
									(iproof-node-nil-term)
									#'visit))
		       (let* ((old-parent-node (find-node-at-address
						old-proof
						(butlast (address-of-proof-node x))))
			      (old-subgoals (map-isexpr-to-list (subgoals-of-iproof-node-term old-parent-node)
								(iproof-node-cons-op)))
			      (matches (filter
					#'(lambda (n)
					    (alpha-equal-terms-p (goal-of-iproof-node-term n)
								 (goal-of-iproof-node-term x)))
					old-subgoals)))
			 (if matches (car matches) x)))))))                               
  
    (visit new-proof old-proof)))

(defun edit-proof-kreitz-refine (v term)
  (edit-proof-refine-aux v term #'edit-step-refine-term
			 (kreitzed-tactic-of-iproof-node-term term nil) t))

(defun edit-proof-dekreitz-refine (v term)
  (edit-proof-refine-aux v term #'edit-refine-tttt-tree-term
			 (dekreitzed-tactic-of-iproof-node-term term) t))    

(defun edit-proof-step-refine (v term)
  (edit-proof-refine-aux v term #'edit-step-refine-term
			 (icar (refinement-of-iproof-node-term term))))

;; if produces same subs, simply puts them in completed
(defun edit-proof-refine (v term)
  (edit-proof-refine-aux v term #'edit-refine-term
			 (icar (refinement-of-iproof-node-term term))))
	     
(defun edit-proof-tree-refine (v term)
  (edit-proof-refine-aux v term #'edit-refine-tree-term (iproof-node-term-to-tgt term) t))

(defun edit-proof-tttt-refine (v term)
  (edit-proof-refine-aux v term #'edit-refine-tttt-tree-term
			 (tactic-tree-term-to-ttt (term-stack-peek-r)) t))

(defun edit-proof-edit-refine (v term)
  (edit-proof-refine-aux v term #'edit-step-refine-term
			 (icar (refinement-of-iproof-node-term term))
			 nil #'merge-matching-subgoals))

(defun edit-proof-edit-tree-refine (v term)
  (edit-proof-refine-aux v term #'edit-refine-edit-tree-term
			 (iproof-node-term-to-tgt term)
			 t #'merge-matching-subgoals))

(defun edit-proof-move (v)
  (let* ((term (term-stack-peek-r))
	 (address (oed-filter-point-mark term)))
    (unless (or (ipui-addr-cons-term-p address) (ipui-addr-term-p address))
      (raise-error (error-message '(not an address) term)))
    (find-node-at-address (eddtop-proof-of-proof-object (object-of-view v))
			  (ipui-addr-term-to-list address))))


;;;; history

(defun proof-history-init (o)
  ;;(format t "h init ")
  (setf (proof-object-history o) (new-proof-history)))

(defun proof-history-reset (o)
  (setf (proof-object-history o) nil))

(defun proof-begin-history-walk (o)

  (let* ((h (or (history-of-proof-object o) (proof-history-init o)))
         (w (history-walk-begin h)))

    (setf (proof-object-history-walk o) w)
    w))

(defun proof-end-history-walk (o)

  (let ((w (history-walk-of-proof-object o)))
    (when w
      (unless (null w)
	(setf (proof-object-history-walk o) nil)))))

(defun proof-update-history (o item)

  (let ((h (history-of-proof-object o)))
    (unless h
      (format t "no history") (proof-history-init o))
    (history-update (or h (history-of-proof-object o)) item)
    (proof-end-history-walk o)))

  
;; #'s represent (?) size of history to maintain
(defun new-proof-history ()
  (new-history
   (cons (list* 64 7 #'(lambda (x) x))
	 '((32 15) (16 15) (16)))))

(defun next-proof (v)
  (let* ((o (object-of-view v))
	 (w (or (history-walk-of-proof-object o)
		(proof-begin-history-walk o))))
    (history-walk-backward w)
    (let ((p (history-walk-peek w)))
      (unless p (raise-error (error-message '(history next none))))
      (setf (proof-object-eddtop-proof o) p)
      p)))

(defun previous-proof (v)
  (let* ((o (object-of-view v))
	 (w (or (history-walk-of-proof-object o)
		(proof-begin-history-walk o))))
    ;;(format t " wf: ~s" (history-walk-forward w))
    
    (history-walk-forward w)
    ;;(setf ww w)
    (let ((p (history-walk-peek w)))
      (unless p (raise-error (error-message '(history prev none))))
      (setf (proof-object-eddtop-proof o) p)
      p)))



(defvar *ml-mp-save-term* nil)

(defun edit-mp-save (v)
  ;;(break "ems")
  ;;(edit-save state) ;;really should save and see if goal was edited?

  (let* ((oid (oid-of-view v)))
    (unless (oid-p oid)
      (raise-error (error-message '(edit mp-save-term oid not))))

    (funmlcall (or *ml-mp-save-term* (ml-text "mp_save"))
	       oid))
  v)

;;only for mp fttb
(defun edit-refine-undo (v)
  ;;(break "eru")
  ;;(edit-save state)

  (let ((obj (object-of-view v)))
    (unless (proof-object-p obj)
      (let ((oid (with-ignore (oid-of-view v))))
	(raise-error (oid-error-message (if oid (list oid) nil)
					'(edit refine undo proof not)))))

    (with-appropriate-transaction (t nil) ;; ??? maybe could be (nil t)

      (let ((r (edit-ref-undo (oid-of-view v))))    
	;;(setf rt r) (break)
	(setf (proof-object-top-proof obj) r)
	(setf (proof-object-eddtop-proof obj) r)
	(set-proof-view-refresh-term v r))))	
      v)

(defun edit-refiner-primitives ()
  (mapc  #'(lambda (pair) (define-edit-primitive (car pair) (cdr pair)))
	 '(("inf-top" . "edit-inf-top")
	   ("inf-undo" . "edit-inf-undo")
	   ("inf-up" . "edit-inf-up")
	   ("inf-down" . "edit-inf-down")
	   ("refine" . "edit-refine")
	   ("refundo" . "edit-refine-undo")
	   ("mp-save" . "edit-mp-save"))))
	

(defunml (|edit_proof_refine| (v term)) 
    (view -> (term -> term))
  (edit-proof-refine v (oed-filter-point-mark term)))
  
(defunml (|edit_proof_step_refine| (v term)) 
    (view -> (term -> term))
  (edit-proof-step-refine v (oed-filter-point-mark term)))
  
(defunml (|edit_proof_kreitz_refine| (v term)) 
    (view -> (term -> term))
  (edit-proof-kreitz-refine v (oed-filter-point-mark term)))
  
(defunml (|edit_proof_dekreitz_refine| (v term)) 
    (view -> (term -> term))
  (edit-proof-dekreitz-refine v (oed-filter-point-mark term)))
  
(defunml (|edit_proof_tree_refine| (v term)) 
    (view -> (term -> term))
  (edit-proof-tree-refine v (oed-filter-point-mark term)))

(defunml (|edit_proof_tttt_refine| (v term)) 
    (view -> (term -> term))
  (edit-proof-tttt-refine v (oed-filter-point-mark term)))

;;below 2 are wips
(defunml (|edit_proof_edit_refine| (v term)) 
    (view -> (term -> term))
  (edit-proof-edit-refine v (oed-filter-point-mark term)))
  
(defunml (|edit_proof_edit_tree_refine| (v term)) 
    (view -> (term -> term))
  (edit-proof-edit-tree-refine v (oed-filter-point-mark term)))
;;end wips 


(defunml (|edit_proof_move| (v)) 
    (view -> term)
  (edit-proof-move v))
  
(defun top-proof-of-view (v)
  (let ((obj (object-of-view v)))
    (unless (proof-object-p obj)
      (let ((oid (with-ignore (oid-of-view v))))
	(raise-error (oid-error-message (when oid (list oid)) '(top proof of view not)))))
    (top-proof-of-proof-object obj)))

(defunml (|first_unrefined_prf_of_view| (v)) 
    (view -> term)
  (let ((obj (object-of-view v)))
    (unless (proof-object-p obj)
      (let ((oid (with-ignore (oid-of-view v))))
	(raise-error (oid-error-message (when oid (list oid)) '(top proof of view)))))
    (first-unrefined-node-of-proof (top-proof-of-proof-object obj))))

(defunml (|proof_jump_up_aux| (address node)) 
    ((int list) -> (term -> term))
  (upmost-unrefined-node-of-proof address node))

(defunml (|next_unrefined_prf| (address v)) 
  ((int list) -> (view -> term))

  (let ((obj (object-of-view v)))
    (unless (proof-object-p obj)
      (let ((oid (with-ignore (oid-of-view v))))
	(raise-error (oid-error-message (when oid (list oid)) '(top proof of view)))))
    (next-unrefined-node-of-proof address (top-proof-of-proof-object obj))))

(defunml (|prf_filter| (p v)) 
  ((term -> (term -> ((term list) -> bool))) -> (view -> ((int list) list)))

  (let ((obj (object-of-view v)))
    (unless (proof-object-p obj)
      (let ((oid (with-ignore (oid-of-view v))))
	(raise-error (oid-error-message (when oid (list oid)) '(top proof of view)))))

    (let ((addresses
	   (mapcar #'car
		   (filter-trees-of-proof
		    #'(lambda (tree)
			(let ((goal (sequent-of-iinf-goal-term (goal-of-iproof-node-term tree)))
			      (tac (icar (refinement-of-iproof-node-term tree)))
			      (subgoals (mapcar #'goal-of-iproof-node-term
						(map-isexpr-to-list (subgoals-of-iproof-node-term tree)
								    (iproof-node-cons-op)))))
			  ;;(setf -tree tree -goal goal) (break "psn")
			  (funmlcall p goal tac subgoals)))
		    (top-proof-of-proof-object obj)))))
      ;;(setf -addresses addresses -v v -obj obj) (break "pf")
      addresses)))

      

(defunml (|stm_of_view| (v)) 
    (view -> object_id)  
  (let ((obj (object-of-view v)))
    (unless (proof-object-p obj)
      (let ((oid (with-ignore (oid-of-view v))))
	(raise-error (oid-error-message (when oid (list oid)) '(stm of view)))))
    (car (oid-list-of-proof-object obj))))

(defunml (|top_proof_of_view| (v)) 
    (view -> term)  
  (top-proof-of-view v))

(defunml (|top_edit_proof_of_view| (v)) 
    (view -> term)
  (let ((obj (object-of-view v)))
    (unless (proof-object-p obj)
      (let ((oid (with-ignore (oid-of-view v))))
	(raise-error (oid-error-message (when oid (list oid)) '(top proof of view)))))
    (eddtop-proof-of-proof-object obj)))

(defunml (|top_goal_of_view| (v)) 
    (view -> term)
  
  (let ((obj (object-of-view v)))
    (unless (proof-object-p obj)
      (let ((oid (with-ignore (oid-of-view v))))
	(raise-error (oid-error-message (when oid (list oid)) '(top goal of view)))))

    (goal-of-iproof-node-term (top-proof-of-proof-object obj))))

(defunml (|oid_of_view| (v)) 
    (view -> object_id)
  (or (oid-of-view v) (raise-error (error-message '(view oid not)))))

(defunml (|find_node_at_address| (node address)) 
    (term -> ((int list) -> term))
  (find-node-at-address node address))

(defunml (|replace_proof_node| (target node)) 
    (term -> (term -> term))
  (replace-proof-node target node))

(defunml (|replace_proof_node_top| (v node)) 
    (view -> (term -> term))
  (let ((proof (top-proof-of-view v)))
    (replace-proof-node proof node)))

(defunml (|oed_filter| (term)) 
    (term -> term)
  (oed-filter-temp-point-mark term))

(defunml (|proof_editor_wrap| (term)) 
    (term -> term)
  (proof-editor-wrap-with-depth term))

(defunml (|proof_editor_unwrap| (term v)) 
    (term -> (view -> term))
  (let ((obj (object-of-view v)))
    (unless (proof-object-p obj)
      (let ((oid (with-ignore (oid-of-view v))))
	(raise-error (oid-error-message (when oid (list oid)) '(eddtop unwrap)))))
      
    (proof-editor-unwrap-with-depth term (eddtop-proof-of-proof-object obj))))

(defunml (|set_top_proof| (term v)) 
    (term -> (view -> unit))
  (let ((obj (object-of-view v)))
    (setf (proof-object-top-proof obj) term)
    nil))

(defun set-node-status-unknown (node)
  (modify-proof-node node
		     :status (iproof-status-term (intern-system "unknown"))
		     :subgoals (map-list-to-isexpr
				(map-isexpr-to-list (subgoals-of-iproof-node-term node)
						    (iproof-node-cons-op)
						    #'set-node-status-unknown)
				(iproof-node-nil-term))))

(defunml (|set_top_proof_goal| (goal v))
    (term -> (view -> term))
  (let* ((obj (object-of-view v))
	 (proof (modify-proof-node (top-proof-of-proof-object obj) :goal goal
				   :status (iproof-status-term (intern-system "unknown")))))
    (setf (proof-object-top-proof obj) proof)
    (setf (proof-object-eddtop-proof obj) proof)))
	  
(defunml (|edd_make_new_proof| (term poid soid name)) 
    (term -> (object_id -> (object_id -> (tok -> view))))

  (let* ((oid (funmlcall (ml-text "lib_backup_proof ") term poid soid name))
         (vobj (instantiate-proof-object oid (list soid poid) term))
         (v (new-view nil)))
  
    (view-assign-oid v oid)
    (set-view-object v vobj)
    (set-view-title v (new-view-titles v))
    (view-refresh-title v (cons (string name) (string name)))     
    (view-dtree-init v (term-of-vobject vobj) (implicit-of-vobject vobj))
    (view-open-window v)
    v))

(defunml (|edd_interior_proof| (term poid v name))
    (term -> (object_id -> (view -> (tok -> unit))))

  (let* ((term2 (funmlcall (ml-text "lib_interior_proof ") term poid name))
	 (oid (oid-of-ioid-term (subterm-of-term term2 (list 0))))
	 (node (subterm-of-term term2 (list 1)))
	 (vobj (instantiate-proof-object oid nil node)))
     
    (set-view-object v vobj)
    (set-view-title v (new-view-titles v))
    (view-refresh-title v (cons (string name) (string name)))     
    (view-dtree-init v
		     (term-of-vobject vobj)
		     (implicit-of-vobject vobj))))

(defunml (|edd_primitive_proof| (poid term v name)) 
    (oid -> (term -> (view -> (tok -> unit))))

  (let* ((node (funmlcall (ml-text "lib_primitive_prooft ")
			  poid (unnumber-node term nil)))
	 (vobj (instantiate-proof-object *null-oid* nil node)))
     
    (set-view-object v vobj)
    (set-view-title v (new-view-titles v))
    (view-refresh-title v (cons (string name) (string name)))     
    (view-dtree-init v
		     (term-of-vobject vobj)
		     (implicit-of-vobject vobj))))

(defunml (|eddtop_save_obj| (term v)) 
    (term -> (view -> unit))  
  (let ((obj (object-of-view v)))
    (unless (proof-object-p obj)
      (let ((oid (with-ignore (oid-of-view v))))
	(raise-error (oid-error-message (when oid (list oid)) '(eddtop save)))))
    
    (let ((proof (eddtop-proof-of-proof-object obj)))
      (setf (proof-object-eddtop-proof obj) 
	    (replace-proof-node proof (label-unwrap term)))))      
  nil)

(defunml (|next_proof| (v)) 
    (view -> term)
  (next-proof v))

(defunml (|prev_proof| (v)) 
    (view -> term)
  (previous-proof v))

;; adds point label to tactic of proof node term, returns node
(defunml (|edd_add_point_label| (node))
    (term -> term)
  (let* ((r (refinement-of-iproof-node-term node)))
    (modify-proof-node node
		       :refinement
		       (icons-left-term (ilabel-term 'point (left-of-icons-left-term r))
					(right-of-icons-left-term r)))))

(defunml (|edd_add_point| (term))
    (term -> term)
  (ilabel-term 'point term))

(defunml (|unnumber_proof_node| (node v))
    (term -> (view -> term))

  (let ((obj (object-of-view v)))
    (unless (proof-object-p obj)
      (let ((oid (with-ignore (oid-of-view v))))
	(raise-error (oid-error-message (when oid (list oid)) '(proof not)))))

    (unnumber-node (oed-filter-point-mark node)
		   (address-of-proof-node-term (top-proof-of-proof-object obj)))))

(defunml (|first_subgoal_of_proof| (node))
    (term -> term)
  (or (car (map-isexpr-to-list (subgoals-of-iproof-node-term node)
			       (iproof-node-cons-op)))
      (raise-error (error-message '(first subgoal not)))))

(defun build-proof-variant (view term)
  (let ((subs (map-isexpr-to-list (subgoals-of-iproof-node-term term)
				  (iproof-node-cons-op))))
    (modify-proof-node term
		       :view view
		       :subgoals
		       (map-list-to-isexpr subs (iproof-node-nil-term)
					   #'(lambda (x) (build-proof-variant view x))))))      
		       
(defunml (|get_proof_variant| (mode v))
  (tok -> (view -> term))
  ;;(break)
  (let ((obj (object-of-view v)))
    (unless (proof-object-p obj)
      (let ((oid (with-ignore (oid-of-view v))))
	(raise-error (oid-error-message (when oid (list oid)) '(proof not)))))

    (let ((variants (variants-of-proof-object obj)))
      (if (cdr (assoc mode variants))
	  (setf (proof-object-eddtop-proof obj)
		(build-proof-variant mode (eddtop-proof-of-proof-object obj)))
	;;(let ((p (build-proof-variant mode (eddtop-proof-of-proof-object obj))))
	;;(setf (proof-object-variants obj) (acons mode p variants))
	;;p)
	(eddtop-proof-of-proof-object obj)))))

(defun replace-proof-refinement (addr term &optional anno top-address)
  (labels ((visit (node address)
	     (if (eq address addr)		 
		 (let ((r (refinement-of-iproof-node-term node)))
		   (ilabel-term 'point
				(modify-proof-node
				 node
				 :refinement
				 (icons-left-term (ipending-refinement-term
						   (left-of-icons-left-term r))
						  (right-of-icons-left-term r))
				 :annotations
				 (when anno (iannotation-cons-term
					     anno
					     (annotations-of-iproof-node-term node))))))
  
		 
		 (let* ((children (map-isexpr-to-list (subgoals-of-iproof-node-term node)
						      (iproof-node-cons-op)))
			(p (nth (1- (car address)) children))
			(v (visit p (cdr address))))
		     
		   (modify-proof-node node
				      :subgoals
				      (map-list-to-ilist (replace-in-list v (car address) children)
							 (iproof-node-nil-term))
				      )))))
	     
    (visit term (if top-address
		    (nthcdr (length top-address) (address-of-proof-node-term term))
		    addr))))

(defunml (|show_refine_pending| (node))
    (term -> term)
  
  (let ((refinement (refinement-of-iproof-node-term node)))
    (modify-proof-node node
		       :refinement
		       (icons-left-term (label-wrap (ipending-refinement-term
						     (left-of-icons-left-term refinement)))
					(right-of-icons-left-term refinement)))))

(defun asynch-edd-unpend (top address)
  (let* ((node (find-node-at-address top address))
	 (refinement (refinement-of-iproof-node-term node))
	 (newnode (modify-proof-node node
				     :refinement (icons-left-term
						  (label-wrap (tactic-of-ipending-refinement-term
						   (left-of-icons-left-term refinement)))
						  (right-of-icons-left-term refinement)))))
    (replace-proof-node top newnode)))

(defunml (|edd_update_prf| (view))
    (view -> term)
  (let ((obj (object-of-view view))
	(term (get-inf-term (oid-of-view view))))
    (setf (proof-object-top-proof obj) term
	  (proof-object-eddtop-proof obj) term)
    term))

(defun label-node-at-address (top address)
  (replace-proof-node top (label-wrap (find-node-at-address top address))))

(defunml (|label_node_at_address| (top address))
    (term -> int list)
  (label-node-at-address top address))

(defun edd-update-prf-at-address (view address &optional failure-p)
  ;;(setf -v view -a address) (break "eu")
  (let* ((obj (object-of-view view))
	 (top (eddtop-proof-of-proof-object obj))
	 (view-top (address-of-proof-node-term (proof-editor-unwrap (term-of-view view))))
	 (oid (oid-of-view view))
	 (node (if failure-p
		   (set-proof-node-status-to-fail
		    (find-node-at-address top (ipui-addr-term-to-list address)))
		   (get-inf-term-at-address oid address)))
	 (refinement (refinement-of-iproof-node-term node))
	 (newnode (modify-proof-node
		   node
		   :refinement (icons-left-term
				(label-wrap (label-unwrap
					     (left-of-icons-left-term refinement)))
				(right-of-icons-left-term refinement))))		     	 
	 (newtop (replace-proof-node
		  top
		  (adjust-address-of-iproof-node-term newnode (address-of-proof-node-term top))
		  t)))

    ;; (setf -top top -newtop newtop -newnode newnode -node node -view-top view-top) (break "update")
    (unless failure-p (setf (proof-object-top-proof obj) newtop))
    (setf (proof-object-eddtop-proof obj) newtop)
    (find-node-at-address newtop view-top)))

(defunml (|edd_update_prf_at_address| (view address))
  (view -> (term -> term))
  (edd-update-prf-at-address view address))

(defunml (|edd_update_failure| (view address))
  (view -> (term -> term))
  (edd-update-prf-at-address view address t))
