
%
Print format for proof node is schematically as follows:

[...branches....]?\
[...branches....]? 1. [id. (if not NIL)][...Hypothesis term.1........]
[...branches....]?    [............Hypothesis term.1.................]
       .                   .                             .
       .                   .                             .
[...branches....]?    [............Hypothesis term.1.................]
       .                   .                             .
       .                   .                             .
       .                   .                             .
[...branches....]? n. [id. (if not NIL)][...Hypothesis term.n........]
[...branches....]?    [............Hypothesis term.n.................]
       .                   .                             .
       .                   .                             .
[...branches....]?    [............Hypothesis term.n.................]
[...branches....]? >> [............Conclusion term...................]
[...branches....]?    [............Conclusion term...................]
       .                   .                             .
       .                   .                             .
[...branches....]?    [............Conclusion term...................]
[...branches....]? BY [............Refinement rule...................]
[...branches....]? !  [............Refinement rule...................]
       .                   .                             .
       .                   .                             .
[...branches....]? !  [............Refinement rule...................]

NOTE
1. Branch at ?'s printed only if node is not last sibling.
2. branch at !'s printed only if node has children.
3. term and rule text is folded so as to always stay in the appropriate
   region. 
4. A hypothesis is only printed if it is textually different from the
   same numbered hypothesis in the parent node.
5. If a hypothesis is hidden, []'s are put around the hypothesis number.
6. The root node does not have a branch entering the node.

The following functions are useful:

a. Invoked by typing into the transformation tactic window:

   Print                           ..pretty print proof to snapshot file.
   PrintToFile <file_name (a token)>  ..set snapshot file to <filename> then ppp.

b. Options:

   set_pp_width i       ..set number of columns to print proof with.
   set_pp_all_hyps b    ..b = true: print all hyps at each node of tree
                        ..b = false: print only hyps different from parent.
%


letref rule_width = 80;;

let turnstile_char = int_to_char 140;;

letref pp_width = 95 ;;
let set_pp_width i = pp_width := i ; () ;;

letref pp_same_hyps = false ;;
let set_pp_same_hyps b = pp_same_hyps := b ; () ;;

letref pp_same_concl = false ;;
letref pp_always_indent = false ;;
letref pp_addrs = false ;;

%
pp_divide_into_lines : string -> int -> string list
%

let pp_divide_into_lines s n =
  let toks = strip_last_newline (explode_string s) in
  let split_at_ret l = 
    let a,b = split (find_newline_position l - 1) l in a, (tl b) 
  in
  let natural_tok_lines =
    unreduce1 split_at_ret toks 
  in
  let clipped_tok_lines = 
  ( if n > 0 then
      flatten
        (map
          (unreduce1 (\l.if length l > n then split n l else fail))
          natural_tok_lines
        )
    else
      natural_tok_lines
  )
  in
    map toks_to_string clipped_tok_lines
;;



% nb pp_format_branches expects branches in right to left order 
  Bunch up branches if pp_always_indent is false
%

let pp_format_branches branch_list = 
  concatenate_strings 
    (map (\x. if x then "| " else "  ") (rev branch_list))
;;

% format numbered branches: helps improve readability of long scripts %

letref pp_number_branches = true;;

let pp_format_nbranches branch_list = 
  if not pp_number_branches then 
    pp_format_branches branch_list 
  else
  letrec aux n bs = 
    if null bs then []
    if not (hd bs) then "  " . aux n (tl bs)
    if n = 9 then "9 " . aux 0 (tl bs)
    else (int_to_string n J " ") . aux (n+1) (tl bs)
  in
   concatenate_strings (aux 1 (rev branch_list))
;;


let pp_format_preceding_lines branches inline=
  if inline then
    [pp_format_branches branches J "|"]
  if null branches then 
    []
  else
      [append_strings
         (pp_format_branches (tl branches))
         (if hd branches then "|\\" else " \\")
      ]
;;

% 
expects rest_line_header to be no longer than first_line_header.
%

let pp_format_block
      first_line_header
      rest_line_header
      body
      =
  let first_header_length = length (explode_string first_line_header) in
  let rest_header_length = length (explode_string rest_line_header) in
  let padded_rest_line_header = 
  ( if rest_header_length < first_header_length then
      concatenate_strings 
        (rest_line_header 
         . replicate " " (first_header_length - rest_header_length)
        )
    else
      rest_line_header
  )
  in
  let folded_body = pp_divide_into_lines body (pp_width - first_header_length) 
  in
    (append_strings first_line_header (hd folded_body))
    .
    map (append_strings padded_rest_line_header) (tl folded_body)
;;
       
% format_hyp checks if a hypothesis is hidden, and 
  if so encloses hyp number in [] %

let pp_format_hyp branches (i,v,T,h) =
  let branch_string = pp_format_branches branches in
  let numeral_string = 
  ( if h then 
      concatenate_strings ["[";int_to_string i;"]. "]
    else
      append_strings (int_to_string i) ". "
  )
  in
  let id_string = 
  ( if is_visible_var v then 
      append_strings (tok_to_string (var_to_tok v)) ": "
    else
       ""
  )
  in
  let first_line_header = 
      concatenate_strings [branch_string;numeral_string;id_string] 
  in
    pp_format_block
      first_line_header
      branch_string
      (term_to_print_string T (pp_width - string_length first_line_header))
;;


let pp_format_hyps branches p parent_p = 
  let v_T_h_list = dest_full_hyps p in
  let i_v_T_h_list = zip (upto 1 (length v_T_h_list)) v_T_h_list in
  let parent_v_T_h_list = dest_full_hyps parent_p in
  let is_unchanged_hyp (i,v,T_h) =
    ((let parent_v,parent_T_h = nth i parent_v_T_h_list in
       ((is_invisible_var v & is_invisible_var parent_v) or v = parent_v)
       & parent_T_h = T_h
     )? 
     false
    )
  in
  flatten
  ( map
    (\i_v_T_h.
        if pp_same_hyps or not is_unchanged_hyp i_v_T_h then
          pp_format_hyp branches i_v_T_h
        else
          []
    )
    i_v_T_h_list
  )
;;



let pp_format_concl branches p parent_p =
  let c = concl p in
  if c = concl parent_p & not pp_same_concl then []
  else
  let branch_string = pp_format_branches branches in
  let first_line_header = append_strings branch_string 
                            (toks_to_string [turnstile_char;` `]) in
  let rest_line_header = append_strings branch_string "|  " in
  let cstring = term_to_print_string 
                  (concl p) 
                  (pp_width - string_length first_line_header) 
  in
    pp_format_block first_line_header rest_line_header cstring
;;

   
let pp_format_rule branches has_children p =
  let first_line_header = 
        append_strings (pp_format_nbranches branches) "BY "
  in
  let rest_line_header = 
        append_strings 
          (pp_format_branches branches) 
          (if has_children then "|" else "")
  in
  let rule = refinement p
  in let status = status_of_rule rule
  in let rstring = rule_to_print_string (refinement p) rule_width 
  in let comment = 
     ( if status = `INCOMPLETE` then 
         (tok_to_string (newline())) J "(INCOMPLETE RULE)"
       if status = `BAD`  then 
         (tok_to_string (newline())) J "(BAD RULE)"
       else 
         ""
     )
  in
    pp_format_block 
      first_line_header 
      rest_line_header 
      (rstring J comment)
;;



let pp_format_node branches has_children parent_p inline p =
  flatten
    [
     pp_format_preceding_lines branches inline
    ;
     pp_format_hyps branches p parent_p
    ;
     pp_format_concl branches p parent_p
    ;
     [pp_format_branches (true.branches)]
    ;
    (if (has_rule p) then 
       pp_format_rule branches has_children p 
     else 
       [append_strings (pp_format_branches branches) "INCOMPLETE"]
    )
    ]
;;


let null_proof = make_proof_node [] void_term ;;

% returns list of lines of proof %

let pp_format_proof_old p = 
  letrec aux branches parent_p p inline =
    if not (is_refined p) or null (children p) then 
      pp_format_node branches false parent_p inline p
    else
     pp_format_node branches true parent_p inline p
    @
   (let offspring = children p in
    let tagged_offspring =
       zip
         offspring
         (replicate false (length offspring - 1) @ [true])
    in
    let inline_only_offspring = not pp_always_indent & length offspring = 1 in
    let augment_brs last_sib brs = 
      if inline_only_offspring then brs else (not last_sib) . brs
    in

    flatten
     (map
       (\p',is_last_sib.
               aux (augment_brs is_last_sib branches) 
                              p p' inline_only_offspring
       )
       tagged_offspring
     )
   )
  in
    aux [] null_proof p false 
;;

% p is a term %
let pp_format_proof_wip p = 
  letrec aux branches parent_p p inline =
    if not false %(is_refined_term p)% or inil_term_p icons_op % null (children_of_inf_term% (subterm p 2) then 
      pp_format_node branches false parent_p inline p
    else
     pp_format_node branches true parent_p inline p
    @
   (let offspring = children p in
    let tagged_offspring =
       zip
         offspring
         (replicate false (length offspring - 1) @ [true])
    in
    let inline_only_offspring = not pp_always_indent & length offspring = 1 in
    let augment_brs last_sib brs = 
      if inline_only_offspring then brs else (not last_sib) . brs
    in

    flatten
     (map
       (\p',is_last_sib.
               aux (augment_brs is_last_sib branches) 
                              p p' inline_only_offspring
       )
       tagged_offspring
     )
   )
  in
    aux [] null_proof p false 
;;
let pp_format_proof p = 
  letrec aux branches parent_p p inline =
    if not (inil_term_p icons_op (subterm p 2)) then 
      pp_format_node branches false parent_p inline p
    else
     pp_format_node branches true parent_p inline p
    @
   (let offspring = children p in
    let tagged_offspring =
       zip
         offspring
         (replicate false (length offspring - 1) @ [true])
    in
    let inline_only_offspring = not pp_always_indent & length offspring = 1 in
    let augment_brs last_sib brs = 
      if inline_only_offspring then brs else (not last_sib) . brs
    in

    flatten
     (map
       (\p',is_last_sib.
               aux (augment_brs is_last_sib branches) 
                              p p' inline_only_offspring
       )
       tagged_offspring
     )
   )
  in
    aux [] null_proof p false 
;;


let pp_output_lines strings =
  map
  (\s. print_to_snapshot_file s ; print_return_to_snapshot_file ())
  strings
  ;
  ()
;;
 
let pp_print_proof p = pp_output_lines (pp_format_proof p) ;;

let Print p =
  open_snapshot_file true
  ; pp_print_proof p 
  ; close_snapshot_file ()
  ; p
;;

let PrintToFile name p =
   set_snapshot_file name 
   ; Print p
;;


let PrintTexFile name p =
  let prl_file = append_strings name ".prl" in
  let tex_file = append_strings name ".tex"
  in
  (PrintToFile prl_file p ; latexize_file prl_file tex_file ; p)
  
;;

let print_tex_lib from to name =
  let prl_file = append_strings name ".prl" in
  let tex_file = append_strings name ".tex"
  in
  ( print_library from to prl_file pp_width false 
    ; latexize_file prl_file tex_file
  )
;;

% tranformation tactics for displaying proofs %
let display_message s =
 message_emit [`display`] [] [s] [];;

let Disp p = 
  display_message "" ;
  map display_message (pp_format_proof p) ;
  display_message "" ;
  p 
;;



% sticks newline after each line %

let join_lines strs = 
  let toks_list = map (\x.string_to_toks x @ [(newline())]) strs
  in 
    toks_to_string (flatten toks_list)
;;

let proof_to_string p = 
  join_lines (pp_format_proof p)
;;

let proof_to_print_strings p n = 
  let width = pp_width 
  in
    pp_width := n
    ; let strs =  pp_format_proof p
      in
        pp_width := width
        ; strs
;;

