%
*************************************************************************
*                                                                       *
*    Copyrighted Cornell University 2000                                *
*                                                                       *
*                                                                       *
*                Nuprl Proof Development System                         *
*                ------------------------------                         *
*                                                                       *
*   Developed by the Nuprl group, Department of Computer Science,       *
*   Cornell University, Ithaca NY.  See the release notes for a list    *
*   of the members of the group.                                        *
*                                                                       *
*   Permission is granted to use and modify Nuprl provided this notice  *
*   is retained in derived works.                                       *
*                                                                       *
*                                                                       *
*************************************************************************
%

%[
*****************************************************************************
*****************************************************************************
THEORY.ML
*****************************************************************************
*****************************************************************************
Functions related to theory load and dump management.

March 3rd 93:

Changes made:

A Few names of functions modified.
Strings rather than tokens used for theories.
Theory delimiters changed.

Functions defined:

set_theory_filename 
set_theory_ancestors

show_theory_filename
show_theory_ancestors

]%


letref theory_filenames = [] : (string # string) list ;;
letref theory_ancestors = [] : (string # string list) list ;;

letref theory_filename_ext = ".lisp";;

let set_theory_filename name filename =
  theory_filenames := 
       update_alist theory_filenames name filename
  ;
  ()
;;


let set_theory_ancestors name ancestors = 
  theory_ancestors := 
       update_alist theory_ancestors name ancestors
  ;
  ()
;;

let show_theory_filename name = 
  apply_alist theory_filenames name 
    ? failwith 
       `show_theory_filenames: no entry for theory ` ^ (string_to_tok name)
;;

let show_theory_ancestors name = 
  apply_alist theory_ancestors name 
    ? failwith 
       `show_theory_ancestors: no entry for theory ` ^ (string_to_tok name)
;;


letrec get_deps_of_theory deps_sought deps_found thy =
   if member thy deps_sought then
     failwith (`get_deps_of_theory: circularity found involving: ` 
                ^ (string_to_tok thy))
   if member thy deps_found then
     deps_found
   else
   let parent_thys = 
      (apply_alist theory_ancestors thy
       ? [] %failwith (`get_deps_of_theory: no dependencies recorded for ` 
                    ^ (string_to_tok thy))%
      )
   in
   let deps_found' =
     get_deps_of_theories (thy.deps_sought) deps_found parent_thys
   in
     thy . deps_found'

 and get_deps_of_theories deps_sought deps_found thies =
   if null thies then deps_found
   else
   let thy.thies' = thies in
   let deps_found' = get_deps_of_theory deps_sought deps_found thy in
     get_deps_of_theories deps_sought deps_found' thies'
;;

% input: list of independent theories one wants loaded 
  output: list of all theories which have to be loaded, in load order.
%

let get_linear_theory_deps thy_list =
  rev (get_deps_of_theories [] [] thy_list)
;;

letref begin_theory_delimiter = "BEGIN_" ;;
letref end_theory_delimiter = "END_" ;;

letref begin_theory_suffix = "_begin" ;;
letref end_theory_suffix = "_end" ;;

%let mk_old_theory_delimiters thy =
  let thy' = string_to_tok (upcase_string (tok_to_string thy)) in
  let start_com = (string_to_tok begin_theory_delimiter) ^ thy' in
  let end_com = (string_to_tok end_theory_delimiter) ^ thy' in
    start_com,end_com
;;
%
let mk_old_theory_delimiters thy =
  let thy' = upcase_string thy in
  begin_theory_delimiter J thy'
  ,end_theory_delimiter J thy'
;;

let mk_theory_delimiters thy =
  thy J begin_theory_suffix
  ,thy J end_theory_suffix
;;

let exists_object_p_ap = begin_ap "exists_object_p";;
let exists_object_p name = lib_eval_to_bool (token_ap exists_object_p_ap name);;
let is_lib_member = exists_object_p;;

let is_theory_loaded thy = 
 let bn,en = mk_theory_delimiters thy
 in let bo,eo = mk_old_theory_delimiters thy
 in 
   is_lib_member (string_to_tok bn) 
   or is_lib_member (string_to_tok bo) 
;;

let get_theory_delimiters thy = 
 let bn,en = mk_theory_delimiters thy
 in let bo,eo = mk_old_theory_delimiters thy
 in 
 if is_lib_member (string_to_tok bn) & is_lib_member (string_to_tok en) then
   bn,en
 if is_lib_member (string_to_tok bo) & is_lib_member (string_to_tok eo) then
   bo,eo
 else
   failwith 
   ( `get_theory_delimiters: could not find delimiters for thy` 
     ^ string_to_tok thy
   )  
;;

% 
Returns (`begin`,thyname) if begin delimiter.
Returns (`end`,thyname) if begin delimiter.
Returns (`none`,"") if regular object.

Handles both new and old theory naming.
Checks theory names occur in theories argument.
(ignoring case)
%

let analyze_lib_obj_name theories =   
  let theories' = map upcase_string theories
  in let txlist = zip theories' theories
  in let esuffix = string_to_toks end_theory_suffix
  in let bsuffix = string_to_toks begin_theory_suffix
  in let eprefix = string_to_toks end_theory_delimiter
  in let bprefix = string_to_toks begin_theory_delimiter
  in
  \name.
    let name_chars = string_to_toks name
    in let kind,rootname =
    ( if is_prefix bprefix name_chars then
        `begin`,toks_to_string (remove_prefix bprefix name_chars)
      if is_prefix eprefix name_chars then
        `end`,toks_to_string (remove_prefix eprefix name_chars)
      if is_suffix bsuffix name_chars then
        `begin`,toks_to_string (remove_suffix bsuffix name_chars)
      if is_suffix esuffix name_chars then
        `end`,toks_to_string (remove_suffix esuffix name_chars)
      else
        `none`,""
    )
    in 
    let name' = upcase_string rootname
    in
    if kind = `none` then
      `none`,""
    if member name' theories' then
      kind, apply_alist txlist name'
    else
      `none`,""
;;





%

let mk_theory_delimiter_comments thy =
  let s1 = "************ " J (upcase_string thy) J " ************"
  in let s2 = toks_to_string (replicate `*` (length (string_to_toks s1)))
  in
    s1,s2
;;

let mk_text_term = itext_term;;

let add_theory_delimiters_at position thy =
  let start_com,end_com = mk_theory_delimiters thy 
  in let start_text,end_text = mk_theory_delimiter_comments thy 
  in
    create_com start_com position
    ; create_com end_com ("+" J start_com)
    ; term_to_object_body 
        (mk_text_term start_text)
        (string_to_tok start_com)
    ; term_to_object_body 
        (mk_text_term end_text)
        (string_to_tok end_com)
    ;check start_com
    ;check end_com
    ;refresh_lib_window ()
;;

let add_theory_delimiters thy = add_theory_delimiters_at "last" thy
;;
%

% fails if theory already loaded/ or no filename recorded for theory. %

%
let load_theory_at position fully thy =
  let filename = 
    apply_alist theory_filenames thy 
    ? 
    failwith (`load_theory_at: no filename for theory: ` ^ string_to_tok thy)
  in
  if is_theory_loaded thy then
      failwith `load_theory_at: theory already loaded`
  else
   (execute_command_line
     (concatenate_strings
      ["load "
      ;if fully then "fully " else ""
      ;position
      ;" from " 
      ;filename
      ;theory_filename_ext
      ]
     )
    ;
    ()
   )
;;

let load_theory_nr thy = load_theory_at "bot" false thy;;

let load_theory thy = 
  load_theory_nr thy 
  ;jump (fst (get_theory_delimiters thy))
;;


let load_fully_theory_nr thy = load_theory_at "bot" true thy;;
let load_fully_theory thy = 
  load_fully_theory_nr thy 
  ;jump (fst (get_theory_delimiters thy))
;;



let load_theories_with_ancestors theories =
   map 
     (\thy.load_theory_nr thy ? ())
     (get_linear_theory_deps theories)
   ;
   (():unit)
;;

let load_theories thys = load_theories_with_ancestors thys ;;

let load_fully_theories_with_ancestors theories =
   map 
     (\thy.load_fully_theory_nr thy ? ())
     (get_linear_theory_deps theories)
   ;
   ()
;;
%

% Careful not to dump a theory if it isn't loaded. Otherwise one can
clobber a theory.
%

%
let dump_theory thy =
  let start_com,end_com = get_theory_delimiters thy in
  let filename =
  ( apply_alist theory_filenames thy 
    ?
    failwith (`dump_theory: no info on filename for theory: ` ^ 
               (string_to_tok thy))
  )
  in
  if filename = "" then
    ()
  else
  ( execute_command_line
     (concatenate_strings
        ["dump " 
        ;start_com
        ;"-"
        ;end_com
        ;" to "
        ;filename
        ;theory_filename_ext
        ]
     )
     ;
     ()
  )
;;
%
% a quick common command %
%
let dump str = do map dump_theory (string_to_words str) ;; 
%
% only deletes loaded version of theory %
%
let delete_theory thy =
  let start_com,end_com = get_theory_delimiters thy in
  ( execute_command_line
     (concatenate_strings
        ["del " 
        ;start_com
        ;"-"
        ;end_com
        ]
     )
  )
;;

let check_theory thy =
  let start_com,end_com = get_theory_delimiters thy in
    check_objects start_com end_com
;;
%
%LAL temp for ver 5 comparison
let check_theory_thms_only thy =
  let start_com,end_com = get_theory_delimiters thy in
    check_objects start_com end_com
;;
%
% 
goes thru library picking out theory delimiters. Checks that
no objects are outside delimiters, and that delimiters are always paired.
Returns list of theories.
%

letref theory_orphans = ([]:string list) ;;

%
let list_theories (():unit) =
  let analyze = analyze_lib_obj_name (map fst theory_filenames)
  in
  letrec aux status lib_objs =
    if null lib_objs then
      ( if status = "out_of_theory" then
        ( if null theory_orphans then [] else
            failwith 
              (`list theories: ` ^ int_to_tok (length theory_orphans) 
               ^ ` object(s) in no theory. See theory_orphans for details`
              )
        )
        else
          failwith (`list_theories: missing end delimiter for: ` 
                    ^ string_to_tok status
                   )
      )
    else
    let obname = tok_to_string (hd lib_objs)
    in let kind,thy_name = analyze obname
    in
    if status = "out_of_theory" then 

   

     (if kind = `begin` then
        thy_name . aux thy_name (tl lib_objs)
      else
       ( theory_orphans := obname . theory_orphans
         ; aux status (tl lib_objs)
       )
      )
    else



     (if kind = `none` then
         aux status (tl lib_objs)
      if kind = `end` & thy_name = status then
         aux "out_of_theory" (tl lib_objs)
      if kind = `end` & not thy_name = status then
         failwith (`list_theories: theory ` 
                   ^ string_to_tok status
                   ^ ` ends with delimiter for theory ` 
                   ^ string_to_tok thy_name
                  )
      else
         failwith (`list_theories: found beginning delimiter for: `
                   ^ string_to_tok thy_name
                   ^ `nested within theory `
                   ^ string_to_tok status
                  )
      )
  in
    theory_orphans := []
    ; aux "out_of_theory" (library ())
        
;;


let dump_all_theories (():unit) =
  map dump_theory (list_theories ())
  ;
  ()
;;

let update_theory_delimiters theory = 
  let old_s,old_e = mk_old_theory_delimiters theory
  in let new_s,new_e = mk_theory_delimiters theory
  in let com_s,com_e = mk_theory_delimiter_comments theory
  in
  if is_lib_member (string_to_tok old_s) then
  ( rename old_s new_s
    ; rename old_e new_e
    ; term_to_object_body 
        (mk_text_term com_s)
        (string_to_tok new_s)
    ; term_to_object_body 
        (mk_text_term com_e)
        (string_to_tok new_e)
    ; check new_s
    ; check new_e
  )
  else
    (():unit)
;;

let update_all_theory_delimiters () =
  map update_theory_delimiters (list_theories ())
;;
%

%
Creates two files: fname.prl and fname.tex in same directory as the
file containing the library.(fname = filename). 
%

let print_theory thy = 
  let start_com,end_com = get_theory_delimiters thy in
  let lib_fname =
  ( apply_alist theory_filenames thy 
    ?
    failwith (`dump_theory: no info on filename for theory: ` ^ 
               (string_to_tok thy))
  )
  in
    print_tex_lib (string_to_tok start_com) (string_to_tok end_com) lib_fname
;;


%
let print_all_theories (():unit) =
  map print_theory (list_theories ())
  ;
  ()
;;
%
%
IDEAS For extension...

1. Dump modified theories. Keep track of which theories modified and which 
   not
%
%
let list_theory thy =
  let start_com,end_com = get_theory_delimiters thy 
  in
    map 
      tok_to_string
      (lib_segment (string_to_tok start_com) (string_to_tok end_com))
;;

let hide_theory_nr thy = 
  map hide_object (list_theory thy) ; ()
;;

let hide_theory thy = hide_theory_nr thy ; refresh_lib_window () ;;

let unhide_theory_nr thy = 
  map unhide_object (list_theory thy) ; ()
;;

let unhide_theory thy = unhide_theory_nr thy ; refresh_lib_window () ;;

let extract_theory thy = 
  mapfilter extract (list_theory thy) ; ();;

let expand_theory thy = 
  mapfilter expand (list_theory thy) ; ();;
let compress_theory thy = 
  mapfilter compress (list_theory thy) ; ();;
%

% strip all the primitive rule trees out
  of expanded proofs. 
%
%
let strip_theory thy = 
  mapfilter strip_object (list_theory thy) ; ();;
%
%[
***************************************************************************
Touch History Maintenance
***************************************************************************
Fresh objects are objects whose status has changed since last dump.
Take note of theories which library objects are in when they are touched.

This needs thinking about more. It won't work as is.
]%

%
letref fresh_objects = ([]:string list) ;;

let update_fresh_objects info = 
  let obnames = flatten (map (map fst o snd) info
  in
    fresh_objects := obnames @ fresh_objects 
;;

let init_fresh_objects (():unit) = 
 fresh_objects := [] 
 ; ()
;;


add_history_update_funs 
  `theory_changes_monitoring`
  ,init_fresh_objects
  ,update_fresh_objects
  ,id
;;
%

% class_fun 
     obname :string
     thyname:string
   : unit

called on each object obname with thyname , the name of its theory.
%
%
let classify_theory_objects class_fun =
  let analyze = analyze_lib_obj_name (map fst theory_filenames)
  in
  letrec aux status lib_objs =
    if null lib_objs then
      ( if status = "out_of_theory" then
        ( if null theory_orphans then [] else
            failwith 
              (`list theories: ` ^ int_to_tok (length theory_orphans) 
               ^ ` object(s) in no theory. See theory_orphans for details`
              )
        )
        else
          failwith (`list_theories: missing end delimiter for: ` 
                    ^ string_to_tok status
                   )
      )
    else
    let obname = tok_to_string (hd lib_objs)
    in let kind,thy_name = analyze obname
    in
    if status = "out_of_theory" then 

 

     (if kind = `begin` then
        thy_name . aux thy_name (tl lib_objs)
      else
       ( theory_orphans := obname . theory_orphans
         ; aux status (tl lib_objs)
       )
      )
    else

 

     (if kind = `none` then
         aux status (tl lib_objs)
      if kind = `end` & thy_name = status then
         aux "out_of_theory" (tl lib_objs)
      if kind = `end` & not thy_name = status then
         failwith (`list_theories: theory ` 
                   ^ string_to_tok status
                   ^ ` ends with delimiter for theory ` 
                   ^ string_to_tok thy_name
                  )
      else
         failwith (`list_theories: found beginning delimiter for: `
                   ^ string_to_tok thy_name
                   ^ `nested within theory `
                   ^ string_to_tok status
                  )
      )
  in
    theory_orphans := []
    ; aux "out_of_theory" (library ())
        
;;


let list_touched_theories (():unit) = 
  



let dump_touched_theories (():unit) = 
  map dump_theory (list_touched_theories ())
  ; init_fresh_objects ()
;;
%
