
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************

#+cltl2(in-system-package)
#-cltl2(in-package *system-package-name*)


;;;
;;; Functions for dealing with closures.  The format of a closure and the
;;; implementation of ap below should probably be reexamined for each system on
;;; which this is to run.  They depend on factors such as the efficiency of &rest
;;; arguments and the ability to call a function with a runtime variable number
;;; of arguments without consing an argument list.
;;;
;;; The implementation given here is of course tuned for Lisp Machines.  It
;;; exploits cdr-coding to use the smallest possible amount of space for
;;; closures.  It is implemented under the assumption that &rest arguments are
;;; cheap ie not consed.  It exploits the ability to call a function with a
;;; runtime variable number of arguments without consing an argument list.
;;;
;;; In this implentation there are two parts to a closure:
;;;   1) A pair containing the lisp function and its arity.
;;;   2) A list of the previously collected arguments.
;;;
;;; In a system where &rest arguments are consed, there should probably be a
;;; separate ap function for each of the common cases of 1, 2 or 3 arguments.
;;; Also in such a case, there isn't the need for runtime variable number of
;;; arguments function calling as the &rest argument list can be destructively
;;; modified instead.  Also, there isn't the need to cons up the closure; the
;;; argument list can be incorporated.

;;; These are defined later on ...
#|
(eval-when (compile)
  (proclaim '(inline closure-func closure-arity closure-fdescriptor closure-env)))

(defun closure-func (c) (car (first c)))
(defun closure-arity (c) (cdr (first c)))
(defun closure-fdescriptor (c) (first c))
(defun closure-env (c) (cdr c))
|#

(defmacro fast-length (l)
  ;; A version of length which avoids a function call for lists of length < 5.
  `(block length
     (let ((length 0)
	   (l ,l))
       ,@(macrolet ((inc-and-check ()
		      ''((setf l (cdr l))  (incf length)
			 (when (null l) (return-from length length)))))
	   `((when (null l) (return-from length length))
	     ,@(inc-and-check) ,@(inc-and-check)
	     ,@(inc-and-check) ,@(inc-and-check)
	     ,@(inc-and-check)
	     ;; If the end hasn't been reached yet, default to the
	     ;; standard length function.
	     (return-from length (+ length (length l))))))))




;; the arguments should be passed in reverse order,
;; as the ml translator produces the lisp code this way.
;; the calls are not reversed, but the enviroment is reversed.
;; thus, the args are reversed as the environment is built.
#+I-dont-want-this-to-be-defined
(defun ap (f &rest args)
    (do* ((env (closure-env f) (cons (car args) env))
	(args args (cdr args))
	(i (fast-length env) (1+ i)))
       ((or (null args) (= i (closure-arity f)))
	(cond

	  ;; apply
	  ((and (null args) (= i (closure-arity f)))
	   (apply (closure-func f) env))

	  ;; return closure
	  ((null args)
	   (cons (closure-fdescriptor f) env))

	  ;; recurse after apply
	  (t (apply #'ap (apply (closure-func f) env) args))))))







;;;
;;; This file contains an implementation of the closure functions as suggested in 
;;; this comment in a previous version of runtime.lisp file. 

;;; ***********
;;; Functions for dealing with closures.  The format of a closure and the
;;; implementation of ap below should probably be reexamined for each system on
;;; which this is to run.  They depend on factors such as the efficiency of &rest
;;; arguments and the ability to call a function with a runtime variable number
;;; of arguments without consing an argument list.
;;; In this implentation there are two parts to a closure:
;;;   1) A pair containing the lisp function and its arity.
;;;   2) A list of the previously collected arguments.
;;;
;;; In a system where &rest arguments are consed, there should probably be a
;;; separate ap function for each of the common cases of 1, 2 or 3 arguments.
;;; Also in such a case, there isn't the need for runtime variable number of
;;; arguments function calling as the &rest argument list can be destructively
;;; modified instead.  Also, there isn't the need to cons up the closure; the
;;; argument list can be incorporated.
;;; ***********


#-dontinline
(eval-when (compile)
  (proclaim '(inline
	      closure-func
	      closure-arity
	      closure-fdescriptor
	      closure-env)))

(defun closure-func (c) (car (first c)))
(defun closure-arity (c) (cdr (first c)))
(defun closure-fdescriptor (c) (first c))
(defun closure-env (c) (cdr c))


;;; A closure has form
;;;
;;; cl = ((f . n) . env)
;;;
;;; where n : int (>0) is the arity of the lisp function f, and 
;;; env = ej, ..., e1 is the environment of the closure, the arguments already
;;; supplied to f. We always have that j < n.
;;;
;;; An application of a closure has form.

;;; (ap cl a1 ... ak)
;;;
;;;
;;; What we do depends on k:
;;; 1. k+j<n :   Return      ((f . n) ak ... a1 ej ... e1)
;;; 2. k+j=n :   Return    (apply f ak ... a1 ej ... e1)
;;; 3. k+j>n :   let cl' = (apply f an-j ... a1 ej ... e1) 
;;;              Return (ap cl' an-j+1 ... ak)
;;;
;;;
;;; We create special cases of ap for k=1,2 and 3. and use a macro for ap to
;;; select whether to call the general ap function or one of the special
;;; optimized versions.
;;;
;;; for the macro see the file runtime-defs.lisp
;;;
;;; Would a case split speed this up at all??


;; NB: since j<n we have no case for 1+j>n.
;; k = 1

(defun one-ap (f-cl a1)
  (let ((cl-env (closure-env f-cl)))
    (if (= (1+ (fast-length cl-env))
           (closure-arity f-cl))
        ; 1+j=n
       (apply (closure-func f-cl) a1 cl-env)
      ;1+j<n
      (list* (closure-fdescriptor f-cl) a1 cl-env))))


;; k = 2

(defun two-ap (f-cl a1 a2)
  (let ((cl-env (closure-env f-cl)))
    (case (- (closure-arity f-cl) (fast-length cl-env))
          ;2+j = n+1
          ((1)
           (one-ap (apply (closure-func f-cl) a1 cl-env)
                   a2))
          ;2+j = n
          ((2)
           (apply (closure-func f-cl) a2 a1 cl-env))
          (otherwise
          ;2+j < n
           (list* (closure-fdescriptor f-cl) a2 a1 cl-env)))))

;; k = 3

(defun three-ap (f-cl a1 a2 a3)
  (let ((cl-env (closure-env f-cl)))
    (case (- (closure-arity f-cl) (fast-length cl-env))
          ;3+j = n+2
          ((1)
           (two-ap (apply (closure-func f-cl) a1 cl-env)
                   a2
                   a3))
          ;3+j = n+1
          ((2)
           (one-ap (apply (closure-func f-cl) a2 a1 cl-env)
                   a3))
          ;3+j = n
          ((3)
           (apply (closure-func f-cl) a3 a2 a1 cl-env))
          (otherwise
          ;3+j < n
           (list* (closure-fdescriptor f-cl) a3 a2 a1 cl-env)))))

(defun general-ap (f-cl &rest args)
  (let (arg-cons-ptr new-args (cl-env (closure-env f-cl)))
    (loop
     (cond
      ; arg-cons-ptr is pointing to (cons an-j ?).
      ; if k+j<n then arg-cons-ptr will be nil.

      ((setf arg-cons-ptr
              (nthcdr (- (closure-arity f-cl)
                         (fast-length cl-env)
                         1)
                      args))
       ; k+j >= n
       (setf new-args (cdr arg-cons-ptr))
       (setf (cdr arg-cons-ptr) nil)
       (cond (new-args

              ;k+j > n

              (setf f-cl (apply (closure-func f-cl) (nreconc args cl-env)))
              (setf args new-args)
              (setf cl-env (closure-env f-cl)))

              ; loop will carry us around with f-cl, cl-env and args set to new 
              ; values.
              ; The type system guarantees that the new f-cl is a closure.

             (t
              ;k+j = n
              (return (apply (closure-func f-cl) (nreconc args cl-env))))))
           
      (t
       ;k+j < n
       (return (cons (closure-fdescriptor f-cl)
                     (nreconc args cl-env))))))))


;;; Test functions
(defunml (|map_gen| (f l) :error-wrap-p nil)
	  ((%a -> %b) -> ((%a list) -> (%b list)))
  (mapcar #'(lambda (x) (general-ap f x)) l))

(defunml (|map_one| (f l) :error-wrap-p nil)
	  ((%a -> %b) -> ((%a list) -> (%b list)))
  (mapcar #'(lambda (x) (one-ap f x)) l))

(defunml (|gen_ap1| (f x) :error-wrap-p nil)
          ((%a -> %b) -> (%a -> %b))
          (general-ap f x))

(defunml (|ap1| (f x) :error-wrap-p nil)
          ((%a -> %b) -> (%a -> %b))
          (one-ap f x))

(defunml (|gen_ap2| (f x y) :error-wrap-p nil)
          ((%a -> (%b -> %c)) -> (%a -> (%b -> %c)))
          (general-ap f x y))

(defunml (|ap2| (f x y) :error-wrap-p nil)
          ((%a -> (%b -> %c)) -> (%a -> (%b -> %c)))
          (two-ap f x y))

