
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************

#+cltl2(in-system-package)
#-cltl2(in-package *system-package-name*)


;;;;	
;;;;	Allows direct loading of nuprl4 library files.
;;;;	
;;;;	
;;;;	import-nv4 (STRING{fname} <object-id{dir}> TOKEN{name})		: (values)
;;;;	  * creates subdir with name and populates with nv4 objects from lib file.
;;;;	    no interpretation is attempted, however operator migration is performed.
;;;;	
;;;;	
;;;;	combine mappings from withing v4 with mappings in v5
;;;;	want v4 mappings so as to share code with v4  fttb.
;;;;	


;;;;	 still need !dform_break_control{:t} -> !dform_break_control{NIL:t}
(defun v5-v4-term-map (term)

  (when (idform-break-control-term-p term) (setf -term term) (break "map"))
  (cond

    ((and (idform-break-control-term-p term)
	  (eql '|| (type-of-idform-break-control-term term)))
     (idform-break-control-term nil))

    (t term)))


(defunml (|v5_v4_term_map| (term))
    (term -> term)

  (term-walk-ops term
		 #'v5-v4-term-map))





;;;;	
;;;;	
;;;;	
;;;;	

(unless (find-package "NUPRL-VARS")
  (make-package "NUPRL-VARS" :use nil
		:nicknames '("vars" "NUPRL-VARIABLE")))

(unless (find-package "NUPRL-PARAMETER-VARS")
  (make-package "NUPRL-PARAMETER-VARS" :use nil
		:nicknames '("parm-vars" "NUPRL-ABSTRACTION-META-VARIABLE")))

(unless (find-package "NUPRL-TEMPLATE-VARS")
  (make-package "NUPRL-TEMPLATE-VARS" :use nil
		:nicknames '("template-vars" "NUPRL-DISPLAY-META-VARIABLE")))

(defparameter *nuprl-vars-package* (find-package 'nuprl-variable))
(defparameter *nuprl-template-vars-package* (find-package 'nuprl-display-meta-variable))
(defparameter *nuprl-parameter-vars-package* (find-package 'nuprl-abstraction-meta-variable))

;;
;;	would like to share migration code between v4 and v5 so can migrate by either
;;	export from v4 or
;;	load directly in v5 from dump files.
;;	misguided to attempt share see comments at head of mig-v4-5.
;;

(defun construct-term (op &optional bts)
  (instantiate-term op bts))

(defun construct-bound-term (term &optional bindings)
  (instantiate-bound-term term bindings))

(define-primitive template ((variable . id)))
;;(define-primitive |!dform_cons| nil (0 0))
(define-primitive |!dform_lhs_cons| nil (car cdr))
(define-primitive |!dform_child_cons| nil (car cdr))
(define-primitive |!dform_parent_cons| nil (car cdr))
(define-primitive |!cond_cons| nil (car cdr))
(define-primitive |void|)

(define-primitive |!dform_child_noelide_p|)
(define-primitive |!dform_child_mode| ((token . mode)))
(define-primitive |!dform_child_library| ((string . variable)))

(define-primitive |!dform_child| ((string . id) (string . descriptor)) (attributes))
(define-primitive |!dform_edit_macro|  ((string . token)))

(define-primitive |!dform_conditions| nil (list))
(define-primitive |!dform_parents| nil (list))
(define-primitive |!dform_hidden_ok|)
(define-primitive |!dform_not_point_condition|)

(defun name-of-idform-address-term (term)
  (value-of-parameter-r (car (last (parameters-of-term term)))))

(defun object-of-idform-address-term (term)
  (value-of-parameter-r (car (parameters-of-term term))))

(defun address-of-idform-child-parentheses-term (term) (pointer-of-idform-child-parentheses-term term))

(defun get-template-variable-id (s) (get-display-meta-variable-id s))

(defvar *isymbolic-dform-address-term-sig*
  (list* '|!dform_address| (list *token-typeid* *string-typeid*) nil))
  
(defun isymbolic-dform-address-term-p (term)
  (equal *isymbolic-dform-address-term-sig* (term-sig-of-term term)))

(defun add-dform-parent-condition (idform &rest r)
  (declare (ignore r))
  idform)

     
(defun inil-term-of-op (op)
  (instantiate-term op nil))

(defun conditionals-of-iabstraction-term (term) (subterm-of-term term '(0)))
(defun expansion-conditionals-of-abstraction (abs) (when abs (conditions-of-abstraction abs)))
(defun construct-parameter-from-string (s typeid)
  (instantiate-parameter-s s (type-id-to-type typeid)))

(defun list-to-ilist-by-op (l op) (map-sexpr-to-ilist l (inil-term-of-op op)) )
(defun ilist-to-list (l &optional (op (icons-op))) (map-isexpr-to-list l op))
(defun map-ilist (ilist f &optional (op (icons-op))) (map-isexpr-to-list ilist op f))

(defun permuted-dforms-of-disp-object (obj-name) (declare (ignore obj-name)) nil)
(defun address-to-dform-p (&rest r) (declare (ignore r)) nil)
(defun term-to-dform-address (idform-addr) (declare (ignore idform-addr)) nil)

(defun rhs-of-idform-term (term) (subterm-of-term term '(2)))
(defun lhs-of-idform-term (term) (subterm-of-term term '(1)))
(defun object-name-of-dform (dform) (declare (ignore dform)) nil)

(define-primitive |!obj_mig| ((tok . name) (tok . kind)) 
 (properties source))

(defun char->ichar (ch &optional (newline-ichar inewline))
  (cond ((char= ch #\newline)
	 newline-ichar)
;	((char= ch #\return)
;	 newline-ichar)
	((char= ch #\tab)
	itab)
	((char-code ch))))


(defun ichar (ch) (char->ichar ch))


(defun istring (x)
  (map 'list #'char->ichar (if (stringp x)
			       x
			       (princ-object-to-string x))))

;; can't be since not defined as parameter type in v5.
(defun parameter-substitution-list-parameter-p (p) (declare (ignore p)) nil)


(defun term-to-v4-sexpr (term)
  (labels ((visit-meta-variable-id (id)
	     (cond
	       ((display-meta-variable-id-p id)
		(intern (string id) *nuprl-template-vars-package*))
	       ((abstraction-meta-variable-id-p id)
		(intern (string id) *nuprl-parameter-vars-package*))
	       (t (raise-error (error-message '(v5 export varible id meta) id)))))

	   (variable-p (term)
	     (and (eql (id-of-term term) '|variable|)
		  (null (cdr (parameters-of-term term)))
		  (let ((p (car (parameters-of-term term))))
		    (and (variable-parameter-p p)
			 (real-parameter-p p)))
		  (null (bound-terms-of-term term))))

	   (parameter-value-to-sexpr (pv type)
	     (let ((v (value-of-parameter-value pv)))
	       (cond
		 ((extended-parameter-value-p v)
		  (cond
		    ((slot-parameter-value-p v)
		     (let ((s (descriptor-of-slot-parameter-value v)))
		       (if (meta-variable-id-p s)
			   (cons (cond
				   ((display-meta-variable-id-p s) 'd)
				   ((abstraction-meta-variable-id-p s) 'a)
				   (t (break)))
				 'slot)
			   'slot)))))
		 ((meta-parameter-value-p v)
		  (visit-meta-variable-id v))
		 ((real-parameter-value-p v type)
		  (real-parameter-value-to-string v type))
		 (t (break "foo" )))))

	   (parameter-to-sexpr (p)
	     (setf -p p)
	     (cons (parameter-value-to-sexpr (parameter-value p) (type-of-parameter p)) (type-id-of-parameter p)))

	   (operator-to-sexpr (term)
	     (cons (id-of-term term)
		   (mapcar #'parameter-to-sexpr (parameters-of-term term))))

	   (term-to-sexpr (term)
	     (setf -term term)
	     (if (variable-p term)
		 (string (id-of-variable-term term))
		 (cons (operator-to-sexpr term)
		       (mapcar #'bound-term-to-sexpr (bound-terms-of-term term)))))
	   
	   (binding-to-sexpr (binding)
	     (parameter-value-to-sexpr binding *variable-type*))

	   (bound-term-to-sexpr (bt)
	     (cons (mapcar #'binding-to-sexpr (bindings-of-bound-term bt))
		   (term-to-sexpr (term-of-bound-term bt))))
	   )

    (term-to-sexpr term)
  ))


(defun v4-sexpr-to-term (sexpr)
  ;;(setf -sexpr sexpr) (break "v4stt")
  (labels ((visit-parameter-value (sexpr type)
	     (if (and (consp sexpr)
		      (not (member (car sexpr) '(a d))))

		 ;; old
		 (progn (setf -sexpr sexpr) (break "v4sttc"))
		 
		 (cond

		   ((consp sexpr)
		    (slot-parameter-value
		     (case (car sexpr)
		       (a (get-abstraction-meta-variable-id "slot"))
		       (d (get-display-meta-variable-id "slot"))
		       (otherwise (raise-error (error-message '(v4 import parameter slot) sexpr))))))
		   
		   ((stringp sexpr)
		    (let ((istr (istring sexpr)))
		      (if (and (eql ipercent (car istr))
			       (eql ipercent (cadr istr)))
			  ;;old
			  (progn (setf -sexpr sexpr) (break "v4stts"))
			  (maybe-string-to-parameter-value sexpr type))))

		   ((eql 'slot sexpr)
		    (slot-parameter-value "slot"))

		   ((symbolp sexpr)	; meta variable.

		    ;;(setf -vsexpr sexpr)  (break "v4sttm")

		    (let ((id sexpr))
		      (cond
			;;((eq (symbol-package id) *nuprl-vars-package*) (get-variable-id id))
			((and (eq (symbol-package id) *nuprl-parameter-vars-package*)
			      (level-expression-type-p type))
			 (format t "level-expression fudge~%")
			 (intern-system id))
			((eq (symbol-package id) *nuprl-parameter-vars-package*) (get-abstraction-meta-variable-id id))
			((eq (symbol-package id) *nuprl-template-vars-package*) (get-display-meta-variable-id id))
			(t (raise-error (error-message '(v4 import parameter meta) sexpr)))))))))


	   (visit-parameter (sexpr)
	     (let ((type-id (lookup-typeid (cdr sexpr))))
	       (if type-id
		   (let ((type (type-id-to-type (cdr sexpr))))
		     (instantiate-parameter (visit-parameter-value (car sexpr)
								   type)
					    type))
		   (string-parameter (format-string "nuprl4 migration unknown type-id ~a" (cdr sexpr))))))

	   (visit-term (sexpr)
	     (if (stringp sexpr)
		 (variable-term (get-variable-id sexpr))
		 (instantiate-term (instantiate-operator (caar sexpr)
							 (mapcar #'visit-parameter (cdar sexpr)))
				   (mapcar #'(lambda (btsexpr)
					       (instantiate-bound-term
						(visit-term (cdr btsexpr))
						(mapcar #'(lambda (bsexpr)
							    (visit-parameter-value bsexpr
										   *variable-type*))
							(car btsexpr))))
					   (cdr sexpr))))))
    (visit-term sexpr)))


	   
(defun v4-migrate-abs (term name)

  (let ((iabs-found-count 0))

    (labels ((search-iabs (term)
	       (if (iabstraction-term-p term)
		   (progn
		     (incf iabs-found-count)
		     (migrate-iabstraction term))
		   (construct-term (operator-of-term term)
				   (mapcar #'(lambda (bt)
					       (construct-bound-term (search-iabs (term-of-bound-term bt))
								     (bindings-of-bound-term bt)))
					   (bound-terms-of-term term))))))

      (prog1
	  (search-iabs term)
	(unless (onep iabs-found-count)
	  (format t
		  "Warning[~a] : found ~a !abstraction(0;0;0) terms in abs object when only one expected."
		  name
		  iabs-found-count))))))


(defun geo-to-iprop (geo)
  (iproperty-term 'geometry (map-list-to-ilist geo (inil-term) #'iint-term)))

(defvar *tactic-tree-nil-term* (instantiate-term (instantiate-operator '|tactic_tree|)))
(defun rbt-to-tttt (rbt)
  (icons-term (v4-sexpr-to-term (car rbt))
	      (map-list-to-ilist (cdr rbt) *tactic-tree-nil-term*
				 #'rbt-to-tttt)))


(defvar *v4-raw-import-p* nil)

(defun v4-migrate-object (name kind term rest metrics)
    
  (let ((props nil))

    (let ((geo metrics)
	  (data
	   (case kind
	     (com	term)
	     (rule	term)
	     (ml	term)

	     (abs	 (if *v4-raw-import-p* ;; nil
			     term
			     (v4-migrate-abs term name))
			 )

	     (disp	(if (and (not *v4-raw-import-p*) ;; nil
				 (or (idform-term-p term)
				     (idform-cons-term-p term)))
			    (migrate-dform-isexpr term nil name)
			    term))
	     
	     (lat	(if (and (not *v4-raw-import-p*) ;; nil
				 (or (iprecedence-ordered-term-p term)
				     (iprecedence-unrelated-term-p term)
				     (iprecedence-equal-term-p term)))
			    (migrate-prec-isexpr term)
			    term))
	     
	     (thm
	      (let ((rbt (cadr rest)))
		(setf -term term -rest rest)
		(setf -rbtt (rbt-to-tttt rbt))
		;;(break "v4mo")
		(icons-term term (rbt-to-tttt rbt))
		))	      ;; rest  rbt??
	       
	     (otherwise
	      (raise-error (error-message '(v4 import migrate kind) kind name))))))

      (when geo
	(push (geo-to-iprop geo) props))
	   
      (iobj-mig-term name kind
		     (map-list-to-ilist props (inil-term))
		     data))))


(defun v4-load-object (sexpr)
  (let ((name (first sexpr))
	(kind (second sexpr))
	(term (v4-sexpr-to-term (fourth sexpr)))
 	(rest (fifth sexpr))
	(metrics (sixth sexpr)))
    

    (v4-migrate-object name kind term rest metrics)

    ))



(defun v4-lib-load (load-port)

  (let ((acc nil)
	)
    
    (let ((dump-format (read load-port)))

      ;; require latest v4 format. If not in that format then need to load then dump
      ;; from v4.
      (unless (or (eql dump-format 'PRL-LIB-DUMP-V4.0.10))
	(raise-error (error-message '(v4 import lib version) dump-format)))

      (do ((dump-obj (read load-port nil nil) (read load-port nil nil)))
	  ((null dump-obj))
	
	(push (v4-load-object dump-obj) acc))

      (nreverse acc))))


(defunml (|nuprl4_import| (fname))
    (string -> (term list))

 (with-open-file (port fname :direction :input)
		  (v4-lib-load port))
  )
			  






;;;;	
;;;;	Export : desire ability  to export to v4 and retain as much structure as possible
;;;;	  - names : rename if ambiguous
;;;;	  - object-ids
;;;;	     * store map from names to object ids in special comment object to use to reconstruct at v5 load.
;;;;	  - property lists :
;;;;	      * store map from names to properties in some set of comment objects using some fixed criteria
;;;;		to limit size of objects.
;;;;	  - directory trees :
;;;;	      * flatten tree via some algorithm to allow for rebuilding of tree at import.
;;;;	      * directories become comments.
;;;;	      * add begin/end dir comment objects.
;;;;	  - disp, prec, term, inf objects
;;;;	     * hide in comment obj.
;;;;	  - code objects
;;;;	     * if not ml then hide in comment obj.
;;;;	
;;;;	use tree listing to get flat list of directories?
;;;;	
;;;;	
;;;;	
;;;;	  - just refiner ml, abs, and thms
;;;;	  - single prf 
;;;;	  - at read, overwrite existing data
;;;;	
;;;;	
;;;;	
;;;;	
;;;;	

(define-primitive |pfterm_to_tactic| nil (tree))
(define-primitive |pfdisp_step| ((string . show)) (goal tac subgoals addr))
(define-primitive |!seq| () (annos sequent))
(define-primitive |red_hyp| ((natural . num) (token . hidden) (string . some)) (type (1 . sequent)))
(define-primitive |pfdisp_concl| ((string . show)) (concl))
(define-primitive |pfdisp_subgoals| () (car cdr))
(define-primitive |proof_anno_rep| ((token . label) (string . value)) (term))
(define-primitive |pfterm_addr_extend| ((natural . n)) (rest))
(define-primitive |pfterm_addr_top| ())


(defun dump-v4-theorem-object (objc)

  (let* ((prf (car (proofs-of-statement-source (source-of-objc objc))))
	 (pobjc (when prf (oc prf))))

    (let ((stu-ml-nil (instantiate-term (instantiate-operator '|ml_nil| nil))))
      (labels
	  ((object-status ()

	     ;; no prf or incomplete proof -> prf then partial.
	     ;; otherwise complete.
	       
	     ;;(setf -pobjc pobjc) (break "dto")
	     (if pobjc
		 (or (with-ignore (tok-upcase (token-of-itoken-term (itree-status (inf-tree-of-proof-source-r (source-of-objc pobjc))))))
		     'incomplete)
		 'incomplete))

	   (dump-rule-body-tree (tree)
	     (setf -tree tree)
	     (cons (when (car tree)
		     (replace-oids (term-to-v4-sexpr (car tree))))
		   (mapcar #'dump-rule-body-tree (cdr tree))))


	   (seq-to-stu-seq (i seq)
	     (if (iinf-sequent-term-p seq)
		 (red-hyp-term i
			       (if (hidden-of-iinf-sequent-term seq) '|t| '|f|)
			       ""
			       (type-of-iinf-sequent-term seq)
			       (binding-of-sequent-of-iinf-sequent-term seq)
			       (seq-to-stu-seq (1+ i) (sequent-of-iinf-sequent-term seq)))
		 (pfdisp-concl-term "show" seq)))


	   (annos-to-stu-annos (annos)
	     (if annos
		 (let ((ml-annos (filter #'(lambda (a) (and (iannotation-term-p a)
							    (eql `ml-annotation (label-of-iannotation-term a))))
					 annos)))
		   (if ml-annos
		       (progn
			 (setf -ml-annos ml-annos)
			 (when (cdr ml-annos) (break "annos +"))
			 (let ((alabel (label-of-iml-annotation-term (term-of-iannotation-term (car ml-annos)))))
			   (proof-anno-rep-term (name-of-ianno-label-term alabel)
						(let ((val (value-of-ianno-label-term alabel)))
						  (if (inatural-term-p val)
						      (princ-to-string (numeral-of-inatural-term val))
						      ""))
						stu-ml-nil)))
		       (proof-anno-rep-term '|main| "" stu-ml-nil)))
		 (proof-anno-rep-term '|main| "" stu-ml-nil)))

	   (goal-to-stu-goal (goal)
	     (let ((term (sequent-of-goal goal))
		   (annos (annotations-of-goal goal)))
	       (iseq-term (annos-to-stu-annos annos) 
			  (seq-to-stu-seq 1 term))))

	   (stu-addr (addr)
	     (if addr
		 (pfterm-addr-extend-term (car addr)
					  (stu-addr (cdr addr)))
		 (pfterm-addr-top-term)))

	   (itree-to-stu-tree (itree addr)
	     (let* ((objc (with-ignore (objc-of-inf-tree itree)))
		    (istep (when objc (step-of-inf-objc objc))))
	       
	       (let ((tac (or (when (and istep (refined-inf-step-p istep))
				(tactic-of-inf-step istep))
			      (itext-term "")
			      )))
		 (pfdisp-step-term "show"
				   (goal-to-stu-goal (goal-of-inf-step istep))
				   tac
				   (let ((i 0))
				     (let ((subgoals (mapcar #'(lambda (tree)
								 (incf i)
								 (itree-to-stu-tree tree (cons i addr)))
							     (children-of-inf-tree itree))))
				       (map-list-to-isexpr (reverse subgoals)
							   (instantiate-term (pfdisp-subgoals-op) nil)
							   )))
				   (stu-addr addr)))))
	     
	   (dump-stu-body-tree (itree)
	     (cons (term-to-v4-sexpr (replace-oids (pfterm-to-tactic-term (itree-to-stu-tree itree nil))))
		   nil
		   ))

	   (rule-body-tree-of-proof (itree)
	       
	     (let* ((objc (with-ignore (objc-of-inf-tree itree)))
		    (istep (when objc (step-of-inf-objc objc))))
	       (cons (when (and istep
				(refined-inf-step-p istep))
		       (tactic-of-inf-step istep))
		     (when objc
		       (mapcar #'rule-body-tree-of-proof
			       (children-of-inf-tree itree))))
	       ))
	   )


	;;(break "dto")

	;; if not active should be rolled up as comment? or ignored? or sent as usual.

	(list (object-status)
	      (dump-stu-body-tree (inf-tree-of-proof-source-r (source-of-objc pobjc)))
	      ;;(dump-rule-body-tree (rule-body-tree-of-proof (inf-tree-of-proof-source-r (source-of-objc pobjc))))
	      ;; skip extract fttb:
	      (when nil;;(theorem-object-extract obj)
		;;(term-to-v4-sexpr (theorem-object-extract obj))
		)
	      nil)))))
	

(defun sanitize-name (n) (intern (sanitize-name-string n)))

(defunml (|sanitize_v4_object_name| (n))
    (string -> string)
  (sanitize-name-string n))

(defun termof-to-termof4 (tot)
  (let ((oid (oid-of-termof-term tot))
	(les (level-expressions-of-termof-term tot)))
    (instantiate-term (instantiate-operator (intern-system (sanitize-name (name-of-oid oid)))
					    les))))

(defun replace-oids (term)
  (if (is-termof-id term)
      (termof-to-termof4 term)
      (if *convert-oids-to-strings-p*  
	  (maybe-instantiate-term
	   term
	   (instantiate-operator (id-of-term term)
				 (mapcar #'(lambda (p)
					     (if (oid-parameter-p p)
						 (string-parameter (string (name-of-oid (value-of-parameter-r p))))
						 p))
					 (parameters-of-term term)))
	   (mapcar #'(lambda (bt)
		       (maybe-instantiate-bound-term
			bt
			(bindings-of-bound-term bt)
			(replace-oids (term-of-bound-term bt))))
		   (bound-terms-of-term term)))
	  term)))

(defun dump-object (objc)
  ;;(let ((*dumping* t)))
  (list (let ((iprop (property-of-objc objc 'name)))
	  (when iprop
	    (format t "~%DumpObject ~a " (token-of-itoken-term iprop))
	     (sanitize-name (token-of-itoken-term iprop))))
	(case (kind-of-objc objc)
	  (code 'ml)
	  (stm 'thm)
	  (term 'com)
	  (otherwise (kind-of-objc objc)))
	nil
	(let ((rterm (source-reduce (term-of-source (source-of-objc objc))
				    '(|EditEphemeral|)))) 
	  (case (kind-of-objc objc)
	    (disp (term-to-v4-sexpr (migrate-5to4-idform-op-sweep rterm)))
	    (com  (term-to-v4-sexpr (migrate-oid-op-sweep rterm)))
	    (otherwise (term-to-v4-sexpr (replace-oids rterm)))))
	
	(case (kind-of-objc objc)
	  (stm (dump-v4-theorem-object objc))
	  (otherwise nil))
	(let ((iprop (property-of-objc objc 'geometry)))
	  (when iprop
	    (map-isexpr-to-list iprop (icons-op) #'numeral-of-inatural-term)))))

(defun lib-dump-term (kind name term)
  ;;(let ((*dumping* t)))
  (list name kind nil
	 (term-to-v4-sexpr term)
	nil nil))


(defun lib-dump-term-hook (h pstream)

  (let ((ndumped	0))

    ;; Lib-list will be nil if object not found or range out of order.
    (prl-char-stream-write-string pstream (format-string "~a~%" 'PRL-LIB-DUMP-V4.0.10))

    (format t "lisp calls hook")
    (prog1 (funmlcall h
		      (mlclosure
		       #'(lambda (knameterm)
			   ;;(format t "ml called hook")
			   (let ((kind (car knameterm))
				 (name (cadr knameterm))
				 (term (cddr knameterm)))
			     (handle-process-err #'(lambda (err-str)
						     (process-err
						      (format-string "DumpError[~a]: ~a"
								     name err-str))
						     nil)
						 (prl-stream-write (lib-dump-term kind name term) pstream)
						 (incf ndumped)
						 (when (zerop (mod ndumped 100))
						   (format t "~%dumped ~a   " ndumped))
						 nil
						 )) )
		       1))

      (format t "~a objects dumped." ndumped))))

(defunml (|dump_terms_v4| (fname h))
    (string -> ((((tok |#| (tok |#| term)) -> unit) -> *) -> *))

  (with-prl-open-file (pstream fname out t nil)
    (when (null pstream)
      (process-err
       (format-string "Could not open output file '~a'." fname)))

    (lib-dump-term-hook h pstream)))


(defun lib-dump-list (objects pstream)

  (let ((ndumped	0))

    ;; Lib-list will be nil if object not found or range out of order.
    (prl-char-stream-write-string pstream (format-string "~a~%" 'PRL-LIB-DUMP-V4.0.10))

    (dolist (nameoid objects)
      (let ((name (car nameoid))
	    (oid (cdr nameoid)))
      
	(let ((objc (oc oid)))
	  (handle-process-err #'(lambda (err-str)
				  (process-err
				   (format-string "DumpError[~a]: ~a"
						  name err-str))
				  nil)
			      (if objc
				  (progn
				    (prl-stream-write (dump-object objc) pstream)
				    (incf ndumped))
				  (format t "Object [~a] not in library." name))))))

    (if (= ndumped 0)
	(format t "Object(s) not found or improper range.")
	(format t "~a objects dumped." ndumped))))



(defunml (|dump_theory_v4| (fname l))
    (string -> (((tok |#| object_id) list) -> unit))

  ;; Open the file, return any problems via a 'throw' to cmd-err.
  ;;(setf -a fname fname -l l) (break "dtv4")

  (with-prl-open-file (pstream fname out t nil)
    (when (null pstream)
      (process-err
       (format-string "Could not open output file '~a'." fname)))

    (lib-dump-list l pstream))

    ;;; no error if here.
  (format t "Dump to ~a completed." fname)
  nil
  )

