
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************

#+cltl2(in-system-package)
#-cltl2(in-package *system-package-name*)

;;; export substance
;;;;

(defun map-objc-to-table-type (objc)

  (case (kind-of-objc objc)

    (com	'comments)
    (abs	'abstractions)
    ;;(prf	'abstractions)
    (rule	'rules)
    (stm	'statements)
    (disp	'dforms)
    (prec	'precedences)
    (code	'code)
    (term	'terms)

    (otherwise nil)))


;;;;
;;;; 	Translate
;;;;

;; TODO : at the moment, it does not appear much is done to presever substance
;;  in cases where there is no substantive change. Also need a way to compare
;;  substances in ML so that ML code can judge is change substantive.
;; compare result with substance to judge substantive change.
;; Translation could be done in two stages. Modify could do the translation
;; implicitly, non-force fails if translation not completed.
;; 

;;;;	
;;;;	translation : 
;;;;	  - Lib : term-of-source -> source-reduce -> term-of-substance
;;;;	      * 'SOURCE references.
;;;;	    if references and term-of substance same, then update
;;;;	    create translation by inheriting

  
(defvar *abs-reference-static* nil)

(defun lib-abs-reference-static (desc re term)
  (funmlcall (or *abs-reference-static*
		 (setf *abs-reference-static* (ml-text "lib_abs_reference_static")))
	     desc re term))

;; TODO :: somehow seems like this should be more automatic, particular the wrapping and returning
;;   should be implicit in rsp or noted upon receipt of inform msg.
(defun note-static-abstraction-references (description re term)
  (let ((sdeps (if (or (null description) 
		       (equal (system-of-description description) (intern "metaprl")))
		   (reference-static-abstractions term)
		   (lib-abs-reference-static (description-to-term description)
					     re
					     ;;(rhs-of-iabstraction-term term)
					     term
					     ))))

    ;;(setf -sdeps sdeps) (break "lars")
    (unless (ivoid-term-p sdeps)
      (dependency-note-environment (term-to-environment-dependencies sdeps)))
    ))

(defun objc-abs-translate (source term description)
  ;;(declare (ignore description))
  ;;(break "oat")

  (unless (or (null description) 
		  (equal (system-of-description description) (intern "metaprl"))
		  (equal (system-of-description description) (intern "Metaprl")))
      (abstraction-translate-source-r term))

  (prop-substance term (substantive-properties-of-source source)))
  

(defun objc-com-translate (source term description)
  (declare (ignore source description))

  (substance term))


;; proofs listed in stm substance are those
;; of source which match and are complete.
(defun objc-stm-translate (source term description)
  (declare (ignore description))

  (when nil ;;(term-walk-p term #'iplaceholder-term-p)  ;; nil ?
    (raise-error (error-message '(stm translate placeholder) term)))

  (let ((prf-acc nil)
	(extp (extract-required-p source))
	(ext nil))

    (dolist (prf (proofs-of-statement-source source))

      ;; TODO: if prf oid has no object then there will be no reference recorded.
      ;; then if obj assiged to oid, there is no method of detecting the
      ;; source ref and causing retranslation ? Maybe some static check
      ;; will record the ref ???
      ;;(break "tsp")

      ;; handle err
      (with-handle-error (('(translate stm prf)))

	(let* ((prf-obj (library-lookup prf))
	       (prf-objc (objc-of-library-object prf-obj)))
	
	  (if (library-object-active-p prf-obj)
	      (let ((goal (goal-of-prf-objc-r prf-objc)));; LAL returns cons seq . anno

		;;(setf -goal goal -prf-objc prf-objc -term term) (break "ost")
		(if (equal-terms-p term (car goal))
		    (progn
		      (push prf prf-acc)
		      ;; ?want refs to improper proofs so that retranslation happens if they change.
		      ;; but not proof refs, instead use access-definition-object-id ?? or make up some other ref-type
		      (dependency-note-reference 'proof
						 (dependency-of-definition prf-obj))
		      (when (and extp (null ext))
			(let ((next (maybe-extract-of-prf-objc prf-objc)))
			  (when (and next (not (ivoid-term-p next)))
			    (setf ext next)))))
		    (message-emit (oid-warn-message (list prf) '(goal match not) term goal))))
	      (message-emit (oid-warn-message (list prf) '(active not))))
	  )))

    (new-statement-substance term
			     (substantive-properties-of-source source)
			     (nreverse prf-acc)
			     ext)))


(defun nuprl4-description-p (d)
  (and (eql (system-of-description d) 'nuprl)
       (= 4 (car (version-of-description d)))))

(defun nuprl5-description-p (d)
  (and (eql (system-of-description d) 'nuprl)
       (= 5 (car (version-of-description d)))))

(defun fdl0-description-p (d)
  (and (eql (system-of-description d) 'fdl)
       (= 0 (car (version-of-description d)))))

(defun objc-disp-translate (source term description)
  ;;(setf a description) (break)
    
  (cond
    ((null description)
     (raise-error (error-message '(translate disp description not))))

    ((nuprl5-description-p description)
     (let ((prop (property-of-source source 'name)))
       (if prop
	   (display-substance-wname term
				    (substantive-properties-of-source source)
				    (disp-translate-source term)
				    (token-of-itoken-term prop))
	   (display-substance term
			      (substantive-properties-of-source source)
			      (disp-translate-source term)))))

    ((nuprl4-description-p description)
     (raise-error (error-message '(translate disp v4 not yet done))))

    (t (raise-error (error-message '(translate disp v?))))))

(defun objc-prec-translate (source term description)
  (declare (ignore source))
    
  (cond
    
    ((or (null description)
	 (nuprl5-description-p description))
     (prec-translate-source-r term)
     (substance term))

    ((nuprl4-description-p description)
     (raise-error (error-message '(translate prec v4 not yet done))))

    (t (raise-error (error-message '(translate prec v?))))))


;;;;	
;;;;	
;;;;	Translate :
;;;;	  - determines if source has changed such that new translation is required.
;;;;	  - produces a dependency event.
;;;;	  - produces substance for activation.
;;;;	
;;;;	
;;;;	
;;;;	
;;;;	code : 
;;;;	
;;;;	fruitless to attempt to hold binary data in lib, except as file proxies.
;;;;	IE, we manage code by having code objects contain file names for both 
;;;;	source and bin files. Also, code objects may contain source and lib may
;;;;	produce files representing multiple objects.
;;;;	
;;;;	Two modes:
;;;;	  - active : lib manages files. Aware of comment syntax (== can see ast?).
;;;;	  - passive : lib is dumb.
;;;;	FTTB, assume active mode.
;;;;	
;;;;	data :
;;;;	  - source : term and/or file.
;;;;	  - binaries : file (names and dates).
;;;;	  - explicit(hard-coded,maybe produced by ast scan)  object dependencies.
;;;	      could be in source if oid part of code-ids or could be property.
;;;;	  - xref : event ? dependencies or symbols?
;;;;	      For various uses of symbols object dependencies are declared.
;;;;	      Ie if object a contains call of fun in object b then there is a call (b) dependency in a.
;;;;	      * distinction between fun defs and funcalls? fun def is a call!
;;;;	        
;;;;	      * object ids?
;;;;	      * Need to be able to detect redefinition for dependencies.
;;;;		what if two objects define same func, which is depended upon?
;;;;		last activated? but then lib must know which was last activated.
;;;;		error? 
;;;;		undefined? sensible to redefine funcs but require explicit dependency!
;;;;		  then if a touched a' compiled if a
;;;;	        have graph of ids and know which oid contains which ids. but refdefinition means same id
;;;;		so graph must be more than id ie id.oid. When building graph oid of fun called if fun
;;;;		defined in two places will be? (compiler tells us? or we arrange so compiler uses right one?)
;;;;	
;;;;		need some kind of a-priori order as can not compile (could use comp failure as search predicate)
;;;;		initially without an order. Ie need order as input to translate but use ddag as input
;;;;		to bag(group) activate.
;;;;		could use ast to produce compile order, but that would not be built-in step?
;;;;	
;;;;	
;;;;	bag activate order is general problem and should be determined by dag at time of activate 
;;;;		(after translate), cycles are possible (probable) need to identify types of dependencies.
;;;;	ie identify a list of dependency tags which determine activate order. '(ast calls activate/translate)
;;;;	
;;;;	  - file time-stamps.
;;;;	  - compilation list : 
;;;;	      * architecture, eg sunos-sparc, linux86.
;;;;	      * compiler, eg acl, lucid, cmucl, etc.
;;;;		Language? is substantive property already?
;;;;		compiler version?? should be included in compiler
;;;;	      * bin filename : identifiable to GC. derived from substance stamp?
;;;;		understandable by clients, determined/managed by lib.
;;;;	  - identifiers : assume merge or working maps manages language identifiers.
;;;;	    however, if (like lisp) an excutable supports multiple working maps, then
;;;;	    must manage to avoid clashing between working maps.
;;;;	      * ML : each environment has distinct ml environment.
;;;;	      * Object-ids : all object id's within all environments distinct and languge ids
;;;;		derived from object ids.
;;;;
;;;;	issues :
;;;;	  - translation required
;;;;	  - object-ids.
;;;;	  - multiple binaries.
;;;;	  - multiple languages.
;;;;
;;;;	
;;;;	Translate :  want substance stamp to survive comment changes to source.
;;;;	  - supports a participant to report source diff.
;;;;	  - requires a participant to compile and produce xref.
;;;;	  - substance and source contain xref and used by translation required
;;;;	  
;;;;	Allow source to be updated with compilation. Translation should migrate
;;;;	compilation data to substance without updating substance stamp.
;;;;	Why do we trust such an update? 
;;;;	
;;;;	
;;;;	Activate : observer must be prepared to compile if not compiled for 
;;;;	observers platform. 
;;;;
;;;;	
;;;;	when to compile, when to load ?
;;;;	
;;;;	maybe have seperate scan pass which produces xref and can be used
;;;;	to fdetermine compilation order.
;;;;	  also maybe scan can detect identity with previous version by ignoring 
;;;;	  comments at language level. source-reduce ignore's comment at term level
;;;;	  but comments may be embedded in text in syntax of language.
;;;;
;;;;	
;;;;	dup defs make derived order hard.
;;;;	FTTB expect to be told compile order (for inactive), but be able to compute an
;;;;	activate order.
;;;;
;;;;	  - source 
;;;;	      * file : source in object is file-pointer.
;;;;	      * direct : term is text rep of source.
;;;;	
;;;;	FTTB : ignore files.
;;;;	  - source : term
;;;;	  - property : 'require
;;;; 	      * hardcode.
;;;;	      * ? produce by examining ast or xref dag? 
;;;;	  - identifier dag (xref). xref of id.<dependency> pairs.
;;;;	      * FTTB from translate compile.
;;;;	
;;;;	  FTTB : no binaries.
;;;;	  - translate fails if no compiler available.
;;;;	  - activate recompiles.
;;;;	
;;;;	assume translate compiles but activate can test if compiled and avoid recompilation if so.
;;;;	ie maybe translate updates code table so activate can relate.
;;;;	assume activate can test if compilation needed?
;;;;	
;;;;	translation-required 
;;;;	source should contain compilation info. ie xref and bin file pointer.
;;;;	then if recompiled and no visible change, substance stamp not updated.
;;;;	  - why would we be recompiling if no change? 
;;;;	
;;;;	we require that obj be compiled/loaded in dest process prior to activation,
;;;;	then activate does not need to fail
;;;;	
;;;;	
;;;;	code 
;;;;	
;;;;	source
;;;;	  - source
;;;;	      * literal source term.
;;;;	      * proxy filename.
;;;;	  - xref : should be same for all compilations.
;;;;		by id.
;;;;	  - compilation list : 
;;;;	     * bin filename : identifiable to GC.
;;;;	     * architecture, ie sunos-sparc, linux86, etc
;;;;	     * compiler, ie acl, lucid, cmucl, etc.
;;;;	
;;;;	if active then must have be compiled. However, may not 
;;;;	be compiled on with compiler and hardware in use.
;;;;	But, compilation should not fail.
;;;;	But, must be prepared for such failure.
;;;;
;;;;	substance :
;;;;	  - compilation list.
;;;;	  - adding to list should not update stamp.
;;;;	  - how to add to list once distributed? delete and re-insert.
;;;;	    do not broadcast mod. Have lib update source with compile info
;;;;	    and at next translate compile info will migrate to substance.
;;;;	
;;;;	lexical object_id scoping vs dynamic edit object_id scoping :
;;;;	  ml-id is pair of object-id and identifier. object id can be 
;;;;	  dynamically inserted by editor during edit. Or could be
;;;;	  declared lexically in code.
;;;;	  - Or could be hard coded in code.
;;;;	  - Or could be implicit in ml global environment, ie take latest.
;;;;	
;;;;	lexical-scoping: all ids defined in a file assume obid of proxy object.
;;;;	
;;;;	
;;;;	
;;;;	
;;;;
;;;;
;;;;	It is not easy to include obid's in ascii code in files.
;;;;	maybe some convention within the id's. eg foo::id means obid with
;;;;	name property foo containing identifier id.
;;;;	
;;;;	
;;;;	
;;;;	xref is <dependency>.<term{ident}> list
;;;;	 
;;;;   At compile time, xref is produced by compiler+code-table
;;;;	code-table has map from ident -> dependency, update of map is 
;;;;	parameterized by Language property of code object. Thus differing
;;;;	update methods are possible for various languages.
;;;;	
;;;;	tis possible for defs to be snuck past the code table so that the
;;;;	depenedencies are innaccurate. Can modify compiler to keep track
;;;;	of dependencies. Or accept inaccuracies. Code table can supply
;;;;	hook to compiler so compile can inform code table of sneaks.
;;;;	Or expect each compile to be parameterized by code-table to prevent sneaking.
;;;;	so true for all dynamic compiles. Compiles done at system level missed, ie
;;;;	done outside of any environment.
;;;;	
;;;;	goes hand in hand with global-env being part of environment.
;;;;
;;;;	TODO fix shared ref variables : see bml-defs



;;;;	code activation : 
;;;;	
;;;;	code translation : 
;;;;	  - send to any code listener.
;;;;	  - translate and compile, but do not load.
;;;;	  - return xref and bin filename. bin-filename
;;;;	    should be stamp-based and in db? 
;;;;	    bin-file reference must be recognized by db gc and bin-files should be collected.
;;;;	  - in order to allow bin files, terms must be embedded, if embedded as pointers to term files
;;;;	    then gc must recognize.
;;;;	
;;;;	
;;;;	
;;;;	
;;;;	

;; twould be nice to get xref info back in rsp.
;; twould be nice to get bin file name back too.
;;  to produce bin file need way of encoding terms so they load well.
;;  inline terms as standard char string or byte array and then parse
;;  at load time.

   ;; this requires substance to be loaded twould be nice if not.
   ;; maybe compile is post activate! then substance there but still in transaction!
   ;; if after then no ability to collect dependencies from compile.
   ;; fttb not important.


;;  (orb-eval-by-description
;;   (description-property-term-of-objc objc)
;;   (term-of-substance (substance-of-objc objc))
;;   'all)


(defvar *lib-ml-code-compile* nil)
;; dependency . proxy
(defvar *code-order-sort-dependency-kinds*
  '((compile static |reference_environment|) . (|reference_environment_index|)))

(defun lib-ml-code-compile (desc reterm term)
  (funmlcall (or *lib-ml-code-compile*
		 (setf *lib-ml-code-compile* (ml-text "lib_ml_code_compile ")))
	     desc reterm term))


(defvar *lib-lisp-code-compile* nil)

(defun lib-lisp-code-compile (desc reterm term)
  (funmlcall (or *lib-lisp-code-compile*
		 (setf *lib-lisp-code-compile* (ml-text "lib_lisp_code_compile ")))
	     desc reterm term))



(defvar *lib-sql-code-query* nil)
(defvar *lib-sql-code-exec* nil)

(defun lib-sql-code-query (desc term schema)
  (funmlcall (or *lib-sql-code-query*
		 (setf *lib-sql-code-query* (ml-text "lib_sql_code_query ")))
	     desc term schema))

(defun lib-sql-code-exec (desc term)
  (funmlcall (or *lib-sql-code-exec*
		 (setf *lib-sql-code-exec* (ml-text "lib_sql_code_exec ")))
	     desc term))


(defun code-compiled-p (objc)
  (let ((source (source-of-objc objc)))
    (and (typep source 'code-source)
	 (let ((reduced (reduced-of-code-source source)))
	   ;;(setf -reduced reduced -source source -objc objc) (break "ccp")
	   (and reduced
		(bins-of-code-source source)
		(compare-terms-p (source-reduce (term-of-source source)
						(cons (kind-of-objc objc) (reduce-property-of-objc objc)))
				 reduced)))

	 ;;nil
	 ;;(format t "ccp~%")
	 )))


(defun reference-environment-property-of-objc (objc)
  (let ((re (property-of-objc objc '|reference_environment|)))
    (if re
	(ml-inr (oid-of-ioid-term re))
	(ml-inl nil))))

;; see also reference-environment-property-of-substance in com-orb 
;;
(defvar *re-default-prop* '|reference_environment_relative_minimal|)

(defunml (|compile_with_re_flavor| (flavor f a))
    (tok -> ((* -> **) -> (* -> **)))
    (let ((*re-default-prop* flavor))
      (funmlcall f a)))

(defun reference-environment-term-of-objc (objc)
  (or (property-of-objc objc *re-default-prop*)
      (property-of-objc objc '|reference_environment_relative_minimal|)
      (property-of-objc objc '|reference_environment_theories_minimal|)
      (property-of-objc objc '|reference_environment_minimal|)
      (property-of-objc objc '|reference_environment|)
      (ivoid-term)))


;;;;	
;;;;	
;;;;	compile happens prior to translation.
;;;;	
;;;;	
;;;;	source-reduce : 
;;;;	  - objc-translate
;;;;	  - translation-required-p
;;;;	  - compile-code-objc-source
;;;;	  - inf-preprocess-reduce
;;;;	  - inf-objc-refine-preprocess

;; objc -> objc
(defun compile-code-objc-source (objc)
  (let ((source (source-of-objc objc))
	(language (language-property-of-objc objc))
	(reduce-dependencies nil)
	(reduced nil)
	(rreduced nil))

    ;; whose catching references made by source-reduce ??
    (with-dependencies-vertical ('(compile-code source-reduce))
      (setf reduced (source-reduce (or (term-of-source source) (raise-error (error-message '(compile code source not))))
				   (cons (kind-of-objc objc) (cons 'LIB (reduce-property-of-objc objc)))))

      ;; apply hooks and reduce again
      ;;(setf rreduced (source-reduce (setf -hooked (apply-compile-code-process-hooks language reduced))
      ;;(cons (kind-of-objc objc) (reduce-property-of-objc objc))))
      
      (setf reduce-dependencies (setf -reduce-dependencies (event-dependencies-collected-term))))
    

    ;; TODO no-op if reduced same as reduced-of-code-source?
    ;; maybe caller should check that.
    ;;(setf -objc objc  -red reduced -rred rreduced) (break "ccos")

    (case language
      (ml (let* ((compiled (lib-ml-code-compile (description-property-term-of-objc objc)
						(reference-environment-term-of-objc objc)
						reduced)))

	    ;;(setf -rd reduce-dependencies -c compiled -reduced reduced) (break "rd")
	    (let ((deps (unless (ivoid-term-p reduce-dependencies) (list reduce-dependencies)))
		  (cdeps (dependencies-of-iml-compiled-term compiled)))
	      (unless (ivoid-term-p cdeps)
		(push cdeps deps))
		
	      (setf -a objc -b reduced -c source r-t-d compiled -e reduce-dependencies -f deps) ;; (break "ccos")

	      (objc-modify-source objc
				  (new-code-source-compiled source
							    language
							    reduced
							    (term-to-xrefs (xref-of-iml-compiled-term compiled))
							    (list (bin-of-iml-compiled-term compiled))
							    deps)))))

      (lisp;;(break "translate lisp code")
       (let* ((compiled (lib-lisp-code-compile (description-property-term-of-objc objc)
					       (reference-environment-term-of-objc objc)
					       reduced)))

	 ;;(setf -rd reduce-dependencies -c compiled -reduced reduced) (break "rd")
	 (let ((deps (unless (ivoid-term-p reduce-dependencies) (list reduce-dependencies)))
	       (cdeps (dependencies-of-icode-compiled-term compiled))
	       )
	   (unless (ivoid-term-p cdeps)
	     (push cdeps deps))
		
	   ;;(setf -a objc -b reduced -c source -d compiled -e reduce-dependencies -f deps) (break "ccos")

	   (objc-modify-source objc
			       (new-code-source-compiled source
							 language
							 reduced
							 (term-to-xrefs (xref-of-icode-compiled-term compiled))
							 (list (bin-of-icode-compiled-term compiled))
							 deps))))
       )

      (sql
       (let ((schema (property-of-objc objc 'schema)))

	 ;; schema forced by inability to examine database field types.
	 ;; this is probably not an inherent limitation of database
	 ;; but a limitation of my understanding and use. Thus may change.

	 
	 ;; if tuples property then do query otherwise exec.
	 ;; if query substance is tuples otherwise nada.

	 (let ((table (if schema
			  (list (irdb-table-term
				 (lib-sql-code-query (description-property-term-of-objc objc)
						     reduced
						     schema)))
			  (progn
			    (lib-sql-code-exec (description-property-term-of-objc objc)
					       reduced)
			    nil))))

	   (objc-modify-source objc
			       (new-code-source-compiled source
							 language
							 reduced
							 nil
							 table
							 (unless (ivoid-term-p reduce-dependencies) (list reduce-dependencies))))
	   ;;(break "ctsql")
	   )))

      (otherwise (raise-error (error-message '(objc compile code language unknown )))))))
  

(defunml (|code_compiled_p| (objc))
    (object_contents -> bool)
  
  (code-compiled-p objc))

(defunml (|compile_code_objc_source| (objc))
    (object_contents -> object_contents)
  
  (compile-code-objc-source objc))

;;  (map_lib ``CODE`` true (\oid oc. not (check_code_idents oc)))
;; returns list of objects whose substance does not appear to have correct idents.
(defunml (|check_code_idents| (objc))
    (object_contents -> bool)
  
  (let ((src (source-of-objc objc))
	(sub (substance-of-objc objc)))

    (let ((ridents (idents-of-code-source src))
	  (uidents (idents-of-code-substance sub)))

      ;;(setf -objc objc  -ridents  ridents  -uidents uidents) (break "ccsi")
      (null (set-difference ridents uidents))
      )))
  

;; activate should do compile, not translate?
(defun objc-code-translate (source term description)
  (declare (ignore description))
    
  (unless (typep source 'code-source)
	  (raise-error (error-message '(translate code ml source compiled never))))
  
  (let ((l (language-of-code-source source)))

    (let ((reduced (reduced-of-code-source source)))
      (unless (and reduced (compare-terms-p term reduced))
	(setf -term term -reduced reduced)
	(raise-error (error-message '(translate code ml source compiled not)))))
	 
    ;;(setf -source source -term term -description description) ;; (break "oct")
    (case l

      ((ml lisp)
       (let ((xdeps (xref-dependencies-of-code-source source)))
	 (when xdeps
	   (dependency-note-dependencies (list (new-dependencies 'compile xdeps)))))

       ;; probably not the best way of doing this but still effective.
       (dolist (event-term (dependencies-of-code-source source))
	 (dolist (env-deps (list-of-stamp-dependencies (term-to-event-dependencies event-term)))
	   (dependency-note-environment env-deps)))
	     
       ;;(setf -a source) (break "oct")
       (new-code-substance term
			   (idents-of-code-source source)
			   (bins-of-code-source source)
			   (substantive-properties-of-source source)))
      
      (sql
       ;;(break "ctsql")
       (new-code-substance term
			   nil
			   nil
			   (substantive-properties-of-source source)))


      (t (raise-error (error-message '(translate code language unknown)))))))
	 
(defunml (|table_of_sql_code| (objc))
    (object_contents -> term)
  (let ((source (source-of-objc objc)))
    ;;(setf -substance substance) (break "tsqlcs")
    (car (bins-of-code-source source))))


(defun objc-term-translate (source term description)
  (declare (ignore description))

  ;;(break)
  (with-dependencies
      (walk-note-oid-dependencies 'static term))

  ;;(break "ott")
  (prop-substance term (substantive-properties-of-source source))

  )


(defun objc-rule-translate (source term description)
  (declare (ignore source description))

  (let ((interp-term (rule-definition-translate-source term)))
      (rule-definition-check interp-term)
      (rule-substance term interp-term)))



#|
(defun prf-tree-other-dependencies (inf-tree)
  (list (squash-dependency-store-list-to-event-dependencies
	 '(translate prf inf_tree)
	 (inf-tree-collect-dependencies inf-tree))))
|#


(defun prf-tree-xref (inf-tree)
  (let ((acc nil))
    (labels ((visit (inf-tree)
	       (push (xref-of-source (source-of-objc (objc-of-inf-tree inf-tree)))
		     acc)
	       (mapc #'visit (children-of-inf-tree inf-tree))))
      (visit inf-tree))

    (map-list-to-ilist (reduce #'nconc
			       (mapcar #'(lambda (x)
					   (map-isexpr-to-list x (ixref-cons-op)))
				       (nreverse acc)))
		       (ixref-nil-term))))

(defun show-brief-dependencies (oid)
  (mapcar #'(lambda (x)
	      (cons  (tag-of-dependencies x) (length (list-of-dependencies x))))
	  (dependencies-of-dependency-store (translation-of-objc (oc oid)))))


(defun objc-prf-translate (source term description t-data)
  (declare (ignore description term t-data))


  ;; t-data should contain dependendencies and extract from source.
  ;; <t-data> : <dependency-store> . extract ????

  ;; TODO TODO TODO : prf-translate dependencies ???
  ;; need to extract dependencies from tree as well.
  ;; or should we consider those source dependencies?
  ;; no, statement, proof, rewrite, abstraction, rule dependencies
  ;; should be extracted from source and saved in substance and
  ;; a difference in these semantic dependencies should result in retranslation
  ;; retranslation consist of extracting dependencies and extract and new stamp.

  ;; prescence of substance does not imply complete proof.

  ;; some post processing needs to be done on extract to remove undeclared
  ;; level variables. this should be done here or
  
  ;; TODO TODO TODO
  ;; - must ensure inf-tree is valid, ie goals and subtree's match, etc.
  ;;   or is it assured already? ie can we build a bad inf-tree.
  ;; - if we allow links to object-id's then if oid updated then
  ;;   dependency dissonance that dependency in tree will not match.

  ;; prf substance dependencies is union of dependencies of inf substances
  ;; prf translation included union of inf translations.

  ;; inf-tree-proxy-summary
  ;;  - complete
  ;;  - goal : 
  ;;  - xref : part of proof source.
  ;;  - dependencies 
  ;;  - extract
  (let ((inf-tree (inf-tree-or-proxy-of-proof-source source)))
    
    (unless inf-tree
      (raise-error (error-message '(translate proof inf-tree not))))

    (let ((g (goal-of-inf-tree-or-proxy inf-tree)))
      
      (when (exists-p #'(lambda (term) (term-walk-p term #'iplaceholder-term-p)) g)  ;; nil ?
	(raise-error (error-message '(prf translate goal placeholder) g)))

      
      (unless (or (complete-inf-tree-proxy-p inf-tree)
		  (verify-proof inf-tree g))
	;;(setf -inf-tree inf-tree -g g) (break "vp")
	(raise-error (error-message '(translate proof inf-tree verified not))))
    
      (set-proof-source-xref-or-proxy inf-tree source)

      ;; get lemma and termof dependencies into termof
      ;;(setf -inf-tree inf-tree) (break "tp")
      (dolist (envdeps (list-of-stamp-dependencies (stm-prf-dependencies-of-inf-tree-or-proxy inf-tree)))
	(dependency-note-environment envdeps))

      ;;(setf -a g -b inf-tree) (break "tp")
      (new-proof-substance (substantive-properties-of-source source)
			   g
			   (let ((ext (extract-of-inf-tree-or-proxy inf-tree)))
			     (when ext
			       (extract-le-fixup (sequent-of-goal g)
						 ext)))) )))

    
;;;;	
;;;;	inf refine : all refinements should be via objc in order to collect
;;;;		dependencies.
;;;;	
;;;;	what does it mean for inf to be active.
;;;;	  - refinement present
;;;;	      * consistent with preprocess (reduced matches tactic of inf-step)
;;;;	      * goal of step matches goal of inf-objc
;;;;	  - translated :
;;;;	      * collected dependencies
;;;;	      * collected xref
;;;;	  - partial extract available
;;;;

;;;;	
;;;;	translation : makes abbreviated proof substance
;;;;	
;;;;	
;;;;	
;;;;	



;; xrefs will come back as annotation of step.

(defun inf-preprocess-reduce (term)
  (source-reduce term '(ml inf ref)))

(defun inf-objc-refine-preprocess (objc)

  (let ((source (source-of-objc objc))
	(treduced nil) ;; like done by translation.
	(nobjc nil)
	)

    ;; should only be one reduce call and that should reduced term.
    ;; but editor incorrectly pulls tactic from inf-step instead of source thus
    ;; fttb put mostly unmodified term into step.
    (with-dependencies-vertical ('(refine-inf source-reduce))
      (prog1
	  (setf treduced (source-reduce (term-of-source source)
					(cons (kind-of-objc objc) (reduce-property-of-objc objc))))
	;; 'ml is not necessarily right,  it is a sensible default
	;; but might be better to deduce from desc.
	(setf nobjc (preprocess-objc-modify-source objc
						   (inf-preprocess-reduce treduced)
						   (event-dependencies-collected (new-event-description *system* *version* *dependency-event-tags*)
										 (transaction-stamp))))))

    nobjc))


(defun inf-objc-update-refine (objc nstep)
  (setf -nstep nstep -objc objc
	-nobjc (inf-objc-modify-step (if (refined-inf-step-p nstep)
					 (xref-objc-modify-source objc (xrefs-of-inf-step nstep))
					 objc)
				     nstep)
	)
  ;;(break "iour")
   -nobjc
   )

(defun inf-objc-refine (objc envoid oids)
  (let ((nobjc (inf-objc-refine-preprocess objc)))
    (let ((src (source-of-objc nobjc))
	  (step (step-of-inf-objc-r nobjc)))
      (let ((tactic (reduced-of-source src)))

      (inf-objc-update-refine
       nobjc
       (inf-step-refine (description-property-term-of-objc objc)
			step
			tactic
			envoid
			oids))))))

(defun inf-objc-refinet (objc envterm)
  (let ((nobjc (inf-objc-refine-preprocess objc)))
    (let ((src (source-of-objc nobjc))
	  (step (step-of-inf-objc-r nobjc)))
      (let ((tactic (reduced-of-source src)))

      (inf-objc-update-refine
       nobjc
       (inf-step-refinet (description-property-term-of-objc objc)
			step
			tactic
			envterm
			))))))

;; a backdoor to allow multiple building proof after independent refinement.
;; seems a heck of a lot easier than the usual method.
(defun inf-objc-refined-aux (objc iinf)
  (when (not (iinf-tree-term-p iinf))
    ;;(setf -iinf iinf) (break "ior")
    (raise-error (error-message '(|inf_objc_refined| |!inf_tree| not) (list iinf))))

  (let ((nobjc (inf-objc-refine-preprocess objc)))
    (let ((src (source-of-objc nobjc)))
     (let ((tactic (reduced-of-source src)))
       (inf-objc-update-refine nobjc
			       (inf-step-update-refinement (step-of-inf-objc-r nobjc)
							   iinf
							   tactic)) ))))

(defunml (|inf_objc_refined| (objc iinf))
    (object_contents -> (term -> object_contents))

  (inf-objc-refined-aux objc iinf))

(defunml (|inf_objc_refined_tree| (objc deps rsrc rtt))
  (object_contents -> (term -> (term -> (term -> object_contents))))

 (let ((iinf (refinement-tree-to-iinf-tree-term deps rsrc rtt)))

   (inf-objc-refined-aux objc iinf)))


(defun inf-objc-refine-begin (objc)
  (let ((nobjc (inf-objc-refine-preprocess objc)))
    (let ((src (source-of-objc nobjc))
	  (step (step-of-inf-objc-r nobjc)))
      (let ((tactic (reduced-of-source src)))

	(list* (description-property-term-of-objc nobjc)
	       (goal-to-term (goal-of-inf-step step))
	       tactic
	       (add-notify #'(lambda (term)
			       (inf-objc-update-refine nobjc
						       (inf-step-update-refinement (step-of-inf-objc-r objc)
										   term
										   tactic)))))))))

(defunml (|inf_objc_refine_begin| (objc))
    (object_contents -> (term |#| (term |#| (term |#| term))))

  (inf-objc-refine-begin objc))

(defunml (|inf_objc_refine_complete| (cookie result))
    (term -> (term -> object_contents))

  (notify cookie result))

(defun objc-other-translation-dependencies (objc)
  (case (kind-of-objc objc)
    (abs nil)
    (inf (let ((source (source-of-objc objc)))
	   (list
	    (event-dependencies (current-transaction-stamp)
				(new-event-description *system* *version* '(preprocess inf))
				(append
				 (list-of-stamp-dependencies (reduce-dependencies-of-source source))
				 (let ((xdeps (dependencies-of-xrefs (xrefs-of-source source))))
				   (when xdeps
				     (list 
				      (environment-dependencies (current-transaction-stamp)
								(list (new-dependencies 'compile xdeps))))))
				 )))))
    (prf (other-dependencies-of-inf-tree-or-proxy
	  (inf-tree-or-proxy-of-proof-source (source-of-objc objc))))
    (stm nil)
    (com nil)
    (rule nil)
    (disp nil)
    (prec nil)
    (code nil)
    (term nil)
    (otherwise nil)
    )
  )

(defun objc-inf-translate (source term description t-data)
  (declare (ignore description term))

  (let ((astep (if (inf-step-p t-data)
		   t-data
		   (inf-step-abbreviate (step-of-inf-source source)))))

    (let ((goal (goal-of-inf-source source)))
      (unless (or (null goal)
		  (similar-goals-p goal (goal-of-inf-step astep)))
	(raise-error (error-message '(translate inf goal similar not) goal))))

    ;; dependencies ? reduce dependencies + dependencies of xref + dependencies of abbrev step.

    #|
    (mapc #'dependency-note-environment
	  (list-of-stamp-dependencies (reduce-dependencies-of-source source)))
   
    (let ((xdeps (dependencies-of-xrefs (xrefs-of-source source))))
      (when xdeps
	(dependency-note-dependencies (list (new-dependencies 'compile xdeps)))))
    |#
    
    (let ((de (references-of-inf-step astep)))
      ;;(setf -de de -source source) (break "oit")
      (dependency-note-dependencies (list-of-stamp-dependencies de)))

    (new-inf-substance astep (substantive-properties-of-source source))))



;;;;	
;;;;	
;;;;	A dependency is touched (stale) if the object id is no longer bound or the
;;;;	substance stamp of the dependency does not match the current substance
;;;;	of the object.
;;;;	
;;;;	
;;;;	static dependency : ie occurence of object id in substance of object.
;;;;	  touch of such dependency does not necessarily require retranslation.
;;;;	  Such a dependency should be present in translation though to make it
;;;;	  visible in dependency graph.
;;;;	
;;;;	
;;;;	subtype dependencies :
;;;;	  translation allows more specific methods for each kind of object.
;;;;	  these methods may generate references.
;;;;	
;;;;	
;;;;	Translation : 
;;;;	  - reduce source : translate-souce event.
;;;;	      * term
;;;;	      * source dependencies.
;;;;	      * can ignore static dependencies in source.
;;;;	    If no source dependencies touched then might skip reduction. However, this would
;;;;	    preclude the possibility of a new abstraction taking effect, ie the dependencies
;;;;	    may enlarge. Sentinels ala sfa would fix.
;;;;
;;;;	  - translation required : check if translation args match previous translation.
;;;;	    ARGs:
;;;;	      * reduced source
;;;;	      * substantive properties
;;;;	    If some object depended upon by a translation arg has been touched since
;;;;	    last translation then translation required. Tis possible that some
;;;;	      
;;;;	  - translate : translate event
;;;;	      check consistency of object.
;;;;	      * syntax
;;;;	      * stm - check referenced proofs match stm goal.
;;;;	      * prf - complete tree with frontiers matching subgoals.
;;;;	      * inf - refined.
;;;;	    include static dependencies of reduced and all substantive properties?
;;;;	
;;;;	should be possible to return arg objc if translation is a no-op.
;;;;	  - translation is no-op if 
;;;;	      * reduced term same.
;;;;	      * reduction dependencies equivalent.
;;;;	
;;;;	

(defun stale-dependency-p (d)
  (let ((oid (oid-of-dependency d)))
    (let ((objc (library-lookup oid)))
      (or (null objc)
	  (let ((sub (substance-of-objc objc)))
	    (or (null sub)
		(not (equal-stamps-p (stamp-of-data sub) (data-stamp-of-dependency d)))))))))

;; check for stale dependencies in depedencies of indicated tags.
;; tags should catch
;;   - code objects for referenced ml functions
;;   - inf objects of prfs. 'inference ?
;;   - 'proof :  prf objects of stms
;;
;; RLE TODO TODO TODO detecting stale dependencies

(defun dependencies-touched-p (dependency-store tags)
  (find-first #'(lambda (dependencies)
		  (when (or (eql t tags)
			    (member (tag-of-dependencies dependencies) tags))
		    (find-first #'(lambda (d)
				    (stale-dependency-p d))
				(list-of-dependencies d))))
	      (dependencies-of-dependency-store dependency-store)))


(defun objc-translate (objc &optional force)

  ;;(format t "objc_translate1")
  (let ((new-objc (clone objc))
	(source (source-of-objc objc))
	(substance (substance-of-objc objc t)))

    ;; first stage : source-reduce source.
    ;; collect abs references as event.
    (let* ((event-dependencies nil)
	   (o-desc (description-property-of-objc objc))
	   (re (reference-environment-term-of-objc objc))
	   (rtags (reduce-property-of-objc objc))
	   (t-data-p nil)
	   (unreduced (term-of-source source))
	   (reduced nil))

      (not-without-dependencies
       (with-dependency-event ('(translate-source))
	 (with-dependency-environment
	     (setf reduced (when unreduced
			     (source-reduce unreduced
					    (cons (kind-of-objc objc)
						  (cons 'LIB rtags)))))

	   (with-dependencies

	       (setf t-data-p (translation-required source substance reduced))
	      
	     ;;(setf -a t-data-p -b source -c substance -d reduced )(break "tdp")
							    
	     (dependency-note-reference 'source
					(dependency *null-oid*
						    (stamp-of-objc objc)
						    (stamp-of-data source)))


	     ;; kludge to support sorting of ml objects that alter ml reference variables
	     ;; we maintain order from ver. 4 to guarantee replay of proofs
	     (let ((prop (property-of-objc objc 'code-order)));;strict ver4 order
	       (when prop
		 (dependency-note-reference 'compilev4 (dependency (oid-of-ioid-term prop) nil nil)))) )


	   ;; want to record dependency on order of ref vars, todo
	   ;;(let ((prop (property-of-objc objc 'compile-order)))
	   ;;(when prop
	   ;;(mapc #'(lambda (o)
	   ;;(dependency-note-reference 'compile (dependency o (stamp-of-objc objc)
	   ;;(stamp-of-data source))))
	   ;; (oids-of-ioids-term prop))))
		      

	   (setf event-dependencies
		 (event-dependencies-collected
		  (new-event-description *system* *version* *dependency-event-tags*)
		  (transaction-stamp))))))

      ;;(format t "objc_translate2")

      (advance-sequence)
      (let ((t-descr (if o-desc
			 (new-event-description (system-of-description o-desc)
						(version-of-description o-desc)
						(cons 'translate (purposes-of-description o-desc)))
			 (new-event-description *system* *version* '(translate))))
	    (s-descr (when substance
		       (description-of-event-dependencies (dependencies-of-substance substance))))

	    ;; susbstance dependencies are the translate event. ???
	    ;;(find-first #'(lambda (event-dependencies)
	    ;;(when (member 'translate
	    ;;(purposes-of-description
	    ;;(description-of-event-dependencies event-dependencies)))
	    ;;event-dependencies))
	    ;;((dependencies-of-substance substance)))
	    )

	;;(setf -ed event-dependencies -substance substance -s-descr s-descr -t-descr t-descr -t-data-p t-data-p) (break "ot")
	;; (view-show (event-dependencies-to-term -ed))
      
	(cond
	  ((and (not force)
		substance
		(description-match s-descr t-descr)
		(null t-data-p))

	   ;;(format t "objc_translate3")
	   ;; here is where we share substances.
	   (set-objc-translation (dependency-store
				  (list* event-dependencies
					 (dependencies-of-substance substance)
					 (static-dependencies-of-substance substance)
					 (objc-other-translation-dependencies new-objc)))
				 new-objc))


	  ((and (null reduced) 
		(not (or (eql 'prf (kind-of-objc objc))
			 (eql 'stm (kind-of-objc objc)))))
	   ;;(setf -objc objc -source source)(break "otsrn") 
	   (raise-error '(error-message '(objc translate source term not))))


	  (t (with-dependency-event (nil t-descr)
	       (with-dependencies 
		   (let ((substance
			  (with-dependency-environment
			      ;;(break "translate")
			    (case (kind-of-objc objc)
			      (abs (objc-abs-translate source reduced o-desc))
			      (inf (objc-inf-translate source reduced o-desc t-data-p))
			      (prf (objc-prf-translate source reduced o-desc t-data-p))
			      (stm (objc-stm-translate
				    source
				    (or reduced
					(progn (set-source-term (iplaceholder-term) source)
					       (iplaceholder-term)
					       ;;(set-source-term (itext-term "lori") source)
					       ;;(itext-term "lori")

					       ))
				    o-desc))
			      (com (objc-com-translate source reduced o-desc))
			      (rule (objc-rule-translate source reduced o-desc))
			      (disp (objc-disp-translate source reduced o-desc))
			      (prec (objc-prec-translate source reduced o-desc))
			      (code (objc-code-translate source reduced o-desc))
			      (term (objc-term-translate source reduced o-desc))
			      (otherwise (raise-error (error-message '(objc translate kind) (kind-of-objc objc))))
			      ))))
		      

		     ;; desire translate make dependencies on static abstraction occurences.
		     ;; this requires translation by component matching description.
		     (unless (member (kind-of-objc objc) '(com term))
		       ;; ??? may not be best idea to restrict types, mainly a performace issue
		       ;;     with TERM kind.
		       ;; need to eval in scope of reference environment!.
		       (when reduced
			 ;;(setf -re re) (break "aus")
			 (note-static-abstraction-references o-desc re reduced)))

		     ;; not certain this is desired. maybe want collected to go to translation
		     ;; and let sub translate routines set dependencies of substance.
		     ;; translation looses event tag of dependencies, so want all tracked
		     ;; elsewhere and translation to be summary.
		     (set-substance-dependencies (event-dependencies-collected *dependency-event-description*
									       (transaction-stamp))
						 substance)
		 
		     ;;(setf a substance) (break "foo")
		     ;;(format t "objc_translate4")
		  
		     (set-objc-translation (dependency-store
					    (list* event-dependencies
						   (dependencies-of-substance substance)
						   ;; static could get expensive if frequently modifying
						   ;; some long list of objects (like a dir) but should be
						   ;; collected quickly too.
						   (static-dependencies-of-substance substance)
						   (objc-other-translation-dependencies new-objc)))
					   new-objc)

		     ;;(format t "objc_translate5")

		     ;;(setf a substance b new-objc) (break "foo2")
		     (set-objc-substance substance new-objc))))))))
    new-objc))


;; if translation is not required then means that we have not
;; changed source in any way which would change substance if
;; retranslated.

;;;;	
;;;;	twould be nice to only call translate if really needs translation since
;;;;	must first deactivate object before translating :
;;;;	  - why : to prevent orphaned substances ie no way to deact an orphaned substance
;;;;	  - checked : not that I could find. When activated need to flag objc then have translate
;;;;		check and fail. TODO !!!
;;;;	
;;;;	
;;;;	may still desire translation to redo source reduction if source dependencies touched.
;;;;	
;;;;	some mods do not nullify translation. This function allows translation to be a???
;;;;	purpose of this function is to detect if translation would occur without actually
;;;;	invoking translation. This prevents needless unbinding  and rebinding after non-substantive
;;;;	mods.
;;;;	
;;;;	
;;;;	

(defun translation-required-p (objc)
  ;;(break "force translate?")
  (let ((source (source-of-objc objc))
	(substance (substance-of-objc objc t)))

    ;;(break "afji")

    ;; TODO check stale dependencies.
    (with-handle-error-and-message (nil #'(lambda (m) (declare (ignore m)) ;;(break "hello")
						  t))
      (or (null (translation-of-objc objc))
	  (and (translation-required source substance
				     (let ((term (term-of-source source)))
				       (when term
					 (setf -source (source-reduce term
								      (cons (kind-of-objc objc)
									    (cons `LIB
										  (reduce-property-of-objc objc))))))))
	       t)))))


(defunml (|translation_required_p| (oc))
    (object_contents -> bool)
  (translation-required-p oc))
			       
  
(defunml (|term_to_objc| (term))
    (term -> object_contents)
  (term-to-objc term))

(defunml (|objc_contents| (kind))
    (token -> object_contents)
  (object-contents kind nil))

(defunml (|objc_kind| (objc))
    (object_contents -> tok)
  (kind-of-objc objc))

(defvar temp nil)

(defunml (|set_oid_temp| (oid))
    (oid -> unit)
  (setf temp oid))

(defunml (|objc_source| (objc))
    (object_contents -> term)
  (let ((term (term-of-source (source-of-objc objc))))
    (or term
	(raise-error (error-message '(objc source term not))))))

(defunml (|objc_substance_term| (objc))
    (object_contents -> term)
  (let ((term (term-of-substance (substance-of-objc objc))))
    (or term
	(raise-error (error-message '(objc substance term not))))))

(defunml (|objc_reduced_source| (objc))
    (object_contents -> term)
  (let ((src (term-of-source (source-of-objc objc))))
    (if src
	(source-reduce (term-of-source (source-of-objc objc))
		       (cons (kind-of-objc objc) (cons 'LIB (reduce-property-of-objc objc))))
	(raise-error (error-message '(objc reduced source not))))))

(defunml (|objc_modify_source| (objc term))
    (object_contents -> (term -> object_contents))
  (objc-modify-source-term objc term))



;;;;	
;;;;	Desire ability to browse database.
;;;;	
;;;;	  dbread	: term{link} -> term
;;;;	
;;;;	
;;;;	Desire weak links in database.
;;;;	  - a database link is one of
;;;;	      * !data_persist{<type>:t)(<stamp>)
;;;;	      * !data_persist_inline{<type>:t}(<stamp>; <term>)
;;;;	  - a weak link does not preclude collection of data referenced.
;;;;	  - can make a weak link by quoting or exploding a link.
;;;;	  
;;;;	  blot		: term{link} -> term{blot}
;;;;	  unblot	: term{blot} -> term{link}
;;;;
;;;;	
;;;;	Could allow linking and recovery of any datatype but need specific
;;;;	functions for each for ML type system:
;;;;	
;;;;	  recover_objc	: term{link} -> object_contents
;;;;	  link_of_objc	: object_contents -> term{link} 
;;;;	
;;;;	

(defunml (|blot| (term))
    (term -> term)
  (format t "~%blot")
  (opquote-term term 'dblink))

(defunml (|objc_to_term| (objc))
    (object_contents -> term)
  (data-export objc (ivoid-term)))


(defunml (|dbread| (term))
    (term -> term)

  (term-of-idata-persist-term (un-opquote-term term '(dblink))))

(defunml (|library_object_term| (oid))
    (object_id -> term)
  (let ((lobj (library-lookup oid)))
    (icons-term (dependency-to-term (library-object-dependency lobj))
		(persist-data (objc-of-library-object lobj)))))

;; creates a reference to an objc which the garbage collector can not see.
;; if not persistent on disk then force inline! or clone ? weak_bind ??
;; weak bind sounds like the winner.
;; only reason not on disk is due to previous error, but need to
;; support.
(defunml (|link_of_objc| (objc))
    (object_contents -> term)

  ;;(setf -objc objc) (break "loo")
  (persist-data objc)

  #|
  ;; clones since db (rightly) refuses to write to other directories than our own.
  ;; need to call persist-data to insure not inlined data. could force all objc's to not be inline.
  ;; then just need to remember stamp.  this is a little lazier but may result in duplication.

  ;; Major PERF problems here. subsequent update of property results in write of object, and
  ;; persist data here results in write. Really hope for a lower cost method.
  ;; currently calling at activate time. Maybe if force objc to not inline and update
  ;; property implicit with source update would be more efficient.
  (let ((stamp (stamp-of-data objc)))
    (if (db-persistent-not-inline-p stamp 'objc)
	(stamp-to-term (stamp-of-data objc))
	(if (eql (process-id) (process-id-of-stamp stamp))
	    (stamp-of-idata-persist-term (persist-data objc))
	    (progn (format t "~%cloning to blot for history~%")
		   (stamp-of-idata-persist-term (persist-data (clone objc)))))))
  |#
  )


(defunml (|unblot_objc| (term))
    (term -> object_contents)

  ;; may fail due to subcomponents being collected?
  ;; if so then pass failure up. 
  (provide-data (data (term-to-stamp term) 'objc) 'objc))


;; history is data-persist term. (could allow inline).
(defunml (|recall_objc| (objc))
    (object_contents -> term)

  (let ((h (objc-history objc)))
    (or h
	(raise-error (error-message '(recall objc none))))))


(defunml (|objc_substance| (objc))
    (object_contents -> term)
   (term-of-substance (substance-of-objc objc)))



(defunml (|objc_dependencies_list| (tags objc))
    ((tok list) -> (object_contents -> ((object_id list) list)))

  (let ((depstore (dependencies-of-dependency-store (translation-of-objc-r objc))))
    (mapcar #'(lambda (tag)
		(find-first #'(lambda (deps)
				(when (eql (tag-of-dependencies deps) tag)
				  (mapcar #'oid-of-dependency
					  (list-of-dependencies deps))))
			    depstore))
	    tags)))

(defunml (|objc_dependencies_listx| (tag objc))
    (tok -> (object_contents -> (tok list)))

  (let ((depstore (dependencies-of-dependency-store (translation-of-objc-r objc))))
    ;;(setf -depstore depstore) (break "odlx")
    (mapcan #'(lambda (deps)
		(when (eql (tag-of-dependencies deps) tag)
		  (list tag)))
	    depstore)))


(defunml (|dependent_oids_of_objc| (tag objc))
    (tok -> (object_contents -> (object_id list)))

  (find-first #'(lambda (deps)
		  (when (eql (tag-of-dependencies deps) tag)
		    (mapcar #'oid-of-dependency
			    (list-of-dependencies deps))))
	      (dependencies-of-dependency-store (translation-of-objc-r objc))))

(defunml (|dependencies_size| (objc))
    (object_contents -> int)

  (term-op-count (dependency-store-to-term (translation-of-objc-r objc))))


(defunml (|objc_dependencies| (objc))
    (object_contents -> term)
  (event-dependencies-to-term (dependencies-of-substance (substance-of-objc objc t))))

(defunml (|objc_add_property| (objc tok term) :error-wrap-p t)
    (object_contents -> (tok -> (term -> object_contents)))

  (objc-add-property objc tok term))

(defunml (|objc_remove_property| (objc tok) :error-wrap-p t)
    (object_contents -> (tok -> object_contents))

  (objc-remove-property objc tok))

    
(defunml (|objc_property| (objc tok) :error-wrap-p t)
    (object_contents -> (tok -> term))

  (or (property-of-objc objc tok)
      (raise-error (error-message '(objc property tok not) tok))))



(defunml (|objc_get_properties| (objc) :error-wrap-p t)
    (object_contents -> ((tok |#| term) list))
  (format t "~%objc-get-properties")
  (source-properties-of-objc objc))

(defunml (|objc_set_properties| (objc properties) :error-wrap-p t)
    (object_contents -> (((tok |#| term) list) -> object_contents))

  (objc-set-properties objc properties))

(defunml (|objc_translated_p| (objc) :error-wrap-p t)
    (object_contents -> bool)

  (objc-translated-p objc))

;;;
;;;	INF
;;;

(defunml (|inf_objc_src_step| (objc))
    (object_contents -> inf_step)

  (require-objc-kind objc 'inf)
  (or (step-of-inf-objc objc)
      (raise-error (error-message '(step not)))))

(defunml (|inf_objc_substance_step| (objc))
    (object_contents -> inf_step)

  (require-objc-kind objc 'inf)
  (let ((s (substance-of-objc objc t)))
    (if (null s)
	(raise-error (error-message '(inf substance not)))
	(or (step-of-inf-substance s)
	    (raise-error (error-message '(step not)))))))

(defunml (|inf_objc_src_modify_step| (objc step))
    (object_contents -> (inf_step -> object_contents))

  (require-objc-kind objc 'inf)
  (inf-objc-modify-step objc step))

(defunml (|inf_objc_src_delete_step| (objc))
    (object_contents -> object_contents)

  (require-objc-kind objc 'inf)
  (inf-objc-modify-step objc nil))

(defunml (|inf_objc_src_modify_step_goal| (objc step goal))
    (object_contents -> (inf_step -> ((term |#| (term list)) -> object_contents)))

  (require-objc-kind objc 'inf)
  (inf-objc-modify-step-goal objc step goal))

(defunml (|inf_objc_src_modify_goal| (objc goal))
    (object_contents -> ((term |#| (term list)) -> object_contents))

  (require-objc-kind objc 'inf)
  (inf-objc-modify-src-goal objc goal))

(defunml (|objc_remove_substance| (objc))
    (object_contents -> object_contents)

  (let ((nobjc (clone objc)))
    (set-objc-substance nil nobjc)
    (set-objc-translation nil nobjc)
    nobjc))

;;;	
;;;	STM
;;;	


(defunml (|stm_objc_src_modify_proofs| (objc oids))
    (object_contents -> ((object_id list) -> object_contents))

  (require-objc-kind objc 'stm)
  
  (let ((source (clone (source-of-objc objc))))
    (set-statement-source-proofs oids source)

    (objc-modify-source objc source)))


(defunml (|stm_objc_src_proofs| (objc))
    (object_contents -> (object_id list))

  (require-objc-kind objc 'stm)

  (proofs-of-statement-source (source-of-objc objc)))


(defunml (|stm_objc_goal| (objc))
    (object_contents -> term)

  (require-objc-kind objc 'stm)

  (term-of-substance (substance-of-objc objc)))


(defunml (|term_objc_term| (objc))
    (object_contents -> term)

  (require-objc-kind objc 'term)

  (term-of-substance (substance-of-objc objc)))


(defunml (|stm_objc_proofs| (objc))
    (object_contents -> (object_id list))

  (require-objc-kind objc 'stm)

  (proofs-of-statement-substance (substance-of-objc objc)))

(defunml (|stm_objc_extract| (objc))
    (object_contents -> term)

  (require-objc-kind objc 'stm)

  (or (extract-of-statement-substance (substance-of-objc objc))
      (raise-error
       (error-message '(stm objc extract)))))



;;;	
;;;	PRF
;;;	

(defunml (|prf_objc_src_inf_tree| (objc))
    (object_contents -> inf_tree)

  (require-objc-kind objc 'prf)

  (inf-tree-of-proof-source-r (source-of-objc objc)))

(defunml (|prf_objc_src_inf_tree_proxied_p| (objc))
    (object_contents -> bool)

  (require-objc-kind objc 'prf)
  (complete-inf-tree-proxy-p (inf-tree-or-proxy-of-proof-source (source-of-objc objc))))

(defunml (|prf_objc_src_inf_tree_proxizeable_p| (objc))
    (object_contents -> bool)

  (require-objc-kind objc 'prf)
  ;;(setf -objc objc) (break "positpp")

  ;; don't know why this was here rle 3/2004
  ;;  - at dump time may want to proxize proofs.
  ;;  - if will proxize then do not want inf obids
  ;;    if proxiable is true then do not include inf obids.
  ;;    thus if *proxize-inf-tree-dynamically* is false than
  ;;      prf_objc_src_inf_tree_proxizeable_p is false and dump will include inf obids.
  (and ;; *proxize-inf-tree-dynamically* 
       (let* ((src (source-of-objc objc))
	      (itree (inf-tree-of-proof-source src)))
	 (when (inf-tree-p itree)
	   (when (null (xref-of-proof-source src)) (format t "~%no xref ~a" (name-of-objc objc)))
	   (let ((g (goal-of-inf-tree itree)))
	     (and (xref-of-proof-source src)
		  (not (exists-p #'(lambda (term) (term-walk-p term #'iplaceholder-term-p)) g))
		  (verify-proof itree g)))))))

(defunml (|prf_objc_src_inf_tree_eph| (objc))
    (object_contents -> inf_tree)

  (require-objc-kind objc 'prf)

  (inf-tree-of-proof-source-r (source-of-objc objc) t))

(defunml (|prf_objc_src_inf_tree_term| (objc))
    (object_contents -> term)

  (require-objc-kind objc 'prf)

  (inf-tree-term-of-proof-source (source-of-objc objc)))


(defunml (|prf_objc_src_import_inf_tree| (objc itree))
    (object_contents -> (term -> object_contents))
  
  (require-objc-kind objc 'prf)

  (prf-objc-modify-inf-tree objc (iinf-tree-term-to-inf-tree itree nil)))


  



(defunml (|prf_objc_src_modify_inf_tree| (objc itree))
    (object_contents -> (inf_tree -> object_contents))
  
  ;;(setf -objc objc -itree itree) (break "posmit")

  (require-objc-kind objc 'prf)
  (prf-objc-modify-inf-tree objc itree))
  

(defunml (|prf_objc_src_delete_inf_tree| (objc))
    (object_contents -> object_contents)
  
  (require-objc-kind objc 'prf)

  (prf-objc-delete-inf-tree objc))
  

(defunml (|prf_objc_goal| (objc))
    (object_contents -> term)
  
  (require-objc-kind objc 'prf)

  (sequent-of-goal (goal-of-prf-objc-r objc)))


(defunml (|prf_objc_extract| (objc))
    (object_contents -> term)
  
  (require-objc-kind objc 'prf)

  (extract-of-prf-objc-r objc))

  


;;;	
;;;	DESTRUCTIVE to input list
;;;	
;;;	Input : (<oid>  . <objc>) list
;;;	
;;;	Remainder : ((<oid> . <dependency> list) . <objc>) list
;;;	  lal: rle I think you mean Remainder : ((<dependency> list) . (<oid . <objc>)) list
;;;	
;;;	
;;;	Kludge alert :
;;;	  file-proxy property 
;;;	
;;;	
;;;	
;;;	
;;;	
;;;	
;;;	
;;;	

(defun theories-objc-order (objcs)
  ;; try ignoring theory order 
  #+cmu objcs ;; i think cmu's problem is with the macro with-ignore below
  #-cmu 
  (if t 
      objcs
    (let ((avoidoids (ml-text "([descendent_s ``system mill Examples``] ? nil)")))
	(let ((toid (dag-root '|theories|)))
	  (if (null toid)
	      (values objcs nil)
	      (let ((oids (subtree-oids toid #'(lambda (oid)
						 (member oid avoidoids :test #'equal-oids-p)))))
		(let ((ohash (make-hash-table)))
		  (dotimeslist (i o oids)
			       (with-ignore (setf (gethash (oc o) ohash) i)))
	    
		  (let ((m nil)
			(n nil))
		    (dolist (oc objcs)
		      (if (gethash (cdr oc) ohash)
			  (push oc m)
			  (push oc n)))
	
		    (values (reverse n)
			    (sort m #'(lambda (a b)
					(< (gethash (cdr a) ohash) (gethash (cdr b) ohash)))))))))))))
	
(defvar *n-objc-sort-remainder* nil)


;;;;	
;;;;	we want any object in tree of refenv (ie objects listed in a refenv field) to be a dependency
;;;;	of code object defining refenv or object with explicit refenv property
;;;;	if object in tree of refenv is a dir then depending on dir should be sufficient.
;;;;	
;;;;	but do not want to willy nilly add dirs to graph determining code order as
;;;;	there may be static refs to dirs which are not meant to affect order.
;;;;	
;;;;	code may ref the dir containing code
;;;;	eg RE_final proper-re declares theory dir as index and thus refs dir which contains it.


;;;;	sorts on compile dependencies.
;;;;	  - need more robust sorting.
;;;;	  - refenvs need static + proxy sort
;;;;	  - static sort in general may be problem if code object contains
;;;;	    static link to dir containing other not necessarily earlier code.
;;;;	
;;;;   start with code and produce code graph.
;;;;   find proxies then reduce,
;;;    then sort.
;; to graph ref-renvs need proxies?
;; can only find proxy by looking at static deps of refenv


(defun flatten-some-dependencies (tags oid store)
  (mapcan #'(lambda (tag)
	      (find-first #'(lambda (deps)
			      (when (eql (tag-of-dependencies deps) tag)
				;;(format t "~a~%" tag)
				;; need to copy anyways for outer mapcan
				(mapcan #'(lambda (d)
					    (let ((doid (oid-of-dependency d)))
					      (unless (equal-oids-p oid doid)
						(list doid))))
					(list-of-dependencies deps))))
			  (dependencies-of-dependency-store store)))
	  tags))


(defun objc-graph-by-dependencies (depprox objcs)

  (let ((tags (car depprox))
	(ptags (cdr depprox)))
    (setf *n-objc-sort-remainder* nil)
    (let ((fp-objcs (sort objcs
			  #'(lambda (a b)
			      (declare (ignore b))
			      (and (property-of-objc (cdr a) 'file-proxy)
				   t)))))

      ;;(setf -fp-objcs fp-objcs) (break "nosbd")

      (graph-reduce-proxy
       #'equal-oids-p
       (mapcar #'(lambda (i)
		   (let ((oid (car i)))
		     (with-translation-of-objc (trans (cdr i))
		       (let ((dependencies (flatten-some-dependencies tags oid trans))
			     (proxies (flatten-some-dependencies ptags oid trans)))
			 (cons (cons oid proxies) dependencies)))))
	       fp-objcs)))))

(defunml (|code_order_dependencies| (oid))
    (object_id -> ((object_id list) |#| (object_id list)))

  ;;(setf -oid oid) (break "cod")
  (let ((depprox *code-order-sort-dependency-kinds*))
    (let ((tags (car depprox))
	  (ptags (cdr depprox)))

      (let ((o (library-lookup oid)))
	(with-objc-of-library-object (objc o)
	  (with-translation-of-objc (trans objc)
	    (let ((dependencies (flatten-some-dependencies tags oid trans))
		  (proxies (flatten-some-dependencies ptags oid trans)))
	      (cons (cons oid proxies) dependencies)))))
      )))
  

(defun n-objc-closure (depprox aobjcs seeds)
  (let ((g (objc-graph-by-dependencies depprox aobjcs)))
    (graph-sort g (graph-closure g nil seeds))))


;; would like to include refenv data in dependencies
;; so as to avoid having to declare refenv dependencies
;;  - easy for 
;;  - flatten refenv trees in lib at translate/ ie substantive properties processed?


;; tok list # tok list -> (oid # objc) list -> (oid # objc) list
(defun n-objc-sort-by-dependencies (depprox aobjcs)

  (setf *n-objc-sort-remainder* nil)
  (mlet* (((objcs theories) (theories-objc-order aobjcs)))

	 (let ((g (objc-graph-by-dependencies depprox objcs)))
	   ;;(setf -objcs objcs -theories theories -g g) (break "nosbd")

	   (setf -g g)
	   (mlet* (((layers cycles) (graph-layers g)))

		  (let ((h (new-oid-table)))

		    (dolist (aobjc objcs)
			    (hashoid-set h (car aobjc) (cdr aobjc)))

		    (let ((acc (mapcar #'(lambda (oid)
					   (cons oid (hashoid-get h oid)))
				       cycles)))

		      (dolist (l (reverse layers))
			      (dolist (oid l)
				      (push (cons oid (hashoid-get h oid))
					    acc)))

		      (when t
			(setf -acc acc -layers layers -cycles cycles
			      -l (mapcan #'(lambda (l) (copy-list l))
					 -layers))
			(format t "#layers  ~a #layered  ~a #cycles ~a~%" (length layers) (length -l) (length cycles))
			(dolist (l layers)
				(format t " ~a " (length l)))
			(terpri)
			;;(break)
			)

		      acc))))))


(define-primitive |!oid_cons| () (car cdr))

;;  LISP[(lib)]> (show-cut-list 'rich (mapcar #'car (code-order (term-to-description (ml-text  "nuprl5_refiner_description_term")))))
;;  LISP[(lib)]> (show-cut-list 'rich (mapcar #'car (code-order (term-to-description (ml-text  "nuprl5_edit_description_term")))))

;;  LISP[(lib)]> (show-cut-list 'rich (graph-cycle -g));;
;;  (show-code-order-remainder 'rich);;

(defunml (|duplicate_named_oids| (desc oids))
    (term -> ((object_id list)  -> (object_id list)))
  
  (without-dependencies
   (let ((consumer-descr (term-to-description desc)))
     (let ((acc nil))


       (let ((subtree-names nil)
	     )

	 ;;(break "dsoo")

	 (mapc #'(lambda (oid)
		   (with-ignore
		       (let ((o (library-lookup oid)))
			 ;;(setf -c o -e oid) ;;(break "lpbt")
			 (let* ((objc (oc oid)))
	    
			   (let ((desc (description-term-of-objc objc)))
			     ;;(setf -desc desc -cdesc consumer-descr -objc objc -oid oid)
			     (when (and (library-object-active-p o)
					(not (member (kind-of-objc objc) '(prf inf)))
					(match-descriptions-p desc consumer-descr))
			       (let ((n (name-of-oid oid)))
				 (push n subtree-names))))))))
    
	       oids)
	 ;;(break "dsoo")

	 (definition-table-map (resource 'library) (current-transaction-stamp)
	   #'(lambda (oid o)
	       ;;(setf c o e oid) (break "lpbt")
	       (let* ((objc (objc-of-library-object o)))
	    
		 (let ((desc (description-term-of-objc objc)))
		   (when (and (library-object-active-p o)
			      (not (member (kind-of-objc objc) '(prf infs)))
			      (match-descriptions-p desc consumer-descr))
		  
		     (let ((n (name-of-oid oid)))
		       (when (and (member n subtree-names)
				  (not (member oid oids :test #'equal-oids-p)))
			 (push oid acc))))))))

	 ;;(setf -acc acc) (break "dso")
	 acc)))))


(defunml (|bad_code_oids| (unit))
    (unit -> (object_id list))
  
   (without-dependencies
    (let ((consumer-descr (term-to-description (ml-text  "nuprl5_refiner_description_term"))))
    (let ((acc nil)
	  (bad nil))

      (let ((subtree-names nil)
	    (oids (subtree-oids (ml-text "descendent_s ``theories markb``"))))

	(progn (mapc #'(lambda (oid)
		  (let ((o (library-lookup oid)))
		    ;;(setf c o e oid) (break "lpbt")
		    (let* ((objc (oc oid)))
	    
		      (let ((desc (description-term-of-objc objc)))
			(when (and (library-object-active-p o)
				   (eql 'code (kind-of-objc objc))
				   (match-descriptions-p desc consumer-descr))
			  (let ((n (name-of-oid oid)))
			    (push n subtree-names)))))))
    
	      oids)

	(definition-table-map (resource 'library) (current-transaction-stamp)
	  #'(lambda (oid o)
	      ;;(setf c o e oid) (break "lpbt")
	      (let* ((objc (objc-of-library-object o)))
	    
		(let ((desc (description-term-of-objc objc)))
		  (when (and (library-object-active-p o)
			     (eql 'code (kind-of-objc objc))
			     (match-descriptions-p desc consumer-descr))
		  
		    (let ((n (name-of-oid oid)))
		      (when (and (member n subtree-names)
				 (not (member oid oids :test #'equal-oids-p)))
			(push oid bad))))))))
	bad))))))


(defun show-cut-list (who l)
  (let ((i 0))
    (view-show
     (map-list-to-ilist l
			(instantiate-term (instantiate-operator '|!cut_cons| nil))
			;;(instantiate-term (instantiate-operator '|oid_cons| nil))
			#'(lambda (o)
			    (instantiate-term
			     (instantiate-operator '|index|
						   (list (natural-parameter (incf i))))
			     (list (instantiate-bound-term (ioid-term o))))))
     who)))


(defun code-order (consumer-descr)

  (let ((code-acc nil))

    (definition-table-map (resource 'library) (current-transaction-stamp)
      #'(lambda (oid o)
	  ;;(setf c o e oid) (break "lpbt")
	    
	  (when (library-object-active-p o)
	    (with-objc-of-library-object (objc o)
	      (let ((desc (description-term-of-objc objc)))
		(when (or (and (eql 'code (kind-of-objc objc))
			       (match-descriptions-p desc consumer-descr))
			  ;(dag-directory-p oid)
			  )
		  (push (cons oid objc) code-acc)  ))))))

    ;;(setf -code-acc code-acc)
    (n-objc-sort-by-dependencies *code-order-sort-dependency-kinds* code-acc)
    ))

(defunml (|code_order| (descterm))
    (term -> (object_id list))
  (mapcar #'car (code-order (term-to-description descterm))))
    

;; create terms without io to gather stats
;; (lib-producer-ghost-bound-terms '(ostates) (term-to-description (ml-text "nuprl5_edit_description_term")))
(defun lib-producer-ghost-bound-terms (table-types consumer-descr)


  (let ((least-max 0)
	(terms nil))

    (dotimes (i 10) (push (cons 0 (ivoid-term)) terms))

    (labels ((max-check (term)
	       (let ((c (term-op-count term))) 
		 (when (> c least-max)
		   (let ((foundp nil)
			 (new-least-max c))
		     ;;(setf -c c -term term -least-max least-max -terms terms)  (break "lpgbti")
		     (setf terms (mapcar #'(lambda (a)
					     (if (and (not foundp) (eql (car a) least-max))
						 (progn
						   (setf foundp t)
						   (cons c term))
						 (progn
						   (when (< (car a) new-least-max)
						     (setf new-least-max (car a)))
						   a)))
					 terms))
		     (setf least-max new-least-max))))))

      (let ((*produce-max-check* #'max-check))
	(lib-producer-bound-terms table-types consumer-descr
				  #'(lambda (x) (max-check (cdr x)))))
								      

      (break "lbgbt")
      (setf -max least-max -terms terms)
      (break "lbgbt2")
      nil
      )))
  

(defun lib-producer-bound-terms (table-types c-descr acc-f &optional oids synchro)

  ;;(setf -c-descr c-descr) (break "lpbt")
  
  (let* ((db-buffering-p (when *io-db-buffering*
			   (stop-db-buffering)
			   t))

	 (lib (environment-resource 'library))
	 (stamp (table-stamp-term-of-definition-table lib))
	 (attr-assoc nil)
	 (consumer-descr (if (description-p c-descr)
			     c-descr
			     (if (idescription-term-p c-descr)
				 (term-to-description c-descr)
				 (raise-error (error-message '(lib-producer-bound-terms consumer description) c-descr)))))
	 (synchro-stamp synchro)
	 (checkpoint (when synchro-stamp
		       (find-first #'(lambda (oid-pe)
				       (let ((pe (cdr oid-pe)))
					(when (and (ievent-checkpoint-term-p pe)
						   (compare-terms-p synchro-stamp
								    (event-of-ievent-checkpoint-term pe)))
					  pe)))
				   (persistent-events))))
	 (log-checkpoints (when checkpoint
			    (map-ilist-to-list 
			     (logs-of-ievent-checkpoint-term checkpoint)
			     (icons-op)
			     #'(lambda (lchk)
				 (cons (kind-of-ilog-checkpoint-term lchk)
				       (stamp-of-ilog-checkpoint-term lchk))))))
	 
	 (touched-oids (when synchro-stamp
			 (commit-event-touched (term-to-stamp synchro-stamp))))
	 (codeg nil))

    ;;(setf -synchro-stamp synchro-stamp -checkpoint checkpoint -log-checkpoints log-checkpoints) (break "fuf")

    (when (and touched-oids oids)
      ;; not expecting this yet maybe when quickstarting libs.
      (break "lpbt touched"))

    (setf oids touched-oids)
	  
    (without-dependencies;;with-default-dependencies ('broadcast t)

     ;; fttb: def could have position property used to sort for activate
     ;; need some order for ml objects.
       (let ((library-p (and (member 'library table-types) t))
	     (code-acc nil))

	 (let ((accumulate-states nil))
	    
	   

	   (labels
	    ((accumulate (term)
			 ;;(setf b term) (break "accum")
			 (funcall acc-f (instantiate-bound-term term)))

	     (accumulate-log (kind log)
			     (let ((ckp (cdr (assoc kind log-checkpoints))))	      
			       (push (instantiate-bound-term (idefinition-term (ivoid-term)
									       (if ckp
										   (ilog-checkpointed-term ckp log)
										 log)))
				     (cdr (assoc kind attr-assoc)))))

	     (mk-pass (table-type desc oid objc)
		      (ipassport-term table-type
				      stamp
				      desc
				      (idefinition-insert-term
				       (current-sequence)
				       (let ((data (objc-substance objc)))
					 (idefinition-term 
					  (idependency-term oid
							    (stamp-to-term (stamp-of-data objc))
							    (stamp-to-term (stamp-of-data data)))
					  (persist-data data nil t))))))

	     (all-code (oid o)
		       (when (library-object-active-p o)

			 ;; insert
			 (let* ((objc (objc-of-library-object o))
				(table-type (map-objc-to-table-type objc)))

			   ;;(setf b consumer-descr c table-type d table-types) (break "pbt4")
			   (when (member table-type table-types)
			     (let ((desc (description-property-term-of-objc objc)))
			       (when (match-descriptions-p desc consumer-descr)
				 (when (eql 'code (kind-of-objc objc))
				   (cons oid objc))))))))

			
	     (visit (oid o)
		     
		    (when accumulate-states
		      (accumulate-object-attrs accumulate-states oid o))

		    ;; perf todo : combining bind and active in single msg would save at other end.
		    (when library-p
		      (advance-sequence)
		      ;;(when -loabl (setf -o o) (break "pbt3"))
		      (accumulate 
			
		       (ipassport-term 'library
				       stamp
				       *lib-description*
				       (idefinition-insert-term
					(current-sequence)
					(idefinition-term (dependency-to-term
							   (dependency-of-definition o))
							  (persist-data (library-object-objc o) nil t)
							  ;;(persist-data (objc-of-library-object o) nil t)
							  ))))
		      
		      (unless (library-object-collectable-p o)
			(advance-sequence)
			(accumulate
			 (ipassport-term 'library
					 stamp
					 *lib-description*
					 (idefinition-disallow-collection-term (current-sequence) oid)))))
		     
		    (when (library-object-active-p o)
		      (when (member 'library table-types)
			(advance-sequence)
			(accumulate
			 (ipassport-term 'library
					 stamp
					 *lib-description*
					 (idefinition-activate-term (current-sequence) oid))))
		 
		      ;; insert
		      (let* ((objc (objc-of-library-object o))
			     (table-type (map-objc-to-table-type objc)))

			;;(setf b consumer-descr c table-type d table-types) (break "pbt4")
			(when (member table-type table-types)
			  (let ((desc (description-property-term-of-objc objc)))
			    ;;(setf a desc b consumer-descr c table-type)
			    ;;(when (member table-type '(abstractions code)) (break "pbt5"))
			    ;;(when -loab (setf -o o -objc objc -table-type table-type -consumer-descr  consumer-descr) (break "pbt6"))
			    (when (match-descriptions-p desc consumer-descr)

			      (if (eql 'code (kind-of-objc objc))
				  (push (cons oid objc) code-acc)

				(progn
				  (advance-sequence)

				  (accumulate
				   (mk-pass table-type desc oid objc))))))))))
	     )


	    (mapc #'(lambda (kind)
		      (when (member kind table-types)
			(setf attr-assoc (acons kind nil attr-assoc))
			(let ((aahook (produce-object-attr kind #'accumulate-log)))
			  (when aahook
			    (push (cons kind aahook) accumulate-states)))))
		  *object-attr-table-types*)		   

	   ;;(setf -attr-assoc attr-assoc -accumulate-states accumulate-states) (break "pbt1")

	   ;;collect binds then activates?
	   (definition-table-map lib (current-transaction-stamp)
	     #'(lambda (oid o)
		 ;;(setf c o e oid) (break "lpbt2")
	      
		 (with-handle-error-and-message
		     (('(lib producer))
		      #'(lambda (msg)
			  (setf -m msg);;(break)
			  (message-emit msg)
			  nil))
		  
		     (cond
		      ((not (null oids))
		       (let ((ag (all-code oid o)))
			 (when ag 
			   (push ag codeg)))
		       (when (hashoid-get touched-oids oid)
			 (visit oid o)))
		       
		      (t (visit oid o))))))

	   (when library-p
	     ;; kludge to keep ostate table log in env.
	     (let ((lib-sub (sub-of-environment (current-environment))))
	       (dolist (a (library-sub-environment-table-logs lib-sub))
		 ;;(setf -a a) (break "otl")
		 (accumulate
		  (ienvironment-state-term (or (term-of-table-log (cdr a))
					       (itable-log-term (car a) 0 (stamp-of-table-log (cdr a)))))))))

	   (when code-acc
	     (if checkpoint
		 (progn
		   ;;(setf -code-acc code-acc -codeg codeg)
		   ;;(setf -codeo (icode-order-term (icut-oid-list (n-objc-closure *code-order-sort-dependency-kinds* codeg (mapcar #'car code-acc)))))
		   ;;(break "fufu")
		   (dolist (code code-acc)
		     (let ((oid (car code))
			   (objc (cdr code)))
		    
		       (advance-sequence)
		       (accumulate
			(mk-pass (map-objc-to-table-type objc)
				 (description-property-term-of-objc objc)
				 oid objc))))

		   (accumulate
		    (ipassport-term 'code
				    stamp
				    (description-to-term consumer-descr)
				    (icode-order-term
				     (icut-oid-list
				      (n-objc-closure *code-order-sort-dependency-kinds*
						      codeg
						      (mapcar #'car code-acc)))))))

	       (let ((sorted (n-objc-sort-by-dependencies *code-order-sort-dependency-kinds* code-acc)))

	       (dolist (code sorted)
		 (let ((oid (car code))
		       (objc (cdr code)))
		    
		   (advance-sequence)
		   (accumulate
		    (mk-pass  (map-objc-to-table-type objc)
			      (description-property-term-of-objc objc)
			      oid objc)))))))


	   (setf -attr-assoc attr-assoc) 
	   ;;(break "pbte7")

	   (dolist (a attr-assoc)
	     (when (cdr a)
	       (advance-sequence)
	       (accumulate
		(ipassport-term (car a)
				(table-stamp-term-of-definition-table (resource 'library))
				(description-to-term (new-description (list (car a))))
				(idefinition-replace-term (current-sequence)
							  (instantiate-term *object-attr-states-op* (cdr a)))))))
	   ))))

    (when db-buffering-p
      (start-db-buffering))))




(defunml (|objc_translate| (objc))
    (object_contents -> object_contents) 
 (objc-translate objc t))
			   
(defunml (|objc_translate_force| (objc))
    (object_contents -> object_contents) 
 (objc-translate objc t))
			   
(defunml (|objc_translated_p| (objc))
    (object_contents -> bool)
  
  ;;(break)
  ;; should check for 
  (and (translation-of-objc objc) t))

(defunml (|objc_translation| (oc))
    (object_contents -> term)

  ;;(setf -oc oc) (break "ot")
  (translation-term-of-objc-r oc))


;; both must have substances and both substances must have same stamp.
;; both must be translated?


(defun objc-similar-p (objca objcb)

  ;; weak mods leave objc translated after mod otherwise mod results in untranslated objc.

  (and (objc-translated-p objca)
       (objc-translated-p objcb)

       (not (translation-required-p objca))
       (not (translation-required-p objcb))

       (let ((sa (substance-of-objc objca t))
	     (sb (substance-of-objc objcb t)))

	 (unless (and sa sb)
	   ;; this shouldn't be possible as we would expect a translated objc to have a substance.
	   (system-error (error-message '(substance not))))
    
	 (equal-stamps-p (stamp-of-data sa)
			 (stamp-of-data sb)))))


(defunml (|objc_similar_p| (objca objcb))
    (object_contents -> (object_contents -> bool))

  (unless (and (objc-translated-p objca)
	       (objc-translated-p objcb))
    (raise-error (error-message '(translated not))))

  
  (objc-similar-p objca objcb))


;; need to allow have non-substantive prop updates with new stamps but all else the same.
;; no translation-required and same substances should be sufficient
(defunml (|objc_very_similar_p| (objca objcb))
    (object_contents -> (object_contents -> bool))

  ;; same objc stamps
  (very-similar-objcs objca objcb))


(defun active-orphaned (kind)
  (let ((lib (resource 'library))
	(acc nil)
	(other 0))

    (definition-table-map lib (current-transaction-stamp)
      #'(lambda (oid o)
	  (when (library-object-active-p o)
	    (let ((objc (objc-of-library-object o)))
	      (when (eql kind (kind-of-objc objc))
		(if (null (oid-find-paths oid))
		    (push oid acc)
		    (incf other)
		  ;(setf -oid oid -o o -objc objc) (break "aoa")
		  ))))))

    ;;(setf -acc acc)
    (cons other acc)))

(defunml (|active_orphaned| (kind))
    (tok -> (int |#| (object_id list)))

  (active-orphaned kind))
