%
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************
%


%
;;;;	
;;;;	
;;;;	tactic	: !tactic(!text{<tac>:s}) | void
;;;;	
;;;;	<sequent>	: !inf_sequent{<hidden>:bool}(<type>; <id>.<sequent>)
;;;;	
;;;;	msequent -> concl + assumptions
;;;;	leave hidden false and id empty ie `|| 
;;;;	 <type> will hold an assumption, <sequent> is rest of sequent.
;;;;	 last <sequent> will be  concl.
;;;;	 eg !inf_sequent{nil:b}(<assum1>; .!inf_sequent{nil:b}(<assum2>; .<concl>)
;;;;	
;;;;	

;;;;
;;;;	libmp_open_proof: downloads entire proof tree - creates proof.
;;;;	import_theorems	: downloads top level goal - creates lemma.
;;;;
;;;;	
;;;;	
;;;;	implemented:
;;;;	
;;;;	SYMBOLICADDRESS	: !token{t} !tok_cons list.
;;;;	
;;;;	libmp_import_root	: unit -> unit
;;;;	  * imports all the root modules (just gets all module names, not their contents)
;;;;	libmp_import_theorems	: (object_id -> unit)
;;;;	  * given a module oid from root command above, 
;;;;        import its contents (statements and abs, no proofs)
;;;;	libmp_import_proofs	: (object_id -> unit)
;;;;	  * given a module oid from root command above, 
;;;;        import its contents (statements and abs and proofs)
;;;;	
;;;;	libmp_open_proof	: (object_id {STM} -> object_id{PRF})
;;;;      * called by stm_to_prf, will import proof if not already imported.
;;;;	libmp_close_proof	: (object_id {STM} -> unit)
;;;;	  * not called, but could add explicit call to navigator/editor.
;;;;	
;;;;	uses:
;;;;	
;;;;	mp_list_root_module	: unit -> tok list
;;;;	mp_list_module		: tok list -> (tok list) list {dependencies} # (tok # term) list {thms}
;;;;	mp_thm_set		: tok list -> term {sequent} -> unit
;;;;	  * called implicitly by save.
;;;;	mp_lookup_proof		: tok list -> term {itree}
;;;;	
;;;;	mp_undo?
;;;;
;;;;	mp_list_display		: (tok list) list -> (prec? # (tok # term) list) list
;;;;	
;;;;	
;;;;	
%



%
metaprl stubs for testing
%

let iml_op = (`!ML`, [false_bool_parameter; true_bool_parameter]);;
let iml_expression_term cmd args = 
  make_term (`!expression`,nil)
    [[], make_term iml_op
          (([],cmd) . (map (\t. ([],t)) args))
    ]
;;


% fns for metaprl refinements %

let symbolic_address_of_object oid = 
 map_isexpr_to_list icons_op token_of_itoken_term (get_property oid `SYMBOLICADDRESS`)
;;


% commands for creating metaprl objects %
					       
let mp_create_stm oid name = 
  let saddr = symbolic_address_of_object oid in
  let mname = hd (tl (rev saddr)) in
  mp_eval_command (mp_command `!mp_create` [(itoken_term mname);(itoken_term name)]);
  ()
;;
						       
let mp_create_rw oid name = 
  let saddr = symbolic_address_of_object oid in
  let mname = hd (tl (rev saddr)) in
  mp_eval_command (mp_command `!mp_create_rw` [(itoken_term mname);(itoken_term name)]);

  %let coid = create_with_term `ABS` (imp_msequent_term imp_nil_term (placeholder_term "goal"))
		        [`SYMBOLICADDRESS`, (symbolic_address_term saddr) 
			; `NAME`, (itoken_term name) 
			; `DESCRIPTION`, metaprl_refiner_description_term
			] in
			        
   save oid (dag_add_child oid name coid)
   ; activate oid
   ;% ()
;;

let libmp_import_root_aux doid l = 
  let saddr = symbolic_address_of_object doid in
  letref acc = nil in
  let aux name = 
    let coid = create_object_id () in
    (((lib_bind coid
      (objc_modify_source
 	(objc_set_properties (objc_contents `TERM`)
	  [ `SYMBOLICADDRESS`, symbolic_address_term (append saddr [name])
	  ; `NAME`, (itoken_term name)
          ; `DESCRIPTION`, nuprl5_library_description_term
	  ])
	(itoken_term name))) ());
     acc := ((name, coid) . acc);
     ()) in

    map aux l;
    dag_inserts doid acc 
;;

let mp_display_directory () = descendent_s ``theories metaprl Modules Display``;;

let libmp_display_loaded_p saddr_suffix = 
   %  let mod = descendent (root `Modules`) saddr_suffix in %
   can (descendent (mp_display_directory ())) saddr_suffix
;;


% assumes that loaded_p above is true %
let libmp_display_loaded saddr_suffix = 
   if (not (can (descendent (descendent (root `Modules`) saddr_suffix)) ``Display``))
    then dag_insert (descendent (root `Modules`) saddr_suffix)
		    `Display`
		    (descendent (mp_display_directory ()) saddr_suffix)
  ; ()
;;


let libmp_make_prec saddr_suffix name pterm = 
 lib_add
      (objc_modify_source
 	(objc_set_properties (objc_contents `PREC`)
	  [ `SYMBOLICADDRESS`, symbolic_address_term (append (append ``Modules`` saddr_suffix) [name])
	  ; `NAME`, (itoken_term name)
          ; `DESCRIPTION`, metaprl_refiner_description_term
	  ])
	pterm)
;;


let rename_duplicates l =
  letrec rename seen n i =
   let nn = concat n (int_to_tok i) in
    if (mem nn seen)
        then rename seen n (1 + i)
        else nn
   in

  letrec aux seen rem =
   if rem = [] then []
     else 
     let n = hd rem in
       if (mem n seen) then
         (let nn = rename seen n 0 in
           (nn . aux (nn . seen) (tl rem)))
         else (n . aux (n . seen) (tl rem))

 in aux [] l
;;


% saddr is address of module whose dforms are being loaded. %
let libmp_load_display saddr_suffix dmod = 
 let prec, dforms = dmod in
  let (onames, dterms) = unzip dforms in 
  let names = rename_duplicates onames in

  % load display %
  let aux name dterm = 
    (name,
     (lib_add
      (objc_modify_source
 	(objc_set_properties (objc_contents `DISP`)
	  [ `SYMBOLICADDRESS`, symbolic_address_term (append (append ``Modules`` saddr_suffix) [name])
	  ; `NAME`, (itoken_term name)
          ; `DESCRIPTION`, nuprl5_edit_description_term
	  ])
	(mp_import_dform_term name dterm))))
    in

    % create dir for dforms and add to display directory %
    let mname = last saddr_suffix in
    let doc = objc_add_properties (create_dag_directory_objc ())
			[ `NAME`, (itoken_term (last saddr_suffix))
			] in
       
       (let doid = lib_add
		    (objc_modify_source doc
		     	    (dag_add_children_src nil (objc_source doc) 
			      (((concat mname `_prec`), libmp_make_prec saddr_suffix mname prec)
			       . (map2 aux names dterms)))) 
         in
	    activate doid
          ; (dag_insert (mp_display_directory ()) (last saddr_suffix) doid ? ())
        )
       ; ()
;;

letref debugload = ([``a``], [``b``]);;
letref debugacc = [(`foo`, dummy_object_id())];;
     
let libmp_import_aux f oid= 
  let saddr = symbolic_address_of_object oid in
  let mname = last saddr in
  letref acc = nil in

  let import_stm (name, term) =
    (let coid = create_with_term `STM` term
	[`SYMBOLICADDRESS`, (symbolic_address_term (append saddr [name])); 
	  `NAME`, (itoken_term name) ;
	  `DESCRIPTION`, metaprl_refiner_description_term;
	] in
    acc := ((name, coid) . acc);
    ())
      
  and import_abs (name, term) =
    (let coid = create_with_term `ABS` term
	[`SYMBOLICADDRESS`, (symbolic_address_term (append saddr [name])); 
	  `NAME`, (itoken_term name);
	  `DESCRIPTION`, metaprl_refiner_description_term;
	] in
    acc := ((name, coid) . acc);
    ())
      
  and import_term (name, term) =
    (let coid = create_with_term `TERM` term
	[`SYMBOLICADDRESS`, symbolic_address_term (append saddr [name]) ;
	  `NAME`, (itoken_term name) ;
	  `DESCRIPTION`, metaprl_refiner_description_term
	] in
    acc := ((name, coid) . acc);
    ())
  in
 
  let (dependencies, (rws, crws, thms, rls)) = f saddr in
    map import_abs rws;
    map import_abs crws;
    map import_stm thms;
    map import_stm rls; %maybe primitive rules should be terms since they cannot be modified in mp%
    
    delete_strong oid;
    debugacc := acc;
    (lib_insert oid 
       (let doc = (objc_add_properties (create_dag_directory_objc ())
	    	     [ `SYMBOLICADDRESS`, (symbolic_address_term saddr);
		       `NAME`, (itoken_term mname)
		     ]) in
       (objc_modify_source doc
	  (dag_add_children_src nil (objc_source doc) acc))));

    activate oid;

    let (loaded, loading) = divide_list libmp_display_loaded_p (rev dependencies) in
      map2 libmp_load_display loading (mp_list_display loading);
    ()
;;

let libmp_import_theorems = libmp_import_aux mp_list_module;;
 
let libmp_import_root dir =

  put_properties dir [ `SYMBOLICADDRESS`, (symbolic_address_term ``Modules``);
		       `NAME`, (itoken_term `Modules`);
		       `DESCRIPTION`, object_id_dag_description_term
		     ];
  libmp_import_root_aux dir (mp_list_root_module ());
  dag_make_directory dir `Display`;
  put_property (mp_display_directory ()) `NAME` (itoken_term `Display`);
  ()
;;

let libmp_object_p oid = 
 (can symbolic_address_of_object oid)
 & ((`metaprl` = system_of_description_term (description_of_objc (oc oid)))
   ? false)
;;
 
 
let libmp_put_term_hook oid term =
  %term_break `hook`;%
   (let kind = objc_kind (oc oid) in
    if kind = `STM` then
        (if libmp_object_p oid then
          (mp_save oid;
	   mp_thm_set (symbolic_address_of_object oid) term
	   ))
    else
    if kind = `ABS` then
        (if libmp_object_p oid then
           (mp_save oid;
	    mp_rw_set (symbolic_address_of_object oid) term
	   ))
	   
   ? ())
;;

put_term_hook := inr libmp_put_term_hook;;

let libmp_stm_to_prf_hook soid soc = 
  let saddr = symbolic_address_of_object soid in
   
  let oid = create_object_id () in
  let oc = (objc_set_properties (prf_objc_src_import_inf_tree (objc_contents `PRF`) 
						      (mp_lookup_proof (tl saddr)))
	    	        [ `SYMBOLICADDRESS`, (symbolic_address_term saddr)
			; `NAME`, (itoken_term (last saddr))
			; `DESCRIPTION`, metaprl_refiner_description_term
			])   
     in lib_insert oid oc;
     oid
;;    

%note: below assumes you already called libmp_import_theorems%
let libmp_import_proofs doid = 
  
  let soids = lib_statements (map snd (directory_children doid)) in 
  map (\x. let soc = oc x in let prfs = stm_objc_src_proofs soc in
           if (prfs = []) then 
	     let poid = libmp_stm_to_prf_hook x soc in
             save x (stm_objc_src_modify_proofs soc [poid]) else
             ())
    soids;
  ()
;;


let libmp_import_theorems_all () = 
 let modules = (map snd (directory_children (descendent (root `theories`) ``metaprl Modules``))) in
  map (\x. 
       if (equal_oids_p x (mp_display_directory ())) then ()
       else libmp_import_theorems x)
    modules;
 ()
;;

let libmp_import_proofs_all () = 
 let modules = (map snd (directory_children (descendent (root `theories`) ``metaprl Modules``))) in
  map (\x. 
       if (equal_oids_p x (mp_display_directory ())) then ()
       else libmp_import_proofs x)
    modules;
 ()
;;

%LAL change this to get the list from mp_import_root instead of the dir since users could change this, so could mp.%

let libmp_import_all () = 
  let modules = (map snd (directory_children (descendent (root `theories`) ``metaprl Modules``))) in
  map (\x. 
       if (equal_oids_p x (mp_display_directory ())) then ()
       else libmp_import_theorems x)
    modules;
  %below may need to be in separate transaction%
  map (\x. 
       if (equal_oids_p x (mp_display_directory ())) then ()
       else libmp_import_proofs x)
    modules;
  ()
;;

let libmp_open_proof soid =

  let soc = oc soid in
  let prfs = stm_objc_src_proofs soc in
  if not (prfs = []) then 
    hd prfs

  else

    let poid = libmp_stm_to_prf_hook soid soc in
    save soid (stm_objc_src_modify_proofs soc [poid]);
    poid
;;    


let libmp_close_proof soid = 
  
  let soc = oc soid in
  let prfs = stm_objc_src_proofs soc in
    (if not (prfs = []) then 
       (save soid (stm_objc_src_modify_proofs soc []);())
       ; map delete_strong prfs)
  ; ()
;;
    
let libmp_some_open_proof (soid, soc) = 
  if (libmp_object_p soid) then
      inr (libmp_stm_to_prf_hook soid soc)
  else inl ()
;;
     
stm_to_prf_hooks := [libmp_some_open_proof; stm_to_prf_default_hook] ;;


%at lib toploop, purge_directory (descendent (root `Modules`) ``itt_logic``);;%
letrec purge_directory doid =

  let poc = dag_directory_objc doid in
  if not (dag_objc_directory_p poc)
  then (raise_error [doid] [`child`; `not`; `directory`] [])
  else let idir = (objc_source poc) in
  (search_isexpr idag_cons_op 
		 (\t. true)
		 (\t. (let oid = (oid_of_idag_child_term t) in
	    if dag_objc_directory_p (lib_object_contents oid)
	    then purge_directory oid
	    else (deactivate oid; delete_strong oid)))
     (children_of_idirectory_term idir);

   ();
   let name = token_of_itoken_term (get_property doid `NAME`) in
   deactivate doid;
   save doid 
	(objc_modify_source
	  (objc_set_properties (oc doid)
	  [ `SYMBOLICADDRESS`, symbolic_address_term (append ``Modules`` [name]);
	   `NAME`, (itoken_term name);
           `DESCRIPTION`, nuprl5_library_description_term
	  ])  
	   (itoken_term name));
    ())
	
;;

let compile_mp_obj oid =
  let code = objc_substance (oc oid) in
  let term = mp_eval_command_to_term (mp_command `!mp_compile` [code]) in
  term
;;



