
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************

#+cltl2(in-system-package)
#-cltl2(in-package *system-package-name*)


;;;;
;;;;	write-term-to-ascii-stream (<term> <stream>)	: NULL
;;;;	read-term-from-ascii-stream (<stream>)		: <term>
;;;;
;;;;	new-term-stream-link ( (in-socket . out-socket) list)	: <link>
;;;;	new-term-tt-link (<bool{server?}>)				: <link>
;;;;

(defconstant csid-level (character-to-code #\l))
(defconstant csid-term (character-to-code #\t))
(defconstant csid-break (character-to-code #\newline))

(defun write-term-to-ascii-stream (aterm stream)
  (let ((term (if *link-term-hook* (funcall *link-term-hook* aterm) aterm)))
    (prog2
	(when (and *ascii-echo-p* *process-break*) (terpri))
	(walk-term-ascii term
			 #'(lambda (byte)
			     (when (and *ascii-echo-p* *process-break*)
			       (format t "~a,~a." byte (code-char byte)))
			     (prl-stream-write byte stream)))
      (prl-stream-write ieot stream)
      (when (and *ascii-echo-p* *process-break*) (terpri))  )))

(defun read-term-from-ascii-stream (stream)
  (prog2
     (when (and *ascii-echo-p* *process-break*) (terpri))
     (with-stream-scanner (stream)
       (prog1 (scan-term-ascii)
	 (scan-rest ieot)))
     (when (and *ascii-echo-p* *process-break*) (terpri))
     ))


(defun write-term-to-standard-ascii-stream (term stream)
  (prog1 (walk-term-ascii-standard term
				   #'(lambda (byte)
				       (when (and *ascii-echo-p* *process-break*)
					 (format t "~a,~a." byte (code-char byte)))
				       (prl-stream-write byte stream)))
    (prl-stream-write ieot stream)))

(defun read-term-from-standard-ascii-stream (stream)
  (with-stream-scanner (stream)
    (prog1 (scan-term-ascii-standard)
      (scan-rest ieot))))




;;;;	
;;;;	Extend prl-stream to encode compression.
;;;;	  
;;;;	cprl-stream : compressed-prl-stream 
;;;;	
;;;;	with-cprl-open-out file ((stream fname levels leveln-p &optional append-p) &body)
;;;;	  * leveln-p : if nil, prohibits leveln at write. logs would normally avoid leveln.
;;;;	  * If append-p t then expects levels to be identical to levels at
;;;;	    last close. If not append then writes level stamp.
;;;;	
;;;;	with-cprl-open-in-file ((stream fname &optional position levels) &body)
;;;;	  * If position specified, expect levels to be identical to levels at
;;;;	    last close.
;;;;	  
;;;;	new-cprl-in-stream (stream)			: <cprl-stream>
;;;;	new-cprl-out-stream (stream levels leveln-p)	: <cprl-stream>
;;;;	
;;;;	cprl-stream-read (<cprl-stream>)		: <term>
;;;;	cprl-stream-write(<cprl-stream> <term>)		: NULL
;;;;	
;;;;	push-cprl-stream-level (<cprl-stream> <compression-level>)
;;;;	pop-cprl-stream-level (<cprl-stream>)
;;;;	  * vapor : <compression-level> arg may be compression descr ???
;;;;	  * only valid on out-streams.
;;;;
;;;;	set-cprl-stream-levels (<cprl-stream> <compression-levels>)
;;;;	  * writes level stamp, updates cprl-stream levels.
;;;;	
;;;;
;;;;	assume multiple read/writes ok. But may have to fix underlying bsc scanner
;;;;	for multiple reads to work.
;;;;	
;;;;	TODO (position) :  reopen and set file-pointer ??? 
;;;;	  - desire ability to pickup reading from a byte index point in the file.
;;;;	      * requires ability to get byte index from stream.
;;;;	      * requires ability to set file-pointer in stream.
;;;;	
;;;;	
;;;;	TODO ??? it might be better if scanner where arg rather than global.
;;;;	 to avoid milling, maybe make duplicate scan functions so that new take arg
;;;;	 but and then implement old using new + global.
;;;;	
;;;;	Stream channels share compression with two streams.
;;;;	that's up to channel to manage. 
;;;;	
;;;;	nifty way to manage: server updates send-side and sends client level, client updates recv-side
;;;;	and send-side and sends server level, server updates recv-side.
;;;;	
;;;;	
;;;;	
;;;;	
;;;;	
;;;;	
;;;;	internal to cprl.
;;;;	
;;;;	new-cprl-stream (stream levels leveln-p)	: <cprl-stream>
;;;;	


(defstruct cprl-stream
  prl-stream
  levels-update-hook	;; in case of duplex stream allows updating of send side
  		        ;; by recv side when levels received.
  levels
  leveln-p
  read-ahead)

(defun prl-stream-of-cprl-stream (s) (cprl-stream-prl-stream s))
(defun levels-of-cprl-stream (s) (cprl-stream-levels s))

(defun cprl-stream-file-position (cprl pos)
  (prl-stream-file-position (prl-stream-of-cprl-stream cprl) pos))

(defun cprl-stream-update-levels (s levels)
  (setf (cprl-stream-levels s) levels)
  (when (cprl-stream-levels-update-hook s)
    (funcall (cprl-stream-levels-update-hook s) levels))
  levels)


(defun new-cprl-in-stream (pstream &optional levels-update-hook)

  (make-cprl-stream :levels-update-hook levels-update-hook
		    :prl-stream pstream))

(defun new-cprl-out-stream (pstream &optional levels leveln-p not-send-p)

  ;;(break "ncos")
  (let ((s (make-cprl-stream :prl-stream pstream
			     :levels levels
			     :leveln-p leveln-p)))

    (when (and levels (not not-send-p))
      (cprl-stream-send-levels s levels))
    
    s))

(defun new-cprl-io-stream (pstream &optional levels leveln-p not-send-p levels-update-hook)

  ;;(break "ncos")
  (let ((s (make-cprl-stream :levels-update-hook levels-update-hook
			     :prl-stream pstream
			     :levels levels
			     :leveln-p leveln-p)))

    (when (and levels (not not-send-p))
      (cprl-stream-send-levels s levels))
    s))


(defun cprl-stream-out-update-levels (cstream levels)
  (cprl-stream-send-levels cstream levels)
  (setf (cprl-stream-levels cstream) levels))
  


#|
(defun session-levels-to-stream (term closure)
  (funcall closure csid-level)
  (with-levels-out (levels)
    (walk-term-ascii-compress term closure nil))
  (funcall closure csid-level))
|#
      

(defun cprl-stream-send-levels (cstream levels &optional index)
  (let ((pstream (prl-stream-of-cprl-stream cstream)))
    (prl-stream-write csid-level pstream)
    (with-levels-out (levels)
      (walk-term-ascii-compress (short-term-of-compression-levels levels index)
				pstream
				nil))
    (prl-stream-write csid-level pstream)))


(defmacro with-cprl-open-out-file ((stream fname levels leveln-p &optional append-p) &body body)
  (let ((pstream (gentemp)))
    `(with-safe-prl-open-file (,pstream ,fname (if ,append-p 'append 'out))
      (let ((,stream (new-cprl-out-stream ,pstream ,levels ,leveln-p ,append-p)))
	(multiple-value-prog1 (progn ,@body)
	  (prl-stream-finish-output ,pstream))))) )

(defmacro with-cprl-open-in-file ((stream fname &optional position) &body body)
  (let ((pstream (gentemp)))
    `(with-prl-open-file (,pstream ,fname in)
      (let ((,stream (new-cprl-in-stream ,pstream)))
	(when ,position
	  (break "cprl-open-in-file"))
	,@body))))

(defmacro with-cprl-open-io-file ((stream fname &optional position) &body body)
  (let ((pstream (gentemp)))
    `(with-prl-open-file (,pstream ,fname io)
      (let ((,stream (new-cprl-io-stream ,pstream)))
	(when ,position
	  (break "cprl-open-io-file"))
	,@body))))


;; returns nil on eof if blockp t. 
;; eof not detectable with if not blocking.
(defun cprl-stream-read-maybe (cstream &optional blockp)

  (let ((levels (levels-of-cprl-stream cstream))
	(pstream (prl-stream-of-cprl-stream cstream)))
    
    (with-stream-scanner (pstream nil blockp)

      (labels
      
	  ((aux ()
	     ;;(unless (scan-eof-p) (format t "csrm ~a~%" (scan-cur-char)))
	     (cond
	       ((scan-at-byte-p csid-level)
		(scan-byte csid-level)

		(setf levels
		      (cprl-stream-update-levels cstream
						 (read-level (scan-compressed-term-ascii nil)
							     levels)))

		(scan-byte csid-level)

		(when (or blockp (prl-stream-listen pstream))
		  (aux)))

	       ((scan-at-byte-p csid-term)
		(scan-byte csid-term)
		(prog1
		      (if (compression-training-tables-p)
			  (let ((term (scan-compressed-term-ascii (new-compression-levels levels 'in))))
			    (add-term-to-training-tables term)
			    term)
			  (scan-compressed-term-ascii (new-compression-levels levels 'in)))

		  ;;(progn (setf -tt *** ) (break "tt") -tt)
		  (scan-rest csid-term)))

	       ((scan-eof-p)
		(if blockp
		    nil
		    (raise-error (error-message '(cprl stream read eof)))))

	       ((scan-at-byte-p csid-break)
		(scan-byte csid-break)
		(aux))
	     
	       (t ;;(format t "~%cprl-stream-read-maybe")
		  (break "cprl-stream-read-maybe")
		  ))))

	(aux)))))


;; returns nil on eof.

(defvar *debug-show-term-op-count-p* nil)
;; (setf  *debug-show-term-op-count-p* nil)
;; (setf  *debug-show-term-op-count-p* t)

(defun cprl-stream-read (cstream)
  (let ((term 
	 (if (cprl-stream-read-ahead cstream)
	     (prog1
		 (cprl-stream-read-ahead cstream)
	       (setf (cprl-stream-read-ahead cstream) nil))
	     (cprl-stream-read-maybe cstream t))))
  
    (when *debug-show-term-op-count-p*
      (if term
	  (let ((count (term-op-count term)))
	    (format t "read term, op count : ~11:D.~%" count)
	    (when (and nil (> count 10000))
	      (break)
	      ))
	  (format t "~%read term NIL~%")))
    
    term))



;; listen if level data may not return. after read.
;;  if level data only then do not want to return t.
;;  thus read level data then listen on pstream.
;; or caller of read must accept nil if no term after level.

(defun cprl-stream-listen (cstream)
  (if (cprl-stream-read-ahead cstream)
      t
      (when (prl-stream-listen (prl-stream-of-cprl-stream cstream))
	(let ((term (cprl-stream-read-maybe cstream nil)))
	  (when term
	    (setf (cprl-stream-read-ahead cstream) term)
	    t)))))


(defvar *big-term* (cons 1 (instantiate-term (instantiate-operator `!void))))
(defun big-term (term)
  (let ((c (term-op-count term)))
    (when (> c (car *big-term*))
      (setf *big-term* (cons c term))))
  term)


(defun cprl-stream-write (cstream aterm &optional linebreakp )

  (let ((term (if *link-term-hook* (funcall *link-term-hook* aterm) aterm)))
  
    (when *debug-show-term-op-count-p*
      (format t "write term, op count : ~11:D.~%" (term-op-count term)))
  
    ;;(setf -cstream cstream -term term) (break "csw")
  
    (when (compression-training-tables-p)
      (add-term-to-training-tables term))

    (let ((pstream (prl-stream-of-cprl-stream cstream))
	  )

      (when linebreakp (prl-stream-write csid-break pstream))
      (prl-stream-write csid-term pstream)
      
      ;;(format t "~%term write ~a" (term-op-count term))
      (term-out-with-compression-level pstream
				       (levels-of-cprl-stream cstream)
				       (cprl-stream-leveln-p cstream)
				       term
				       )
      
      (prl-stream-write csid-term pstream))))



(defun new-cprl-stream-link (primary-sockets &optional levels)
  ;;(break "ncprl")
  (new-stream-link primary-sockets levels))

(defun cprl-stream-finish-output (s)
  (prl-stream-finish-output (prl-stream-of-cprl-stream s)))

;; TODO this will surely cause trouble to have to change this list
;; to force standard ascii.
(defun new-prl-stream-link (primary-sockets &optional api)
  (if (and api (member api '(FDL0)))
      (new-standard-ascii-stream-link primary-sockets)
      (new-uncompressed-stream-link primary-sockets)))


;; old version
;;(defun new-term-stream-link (ports server-p)
;;  (new-stream-link ports server-p #'session-term-to-stream #'session-stream-to-term))

;; (defun new-term-stream-link (socket-pairs)
;;  (new-stream-link socket-pairs #'session-term-to-stream #'session-stream-to-term))

(defun new-term-mathbus-link (primary-sockets)
  (new-mathbus-link primary-sockets
		    #'(lambda (term stream) (write-node (term-to-mbterm term) stream))
		    #'(lambda (stream) (mbterm-to-term (read-node stream)))))




;;;;	
;;;;	cprl-stream strings for testing.
;;;;	
;;;;	cprl-stream-write-to-string (term )
;;;;	

(defun cprl-stream-read-from-string (string)
  (with-input-from-string (sstream string)
    (let ((pstream (new-prl-in-stream sstream
				      #'(lambda (s eofp eof) (char-code (read-char s eofp eof))))))
      (cprl-stream-read (new-cprl-in-stream pstream)))))


(defun cprl-stream-write-to-string (term &optional levelnp)
  (with-output-to-string
    (sstream)
    
    (let ((pstream (new-prl-out-stream sstream
				       #'(lambda (b stream)
					   (and t
						(write-char (code-char b) stream)))
				       #+fbbstring
				       #'(lambda (stream a offset count)
					   (write-string a stream :start offset :end (+ offset count)))
				       #-fbbstring
				       #'(lambda (stream a offset count)
					   (dotimes (i count)
					     (write-char (code-char (aref a (+ i offset))) stream))))))

      
      (let ((cstream (new-cprl-out-stream pstream (cdadr *levels-cache*) levelnp)))

	
	(cprl-stream-write cstream term)
	(cprl-stream-finish-output cstream)
	))))

			 

(defun test-file-output (fname term profile-p)
  (if profile-p
      (with-profile (:time test-file-output t nil)
	(with-cprl-open-out-file (stream fname (get-levels) t)
	  (cprl-stream-write stream term)))
      (time (with-cprl-open-out-file (stream fname (get-levels) t)
	      (cprl-stream-write stream term)))))

(defun write-term-to-file (fname term)
  (with-cprl-open-out-file (stream fname nil t)
    (cprl-stream-write stream term)))


(defun read-term-from-file (fname)
  (with-cprl-open-in-file (stream fname)
    (cprl-stream-read stream)))

(defun write-terms-to-file (fname terms)
  (with-cprl-open-out-file (stream fname nil t)
    (dolist (term terms)
      (cprl-stream-write stream term))))

(defun read-terms-from-file (fname)
  (let ((acc nil))
    (with-cprl-open-in-file (stream fname)
      (do ((term (cprl-stream-read stream) (cprl-stream-read stream)))
	  ((null term))
	(push term acc)))
    (nreverse acc)))
	
  
;; (defvar *big-term* nil)

;; (progn (setf tt (test-file-input "~/fu2.term")) nil)
;; (progn (setf *big-term* (cons (term-op-count tt)  tt)) nil)

(defun test-file-input (fname)
  (with-cprl-open-in-file (stream fname)
    (cprl-stream-read stream)))
  

;;;;	
;;;;	Assoc list on disk.
;;;;	
;;;;	
;;;;	
;;;;	
;;;;	
;;;;	<term{key}> <term{data}>
;;;;	
;;;;	disk-list-assoc
;;;;	

;;;;	
;;;;	levels-assoc : assoc of l{<i>:n} with stamp.
;;;;	  - files referencing levels will use short form.
;;;;	  - get-levels will take short form or long form as arg.
;;;;	    if short form lookups in level assoc.
;;;;	    update local cache with assoc using this file.
;;;;	  - put-levels will assign <i> and append to levels-assoc.


;;;;	
;;;;	RLE TODO : very primitive-match for search on terms. Needs to be more robust, 
;;;;	RLE TODO : but simplicity is attractive.
;;;;	


(defun placeholder-op-match (match instance)
  (and (eql (id-of-operator match) (id-of-operator instance)) 
       (forall-p #'(lambda (match instance)
		     (and (equal-types-p (type-of-parameter match) (type-of-parameter instance))
			  (or (slot-parameter-value-p (value-of-parameter match))
			      (equal-parameters-p match instance))))
		 (parameters-of-operator match)
		 (parameters-of-operator instance))))

(defun placeholder-match (match instance)
  ;;(setf a match b instance) (break)
  (if (iplaceholder-term-p match)
      t
      (and (placeholder-op-match (operator-of-term match)
				 (operator-of-term instance))
	   (forall-p #'(lambda (bt-m bt-i)
			 (and (forall-p #'(lambda (b-m b-i)
					    (equal-parameter-values-p b-m b-i *variable-type*))
					(bindings-of-bound-term bt-m)
					(bindings-of-bound-term bt-i))
			      (placeholder-match (term-of-bound-term bt-m) (term-of-bound-term bt-i))))
		     (bound-terms-of-term match)
		     (bound-terms-of-term instance)))))


(defun disk-list-append (list-name term)
  (with-cprl-open-out-file (stream (ifilename-to-filename '|FDLdb| nil list-name "lst")
				   nil nil t)
    (cprl-stream-write stream term)))

(defun disk-list-find-last (list-name &optional match)
  (let ((last-match nil))
    (with-ignore
	(with-cprl-open-in-file (stream (ifilename-to-filename '|FDLdb| nil list-name "lst"))

	  (do ((term (cprl-stream-read stream) (cprl-stream-read stream)))
	      ((null term))
    
	    (when (or (null match) (placeholder-match match term))
	      (setf last-match term)))))

    (when (null last-match)
      ;;(setf -list-name list-name)
      (break "disk-list")
      (raise-error (error-message '(disk-list find not) match)))
    
    last-match))


(defun disk-list-read (list-name)
  (with-cprl-open-in-file (stream (ifilename-to-filename '|FDLdb| nil list-name "lst"))
      
    (do ((term (cprl-stream-read stream) (cprl-stream-read stream))
	 (terms nil (cons term terms)))
	((null term)
	 (nreverse terms)))))

(defun disk-list-pop (list-name)
  (let ((list (butlast (disk-list-read list-name))))
    
    (with-cprl-open-out-file (stream
			      (ifilename-to-filename '|FDLdb| nil list-name "lst")
			      nil nil)
      (dolist (term list)
	(cprl-stream-write stream term)))))

(defun put-level-file-pointer (pt)
  (with-cprl-open-out-file (s (db-extend-pathname nil "COMPRESSION" "CURRENT") nil nil)
    (cprl-stream-write s pt)))

(defun get-level-file-pointer ()
  (with-cprl-open-in-file (s (db-extend-pathname nil "COMPRESSION" "CURRENT"))
    (cprl-stream-read s)))

(defun write-level (levels index type &optional heuristics)
  
  (advance-sequence)
  (let ((stamp (transaction-stamp)))

    (let ((new-levels 
	   (case type
	     (static (with-cprl-open-out-file (cstream (stamp-to-pathname stamp type) nil t)
		       (compression-write-static-level cstream levels index stamp heuristics)))

	     (otherwise (raise-error (error-message '(compression level write type unknown)
						    type index))))))

      new-levels)))


(define-primitive |!level_assoc| () (short long))
(define-primitive |!l| ((natural . index)) ())

(defun levels-list-add (long)
  (let* ((l (with-ignore (disk-list-find-last "levels")))
	 (short (il-term (if l
			     (1+ (index-of-il-term
				  (short-of-ilevel-assoc-term l)))
			     0)))
	 (aterm (ilevel-assoc-term short long)))
    
    (disk-list-append "levels" aterm)
    aterm))

(defun levels-list-assoc (short)
  (let* ((l (disk-list-find-last "levels" (ilevel-assoc-term short (iplaceholder-term)))))
    (if l
	(long-of-ilevel-assoc-term l)
	(raise-error (error-message '(levels assoc found not) short)))))
    
;; (levels-list-add (persist-term-of-compression-levels (cdar *levels-cache*)))

;;;;	Psuedo-orb in concert with start/stop offers fine grained control over
;;;;	routing of broadcasts.
;;;;
;;;;	Some components may connect without use of a psuedo orb, eg, may connect
;;;;	directly to tooltalk. In such a case, coarser categories of messages are
;;;;	convenient.
;;;;	
;;;;	The orb-tooltalk connect will be mininal until the viability of
;;;;	tooltalk is established. The categories of messages the bus expects from
;;;;	tooltalk will be fixed at connection. They will be supplied as args to
;;;;	the connection procedure.
;;;;
;;;;	In order to avoid complexity of rejecting (or forwarding) messages not
;;;;	bound for a local environment, assume no more than one component of any
;;;;	category server is connected to tooltalk at one time.
;;;;	
;;;;	Current server categories : LIB, DMS.
;;;;	
;;;;	Req and Broadcast iobs have category field.
;;;;	Category tag will be prepended to request's destination address during transmission.
;;;;	Category tag will be parameter of broadcast term during tranmission.
;;;;
;;;;	IO layer examines req or broadcast term to determine category when needed.
;;;;	
;;;;	SYNGEN TODO : be prepared for env address changes, ie, type prepended to address.
;;;;
;;;;	Twould be nice not to have to include type in broadcast or request terms. However,
;;;;	as we could handle as function args in source orb, if it will be forwarded,
;;;;	info is lost once sent down stream link. May be able to streamline at a later date, 
;;;;	ie, tooltalk not supported, or tooltalk pattern ops expanded.
;;;;	


#|

(defun session-term-to-stream (inlevels term closure &optional leveln-protect)
  (when (compression-training-tables-p)
    (add-term-to-training-tables term))

  (let ((levels inlevels))
	  
    (when (and (not leveln-protect)
	       (term-op-count-exceeds-p term *compression-op-count-threshold*))

      (setf levels (new-compression-levels inlevels 'out)))

    (when levels
      (add-structure-to-compression-levels levels term *compression-term-type*)
      (add-indices-to-compression-levels term levels)
      )
    
    (funcall closure csid-term)
    (walk-term-ascii-compress levels term closure)
    (funcall closure csid-term)))



;; level-update-f (<compression-levels>)	: NULL
;;

(defun session-stream-to-term (inlevels level-update-f &optional (endp t))

  (let ((levels inlevels))

    (labels
      
	((aux ()
	   (cond
	     ((scan-at-byte-p csid-level)
	      (scan-byte csid-level)

	      (setf levels (read-level levels))
	      (funcall level-update-f levels)

	      (scan-byte csid-level)
	      (aux))

	     ((scan-at-byte-p csid-term)
	      (scan-byte csid-term)
	      (prog1
		  (scan-compressed-term-ascii (new-compression-levels levels 'in))
		(if endp
		    (scan-rest csid-term)
		    (scan-byte csid-term))))
	     
	     (t (break "session-stream-to-term") ))))

	   (aux))))





;;;	1 Expression | connect
;;;	2 Broadcast	
;;;	3 Message
;;;	4 Result | connected
;;;	5 Other : !ping !pong !void ???     

;;;	6 DMS-Broadcast  !broadcast with table DMS-DFORMS or DMS-PRECEDENCES
;;;	7 DMS-Expression  !dms_command ?? !command{DMS}(*)

(defun io-type (term)
  (cond
    ((iinterrupt-term-p term)
     1)
    ((ireq-aux-term-p term)
     (if (eql (type-of-ireq-term term) 'dms)
	 7
	 1))
    ((ibroadcasts-term-p term)
     (if (eql (type-of-ibroadcasts-term term) 'dms)
	 6
	 2))
    ((iinterrupted-term-p term)
     4)
    ((irsp-term-p term)
     4)
    ((imsg-term-p term)
     3)
    (t 5
       ;;(setf a term) (break)
       (raise-error (error-message '(io type term) term))
       )))

(defun new-term-tt-link (types compress-p &optional session)
  (new-tt-link types
	       ;; expects to be called from inside of with-byte-accumulator.
	       (if compress-p
		   #'(lambda (term)
		       (session-term-to-stream term #'accumulate-byte))
		   #'(lambda (term) 
		       (walk-term-ascii term #'accumulate-byte)))
	       (if compress-p
		   #'session-stream-to-term
		   #'read-term-from-ascii-stream)
	       #'io-type
	       session))
|#

