
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************

#+cltl2(in-system-package)
#-cltl2(in-package *system-package-name*)

(defun com-ml-comp (&optional printp)
  ;;(break "cmc")
    (inml (ml-text
	   (list
	    (format-string "load_system true false ~a [\"library\"]"
			   (if printp "true" "false"))
	    "[([\"top\"], [\"general\"; \"primitives\"; \"term\"; \"subst\"; \"comm-bsc\";"
	    "   \"comm-def\"; \"level-exp\"; \"match\"; \"compute-aux1\"])"

	    "; ([\"orb\"], [\"orbm-bsc\"])]"))))

;;(defun extend-system-path (dirs)
;;  (prl-extend-pathname `system dirs))
      
(defun find-db-path ()
  (or (car (get-config-data 'dbpath))
      (let ((dbpath (get-config-data 'dbsystempath)))
	(when dbpath
	  (extend-system-path dbpath)))
      (extend-system-path (list "FDLdb"))))
   

(defun init-com-base ()

  (init-process-id)
  (setf *transaction-sequence* 0)
  (setf *transaction-id* nil)
  (setf *sequence-count* 1)
  (advance-transaction-sequence)

  (setf *transactions-active* nil)
  (setf *bus-pend* nil)
  
  (setf *orb-address* nil)
  )

(defvar *reset-orb-name-hook* nil)

(defun reset-orb-name ()
  (let ((times (princ-to-string (get-universal-time)))
        (iam (car (get-config-data 'iam)))
	(host (local-host)))
    
    (set-orb-name (intern-system (let ((s (if host
					     (concatenate 'string host "-" times)
					   times)))
				  (if iam
				      (concatenate 'string (string iam) "@" s)
				    s))))

    (when *reset-orb-name-hook*
      (funcall *reset-orb-name-hook*))
    ))

;; yanked this kludge from some mail thread for cmu18e
#+cmu18e
(defvar *reload-global-alist* nil)

(defun set-reload-global-alist ()
  #+cmu18e
  (setf *reload-global-alist*
	(list
	 (cons (complete-system-path (list "binaries" "linux86") "ffi-unix" "so") "ffi-unix")
	 (cons (complete-system-path (list "binaries" "linux86") "ffi-io" "so") "ffi-io")
	 (cons (complete-system-path (list "binaries" "linux86") "ffi-bsc" "so") "ffi-bsc")))
  #-cmu18e
  nil)
#+cmu18e
(defun reload-global-table ()
  (labels ((newpath (name)
	     (let ((np (extend-pathname cl-user:*system-path-prefix*
					(append '("binaries")
						(butlast (cl-user:standard-binary-directories)))
					name
					(cdr make::*foreign-filename-extensions*))))
	       (format t "NewPath ~a~%" (namestring np))
	       np)))
    (loop :for lib-entry in sys::*global-table*
	  :for (sap . lib-path) = lib-entry
	  :when lib-path :do
	  (let ((new-sap (sys::dlopen (namestring
				       (newpath (cdr (assoc (namestring lib-path) *reload-global-alist* :test #'string=))))
				      (logior sys::rtld-now sys::rtld-global))))
	    (when (zerop (sys:sap-int new-sap))
	      (error "Couldn't open library ~S: ~S" lib-path (sys::dlerror)))
	    (setf (car lib-entry) new-sap)))
    (alien:alien-funcall (alien:extern-alien "os_resolve_data_linkage"
					     (alien:function c-call:void)))))

(defun cmu-ff-restart-fixup ()
  ;;(break "cfrf")
  ;;(loadnff "ffi-bsc")
  ;;(loadnff "ffi-unix")
  ;;(loadnff "ffi-io")
  #+cmu18e(reload-global-table)
  #-cmu18e(eval '(progn 
		  (defunff (make-nonblocking-socket ffi-make-socket) int ((port int) (queue-size int)))
		  (defunff (establish-connection-server ffi-make-socket-b) int ((port int)))
		  (defunff (establish-connection ffi-open-client) int ((port int) (host string)))
		  (defunff (server-accept ffi-open-server) int ((sock int)))
		  (defunff (close-connection ffi-close-connection) int ((fd int)))

		  ;;from io-db
		  (defunff (libc-mkdir) int ((name string)))
		  (defunff (umask libc-umask) int ((name int)))
		  (defunff (rename libc-rename) int ((old string) (new string)))  
		  (defunff (read-lock-test) int ((fd int)))
		  (defunff (write-lock-test) int ((fd int)))
		  (defunff (read-lock) int ((fd int)))
		  (defunff (write-lock) int ((fd int)))
		  (defunff (isread-lock) int ((fd int)))
		  (defunff (iswrite-lock) int ((fd int)))
		  (defunff (iswrite-lock-file) int ((name string)))
		  (defunff (un-lock unlock-fd) int ((fd int)))
		  (defunff (un-lock-read unlock-fd-read) int ((fd int)))
		  (defunff (owner-uid) int ((name string)))
		  (defunff (libc-atime) int ((name string)))
		  (defunff (libc-asctime) int ((name string)))
		  (defunff (libc-create) int ((name string)))
		  (defunff (libc-rmdir) int ((name string)))

		  )))

(defvar *disksave-config* nil)

(defun init-com-restart (&optional (mlp t) (hotp nil))

  (init-com-base)

  (io-echo-off)
  (setf *eval-print-p* nil)
  
  (format t ";;; Loading user config~%")
  (load-user-config t)

  ;; needs to happen after load-user-config as load-user-config sets system-root which loadnff needs to find files.
  ;; isn't a problem unless disksave is run when system-root differs from system-root at build.
  #-CMU18E (progn
	     (format t ";; loadnff restart")
	     (loadnff "ffi-bsc")
	     (loadnff "ffi-unix")
	     (loadnff "ffi-io"))

  (push-bus-message-filter (message-filter-by-prefix `(INFORM DEPENDENCIES DEFAULT EVAL)))

  ;; (break "disksave-config")
  (when *disksave-config*
    (let ((fname (complete-system-path (list "bin")
				       (format-string "~a.config" *system-kind*)
				       *disksave-config*)))
      (when (probe-file fname)
	(read-config fname))))

  #+cmu (progn
	    (format t ";;; CMU ff restart~%")
	    (cmu-ff-restart-fixup))

  (patch)

  (add-asynch-link)

  (reset-orb-name)

  (when mlp
    (reset-ml)

    (rehash-ml-primitive-types)
    )
  )

(defun reset-dbpath ()
  (db-init (namestring (extend-system-path (list "FDLdb")))))

(defun init-com-top (&optional (mlp t))

  ;; maybe should be in restart??
  #+lucid(load-foreign-libraries nil)
  ;;(setf -mlp mlp) (break "ict")
  (when mlp
    (com-ml-comp)))

(defun do-init-com (&optional (mlp t))

  ;;(setf -mlp mlp) (break "dic")
  (init-com-restart mlp)

  ;; TODO : specify method of including Level0 compression in links and db.
  (init-com-top mlp))


(defun cnf (dir name)
  (let ((binfile (complete-system-path (list* "binaries" (system-bin-path-list))
				      name
				      (bin-file-extension))))

    (compile-file (complete-system-path (list "sys" "library" dir)
				       name
				       "lsp");;(lisp-file-extension) ?? not right gives "lisp"
		  :output-file binfile)

    (load binfile)))


(defun cnmf (dir name &optional srcp)
  (inml (ml-text
	 (list
	  (if srcp
	      (format-string "load_system true true false [\"library\"] " dir)
	      "load_system true true false [\"bin\"; \"patches\"]")
	  "[("
	  (if srcp
	      (format-string "[\"~a\"]" dir)
	      "[]")
	  ", [\""
	  (string name)
	  "\"])]")))
  )

(defun print-client-banner (name)

  (format t "~%;;;~%;;;	")
  (format t "~%;;;	FDL ~a~%;;;~%" name) 

  (let ((ports (ml-text "lib_ports_connected := (config_sockets() ? (0, 0))"))
	(host  (ml-text "lib_host := config_libhost() ? \"\""))
	(env (ml-text "lib_env := config_libenv() ? \"\"")))

    (if (and
	 (not (zerop (car ports)))
	 (not (zerop (cdr ports)))
	 (not (string= "" host))
	 )
	(progn
	  (format t ";;;	The ~a will connect to lib on host ~a using ports ~a, ~a.~%" name host (car ports) (cdr ports))
	  )
	(progn
	  (format t ";;;	The current connection parameters are not valid : ports ~a, ~a, host ~a.~%" host (car ports) (cdr ports))
	  ))
	
    (if (not (string= "" env))
	(format t ";;;	The default environment is ~a.~%" env)
	(format t ";;;	A default environment was not specified.~%"))
    (format t ";;;	~%")
    
    (format t ";;;	At startup the default connection parameters are read from")
    (format t "~~/.~a.config"  *system-kind*) 
    (format t "~%;;;	using keywords: sockets, libhost, libenv.~%")
    (format t ";;;	You may override these parameters via setup_connect: int -> int -> string -> unit~%;;;	and set_lib_env: string -> unit.~%")
    (print-config-data)
    )

  (format t ";;;~%;;;	(top)~%")
  (format t ";;;	go.~%;;;~%")

;;  (format t ";;;	myopen();;~%")
;;  (format t ";;;	myclose();;~%;;;~%")

;;  (format t ";;;	dc();;~%")
;;  (format t ";;;	open_lib `<lib-name>`;;~%;;;~%")
;;  (format t ";;;	close_lib `<lib-name>`;;~%")
;;  (format t ";;;	dd();;~%;;;~%")
  )


;;;;	
;;;;	Patches : use config file syntax, ie s-sexprs.
;;;;
;;;;	  - find restart file for disksave.
;;;;	     * restart-<kind>-<version>
;;;;		(patch-level . 0)
;;;;	  - read patch level.
;;;;	     * restart-<kind>-<version>-<level>
;;;;		(patches . (meta-le gcv case))
;;;;	  - apply patches in patch level.
;;;;	     * each patch name will have corresponding .lsp file in patches dir.
;;;;	
;;;;	Indirection via level file allows easy modification of patch level
;;;;	if patch problems. EG, might allow /home/<user>/<system>.config to set
;;;;	patch level for a particular kind and disksave version.
;;;;	
;;;;	Scenario : adding a patch and patch level.
;;;;	  - copy current restart level file and add patch name.
;;;;	  - save with level incremented.
;;;;	  - modify restart version file to level increment.
;;;;	
;;;;	could use file-write-date to dectect stale patches.

(defun versions-filename (type kind v &optional name)
  (apply #'concatenate 'string
	 (if name (string name) "")
	 (if name "-" "")
	   "FDL"
	   (princ-to-string *system-major-version*)
	   ;;"."
	   ;;(princ-to-string *system-minor-version*)
	   "-"
	   (string kind)
	   "-"
	   (string type)
	   (cond
	     ((null v) nil)
	     ((consp v)
	      (mapcan #'(lambda (i) (list "-" (princ-to-string i))) v))
	     (t (list "-" (princ-to-string v))))))


(defun disksave-filename (v &optional name)
  (versions-filename "disksave" (string-upcase *component-kind*) v name))

(defun restart-filename ()
  (or (cl-user:getenv (format-string "~aRESTARTFILE" *system-kind*))
      (versions-filename "restart" (string-upcase *component-kind*) *disksave-version*)))

(defun patches-filename (v level)
  (versions-filename "restart"
		     (string-upcase *component-kind*)
		     (list *disksave-version* level)))


(defvar *restart-patch-level* nil)

;; <name> . <write-date> list
(defvar *restart-patches-loaded* nil)
(defvar *restart-patch-names* nil)

(defun patches-restart-filename ()
  (complete-system-path (list "bin" "patches")
		       (restart-filename)))

(defun system-patches-filename ()
  (complete-system-path (list "bin" "patches")
		       (patches-filename *disksave-version*
					 *restart-patch-level*)))

(defun load-patch-level ()
  (setf *restart-patch-level*
	(or (get-config-data 'patch-level)
	    (let ((fname (patches-restart-filename)))
	      (when (probe-file fname)
		(read-config fname)
		(get-config-data 'patch-level)
		)))))

(defun load-patch-names ()
  (unless *restart-patch-level* (load-patch-level))
  (setf *restart-patch-names*
	(or (cdr (assoc 'patches (config-data)))
	    (when *restart-patch-level*
	      (let ((fname (system-patches-filename)))
		(when (probe-file fname)
		  (read-config fname)
		  (get-config-data 'patches)))))))


(defun load-patch (name)
  (let ((loaded (cdr (assoc name *restart-patches-loaded*))))
    (let ((fname (complete-system-path '("bin" "patches")
				       (string-downcase (string name)) "lsp")))
      (if (probe-file fname)
	  (let ((wd (file-write-date fname)))
	    (if (or (null loaded)
		    (not (eql loaded wd)))
		(progn
		  (format t ";;;;	loading patch ~a~%" fname)
		  (load fname)
		  (push (cons name wd) *restart-patches-loaded*)
		  (list name))
		(progn
		  (format t ";;;;	Not loading patch ~a~%" fname)
		  nil)))
	  (progn
	    (warn (format nil "Patch file ~a could not be loaded." fname))
	    nil)))))


(defun patch (&optional (force t))

  (load-patch-level)
  
  ;; need to check if patch level updated
  (when force
    (setf *restart-patch-names* nil
	  *restart-level* nil))

  (unless *restart-patch-names* (load-patch-names))

  (mapcan #'load-patch *restart-patch-names*))



(defun read-disksave-version (fname)
  (if (probe-file fname)
      (with-open-file (stream fname :direction :input)
	(read stream))
      0))

(defun write-disksave-version (v fname)
  (with-open-file (stream fname
			  :direction :output
			  :if-exists :supersede
			  :if-does-not-exist :create)
    (write v :stream stream)
    (terpri stream)))


#+(and LINUX86 :ALLEGRO-V4.3)
(defun complete-local-path (directories file)  
  #+(or symbolics unix mach)
  (flet ((add-separator (x) (concatenate 'string x #+symbolics ";" #+(or unix mach) "/")))
    (reduce #'(lambda (x y) (concatenate 'string x y))
	    `("/usr/local/lisp/acl43disksaves/"
	      ,@(mapcar #'add-separator (mapcar #'string directories))
	      ,(string file))))
  #-(or symbolics unix mach)
  file)

(defun system-disk-path ()
  (prl-extend-pathname 'system (cons "disksaves" (system-bin-path-list))))

(defun system-version-filename ()
  (prl-make-filename  'system (list "disksaves")
		     "system" "version"))

(defvar *disksave-filename* nil)
(defvar *disksave-datetime* nil)

(defun clean-environment (env)
  (setf (environment-broadcast-states env) nil
	(environment-broadcast-sentry env) nil
	(environment-tent-order env) nil
	(environment-server-address env) nil))

(defun dodisk-aux (restart-f name version)
  (quit-request-loop)

  (unless (or (null *bus-links*)
	      (and (null (cdr *bus-links*))
		   (asynch-blink-p (car *bus-links*))))
    ;;(break "dble")
    (raise-error (error-message '(dodisk bus links exists))))

  (mapc #'clean-environment *component*)
  
  (set-reload-global-alist)
  (let ((diskpath (system-disk-path)))
    (unless (probe-file diskpath)
      (format t "~%diskpath does not exist ~a~%" diskpath)
      (raise-error (error-message '(dodisk diskpath not) diskpath)))

    (let ((disksave-fname (prl-make-filename diskpath nil
					     (disksave-filename version name)))
	  (disksave-datetime (sortable-datetime-string (get-universal-time))))
      
      (setf *disksave-filename* disksave-fname
	    *disksave-datetime* disksave-datetime)

      (when (probe-file disksave-fname)
	(raise-error (error-message '(dodisk save-version odd) diskpath version)))

      (format t "~%;;;~%;;; Disksave filename ~a.~%;;; ~a~%;;;~%" disksave-fname disksave-datetime)

      ;;(gc t)
      (unloadnff "ffi-bsc")
      (unloadnff "ffi-unix")
      (unloadnff "ffi-io")
	
      (image-save disksave-fname
		  #'(lambda ()
		      (eval `(in-package ,*system-package-name*))
		      (format t "~%;;;~%;;; Disksave filename ~a.~%;;; ~a~%;;;~%" *disksave-filename* *disksave-datetime*)
		      ;;(patch)
		      (funcall restart-f))))))

(defun dodisk-kind (name restart-f)
  (setf *disksave-config* (string name))
  (dodisk-aux 
   #'(lambda ()
       (setf *reset-orb-name-hook*
	     #'(lambda ()
		 (let ((e (match-environment-in-list (list name) *component*)))
		   (let ((addr (list *sys-version* (orb-name) *component-kind* name)))
		     (setf (environment-base-address e) addr
			   (environment-base-address-parameters e) (toks-to-parameters addr)) ))))
       (funcall restart-f))
   (intern-system (concatenate 'string (string name) "_" (sortable-datetime-string (get-universal-time))))
   (or *disksave-version* 0)))
  

(defun dodisk (restart-f &optional overwrite name)
  (let ((diskpath (system-disk-path)))

    (unless (probe-file diskpath)
      (format t "~%diskpath does not exist ~a~%" diskpath)
      (raise-error (error-message '(dodisk diskpath not) diskpath)))


    (set-reload-global-alist)
    (let ((disksave-version-fname (system-version-filename)))
      (let ((disksave-version (read-disksave-version disksave-version-fname)))

	(let ((disksave-fname (prl-make-filename diskpath nil
						 (disksave-filename disksave-version name)))

	      (disksave-datetime (sortable-datetime-string (get-universal-time))))
      
	  #+(and LINUX86 :ALLEGRO-V4.3)
	  (setf disksave-fname
		(complete-local-path nil ;;(list "local-disksaves")
				     (disksave-filename disksave-version)))

	  (when (and (not overwrite) (probe-file disksave-fname))
	    (incf disksave-version)
	    (write-disksave-version disksave-version disksave-version-fname)
	    (setf disksave-fname (prl-make-filename diskpath nil
						    (disksave-filename disksave-version))))

	  (setf *disksave-filename* disksave-fname
		*disksave-datetime* disksave-datetime)
	  (setf *disksave-version* disksave-version)


	  ;; fail if exists
	  (when (and (not overwrite) (probe-file disksave-fname))
	    (raise-error (error-message '(dodisk save-version odd) diskpath disksave-version)))

	  (format t "~%;;;~%;;; Disksave filename ~a.~%;;;~%" disksave-fname)
	  (format t "~%;;;~%;;; Disksave filename ~a.~%;;; ~a~%;;;~%" disksave-fname disksave-datetime)

	  ;;#+cmu(extensions:save-lisp disksave-fname)
	  ;;#-cmu
	  (image-save disksave-fname
			   #'(lambda ()
			       (eval `(in-package ,*system-package-name*))
			       (format t "~%;;;~%;;; Disksave filename ~a.~%;;; ~a~%;;;~%" *disksave-filename* *disksave-datetime*)
			       ;; patch needs to be caled after system-root reset.
			       ;;(patch)
			       (funcall restart-f)
			       ))

	  )))))


;;;;	
;;;;	
;;;;	

(defun go-client ()
  (unless (match-bus-environment-p (list (ml-text "lib_env")))
    (with-environment ((address-of-environment (match-environment-in-list (list '|ORB|) *component*)))
      (with-appropriate-transaction (nil nil)
	(funmlcall (ml-text "go_open") *component-kind*)))))

;; if lib then start_accept library_open
;; if ref then myopen.
;; if edd then myopen then win.
(defun go-go ()
  (case *component-kind*
    ((|www| www) (go-www))
    ((|FDL| fdl) (go-client))
    (|edd| (go-edd))
    (|lib| (go-lib))
    (|ref| (go-client))
    (|ref501| (go-client))))

(defun stop-client ()
  ;; gross kludge, proper method would be too map over non orb environments
  ;; and pass address to stop_close.
  (let ((env (match-environment (list (intern-system (ml-text "lib_env"))))))
    (with-environment-actual env
      (with-transaction-client (t)
	(ml-text "stop_close ()"))))

  (quit-request-loop)
  (toploop-remove-bus-link))

(defun stop ()
  (case *component-kind*
    ((|www| www) (stop-client))
    ((|FDL| fdl) (stop-client))
    (|edd| (stop-client))
    (|lib| (stop-lib))
    (|ref| (stop-client))
    (|ref501| (stop-client)))
  
  ;;(orb-cleanup t)
  ;;(update-process-activity-log "stop")
  ;;(quit)
  )

#+cmu
(defun read-eval-print-loop ()
  (let ((wot 
	 (catch '%end-of-the-world
	   (loop
	     (lisp::%top-level)
	     (write-line "You're certainly a clever child.")))))
    (unix:unix-exit wot)))


;; beware for cmu: for some reason if edd-restart calls db-root-path
;; such that extend-pathname is called then the room below crashes cmu at restart.
;; modified find-db-path to avoid extend-pathname.
(defun edd-expand (profile)

  (if t
      (progn
	(format t "~%;;;~%;;;~tExpand for edd ~a~%;;;~%" profile)
	(case profile
	  (laptop	(large-expand))
	  (cs671	(large-expand))
	  (lite		(large-expand))
	  (nps		(cl-user::expand-space 128 96))
	  (otherwise	(huge-expand)))
	(room))
      (format t "~%;;;~%;;;~tActually skipped Expand for edd~%")))

(defun ref-expand (profile)
  (format t "~%;;;~%;;;~tExpand for ref ~a~%;;;~%" profile)
  (case profile
	(laptop (expand))
	(lite   (expand))
	(nps (cl-user::expand-space 64 48))
	(otherwise (expand)))
  (room)
  )

(defun lib-expand (profile)
  (format t "~%;;;~%;;;~tExpand for lib ~a~%;;;~%" profile)
  (case profile
	(laptop (large-expand))
	(lite (large-expand)
	      ;;(cl-user::expand-space 64 64)
	      )
	(nps (cl-user::expand-space 64 64))
	(otherwise (very-large-expand)))
  (room)
  )

