%
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************
%

%
************************************************************************
************************************************************************
Level Expression manipulations.
************************************************************************
************************************************************************
%

%
-----------------------------------------------------------------------
level expression simplification
-----------------------------------------------------------------------
%

let mk_simple_level_exp v_i_prs = 
  let mk1 (v,i) = 
    if v = unit_le_id then mk_const_level_exp (i+1)
    if i = 0          then mk_var_level_exp v 
                      else mk_inc_level_exp (mk_var_level_exp v) i
  in
  let maybe_thin_floor vis = 
    let (v,i).vis' = vis
    in 
    if not v = unit_le_id or null vis' then 
      vis
    if i LE reduce1 max (map snd vis') then
      vis'
     else vis
  in      
  if null v_i_prs then failwith `mk_simple_level_exp: null input` else
  let les = map mk1 (maybe_thin_floor v_i_prs)
  in
    if null (tl les) then hd les
    else make_max_level_expression les
;;


let simplify_level_exp le = 
  mk_simple_level_exp (dest_level_exp le) 
;;





%
-----------------------------------------------------------------------
level expression matching 
-----------------------------------------------------------------------
Unless otherwise stated, all level expressions are represented as lists of
form:

  [`(1)`,k
  ;v1,i1
  ...
  ;vn,in
  ]

where n>=0, and k >= max(i1...in)

k+1 is then the `floor' value of the level expression. For any 
instantiation of v1...vn with constants >=1, the value of the expression
is always greater than this floor value.

All functions are supposed to maintain the invariants about the position
of the `(1)`, and the value of k.

Another invariant is that v1...vn are distinct. 
%

% 
Input: list of substitutions s1 ... sn where
       si = v1,e1 ... vm,em  v = variable, e = level expression.

Output: bindings sorted by variable.
       b1 ... bj where
       bi = v, ei-1 ... ei-ki
%

%
Two versions. First is much harder to understand,
and returns wrong order
%

%
let lem_semi_merge_alists as =
  accumulate
    (\y a.
       accumulate
         (\z (v,e).
           modify_or_add_alist_entry
             ($. e)
             [e]
             z
             v
         )
         y
         a
    )
    []
    as
;;
%

let lem_semi_merge_alists as = 
   group_alist_entries (flatten as)
;;


% 
Input: list of level expressions
Output: level expression.
%

let lem_intersect_exps exps =
  reduce1
    (\e1 e2.
        mapfilter
          (\v,i. v, min i (apply_alist e2 v))
          e1
    )
    exps
;;

let lem_union_exps exps =
  map
    (id # reduce1 max)
    (lem_semi_merge_alists exps)
;;

% 
Input list of le substitutions
Output: single le substitution
%


let lem_intersect_subs subs =
  map 
    (id # lem_intersect_exps)
    (lem_semi_merge_alists subs)
;;


let lem_union_subs subs =
  map 
    (id # lem_union_exps)
    (lem_semi_merge_alists subs)
;;




%
Returns sub s.t. pat sub <= inst
Fails if no such sub.

Uses invariant that constant inc occurs first
and is floor of level exp for both pat and inst.

Does generate sub with exps in normal form. (All exps have some initial 
floor element)

Never fails to get match if floor(pat) =< floor(inst)
%


let lem_lower_match pat inst =
  let ((),kp).pat' = pat 
  in let ((),ki).inst' = inst 
  in
  if ki < kp then
    failwith `lem_lower_match: floor of pat too high`
  else
  % max (increments of pat) =< kp =< ki %
  map
    (id
     #
     \ip.
         mapfilter

         % this never fails on first element of inst, since
           on this element ii = ki which is >= any ip
         %
         (id # \ii. if ii < ip then fail else ii-ip)
         inst
    ) 
    pat'
;;

%
lem_check_upper_sub pat inst sub

returns true if

    1.  domain sub >= vars pat  ie. sub covers pat
and 2.  pat sub >= inst

1 included to ensure that sub is full. (Seems a desirable property.
want to avoid left over pat le vars after sub.)
%

%
Uses invariant that floor in e is first element
%

let lem_sub_dom_covers_exp s e =
  subset (map fst (tl e)) (map fst s)
;;

let lem_is_upper_sub sub pat inst =
  level_exp_less_equal
    (mk_level_exp inst)
    (level_exp_subst (map (id # mk_level_exp) sub) (mk_level_exp pat))
;;


let lem_check_upper_sub pat inst sub =
  lem_sub_dom_covers_exp sub pat
  & lem_is_upper_sub sub pat inst
;;



%
Returns sub s.t. pat sub >= inst
Never fails. Will not necessarily return sub s.t. pat sub = inst if
such a sub exists.

Each exp in sub satisfies invariant about floor element.
%

let lem_upper_match pat inst =
  map
    (\u,j.
       u
       ,
       map
         (\v,k.if k < j then v,0 else v,(k-j))
         inst
    ) 
    pat
;;


%
Returns sub', on domain disjoint from sub  s.t. pat (sub + sub') >= inst.
Fails if no such sub. Tries first to use lem_lower_match to find
an equality match . If this fails, then uses lem_upper_match to find 
match poss with inequality.

Always called with sub not covering every var in pat,
so lem_upper_match always returns a good sub', one with at least one
var binding that guarantees that pat (sub + sub') >= inst.

Uses invariant that pat has floor pair as first element.

%

let lem_get_upper_match_inc pat inst sub =
  let pat' = 
    hd pat
    .
    mapfilter
      (\v,i. if is_bound v sub then fail else v,i)
      (tl pat)
  in
  let sub' = lem_lower_match pat' inst ? []
  in
  if lem_check_upper_sub pat inst (sub' @ sub) then
    sub'
  else
    lem_upper_match pat' inst
;;

%
Hopefully, always returns a sub with expressions in normal form.
%

let lem_match_dset upper_p_i_prs lower_p_i_prs =
  let lower_sub =
    lem_intersect_subs
    (map (uncurry lem_lower_match) lower_p_i_prs)
  in
  % lower sub should have expressions satisfying invariant, so
    don't need to check that they are not null lists.
  %
  if 
    not all (lem_sub_dom_covers_exp lower_sub) (map fst lower_p_i_prs) 
  then  
     failwith `lem_match_dset: level exp lower match failed`
  else
  let covered_upper_p_i_prs,uncovered_upper_p_i_prs =
    divide_list
      (\p,i. lem_sub_dom_covers_exp lower_sub p)
      upper_p_i_prs 
  in
  if not all (\p,i. lem_is_upper_sub lower_sub p i) covered_upper_p_i_prs then
    failwith `lem_match_dset: level exp upper match failed`
  else
  let upper_sub_inc =
    lem_union_subs
    (map (\p,i. lem_get_upper_match_inc p i lower_sub)
         uncovered_upper_p_i_prs
    )
  in
    upper_sub_inc @ lower_sub
;;

