%
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************
%

%[
****************************************************************************
****************************************************************************
PRIMITIVES.ML
****************************************************************************
****************************************************************************
This file contains construction and analysis functions for terms and
term parts. (operators, bound terms and variables.)

Terms fall into 2 basic classes:
1. terms of Nuprl's type theory - called prl-terms from here on.
2. system terms -called sys-terms from here on.

sys-terms are used for constructing the super-structure of objects in a
library. They can usually be distinguished by having an opid starting
with a ! character.

As of Feb 94:

Abstract and primitive terms now have the same structure.
(It used to be that abstractions had opid `abstraction` and the first 
parameter was always a token parameter which named the abstraction.  )
Abstract terms are now only distinguished by having an unfolding 
defined in some abstraction object.

Functions dealing with specific terms are defined mostly for
compatibility with Nuprl V3. Functions dealing with term classes
should be used preferentially. Some of the specific functions for
common terms (e.g. vars) do come in handy.

Word Usage
~~~~~~~~~~
primitive terms: any prl-terms non-abstract prl-terms.

Naming Conventions
~~~~~~~~~~~~~~~~~~
fo : first order
so : second order (including first order).

Sometimes functions just deal with fo terms will drop the fo. e.g. dest_var.

]%

%[
******************************************************************************
Defined in lisp
******************************************************************************

destruct_term : term -> (tok |#| parameter list) |#| ((variable list |#| term) list)
parameters_of_term : term -> parameter list
arity_of_term : term -> int list
operator_id_of_term : term -> tok
parameter_of_term : term -> int -> parameter
natural_parameter_of_term : term -> int -> int
token_parameter_of_term : term -> int -> tok
variable_parameter_of_term : term -> int -> variable
operator_of_term : term -> tok |#| (parameter list)
bound_terms_of_term : term -> (variable list |#| term) list
subterm_of_term : term -> int -> term
bindings_of_subterm_of_term : term -> int -> variable list
make_term : (tok |#| (parameter list)) -> (variable list |#| term) list -> term
tok_to_variable : tok -> variable
variable_to_tok : variable -> tok
type_of_parameter : parameter -> tok
destruct_natural_parameter : parameter -> int
make_natural_parameter : int -> parameter
destruct_token_parameter : parameter -> tok
make_token_parameter : tok -> parameter
destruct_variable_parameter : parameter -> variable
make_variable_parameter : variable -> parameter
destruct_parameter_variable : parameter -> tok
make_parameter_variable : tok -> tok -> parameter
term_to_ugly_term : term -> tok
abstraction_definition_of_object : tok -> term # term
abstraction_definition_of_term : term -> term # term
name_of_abstraction_object_of_term : term -> tok

free_variables : term -> variable list
second_order_free_variables : term -> (variable # int) list
first_order_substitute : term -> (variable # term) list -> term
second_order_substitute
   : term -> (tok # parameter) list -> (variable # (variable list # term)) list -> term
quick_second_order_substitute
   : term -> (tok # parameter) list -> (variable # (variable list # term)) list -> term

destruct_assumption : declaration -> variable # term
make_assumption : variable -> term -> declaration
]%

% common defs %
let subtermn i t = subterm_of_term t i;;
let parametern i t = parameter_of_term t i;;
			      

%[
******************************************************************************
Variable manipulation
******************************************************************************
]%

lettype var = variable ;;

let tok_to_var = tok_to_variable ;;
let var_to_tok = variable_to_tok ;;

let mkv = tok_to_var ;;

let null_var = string_to_variable "" %dummy_variable% ;;

let is_null_var x = x = null_var ;;

let cons_if_real_var v vs = if is_null_var v then vs else v.vs ;;
let remove_null_vars vs =
  remove_if is_null_var vs ;;

% 

Meta variables (or mvars) have type var, but different `flavour': 
  1. display : 
  2. abstraction: 

disp-mvars are used on the rhs of display forms for binding vars which 
correspond to slots on the lhs.

Both are used in constructing meta parameters.

NB: meta_variable_type: var -> tok ; one of DISPLAY, ABSTRACTION, or NIL

Normal variables are still created by tok_to_var.
var_to_tok gives id of any variable.
%

%New:%

let mk_ab_mvar =  make_abstraction_meta_variable ;;
let mk_disp_mvar =  make_display_meta_variable ;;

let is_ab_mvar v = meta_variable_type v = `ABSTRACTION` ;;
let is_disp_mvar v = meta_variable_type v = `DISPLAY` ;;
let is_mvar v = is_ab_mvar v or is_disp_mvar v ;;

% assume not meta...%

let var_lt a b = 
  tok_lt (var_to_tok a) (var_to_tok b)
;;

%[
******************************************************************************
Level Expression manipulation
******************************************************************************
]%

lettype level_exp = level_expression ;;
%
defined in lisp:

normalize_level_expression : level_expression -> level_expression
equal_level_expression     : level_expression -> level_expression -> bool
less_level_expression      : level_expression -> level_expression -> bool
   ;; true when first expr is less than second.
equal_less_level_expression      : level_expression -> level_expression -> bool
   ;; true when first expr is less than or equal to second.
substitute_in_level_expression :
    level_expression -> (tok |#| level_expression) list -> level_expression
%

let normalize_level_exp = normalize_level_expression ;;

let level_exp_equal = equal_level_expression ;;
let level_exp_less = less_level_expression ;;
let level_exp_less_equal = equal_less_level_expression ;;

let level_exp_subst sub t = substitute_in_level_expression t sub ;;

let mk_max_level_exp = make_max_level_expression ;;
let mk_inc_level_exp = make_increment_level_expression;;
let mk_var_level_exp = make_variable_level_expression;;
let mk_const_level_exp = make_constant_level_expression;;

let dest_max_level_exp = destruct_max_level_expression ;;
let dest_inc_level_exp = destruct_increment_level_expression;;
let dest_var_level_exp = destruct_variable_level_expression;;
let dest_const_level_exp = destruct_constant_level_expression;;


% 
Normalized ML format for level expressions is 

[`(1)`,k;v1,i1;...;vn,in] n >= 0

`(1)`,k is interpreted as the constant level expression k+1.

k+1 is the floor of the level expression, the min value it ever could have.
Always true that k >= max i1...in.
Also, in normalized le's v1...vn are always distinct.

dest_level_exp always returns le's in this format. 
mk_level_exp is more liberal in what it accepts as input. If given
normalized ml format le, returns normalized le.
%

let unit_le_id = `(1)` ;;
let base_level = mk_const_level_exp 1 ;;


let mk_level_exp v_i_prs = 
  let mk1 (v,i) = 
    if v = unit_le_id then
      mk_const_level_exp (i+1)
    else
      mk_inc_level_exp (mk_var_level_exp v) i
  in
    make_max_level_expression (map mk1 v_i_prs)
;;



% Always returns level exps in normal form %

let dest_level_exp e =
  let dest1 le = 
    ((dest_var_level_exp # id) o dest_inc_level_exp) le
    ?
    unit_le_id, dest_const_level_exp le - 1
  in
 (map dest1
  o dest_max_level_exp
  o normalize_level_exp
 ) e
;;


  
%[
******************************************************************************
Bound term manipulation 
******************************************************************************
A bound term is a pair of a variable list and a term. Free occurrences in the
term of listed variables are considered bound in the bound term. The
binding variables are listed in outer to inner order. Usually we will assure
that no variables are duplicated so we need not be concerned about this order.

Might want to make bterm an abstract type.
]%

lettype bterm = variable list # term ;;

let term_of_bterm = snd ;;
let bvars_of_bterm = fst ;;
let term_to_bterm t = [],t ;;
let bterm_to_term (vs,t) = if null vs then t else failwith `bterm_to_term` ;;

let terms_of_bterms bterms = map snd bterms ;;
let terms_to_bterms terms = map (\t.[],t) terms ;;
let bterms_to_terms bterms = map bterm_to_term bterms ;;

let arity_of_bterm t = length (fst t) ;;

let is_fo_bterm t = null (fst t) ;;
let is_non_fo_bterm t = not (null (fst t)) ;;

let dest_bterm = id ;;
let mk_bterm = pair ;;
%[
******************************************************************************
Parameter manipulation 
******************************************************************************
]%

lettype parm = parameter ;;

let dest_token_parm = destruct_token_parameter ;;
let dest_string_parm = destruct_string_parameter ;;
let dest_natural_parm = destruct_natural_parameter ;;
let dest_variable_parm = destruct_variable_parameter ;;
let dest_level_exp_parm = destruct_level_expression_parameter ;;
let dest_bool_parm = destruct_bool_parameter ;;

let type_of_parm = type_of_parameter ;;

let is_natural_parm p = type_of_parameter p = `natural` ;;
let is_string_parm p = type_of_parameter p = `string` ;;
let is_token_parm p = type_of_parameter p = `token` ;;
let is_variable_parm p = type_of_parameter p = `variable` ;;
let is_level_exp_parm p = type_of_parameter p = `level-expression` ;;
let is_bool_parm p = type_of_parameter p = `bool`;;

let mk_natural_parm i = make_natural_parameter i ;;
let mk_token_parm tok = make_token_parameter tok ;;
let mk_string_parm s = make_string_parameter s ;;
let mk_variable_parm var = make_variable_parameter var ;;
let mk_level_exp_parm le = make_level_expression_parameter le ;;
let mk_bool_parm b = make_bool_parameter b ;;

% 
meta parameters come in 2 flavours:
  1. display meta: used on rhs of display forms 
  2. ab meta: used on lhs and rhs of abstractions

NB: any level expression consisting of a single level exp var, is considered
to be an ab-meta parameter of level-exp type.
%


let id_of_mparm p = var_to_tok (destruct_meta_parameter p) ;;

let is_mparm = meta_parameter_p ;;

let is_level_expression_parameter p = `level-expression` = (type_of_parameter p);;

let is_ab_level_mparm p =
  ( is_level_expression_parameter p
  & variable_level_expression_p (destruct_level_expression_parameter p))
 ? false
;;

let is_ab_mparm p =
  (  if (is_mparm p)
	then (is_ab_mvar (destruct_meta_parameter p))
	else false % (is_ab_level_mparm p) twould seem we'd want this but messes stuff up. % 
  ) ? false
;;

let is_disp_mparm p = 
  ((is_mparm p) & is_disp_mvar (destruct_meta_parameter p)) ? false
 ;;

let mk_ab_mparm type id = make_meta_parameter type (mk_ab_mvar id) ?
    failwith (`mk_ab_mparm: can\'t make meta-parameter: ` ^ id);;
let mk_disp_mparm type id = make_meta_parameter type (mk_disp_mvar id) ?
    failwith (`mk_disp_mparm: can\'t make meta-parameter: ` ^ id);;



% 
For backward compatibility. These definitions should be phased out sometime.
%

let id_of_parm_variable = id_of_mparm ;;
let is_parm_variable = is_ab_mparm ;;
let mk_parm_variable = mk_ab_mparm ;;

%
The original bindings. The primitive functions on the rhs no longer exist.
%

%
let id_of_parm_variable = destruct_parameter_variable ;;
let is_parm_variable p = (id_of_parm_variable p; true) ? false ;;
let mk_parm_variable = make_parameter_variable ;;
%

% parameter ordering functions %

let parm_types = 
  ``
  natural
  string
  token
  variable
  level-expression
  bool
  ``
;;

let parm_lt a b = 
  let ta = type_of_parameter a and tb = type_of_parameter b
  in let nta = position ta parm_types and ntb = position tb parm_types
  in 
  if nta < ntb then true
  if ntb < nta then false
  if ta = `natural` then
    dest_natural_parm a < dest_natural_parm b
  if ta = `string` then
    string_lt (dest_string_parm a) (dest_string_parm b)
  if ta = `token` then
    tok_lt (dest_token_parm a) (dest_token_parm b)
  if ta = `variable` then
    tok_lt 
      (var_to_tok (dest_variable_parm a)) 
      (var_to_tok (dest_variable_parm b))
  if ta = `level-expression` then
    level_exp_less (dest_level_exp_parm a) (dest_level_exp_parm b)
  if ta = `bool` then
    not (dest_bool_parm a) & (dest_bool_parm b)
  else
    failwith `parm_lt: unrecognized parameter type`
;;


%[
******************************************************************************
Primitive term database
******************************************************************************
]%
%
Missing special case of `variable` which doesn't
have fixed arity. 
%

let primitive_opid_parm_types_and_arities =
[`axiom`, [], [];
 `equal`, [], [0;0;0];
 `sqequal`, [], [0;0];
 `term_sq`, [], [];
 `universe`, [`level-expression`], [];
 `U`, [`natural`], [];
 `void`, [], [];
 `any`, [], [0];
 `atom`, [], [];
 `token`, [`token`], [];
 `int`, [], [];
 `object`, [], [];
 `natural_number`, [`natural`], [];
 `minus`, [], [0];
 `add`, [], [0;0];
 `subtract`, [], [0;0];
 `multiply`, [], [0;0];
 `divide`, [], [0;0];
 `remainder`, [], [0;0];
 `ind`, [], [0;2;0;2];
 `list`, [], [0];
 `nil`, [], [];
 `cons`, [], [0;0];
 `list_ind`, [], [0;0;3];
 `union`, [], [0;0];
 `inl`, [], [0];
 `inr`, [], [0];
 `decide`, [], [0;1;1];
 `product`, [], [0;1];
 `pair`, [], [0;0];
 `spread`, [], [0;2];
 `function`, [], [0;1];
 `rfunction`, [], [0;2];
 `isect`, [], [0;1];
 `lambda`, [], [1];
 `apply`, [], [0;0];
 `quotient`, [], [0;2];
 `set`, [], [0;1];
 `less_than`, [], [0;0];
 `atom_eq`, [], [0;0;0;0];
 `int_eq`, [], [0;0;0;0];
 `less`, [], [0;0;0;0];
 `rec_ind`, [], [0;2];
 `rec`, [], [1];
 `tag`, [`natural`], [0]
] 
;;


% Will later be an ML primitive. %

let primitive_opids = 
 `variable` . map fst primitive_opid_parm_types_and_arities 
;;

let is_primitive_opid id = 
   member id primitive_opids
;;

%[
******************************************************************************
Basic term functions.
******************************************************************************
These functions do not make check whether term opids and arities correspond
to those recognized by any part of the system.

defined in lisp:

arity_of_term
]%



%[
constructors
-----------
]%

let mk_term = make_term ;;

%[
destructors
-----------
]%

let dest_term = destruct_term ;;

let dest_term_with_opid opid t =
  let (id,ps), bts = dest_term t in
    if id = opid then
      ps,bts
    else 
      failwith `dest_term_with_opid`
;;

let op_of_term = fst o dest_term ;;
let id_of_op = fst ;;
let parms_of_op = snd ;;

let opid_of_term = operator_id_of_term ;;

let parms_of_term = parameters_of_term ;;

let parm_of_term t i = 
  nth i (parms_of_term t) 
;;

let bterms_of_term = snd o dest_term ;;

let subterms_of_term t = map snd (bterms_of_term t) ;; 
let subterms = subterms_of_term ;;
let sub_term = subtermn;;
let subterm = subterm_of_term;;

let bvars_of_term t i =
  fst (nth i (bterms_of_term t))
;;

let bvar_of_term t i j =
  nth j (bvars_of_term t i)
;;

%[
recognisers
~~~~~~~~~~~
Assume all terms can be indentified by opid alone...
]%

let is_term opid t = opid_of_term t = opid ;;
let is_terms opids t = member (opid_of_term t) opids ;;

% Convenient to allow new `primitives' to be introduced
  by abstraction objects.
%

let is_abstraction_term = expandable_abstraction_instance_p
;;

let is_ab_term = is_abstraction_term
;;



%[
******************************************************************************
Manipulation functions for classes of terms.
******************************************************************************
Classes of terms dealt with are:

std: standard terms: no parameters.
simple: no parameters or binding variables.
np: has 1 natural parameter and no subterms.
tp: has 1 token parameter and no subterms.
sp: has 1 string parameter and no subterms.
lp: has 1 level-expression parameter and no subterms.
]%

%[
destructors
~~~~~~~~~~~
]%
let dest_std_term t = (let (opid,[]),bterms = dest_term t in opid,bterms
                      ) ? failwith `dest_std_term`
;;

let dest_std_term_with_opid opid t =
     if is_term opid t then bterms_of_term t 
     else failwith `dest_std_term_with_opid`
;;

let dest_simple_term t = (let (opid,[]),bterms = dest_term t 
                          in opid, (bterms_to_terms bterms)
                         ) ? failwith `dest_simple_term`
;;

let dest_simple_term_with_opid opid t =
     if is_term opid t then subterms_of_term t 
     else failwith `dest_simple_term_with_opid`
;;

let dest_np_term t =
  (let (opid,[p]),[] = dest_term t in
     opid, dest_natural_parm p
  ) ? failwith `dest_np_term`
;;

let dest_lp_term t =
  (let (opid,[p]),[] = dest_term t in
     opid, dest_level_exp_parm p
  ) ? failwith `dest_lp_term`
;;

let dest_tp_term t =
  (let (opid,[p]),[] = dest_term t in
     opid, dest_token_parm p
  ) ? failwith `dest_tp_term`
;;

let dest_sp_term t =
  (let (opid,[p]),[] = dest_term t in
     opid, dest_string_parm p
  ) ? failwith `dest_sp_term`
;;

%[
constructors
~~~~~~~~~~~~
]%

let mk_std_term opid bterms = mk_term (opid,[]) bterms ;;

let mk_simple_term opid subterms = 
  mk_term (opid,[]) (terms_to_bterms subterms)
;;

let mk_np_term opid i =
  mk_term (opid,[mk_natural_parm i]) []
;;

let mk_tp_term opid i =
  mk_term (opid,[mk_token_parm i]) []
;;

let mk_lp_term opid i =
  mk_term (opid,[mk_level_exp_parm i]) []
;;

let mk_sp_term opid i =
  mk_term (opid,[mk_string_parm i]) []
;;


%[
abbreviations
~~~~~~~~~~~~~
]%

let mk_tm = mk_simple_term ;;
let dest_tm = dest_simple_term ;;


%[
******************************************************************************
Recognisers for specific terms 
******************************************************************************
Assume for the most part that primitive terms can be identified by opid alone.
(Except for variables: check arity).
]%

let mk_primitive_op_instance_recognizer opid t = 
  opid_of_term t = opid
;;

let is_axiom_term = mk_primitive_op_instance_recognizer `axiom` 
% redefined later
and is_U_term =  mk_primitive_op_instance_recognizer `U` 
% and is_void_term = mk_primitive_op_instance_recognizer `void` 
and is_any_term =  mk_primitive_op_instance_recognizer `any` 
and is_atom_term = mk_primitive_op_instance_recognizer `atom` 
and is_token_term = mk_primitive_op_instance_recognizer `token` 
and is_int_term = mk_primitive_op_instance_recognizer `int` 
and is_object_term = mk_primitive_op_instance_recognizer `object` 
and is_natural_number_term = mk_primitive_op_instance_recognizer `natural_number` 
and is_minus_term = mk_primitive_op_instance_recognizer `minus` 
and is_add_term =  mk_primitive_op_instance_recognizer `add` 
and is_subtract_term = mk_primitive_op_instance_recognizer `subtract` 
and is_multiply_term = mk_primitive_op_instance_recognizer `multiply` 
and is_divide_term = mk_primitive_op_instance_recognizer `divide` 
and is_remainder_term = mk_primitive_op_instance_recognizer `remainder` 
and is_ind_term = mk_primitive_op_instance_recognizer `ind` 
and is_list_term = mk_primitive_op_instance_recognizer `list` 
and is_nil_term = mk_primitive_op_instance_recognizer `nil` 
and is_cons_term = mk_primitive_op_instance_recognizer `cons` 
and is_list_ind_term = mk_primitive_op_instance_recognizer `list_ind` 
and is_union_term = mk_primitive_op_instance_recognizer `union` 
and is_inl_term = mk_primitive_op_instance_recognizer `inl` 
and is_inr_term = mk_primitive_op_instance_recognizer `inr` 
and is_decide_term = mk_primitive_op_instance_recognizer `decide` ;;
% break because of Lucid bug %
let is_product_term = mk_primitive_op_instance_recognizer `product` 
and is_pair_term = mk_primitive_op_instance_recognizer `pair` 
and is_spread_term = mk_primitive_op_instance_recognizer `spread` 
and is_function_term = mk_primitive_op_instance_recognizer `function` 
and is_rfunction_term = mk_primitive_op_instance_recognizer `rfunction`
and is_isect_term = mk_primitive_op_instance_recognizer `isect`
and is_lambda_term = mk_primitive_op_instance_recognizer `lambda` 
and is_apply_term = mk_primitive_op_instance_recognizer `apply` 
and is_quotient_term = mk_primitive_op_instance_recognizer `quotient` 
and is_set_term = mk_primitive_op_instance_recognizer `set` 
and is_less_than_term = mk_primitive_op_instance_recognizer `less_than` 
%
and is_term_of_term = mk_primitive_op_instance_recognizer `term_of` 
%
and is_atom_eq_term = mk_primitive_op_instance_recognizer `atom_eq` 
and is_int_eq_term = mk_primitive_op_instance_recognizer `int_eq` 
and is_less_term =  mk_primitive_op_instance_recognizer `less` 
and is_rec_ind_term = mk_primitive_op_instance_recognizer `rec_ind` 
and is_rec_term = mk_primitive_op_instance_recognizer `rec` 
and is_tag_term = mk_primitive_op_instance_recognizer `tag` 
and is_equal_term = mk_primitive_op_instance_recognizer `equal` 
and is_sqequal_term = mk_primitive_op_instance_recognizer `sqequal`
and is_term_sq_term = mk_primitive_op_instance_recognizer `term_sq`
;;

% special cases for variable arity terms %

let is_variable_term t = 
   (let (opid,()),[] = destruct_term t in
      opid = `variable`)
   ? false

and is_so_variable_term t = opid_of_term t = `variable`

;;

% abbreviations %

let is_inteq_term =  is_int_eq_term 
and is_atomeq_term = is_atom_eq_term 
and is_int_less_term =  is_less_term 
and is_var_term = is_variable_term 
and is_so_var_term = is_so_variable_term ;;

let is_fo_var_term = is_var_term ;;
let is_non_fo_var_term t = is_so_var_term t & not is_var_term t ;;
  
%[
******************************************************************************
Specific term destructors
******************************************************************************
]%

let one_subterm t =
  let l = (bound_terms_of_term t) in
    if not (null (tl l)) then failwith `one_subterm_more`;
    snd (hd l)
and two_subterms t =
  let l = (bound_terms_of_term t) in
   ( (snd (hd l))
   , let ll = (tl l) in
      if not (null (tl ll)) then failwith `two_subterms_more`;
      (snd (hd ll)))
and three_subterms t =
  let l = (bound_terms_of_term t) in
   ( (snd (hd l))
   , let ll = (tl l) in
   ( (snd (hd ll))
   , let lll = (tl ll) in
      if not (null (tl lll)) then failwith `three_subterms_more`;
     (snd (hd lll))))
and four_subterms t = 
  let l = (bound_terms_of_term t) in
   ( (snd (hd l))
   , let ll = (tl l) in
   ( (snd (hd ll))
   , let lll = tl ll in
   ( (snd (hd lll))
   , let llll = tl lll in
      if not (null (tl llll)) then failwith `four_subterms_more`;
      (snd (hd llll)))))
;;

let % redefined later
(dest_U: term -> int) t =
  if is_U_term t then dest_natural_parm (hd (parms_of_term t))  
  else failwith `dest_U`
and % (dest_any: term -> term) t =
  if is_any_term t then one_subterm t
  else failwith `dest_any`
and (dest_token: term -> tok) t =
  if is_token_term t then dest_token_parm (hd (parms_of_term t))  
  else failwith `dest_token`
and (dest_natural_number: term -> int) t =
  if is_natural_number_term t then dest_natural_parm (hd (parms_of_term t))      
  else failwith `dest_natural_number`
and (dest_minus: term -> term) t =
  if is_minus_term t then one_subterm t
  else failwith `dest_minus`
and (dest_add: term -> (term # term)) t =
  if is_add_term t then two_subterms t
  else failwith `dest_add`
and (dest_subtract: term -> (term # term)) t =
  if is_subtract_term t then two_subterms t
  else failwith `dest_subtract`
and (dest_multiply: term -> (term # term)) t =
  if is_multiply_term t then two_subterms t
  else failwith `dest_multiply`
and (dest_divide: term -> (term # term)) t =
  if is_divide_term t then two_subterms t
  else failwith `dest_divide`
and (dest_remainder: term -> (term # term)) t =
  if is_remainder_term t then two_subterms t
  else failwith `dest_remainder`
and (dest_ind: term -> (term # (bterm # (term # bterm)))) t =
  if is_ind_term t then (let [(),a; b; (),c; d] = bterms_of_term t in a,b,c,d)
  else failwith `dest_ind`
and (dest_list: term -> term) t =
  if is_list_term t then one_subterm t
  else failwith `dest_list`
and (dest_cons: term -> (term # term)) t =
  if is_cons_term t then two_subterms t
  else failwith `dest_cons`
and (dest_list_ind: term -> (term # (term # bterm))) t =
  if is_list_ind_term t then (let [(),a; (),b; c] = bterms_of_term t in a,b,c)
  else failwith `dest_list_ind`
and (dest_union: term -> (term # term)) t =
  if is_union_term t then two_subterms t
  else failwith `dest_union` ;;
% break because of a Lucid bug %
let (dest_inl: term -> term) t =
  if is_inl_term t then one_subterm t
  else failwith `dest_inl`
and (dest_inr: term -> term) t =
  if is_inr_term t then one_subterm t
  else failwith `dest_inr`
and (dest_decide: term -> (term # (bterm # bterm))) t =
  if is_decide_term t then (let [(),a; b; c] = bterms_of_term t in a,b,c)
  else failwith `dest_decide`
and (dest_product: term -> (variable #  (term # term))) t =
  if is_product_term t then (let [(),a; [x],b] = bterms_of_term t in x,a,b)
  else failwith `dest_product`
and (dest_pair: term -> (term # term)) t =
  if is_pair_term t then two_subterms t
  else failwith `dest_pair`
and (dest_spread: term -> (term # bterm)) t =
  if is_spread_term t then (let [(),a; b] = bterms_of_term t in a,b)
  else failwith `dest_spread`
and (dest_function: term -> (variable # (term # term))) t =
  if is_function_term t then (let [(),a; [x],b] = bterms_of_term t in x,a,b)
  else failwith `dest_function`
and (dest_rfunction: term -> (variable # (variable # (term # term)))) t =
  if is_rfunction_term t then (let [(),A; [f;x],B] = bterms_of_term t in f,x,A,B)
  else failwith `dest_rfunction`
and (dest_isect: term -> (variable # (term # term))) t =
  if is_isect_term t then (let [(),a; [x],b] = bterms_of_term t in x,a,b)
  else failwith `dest_isect`
and (dest_lambda: term -> (variable # term)) t =
  if is_lambda_term t then (let [[x],b] = bterms_of_term t in x,b)
  else failwith `dest_lambda`
and (dest_apply: term -> (term # term)) t =
  if is_apply_term t then two_subterms t
  else failwith `dest_apply`
and (dest_quotient: term -> (variable # (variable # (term # term)))) t =
  if is_quotient_term t then (let [(),a; [x;y],b] = bterms_of_term t in x,y,a,b)
  else failwith `dest_quotient`
and (dest_set: term -> (variable # (term # term))) t =
  if is_set_term t then (let [(),a; [x],b] = bterms_of_term t in x,a,b)    
  else failwith `dest_set`
and (dest_equal: term -> (term # (term # term))) t =
  if is_equal_term t then three_subterms t
  else failwith `dest_equal`
and (dest_sqequal: term -> (term # term)) t =
  if is_sqequal_term t then two_subterms t
  else failwith `dest_sqequal`
and (dest_less_than: term -> (term # term)) t =
  if is_less_than_term t then two_subterms t
  else failwith `dest_less_than`
and (dest_variable: term -> variable) t =
  if is_var_term t then dest_variable_parm (hd (parms_of_term t))  
  else failwith `dest_var`
and (dest_so_variable: term -> (variable # term list)) t =
(  let (opid,[p]), ts = destruct_term t in
   if opid = `variable`  &  all (null o fst) ts 
   then dest_variable_parm p, map snd ts 
   else fail
) ? failwith `dest_so_variable`
%
and (dest_term_of: term -> tok # level_exp list) t =
 (if is_term_of_term t then 
  (let tok_parm.le_parms = parms_of_term t  
   in
     dest_token_parm tok_parm, map dest_level_exp_parm le_parms
  )
  else fail
 )
  ? failwith `dest_term_of`
%
and (dest_atomeq: term -> (term # (term # (term # term)))) t =
  if is_atomeq_term t then four_subterms t    
  else failwith `dest_atomeq`
and (dest_inteq: term -> (term # (term # (term # term)))) t =
  if is_inteq_term t then four_subterms t    
  else failwith `dest_inteq`
and (dest_less: term -> (term # (term # (term # term)))) t =
  if is_less_term t then four_subterms t    
  else failwith `dest_less`
and (dest_rec: term -> (variable#term)) t =
  if is_rec_term t then (let [[x],A] = bterms_of_term t in x,A)
  else failwith `dest_rec`
and (dest_rec_ind: term -> (term#bterm)) t =
  if is_rec_ind_term t then (let [[],e; b] = bterms_of_term t in e,b)
  else failwith `dest_rec_ind`
and (dest_tag: term -> (int # term)) t =
  if is_tag_term t then dest_natural_parm (hd (parms_of_term t)), one_subterm t
  else failwith `dest_tag`
;;

let destroy_tag = one_subterm;; 				  

% abbreviations %

let dest_var = dest_variable
and dv = dest_variable 
and dest_so_var = dest_so_variable 
and var_of_so_var_term = fst o dest_so_variable
and args_of_so_var_term = snd o dest_so_variable
and var_and_arity_of_so_var_term = (id # length) o dest_so_variable
;;

%[
******************************************************************************
Specific Constructors
******************************************************************************
]%

let % redefined later.
     (mk_U_term: int -> term) level =
  make_term (`U`, [mk_natural_parm level]) []
and %mk_void_term = mk_simple_term `void` [] 
and (mk_any_term: term -> term) term = mk_simple_term `any` [term] 
and mk_atom_term = mk_simple_term `atom` [] 
and (mk_token_term: tok -> term) atom = 
  make_term (`token`, [mk_token_parm atom]) [] 
and mk_int_term = mk_simple_term `int` [] 
and (mk_natural_number_term: int -> term) int =
  make_term (`natural_number`, [mk_natural_parm int]) [] 
and (mk_minus_term: term -> term) term = mk_simple_term `minus` [term] 
and mk_add_term t t' = mk_simple_term `add` [t;t'] 
and mk_subtract_term t t' = mk_simple_term `subtract` [t;t'] 
and mk_multiply_term t t' = mk_simple_term `multiply` [t;t'] 
and mk_divide_term t t' = mk_simple_term `divide` [t;t'] 
and mk_remainder_term t t' = mk_simple_term `remainder` [t;t'] 
and (mk_ind_term: term -> (bterm -> (term -> (bterm -> term))))
    value downterm baseterm upterm = 
  make_term (`ind`,[]) [[],value; downterm; [],baseterm; upterm]
and (mk_list_term: term -> term) type = mk_simple_term `list` [type] 
and mk_nil_term = mk_simple_term `nil` [] 
and (mk_cons_term: term -> (term -> term)) head tail = 
  mk_simple_term `cons` [head;tail]
and (mk_list_ind_term: term -> (term -> (bterm -> term))) 
    value baseterm upterm = 
  make_term (`list_ind`,[]) [[],value; [],baseterm; upterm]
and (mk_union_term: term -> (term -> term)) lefttype righttype = 
  mk_simple_term `union` [lefttype; righttype]
and (mk_inl_term: term -> term ) term = mk_simple_term `inl` [term] 
and (mk_inr_term: term -> term) term = mk_simple_term `inr` [term] 
and (mk_decide_term: term -> (bterm -> (bterm -> term))) 
       value leftterm rightterm = 
  make_term (`decide`,[]) [[],value; leftterm; rightterm]
and (mk_product_term: variable -> (term -> (term -> term))) 
  bound_id lefttype righttype = 
   make_term (`product`,[]) [[],lefttype; [bound_id],righttype] ;;
let (mk_pair_term: term -> (term -> term)) leftterm rightterm = 
   mk_simple_term `pair` [leftterm; rightterm] 
and (mk_spread_term: term -> (bterm -> term)) value term = 
  make_term (`spread`,[]) [[],value; term]
and (mk_function_term: variable -> (term -> (term -> term))) 
       bound_id lefttype righttype = 
   make_term (`function`,[]) [[],lefttype; [bound_id],righttype] 
and (mk_rfunction_term: variable -> (variable -> (term -> (term -> term))))
    fun_id bound_id lefttype righttype =
   make_term (`rfunction`,[]) [[],lefttype; [fun_id; bound_id],righttype]
and (mk_isect_term: variable -> (term -> (term -> term)))
   bound_id lefttype righttype =
   mk_term (`isect`,[]) [[],lefttype; [bound_id],righttype]
and (mk_lambda_term: variable -> (term -> term)) bound_id term = 
  make_term (`lambda`,[]) [[bound_id],term]
and (mk_apply_term: term -> (term -> term)) function arg = 
   mk_simple_term `apply` [function; arg]
and (mk_quotient_term: variable -> (variable -> (term -> (term -> term)))) 
    id1 id2 lefttype righttype = 
  make_term (`quotient`,[]) [[],lefttype; [id1;id2],righttype]
and (mk_set_term: variable -> (term -> (term -> term))) 
       bound_id lefttype righttype = 
   make_term (`set`,[]) [[],lefttype; [bound_id],righttype]   
and (mk_equal_term: term -> (term -> (term -> term))) T t t' = 
   mk_simple_term `equal` [T;t;t']
and (mk_sqequal_term: term -> (term -> term)) t t' =
   mk_simple_term `sqequal` [t;t']
and mk_term_sq_term = mk_simple_term `term_sq` []
and mk_axiom_term = mk_simple_term `axiom` [] 
and (mk_less_than_term: term -> (term -> term)) t t' =  
  mk_simple_term `less_than` [t;t']
and (mk_atomeq_term: term -> (term -> (term -> (term -> term)))) a b c d =
   mk_simple_term `atomeq` [a;b;c;d]
and (mk_inteq_term: term -> (term -> (term -> (term -> term)))) a b c d =
   mk_simple_term `inteq` [a;b;c;d]
and (mk_less_term: term -> (term -> (term -> (term -> term)))) a b c d =
   mk_simple_term `less` [a;b;c;d]
and (mk_variable_term: variable -> term) tok = 
  make_term (`variable`, [mk_variable_parm tok]) []
and mk_so_variable_term p args = 
  make_term (`variable`, [mk_variable_parm p]) (map (\x. [],x) args) 
%
and (mk_term_of_term: tok -> level_exp list -> term) name level_exps = 
  make_term (`term_of`
             , (mk_token_parm name.map mk_level_exp_parm level_exps)
            ) 
            []   
%
and mk_object_term = mk_simple_term `object` [] 
and (mk_rec_term: variable -> term -> term) z T = 
  mk_std_term `rec` [[z],T]
and (mk_rec_ind_term: term -> bterm -> term) t b =
  mk_std_term `rec_ind` [[],t;b]
and (mk_tag_term: int -> (term -> term)) tag term = 
  make_term (`tag`,[mk_natural_parm tag]) [[],term]
;;

let mk_var_term = mk_variable_term
and mvt = mk_variable_term 
and mk_so_var_term = mk_so_variable_term
;;
let atom_term = mk_atom_term
and int_term = mk_int_term
and void_term = mk_void_term 
and axiom_term = mk_axiom_term
;;

%
Fixes for new universe terms with level expressions.
%

let mk_U_term le = 
  make_term (`universe`,[mk_level_exp_parm le]) [] 
and dest_U t =
 ((let [p],[] = dest_term_with_opid `universe` t in
   dest_level_exp_parm p 
  ) ? failwith `dest_U`
 )
and is_U_term t = is_term `universe` t
;;

   
  
%[
******************************************************************************
Term well formedness checking
******************************************************************************
A proper term is one of
1. a fixed arity primitive term with correct arity, and parameter types.
2. a variable term with correctly typed parameter and no subterms.
3. an abstraction term.

A proper second order term is one of
1. a proper term
2. a variable term with correctly typed parameter and one or more subterms
   without binding variables.
]%

let is_spec_term opid parm_types arity t =
   opid_of_term t = opid
   & arity_of_term t = arity 
   & map type_of_parm (parms_of_term t) = parm_types  
;;

let is_proper_vo_term so_vars_ok t =
  let (opid,parms),bterms = destruct_term t in
  let arity = map (length o fst) bterms in
  let arity_sum = sum arity in
  let parm_types = map type_of_parm parms 
  in
    member (opid,parm_types,arity) primitive_opid_parm_types_and_arities
  or
    (opid = `variable` & parm_types = [`variable`] & arity_sum = 0 
     & (so_vars_ok or arity = []))
  or 
    opid = `abstraction`
;;

let is_proper_term t = is_proper_vo_term false t ;;
let is_proper_so_term t = is_proper_vo_term true t ;;


%[
******************************************************************************
Sys-term functions
******************************************************************************
]%

let mk_text_term str = 
  mk_term (`!text`,[mk_string_parm str]) []
;;

let dest_text_term t = 
 (let [p],[] = dest_term_with_opid `!text` t
  in
    dest_string_parm p
 )
 ? failwith `dest_text_term`
;;

% Put a prl term wrapper around a term. These
  are used in ML to inject prl terms in ML term sequences.
%

let mk_prl_term t = 
  mk_term (`!prl_term`,[]) [[],t]
;;


let mk_disp_meta_term id = 
  mk_term (`!template`,[mk_string_parm (tok_to_string id)]) [] ;;


%[
************************************************************************
Extract term constructor/destructor/tester
************************************************************************
Assume only extract terms have opids that are names of theorems.

These functions hide the existence of the extra level expression
argument that all extracts take.
]%


letref default_le_var_in_extract_terms = `\\v`;;

let is_extract_term = is_termof
;;




let dest_extract t = 
 (if is_extract_term t then
    let (id,ps),[] = dest_term t
    in 
     (if (id = `TERMOF`)
         then let oid = destruct_object_id_parameter (hd ps) in
	       (oid, map dest_level_exp_parm (tl (tl ps)))
      else fail %id, map dest_level_exp_parm (tl ps)%
      )
  else
    fail
 ) ? failwith `dest_extract: not extract term`
;;

let dest_extract_with_extra_le t = 
 (if is_extract_term t then
    let (id,ps),[] = dest_term t
    in
     (if (id = `TERMOF`)
         then let oid = destruct_object_id_parameter (hd ps) in
	       (oid, map dest_level_exp_parm (tl ps))
      else fail %id , map dest_level_exp_parm ps%
      )
  else
    fail
 ) ? failwith `dest_extract_with_extra_le: not extract term`
;;


let mk_extract_term oid le_exps =
  mk_term 
   (`TERMOF` 
    , ( (make_object_id_parameter oid)
      . map mk_level_exp_parm 
         (mk_var_level_exp default_le_var_in_extract_terms . le_exps))
   ) 
   []
;;




let alpha_equal_bterms x y =
 let xvars, x' = dest_bterm x and yvars, y' = dest_bterm y in
 xvars = yvars & alpha_equal_terms x' y'
;;

% defined in comm-def %
%			 
letrec equal_lists_p x y equal =
 if null x then
   if null y then true
   else false
 else equal (hd x) (hd y) & equal_lists_p (tl x) (tl y) equal
;;
% 

%[
******************************************************************************
Obselete Functions
******************************************************************************
Just in case these functions are used in ML objects in any libraries.
]%



%[
constructors
------------
]%

let mk_ab_term = mk_term ;;
let mk_prim_term = mk_term ;;

let mk_std_ab_term = mk_std_term ;;
let mk_simple_ab_term = mk_simple_term ;;

let mk_std_prim_term = mk_std_term ;;
let mk_simple_prim_term = mk_simple_term ;;

let mk_np_ab_term = mk_np_term ;;
let mk_tp_ab_term = mk_tp_term ;;
let mk_lp_ab_term = mk_lp_term ;;
let mk_sp_ab_term = mk_sp_term ;;

let mk_np_prim_term = mk_np_term ;;
let mk_tp_prim_term = mk_tp_term ;;
let mk_lp_prim_term = mk_lp_term ;;
let mk_sp_prim_term = mk_sp_term ;;
