%
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************
%

%[
************************************************************************
************************************************************************
 SUBSTITUTION
************************************************************************
************************************************************************
]%

%[
************************************************************************
Alpha conversion
************************************************************************
3 versions of functions are defined. e.g. 

1. alpha_conv_term   :   (var # var) list -> term -> term
2. alpha_conv_term_d :   (var # var) list -> term -> term
3. alpha_conv_term_mi:   (var # var) list -> term -> term

1: successive bindings for the same var, are applied to successive occurrences.
2: first binding for each var applied to all occurrences.
3: As 2, with additional renaming done to maintain identities between 
   bound variables in term constructors.
]%



%
nvis : new var int prs. (integers allow unique identification of vars.)
fvis : free var int prs.
csub : (var # (var # int)) list
%


% Recursive procedure to apply an alpha conversion to a term.
  Procedure fails if alpha conversion results in conflict.
%

% assumes var-int mappings in vis1 and vis2 are 1-1. %

letref ac_conflict = (tok_to_var `x`,1,[]) : var # int # var list
;;

let detect_conflict (v,i) fvis = 
  let conflict_is = mapfilter (\w,j.if w = v & not j = i then j else fail) fvis 
  in 
    if null conflict_is then
      false
    else
    ( ac_conflict := v,i,map fst fvis
    ;
      true
    )
;;

letref ac_null_var_index = 0;;

letrec ac_tm tm nvis csub = 
  if is_var_term tm then
  ( let v = dest_var tm 
    in let fvi = (apply_alist csub v ? v,0) 
    in let tm' = (mk_var_term (fst fvi) 
                  ? (ac_null_var_index := snd fvi
                     ; failwith `ac_tm: cannot make null var term`
                    )
                 )
    in 
      tm',nvis,[fvi]
  )
  else
  ( let op,btms = dest_term tm 
    in let btms', nvis', fvis = ac_btms btms nvis csub
    in let tm' = mk_term op btms'
    in 
      tm',nvis',fvis
  )
and ac_btms btms nvis csub = 
  if null btms then
    [],nvis,[]
  else
  ( let btm.btms' = btms 
    in let btm',nvis',fvis = ac_btm btm nvis csub
    in let btms'',nvis'',fvis' = ac_btms btms' nvis' csub
    in 
      (btm'.btms''),nvis'',(fvis @ fvis')
  )  
and ac_btm btm nvis csub =
  let vs,tm = btm 
  in 
  if null vs then
  ( let tm',nvis',fvis = ac_tm tm nvis csub
    in
      ([],tm'),nvis',fvis
  )
  else
  ( let v.vs' = vs 
    in let nvi.nvis' = nvis 
    in let csub' = (v,nvi).csub 
    in let btm',nvis'',fvis = ac_btm (vs',tm) nvis' csub'
    in
    if detect_conflict nvi fvis then
      failwith `ac_btm: conflict`
    else
    let fvis' = remove_if ($= nvi) fvis
    in let btm'' = (fst nvi . (fst btm') , (snd btm'))
    in 
       btm'',nvis'',fvis'
  )
;;

% new_vars are variables to replace (bound_vars t) %

let apply_exact_alpha_conv new_vars t =
  fst (ac_tm t (zip new_vars (upto 1 (length new_vars))) [])
  ?\x 
  if x = `ac_btm: conflict` then
    failwith `apply_exact_alpha_conv: conflict found`
  if x = `ac_tm: cannot make null var term` then
    failwith `apply_exact_alpha_conv: cannot make null var term`
  else
    failwith (`apply_exact_alpha_conv` ^ x)
;;

% This should always terminate since each iteration eliminates a
  conflict and can only introduce conflicts lower in the term tree.
%

letrec apply_alpha_conv new_vars t =
  apply_exact_alpha_conv new_vars t
  ?? [`apply_exact_alpha_conv: conflict found`]
  let v,i,vs = ac_conflict
  in let new_vars' = replace_nth_by i new_vars (new_var v vs)
  in
    apply_alpha_conv new_vars' t
  ?? [`apply_exact_alpha_conv: cannot make null var term`]
  let new_vars' = 
    replace_nth_by 
      ac_null_var_index 
      new_vars 
      (new_var (tok_to_var `x`) new_vars)
  in
    apply_alpha_conv new_vars' t
;;



let format_alpha_conv_sub sub t =
  letrec aux sub bvars =
    if null bvars then []
    else
    let bv.bvars' = bvars 
    in let new_v,sub' = 
         (apply_alist sub bv, remove_alist_entry sub bv ? bv, sub)
    in
      new_v . aux sub' bvars'
  in
    aux sub (bound_vars t)
;;

let alpha_conv_term vsub (t:term) =
  apply_alpha_conv (format_alpha_conv_sub vsub t) t
  ?
  failwith `alpha_conv_term`
;;


let format_alpha_conv_sub_d sub t =
  map
    (\v.apply_alist sub v ? v)
    (bound_vars t)
;;

let alpha_conv_term_d vsub (t:term) =
  apply_alpha_conv (format_alpha_conv_sub_d vsub t) t
  ?
  failwith `alpha_conv_term_d`
;;



% variant on above, maintaining binding variable identities within terms. %

letrec apply_alpha_conv_mi new_vars t =
  apply_exact_alpha_conv new_vars t
  ?? [`apply_exact_alpha_conv: conflict found`]
  let bv_tag_prs = bound_vars_with_tm_tags t
  in let v,i,vs = ac_conflict
  in let oldv,tm_id =  (nth i bv_tag_prs) 
  in let tm_bvs = mapfilter (\v,id.if id = tm_id then v else fail) bv_tag_prs
  in let v' = new_var v (tm_bvs @ vs)
  in let new_vars' = 
    map2 
      (\x (y,id).if id = tm_id & y = oldv then v' else x) 
      new_vars 
      bv_tag_prs
  in
    apply_alpha_conv_mi new_vars' t
  ?? [`apply_exact_alpha_conv: cannot make null var term`]
  let new_vars' = 
    replace_nth_by 
      ac_null_var_index 
      new_vars 
      (new_var (tok_to_var `x`) new_vars)
  in
    apply_alpha_conv new_vars' t
;;

let alpha_conv_term_mi vsub (t:term) =
  apply_alpha_conv_mi (format_alpha_conv_sub_d vsub t) t
  ?
  failwith `alpha_conv_term_mi`
;;



%
************************************************************************
Second Order terms.
************************************************************************
Second order terms are terms with empty slots, which can be filled by
either normal terms or other second order terms. 

Terminology:

Second Order Variables
~~~~~~~~~~~~~~~~~~~~~~

A second order (so) variable is a term with opid `variable`:tok, a
variable parameter and 0 or more subterms. A non first order (fo)
variable has 1 or more subterms.  For example in standard syntax:

variable{<name>:var}(t1;t2;t3)


Second order variables appear in abstraction definitions and are
recognised by the full_match and full_subst functions. 

Second Order Applications 
~~~~~~~~~~~~~~~~~~~~~~~~~

Second order variables cannot appear in proofs. Instead in proofs we use
a special abstraction called so_apply. 

so_apply(v;t1;t2;...;tn) =def apply(...apply(apply(v;t1);t2)...tn)
                           =def v t1 t2 ... tn

The so_apply term:

so_apply(variable{<name>:var};t1;...;tn) 

models the second order variable:

variable{<name>:var}(t1;t2;...;tn).


Second Order Lambdas
~~~~~~~~~~~~~~~~~~~~
The bterms or bound terms which occur in second order matching and 
substitution are modelled by an so_lambda abstraction. the bterm

x1,...,xn.t is modelled by the so_lambda term

so_lambda(x1,...,xn.t)


Second order substitution is accomplished by first order substitution
of so_lambda terms for the variables in so_apply terms, abstraction
expansion and beta reduction.

NB: 1. As of 4 April 1992, so_lambda abstractions are not used everywhere
       so substitution is desired.
    2. Eventually we might encode in the same way, third or higher order
       substitution.

%

%
destructors for so applications of variables
--------------------------------------------
%

let dest_so_apply_of_var t =
 (let var_term.args = dest_simple_term_with_opid `so_apply` t in
    dv var_term, args
 ) ? failwith `dest_so_apply_of_var`
;;

let var_and_arity_of_so_apply_of_var t =
  let var,args = dest_so_apply_of_var t in
    var, length args
;;


let is_so_var_or_so_apply_of_var t =
  is_so_var_term t 
  or ((dest_so_apply_of_var t ; true) ? false)
;;

let var_and_arity_of_so_var_or_so_apply_of_var t =
  var_and_arity_of_so_apply_of_var t
  ? var_and_arity_of_so_var_term t
;;

% 
so-izing functions for terms
~~~~~~~~~~~~~~~~~~~~~~~~~~~~

change so applications of variables in var arity list to true second
order variables.
%

let so_ize_so_apply_of_var so_ap_of_v =
  let v,args =  dest_so_apply_of_var so_ap_of_v in
    mk_so_var_term v args
;;

let so_ize_so_ap_of_var_under_bvars vars_to_consider bvars t =
( let v,args = dest_so_apply_of_var t in
  if member v vars_to_consider & not member v bvars then
    mk_so_var_term v args
  else fail
) ?
    failwith `so_ize_so_ap_of_var_under_bvars`
;;

let so_ize_so_aps_of_vars vars_to_consider t =
  sweep_down_map_with_bvars 
    (so_ize_so_ap_of_var_under_bvars vars_to_consider)
    t
;;

%
so_lambda constructor and destructor
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
These handle first order special cases
%

let bterm_to_so_lambda (xs,t) =
  if null xs then t
  else
    mk_std_term `so_lambda` [xs,t]
;;

let so_lambda_to_bterm t =
 hd (dest_std_term_with_opid `so_lambda` t) ? [],t
;;


%
************************************************************************
Bound variable matching...
************************************************************************
%
%
Walks pattern and instance term, returning list of correspondences between
bound variables.

Unused anywhere as of 12th Sept 92.
%

letrec bvar_match pat inst = 
  if is_so_var_or_so_apply_of_var pat then
    []
  else
  let bts = bterms_of_term pat in
  let bts' = bterms_of_term inst in
   (zip (flatten (map fst bts)) (flatten (map fst bts')) ? [])
   @ flatten
       (map2 bvar_match (map snd bts) (map snd bts'))
;;


%
Generate preorder newvar sequence for instance term, given 
pattern term, and substitution for (some of) bvars in pattern term.
%

letrec format_bvar_pl_from_pat sub pat inst =
  if is_so_var_or_so_apply_of_var pat 
     or is_apply_term pat & not is_apply_term inst
  then
    bound_vars inst
  else
  let bts = bterms_of_term pat in
  let bts' = bterms_of_term inst in
    flatten
      (map2
        (\(vs,t) (vs',t').
          map (\v.apply_alist sub v ? v) vs @ format_bvar_pl_from_pat sub t t'
        )
        bts
        bts'
        ?? ``map2`` failwith `format_bvar_pl_from_pat`
      )
;;



let level_exp_sub_to_parm_sub s =
    map (id # mk_level_exp_parm) s
;;
let parm_sub_to_term_sub s =
  map (tok_to_var # mk_parm_term) s
;;
let level_exp_sub_to_term_sub = 
  parm_sub_to_term_sub o level_exp_sub_to_parm_sub 
;;

let term_sub_to_parm_sub s =
  map (var_to_tok # (hd o parms_of_term)) s
;;
let parm_sub_to_level_exp_sub s =
    map (id # dest_level_exp_parm) s
;;
let term_sub_to_level_exp_sub =
  parm_sub_to_level_exp_sub o term_sub_to_parm_sub
;;

let parm_term_for_null_var =
  mk_parm_term (mk_variable_parm (tok_to_var `%**`))
;;

let mk_bvar_binding (v,v') = 
  if v' = null_var then
    v, parm_term_for_null_var
  else
    v,mk_parm_term (mk_variable_parm v')
;;

let dest_bvar_binding (v,t) = 
  if alpha_equal_terms t parm_term_for_null_var then
    v, null_var
  else
    v, dest_variable_parm (dest_parm_term t)
;;


%
Returns one of:

`natural`
`variable`
`level-expression`
`token`
`regular`
%

let sub_kind_of_term t =
  type_of_parm_term t
  ?
  `regular`
;;

%
************************************************************************
Utility functions
************************************************************************
%
let bterm_to_lambdas_term (bvs,t) =
  mk_iterated_lambda bvs t ;;

let lambdas_term_to_bterm = dest_iterated_lambda 
;;

% unused... delete? %

let lambdas_term_to_bterm_with_arity arity t =
  let vs,t = lambdas_term_to_bterm t in
  let n = length vs in
  if n = arity then vs,t
  if n < arity then failwith `so_lambdas_term_to_bterm`
  else let bvs,extra_vs = split arity vs in
    bvs,bterm_to_lambdas_term (extra_vs,t)
;;

let aequal_bterms bt bt' =
  (alpha_equal_terms (bterm_to_so_lambda bt) (bterm_to_so_lambda bt'))
;;

%
Used if we have two bound terms [x1;...;xn].u and [y1;...;yn].v, we
have some variable x free in u and we desire the name of the corresponding
variable y free in v.
Should have l= [xn,yn;...;x1,y1]
%

letrec translate_binding l x =
  if null l then 
    failwith `translate_binding: binding not found` 
  else
  let (a,b).t = l in
  if x = a then b else
  let y = translate_binding t x in
  if y = b then 
     failwith `translate_binding: binding found, but it is shadowed` 
  else y
;;

%
************************************************************************
Substitution
************************************************************************
The main functions are as follows:

fo_subst   : (var # term) list -> term -> term
subst      : (var # term) list -> term -> term
full_subst : (var # term) list -> term -> term

They act on the following kinds of bindings:

1. v,so_lambda(...)           a higher order term binding.
2. v,parameter{le:level_exp}  a binding for a level expression variable
3. v,parameter{var:variable}  a binding for a bound variable
4. v,parameter{...}           a binding for a parameter variable
5. v,t                        a first order binding

The substitutions of kind 3 are used for renaming bound variables.
The encoding of substitutions of kind 3 is unambiguous, if we assume
that no substitutions turn up with bindings for parameter variables of
type `variable'. (Parameter variables are currently only used for
level expressions and integers.)


Function:   Binding kinds recognised:

fo_subst    5
subst       1,3,5
full_subst  1,2,3,4,5

so substitution is carried out for both second order variables and so 
applications of fo variables.
%

%
binding recognition functions
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
%
let is_term_binding ((v:var),t) = 
  not is_term `parameter` t
;;

let is_bvar_binding ((v:var),t) = 
  ( let [p],() = dest_term_with_opid `parameter` t
    in 
      is_variable_parm p
  ) ?
  false
;;

letref substitutions_maintain_bound_var_identities = true ;;



% 
Doesn't handle 
 1. substituting in level expression parameters more complicated than 
    le vars . 
 2. Bound variable renaming
%

let subst_parms_terms (ps,vs) t = 
    bimodal_second_order_substitute
      substitutions_maintain_bound_var_identities 
      ps
      vs
      t
;;

%
subst ensures that any so applications of variables in the substitution
are converted to true so variables before invoking subst_parms_terms.

subst
  sub : (var # term) list
  t   : term
  =
  t'  : term
%


let subst sub pat = 
   %mlbreak `s `;% let tsub,bv_sub = 
    divide_list (\vt.sub_kind_of_term (snd vt) = `regular`)
    sub
  in let so_sub = map (id # so_lambda_to_bterm) tsub 
  in let bv_sub' = 
      (map dest_bvar_binding bv_sub 
       ?
       failwith `subst: unexpected parameter binding`
      )
  in let inst = 
    subst_parms_terms
      ([],so_sub)
      (so_ize_so_aps_of_vars (map fst tsub) pat)
  in
  let inst_bvars = format_bvar_pl_from_pat bv_sub' pat inst
  in
    apply_alpha_conv_mi inst_bvars inst
;;



let fo_subst sub t = 
      bimodal_first_order_substitute 
        substitutions_maintain_bound_var_identities
        t
        sub ;;


% 
Applies a parameter substitution to a parameter. Invokes level expression
substitution when needed. (Should it simplify resulting level expression?)
%

let parm_subst psub =
  let le_psub,psub' = divide_list (is_level_exp_parm o snd) psub in
  let le_sub = map (id # dest_level_exp_parm) le_psub in
  \p.
  if is_level_exp_parm p then
    (mk_level_exp_parm o level_exp_subst le_sub o dest_level_exp_parm) p
  if is_parm_variable p then
    (let p' = apply_alist psub' (id_of_parm_variable p) in
     if type_of_parm p = type_of_parm p' then
       p'
     else
       failwith `parm_subst: wrong parameter type in substitution`
    )
  else
    p
;;

let apply_psub_to_term psub t =
  if null psub then t else
  sweep_up_map
    (\t'.let (opid,parms),btms = dest_term t' in
          mk_term (opid,map (parm_subst psub) parms) btms
    )
    t
;;


% 
TO CHANGE:
there is no need here to separate off the  parameter substitution,
since the lisp sub function does do substitution of level expression
variables within level expressions.
%

let full_subst sub t =
  let tsub,psub = 
     divide_list
       (\v,t'.member (sub_kind_of_term t') ``regular variable``)
       sub
  in
  let t' = apply_psub_to_term (term_sub_to_parm_sub psub) t 
  in
    subst tsub t'
;;
      

% Only used in match_dset function defined later in this file%

let so_subst_in_bterm s bterm =
  so_lambda_to_bterm 
   (subst_parms_terms 
      ([],s) 
      (so_ize_so_aps_of_vars (map fst s) (bterm_to_so_lambda bterm))
   )
;;

