%
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************
%

%[
************************************************************************
************************************************************************
TERM.ML
************************************************************************
************************************************************************
term analysis  (independent of kinds of terms.)

These functions generalise the map and accumulate functions for lists.

n.b. sweep map functions catch all failures of the functions being mapped.
]%


%[
Map functions ignoring binding structure
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
]%

let map_on_immediate_subterms_old f t =
  let op,bterms = dest_term t in 
  let new_bterms = 
    map (\vs,t.vs,f t) bterms 
  in
    mk_term op new_bterms
;;

letref mois_same = 0;;
letref mois_diff = 0;;

let map_on_immediate_subterms_alt f t =
 letrec aux i bts =
   if null bts then nil 
   else let t = snd (hd bts) in
        let nt = f t in
          if eq_terms t nt then aux (1 + i) (tl bts)
          else ((i, fst (hd bts), nt) . (aux (1 + i) (tl bts)))
  in
 letrec build i nbts bts =
   if null nbts then bts
   else if null bts then (tty_print "map_on_immediate_subterms_alt error"; fail)
   else let nbt = hd nbts in				     
   else if (i = fst nbt) then ((snd nbt) . (build (1 + i) (tl nbts) (tl bts)))
   else (hd bts) . (build (1+i ) nbts (tl bts)) in

 let bts = (bound_terms_of_term t) in
 let nbts = aux 1 bts in
  if null nbts then (mois_same := 1 + mois_same; t)
  else (mois_diff := (1 + mois_diff); mk_term (operator_of_term t) (build 1 nbts bts))
;;  

let map_on_immediate_subterms = map_on_immediate_subterms_alt;;

% bottom up order. %

let sweep_up_map f t =
  letrec aux t = 
           let t' = map_on_immediate_subterms aux t in
             f t' ? t'
  in
    aux t
;;


% top down order. %

let sweep_down_map f t =
  letrec aux t = 
           let t' = f t ? t in
             map_on_immediate_subterms aux t' 
  in
    aux t
;;

% like sweep down but don't descend into new subterms. %

let higher_map f t =
  letrec aux t = f t ? map_on_immediate_subterms aux t 
  in
    aux t
;;

%[
Map functions ignoring binding structure,
applied to two terms simulataneously.
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
]%

let map_on_immediate_subterms2 f t1 t2 =
  let op1, bterms1 = dest_term t1 in
  let op2, bterms2 = dest_term t2 in
  let new_bterms = map2 (\(vs1,t1) (vs2, t2). vs1, f t1 t2) bterms1 bterms2
  in
      mk_term op1 new_bterms
;;

%
| bottom up order.
%
let sweep_up_map2 f t1 t2 =
  letrec aux t1 t2 = 
      let t' = map_on_immediate_subterms2 aux t1 t2 in
	  f t' t2 ? t'
  in
      aux t1 t2
;;

%
| top down order.
%
let sweep_down_map2 f t1 t2 =
  letrec aux t1 t2 = 
      let t' = f t1 t2 ? t1 in
	  map_on_immediate_subterms2 aux t' t2
  in
      aux t1 t2
;;

%
| like sweep down but don't descend into new subterms.
%
let higher_map2 f t1 t2 =
  letrec aux t1 t2 = f t1 t2 ? map_on_immediate_subterms2 aux t1 t2 
  in
      aux t1 t2
;;

%[
Map functions specific to a single subterm
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
]%

let apply_to_nth_immediate_subterm f n t =
    let op,bterms = dest_term t in
    let vs,t' = select n bterms ? failwith `apply_to_nth_immediate_subterm` in
    let t'' = f t' in
      mk_term 
        op
        (replace_nth_by n bterms (vs,t''))
;;

let apply_to_addressed_subterm f address t =
 (letrec aux addr t =
    if null addr then f t else
    apply_to_nth_immediate_subterm (aux (tl addr)) (hd addr) t
  in
    aux address t
 ) ?? [`apply_to_nth_immediate_subterm`]
      failwith `apply_to_addressed_subterm: invalid address`
;;



%[
Map functions keeping track of binding variables
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
]%

% n.b. bvar lists are maintained in reversed order %

let map_on_immediate_subterms_with_bvars f vs t =
  let op, bterms = dest_term t in
  mk_term op (map (\bvars,u. bvars, f (rev bvars @ vs) u) bterms)
;;


let sweep_up_map_with_bvars f t =
  letrec g vs t =
    let t' = map_on_immediate_subterms_with_bvars g vs t in
      f vs t' ? t'
  in
    g [] t
;;

let sweep_down_map_with_bvars f t =
  letrec g vs t =
    let t' = f vs t ? t in
       map_on_immediate_subterms_with_bvars g vs t' 
  in
    g [] t
;;

let higher_map_with_bvars f t =
  letrec g vs t = f vs t ? map_on_immediate_subterms_with_bvars g vs t 
  in
    g [] t
;;




%[
Accumulate functions keeping track of binding variables
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
]%

% output = f ivs t input  %

let accumulate_over_immediate_subterms_with_bvars f vs t input =
  let bterms = bterms_of_term t in
    (accumulate 
       (\ain (bvars,u).f (rev bvars @ vs) u ain)
       input
       bterms
    )
;;

let preorder_accumulate_with_bvars f input t =
  letrec g vs t ain =
    let amid = f vs t ain in
      accumulate_over_immediate_subterms_with_bvars g vs t amid
  in
    g [] t input
;;

% output = f ivs addr t input  %

let accumulate_over_immediate_subterms_with_bvars_and_addr f vs addr t input =
  let bterms = bterms_of_term t in
    (accumulate 
       (\ain (i,bvars,u).f (rev bvars @ vs) (i.addr) u ain)
       input
       (zip (upto 1 (length bterms)) bterms)
    )
;;

% NB: vars and addr come in inside out order. %

let preorder_accumulate_with_bvars_and_addr  f input t =
  letrec g vs addr t ain =
    let amid = f vs addr t ain in
      accumulate_over_immediate_subterms_with_bvars_and_addr g vs addr t amid
  in
    g [] [] t input
;;

%[
Accumulate function ignoring binding variables
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
]%

%  f:t->*->*  accumulator input 2nd %

let preorder_accumulate f input t =
  preorder_accumulate_with_bvars (\x.f) input t
;;




%[
************************************************************************
Subterm selection functions
************************************************************************
]%

letrec get_addressed_subterm addr t =
  if null addr then t
  else
    get_addressed_subterm 
      (tl addr) 
      (subterm_of_term t (hd addr)
       ? failwith `get_addressed_subterm: invalid address`)
;;

% retrieves first in preorder. %

let find_subterm (P:bterm -> bool) t = 
  let t' = 
    preorder_accumulate_with_bvars
     (\vs tm ain.
        if isl ain & P (vs,tm) then inr tm  else ain
     )
     (inl ())
     t
  in
  if isl t' then
    failwith `find_subterm: no subterm satisfying predicate`
  else
     outr t'
;;

% retrieves all addresses of subterms satisfying P in preorder %

let find_subterms_with_addrs (P:bterm -> bool) t = 
  rev
   (preorder_accumulate_with_bvars_and_addr
     (\vs addr tm ain.
        if P (vs,tm) then (tm,rev addr).ain else ain
     )
     []
     t
   )
;;

    
%[
************************************************************************
Term pruning 
************************************************************************
]%

%
cut off all subterms, replacing with trivial terms. (e.g. void_type)
%

let prune_subterms t = 
  map_on_immediate_subterms (\x.void_term) t
;;




%[
************************************************************************
Recognisers for classes of terms.
************************************************************************
]%

% These are ghastly as is. Use member or memq on opid... %

let is_canonical_nontype =
  satisfies_one
  [is_token_term; is_natural_number_term; is_axiom_term; is_nil_term; 
   is_cons_term; is_inl_term; is_inr_term; is_lambda_term; is_pair_term]
;;

let is_canonical_non_U_type =
  satisfies_one
  [is_void_term; is_atom_term; is_int_term; is_list_term; 
   is_union_term; is_product_term; is_function_term;
   is_rfunction_term; is_isect_term; is_quotient_term; 
   is_set_term; is_equal_term; is_sqequal_term; is_term_sq_term;
   is_less_than_term; is_rec_term; is_object_term]
;;

let is_canonical_type t = is_U_term t or is_canonical_non_U_type t ;;
let is_canonical_term t = is_canonical_nontype t or is_canonical_type t ;;

let is_noncanonical_term = 
  satisfies_one
  [is_ind_term; is_list_ind_term; is_decide_term; is_spread_term;
   is_apply_term; is_atomeq_term; is_inteq_term; is_less_term ; 
   is_minus_term; is_add_term; is_subtract_term; is_multiply_term; 
   is_divide_term; is_remainder_term; is_rec_ind_term ]
;;



% Doesn't include decide. %
let is_decision_term = 
  satisfies_one [is_atomeq_term; is_inteq_term; is_less_term]
;;

%[
************************************************************************
Integer expression constructors, destructors, recognisers
************************************************************************
]%

% :int->term %
let mk_integer_term n =
  if n<0
    then mk_minus_term (mk_natural_number_term (-n))
    else mk_natural_number_term n
;;

let is_integer_term t = 
  is_natural_number_term t 
  or ( is_minus_term t & is_natural_number_term (dest_minus t) )
;;

let dest_integer t =
  if is_minus_term t
    then (- dest_natural_number (dest_minus t))
    else dest_natural_number t
  ? failwith `dest_integer`
;;

let is_int_exp t = 
  member (opid_of_term t) 
  ``natural_number minus add subtract multiply divide remainder``
;;

%[
************************************************************************
Universe, squash, equality and membership manipulations.
************************************************************************
]%


% squash(t) == {0 in int | t} %

let mk_squash_term t = mk_simple_term `squash` [t]
and is_squash_term = is_term `squash` 
and dest_squash = list_to_pair o dest_simple_term_with_opid `squash` 
;;

%
let map_on_equality_type f T =
   let equands,T = dest_equal T  in  mk_equal_term (f T) equands
;;
%

let mk_member_term T t = mk_simple_term `member` [T;t]
and is_member_term = is_term `member` 
and dest_member = list_to_pair o dest_simple_term_with_opid `member`  
;;

let is_member_or_equal_term t =
  let opid = opid_of_term t in
    opid = `member` or opid = `equal`
;;

let dest_member_or_equal t =
 dest_equal t
 ?
 let S,s = dest_member t in S,s,s
;;

let mk_label_term label t = mk_term (`label`,[mk_token_parm label]) [[],t] 
and is_label_term label t = 
  let opid,parms = op_of_term t in
    opid = `label` & (dest_token_parm o hd) parms = label
;;

%[
************************************************************************
Basic propositional and predicate term manipulations.  
************************************************************************
]%

%
Prop Universe
~~~~~~~~~~~~~~
%

let mk_prop_term le = mk_term (`prop`,[mk_level_exp_parm le]) []
and dest_prop t = 
  if is_term `prop` t then dest_level_exp_parm (hd (parms_of_term t))  
  else failwith `dest_prop`
;;

%
Propositional definitions
~~~~~~~~~~~~~~~~~~~~~~~~~
%

% true == (0 in int)   false == void %

let is_true_term = is_term `true`  
and true_term = mk_simple_term `true` [] ;;

let is_false_term = is_term `false`  
and false_term = mk_simple_term `false` [] ;;


% and(p;q) == p#q %
let is_and_term = is_term `and`  
and dest_and = list_to_pair o dest_simple_term_with_opid `and`  
and mk_and_term s t = mk_simple_term `and` [s;t]
;;

% or(p;q) == p|q %
let is_or_term = is_term `or` 
and dest_or = list_to_pair o dest_simple_term_with_opid `or` 
and mk_or_term s t = mk_simple_term `or` [s;t]
;;

% implies(p;q) == p->q % 
let is_implies_term = is_term `implies` 
and dest_implies = list_to_pair o dest_simple_term_with_opid `implies` 
and mk_implies_term s t = mk_simple_term `implies` [s;t]
;;

% not(p) == implies(p;false) %
let mk_not_term t = mk_simple_term `not` [t]
and is_not_term t = is_term `not` t
and dest_not = hd o dest_simple_term_with_opid `not` 
;;

% iff(p;q) == p=>q & p<=q % 
let is_iff_term = is_term `iff` 
and dest_iff = list_to_pair o dest_simple_term_with_opid `iff` 
and mk_iff_term s t = mk_simple_term `iff` [s;t]
;;

%
Predicate term manipulations
~~~~~~~~~~~~~~~~~~~~~~~~~~~~
%

% exists(a;x.b[x]) == x:a # b[x] %
let mk_exists_term var type prop  =
  mk_std_term `exists` [[],type; [var],prop] 
and is_exists_term = is_term `exists`
and dest_exists t = 
 (let [[],type;[var],prop] = dest_std_term_with_opid `exists` t in
  var,type,prop
 ) ? failwith `dest_exists`
;;



% all(a;x.b[x]) == x:a->b[x] %
let mk_all_term var type prop =
  mk_std_term `all` [[],type; [var],prop] 
and is_all_term = is_term `all` 
and dest_all t = 
 (let [[],type;[var],prop] = dest_std_term_with_opid `all` t in
  var,type,prop
 ) ? failwith `dest_all`
;;



% To satisfy some people's taste... %

let mk_some_term = mk_exists_term ;;

%[
************************************************************************
Iterated propositional term constructors and destructors
************************************************************************
The dest_iterated_* functions assume operator is associated to right.

dest_conjunction and dest_disjunction work with arbitrary associations.
]%

let dest_assoc_binop (dest : term-> (term # term)) t =
  letrec aux t =
    (let ta,tb = dest t in
       aux ta @ aux tb 
    ) ? [t]
  in 
    aux t
;;

% [t1;t2;...] --> t1 or (t2 or ... %
% nondisjunction t --> [t] %

let mk_iterated_or = fail_with `mk_iterated_or` (reduce1 mk_or_term) 
and dest_iterated_or = unreduce1 dest_or
;;

let dest_disjunction = dest_assoc_binop dest_or ;;


let mk_iterated_and = fail_with `mk_iterated_and` (reduce1 mk_and_term)
and dest_iterated_and = unreduce1 dest_and
;;

let dest_conjunction = dest_assoc_binop dest_and ;;

let mk_iterated_implies = fail_with `mk_implication` (reduce1 mk_implies_term)
and dest_iterated_implies = unreduce1 dest_implies
;;


%[
************************************************************************
General purpose iterated term constructors and destructors
************************************************************************
]%

let iterate_dest_quantifier dest %:term -> var#term#term% =
  unreduce (\t. let x,y,z = dest t in (x,y),z) 
;;

% Variant on above, which removes exactly n quantifiers. %

letrec iterate_dest_quantifier_for_n dest n t = 
  if n=0 then [],t
  else
  let x,A,t' = dest t
  in ($. (x,A) # id) (iterate_dest_quantifier_for_n dest (n-1) t')
;;



let iterate_mk_quantifier (mk: var->term->term->term) =
  C (reduce (\(x,A) P. mk x A P))
;;
%[
************************************************************************
Iterated primitive term manipulations.  
************************************************************************
For product, function, lambda, and application terms.

Need more systematic naming convention...
e.g

mk_iterated_<basic term name>
dest_iterated_<basic term name>

Although standard names like conjunction and disjunction are OK. (?)
]%

let explode_product = iterate_dest_quantifier dest_product 
and implode_product = iterate_mk_quantifier mk_product_term
;;

let explode_function = iterate_dest_quantifier dest_function 
and implode_function = iterate_mk_quantifier mk_function_term 
;;

let mk_iterated_function = implode_function
and mk_iterated_product = implode_product
;;
% Filters out no_var ids %
let mk_iterated_lambda vars t = 
  reduce (\x b. if is_null_var x then b else mk_lambda_term x b) t vars 
and dest_iterated_lambda = unreduce dest_lambda
;;

% :term->term list.   Let x1(x2)...(xn) be a breakdown of t maximal for its 
  form.  Return [x1;x2;...;xn].   %
let dest_iterated_apply t =
  letrec aux fun args =
         (let f,a = dest_apply fun in aux f (a . args))
         ? fun . args  in
  aux t []
;;


let dest_proper_iterated_apply t = 
  let l = dest_iterated_apply t in 
  if length l < 2 then failwith `dest_proper_iterated_apply` else l
;;

% return x1 (see comment on decompose_application) %
letrec head_of_application t =
  head_of_application (fst (dest_apply t))  ?  t
;;

let mk_iterated_apply ts = accumulate1 mk_apply_term ts ;;

let mk_application = mk_iterated_apply ;;
let mk_ap = mk_application ;;

letrec arity_of_application t =
  arity_of_application (fst (dest_apply t)) + 1  ?  0
;;

let arity_of_iterated_apply = arity_of_application ;;


%[
************************************************************************
Iterated logic term manipulations
************************************************************************

]%

let mk_iterated_all = iterate_mk_quantifier mk_all_term  ;;

let dest_iterated_all_for_n n = 
  iterate_dest_quantifier_for_n dest_all n
;;


%[
************************************************************************
Simple primitive term manipulations.  
************************************************************************
]%
let equands = snd o dest_equal ;;
let first_equand = fst o equands ;;
let eq_type = fst o dest_equal;;


let is_independent_product_term t =
  is_product_term t & is_null_var (fst (dest_product t))
;;

%[
************************************************************************
term manipulation for computation purposes
************************************************************************
]%
let mtt = mk_tag_term ;; 

% :term->bool.  -n is a redex exactly when n is -k for k a canonical
  natural number.  A rec_ind term is not considered a redex unless its
  "principal argument" is in canonical form, since otherwise
  normalization would almost always diverge.  (Note, however, that
  evaluation of rec_inds is lazy.)
%
let is_primitive_redex t =
  let are_integers s t = is_integer_term s & is_integer_term t  in
  not (is_canonical_term t)
  &
  if is_apply_term t then is_lambda_term (fst (dest_apply t)) 
  if is_spread_term t then is_pair_term (fst (dest_spread t))
  if is_atom_eq_term t then 
    (let l,r,(),() = dest_atomeq t in is_token_term l & is_token_term r)
  if is_int_eq_term t
    then (let l,r,(),() = dest_inteq t in are_integers l r)
  if is_less_term t 
    then (let l,r,(),() = dest_less t in are_integers l r)
  if is_decide_term t
    then (let e,(),() = dest_decide t in  is_inl_term e or is_inr_term e)
  if is_ind_term t then
    (let e,(),(),() = dest_ind t in is_integer_term e)
  if is_list_ind_term t then
    (let e,(),() 
       = dest_list_ind t in is_nil_term e or is_cons_term e)
  if is_minus_term t then (dest_integer (dest_minus t) < 0
                          ? false)
  if is_int_exp t & not is_natural_number_term t then
    (let [a;b] = subterms_of_term t  in
    are_integers a b & (is_remainder_term t or is_divide_term t
                        => not dest_integer b = 0 | true))
  if is_rec_ind_term t then
    is_canonical_term (fst (dest_rec_ind t))
  else false
;;

let is_prim_redex = is_primitive_redex ;;

let is_prim_redex_id id = 
  member id 
 ``apply
   spread
   atom_eq
   int_eq
   less
   decide
   ind
   list_ind
   minus
   add
   multiply
   subtract
   divide
   remainder
 ``
;;


let is_beta_redex t = 
  is_apply_term t & is_lambda_term (fst (dest_apply t)) 
;;

% 
is_principle_arg t i = true if t is a non-canonical term and the ith subterm 
is principle. Used to control lazy reduction of terms in term rewriting.
%

letref principle_arg_alist =
[
    `minus`,[1] ;
    `add`,[1;2] ;
    `subtract`,[1;2] ;
    `multiply`,[1;2] ;
    `divide`,[1;2] ;
    `remainder`,[1;2] ;
    `ind`,[1] ;
    `list_ind`,[1] ;
    `decide`,[1] ;
    `spread`,[1] ;
    `apply`,[1] ;
    `atom_eq`,[1;2] ;
    `int_eq`,[1;2] ;
    `less`,[1;2] 
]
;;

let update_principle_arg_alist name args =
  principle_arg_alist := update_alist principle_arg_alist name args 
;;

let principle_arg_nums_of_term t = 
  apply_alist principle_arg_alist (opid_of_term t) ? []
;;

let is_principle_arg t =
  let principle_arg_nums = principle_arg_nums_of_term t
  in
    (\i. member i principle_arg_nums)
;;

%[
************************************************************************
Guarded term constructor/destructor/tester
************************************************************************
]%
let is_guard_term = is_term `guard` ;;
let dest_guard = hd o dest_simple_term_with_opid `guard` ;;
let mk_guard_term t = mk_simple_term `guard` [t] ;;


%[
******************************************************************************
Analysis functions for free and bound variables in terms
******************************************************************************
]%

let free_fo_vars = free_variables ;;
let free_term_vars_with_arities = 
  second_order_free_variables ;;

let free_vars t = map fst (free_term_vars_with_arities t);;

let level_vars t = 
   (level_variables_of_term t)
;;

% tl throws away the constant level exp %

let vars_of_level_exp le =
  tl (map fst (dest_level_exp le))
;;

let parm_vars t =
  let parms_with_vars = 
    preorder_accumulate 
      (\t' ps.
         (filter (\p.is_parm_variable p or is_level_exp_parm p) 
                 (parms_of_term t')
         )
         @ ps
      )
      []
      t
  in
  let vars_with_dups =
    flatten
      (map
       (\p.
         if is_level_exp_parm p then
           vars_of_level_exp (dest_level_exp_parm p)
         else
           [id_of_parm_variable p]
       )
       (rev parms_with_vars)
      )
  in
    remove_later_duplicates vars_with_dups
;;


% NB: it is important that the next two functions retrieve bound vars in the
  same order
%

letrec bound_vars t =
  flatten
  ( map 
      (\vs,t'. vs @ bound_vars t')
      (bterms_of_term t)
  )
;;


let bound_vars_with_tm_tags t =
  letref n = 0 
  in let getn () = n := n+1
  in letrec aux t =
    let i = getn () 
    in
      flatten
      ( map 
          (\vs,t'. map (\v.v,i) vs @ aux t')
          (bterms_of_term t)
      )
  in
    aux t
;;

%
| Find all vars in a term,
| and list them in reverse preorder.
%
let all_vars term =
    letrec aux term value =
        letrec search_parms parms value =
	    if null parms then
		value
	    else
		let p = hd parms in
		    if is_variable_parm p then
			search_parms
			    (tl parms)
			    (insert (dest_variable_parm p) value)
		    else
			search_parms (tl parms) value
        in
	letrec search_args args value =
	    if null args then
		value
	    else
		let bvars, term = hd args in
		    search_args
		        (tl args)
			(aux term
			    (accumulate
			        (\value v. insert v value)
				value
				bvars))
	in
	let ((), parms), args = dest_term term in
	    search_args args (search_parms parms value)
    in
	aux term [];;

%[
************************************************************************
Term List functions.
************************************************************************
Functions for destructing and constructing term lists.

Term lists are generally built from two operators <opid>() and <opid>(0,0)
representing the nil and cons structures of the list.
]%

letrec dest_list_of_terms list_tm = 
 (let (),[a;b] = dest_simple_term list_tm
  in
    a. dest_list_of_terms b
 )
 ?
 (let (),[] = dest_simple_term list_tm
  in
    []
 )
 ?
 [list_tm]
;;
 

%
Makes a term list using cons_opid for cons and nil_opid for nil.
%

letrec mk_list_of_terms_with_nil nil_opid cons_opid tm_list = 
  if null tm_list then
    mk_simple_term nil_opid []
  else
    mk_simple_term 
     cons_opid [hd tm_list
               ;mk_list_of_terms_with_nil nil_opid cons_opid (tl tm_list)]
;;


let mk_list_of_terms opid tm_list = 
  mk_list_of_terms_with_nil opid opid tm_list ;;

  
%
Makes a term sequence using abstractions with opid opid for cons and nil nodes.
Sequences have no nil's and the unit sequence is the same as the element in
the sequence.
%

let mk_term_seq opid tm_list = 
  letrec aux as = 
    if null (tl as) then 
      hd as
    else
      mk_simple_term 
        opid [hd as;aux (tl as)]
  in
  if null tm_list then
    mk_simple_term opid []
  else
    aux tm_list
;;

% skips over elements which have opid and no subterms. 
Does not assume right normalization
%

let dest_term_seq opid t = 
  letrec aux t = 
   (let subts = dest_simple_term_with_opid opid t
    in 
      if null subts then [] else aux (first subts) @ aux (second subts)
   ) 
   ? [t]
  in
    aux t
;;

let normalize_term_seq opid t = 
  mk_term_seq opid (dest_term_seq opid t)
;;

%
Makes an alternating text/term sequence using abstractions with opid opid 
for cons and nil nodes, using !text terms for odd elements and initial and final elements. 
%

let is_itext_term = is_term `!text` ;;

let mk_itext_term s = mk_sp_term `!text` s ;;

let append_itext_terms t1 t2 = 
  if is_itext_term t1 & is_itext_term t2 then
    mk_itext_term (snd (dest_sp_term t1) J snd (dest_sp_term t2))
  else
    failwith `append_itext_terms`
;;

let mk_text_seq opid tm_list = 

  letrec collapse_text as = 
    if null as or null (tl as) then 
      as
    else
    ( let a1.a2.as3 = as in
      if is_itext_term a1 & is_itext_term a2 then
        collapse_text (append_itext_terms a1 a2 . as3)
      else
        a1 . collapse_text (a2.as3)
    )
  in
  letrec add_odd bs = 
    if null bs then 
      [mk_itext_term ""]
    if is_itext_term (hd bs) then
      hd bs . add_even (tl bs)
    else
      mk_itext_term "" . add_even bs
  
  % always (hd bs) if exists is not text %

  and add_even bs = 
    if null bs then
      []
    else
      (hd bs) . add_odd (tl bs)
  in
    mk_term_seq opid (add_odd (collapse_text tm_list))
     
;;

let normalize_text_seq opid t = 
  mk_text_seq opid (dest_term_seq opid t)
;;

%[
************************************************************************
Term Ordering functions
************************************************************************
Order by
  1. opid
  2. lex on parms
  3. lex on bvars
  4. lex on subterms.
]%
let equal_lists_p p l1 l2 =
 letrec aux l1 l2 = 
   if (l1 = [] & l2 = []) then true
   else if l1 = [] then false
   else if l2 = [] then false
   else if (p (hd l1) (hd l2))
   then aux (tl l1) (tl l2)
   else false  in
 aux l1 l2
;;



letrec term_lt a b = 
  let (ida,psa),btsa = dest_term a and (idb,psb),btsb = dest_term b
  in
  if not ida = idb then tok_lt ida idb 
  if not (equal_lists_p equal_parameters psa psb) then list_lex_lt_p equal_parameters parm_lt psa psb
  else
  let bvsa,tsa = unzip btsa and bvsb,tsb = unzip btsb
  in
  % PERF unzip probably consing more than we would want %
  if not bvsa = bvsb then list_lex_lt (list_lex_lt var_lt) bvsa bvsb
  else
    list_lex_lt_p alpha_equal_terms term_lt tsa tsb
;;

%
************************************************************************
Parameter terms
************************************************************************
These are used to make parameter bindings of type (tok # parm) list,
have type (var # term) list.
%

let mk_parm_term p = 
  mk_term (`parameter`,[p]) []
;;
let is_parm_term t = is_term `parameter` t ;;

let dest_parm_term t = 
  let [p],[] = dest_term_with_opid `parameter` t in
   p
;;

let type_of_parm_term t = 
  type_of_parm (dest_parm_term t)
;;

% used for types of variables which are bound to level exp parm terms. %

let level_exp_type = mk_simple_term `level_exp` [] ;;
let is_level_exp_type = is_term `level_exp` ;;

%[
*****************************************************************************
Gathering stats on terms
*****************************************************************************
]%

% size of a term. %

% bad consing behaviour 12+ cons per node! in test on 32 node term %

%
let term_size t = 
  preorder_accumulate (\t i.i+1) 0 t 
;;

letrec term_size t = 
  reduce (\x y.term_size x+y) 1 (subterms t)
;;
%

let term_size t = 
  letref n = 0
  in
  letrec aux t = 
   n:= n+1
   ; all (\t'.aux t') (subterms t)
  in
    aux t ; n
;;

%
Rough estimate, since in general could be sharing of structure 
%

let conses_in_term t = 
  let num t = 
  ( let (id,ps),bts = destruct_term t
    in let vs = flatten (map fst bts)
    in 
      2
      + 2 * length ps     % in parm list %
      + 2 * length bts    % in subterm list %
      + length vs         % in bvar lists %
  )
  in
    preorder_accumulate (\t i.i+num t) 0 t 
;;

