
;;;************************************************************************
;;;                                                                       *
;;;                Nuprl Proof Development System                         *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the Nuprl group, Department of Computer Science,       *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify Nuprl provided this notice  *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************


#+cltl2(in-system-package)
#-cltl2(in-package *system-package-name*)


(defstruct (direct-computation-definition (:include definition))
  func)

(defun dc-definition (term dc-f)
  (make-direct-computation-definition :keys (list term)
				      :dependency (dependency (new-oid (transaction-stamp)) nil nil)
				      :func dc-f))

(defun function-of-dc-definition (def)
  (direct-computation-definition-func def))

(defun direct-computation-table (tag)
  (define-definition-table;;-make
      (transaction-stamp)
      (list 'direct-computation tag)
    t
    ;; allow only one def per term-sig.
    :key-clash-f #'(lambda (a b) (declare (ignore a b)) t)
    ;;:meta-parameter-p #'abstraction-meta-parameter-p
    :meta-parameter-p #'(lambda (p)
			  (if (level-expression-parameter-p p)
			      (level-variable-p (value-of-parameter p))
			      (abstraction-meta-parameter-p p)))
    ;; only one per term-sig, so must be right one.
    :key-match-f #'(lambda (a b) (declare (ignore a b)) t)
    :no-touch-history t))


(defvar *direct-computation-table*)

(defun insert-direct-computation-function (term dc-f)

  (let ((def (dc-definition term dc-f))
	(seq (new-sequence-stamp)))
    (definition-insert *direct-computation-table*
	def
      (current-transaction-stamp)
      seq)
    (definition-commit *direct-computation-table*
	(oid-of-definition def)
      (transaction-stamp)
      seq) ))


;;; do-indicated-computations: term -> term.  Perform the computations
;;; indicated by the tags in the argument as far as possible, never
;;; failing.   This function, and some of its supporting functions, are
;;; somewhat dependent on term representation.


(defun erase-tags (term)
  (if (tagged-term-p term)
      (erase-tags (term-of-tagged-term term))
      (mapcr-on-subterms-optimized term erase-tags)))

(defvar *step-count*)
(defvar *max-step-count*)
(defvar *wo-pred* nil)

(defvar *tagged-ids* nil)
(defvar *tagged-dir* nil)

(defun increment-and-check-step-count ()
  (incf *step-count*)
  (or (zerop *max-step-count*)
      (<= *step-count* *max-step-count*)))

(defun check-step-count ()
  (or (zerop *max-step-count*)
      (< *step-count* *max-step-count*)))

(defmacro increment-step-count ()
  `(incf *step-count*))

(defun check-computation-sequence (b e cs)
  (let ((prev b)
	(i 1))
    
    (dolist (c cs)
      (let ((dir (car c))
	    (tterm (cdr c)))
	(let ((cterm (do-indicated-computations tterm)))
	  (if dir
	      (progn
		(unless (equal-terms-p prev (erase-tags tterm))
		  (raise-error (error-message '(verify comp-seq fwd) i prev (erase-tags tterm))))
		(setf prev cterm))
	      (progn
		(unless (equal-terms-p prev cterm)
		  (raise-error (error-message '(verify comp-seq rev) i prev cterm)))
		(setf prev (erase-tags tterm))))))
      (incf i))

    (unless (equal-terms prev e)
      (raise-error (error-message '(verify comp-seq end) prev e)))
    ))

(defunml (|check_computation_sequence| (ep cs))
    ((term |#| term) -> (((bool |#| term) list) -> unit))
  (check-computation-sequence (car ep) (cdr ep) cs)
  (values))
				       

(defun do-indicated-computations (term)
  ;;(setf -term term) ;;(break "dic")
  ;;(format t "do-indicated-computations ~a~%" (id-of-term term))
  (if (tagged-term-p term)
      (cond
	((compseq-tagged-term-p term)
	 (compute-compseq (dir-of-compseq-tagged-term term)
			  (do-indicated-computations
			      (term-of-tagged-term term))
			  (compseq-of-compseq-tagged-term term))
	 )
	((obids-tagged-term-p term)
	 (initialize-and-compute (tag-of-tagged-term term)
				 (dir-of-obids-tagged-term term)
				 (ids-of-tagged-term term)
				 (do-indicated-computations
				     (term-of-tag-term term))))
	(t (initialize-and-compute (tag-of-tag-term term)
				   nil nil
				   (do-indicated-computations
				       (term-of-tag-term term)))))
      (mapcr-on-subterms-optimized term do-indicated-computations)))

;;; Used in the ML primitive no_extraction_compute
(defvar *no-extraction-compute* nil)

(defun no-extraction-compute-p ()
  *no-extraction-compute*) 

(defun no-extraction-compute (tag term)
  (raise-error (error-message '(no-extraction-compute dead)))
  (let ((*no-extraction-compute* t))
    (initialize-and-compute tag nil nil term)))


(defun initialize-and-compute (tag dir ids term)
  (let ((*step-count* 0)
	(*max-step-count* tag)
	(*tagged-ids* ids)
	(*tagged-dir* dir)
	;;(*max-step-count*  2)
	)
    (let ((cterm (compute term)))
      ;;(setf a tag b term c cterm) (break)
      (values cterm *step-count*))))

;; tagged dir == t ->   expand only those abstractions in tagged-ids
;; tagged dir == nil -> expand abstractions except those in tagged-ids
;;  thus dir == nil and null list of tagged-ids is no-op

(defun compute (term)
  ;;(setf -term term) (break "c")
  (if (check-step-count)
      (let ((def (definition-lookup *direct-computation-table* term)))
	;;( setf a term b def) (break "c")
	(if def
	    (funcall (function-of-dc-definition def) term)
	    ;; kludge for v4 migrated objects, real fix would be to mill termof instances to be like v5 expects.
	    ;;(let ((proof (lookup-proof-by-opid term)))
	    ;;(if (and proof (expansion-of-abstraction proof))
	    ;;(compute-termof term proof)))
	    (compute-abstraction term)))
      term))

(defun compute-abstraction (term)
  (let ((xterm (maybe-expand-term *tagged-dir* *tagged-ids* *wo-pred* term)))
    (if (eql xterm term)
        term
	(progn
	  (increment-step-count)
	  (when *computed-hook*
	    (funcall *computed-hook* term))
	  (compute xterm)))))

(define-primitive |compseq| () (left right))

(defvar *verify-compseq-hook* #'(lambda (x) (raise-error '(verify-compseq-hook not))))

;; h must not be partial application for this to work.
;; ie "set_verify_compseq_hook (can lookup_compseq)" is bad.
;; and "let verify_compseq = (can lookup_compseqp);; set_verify_compseq_hook verify_compseq;;" is bad.
;; but "let verify_compseq ep = (can lookup_compseq ep);; set_verify_compseq_hook verify_compseq;;" is good.
(defunml (|set_verify_compseq_hook| (h))
    (((term |#| term) -> bool) -> unit)
  ;;(setf -h h) (break "svch")
  (setf *verify-compseq-hook* (caar h)))


(defun require-compseq (compseq)
  ;;(setf -compseq compseq) (break "rc")
  (unless (funcall *verify-compseq-hook*
		   (cons (left-of-compseq-term compseq) (right-of-compseq-term compseq)))
    (raise-error (error-message '(compseq verify not) compseq))))

(defun compute-compseq (dir term compseq)
  (require-compseq (rhs-of-abstraction (abstraction-super-of-term compseq)))

  (let ((ecompseq (expand-term compseq)))
    ;;(setf -ecompseq ecompseq -term term) (break "cc")
    (if dir
	(if (equal-terms-p (left-of-compseq-term ecompseq) term)
	    (right-of-compseq-term ecompseq)
	    (raise-error (error-message '(compseq compute not)
					term
					(left-of-compseq-term ecompseq))))

	(if (equal-terms-p (right-of-compseq-term ecompseq) term)
	    (left-of-compseq-term ecompseq)
	    (raise-error (error-message '(compseq compute not)
					term
					(right-of-compseq-term ecompseq)))))))


(defun compute-apply (term)
  (let ((computed-function (compute (function-of-apply-term term))))
    (if (and (lambda-term-p computed-function)
	     (increment-and-check-step-count))
	(compute (substitute (term-of-lambda-term computed-function)
			     (list (cons (binding-of-term-of-lambda-term computed-function)
					 (arg-of-apply-term term)))))
	(apply-term computed-function (arg-of-apply-term term)))))


(defun compute-spread (term)
  (let ((computed-value (compute (value-of-spread-term term))))
    (if (and (pair-term-p computed-value)
	     (increment-and-check-step-count))
	;; a little optimization: x,y.x or x,y.y avoids subst call.
	(compute
	 (let ((sterm (term-of-spread-term term))
		(bindings (bindings-of-term-of-spread-term term)))

	  (if (variable-p sterm)
	      (let ((id (id-of-variable-term sterm)))
		(cond
		  ((eql id (car bindings)) (leftterm-of-pair-term computed-value))
		  ((eql id (cadr bindings)) (rightterm-of-pair-term computed-value))
		  (t sterm)))
		(substitute sterm
			    (list (cons (car bindings)
					(leftterm-of-pair-term computed-value))
				  (cons (cadr bindings)
					(rightterm-of-pair-term computed-value)))))))

	(if (eq computed-value (value-of-spread-term term))
	    term
	    (spread-term computed-value 
			 (bindings-of-term-of-spread-term term)
			 (term-of-spread-term term))))))


(defun compute-decide (term)
  (let ((computed-value (compute (value-of-decide-term term))))

    (if (and (injection-term-p computed-value)
	     (increment-and-check-step-count))
	
	(if (inl-term-p computed-value)
	    (compute (substitute (leftterm-of-decide-term term)
				 (list (cons (binding-of-leftterm-of-decide-term term)
					     (term-of-injection-term computed-value)))))
	    ; injection-term-p and not inl-term-p -> inr-term-p
	    (compute (substitute (rightterm-of-decide-term term)
				 (list (cons (binding-of-rightterm-of-decide-term term)
					     (term-of-injection-term computed-value))))))
	  
	(if (eq computed-value (value-of-decide-term term))
	    term
	    (instantiate-term (operator-of-term term)
			    (cons (instantiate-bound-term computed-value)
				  (cdr (bound-terms-of-term term))))))))


(defun compute-ind (term)
  (let ((computed-value (compute (value-of-ind-term term))))

    (if (and (integer-term-p computed-value)
	     (increment-and-check-step-count))

	(let ((number (integer-of-integer-term computed-value)))
	  (return-from compute-ind
	    (cond

	      ((zerop number)
	       (compute (baseterm-of-ind-term term)))

	      ((plusp number)
	       (compute 
		 (substitute 
		   (upterm-of-ind-term term)
		   (list (cons (car (bindings-of-upterm-of-ind-term term))
			       computed-value)
			 (cons (cadr (bindings-of-upterm-of-ind-term term))
			       (instantiate-term
				 (operator-of-term term)
				 (cons (instantiate-bound-term (natural-number-term (1- number)))
				       (cdr (bound-terms-of-term term)))))))))

	      ((minusp number)
	       (compute
		 (substitute
		   (downterm-of-ind-term term)
		   (list
		     (cons (car (bindings-of-downterm-of-ind-term term))
			   computed-value)
		     (cons (cadr (bindings-of-downterm-of-ind-term term))
			   (instantiate-term
			     (operator-of-term term)
			     (cons (instantiate-bound-term (integer-term (1+ number)))
				   (cdr (bound-terms-of-term term))))))))))))

	(if (eq computed-value (value-of-ind-term term))
	    term
	    (instantiate-term
	      (operator-of-term term)
	      (cons (instantiate-bound-term computed-value)
		    (cdr (bound-terms-of-term term))))))))


(defun compute-atomeq (term)
  (let ((computed-e1 (compute (leftterm-of-atom-eq-term term)))
	(computed-e2 (compute (rightterm-of-atom-eq-term term))))

    (if (and (token-term-p computed-e1)
	     (token-term-p computed-e2)
	     (increment-and-check-step-count))

	  (if (equal (atom-of-token-term computed-e1)
		     (atom-of-token-term computed-e2))
	      (compute (if-term-of-atom-eq-term term))
	      (compute (else-term-of-atom-eq-term term)))

	  (if (and (eq computed-e1 (leftterm-of-atom-eq-term term))
		   (eq computed-e2 (rightterm-of-atom-eq-term term)))
	      term
	      (instantiate-term (operator-of-term term)
			      (cons (instantiate-bound-term computed-e1)
				    (cons (instantiate-bound-term computed-e2)
					  (cddr (bound-terms-of-term term)))))))))

			     
(defun compute-less (term)
  (let ((computed-e1 (compute (leftterm-of-int-eq-term term)))
	(computed-e2 (compute (rightterm-of-int-eq-term term))))

    (if (and (integer-term-p computed-e1)
	     (integer-term-p computed-e2)
	     (increment-and-check-step-count))

	(if (< (integer-of-integer-term computed-e1)
	       (integer-of-integer-term computed-e2))
	    (compute (if-term-of-less-term term))
	    (compute (else-term-of-less-term term)))

	(if (and (eq computed-e1 (leftterm-of-less-term term))
		 (eq computed-e2 (rightterm-of-less-term term)))
	    term
	    (instantiate-term (operator-of-term term)
			    (cons (instantiate-bound-term computed-e1)
				  (cons (instantiate-bound-term computed-e2)
					(cddr (bound-terms-of-term term)))))))))


(defun compute-inteq (term)
  (let ((computed-e1 (compute (leftterm-of-int-eq-term term)))
	(computed-e2 (compute (rightterm-of-int-eq-term term))))

    (if (and (integer-term-p computed-e1)
	     (integer-term-p computed-e2)
	     (increment-and-check-step-count))

	(if (= (integer-of-integer-term computed-e1)
	       (integer-of-integer-term computed-e2))
	    (compute (if-term-of-int-eq-term term))
	    (compute (else-term-of-int-eq-term term)))

	(if (and (eq computed-e1 (leftterm-of-int-eq-term term))
		 (eq computed-e2 (rightterm-of-int-eq-term term)))
	    term
	    (instantiate-term (operator-of-term term)
			    (cons (instantiate-bound-term computed-e1)
				  (cons (instantiate-bound-term computed-e2)
					(cddr (bound-terms-of-term term)))))))))


(defun compute-incomplete (term)
  (raise-error (error-message '(compute incomplete) term)))


(defun compute-minus (term)
  (let ((computed-term (compute (term-of-minus-term term))))

    (if (and (integer-term-p computed-term)
	     (check-step-count))

	;; not null num -> computed-term == (minus (natural-number n)) | (natural-number n) where n>=0.
	(if (minusp (integer-of-integer-term computed-term))
	    ;; minusp num -> computed-term == (minus (natural-number n))
	    (progn (increment-step-count)
		   (term-of-minus-term computed-term))
	    ;; not minusp num -> computed-term == (natural-number n)
	    (minus-term computed-term))

	(if (eq computed-term (term-of-minus-term term))
	    term
	    (minus-term computed-term)))))



(defun compute-binary (term)
  (let ((computed-e1 (compute (leftterm-of-int-eq-term term)))
	(computed-e2 (compute (rightterm-of-int-eq-term term))))

    (if (and (integer-term-p computed-e1)
	     (integer-term-p computed-e2)
	     (not (and (or (divide-term-p term)
			   (remainder-term-p term))
		       (zerop (integer-of-integer-term computed-e2))))
	     (increment-and-check-step-count))

	(let ((num-e1 (integer-of-integer-term computed-e1))
	      (num-e2 (integer-of-integer-term computed-e2)))
	  (constant-case (id-of-term term)
	    (*add*		(integer-term (+ num-e1 num-e2)))
	    (*subtract*		(integer-term (- num-e1 num-e2)))
	    (*multiply*		(integer-term (* num-e1 num-e2)))
	    (*divide*		(integer-term (if (zerop num-e2) 
						  0
						  (truncate num-e1 num-e2))))
	    (*remainder*	(integer-term (if (zerop num-e2)
						  0
						  (rem num-e1 num-e2))))))
	
	(if (and (eq computed-e1 (leftterm-of-binary-term term))
		 (eq computed-e2 (rightterm-of-binary-term term)))
	    term
	    (instantiate-term (operator-of-term term)
			    (list (instantiate-bound-term computed-e1)
				  (instantiate-bound-term computed-e2)))))))


(defun compute-list-ind (term)
  (let ((computed-value (compute (value-of-ind-term term))))
    (if (and (or (nil-term-p computed-value)
		 (cons-term-p computed-value))
	     (increment-and-check-step-count))
	(if (nil-term-p computed-value)
	    (compute (baseterm-of-list-ind-term term))
	    (compute 
	      (substitute 
		(upterm-of-list-ind-term term)
		(let ((var-ids (bindings-of-upterm-of-list-ind-term term)))
		  (list (cons (car var-ids)
			      (head-of-cons-term computed-value))
			(cons (cadr var-ids)
			      (tail-of-cons-term computed-value))
			(cons (caddr var-ids)
			      (instantiate-term (operator-of-term term)
					      (cons (cadr (bound-terms-of-term computed-value))
						    (cdr (bound-terms-of-term term))))))))))

	(if (eq computed-value (value-of-ind-term term))
	    term
	    (instantiate-term (operator-of-term term)
			    (cons (instantiate-bound-term computed-value)
				  (cdr (bound-terms-of-term term))))))))


(defun compute-rec-ind (term)
  (let ((bindings (bindings-of-term-of-rec-ind-term term)))
    
    (increment-step-count)
    (compute 
      (substitute 
	(term-of-rec-ind-term term)
	(list (cons (car bindings)
		    (lambda-term (cadr bindings) 
				 (instantiate-term 
				   (operator-of-term term)
				   (cons (instantiate-bound-term (variable-term (cadr bindings)))
					 (cdr (bound-terms-of-term term))))))
	      (cons (cadr bindings) (value-of-rec-ind-term term)))))))




(defun compute-tagged (term)
  (declare (ignore term))
  (ref-error " Unexpected tagged term during direct-computation"))

(defun compute-termof (term termof-def)
  ;;(raise-error (error-message '(compute-termof dead)))
  (if (no-extraction-compute-p)
      term
      (progn
	(increment-step-count)
	(let ((exp (expansion-of-abstraction termof-def)))
	  (when (null exp)
	    (raise-error (error-message '(proof termof compute not) term)))
	  ;;(when *computed-hook*
	  ;;(funcall *computed-hook* xterm))
	  (when *computed-hook*
	    (break "compute-term-of-computed hook"))
	  (compute (expand-term-aux term
				    termof-def
				    exp))))))



;; this should be owned by an environment? One global table?
(defun rehash-direct-computation-table ()
  (with-environment-actual (new-environment '(dummy) nil nil nil nil)
    (with-local-transaction 
	(setf *direct-computation-table* (direct-computation-table 'ref))
  
      (insert-direct-computation-function (canonical-rec-ind-term) #'compute-rec-ind)
      (insert-direct-computation-function (canonical-add-term) #'compute-binary)
      (insert-direct-computation-function (canonical-subtract-term) #'compute-binary)
      (insert-direct-computation-function (canonical-multiply-term) #'compute-binary)
      (insert-direct-computation-function (canonical-divide-term) #'compute-binary)
      (insert-direct-computation-function (canonical-remainder-term) #'compute-binary)
      (insert-direct-computation-function (canonical-minus-term) #'compute-minus)
      (insert-direct-computation-function (canonical-iincomplete-term) #'compute-incomplete)
      (insert-direct-computation-function (canonical-int-eq-term) #'compute-inteq)
      (insert-direct-computation-function (canonical-less-term) #'compute-less)
      (insert-direct-computation-function (canonical-atom-eq-term) #'compute-atomeq)
      (insert-direct-computation-function (canonical-list-ind-term) #'compute-list-ind)
      (insert-direct-computation-function (canonical-ind-term) #'compute-ind)
      (insert-direct-computation-function (canonical-decide-term) #'compute-decide)
      (insert-direct-computation-function (canonical-spread-term) #'compute-spread)
      (insert-direct-computation-function (canonical-tag-term) #'compute-tagged)
      (insert-direct-computation-function (canonical-apply-term) #'compute-apply))))






(defunml (|compute| (term))
    (term -> term)
  (initialize-and-compute 0 nil nil term))

(defunml (|compute_wo_pred| (pred term))
    ((tok |#| ((tok list) -> (term -> (term -> bool)))) -> (term -> term))
  (let ((*wo-pred* (cons (car pred)
			 #'(lambda (def)
			     ;;(break "cwp")
			     (funmlcall (cdr pred)
					(conditions-of-abstraction def)
					(lhs-of-abstraction def)
					(rhs-of-abstraction def)
					)))))
    (initialize-and-compute 0 nil nil term)))

(defunml (|do_indicated_computations| (term))
    (term -> term)
  (do-indicated-computations term))


