
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2000                                *
;;;                                                                       *
;;;                                                                       *
;;;                Nuprl Proof Development System                         *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the Nuprl group, Department of Computer Science,       *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify Nuprl provided this notice  *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;************************************************************************

#+cltl2(in-system-package)
#-cltl2(in-package *system-package-name*)

;; RLE TODO ml ddag funcs? Want to allow ML to manage ddags??? They are not functional???
;;  there should be something built on top of ddag and statement and proof tables
;; to do cycle detection for ref.
 
;;; consider RLE MILL  proof-node -> proof-tree.

;;; RLE NAP maintain assumptions as reversed list. Support current
;;; ml interface but add funcs to access reversed list. Then migrate tactics.
;;; unmigrated tactics will be slower but migrated should be faster.


;;;; -docs- (mod ref)
;;;;
;;;;	Ref Rule table :
;;;;
;;;;	Ref module extends rule definition to contain an internal representation.
;;;;
;;;;	A symbol hash table is maintained for lookup by rule id. Thus, it is an
;;;;	error for more than one rule to use the same id.
;;;;
;;;;
;;;;	ref-rule-table (<tag>)			: <definition-table{rules}>
;;;;	ref-rule-insert(<term{export}> list)	: <ref-rule-defintion> list
;;;;	ref-rule-delete(<oa> list)		: NULL
;;;;
;;;;	rule-id-table-lookup(<id>)		: <ref-rule-definition>
;;;;	 * fails if id not in table.
;;;;
;;;;  -page-
;;;;
;;;;	Tactic Interface:
;;;;
;;;;
;;;;	<assumption>		: assumption[<variable-id> <term{type}> <bool{hidden}>]
;;;;
;;;;	<proof-node>		: proof-node[<assumption> list
;;;;					     <term{conclusion}>
;;;;					     <rule>
;;;;					     <proof-node> list]
;;;;
;;;;	<rule>			: rule[PRIMITIVE <id> <arg> list]
;;;;				| rule[TACTIC ML <term{source}> <proof-node{top}>]
;;;;				| rule[FAILED <rule> <message>]
;;;;
;;;;  -page-
;;;;	
;;;;	instantiate-assumption(<variable-id> <term> <bool>)		: <assumption>
;;;;	instantiate-assumption-r(<id> <term> <bool>)			: <assumption>
;;;;	 ** fails if id is not a proper <variable-id>; the dummy variable id is not accepted.
;;;;	
;;;;	id-of-assumption(<assumption>)					: <id>
;;;;	type-of-assumption(<assumption>)				: <term>
;;;;	hidden-assumption-p(<assumption>)				: <bool>
;;;;
;;;;	equal-assumptions-p(<assumption> <assumption>)			: <bool>
;;;;	lex-equal-assumptions-p(<assumption> <assumption>)		: <bool>
;;;;
;;;;  -page-
;;;;	
;;;;	instantiate-proof-node (<assumption> list <term>)		: <proof-node>
;;;;	instantiate-proof-node-r (<assumption> list <term>)		: <proof-node>
;;;;	 ** fails if sequent is not closed.
;;;;
;;;;	conclusion-of-proof-node (<proof-node>)				: <term>
;;;;	assumptions-of-proof-node (<proof-node>)			: <assumption> list
;;;;	rule-of-proof-node(<proof-node>)				: <rule> | nil
;;;;	children-of-proof-node(<proof-node>)				: <proof-node> list
;;;;
;;;;	mark-proof-node(<proof-node> <label> <prop>)			: <proof-node>
;;;;	proof-node-mark-value(<proof-node> <label>)			: <prop>
;;;;
;;;;	copy-sequent(<proof-node>)					: <proof-node>
;;;;	copy-proof-tree(<proof-node>)					: <proof-node>
;;;;
;;;;	alpha-equal-sequents-p(<proof-node> <proof-node>)		: <bool>
;;;;	lex-equal-sequents-p(<proof-node> <proof-node>)			: <bool>
;;;;	equal-sequents-p(<proof-node> <proof-node>)			: <bool>
;;;;	 * this is a bastardized version of the preceeding two in that the ids
;;;;	 * of the assumptions must be identical but the types of the assumptions
;;;;	 * and the conclusion are checked with alpha equality.
;;;;
;;;;	frontier-of-proof(<proof-node>)					: <proof-node> list
;;;;	 * copies of unrefined leaves of proof.
;;;;
;;;;	One can view a proof as a one-dimensional tree by using the following   
;;;;	walk function, which walks through proofs by stepping from unrefined
;;;;	nodes to their continuations:
;;;;
;;;;	walk-proof (<proof>
;;;;		    <closure{primitive-f}>
;;;;		    <closure{tactic-f}>
;;;;		    <closure{unrefined-f}> 
;;;;		    &key <closure{curtail-p}>)
;;;;	 * primitive-f (<proof> <*> list)	: <*>
;;;;	    <*> is result of visiting children.
;;;;	 * tactic-f(<proof> <*>)		: <*>
;;;;	    <*> is result of visiting interior using children as continuations.
;;;;	 * unrefined-f(<proof>)			: <*>
;;;;	 * curtail-p(<proof>)			: <bool>
;;;;	    Avoids recursive descent and purges orphaned continuations.
;;;;	 ** other keys will be added as other rule types are supported.
;;;;
;;;;  -page-
;;;;
;;;;	primitive-extraction : there are three modes for primitive extraction.
;;;;	 - nil   : no extraction.
;;;;	 - eager : partial extraction after tactic application.
;;;;	 - lazy  : partial extraction after refinement complete.
;;;;		   primitive proof tree is accessible after refinement.
;;;;
;;;;
;;;; -doct- (mod ref ml)
;;;;
;;;;
;;;;	ML primitive types : 
;;;;
;;;;	rule
;;;;	argument
;;;;	proof
;;;;	assumption
;;;;
;;;;  -page-
;;;;
;;;;	Proofs:
;;;;
;;;;	make_proof_node			: assumption list -> term -> proof
;;;;	hypotheses 			: proof -> assumption list
;;;;	conclusion			: proof -> term
;;;;	refinement			: proof -> rule
;;;;	children			: proof -> proof list
;;;;
;;;;	alpha_equal_sequents		: proof -> proof -> bool
;;;;	lex_equal_sequents		: proof -> proof -> bool
;;;;	equal_sequents			: proof -> proof -> bool
;;;;
;;;;	copy_proof			: proof -> proof
;;;;
;;;;
;;;;	Proofs may be annotated:
;;;;	
;;;;	annotate_proof			: proof -> * -> proof
;;;;	annotation_of_proof		: proof -> *
;;;;	clear_annotation_of_proof	: proof -> proof
;;;;
;;;;  -page-
;;;;
;;;;	Assumptions :
;;;;
;;;;	make_assumption			: variable -> term -> bool -> assumption
;;;;	destruct_assumption		: assumption -> variable # term # bool
;;;;
;;;;  -page-
;;;;
;;;;	Primitive Rules:
;;;;
;;;;	Args:
;;;;
;;;;	make_assumption_index_argument	: int -> argument
;;;;	make_variable_argument		: variable -> argument
;;;;	make_parameter_argument		: parameter -> argument
;;;;	make_term_argument 		: term -> argument
;;;;	make_bound_term_argument	: variable list -> term -> argument
;;;;	make_assumption_list_argument	: assumption list -> argument
;;;;	make_substitution_list_argument
;;;;	 : (tok # parameter) list -> (variable # term) list -> argument
;;;;
;;;;
;;;;	Recall the rule arg interpreter syntax:
;;;;
;;;;	<rule-arg>		: <term>
;;;;				| !assumption-index{<tok>:t}
;;;;				| !term{<id>:v}
;;;;				| !variable{<tok>:t}
;;;;				| !parameter{<tok>:t}
;;;;				| !bound-id(<id> list.<term>)
;;;;				| !substitution_list{<tok>:t}
;;;;
;;;;	There is an obvious correspondence between the list of rule-definition
;;;;	rule arg terms and the arguments expected by make_primitive_rule.
;;;;
;;;;
;;;;  -page-
;;;;
;;;;	Rules:
;;;;
;;;;	make_primitive_rule 	: tok -> argument list -> rule
;;;;
;;;;	type_of_rule		: rule -> tok
;;;;	 * one of primitive, or tactic.
;;;;
;;;;	id_of_rule		: rule -> tok
;;;;	 * if primitive rule then primitive id;
;;;;	 * if tactic rule then ml or prl;
;;;;
;;;;	make_tactic_rule	: term -> rule
;;;;
;;;;	proof_of_tactic_rule	: rule -> proof
;;;;
;;;;
;;;;  -page-
;;;;
;;;;	Refine: 
;;;;
;;;;	validation		= proof list -> proof
;;;;	tactic			= proof -> (proof list # validation)
;;;;
;;;;	refine			: rule -> proof -> validation
;;;;			        = rule -> tactic
;;;;
;;;;	frontier		: tactic
;;;;
;;;;	refiner_current_object	: unit -> object_address
;;;;
;;;;
;;;;  -page-
;;;;
;;;;	lemma_lookup	: tok -> object_id
;;;;	name_of_lemma	: object_id -> tok
;;;;	
;;;;	refiner_current_objects		: unit -> object_id list
;;;;	refiner_object_current_p	: object_id -> bool
;;;;
;;;;
;;;;  -page-
;;;;
;;;;	Direct Computation :
;;;;
;;;;	compute				: term -> term
;;;;	do_indicated_computations	: term -> term
;;;;	
;;;;
;;;; -doce-

;;;; RLE NAP when rule interpreter spiffed up may want to restore these:
;;;;	type_of_argument : argument -> tok
;;;;	** returns one of : `assumption-index` `variable` `parameter`
;;;;	**		    `term` `bound-term` `substitution-list`
;;;;	destruct_assumption_index_argument	: argument -> int
;;;;	destruct_variable_argument	: argument -> variable 
;;;;
;;;;	destruct_parameter_argument	: argument -> parameter
;;;;	destruct_term_argument		: argument -> term
;;;;	destruct_bound_term_argument	: argument -> (variable list # term)
;;;;	destruct_assumption_list_argument
;;;;	 : argument -> assumption list
;;;;	destruct_substitution_list_argument
;;;;	 : argument -> ((tok # parameter) list # (variable # term) list)
;;;;
;;;;
;;;;
;;;;

;;; RLE NAP 	Check args at make_primitive_rule then assume args proper in
;;;		interpreter as long as definition is  not stale.


;;;;	RLE NAP destruct_parameter_argument	: argument -> parameter

;;; 	RLE NAP	possible to hide type of argument in defstruct type?
;;;		Would this save space??

;;;; RLE NAP : other rules : reflection rule, proof-cache rule and prl term tactics:
;;;;				| rule['prl nil <term> <proof-node>]
;;;;




;;;
;;;	Rules.
;;;

;;;
;;;	No rule to term as all import/export-able rules will be tactics.
;;; RLE NAP	Need tactic access to reflection-rule
;;; RLE NAP	At import, need ability to distinquish prl tactic from ml tactic.
;;;	 - wrapper for prl, maybe both.
;;;	 - maybe give ml acces to prl tactics.



#-(or dontinline dontinlinerefd)
(eval-when (compile)
  (proclaim '(inline
	      definition-of-rule id-of-rule
	      message-of-failed-rule rule-of-failed-rule failed-rule
	      type-of-rule proof-of-top-rule nml-tactic-rule-p nml-tactic-rule
 	      refined-nml-tactic-rule text-of-tactic-rule proof-of-tactic-rule
	      primitive-tree-rule-p primitive-tree-rule extract-of-primitive-tree-rule
	      args-of-primitive-rule results-of-refined-primitive-rule refined-primitive-rule
	      id-of-assumption type-of-assumption instantiate-assumption
	      allocated-proof-count incf-allocated-proof-count
	      assumptions-of-sequent conclusion-of-sequent
	      assumptions-of-proof-node conclusion-of-proof-node
	      rule-of-proof-node children-of-proof-node marks-of-proof-node copy-sequent
	      instantiate-sequent instantiate-sequent-r instantiate-proof-node instantiate-proof-node-r
	      proof-node-mark-value refined-rule-p refined-proof-node-p
	      rule-id-table-lookup
	      independent-product-term dependent-product-term-p independent-product-term-p dependent-set-term-p
	      injection-term-p not-term not-term-p term-of-not-term
	      not-equal-term-p not-less-than-term-p int-equal-term-p reflexive-equal-term
	      reflexive-equal-term-p int-equal-term not-int-equal-term equal-zero-term
	      not-equal-zero-term not-less-than-term less-than-zero-term not-less-than-zero-term
	      injection-term integer-term-p integer-term zero-term zero-term-p
	      integer-of-integer-term binary-integer-term leftterm-of-binary-term rightterm-of-binary-term
	      term-of-injection-term leftterm-of-decision-term rightterm-of-decision-term
	      if-term-of-decision-term else-term-of-decision-term ref-current-object
	      |ml-refiner_current_object|
	      syntax-of-argument value-of-argument
	      do-disequality label-of-eq-node operands-of-eq-node containers-of-eq-node
	      reprcount-of-eq-node next-of-eq-node eligible-of-eq-node node-iden-of-eq-node previous-of-eq-node
	      eq-node get-node-iden$ rot process-assumptions$ not-term$ not-term-body$
	      arith-reflexive-equal-term max-edge-weight$ sum-edge-weight$
	      assert-E-graph-disequality$ E-graph-class$ do-E-disequalities$ 
	      function-of-dc-definition increment-and-check-step-count check-step-count
	      no-extraction-compute-p no-extraction-compute
	      ;;tactic-type-p
	      top-rule-to-itactic
	      |ml-make_proof_node| |ml-hypotheses| |ml-conclusion| |ml-refinement| |ml-children|
	      |ml-clear_annotation_of_proof| |ml-make_assumption| |ml-destruct_assumption|
	      |ml-make_assumption_index_argument| |ml-make_variable_argument| |ml-make_parameter_argument|
	      |ml-make_term_argument| |ml-make_bound_term_argument| |ml-make_assumption_list_argument|
	      |ml-make_substitution_list_argument| |ml-term_to_argument| |ml-argument_to_term|
	      |ml-make_primitive_rule| |ml-make_tactic_rule| |ml-type_of_rule|
	      |ml-id_of_rule| |ml-proof_of_tactic_rule| |ml-equal_sequents|
	      |ml-lex_equal_sequents| |ml-alpha_equal_sequents| |ml-copy_proof|
	      )))




(defvar *refiner-safety* nil)

(defstruct rule
  definition)

;; RLE TODO check if rule stale if so refresh. If refresh fails, then raise error.
(defun definition-of-rule (rule)
  (rule-definition rule))

(defun id-of-rule (rule)
  (id-of-rule-definition (rule-definition rule)))




(defvar *failed-rule-def* (make-rule-definition :name 'failed))

;; not unparseable -> failed. Will be able to distinquish by message.
(defstruct (failed-rule (:include rule))
  rule
  message)

(defun message-of-failed-rule (rule)
  (failed-rule-message rule))

(defun rule-of-failed-rule (rule)
  (failed-rule-rule rule))


(defun failed-rule (rule message)
  (make-failed-rule :definition *failed-rule-def*
		    :rule rule
		    :message message))


(defun type-of-rule (rule)
  (cond
    ((null rule) nil)
    ((failed-rule-p rule) 'failed)
    ((primitive-rule-p rule) 'primitive)
    ((top-rule-p rule) 'tactic)))
  



;;;
;;; tactic-rules
;;;

(defstruct (term-rule (:include rule))
  (term nil)				; text, maybe should be renamed text.
  )

(defstruct (top-rule (:include term-rule))
  (proof-top nil)
  )


(defun proof-of-top-rule (rule)
  (top-rule-proof-top rule))


(defvar *nml-tactic-rule-definition* (make-rule-definition :name 'nml))

(defun nml-tactic-rule-p (rule)
  (and (eql 'nml (id-of-rule rule))
       (not (primitive-rule-p rule))))
  
(defun nml-tactic-rule (term)
  (make-term-rule :definition *nml-tactic-rule-definition*
		  :term term))

(defun refined-nml-tactic-rule (term proof)
  (make-top-rule :definition *nml-tactic-rule-definition*
		       :term term
		       :proof-top proof))

(defun text-of-tactic-rule (rule)
  (term-rule-term rule))

(defun proof-of-tactic-rule (rule)
  (if (top-rule-p rule)
      (proof-of-top-rule rule)
      (raise-error (error-message '(rule tactic proof-top)))))


;;;
;;;  Rules with side-proofs
;;;

(defstruct (proof-rule (:include top-rule)
		       (:print-function  (lambda (rule stream depth)
					   (declare (ignore depth))
					   (format stream "~a" (id-of-rule rule)))))
  (args nil)
  extract)

(defvar *permute-proof-rule-definition* (make-rule-definition :name 'permute-proof))
(defvar *extract-proof-rule-definition* (make-rule-definition :name 'extract-proof))

(defun proof-rule (def args proof)
  (make-proof-rule :definition def
		   :term nil
		   :proof-top proof
		   :args args))

(defun args-of-proof-rule (rule)
  (proof-rule-args rule))

(defunml (|make_collapse_rule| (id map proof))
    (tok -> ((int list) -> (proof -> rule)))

  ;; compute range
  (let ((mn (min map))
	(mx (max map)))
    (break)
  ))

  
;;;
;;;  Rule to represent primitive proof.
;;;

(defvar *primitive-tree-rule-definition* (make-rule-definition :name 'primitive-tree))


(defstruct (primitive-tree-rule (:include term-rule))
  ;; for stats 
  count		; number of prim-rules abbreviated.
  longest	; longest assum list seend 
  )

  
(defun primitive-tree-rule (extract &optional count longest)
  (make-primitive-tree-rule :definition *primitive-tree-rule-definition*
			    :term extract
			    :count count
			    :longest longest))

(defun extract-of-primitive-tree-rule (rule)
  (term-rule-term rule))
		 
(defun longest-of-primitive-tree-rule (rule)
  (primitive-tree-rule-longest rule))
(defun count-of-primitive-tree-rule (rule)
  (primitive-tree-rule-count rule))
		 

(defvar *primitive-extract-mode* 'eager)



;;;
;;; 	primitive-rule 
;;;


(defstruct (primitive-rule (:include rule)
			   (:print-function  (lambda (rule stream depth)
					       (declare (ignore depth))
					       (format stream "~a" (id-of-rule rule)))))
  (args nil))


(defun args-of-primitive-rule (rule)
  (primitive-rule-args rule))

(defstruct (refined-primitive-rule (:include primitive-rule)
				   (:print-function  (lambda (rule stream depth)
						       (declare (ignore depth))
						       (format stream "~a" (id-of-rule rule)))))
  (results nil))

(defun results-of-refined-primitive-rule (r) (refined-primitive-rule-results r))

(defun refined-primitive-rule (rule results)
  (make-refined-primitive-rule :definition (definition-of-rule rule)
			       :args (args-of-primitive-rule rule)
			       :results results))

;;;;
;;;;	Assumptions
;;;;

;;; RLE TODO : implementation needs to be parameterized by assumption list
;;; RLE TODO : so that we may more easily experiment with assumption list implementations.


(defstruct assumption
  (id nil)			; a variable id.
  (type nil))			; a term.


(defun bad-parms-check (term)
  (dolist (p (parameters-of-term term))
    (when (error-parameter-value-p (value-of-parameter-n p))
      (raise-error (error-message '(bad parameter)
				  (message-to-string (message-of-error-parameter-value (value-of-parameter-n p)))
				  term))))
  (dolist (bt (bound-terms-of-term term))
    (bad-parms-check (term-of-bound-term bt))))


(defstruct (hidden-assumption (:include assumption)) )

(defun id-of-assumption (a) (assumption-id a))
(defun type-of-assumption (a) (assumption-type a))

(defun instantiate-assumption (id type &optional hidden)
  (bad-parms-check type)
  (if hidden
      (make-hidden-assumption :id id
			      :type (cache-free-vars type))
      (make-assumption :id id
		       :type (cache-free-vars type))))

(defun instantiate-assumption-r (id type &optional hidden)
  (cond
    ((dummy-variable-id-p id)
     (raise-error (error-message '(refiner assumption instantiate dummy))))
    ((variable-id-p id)
     (instantiate-assumption id type hidden))
    (t (raise-error (error-message '(refiner assumption instantiate) id)))))


(defun lex-equal-assumptions-p (x y)
  (or (eq x y)
      (and (eql (hidden-assumption-p x) (hidden-assumption-p y))
	   (eql (assumption-id x) (assumption-id y))
	   (compare-terms-p (assumption-type x) (assumption-type y)))))

(defun equal-assumptions-p (x y)
  (or (eq x y)
      (and (eql (hidden-assumption-p x) (hidden-assumption-p y))
	   (eql (assumption-id x) (assumption-id y))
	   (equal-terms-p (assumption-type x) (assumption-type y)))))





;;;;
;;;;	Proof nodes.
;;;;

;;; 
;;;  Some abstract proof-node stuff.
;;;
;;;  need to insure that all sequents of proof-nodes are closed.
;;; 
;;; no destructive modification of the assumptions or conclusion of a proof node
;;; should occur. 
;;;
;;; RLE NAP : ensure this is done as efficiently as possible : 
;;; any proof nodes revealed to ML can not be modifed, thus proof nodes seen
;;; by ML must be copied!!. 
;;;


;;; RLE NOTE status can be computed by examining proof and Rule.

;;;
;;;	proof-node statistics.
;;;

(defvar *allocated-proof-count*)
(defvar *rule-count*)

;; hack to count rules invocations and allow aborting if threshold is reached.
(defvar *prim-rule-count* 0)
(defvar *prim-rule-count-max* 0)

(defun prim-rule-count-start (&optional (max nil))
  (setf *prim-rule-count* 0)
  (setf *prim-rule-count-max* max)
  (when max (format t "~%RuleCountStart ~a~%" max)))

(defun prim-rule-count-stop ()
  (prog1 *prim-rule-count*
    (format t "~%RuleCountStop ~a~%" *prim-rule-count*)
    (setf *prim-rule-count-max* nil)))

(defun incf-prim-rule-count ()
  (declare (fixnum *prim-rule-count* *prim-rule-count-max*))
  (incf *prim-rule-count*)
  (when (and *prim-rule-count-max*
	     ;; maybe > would be better, but one shot seems enough
    	     (eql *prim-rule-count* *prim-rule-count-max*))
    (setf *prim-rule-count-max* nil)
    (format t "~%PrimRuleCountMax exceeded. Refinement aborted~%")
    (fooa)
    ))

(defunml (|rule_count_start| (unit) :declare ((declare (ignore unit))))
    (unit -> unit)
  (prim-rule-count-start nil))

(defunml (|rule_count_stop| (unit) :declare ((declare (ignore unit))))
    (unit -> int)
  (prim-rule-count-stop))

(defunml (|rule_count_set_trip| (max))
    (int -> unit)
  (prim-rule-count-start max))

(defvar *tactic-count*)

(defun allocated-proof-count () *allocated-proof-count*)

(defun incf-allocated-proof-count ()
  (when (boundp '*allocated-proof-count*)
    (incf *allocated-proof-count*)))

(defun incf-rule-count ()
  (incf-prim-rule-count)
  (when (boundp '*rule-count*)
    (incf *rule-count*)))

(defun incf-tactic-count ()
  (when (boundp '*tatic-count*)
    (incf *tactic-count*)))

(defmacro with-count-proof-allocation (&body body)
  `(let ((*allocated-proof-count* 0))
    ,@body))

;; accumulate ref stats.
(defvar *ref-stats-acc* nil)
(defvar *ref-stats-report-step* nil)
(defvar *ref-stats-suspended* nil)

(defun start-accumulate-ref-stats (name)
  (setf *ref-stats-acc* (cons (cons name (new-refine-stats-acc)) (delete name *ref-stats-acc* :key #'car))
        *ref-stats-suspended* (delete name *ref-stats-suspended*))
  nil)


(defun refine-stats-accumulate (name stats)
  (when *ref-stats-report-step*
    (report-refine-stats t stats "step"))

 (unless (member name *ref-stats-suspended*)
  (let ((acc (cdr (assoc name *ref-stats-acc*))))
    (when acc
      (refine-stats-acc acc stats)
      (report-refine-stats t acc name)))))

(defun suspend-ref-stats (name)
 (unless (member name *ref-stats-suspended*)
  (setf *ref-stats-suspended* (cons name *ref-stats-suspended*))))

(defun resume-ref-stats (name)
  (setf *ref-stats-suspended* (delete name *ref-stats-suspended*)))



;; fttb expect granularity of ref_refine.
;; desire granularity of refine-tactic.
;; when finer, time will be inclusive but counts will be exclusive???
(defmacro with-refine-stats (update-f &body body)
  (let ((result (gensym)))
    `(let ((,result nil))
      (let ((*allocated-proof-count* 0)
	    (*rule-count* 0)
	    (*tactic-count* 0)
	    )
	(funcall #'(lambda (tstats)
		     (funcall ,update-f
			      (new-refine-stats tstats
						*rule-count*
						*tactic-count*
						*allocated-proof-count*)))
		 (with-time-stats (setf ,result (multiple-value-list ,@body)))
		 )
	(values-list ,result)))
    ))





;; rle ??? if proof nodes treated functionally then maybe worth it to have sequent
;; version, then when refined use full proof.

;; RLE NAP used named structs for time being however may be possible to use unnamed.

(defstruct (sequent (:copier default-sequent-copy))
  assumptions
  conclusion)

(defun proof-print (proof stream depth)
  (declare (ignore depth))
  (format stream "ProofNode: Num Assums: ~a Concl TermSig: ~a Rule: ~a Num Children ~a."
	  (length (proof-node-assumptions proof))
	  (when (proof-node-conclusion proof)
	    (term-sig-of-term (proof-node-conclusion proof)))
	  (proof-node-rule proof)
	  (length (proof-node-children proof))))
  

(defstruct (proof-node (:include sequent)
		       (:print-function proof-print))
  rule
  children)

(defun assumptions-of-sequent (p) (sequent-assumptions p))
(defun conclusion-of-sequent (p) (sequent-conclusion p))

(defun assumptions-of-proof-node (p) (sequent-assumptions p))
(defun conclusion-of-proof-node (p) (sequent-conclusion p))

(defun rule-of-proof-node (p) (when (proof-node-p p) (proof-node-rule p)))
(defun children-of-proof-node (p) (when (proof-node-p p) (proof-node-children p)))
;;;
;;;	proof nodes may be sequents, however use sequents to save space when possible.

;;;
;;;	 Proof nodes may be marked, this should be somewhat rare, thus
;;;	we incur the overhead at mark time. This requires mark to return a new
;;;	structure. Proof-nodes must be treated funtionally anyways so this is not a burden.
;;;	RLE ??? You know though it seems like it would be nice to be able to mark as a side effect
;;;

;; not worth bother of sequent-marks is sequent marked just make it a proof node.

(defstruct (proof-node-marks (:include proof-node))
  (marks (make-marks)))

;;(defvar *dummy-proof-node-marks* (make-proof-node-marks))

;;(defun proof-node-marks-p (p)
;;  (= (array-dimension *dummy-proof-node-marks* 0)
;;     (array-dimension p 0)))

(defun marks-of-proof-node (p)
  (if (proof-node-marks-p p)
      (proof-node-marks-marks p)
      nil))

;;
;; need to prevent destructive updates to marks for proof-annotations.
;;   - share but copy at update.
;;
(defun copy-marks-alist (marks-alist)
  (mapcar #'(lambda (m)
	      (cons (car m) (cdr m)))
	  marks-alist))


;; sequentp t means copy do not copy rule, children.
(defun copy-of-proof-node (proof &optional sequentp unmark)
  ;;(unless (proof-node-p proof) (break))
  (incf-allocated-proof-count)
  (let ((marks (and (proof-node-marks-p proof)
		    (let ((m (marks-alist (marks-of-proof-node proof))))
		      (make-marks :alist
				  (remove 'frontier 
					  (if unmark
					      (remove unmark m :key #'car)
					      m)
					  :key #'car))))))
    (if marks
	(if sequentp
	    (make-proof-node-marks :marks marks
				   :assumptions (assumptions-of-proof-node proof)
				   :conclusion (conclusion-of-proof-node proof))
	    (make-proof-node-marks :marks marks
				   :assumptions (assumptions-of-proof-node proof)
				   :conclusion (conclusion-of-proof-node proof)
				   :rule (rule-of-proof-node proof)
				   :children (children-of-proof-node proof)))
	(if sequentp
	    (make-proof-node :assumptions (assumptions-of-proof-node proof)
			     :conclusion (conclusion-of-proof-node proof))
	    (make-proof-node-marks :assumptions (assumptions-of-proof-node proof)
				   :conclusion (conclusion-of-proof-node proof)
				   :rule (rule-of-proof-node proof)
				   :children (children-of-proof-node proof))))))
	
;; copy sequent to a proof node for later modification of rule,children
(defun copy-sequent (proof)
  (copy-of-proof-node proof t))


;;;;
;;;;
;;;;



;; LAL
#-cmu
(defun alpha-equal-sequents-aux-p (assums-x concl-x assums-y concl-y
				   &optional
				   (dontcheckhidden t) ; probably should default to nil compare to v4 and change then regression test.
				   long-y-ok-p)
  (labels
      ((alpha-equal-bound-term-lists (listx listy)
	 (or (and (null listx) (null listy))
	     (and listx listy
		  (prog1
		      (and (enter-binding-pairs (bindings-of-bound-term-n (car listx))
						(bindings-of-bound-term-n (car listy)))
			   (alpha-equal-terms (term-of-bound-term (car listx))
					      (term-of-bound-term (car listy))))
		    (exit-binding-pairs (bindings-of-bound-term-n (car listx))
					(bindings-of-bound-term-n (car listy))))
		  (alpha-equal-bound-term-lists (cdr listx) (cdr listy)))))

       (alpha-equal-terms (termx termy)
	 (if (and (variable-p termx)
		  (variable-p termy))
	     (equal-bindings (id-of-variable-term termx)
			     (id-of-variable-term termy))
	     (and (equal-operators-p (operator-of-term termx) (operator-of-term termy))
		  (alpha-equal-bound-term-lists (bound-terms-of-term termx)
						(bound-terms-of-term termy)))))
       (alpha-equal-assumption-lists (listx listy)
	 (or (and (null listx)
		  (or long-y-ok-p (null listy))
		  (alpha-equal-terms concl-x concl-y))
	     
	     (and listx (or long-y-ok-p listy)
		  (prog1
		      (and (or dontcheckhidden
			       (eql (hidden-assumption-p (car listx))
				    (hidden-assumption-p (car listy))))
			   (alpha-equal-terms (type-of-assumption (car listx))
					      (type-of-assumption (car listy)))
			   (enter-binding-pair (id-of-assumption (car listx))
					       (id-of-assumption (car listy)))
			   (alpha-equal-assumption-lists (cdr listx)
							 (cdr listy)))
		    (exit-binding-pair (id-of-assumption (car listx))
				       (id-of-assumption (car listy))))))))

    (or (and (eq concl-x concl-y)
	     (or (eq assums-x assums-y)
		 ;; following may be unlikely, but costs little.
		 (apply-predicate-to-list-pair-optimized assums-x assums-y eq)))
	(with-variable-invocation
	    (alpha-equal-assumption-lists assums-x assums-y)))))

(defun alpha-equal-sequents-p (proof-x proof-y)
  (or (eq proof-x proof-y)
      (and proof-x proof-y
	   (alpha-equal-sequents-aux-p
	    (assumptions-of-proof-node proof-x)
	    (conclusion-of-proof-node proof-x)
	    (assumptions-of-proof-node proof-y)
	    (conclusion-of-proof-node proof-y)))))


#+cmu
(defun alpha-equal-sequents-aux-p (assums-x concl-x assums-y concl-y
				   &optional
				   (dontcheckhidden t) ; probably should default to nil compare to v4 and change then regression test.
				   long-y-ok-p)
  (labels
      ((alpha-equal-assumption-lists (listx listy)
	 (or (and (null listx)
		  (or long-y-ok-p (null listy))
		  (alpha-equal-terms concl-x concl-y))
	     
	     (and listx (or long-y-ok-p listy)
		  (prog1
		      (and (or dontcheckhidden
			       (eql (hidden-assumption-p (car listx))
				    (hidden-assumption-p (car listy))))
			   (alpha-equal-terms (type-of-assumption (car listx))
					      (type-of-assumption (car listy)))
			   (enter-binding-pair (id-of-assumption (car listx))
					       (id-of-assumption (car listy)))
			   (alpha-equal-assumption-lists (cdr listx)
							 (cdr listy)))
		    (exit-binding-pair (id-of-assumption (car listx))
				       (id-of-assumption (car listy))))))))

    (or (and (or (eq assums-x assums-y)
		 ;; following may be unlikely, but costs little.
		 (apply-predicate-to-list-pair-optimized assums-x assums-y eq))
	     (eq concl-x concl-y))
	(with-variable-invocation
	    (alpha-equal-assumption-lists assums-x assums-y)))))


(defun check-sequent-closure (assumptions conclusion)
  ;;(break "cs")
  
  ;; check for bad parms in concl
  (bad-parms-check conclusion)
  
  (with-variable-minor-invocation

      (dolist (assum assumptions)
	;; check for unbound variable injection in type.
	;;(unless (free-vars-p (type-of-assumption assum)) (break "csc"))
	(dolist (var-id (mark-value-must (type-of-assumption assum) 'free-vars))
	  (when (not (variable-minor-use-p var-id))
	    (raise-error (error-message '(refiner instantiate sequent not-closed)
					var-id;;assumptions
					conclusion))))

	;; check for duplicate id in assumption list.
	;; (break "NuPRL")  (raise-error (error-message '(refiner instantiate sequent duplicate-ids)
	;;  (id-of-assumption assum) assumptions conclusion))
	(when (set-variable-minor-use (id-of-assumption assum))
	  ;;(break "d")
	  (raise-error (error-message '(refiner instantiate sequent duplicate-ids)
				      (id-of-assumption assum);; assumptions
				      conclusion))))


    ;; check for free vars in conclusion.
    (dolist (var-id (sticky-free-vars conclusion))
      (when (not (variable-minor-use-p var-id))
	(raise-error (error-message '(refiner instantiate sequent not-closed)
				    var-id ;assumptions
				    conclusion))))))
  

(defun instantiate-sequent (assumptions conclusion)
  ;;(when (bad-term-p conclusion) (break "btp is"))
  (incf-allocated-proof-count)
  (make-sequent :assumptions assumptions :conclusion conclusion))

(defun instantiate-sequent-r (assumptions conclusion)
  ;;(when (bad-term-p conclusion) (break "btp isr"))
  (check-sequent-closure assumptions conclusion)
  (instantiate-sequent assumptions conclusion))

(defun instantiate-proof-node (assumptions conclusion)
  ;;(when (bad-term-p conclusion) (break "btp ipn"))
  (incf-allocated-proof-count)
  (make-proof-node :assumptions assumptions :conclusion conclusion))

(defun instantiate-proof-node-r (assumptions conclusion)
  ;;(when (bad-term-p conclusion) (break "btp ipnr"))
  (check-sequent-closure assumptions conclusion)
  (instantiate-proof-node assumptions conclusion))


(defun mark-proof-node (p label value)
  ;;(when (bad-term-p (conclusion-of-proof-node p)) (break "btp mpn"))
  ;;(break "mpn")
  
  (let ((marks (make-marks :alist (let ((m (marks-of-proof-node p)))
				    (when m (copy-marks-alist
					     (marks-alist m)))))))
  
    (let ((q (make-proof-node-marks :assumptions (assumptions-of-proof-node p)
				    :conclusion (conclusion-of-proof-node p)
				    :rule (rule-of-proof-node p)
				    :children (children-of-proof-node p)
				    :marks marks)))
      (mark marks label value)
      q)))

(defun mark-proof-node-n (p label value)
  ;;(break "mpnn")
  ;;(unless (proof-node-p p) (break))
  ;;(when (bad-term-p (conclusion-of-proof-node p)) (break "btp mpnn"))

  (if (proof-node-marks-p p)
      (progn (mark (marks-of-proof-node p) label value)
	     p)
      (let ((q (make-proof-node-marks :assumptions (assumptions-of-proof-node p)
				  :conclusion (conclusion-of-proof-node p)
				  :rule (rule-of-proof-node p)
				  :children (children-of-proof-node p))))
	(mark (marks-of-proof-node q) label value)
	q)))

(defun proof-node-mark-value (p label)
  (let ((marks (marks-of-proof-node p)))
    (when marks (mark-value marks label))))


(defun equal-sequents-p (seq1 seq2)
  (or (eq seq1 seq2)
      (and (or (eql (assumptions-of-proof-node seq1)
		    (assumptions-of-proof-node seq2))
	       (apply-predicate-to-list-pair (assumptions-of-proof-node seq1)
					     (assumptions-of-proof-node seq2)
					     #'equal-assumptions-p))
	   (equal-terms-p (conclusion-of-proof-node seq1)
			  (conclusion-of-proof-node seq2)))))


(defun lex-equal-sequents-p (seq1 seq2)
  (or (eql seq1 seq2)
      (and (or (eql (assumptions-of-proof-node seq1)
		    (assumptions-of-proof-node seq2))
	       (apply-predicate-to-list-pair (assumptions-of-proof-node seq1)
					     (assumptions-of-proof-node seq2)
					     #'lex-equal-assumptions-p))
	   (compare-terms-p (conclusion-of-proof-node seq1)
			  (conclusion-of-proof-node seq2)))))

(defun refined-rule-p (rule)
  (and rule
       (not (failed-rule-p rule))))
	 
(defun refined-proof-node-p (proof-node)
  (refined-rule-p (rule-of-proof-node proof-node)))

;;  This list will not share any modifiable structure with p.
(defun frontier-of-proof-aux (proof)
  (let ((acc nil))
    (labels ((visit (proof)
	       (if (not (refined-proof-node-p proof))
		   (push proof acc)
		   (dolist (proof (children-of-proof-node proof))
		     (visit proof)))))
      (visit proof)
      (nreverse acc))))

;; opportunistically sticks frontier in mark if marks available.
(defun frontier-of-proof (proof)
  (if (and nil (proof-node-marks-p proof))
      (let ((v (proof-node-mark-value proof 'frontier)))
	(if v
	    (cdr v)
	    (let ((l (frontier-of-proof-aux proof)))
	      (mark-proof-node-n proof 'frontier (cons t l))
	      l)))
      (frontier-of-proof-aux proof)))

	    
;; unconditionally sticks frontier in marks
(defun stick-frontier-of-proof (proof)
  (let ((l (proof-node-mark-value proof 'frontier)))
    (if l
      (values proof (cdr l))
      (let ((l (frontier-of-proof-aux proof)))
	(values (mark-proof-node-n proof 'frontier (cons t l)) l)))))


;;;;
;;;;  Walk(proof continuations)
;;;;
;;;;	Walks through proof in all dimensions.
;;;;
;;;;  Let q be a proof node of a proof w.
;;;;  Let p be a proof-top of tactic refining q.
;;;;  Let (a b) be subgoals of q.
;;;;  Let m,n be proof nodes of p.
;;;;  Let r,s be proof-tops of tactics refining m,n.
;;;;  Let (x),(y) be subgoals of m,n.
;;;;  Assume x, y, r, and s each have frontiers of
;;;;  a single proof-node.
;;;;
;;;;  Such a state could be graphically described as follows:
;;;;
;;;;         w  +-------------->  p  +-------------->  r
;;;;        / \/                 / \/                 / \
;;;;       /  /\                /  /\                /   \
;;;;      /  q  \              /  /  \              +-o---+
;;;;     /  a-b  \            / m   n-\------>  s     1
;;;;    /  /\ /\  \          / x-+ +-y \       / \
;;;;   /  /  X  \  \        / /\     /\ \     /   \
;;;;  +-------------+      +-o---------o-+   +---o-+
;;;;                         2         3         4
;;;;  Partial trace of continuations:
;;;;  1 walk w ()
;;;;  2 walk q ()
;;;;  3 walk p ((a b))
;;;;  4 walk m ((a b))
;;;;  5 walk r ((x) (a b))
;;;;  6 o1 - walk x ((a b))
;;;;  7 o2 - walk a ()
;;;;  8 return from 7 : ((b))
;;;;  9 return from 6 : (nil (b))
;;;; 10 return from 5 : ((b))
;;;; 11 return from 4 : ((b))
;;;; 12 walk n (b)
;;;; 13 walk s ((y) (b))
;;;; 14 o4 - walk y ((b))
;;;; 15 o3 - walk b ()
;;;; 16 return from 15 (nil)
;;;; 17 return from 14 (nil nil)
;;;; 18 return from 13 (nil)
;;;; 19 return from 12 (nil)
;;;; 20 return from 3 ()
;;;;

;; curtail would be used in extract when known that subproof can not contribute.

;; rle todo mill callers of walk-proof.
;; rle todo also insure matches spec.
(defun walk-proof (proof
		   primitive-f
		   primitive-tree-f
		   tactic-f
		   proof-f
		   unrefined-f
		   &key
		   (curtail-p #'(lambda (p) (declare (ignore p)) nil))
		   (continuation-f #'(lambda (p x) (declare (ignore p)) x))
		   continuation-enter-f
		   tactic-enter-f
		   )
  
  (labels
      ((purge-continuation-orphans (proof continuations)
	 (let ((rest continuations))
	   (labels
	       ;; stupid name for to avoid confusing weak compilers.
	       ((visit-aux (proof)
		  (if (not (refined-proof-node-p proof))
		      (pop rest)
		      (dolist (proof (children-of-proof-node proof))
			(visit-aux proof)))))
	     (visit-aux proof)
	     rest)))

       (visit (proof-node continuations)
	 (let ((rule (rule-of-proof-node proof-node))
	       (curtail (funcall curtail-p proof-node)))
	   (if curtail
	       (values curtail
		       (cons (purge-continuation-orphans proof-node (car continuations))
			     (cdr continuations)))	
	       (cond
		 ((not (refined-proof-node-p proof-node))
		  (if continuations
		      (mlet* (((x c) (visit (caar continuations)
					    (cdr continuations))))
			     (when continuation-enter-f
			       (funcall continuation-enter-f proof-node))
			     (values (funcall continuation-f proof-node x) (cons (cdar continuations) c)))
		      (values (funcall unrefined-f proof-node) nil)))

		 ((primitive-rule-p rule)
		  (mlet* (((x c) (visit-children (children-of-proof-node proof-node) continuations)))
			 (values (funcall primitive-f proof-node x) c)))

		 ((primitive-tree-rule-p rule)
		  (mlet* (((x c) (visit-children (children-of-proof-node proof-node) continuations)))
			 (values (funcall primitive-tree-f proof-node x) c)))
	       
		 ((nml-tactic-rule-p rule)
		  (when tactic-enter-f
		    (funcall tactic-enter-f proof-node))
		  (mlet* (((x c) (visit (proof-of-tactic-rule rule)
					(cons (children-of-proof-node proof-node)
					      continuations))))
			 (values (funcall tactic-f proof-node x) (cdr c))))

		 ((proof-rule-p rule)
		  (mlet* (((x c) (visit (proof-of-top-rule rule)
					(cons (children-of-proof-node proof-node)
					      continuations))))
			 (values (funcall proof-f proof-node x) (cdr c))))

		 ;;((prl-rule-p rule)
		 ;;(mlet* (((x c) (visit (prl-rule-proof-top rule)
		 ;;(cons (children-of-proof-node proof-node)
		 ;;continuations))))
		 ;;(values (funcall prl-f proof-node x) (cdr c))))

		 ;;((proof-cache-rule-p rule)
		 ;;(process-err "Extract for proof-cache rules not implemented."))


		 ;;((reflection-rule-p rule)
		 ;;(mlet* (((x c) (visit-children (children-of-proof-node proof-node) continuations)))
		 ;;(values (funcall reflection-f proof-node x) c)))
    
		 (t (raise-error (error-message '(walk-proof rule unknown) (id-of-rule rule))))))))


	  (visit-children (children continuations)
			  (if (null children)
			      (values nil continuations)
			      (mlet* (((x c) (visit (car children) continuations))
				      ((rest-x rest-c) (visit-children (cdr children) c)))
				     (values (cons x rest-x) rest-c)))))

    (visit proof nil)))

(defun complete-refine-stats (rstats proof tactic-p)
  (let ((largest 0)
	(longest 0)
	(deepest 0)

	(tactic (if tactic-p 1 0))
	(total (if tactic-p 1 0))
	(continuations (list 0))
	(depth 0)
	)

    (labels ((incf-count (amt depth)
	       (incf (nth depth continuations) amt))) 

      (walk-proof proof
		  ;; primitive
		  #'(lambda (proof results)
		      (declare (ignore proof))
		      (incf total)
		      (incf-count 1 depth)
		      (setf longest (max longest (length (assumptions-of-proof-node proof))))
		      results)

		  ;; primitive-tree-f
		  #'(lambda (proof results)
		      (declare (ignore proof))
		      (let ((pcount (count-of-primitive-tree-rule (rule-of-proof-node proof))))
			(incf total pcount)
			(incf-count pcount depth)
			(setf longest (max longest (longest-of-primitive-tree-rule (rule-of-proof-node proof)))))
		      results)
		    
		  ;; tactic
		  #'(lambda (proof result)
		      (declare (ignore proof))
		      (incf total)
		      (incf tactic)
		      (setf largest (max largest (pop continuations)))
		      (setf deepest (max deepest depth))
		      (decf depth)
		  
		      result)

		  ;; proof
		  #'(lambda (proof result)
		      (declare (ignore proof))
		      (incf total)
		      result)

		  ;; unrefined
		  #'(lambda (proof)
		      (declare (ignore proof))
		      )

		  :tactic-enter-f
		  #'(lambda (proof)
		      (declare (ignore proof))
		      (push 0 continuations)
		      (incf depth)
		      )

		  :continuation-f
		  #'(lambda (proof x) ;;lal 2 args
		      (declare (ignore proof))
		      (incf depth))

		  :continuation-enter-f
		  #'(lambda (proof)
		      (declare (ignore proof))
		      (decf depth))

		  ))

    (setf largest (max largest (pop continuations)))
    (update-refine-stats rstats total tactic largest longest deepest)
    rstats))





;;;;	
;;;;	Lemma table, statement-table plus hash table on lemma-name-4.2 property.
;;;;	
;;;;	lemma def is statement with name?
;;;;	

(defvar *ref-grphash* nil)

(defun ref-group-hash ()

  (when (or (definition-table-flag-touched-p (resource 'statements))
	    ;; need to check abs table to catch prf mods. Unfortunately does not distinquish actual
	    ;; abs, but maybe that will be a feature if abs included in ref-dag.
	    (definition-table-flag-touched-p (resource 'abstractions))
	    (null *ref-grphash*)
	    )
    ;;(break "rgh3")
    (ref-dag-rehash)
    (definition-table-flag-set-touched (resource 'statements) nil)
    (definition-table-flag-set-touched (resource 'abstractions) nil)
    )

  ;; (break "rgh")
  *ref-grphash*
  )

(defun ref-dag-rehash ()
  ;;(break "rdh")
  (without-dependencies
   (let ((acc-valid0 nil)
	 (acc-validnil nil)
	 (acc-other nil)
	 )

     (labels ((accumulate (rdag)
		(case (validity-of-ref-dag rdag)
		  (0 (push rdag acc-valid0))
		  ((nil) (push rdag acc-validnil))
		  (otherwise (push rdag acc-other)))))
	     
       (map-statement-table
	#'(lambda (oid def)
	    (declare (ignore oid))
	    (let ((rdag (ref-dag-of-statement def)))
	      (setf (ref-dag-index rdag) nil)
	      (accumulate rdag))))

       (map-proof-table
	#'(lambda (oid def)
	    (declare (ignore oid))
	    (let ((rdag (ref-dag-of-proof def)))
	      (setf (ref-dag-index rdag) nil)
	      (accumulate rdag))))

       ;; valid0
       (setf -a acc-valid0 -b acc-validnil -c acc-other)
       (let ((grphash (make-hash-table :size (* 2 (length acc-valid0)) :test #'equal)))

	 (setf -grphash grphash)

	 (do ((curindex 1 (1+ curindex))
	      ;; do first pass separate since we know table empty
	      (rest (mapcan #'(lambda (rdag)
				(if (null (refs-of-ref-dag rdag))
				    (progn
				      (setf (ref-dag-index rdag) 0
					    (ref-dag-validity rdag) 0)
				      (hashoid-set grphash (oid-of-ref-dag rdag) rdag)
				      nil)
				    (list rdag)))
			    acc-valid0)))
	     ((null rest))

	   (let ((acc nil))
	     (dolist (rdag rest)
	       (if (forall-p #'(lambda (oid)
				 (let ((dag (hashoid-get grphash oid)))
				   (and dag
					(let ((i (index-of-ref-dag dag)))
					  (and i (< i curindex))))))
			     (refs-of-ref-dag rdag))
		   (progn
		     (setf (ref-dag-index rdag) curindex
			   (ref-dag-validity rdag) 0)
		     (hashoid-set grphash (oid-of-ref-dag rdag) rdag))
		   (push rdag acc)))

	     (if (= (length acc) (length rest))
		 (setf acc-valid0 acc
		       rest nil)
		 (setf rest acc))))

	 (setf *ref-grphash* grphash)

	 ;; TODO :at some point want to do more with valid nil and remainder valid0
	 )))))
;;  
;;  add_lemma_property : tok -> (object_id -> tok -> term -> *) -> (tok -> *)
;;
;;	map_statements_wprops 
;;	  (bool # *)
;;	    -> (tok{prop name} -> object_id -> tok{object name} -> term -> **)
;;	    -> ** -> (bool # *)

(defunml (|destroy_lemma_property| (name))
    (tok -> unit)
  (map-statement-table
   #'(lambda (oid def)
       (declare (ignore oid))
       (abstraction-property-delete name def)))
  nil)

(defun destroy-all-lemma-property-caches ()
  (map-statement-table
   #'(lambda (oid def)
       (declare (ignore oid))
       (setf (abstraction-properties def) nil))))

(defvar *mappable-lemmas* (cons nil nil))

(defunml (|set_mappable_lemmas| (f))
    ((object_id -> (tok -> (term -> bool))) -> unit)

  (let ((tv (get-definition-table-visibility (resource 'statements)))
	(acc nil)
	(i 0))
    
    (map-statement-table
     #'(lambda (oid def)
	 (when (statement-visible-p def tv)
	   (incf i)
	   (with-ml-evaluation map_lemmas
	     (let ((r (ap f oid (name-of-lemma def) (lemma-of-statement def))))
               ;;(format t "mappable ~a " (name-of-lemma def))
	       (when r
                 (format t "set prop ~a ~%" (name-of-lemma def))
		 (push (cons oid def) acc)))))))

    (format t "set_mappable_lemmas, visible: ~a, mappable : ~a~%" i (length acc))
    (setf  *mappable-lemmas* (cons t (nreverse acc)))
    ;;(break "smlw")
    nil
    ))

(defunml (|reset_mappable_lemmas| (unit) :declare ((declare (ignore unit))))
    (unit -> unit)

  (setf (car *mappable-lemmas*) nil))


;; doesn't check visibility, assumes set_mappable did that.
(defun map-mappable-lemmas (init prop f)
			     
  ;;(format t "map_lemmas_mappable ~%")
  (let ((propname (car prop))
	(propf (cdr prop)))

    (block earlyexit
      (let ((r init))
	  (dolist (x (cdr *mappable-lemmas*))
	    (let ((oid (car x))
		  (def (cdr x)))

	      (let ((val (abstraction-property-get propname def)))
                (setf -val val -def def -prop prop)
		(setf r
		      (with-ml-evaluation map_lemmas
			(if val
			    (ap f (cdr val) r)
			    (ap f (abstraction-property-set propname
							    (ap propf oid (name-of-lemma def) (lemma-of-statement def))
							    def)
				r))))
	;;	(setf -r r)(break "mml")
		(unless (car r)
		  (return-from earlyexit (cdr r))))))
	(cdr r))))) 

(defunml (|map_lemmas_wprops| (init prop f))
    ((bool |#| *)
     -> ((tok |#| (object_id -> (tok -> (term -> **))))
	 -> ((** -> ((bool |#| *) -> (bool |#| *))) -> *)))

  (if (car *mappable-lemmas*)
      (map-mappable-lemmas init prop f)
      (let ((propname (car prop))
	    (propf (cdr prop)))
	(format t "map_lemmas_wprops~%")

	(block earlyexit
	  (let ((r init))
	    (let ((tv (get-definition-table-visibility (resource 'statements))))
	      (map-statement-table
	       #'(lambda (oid def)
		   (when (statement-visible-p def tv)
		     (let ((val (abstraction-property-get propname def)))
		       (setf r
			     (with-ml-evaluation map_lemmas
			       (if val
				   (ap f (cdr val) r)
				   (ap f (abstraction-property-set propname
								   (ap propf oid (name-of-lemma def) (lemma-of-statement def))
								   def)
				       r))))
		       (unless (car r)
			 (return-from earlyexit (cdr r))))))))
	    (cdr r))))))

(defun lookup-lemma-property (name oid)
 (let ((def (lookup-statement-def oid nil t)))
   (when (null def)
     (raise-error (oid-error-message (list oid) '(lemma not))))

   (cdr (or (abstraction-property-get name def)
	    (raise-error (oid-error-message (list oid) (cons name '(lemma property not))))))))


(defun name-of-lemma (l) (definition-name l))

(defun import-lemma (term)
  (let* ((substance (term-to-data term))
	 (lsubstance (provide-data substance 'statement-substance)))
    (make-statement :substance lsubstance
		    :name (name-property-of-substance lsubstance)
		    :ref-dag (new-ref-dag (current-object-id)
					  lsubstance
					  (when (proofs-of-statement-substance lsubstance)
					    0)))))

(defunml (|map_lemmas| (init f))
    ((bool |#| *) -> ((object_id -> (tok -> ((bool |#| *) -> (bool |#| *)))) -> *))

  (format t "map_lemmas~%")
  (block earlyexit
    (let ((r init))
      (let ((tv (get-definition-table-visibility (resource 'statements))))
	(map-statement-table
	 #'(lambda (oid def)
	     (when (statement-visible-p def tv)
	       (setf r (ap f oid (name-of-lemma def) r))
	       (unless (car r)
		 (return-from earlyexit (cdr r)))))))
      (cdr r))))

#|
(defunml (|set_statement_table_visibility| (index oids))
    (object_id -> ((object_id list) -> unit))

  (dolist (oid oids)
    (let ((def (lookup-statement-def oid t t)))
      (when def
	(let ((ref-dag (ref-dag-of-statement def)))
	  (setf (ref-dag-visibility ref-dag) index)))))
  nil
  )
|#

(defun lemma-table (stamp tag)
  ;;(break "lt")
  (new-statement-table stamp tag #'import-lemma))

(defun allocate-statements (stamp tag)
  (lemma-table stamp tag))


(defun object-lookup (name)
  (or (let ((defs (or (name-table-lookup (environment-resource 'statements) name (current-transaction-stamp))
		      (name-table-lookup (environment-resource 'abstractions) name (current-transaction-stamp))
		      (name-table-lookup (environment-resource 'rules) name (current-transaction-stamp))
		      ;; (if (resource-p 'comments)
		      ;;(name-table-lookup (environment-resource 'comments) name (current-transaction-stamp)))
		      )))
	(when defs
	      (if (cdr defs)
		  (raise-error (oid-error-message (mapcar #'oid-of-definition defs)
						  '(ref object lookup multiple)
						  name))
		(oid-of-definition (car defs)))))
      (progn (setf nn name);;(break "n")
	     (raise-error (error-message '(ref object lookup none) name)))))

(defun lemma-lookup (name &optional bypass-vischeck nil-ok-p)

  (let ((statements (environment-resource 'statements)))
    (labels ((look ()
	       (name-table-lookup statements name (current-transaction-stamp))))
      
      
      (or (if bypass-vischeck
	      (let ((defs (look)))
		(when defs
		  (if (cdr defs)
		      (progn ;;(setf -defs defs) (break "lemma-lookup ")
			     (raise-error (oid-error-message (mapcar #'oid-of-definition defs)
							     '(ref lemma lookup multiple)
							     name)))
		      (oid-of-definition (car defs)))))

	      (let ((tv (get-definition-table-visibility statements)))
		(let ((defs (mapcan #'(lambda (statement)
					(when (statement-visible-p statement tv)
					  (list statement)))
				    (look))))
		  ;;(setf -tv tv -defs defs -look #'look) (break "ll")
		  (when defs
		    (if (cdr defs)
			(progn ;;(setf -defs defs) (break "lemma-lookup ")
			       (raise-error (oid-error-message (mapcar #'oid-of-definition defs)
							       '(ref lemma lookup multiple)
							       name)))
			(oid-of-definition (car defs)))))))
      
	  (unless nil-ok-p
	    (setf -name name);; (break "n")
	    (raise-error (error-message '(ref lemma lookup none) name)))))))

(defun find-duplicate-lemmas ()
  (let ((table (environment-resource 'statements))
	(duplicates nil))
    
    (name-table-map-2 table (current-transaction-stamp)
		      #'(lambda (x defs) 
			  (when (cdr defs)
			    ;;(setf vv defs) (break)
			    (let ((oids (sort-oids-by-time
					 (mapcan #'(lambda(x) (when x (list (oid-of-definition x))))
						 defs))))
	       
			      ;;(setf ii (cdr oids)) (break)
			      (setf duplicates (append (cdr oids) duplicates))))))
    duplicates))
				
  
(defun find-duplicate-abstractions ()
  (let ((table (environment-resource 'abstractions))
	(duplicates nil))
    
    (name-table-map-2 table (current-transaction-stamp)
		      #'(lambda (x defs) 
			  (when (cdr defs)
			    ;;(setf vv defs) (break)
			    (let ((oids (sort-oids-by-time
					 (mapcan #'(lambda(x) (when x (list (oid-of-definition x))))
						 defs))))
	       
			      ;;(setf ii (cdr oids)) (break)
			      (setf duplicates (append (cdr oids) duplicates))))))		
    duplicates))
				

(defunml  (|lemma_lookup| (name))
    (tok -> object_id)
  (lemma-lookup name))

;; succeeds even if not visible.
(defunml  (|lemma_exists_p| (name))
    (tok -> bool)
  ;;(format t "lemma_exists_p ~a~%" name)
  (and (lemma-lookup name t t)
       t))

(defunml  (|object_lookup| (name))
    (tok -> object_id)
  (object-lookup name))

(defunml  (|name_of_lemma| (oid))
    (object_id -> tok)
  (or (name-of-lemma (lookup-statement-def oid nil t))
      (raise-error (oid-error-message (list oid) '(lemma name not)))))

(defunml  (|find_duplicate_lemmas| (unit))
    (unit -> (object_id list))
  (find-duplicate-lemmas))
  
(defunml  (|find_duplicate_abs| (unit))
    (unit -> (object_id list))
  (find-duplicate-abstractions))
  

;; list of defs.
(defun lemmas-lookup (name)
  (name-table-lookup (environment-resource 'statements)
		     name
		     (current-transaction-stamp)))
  
;;;;	tstamp seq 
;;;;	
;;;;	
;;;;	


#|
(defun lemma-table (stamp tag)
  (name-table stamp tag nil
    :import-f #'(lambda (term)
		  (import-lemma-definition term))
    ))

(defun lemma-table-insert (def)
  (let* ((table (environment-resource 'statements))
	 (lemmas (lemmas-of-lemma-table table))
	 (name (name-of-lemma def)))

    (when name
      (let ((cur-oids (gethash name lemmas)))

	(when cur-oids
	  (message-emit (warn-message '(ref lemma insert name dup) name)))

	(setf (gethash name lemmas) (cons (oid-of-definition def) cur-oids))))))


(defun lemma-table-delete (def)
  (let* ((table (environment-resource 'statements))
	 (lemmas (lemmas-of-lemma-table table))
	 (name (name-of-lemma def)))

    (when name
      (let ((cur-oids (gethash name lemmas)))
	(setf (gethash name lemmas)
	      (remove (oid-of-definition def) cur-oids #'test #'equal-oids-p))))))

    :insert-f #'(lambda (def s i)
		  (declare (ignore s i))
		  (lemma-table-insert def))
    :delete-f #'(lambda (def s i)
		  (declare (ignore s i))
		  (lemma-table-delete def))
|#

#|
(defstruct (rule-table (:include definition-table))
  (rule-ids (make-hash-table))
  )

(defun rule-ids-of-rule-table (table)
  (rule-table-rule-ids table))

(defun rule-id-table-insert (def)
  (let* ((rules (environment-resource 'rules))
	 (rule-ids (rule-ids-of-rule-table rules))
	 (id (id-of-rule-definition def))
	 (cur-def (gethash id rule-ids)))

    (if cur-def
	(message-emit (oid-error-message (list (oid-of-definition def)) '(ref insert id) id))
	(setf (gethash id rule-ids) def))))

(defun rule-id-table-delete (def)
  (let ((rule-ids (rule-ids-of-rule-table (environment-resource 'rules))))
    (let ((hdef (gethash (id-of-rule-definition def) rule-ids)))
      (if (eql hdef def)
	  (setf (gethash (id-of-rule-definition def) rule-ids)
		nil)
	  (message-emit (oid-error-message (list (oid-of-definition def)
						 (oid-of-definition hdef))
					   '(ref delete id) (id-of-rule-definition def)))))))
|#


(defun rule-id-table-lookup (id)
  (let ((defs (name-table-lookup (environment-resource 'rules) id (current-transaction-stamp))))
    (when (null defs)
      (raise-error (error-message '(ref rule lookup) id)))
    (when (cdr defs)
      (raise-error (error-message '(ref rule lookup duplicates) id)))
    (car defs)))


(defun ref-rule-table (stamp tag)
  (name-table stamp (list 'rule tag) nil
	      :import-f #'ref-import-rule-definition))


;;;
;;; primitve terms.
;;;



(define-primitive |universe| ((level-expression . level)))

(define-primitive |void|)

(define-primitive |object|)

(define-primitive |any| () (term))

(define-primitive |atom|)

(define-primitive |token| ((token . atom)))

(define-primitive |int|)

(define-primitive |natural_number| ((natural . number)))

(define-primitive |minus| () (term))

(define-primitive |add| () (leftterm rightterm))

(define-primitive |subtract| () (leftterm rightterm))

(define-primitive |multiply| () (leftterm rightterm))

(define-primitive |divide| () (leftterm rightterm))

(define-primitive |remainder| () (leftterm rightterm))

(define-primitive |ind| () (value (2 . downterm) baseterm (2 . upterm)))

(define-primitive |list| () (type))

(define-primitive |nil|)

(define-primitive |cons| () (head tail))

(define-primitive |list_ind| () (value baseterm (3 . upterm)))

(define-primitive |union| () (lefttype righttype))

(define-primitive |inl| () (term))

(define-primitive |inr| () (term))

(define-primitive |decide| () (value (1 . leftterm) (1 . rightterm)))

(define-primitive |product| () (lefttype (1 . righttype)))

(define-primitive |pair| () (leftterm rightterm))

(define-primitive |equal| () (type leftterm rightterm))

(defun terms-of-equal-term (term)
  (mapcar #'term-of-bound-term-f (cdr (bound-terms-of-term term))))

(define-primitive |spread| () (value (2 . term)))

(define-primitive |function| () (lefttype (1 . righttype)))

(define-primitive |lambda| () ((1 . term)))

(define-primitive |apply| () (function arg))

(define-primitive |quotient| () (lefttype (2 . righttype)))

(define-primitive |set| () (lefttype (1 . righttype)))

(define-primitive |axiom|)

(define-primitive |less_than| () (leftterm rightterm))

;;(define-primitive |term_of| ((token . name)))

(define-primitive |atom_eq| () (leftterm rightterm if-term else-term))

(define-primitive |int_eq| () (leftterm rightterm if-term else-term))

(define-primitive |less| () (leftterm rightterm if-term else-term))

(define-primitive |tag| ((natural . tag)) (term))

(define-primitive |rec| () ((1 . term)))

(define-primitive |rec_ind| () (value (2 . term)))


(defun independent-product-term (l r)
  (product-term l (get-dummy-variable-id) r))
  
(defun dependent-product-term-p (term)
  (and (product-term-p term)
       (not (dummy-variable-id-p
	      (binding-of-righttype-of-product-term term)))))

(defun independent-product-term-p (term)
  (and (product-term-p term)
       (dummy-variable-id-p
	 (binding-of-righttype-of-product-term term))))

(defun dependent-set-term-p (term)
  (and (set-term-p term)
       (not (dummy-variable-id-p
	      (binding-of-righttype-of-set-term term)))))

(defun injection-term-p (term)
  (or (inl-term-p term)
      (inr-term-p term)))

(defun not-term (term)
  (function-term term (get-dummy-variable-id) (void-term)))
  
(defun not-term-p (term)
  (and (function-term-p term)
       (void-term-p (righttype-of-function-term term))))

(defun term-of-not-term (term)
  (lefttype-of-function-term term))

;; is term function(equal(...); (x).void), not is term not equal(...)
(defun not-equal-term-p (term)
  (and (not-term-p term)
       (term-of-not-term (lefttype-of-function-term term))))

;; is term function(less (...); (x).void), not is term not less(...)
(defun not-less-than-term-p (term)
  (and (not-term-p term)
       (less-than-term-p (term-of-not-term term))))

(defun int-equal-term-p (term)
  (and (equal-term-p term)
       (int-term-p (type-of-equal-term term))
       (= 3 (length (bound-terms-of-term term)))))

(defun reflexive-equal-term (type term)
  (equal-term type term term))

(defun reflexive-equal-term-p (term)
  (or (eq (leftterm-of-equal-term term)
	  (rightterm-of-equal-term term))
      (compare-terms-p (leftterm-of-equal-term term)
		       (rightterm-of-equal-term term))))


(defun int-equal-term (t1 t2)
  (equal-term (int-term) t1 t2)) 

(defun not-int-equal-term (t1 t2)
  (not-term (int-equal-term t1 t2)))

(defun zero-term ()
  (natural-number-term 0))

(defun zero-term-p (term)
  (and (natural-number-term-p term)
       (eql 0 (number-of-natural-number-term term))))

(defun equal-zero-term (term)
  (int-equal-term term (zero-term)))

(defun not-equal-zero-term (term)
  (not-term (equal-zero-term term)))


(defun not-less-than-term (t1 t2)
  (not-term (less-than-term t1 t2)))

(defun less-than-zero-term (term)
  (less-than-term term (zero-term)))

(defun not-less-than-zero-term (term)
  (not-term (less-than-zero-term term)))

(defun injection-term (id term)
  (instantiate-term (instantiate-operator id nil)
		  (list (instantiate-bound-term term))))

(defun integer-term-p (term)
  (or (natural-number-term-p term)
      (and (minus-term-p term)
	   (natural-number-term-p (term-of-minus-term term)))))

(defun integer-term (n)
  (if (< n 0 )
      (minus-term (natural-number-term (- 0 n)))
      (natural-number-term n)))

(defun integer-of-integer-term (term)
  (cond
    ((natural-number-term-p term) (number-of-natural-number-term term))
    ((minus-term-p term) (- 0 (number-of-natural-number-term (term-of-minus-term term))))
    (t (error "this might be changed to return nil rather than bomb like this"))))

(defun binary-integer-term (id subterm1 subterm2)
  (instantiate-term (instantiate-operator id nil)
		  (list (instantiate-bound-term subterm1)
			(instantiate-bound-term subterm2))))


(defun leftterm-of-binary-term (term)
  (term-of-bound-term (car (bound-terms-of-term term))))
		    
(defun rightterm-of-binary-term (term)
  (term-of-bound-term (cadr (bound-terms-of-term term))))

(defun term-of-injection-term (term)
  (term-of-bound-term (car (bound-terms-of-term term))))

(defun leftterm-of-decision-term (term)
  (term-of-bound-term (first (bound-terms-of-term term))))

(defun rightterm-of-decision-term (term)
  (term-of-bound-term (second (bound-terms-of-term term))))

(defun if-term-of-decision-term (term)
  (term-of-bound-term (third (bound-terms-of-term term))))

(defun else-term-of-decision-term (term)
  (term-of-bound-term (fourth (bound-terms-of-term term))))


;; tag{n}(0) (term)
;; tag{n,b}(0; 0) (term; abs list)
;; tag{b}(0; 0) (term; compseq)
(defun tagged-term-p (term)
  (and (eql *tag* (id-of-term term))
       (let ((parms (parameters-of-term term)))
	 (and parms
	      (or (natural-parameter-p (car parms))
		  (bool-parameter-p (car parms)))
	      (let ((bts (bound-terms-of-term term)))
		(and bts (null (bindings-of-bound-term (car bts)))
		     (cond
		       ((bool-parameter-p (car parms)) ; ie compseq
			(and (null (cdr parms))
			     (not (null (cdr bts)))
			     (null (bindings-of-bound-term (cadr bts)))
			     (null (cddr bts))))
		       ((null (cdr parms)) ; ie usual
			(null (cdr bts)))
		       ((null (cddr parms)) ; ie abs list
			(and (null (cddr parms))
			     (bool-parameter-p (cadr parms))
			     (not (null (cdr bts)))
			     (null (cddr bts))
			     (null (bindings-of-bound-term (cadr bts)))))
		       (t nil))))))))


(defun tag-of-tagged-term (term)
  (value-of-parameter-r (car (parameters-of-term term))))

(defun term-of-tagged-term (term)
  (term-of-bound-term (car (bound-terms-of-term term))))

(defun dir-of-obids-tagged-term (term)
  (value-of-parameter-r (cadr (parameters-of-term term))))

(defun ids-of-tagged-term (term)
  (map-ilist-to-list (term-of-bound-term (cadr (bound-terms-of-term term)))
		     (icons-op)
		     #'oid-of-ioid-term))

(defun compseq-tagged-term-p (term)
  (bool-parameter-p (car (parameters-of-term term))))

(defun obids-tagged-term-p (term)
  (let ((ps (parameters-of-term term)))
    (and (cdr ps)
	 (bool-parameter-p (cadr ps)))))

(defun dir-of-compseq-tagged-term (term)
  (value-of-parameter-r (car (parameters-of-term term))))

(defun compseq-of-compseq-tagged-term (term)
  (term-of-bound-term (cadr (bound-terms-of-term term))))



(defun occurs-positively (var term)

  (let ((var (value-of-parameter-value var)))

    (unless (variable-id-p var)
      (raise-error (error-message '(occurs-positively)
				  (parameter-value-to-pretty-string var *variable-type*))))

    (labels 
	((occurs-positively-in-bound-term-list (bound-term-list)
	   (mapc #'(lambda (bound-term)
		     (unless (member var (bindings-of-bound-term bound-term))
		       (occurs-positively-in-term (term-of-bound-term bound-term))))
		 bound-term-list))

	 (occurs-positively-in-term (term)
	   (if (let ((abstraction (abstraction-of-term term)))
		 ;; RLE ???
		 (and abstraction (expansion-of-abstraction abstraction)))
	       (progn
		 (when (occurs-free-p var term)
		   (return-from occurs-positively nil))
		 (occurs-positively-in-bound-term-list (bound-terms-of-term term)))
	       (cond
		 ((decide-term-p term)
		  (when (occurs-free-p var (value-of-decide-term term))
		    (return-from occurs-positively nil))
		  (occurs-positively-in-bound-term-list 
		   (cdr (bound-terms-of-term term))))

		 ((ind-term-p term)
		  (when (occurs-free-p var (value-of-ind-term term))
		    (return-from occurs-positively nil))
		  (occurs-positively-in-bound-term-list
		   (cdr (bound-terms-of-term term))))

		 ((list-ind-term-p term)
		  (when (occurs-free-p var (value-of-list-ind-term term))
		    (return-from occurs-positively nil))
		  (occurs-positively-in-bound-term-list 
		   (cdr (bound-terms-of-term term))))
		 
		 ((spread-term-p term)
		  (when (occurs-free-p var (value-of-spread-term term))
		    (return-from occurs-positively nil))
		  (occurs-positively-in-bound-term-list 
		   (cdr (bound-terms-of-term term))))

		 ((function-term-p term)
		  (when (occurs-free-p var (lefttype-of-function-term term))
		    (return-from occurs-positively nil))
		  (unless (eql var (binding-of-righttype-of-function-term term))
		    (occurs-positively-in-term (righttype-of-function-term term))))

		 ((apply-term-p term)
		  (when (occurs-free-p var (arg-of-apply-term term))
		    (return-from occurs-positively nil))
		  (occurs-positively-in-term (function-of-apply-term term)))
	 
		 ((rec-ind-term-p term)
		  (when (occurs-free-p var (value-of-rec-ind-term term))
		    (return-from occurs-positively nil))
		  (unless (member var (bindings-of-term-of-rec-ind-term term))
		    (occurs-positively-in-term (term-of-rec-ind-term term))))

		 (t (occurs-positively-in-bound-term-list (bound-terms-of-term term))) ))))

      (occurs-positively-in-term term)
      t)))



(defvar *ref-current-objects*)

(defun ref-current-objects (&optional unbound-ok-p)
  (if (boundp '*ref-current-objects*)
      *ref-current-objects*
      (unless unbound-ok-p
	(raise-error (error-message '(ref current_objects))))))

(defunml (|refiner_current_objects| (unit) :declare ((declare (ignore unit))) :error-wrap-p nil)
    (unit -> (object_id list))
  (if (boundp '*ref-current-objects*)
      *ref-current-objects*
      (breakout evaluation '|refiner_current_object|)))

(defunml (|refiner_object_current_p| (oid) :error-wrap-p nil)
    (object_id -> bool)
  (if (boundp '*ref-current-objects*)
      (and (member oid *ref-current-objects* :test #'equal-oids-p) t)
      (breakout evaluation '|refiner_current_object|)))



;;;
;;;	ML 
;;; 


(defstruct argument
  (syntax nil)
  (value nil))

(defun syntax-of-argument (a) (argument-syntax a))
(defun value-of-argument (a) (argument-value a))

(defun define-ml-ref-primitive-types ()

  (add-primitive-type '|proof|
		      #'(lambda (p) (declare (ignore p)) "a proof")
		      :member-p #'sequent-p
		      :eq-func #'(lambda (a b)
				   (if *process-break*
				       (break "proof-=")
				       (format t "proof-=~%"))
				   (message-emit (warn-message '(= proof)))
				   (eql a b)))

  (add-primitive-type '|assumption|
		      #'(lambda (a) (declare (ignore a)) "an assumption")
		      :member-p #'assumption-p
		      :eq-func #'(lambda (a b)
				   (break "assumption-=")
				   (if *process-break*
				       (break "assumption-=")
				       (format t "assumption-=~%"))
				   (message-emit (warn-message '(= assumption)))
				   (eql a b)))

  (add-primitive-type '|rule|
		      #'(lambda (r) (format-string "a ~a.~a rule" (type-of-rule r) (id-of-rule r)))
		      :member-p #'rule-p
		      :eq-func #'(lambda (a b)
				   (if *process-break*
				       (break "rule-=")
				       (format t "rule-=~%"))
				   (message-emit (warn-message '(= rule)))
				   (eql a b)))
  
  (add-primitive-type '|argument|
		      #'(lambda (arg) (format-string "a ~a arg" (syntax-of-argument arg)))
  		      :member-p #'argument-p
		      :eq-func #'(lambda (a b)
				   (if *process-break*
				       (break "arg-=")
				       (format t "arg-=~%"))
				   (message-emit (warn-message '(= argument)))
				   (eql a b))))


(define-ml-ref-primitive-types)


(defun count-proof-nodes (proof)
  (+ 1
     (if (and (refined-proof-node-p proof)
	      (nml-tactic-rule-p (rule-of-proof-node proof)))
	 (count-proof-nodes (proof-of-top-rule proof))
	 0)
     (reduce #'+ (mapcar #'count-proof-nodes (children-of-proof-node proof)) :initial-value 0)))




;;;;	
;;;;	Touch History : environment touch history is transaction driven,
;;;;	  while tactic touch history is demand driven. So, this is
;;;;	  intermediary to store history from transactions until demanded
;;;;	  by tactics.
;;;;	
;;;;	
;;;;	
;;;;	
;;;;	



(defstruct refiner-sub-environment
  ;;touch-history
  )

;;(defun touch-history-of-refiner-sub-environment (e) (refiner-sub-environment-touch-history e))


(defunml (|make_refiner_environment| (addr purposes resources table-types reduction-tags))
  ((tok list) -> ((tok list) -> ((tok list) -> ((tok list) -> ((tok list) -> unit)))))

  (when (component-environment-p addr)
	(raise-error (error-message '(environment new exists) addr)))

  (add-environment
   (new-environment addr purposes resources table-types reduction-tags
		    (make-refiner-sub-environment
		     ;;:touch-history (make-touch-history)
		     )))

  nil)

#| 11/2002

(defun refiner-touch-history ()
  (touch-history-of-refiner-sub-environment (sub-of-environment (current-environment))))


(defunml (|touch_history| (unit) :declare ((declare (ignore unit))) :error-wrap-p nil)

    (unit -> ((token |#| (object_id |#| token)) list))

  ;;(when (refiner-touch-history) (format t "<gth>") (break "gth"))

  (list-touch-history (refiner-touch-history)))


(defunml (|clear_touch_history| (unit) :declare ((declare (ignore unit))) :error-wrap-p nil)
    (unit -> unit)

  ;;(when (refiner-touch-history) (format t "<cth>"))
  (clear-touch-history (refiner-touch-history)))
|#

