%
*************************************************************************
*                                                                       *
*    Copyrighted Cornell University 2000                                *
*                                                                       *
*                                                                       *
*                Nuprl Proof Development System                         *
*                ------------------------------                         *
*                                                                       *
*   Developed by the Nuprl group, Department of Computer Science,       *
*   Cornell University, Ithaca NY.  See the release notes for a list    *
*   of the members of the group.                                        *
*                                                                       *
*   Permission is granted to use and modify Nuprl provided this notice  *
*   is retained in derived works.                                       *
*                                                                       *
*                                                                       *
*************************************************************************
%

%[
******************************************************************************
******************************************************************************
ABSTRACTION.ML
******************************************************************************
******************************************************************************
Abstraction manipulation 
]%

%
System defined:
~~~~~~~~~~~~~~
abstraction_definition_of_object : tok -> term # term
abstraction_definition_of_term : term -> term # term
name_of_abstraction_object_of_term : term -> tok

Defined in primitives.ml:
~~~~~~~~~~~~~~~~~~~~~~~~
is_abstraction_term : term -> bool
opid_of_abstraction : term -> tok

Defined in compute-aux.ml
~~~~~~~~~~~~~~~~~~~~~~~~~
is_def_term t
is_def_term_with_opids opids t
tag_abs_for_unfold opids t
tag_any_ab_for_unfold t
fold_and_tag_defs names t

tag_all_abs_for_unfold names t
unfold_abs t

Further functions specific to abstractions should be included in this
file. E.g. soft abstraction handling.
%

% NB. we might allow alternative typing lemmas for primitive terms... %

let abstraction_typing_lemma_name opid =
 opid ^ `_wf`
;;


%[
******************************************************************************
Soft abstractions
******************************************************************************
]%

letref soft_abstractions = [] : tok list ;;

let add_soft_abs new_abs =
   tty_print "add_soft_abs soft_abstractions refvar used, ie not using ref_state";
  soft_abstractions := union new_abs soft_abstractions
 ; () 
;;

let remove_soft_abs old_abs = 
   tty_print "remove_soft_abs soft_abstractions refvar used, ie not using ref_state";
   soft_abstractions := diff soft_abstractions old_abs
 ; ()
;;

letref soft_abstractions_ref_state =
  new_list_ref_state `soft_abstractions` (nil : tok list)
;;

let soft_abstractions_do_updates oid edges oids =
 soft_abstractions_ref_state := ref_state_do_updates soft_abstractions_ref_state oid oids edges
 ; ()
;;

let undeclare_soft_abstractions oid = 
 (soft_abstractions_ref_state := ref_state_remove soft_abstractions_ref_state oid; ())
 ? () 			      
;;

let soft_abstractions_add_data oid data =
 soft_abstractions_ref_state := ref_state_set_data soft_abstractions_ref_state [oid, data]
;;
let soft_abstraction_add oid data =
  reset_ref_environment_data oid;
  add_ref_environment_data oid `soft_abstractions` soft_abstractions_add_data data
;;
let soft_abstraction_add_o oid data =
  reset_ref_environment_data oid;
  add_ref_environment_data oid `soft_abstractions` soft_abstractions_add_data (map name_of_abstraction data)
;;

let ref_add_soft_abs_aux index edges toks =
  soft_abstractions_ref_state :=
    declare_ref_state_data_indirect `soft_abstractions` soft_abstractions_ref_state index toks edges
;;
let ref_add_soft_abs index =
  declare_ref_state_index index `soft_abstractions` index;
  ref_add_soft_abs_aux index
;;

let ref_add_soft_abs_additions toks index edges =
  soft_abstractions_ref_state :=
    ref_state_modify_state soft_abstractions_ref_state toks index edges
;;

let get_soft_abstractions oid = ref_state_get soft_abstractions_ref_state oid;;

let lookup_soft_abstractions () =
  get_soft_abstractions (current_ref_environment_index `soft_abstractions`);;

let default_is_soft_ab t = fail;;

let member_soft_abs l t = member (opid_of_term t) l;;

%let is_soft_ab t =
  member (opid_of_term t)
    (get_soft_abstractions (current_ref_environment_index `soft_abstractions`))
;;
%

letref is_soft_ab = (\t. fail) : term -> bool;;

let build_is_soft_ab uoid =
 is_soft_ab := if (isl uoid) then (\t. fail)
               else member_soft_abs (ref_state_get soft_abstractions_ref_state (outr uoid))
 ; ()
;;

update_ref_environment_cache_hook `soft_abstractions` build_is_soft_ab;;


update_ref_state_view
 (\(). soft_abstractions_ref_state)
 (ref_state_view_list_entry itoken_term)
;;     

let is_hard_ab t = not is_soft_ab t & is_ab_term t
;;

let tag_soft_ab t =
    if is_soft_ab t then tag_term t else failwith `tag_soft_ab`
;;

let unfold_soft_ab t =
  do_computations  (tag_soft_ab t)
;;

let unfold_soft_abs = repeatf unfold_soft_ab ;;

let unfold_all_soft_abs = sweep_down_map unfold_soft_abs ;;

% 9/2002 appears to be supeceded by type-inf
let get_set_type t =
  fst (snd (dest_set t))
;;
%
letrec unfold_soft_abs_with_trace t =
  if is_soft_ab t then 
   ($. (opid_of_term t) # id) 
   (unfold_soft_abs_with_trace (unfold_soft_ab t))
  else
    [],t
;;

% if no soft abstractions, return t %

let unfold_all_but_last_soft_ab t =
  letrec aux t t' =
    if is_soft_ab t' then
      aux t' (unfold_ab t')
    else
      t
  in
  if is_soft_ab t then
    aux t (unfold_ab t)
  else
    t
;;


  

% At some time we should implement this a bit more efficiently. %


let soft_equal a b =
  if (alpha_equal_terms a b) then `true`
  else
  let a' = unfold_soft_abs a in
  let b' = unfold_soft_abs b in
  if (alpha_equal_terms a' b') then `top-soft`
  if opid_of_term a' = opid_of_term b' then
  ( if (alpha_equal_terms (unfold_all_soft_abs a') (unfold_all_soft_abs b')) then
      `soft`
    else
      `false`
  )
  else 
    `false`
;;

%[
******************************************************************************
Attributes
******************************************************************************
NB: abstraction attributes are also known as 
"conditions" by Lisp code.
]%

% PERF: change get_ab_attrs_of_term to use tokens rather than terms %

let get_ab_attrs_of_term term =
  fst (abstraction_of_term term)
  %let (attrs, y) = abstraction_of_term term in
    (map tok_to_string attrs)%
;;

let has_ab_attr t =
  let attrs = get_ab_attrs_of_term t
    in \attr. member (string_to_tok (string_upcase attr)) attrs 
;;

let has_ab_attr_t attr t =
 member attr (get_ab_attrs_of_term t)
;;




%[
******************************************************************************
Match
******************************************************************************
]%



% one simple alternative generation function %

let get_hardened_pr (pat,inst) = 
     if is_soft_ab pat or is_soft_ab inst then
       [unfold_soft_abs pat, unfold_soft_abs inst]
     else
       []
;;

let soft_match = full_match_with_retry get_hardened_pr 0 [] ;;


%
semi_match is provided below for compatibility with existing tactics. 
half_match_with_retry
 defined above is more general, in that it provides for retry 
strategies not involving just unfolding soft abstractions.
%

let semi_match 
  soft polarity meta_parm_vars meta_term_vars pattern instance 
  =
  half_match_with_retry
    (soft => get_hardened_pr | \x.[])
    polarity meta_parm_vars meta_term_vars pattern instance 
;;

let full_soft_match soft pol meta_parm_vars meta_term_vars pattern instance =
  let parm_dset,term_sub =
    semi_match soft pol meta_parm_vars meta_term_vars pattern instance
  in
    parm_sub_to_term_sub (match_parm_dset parm_dset) @ term_sub
;;



%
Simulates so substitution. Useful when bindings for so variables are 
lambda terms rather than bterms.
%

let quasi_so_subst sub t = 
  do_indicated_computations 
   (fo_subst sub (tag_all_so_var_terms (map fst sub) t))
;;


%
Substitution functions for replacing subterm by another subterm or a variable.
Take care to avoid capture.

Also take care to check that none of subt's free vars are bound in t.
%

let replace_subterm_by_new_free_var subt t =
  let v = maybe_new_var (mkv `z`) (bound_vars t @ free_vars t) in
  let vterm = mk_var_term v  
  in
    v, higher_map_with_bvars 
       (\vs t'.if (alpha_equal_terms t' subt) & null (intersection vs (free_vars t')) then 
                 vterm 
               else 
                 fail
       ) t
;;

let replace_subterm u v t =
  let var,t' = replace_subterm_by_new_free_var u t in
    fo_subst [var,v] t'
;;  
