%
*************************************************************************
*                                                                       *
*    Copyrighted Cornell University 2000                                *
*                                                                       *
*                                                                       *
*                Nuprl Proof Development System                         *
*                ------------------------------                         *
*                                                                       *
*   Developed by the Nuprl group, Department of Computer Science,       *
*   Cornell University, Ithaca NY.  See the release notes for a list    *
*   of the members of the group.                                        *
*                                                                       *
*   Permission is granted to use and modify Nuprl provided this notice  *
*   is retained in derived works.                                       *
*                                                                       *
*                                                                       *
*************************************************************************
%

%[
| Create a recursive definition based on the YWType.
| Here is the usage:
|
| 1. Create an ML library object with then name create_<opid>
|
| 2. In this object, call create_yw_type with the following arguments:
|    Operator ID (tok)
|    Prefix ID (tok)
|    Arguments to the type ((var # term) list)
|    Domain (term)
|    Type function (term)
|    Initial argument (term)
|    Well_founded induction tactic on Domain (tactic)
|    Universe for membership (term)
|
| When this function is called, it builds a display form,
| an abstraction object, and a well-formedness theorem object.
|
| We also build desctructors for terms within the recursive type,
| and we try to build tactics that will assist in the well-formedness
| proofs of these destructors.  This is only partially successful,
| since it is too hard in general to figure out what the type is.
]%

%
| Tactic for unfolding a Recursive module.
%
let YRecUnfoldC n opid =
    AllC [UnfoldTopC opid;
	  AddrC [1] (YUnrollC_bo ANDTHENC RedexC);
	  RepeatForC n RedexC;
	  FoldC opid];;

let YRecUnfoldFoldC n opid t1 t2 =
    DoubleMacroC
        (opid ^ `YRec_unfoldC`)
	(YRecUnfoldC n opid)
	t1
	IdC
	t2;;

% Try solving the wf goal %
let YRecModulePiTac n id pi_ids =
    Try (UnivCD
	 THENM RW (RecUnfoldTopC id) (-1)
	 THENM RepeatFor (n - 1) (D (-1))
	 THENM (RepUnfolds pi_ids 0 THEN AbReduce 0)
	 THEN Try (Auto THEN Fail));;

%
| Check for occurrences of the arguments and the recursion term
| in the body.  Figure out which ones are really recursion variables,
| and which ones are parameters.  Return a list of numbers specifying those
| arguments that change during recursion.
%
let analyze_yrec_body ab_term prod_term =
    let (name, params), bterms = dest_term ab_term in
    let occurrences = 
	map fst (find_subterms_with_addrs (\vs,t. is_term name t) prod_term)
    in

    % Now collect a list of booleans for those arguments that change %
    let status =
	accumulate
	   (\status t'.
	        let (), bterms' = dest_term t' in
		let diffs = map2 (\x y. not (alpha_equal_bterms x y)) bterms bterms' in
		    map2 (\x y. x or y) status diffs)
	   (map (\t.false) bterms)
	   occurrences
    in
	status;;

%
| Get the subterms indexed by the status.
%
let yrec_get_subterms status t =
    let (), bt = dest_term t in
    letrec get_them st bt =
        if st = [] then
	    []
	else if hd st then
	    (snd (hd bt)) . (get_them (tl st) (tl bt))
	else
	    get_them (tl st) (tl bt)
    in
	get_them status bt;;

%
| Create the abs object.
%
let create_yrec_abs opid arg_pairs arg_status ab_term prod_type U_term =
    let opids = tok_to_string opid in
    let opidv = tok_to_var opid in
    let opidt = mk_var_term opidv in
    let arg_vars, () = unzip arg_pairs in
    let arg_status_pairs = (zip arg_status arg_vars) in
    let active_vars = mapfilter (\st, i. if st then i else fail) arg_status_pairs in
    let active_terms = map mk_var_term active_vars in
    let new_def =
	if null active_vars then
	    prod_type
	else
	    let new_def1 =
		higher_map
		    (\t. if is_term opid t then
	                     mk_iterated_apply (opidt. yrec_get_subterms arg_status t)
			 else
			     fail)
		    prod_type
	    in
		mk_iterated_apply
		   (mk_simple_term `ycomb` []
		    . mk_iterated_lambda (opidv . active_vars) new_def1
		    . active_terms)
    in
    let ab_wf_thm = mk_iterated_all arg_pairs (mk_member_term U_term ab_term) in
    let wf_text =
	"Unfold `" J
	opids J
	"` 0 THEN Auto"
    in
	[ create_disp_content_for_new_def ab_term (opids J "_df") opids 
        ; create_ab_content_for_new_def ab_term new_def opids
	; create_thm_obj_data ab_wf_thm (itext_term wf_text) (tok_to_string opid J "_wf")
        ]
;;


%
| Create a projection term given an m-product, and we want the
| n'th element.
%
let yrec_make_pi_term term n m =
    if m = 1 then
	term
    else
	letrec f t i =
	    if i = 1 then
		if n = m then
		    t
		else
		    mk_simple_term `pi1` [t]
	    else
		f (mk_simple_term `pi2` [t]) (i - 1)
        in
	    f term n;;
		
%
| Create an abstraction for a projection.
%
let create_yrec_projs opid name arg_pairs prod_pairs term =
    let arg_vars, arg_types = unzip arg_pairs in
    let prod_vars, prod_types = unzip prod_pairs in
    let m = length prod_pairs in
    let v = maybe_new_var `t' (arg_vars @ prod_vars) in
    let names = tok_to_string name in
    let wfrule =
	concatenate_strings
	    ["YRecModulePiTac ";
	     int_to_string m;
	     " `";
	     tok_to_string opid;
	     "` ";
	     mk_string_list
	         "[]" "``" " " "``"
		 (map (\s. names J "_" J (tok_to_string (var_to_tok s))) prod_vars)]
    in
    letrec create_projs sublist ppairs n =
        if not n > m then
	   (let pid, type = hd ppairs in
	    let name' = name %^ `_`% ^ (var_to_tok pid) in
	    let type' = subst sublist type in
	    let ab_term = mk_simple_term name' [mk_var_term v] in
            let ab_def = yrec_make_pi_term (mk_var_term v) n m in
            let ab_wf_thm = mk_iterated_all arg_pairs
	                (mk_all_term v term
			     (mk_member_term type' ab_term))
	    in
	    let subst' = (pid, ab_term).sublist in
	      append   
		[ create_pi_disp_obj_data (tok_to_string name') (tok_to_string name' J "_df")
					  ab_term pid name
		; create_ab_content_for_new_def ab_term ab_def (tok_to_string name')
		; create_thm_obj_data ab_wf_thm (itext_term wfrule) (tok_to_string name' J "_wf")
		]
		(create_projs subst' (tl ppairs) (n + 1))
		)
	else nil
    in
	create_projs [] prod_pairs 1;;


let yrec_proj_AbTac copid m name n =
  letrec PiReduceC n =
	    if n = 1 then
		AbRedexC
	    else
		AddrC [1] (PiReduceC (n - 1)) ANDTHENC AbRedexC
        in		
	    AllC [TryC (AddrC [1] (UnfoldTopC copid));
		  UnfoldTopC name;
		  if m = 1 then
		      IdC
		  else if n = m then
		      PiReduceC (m - 1)
		  else
		      PiReduceC n]
;;


let yrec_AbReduce_build abdata =
 let (opid, [copidp; mp]), [[],args] = destruct_term abdata in
 let copid = destruct_token_parameter copidp in
 let m = destruct_natural_parameter mp in
      map (\name,n. name, yrec_proj_AbTac copid m name n)
          (map_isexpr_to_list icons_op
                              (\t. let (opid, [namep; np]), x = destruct_term t in
			            ( (destruct_token_parameter namep)
			            , (destruct_natural_parameter np)))
			       args)
;;

				   
				   
let mk_ab_reduce_adds opid toks args =
 mk_term (`ab_reduce_adds`, [mk_token_parm opid])
  [ [], indicate_insert_object_id (mk_term (`!oid`, []) [])
  ; [], (mk_tok_func_alist_term toks args)
  ]
;;

let mk_yrec_proj_AbTac_let copid m name n =
 let copids = tok_to_string copid in
 let ucopids = undash_string copids
 and names = tok_to_string name in
 let unames = undash_string names
 and ms = int_to_string m
 and ns = int_to_string n in

 let funcs =  ("yrec_proj_" J ucopids J "_" J ms J "_" J unames J "_" J ns) in

  funcs,
  mk_func_def_term funcs
     [itext_term
       ("yrec_proj_AbTac `" J copids J "` " J ms J " `" J names  J "` " J ns)]
;;

let build_yrec_AbReduce_updates copid m r =
 let upds, letts = unzip (map (\name,n.  mk_yrec_proj_AbTac_let copid m name n) r) in

 let lets_term =
  mk_text_seq `!text_cons`
    (flatten (map (\lets. [lets;  newline_term]) letts)) in

 let upds_term = mk_ab_reduce_adds copid (map fst r) upds in

   mk_defs_updates_content ((tok_to_string copid) J "_AbReduce_conv")
     lets_term upds_term
;;


% Conversions for the projections %
let create_yrec_proj_reductions copid pi_opid data_vars =
  let m = length data_vars in
  letrec aux vars n =
    if null vars then nil
       else let pid.pids = vars in
	    let name = pi_opid %^ `_`% ^ (var_to_tok pid) in
		(name, n) . (aux pids (n + 1))
    in build_yrec_AbReduce_updates copid m (aux data_vars 1)
;;

%
| Create constructors and destructors.
%
let create_yrec_pattern oacc state_flag topid copid arg_pairs term data_pairs =
    let topids = tok_to_string topid in
    let copids = tok_to_string copid in
    let data_vars, data_type = unzip data_pairs in

    % Data contructor %
    letrec create_pattern vars =
	let v.l = vars in
	    if null l then
		mk_var_term (hd vars)
	    else
		mk_pair_term (mk_var_term v) (create_pattern l)
    in
     let pattern = create_pattern data_vars in

    % Display form %
    let break_term = mk_break_term " " in
    let comma_term = mk_text_term "," in
    letrec create_format vars tail =
	if null vars then
	    tail
	else
	    let v.l = vars in
	    let slot = mk_df_slot_format (tok_to_string (var_to_tok v)) "term" "*" in
		if null l then
		    (slot . tail)
		else
		    (slot
		     . comma_term
		     . break_term
		     . create_format l tail)
    in
    let format = 
	(mk_szone_term
	 . mk_pushm_term 0
	 . mk_text_term copids
	 . mk_text_term "("
	 . create_format data_vars [mk_text_term ")";
				    mk_popm_term;
				    mk_ezone_term])
    in

    % Well-formedness theorem for constructor %
    let wf_thm =
	mk_iterated_all arg_pairs
	    (mk_iterated_all data_pairs
	        (mk_member_term term
		    (mk_simple_term copid (map mk_var_term data_vars))))
    in
    let wf_text =
	concatenate_strings
	    ["Unfolds ``";
	     topids;
	     " ";
	     copids;
	     "`` 0 THEN Auto"]
    in
     let ab_terms, pattern_content = (create_pattern_aux oacc state_flag copid pattern update_case_flags format) in
      ab_terms,
      call_oacc pattern_content 
        [create_thm_obj_data wf_thm (itext_term wf_text) (copids J "_wf")]
;;

let yrec_UnfoldFold_additions count topid ab_term prod_type =
  let UnfoldC, FoldC =
     (if count = 0 then
	  UnfoldTopC topid, FoldTopC topid
      else
	  YRecUnfoldFoldC count topid ab_term prod_type)
      ? failwith (`create_rec_module: could not add conversions for: ` ^ topid)
    in
      [(topid, UnfoldC, FoldC)]
;;

let yrec_UnfoldFold_build ufdata =
 let (opid, [topidp; countp]), [[],ab_term; [], prod_type] = destruct_term ufdata in
   yrec_UnfoldFold_additions
     (destruct_natural_parameter countp)
     (destruct_token_parameter topidp)
     ab_term prod_type
;;


let yrec_UnfoldFold_update_term count topid ab_term prod_type =
  mk_ref_update_RecUnfoldFold_term
    "yrec_UnfoldFold_build" 
    (make_term (`args`, [make_token_parameter topid; make_natural_parameter count])
               [[], ab_term; [], prod_type])
;;


let create_yrec_UnfoldFold count topid ab_term prod_type =
  [create_ml_obj_data 
      (include_properties_term [`reference environment additions`, itoken_term `update`]
        (yrec_UnfoldFold_update_term count topid ab_term prod_type))
       ((tok_to_string topid) J "_FoldUnfold_update")]
;;


%
| Now actually create the object.
%
let create_rec_module_aux oacc state_flag topid copid pi_opid arg_pairs data_pairs U_term =
 % Munge the data %
    let n = length data_pairs in
    let arg_vars, arg_types = unzip arg_pairs in
    let data_vars, data_types = unzip data_pairs in
    let data_pairs', [(), last_type] = split (n - 1) data_pairs in
    let prod_type = mk_iterated_product data_pairs' last_type in
    let le_vars = level_vars prod_type in
    let ab_term = mk_term (topid, (map (mk_level_exp_parm o mk_var_level_exp) le_vars))
		          (map (\x. [], mk_var_term x) arg_vars)
    in
    let arg_status = analyze_yrec_body ab_term prod_type in
    let active_count =
	accumulate (\count flag. if flag then count + 1 else count) 0 arg_status

    in
    let content =
              flatten [ (create_yrec_abs topid arg_pairs arg_status ab_term prod_type U_term)
		      ; (create_yrec_projs topid pi_opid arg_pairs data_pairs ab_term)
		      ;	% Add tactics : Modifies global cache. %
		        (create_yrec_UnfoldFold active_count topid ab_term prod_type)
		      ;	% Conversions for the projections %
		        (create_yrec_proj_reductions copid pi_opid data_vars)
		      ] in
      % Add constructor and destructor %
      let pattern_ab_terms, oacc' = create_yrec_pattern (call_oacc oacc content)
                                       state_flag topid copid arg_pairs ab_term data_pairs in
     ( (ab_term, pattern_ab_terms)
     , oacc')
;;


