%
*************************************************************************
*                                                                       *
*    Copyrighted Cornell University 2000                                *
*                                                                       *
*                                                                       *
*                Nuprl Proof Development System                         *
*                ------------------------------                         *
*                                                                       *
*   Developed by the Nuprl group, Department of Computer Science,       *
*   Cornell University, Ithaca NY.  See the release notes for a list    *
*   of the members of the group.                                        *
*                                                                       *
*   Permission is granted to use and modify Nuprl provided this notice  *
*   is retained in derived works.                                       *
*                                                                       *
*                                                                       *
*************************************************************************
%


%
******************************************************************************
Tactic Type Alias
******************************************************************************
This isn't used in this file, but is convenient to put it here.
%

lettype tactic = (proof -> ((proof list) # ((proof list) -> proof)));;


%
******************************************************************************
Sequent manipulation 
******************************************************************************
Proofs can be thought of as having a type defined recursively as follows:

   proof = declaration list # term # refinement # proof list # annotation


If p = [D1;...;Dk],C,R,[p1;...;pn]

Then p is a proof headed by a sequent with declarations D1...Dk, and
conclusion C.  The refinement tactic/rule applied to this sequent is R
and the resulting children proofs are p1...pn. Usually we are working
at the fringe of a proof tree, in which case R and p1...pn are absent.

We deal with here functions for analysing sequents. None of the functions
here do any work on the clauses (types) of a sequent.
%




let concl = conclusion ;;
let hyps = hypotheses ;;
let subproofs_of_proof = children ;;


%
sequent destructors
~~~~~~~~~~~~~~~~~~~
%

let nth_decl = nth_assumption;; 

%
declaration destructors
~~~~~~~~~~~~~~~~~~~~~~~

v,T,b = dest_declaration d 

v is the variable declared by the declaration. (maybe null).
T is the type of the declaration.
b is true if the hypothesis is `hidden'. (i.e. it is not permissible to use
  the computational content of the hypothesis as evidence in an extract term.
%

let dest_declaration = destruct_assumption ;;

let var_of_declaration = id_of_assumption;;
let type_of_declaration = type_of_assumption;;
let term_of_declaration = type_of_assumption;;
let is_hidden_declaration = hidden_assumption_p;;

let var_and_type_of_declaration decl =
  id_of_assumption decl, type_of_assumption decl
;;

% nb null variables are never declared in a hyp list... %

let mk_declaration (v,t,b) = make_assumption v t b ;;


%
more sequent destructors
~~~~~~~~~~~~~~~~~~~~~~~~
%

let declared_vars = map var_of_declaration o hyps ;;

let is_declared p = let vars = declared_vars p 
                    in
                      \v. member v vars ;;

let get_decl_num v p =
  position v (declared_vars p)
;;

let num_hyps proof = length (hyps proof) ;;

let var_of_hyp i p = var_of_declaration (nth_decl i p) ;;
let id_of_type = var_of_hyp ;;

let type_of_hyp i p = type_of_declaration (nth_decl i p) ;;
let h = type_of_hyp ;;

let types_of_hyps p = map type_of_declaration (hyps p) ;;
let hs = types_of_hyps ;;

let dest_hyp i p = var_and_type_of_declaration (nth_decl i p) ;;
let dest_hyps p = map var_and_type_of_declaration (hyps p) ;;
% think again, there's probably a more efficient way.
let dest_full_hyps p = map dest_declaration (hyps p) ;;
%

% If we want this again, we need to rearrange sequent.ml to be loaded after
dest_equal is defined...  might want to give them better names too.
%

%
let hyp_eq_type i p = eq_type (h i p) ;;
let concl_eq_type p = eq_type (concl p);;

let hyp_type i p =
        let A = type_of_hyp i p in
        (fst (dest_equal A)) ? A
;;
let ht = hyp_type ;;

let concl_type p = 
        let A = concl p in
        (fst (dest_equal A)) ? A
;;
%
% 
Clausal Destructors
~~~~~~~~~~~~~~~~~~~
Clauses are a generalised way of handling sequents.
If we have a sequent:

H1,...,Hn |- C

Clause i = 
  (i = 0) C
  (i > 0) H(i)
  (i < 0) H(n+1+i)
%

% perf : multiple accesses to assumptions of proof_node ie num_hyps and callers of get_pos_hyp_num
          maybe unroll some%
let get_pos_hyp_num i p =
  if i GE 0 then i 
  else
    num_hyps p + 1 + i
;;

let clause i p =
  if i = 0 then null_var,concl p 
  else dest_hyp i p
;;

let clause_var i p = fst (clause i p) ;;
let clause_type i p = snd (clause i p) ;;


let has_rule goal =
  ((refinement goal);true)?false;;

let is_refined goal =
  ((children goal);true)?false;;




%
******************************************************************************
manipulation functions for sequent part of proof term
******************************************************************************
%


let dest_sequent p =

   (map dest_declaration (hyps p)),(concl p) ;;

let mk_sequent var_type_hidden_p_trips conc_term =
  make_proof_node
    (map (\v,t,h.make_assumption v t h) var_type_hidden_p_trips)
    conc_term
;;

let mk_simple_sequent var_type_prs conc_term =
  make_proof_node
    (map (\v,t.make_assumption v t true) var_type_prs)
    conc_term
;;

% See bottom of file for additional functions. %

%
******************************************************************************
Proof annotation functions
******************************************************************************
%

abstype arg =
  int + term + term + tok + var + ((var # term) list)
  
  with 
    int_to_arg i        = abs_arg (inl i)
    and tactic_to_arg T = (% mlbreak `tactic_arg`; %failwith `tactic_to_arg`)
    and tactic_text_to_arg T = abs_arg (inr (inl T))
    and term_to_arg t   = abs_arg (inr (inr (inl t)))
    and tok_to_arg k    = abs_arg (inr (inr (inr (inl k))))
    and var_to_arg v    = abs_arg (inr (inr (inr (inr (inl (v))))))
    and sub_to_arg s    = abs_arg (inr (inr (inr (inr (inr (s))))))
 
    and arg_to_int a    = outl (rep_arg a) ? failwith `arg_to_int`
    and arg_to_tactic a = failwith `arg_to_tactic`
    and arg_to_tactic_text a = outl (outr (rep_arg a)) ? failwith `arg_to_tactic_text`
    and arg_to_term a   = outl (outr (outr (rep_arg a))) 
                          ? failwith `arg_to_term`
    and arg_to_tok a    = outl (outr (outr (outr (rep_arg a)))) 
                          ? failwith `arg_to_tok`
    and arg_to_var a    = outl (outr (outr (outr (outr (rep_arg a)))))
                          ? failwith `arg_to_var`
    and arg_to_sub a    = outr (outr (outr (outr (outr (rep_arg a)))))
                          ? failwith `arg_to_sub`
;;



%
We define here an abtract type for proof annotations to allow us to pass extra 
information between tactics. 
(e.g. labels for proofs and arguments for tactics.)
% 

%
Abstract Type for proof annotations
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
%

abstype proof_annotation = 
  (tok # (int + unit))  % for token / integer labels%
  # (tok # arg) list list   % for args to tactics %


 with

  % label funs %

   get_full_label_of_proof_annotation pa =
     %mlbreak `pa`;%
     fst (rep_proof_annotation pa)


   and add_full_label_to_proof_annotation label pa = 
     abs_proof_annotation (label, snd (rep_proof_annotation pa)) 

  % arg funs %
  
   and get_arg_stack_of_proof_annotation pa =
     snd (rep_proof_annotation pa) 

   and add_arg_stack_to_proof_annotation arg_stack pa = 
     abs_proof_annotation (fst (rep_proof_annotation pa),arg_stack) 

  % the empty annotation %

   and null_proof_annotation = abs_proof_annotation ((`main`,inr ()),[])
;;

%
Annotation <-> Proof Interface 
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
defined in lisp:

  annotate_proof : proof -> * -> proof
  annotation_of_proof : proof -> *
  clear_annotation_of_proof : proof -> proof

Using of the first two functions directly is not type safe. Always use the 
functions defined below instead.
%

let get_annotation_of_proof p = 
  (annotation_of_proof p ? null_proof_annotation) 
  : proof_annotation 
;;

let add_annotation_to_proof (a:proof_annotation) p = annotate_proof p a ;; 

%
Label Annotation functions
~~~~~~~~~~~~~~~~~~~~~~~~~~
%

% token label functions %

let add_label_to_proof label p = 
  let pf_ann = get_annotation_of_proof p
  in let (),i = get_full_label_of_proof_annotation pf_ann
  in
    add_annotation_to_proof 
      (add_full_label_to_proof_annotation 
         (label,i)
         pf_ann
      )
      p
;;

let clear_label_of_proof p = 
  add_label_to_proof `main` p
;;

let label_of_proof p =
  %mlbreak `label`;%
  fst (get_full_label_of_proof_annotation (get_annotation_of_proof p))
;;

let is_proof_with_label label p = 
  label_of_proof p = label 
;;

let is_main_proof = is_proof_with_label `main` ;;
let is_aux_proof = $not o is_main_proof ;;

% number label functions %

let add_number_to_proof n p = 
  let pf_ann = get_annotation_of_proof p
  in let tok,() = get_full_label_of_proof_annotation pf_ann
  in
    add_annotation_to_proof 
      (add_full_label_to_proof_annotation 
         (tok,inl n)
         pf_ann
      )
      p
;;
let clear_number_of_proof p = 
  let pf_ann = get_annotation_of_proof p
  in let tok,() = get_full_label_of_proof_annotation pf_ann
  in
    add_annotation_to_proof 
      (add_full_label_to_proof_annotation 
         (tok,inr ())
         pf_ann
      )
      p
;;

let number_of_proof p = 
  outl (snd (get_full_label_of_proof_annotation (get_annotation_of_proof p)))
  ? failwith `number_of_proof`
;;


let is_numbered_proof p = (number_of_proof p ; true) ? false ;;


%
Tactic Argument Annotation functions
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
%

 
let update_current_args_entry i v p =
  let pf_ann = get_annotation_of_proof p in
  let arg_stack = get_arg_stack_of_proof_annotation pf_ann in
  let arg_stack' =
   if null arg_stack then
         [[i, v]] 
   else let h.t = arg_stack in
        (update_alist h i v).t in
  add_annotation_to_proof
    (add_arg_stack_to_proof_annotation arg_stack' pf_ann)
  p
;;

let push_args arg_alist p =
  let pf_ann = get_annotation_of_proof p
  in let arg_stack = get_arg_stack_of_proof_annotation pf_ann
  in
    add_annotation_to_proof 
      (add_arg_stack_to_proof_annotation 
         (arg_alist.arg_stack)
         pf_ann
      )
      p
;;

let get_arg_stack p =
  get_arg_stack_of_proof_annotation 
    (get_annotation_of_proof p)
;;

let get_arg_stack_top p =
  hd (get_arg_stack p) ? failwith `get_arg_stack_top: stack empty`
;;

let get_int_arg i p =
  arg_to_int (apply_alist (hd (get_arg_stack p)) i)
  ? failwith `get_int_arg`
;;
let get_tactic_arg i p =
  text_to_tactic (arg_to_tactic_text (apply_alist (hd (get_arg_stack p)) i))
  ? failwith `get_tactic_arg`
;;
let get_term_arg i p =
  arg_to_term (apply_alist (hd (get_arg_stack p)) i)
  ? failwith `get_term_arg`
;;
let get_tok_arg i p =
  arg_to_tok (apply_alist (hd (get_arg_stack p)) i)
  ? failwith `get_tok_arg`
;;
let get_var_arg i p =
  arg_to_var (apply_alist (hd (get_arg_stack p)) i)
  ? failwith `get_var_arg`
;;
let get_sub_arg i p =
  arg_to_sub (apply_alist (hd (get_arg_stack p)) i)
  ? failwith `get_sub_arg`
;;



% Doesn't fail if arg stack already empty. %
let pop_arg_stack p =
  let pf_ann = get_annotation_of_proof p
  in let arg_stack = get_arg_stack_of_proof_annotation pf_ann
  in
    add_annotation_to_proof 
      (add_arg_stack_to_proof_annotation 
         (tl arg_stack ? [])
         pf_ann
      )
      p
;;

% copy_arg_annotation p p' returns p' with annotations of p %

let copy_arg_annotation p =
  let arg_stack = get_arg_stack p
  in let new_pf_ann = 
       add_arg_stack_to_proof_annotation 
         arg_stack
         null_proof_annotation
  in 
    add_annotation_to_proof new_pf_ann
;;

 

%
******************************************************************************
Proof refinement functions
******************************************************************************
%

% 
when primitive rules are refined, the label annotations are cleared, but the 
argument annotations are propagated.
%

%
let Refine rule_name args p =
  let ps,v = refine (make_primitive_rule rule_name args) p
  in 
    (map (copy_arg_annotation p) ps),v
;;
%
% count refinements, both successes and failures %

letref nrefine_succs = 0 ;;
letref nrefine_fails = 0 ;;

letref refine_succs = [`equality`] ;;
letref refine_fails = [`equality`] ;;
letref refine_rules = [`equality`] ;;

letref debug_refine = false;;

let init_refine_stats =
  nrefine_succs := 0 ;
  nrefine_fails := 0 ;
  refine_succs := [`equality`];
  refine_fails := [`equality`];
  refine_rules := [`equality`]
;;
  
let Refine rule args p = 
  if debug_refine then
   (nrefine_fails := nrefine_fails + 1
  ; refine_rules := append [rule] refine_rules
  ; refine_fails := append [rule] refine_fails; ())
  ; let ps,v = refine (make_primitive_rule rule args) p in 
    (if debug_refine then
       (nrefine_fails := nrefine_fails - 1
      ; nrefine_succs := nrefine_succs + 1
      ; refine_succs := append [rule] refine_succs
      ; refine_fails := tl refine_fails; ())
      ; (map (copy_arg_annotation p) ps),v
    )
;;


let refine_proof p (T:tactic) =
 let ps,v = T p in
   v ps
;;

let refine_proof_with_tactic p tac_string =
  refine_proof 
    p
    (refine (make_tactic_rule (mk_text_term tac_string)))
;;

let mk_proof_refined_by_tactic_term goal Tterm = 
  refine_proof
    (mk_sequent [] goal)
    (refine (make_tactic_rule Tterm))
;;

let mk_proof_refined_by_tactic goal Tstring = 
 mk_proof_refined_by_tactic_term goal (mk_text_term Tstring) 
;;

% transformation tactics run on raw proof type. %

let Frontier = frontier
;; 

let ProofAsTactic p' p =
  if equal_sequents p' p then
    Frontier p' 
  else
    failwith `ProofAsTactic: new sequent not equal to old`
;;

% Use this to run refinement tactics. %

let RunTactic (T:tactic) p = T p
;;

%
******************************************************************************
Manipulating sequents and proofs in library objects
******************************************************************************
%

%
let add_new_theorem_after new_name existing_name =
 if is_lib_member new_name then
   failwith `add_new_theorem: ` ^ new_name ^ ` already exists`
 if not is_lib_member existing_name then
   failwith `add_new_theorem: ` ^ existing_name ^ ` does not exist`
 else
 (execute_command_line
  (concatenate_strings
   ["create " 
   ;tok_to_string new_name
   ;" thm after "
   ;tok_to_string existing_name
   ]
  )
 ? ()
 )      
;;
%

let insert_proof_in_thm_object p name = mlbreak `insert`; failwith `insert_proof_in_thm_object` %transform_theorem name p%;;

%
let add_theorem new_name p position =
  add_new_theorem_after new_name position 
  ;
  insert_proof_in_thm_object 
     p
     new_name
  ;
  ()
;;

let add_theorem_and_run new_name goal tac_string position =
  add_theorem
     new_name
     (refine_proof_with_tactic
        (mk_simple_sequent [] goal)
        tac_string
     )
     position
;;

let add_theorem_with_goal new_name goal position =
  add_theorem
     new_name
     (mk_simple_sequent [] goal)
     position
;;


let copy_theorem old_theorem new_theorem =
  add_theorem_with_goal 
    new_theorem 
    (raw_main_goal_of_theorem old_theorem)
    old_theorem
  ;
  insert_proof_in_thm_object 
    (proof_of_thm_object old_theorem)
    new_theorem
;;

let delete_theorem_proof name =  
  let tname = string_to_tok name
  in let c = raw_main_goal_of_theorem tname
  in
    insert_proof_in_thm_object
      (mk_simple_sequent [] c)
      tname
;;
%

%
******************************************************************************
More sequent functions.
******************************************************************************
%

let extend_sequent p var_type_prs =
  copy_arg_annotation p
  (make_proof_node
    (hyps p @ map (\v,t.make_assumption v t false) var_type_prs)
    (concl p)
  )
;;

let change_concl_of_sequent p new_concl = 
  make_proof_node
    (hyps p)
    new_concl
;;


%
******************************************************************************
Invisible variables
******************************************************************************
%


let mk_invisible_var tok = 
  mkv (`%` ^ tok)
;;

let basic_invisible_var = mkv `%` ;;

let is_invisible_var v =
  let h.t = explode (var_to_tok v) in
    h = `%`
;;
let is_visible_var v = not is_null_var v & not is_invisible_var v
;;


%
******************************************************************************
Sequent Term functions.
******************************************************************************
Mostly for displaying sequents.
Terms:

goal == !sequent(info;seq)

seq == !concl(term)
     | !hyp{i:n}{h:t}(type;x.seq)

1. The variable x is null if no variable is to be shown.
2. h is either t of f: 

    f means hyp is normal.
    t means hyp is "hidden"

3. info is label stuff. 
4. i is the hypothesis number.
%

letref show_labels = true;;

let mk_label_string p = 
  concatenate_strings
    [tok_to_string (label_of_proof p)
    ;is_numbered_proof p 
        => (" " J int_to_string (number_of_proof p))
        |  ""
    ]
;;

%
let mk_concl_term c = 
  mk_simple_term `!concl` [c]
;;

let mk_hyp_term n decl seq = 
  let var,T,hidden = dest_declaration decl 
  in let v = is_invisible_var var =>  null_var | var
  in let h = hidden => `t` | `f`
  in
    mk_term
      (`!hyp`,[mk_natural_parm n;mk_token_parm h])
      [[],T
      ;[v],seq
      ]
;;

let mk_sequent_term info seq = 
  mk_simple_term `!seq` [info;seq]
;;

%
% decls in usual order %

%
let mk_goal_term n_decl_prs conc p = 
  let info_term = 
    show_labels => mk_text_term (mk_label_string p) | void_term
  in
  letrec nest_hyps nd_prs = 
    if null nd_prs then conc else 
    let (n,d).nd_prs' = nd_prs
    in 
      mk_hyp_term n d (nest_hyps nd_prs')
  in
    mk_sequent_term info_term (nest_hyps n_decl_prs)
;;

%
% setting of creation function.
 i = 0: get main goal.
 i > 0: get ith subgoal.

 remove subgoal hyps same as in goal, unless hyps have shrunk.
%


%
let mk_red_goal_term p i =
  let nds = number (hyps p)
  in
  if i = 0 then
    mk_goal_term nds (mk_concl_term (concl p)) p
  else
  ( let p' = nth i (children p)
    in let nds' = number (hyps p')
    in let nds'' = 
       length nds' < length nds 
       =>
       nds' 
       |
       remove_if (\nd.member nd nds) nds'
    in
      mk_goal_term nds'' (mk_concl_term (concl p')) p'
  )
;;


make_red_goal_term :=  (p:proof) (i:int)  ) : term  

make_red_goal_term := mk_red_goal_term ;;
%

let get_decls_and_concl goal = 
  let ((id, parms),
       [([],sequent); ([], pannos)]) = destruct_term goal in

  (map_to_ilist (\h. (fst (snd (dest_declaration h))))
		icons_op
		(filter (\d. let var,T,hidden = dest_declaration d in 
	                 is_invisible_var var)
	                 (assumptions_of_sequent sequent))),
  (conclusion_of_sequent sequent)
;;
