/***************************************************************************
                          algorithm.h  -  description
                             -------------------
    begin                : Tue Aug 5 2003
    copyright            : (C) 2003 by nomore-dg
    email                : nomore-dg@cs.uni-potsdam.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

/*! \file algorithm.h
 *  Declaration of method ColorGraph and classes CNdOperator, COperatorC,
 *  COperatorCH, COperatorD, COperatorDQ and COperatorDH.
 */

#ifndef ALGORITHM_H
#define ALGORITHM_H

/***************************************************************************
 *                                                                         *
 * Includes                                                                *
 *                                                                         *
 ***************************************************************************/
#include "cgraph.h"
#include "coperator.h"
#include "misc.h"

/***************************************************************************
 *                                                                         *
 * Namespace NS_NOMORE                                                     *
 *                                                                         *
 ***************************************************************************/
namespace NS_NOMORE{

/*! \mainpage Source code description of <b>noMoRe++</b>
 *
 *  \section intro Introduction
 *  The source code is distributed into different parts. The next section
 *  give an overview over all written classes and methods interesting for
 *  you.
 *
 *  \section readinglp Reading logic programs
 *  The first step is to read out the logic program from stdin. Easy to use
 *  with the piping mechanism of the operating system. The input structure
 *  is the same like the
 *  <a href="http://www.tcs.hut.fi/Software/smodels/" target="_blank">lparse</a>
 *  output. Use <b>noMoRe++</b> like
 *  <a href="http://www.tcs.hut.fi/Software/smodels/" target="_blank">smodels</a>
 *  on command line. The important classes for internal representation are
 *  CAtom for an simple atom (no literal), CBody for the body of a logic
 *  rule and uses CAtom objects as positive and negative part. The whole
 *  rule is represented by the class CRule which uses as head an atom and
 *  as body a CBody object. For the whole program representation the class
 *  CProgram is written. The important method of this class is
 *  CProgram::Read() which reads the
 *  <a href="http://www.tcs.hut.fi/Software/smodels/" target="_blank">lparse</a>
 *  output from a stream and build up the program.
 *
 */

//! Generate admissible colorings for graphs and print corresponding AS.
/*! \param graph        Graph which should be colored for computing answer
 *                      sets (last coloring is returned).
 *  \param todo         Contains information about all todo lists.
 *  \param op           String which contains the pre-processing operator,
 *                      non-deterministic operator, deterministic operators,
 *                      end-operator and the heuristics for generating
 *                      admissible colorings.
 *  \param as_count     Number of answer sets to search (0 means all
 *                      answer sets).
 *  \param lookahead_op String which contains the deterministic operator
 *                      for the look ahead computation.
 *  \param lookahead    If true look ahead is used for finding best choice
 *                      points.
 *  \return Returns true if the operators are valid.
 */
bool ColorGraph(CGraph* graph,
                CTodoLists *todo,
                string op,
                int as_count,
                string lookahead_op,
                bool lookahead, 
                bool choice_operator);

/***************************************************************************
 *                                                                         *
 * class CNdOperator                                                       *
 *                                                                         *
 ***************************************************************************/

/*! \class CNdOperator algorithm.h
 *  \brief Interface for defining non-deterministic operators like
 *         COperatorC and COperatorD.
 *  \author Christian Anger, Andre Neumann
 *  \date Tue Aug 5 2003
 *
 *  CNdOperator is the base class for all non-deterministic operators. All
 *  these operators must be implements the ChooseNode() method. Use Call()
 *  for computing admissable colorings using some steps below.
 *
 *  -# call deterministic operator (det_op)
 *     - if an error occured then restore coloring if possible and recolor
 *       last choice point (goto step 1)
 *  -# test for total coloring
 *     - if all nodes are colored then write answer set and restore coloring
 *       if possible and recolor the last choice point (goto step 1)
 *  -# choose a new choice point
 *     - without look ahead: choose a node with ChooseNode()       
 *     - with look ahead: call Lookahead()
 *       - if an error occured while computing lookahead then restore
 *         coloring if possible and recolor last choice point (goto step 1)
 *       - if no node is returned and the graph is total colored goto step 1
 *  -# if no node is returned and the graph is not total colored then call
 *     the determistic operator (post_op) else color node as a choice point
 */
class CNdOperator : public COperator {

  //! Output operator.
  friend std::ostream& operator<<(std::ostream& os, CNdOperator& op);

public:
  //! Constructor.
  CNdOperator();

  //! Destructor
	virtual ~CNdOperator();

  //! Starts the answer set computation with deterministic and end-operator.
  /*! \pre  The graph must be generated and the pre processing step is
   *        finished.
   *  \post The answer sets of the graph are computed and written into
   *        parameter sets.
   *  \param det_op       Operator which describe the deterministic part.
   *  \param post_op      Operator which is called if the ChooseNode()
   *                      method fails and the graph is not totally colored.
   *  \param heu          Specifies the heuristic which is used to priorize
   *                      the look ahead nodes.
   *  \param lookahead_op Contains the deterministic operator which is
   *                      used for look ahead tests.
   *  \param graph        Graph which should be colored for computing
   *                      answer sets (last coloring is returned).
   *  \param todo         Contains the todo lists for bp, ps and other
   *                      operators.
   *  \param as_count     Number of answer sets to search (0 means all
   *                      answer sets).
   *  \param lookahead    If true the algorithm use the look ahead to find
   *                      the next choice.
   *  \param sets         Set of TAtomSet's which represent all found
   *                      answer sets.
   *  \return Returns true if it is possible to find more answer sets else
   *          false.
   */
  virtual bool Call(CDetOperator* det_op,
                    CDetOperator* post_op,
                    CHeuristic *heu,
                    CDetOperator* lookahead_op,
                    CGraph* graph,
                    CTodoLists *todo,
                    int as_count,
                    bool lookahead,
                    TSetOfAtomSets* sets);

  //! Restores the graph to the last recolorable choice point.
  /*! \pre  No.
   *  \post The last node onto the stack is a choice point or stack is
   *        empty.
   *  \param graph Specifies the graph which nodes are recolored.
   *  \param todo  Contains the todo lists.
   *  \param stack Defines the stack which holds last changes, after
   *               execution the top element is the plus colored choice
   *               point node. After executing this method the top element
   *               is the last choice point which is colored plus.
   *  \return If no such choice point is found the methode returns false
   *          else true.
   */
  virtual bool RestoreGraph(CGraph* graph,
                            CStack* stack,
                            CTodoLists* todo,
                            bool restore_cps = true);

  //! Prints the answer set and put the answer set to the set of AS's.
  /*! \pre  The graph is totally colored.
   *  \post The current answer set is added to the set of answer sets.
   *  \param nr    Number of the current answer set.
   *  \param graph Graph which is totally colored.
   *  \param sets  Set of answer sets for inserting the current answer set.
   */
  void WriteAnswerSet(int nr,
                      CGraph* graph,
                      TSetOfAtomSets* sets);

  //! Recolores the choice point on the top of the stack.
  /*! \pre  The upper node of the stack is a choice point.
   *        Use RestoreGraph().
   *  \post The upper node is recolored with the new color as choice point.
   *  \param graph Graph on which the last choice node is colored from plus
   *               to minus.
   *  \param stack The top element is the last plus-colored choice point
   *               for recoloring.
   *  \param todo  Contains the todo lists.
   *  \return If the top element is not a plus-colored choice point the
   *          method returns false else true.
   */
  bool RecolorChoicePoint(CGraph* graph,
                          CStack* stack,
                          CTodoLists* todo);

  //! Choose (return) a node as a choice point.
  /*! \pre  No.
   *  \post No.
   *  \param graph Graph contains nodes which are used for finding a
   *               choice point.
   *  \param todo  Contains the todo lists for bp, ps and other operators.
   *  \return Returns the pointer of the choosen body node. If no body node
   *          is choosen the method returns a NULL pointer.
   */
  virtual CNode* ChooseNode(CGraph* graph,
                            CTodoLists* todo) = 0;

  //! Choose (return) a node as a choice point. This method implements lookahead with a given heuristics.
  /*! Execution of lookahead in five steps:
   *  -# first color node plus - if an error occured the node must be
   *     colored minus - if a error occured, too, the lookahead must be
   *     return a null pointer else put all new nodes not yet tested into
   *     lookahead-test-queue
   *  -# second color node minus - if an error occured the node must be
   *     colored plus - if a error occured, too, the lookahead must be
   *     return a null pointer else put all new nodes not yet tested into
   *     lookahead-test-queue
   *  -# save color informations and put the node into a priorized queue
   *  -# goto step 1 until the last node is tested
   *  -# take the best node out of the priorized queue and return true
   *
   *  \pre  No.
   *  \post No.
   *  \param graph        Graph contains nodes which are used for finding
   *                      a choice point.
   *  \param stack        Contains the status of all saved nodes.
   *  \param lookahead_op Deterministic operator which is called to test
   *                      the nodes.
   *  \param det_op       Deterministic operator which is called if can
   *                      only be colored by one color (plus or minus).
   *  \param heu          Heuristics used for generating the rang of a
   *                      tested node.
   *  \param node         Returns the node which is choosen.
   *  \param todo         Contains the todo lists.
   *  \return Returns true if no error occured. If one node is unable to
   *          color plus and minus this method returns false.
   */
  virtual bool Lookahead(CGraph* graph,
                         CStack* stack,
                         CDetOperator* lookahead_op,
                         CDetOperator* det_op,
                         CHeuristic* heu,
                         CNode** node,
                         CTodoLists* todo);
                         
};


/***************************************************************************
 *                                                                         *
 * non-deterministic operator                                              *
 *                                                                         *
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 * class COperatorC                                                        *
 *                                                                         *
 ***************************************************************************/
/*! \class COperatorC algorithm.h
 *  \brief Choose the first body node of the graph which is uncolored.
 *  \author Christian Anger, Andre Neumann
 *  \date Tue Aug 5 2003
 *
 *  COperatorC is the implementation of the choice operator of the paper.
 *  The ChooseNode() method returns the first uncolored body node
 *  (supported or unsupported) of the graph by iterating over all body
 *  nodes.
 */
class COperatorC : public CNdOperator {

public:
  //! Constructor.
	COperatorC();

  //! Destructor.
	virtual ~COperatorC();

  //! Choose the first uncolored node of the graph.
  /*! \pre  No.
   *  \post No.
   *  \param graph Graph contains nodes which are used for finding a
   *               choice point.
   *  \param todo  Contains information about all todo lists.
   *  \return The pointer of the chosen body node, if no body node is
   *          choosen the methode returns NULL.
   */
  virtual CNode* ChooseNode(CGraph* graph, CTodoLists* todo);

  //! Returns an operator specific id which is used for testing of equality.
  inline virtual TType GetType() const {
    
    return type_coperator_c;
    
  }

};

/***************************************************************************
 *                                                                         *
 * class COperatorCQ                                                       *
 *                                                                         *
 ***************************************************************************/
/*! \class COperatorCQ algorithm.h
 *  \brief Choose the top element of the c-choice point queue which is
 *         uncolored (and supported or unsupported).
 *  \author Andre Neumann
 *  \date Tue Aug 5 2003
 *
 *  \attention Use this non-deterministic operator with COperatorPre and
 *             COperatorPs otherwise the c-choice point queue is not filled
 *             with candidates (possible choice points).
 */
class COperatorCQ : public CNdOperator {

public:
  //! Constructor.
	COperatorCQ();

  //! Destructor.
	virtual ~COperatorCQ();

  //! Returns the first node of the choice point list which is uncolored.
  /*! \pre  No.
   *  \post No.
   *  \param graph Graph contains nodes which are used for finding a
   *               choice point.
   *  \param todo  Contains information about all todo lists.
   *  \return The pointer of the chosen body node, if no body node is
   *          choosen the methode returns NULL.
   */
  virtual CNode* ChooseNode(CGraph* graph, CTodoLists* todo);

  //! Restores the graph to the last choice point.
  /*! Also insert new candidates for choice point queue.
   *  \pre  No.
   *  \post The last node onto the stack is a choice point or stack is
   *        empty.
   *  \param graph Specifies the graph which nodes are recolored.
   *  \param stack Defines the stack which holds last changes.
   *  \param todo  Contains information about all todo lists.
   *  \return If no such choice point is found the methode returns false
   *          else true.
   */
  virtual bool RestoreGraph(CGraph* graph,
                            CStack* stack,
                            CTodoLists* todo,
                            bool restore_cps = true);

  //! Returns an operator specific id which is used for testing of equality.
  inline virtual TType GetType() const {

    return type_coperator_cq;

  }

};


/***************************************************************************
 *                                                                         *
 * class COperatorD                                                        *
 *                                                                         *
 ***************************************************************************/
/*! \class COperatorD algorithm.h
 *  \brief Choose the first body node of the uncolored set of a graph which
 *         is support.
 *  \author Christian Anger, Andre Neumann
 *  \date Tue Aug 5 2003
 *
 *  COperatorD is the implementation of the support-driven choice point
 *  operator of the paper. The ChooseNode() method returns the first
 *  supported body node of the uncolored body node set of the graph.
 */
class COperatorD : public CNdOperator {
  
public:
  //! Constructor.
	COperatorD();

  //! Destructor.
	virtual ~COperatorD();

  //! Choose the first uncolored and supported node of the graph.
  /*! \pre  No.
   *  \post No.
   *  \param graph Graph contains nodes which are used for finding a
   *               choice point.
   *  \param todo  Contains information about all todo lists.
   *  \return The pointer of the chosen body node, if no body node is
   *          chosen the methode returns NULL.
   */
  virtual CNode* ChooseNode(CGraph* graph, CTodoLists* todo);

  //! Returns an operator specific id which is used for testing of equality.
  inline virtual TType GetType() const {
    
    return type_coperator_d;
    
  }

};

/***************************************************************************
 *                                                                         *
 * class COperatorDH                                                       *
 *                                                                         *
 ***************************************************************************/
/*! \class COperatorDQ algorithm.h
 *  \brief Choose the top element of the choice point queue which is
 *         supported and uncolored.
 *  \author Andre Neumann
 *  \date Tue Aug 5 2003
 *
 *  \attention Use this non-deterministic operator with COperatorPre and
 *             COperatorPs otherwise the choice point queue is not filled
 *             with candidates (possible choice points).
 */
class COperatorDQ : public CNdOperator {

public:
  //! Constructor.
	COperatorDQ();

  //! Destructor.
	virtual ~COperatorDQ();

  //! Choose the first node of the choice point queue which is supported.
  /*! \pre  No.
   *  \post No.
   *  \param graph Graph contains nodes which are used for finding a
   *               choice point.
   *  \param todo  Contains information about all todo lists.
   *  \return The pointer of the chosen body node, if no body node is
   *          chosen the methode returns NULL.
   */
  virtual CNode* ChooseNode(CGraph* graph, CTodoLists* todo);

  //! Restores the graph to the last choice point.
  /*! Also insert new candidates for choice point queue.
   *  \pre  No.
   *  \post The last node onto the stack is a choice point or stack is
   *        empty.
   *  \param graph Specifies the graph which nodes are recolored.
   *  \param stack Defines the stack which holds last changes
   *  \param todo  Contains information about all todo lists.
   *  \return If no such choice point is found the methode returns false
   *          else true.
   */
  virtual bool RestoreGraph(CGraph* graph,
                            CStack* stack,
                            CTodoLists* todo,
                            bool restore_cps = true);

  //! Returns an operator specific id which is used for testing of equality.
  inline virtual TType GetType() const {

    return type_coperator_dq;

  }

};

/***************************************************************************
 *                                                                         *
 * class COperatorDH                                                       *
 *                                                                         *
 ***************************************************************************/
/*! \class COperatorDH algorithm.h
 *  \brief Choose the best element of the c-choice point queue which is
 *         uncolored (test for best element uses node information set by
 *         lookahead).
 *  \author Andre Neumann
 *  \date Thu Nov 25 2004
 *
 *  \attention Use this non-deterministic operator with COperatorPre
 *             otherwise the c-choice point queue is not filled
 *             with candidates (possible choice points).
 */
class COperatorDH : public COperatorDQ {

public:
  //! Constructor.
	COperatorDH();

  //! Destructor.
	virtual ~COperatorDH();

  //! Returns the first node of the choice point list which is uncolored.
  /*! \pre  No.
   *  \post No.
   *  \param graph Graph contains nodes which are used for finding a
   *               choice point.
   *  \param todo  Contains information about all todo lists.
   *  \return The pointer of the chosen body node, if no body node is
   *          choosen the methode returns NULL.
   */
  virtual CNode* ChooseNode(CGraph* graph, CTodoLists* todo);

  //! Returns an operator specific id which is used for testing of equality.
  inline virtual TType GetType() const {

    return type_coperator_dh;

  }

};

} // end of namespace NS_NOMORE

#endif
