/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         * 
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         * 
 ***************************************************************************/

#ifndef LIBNOMORE_DYNAMIC_HEURISTIC_H
#define LIBNOMORE_DYNAMIC_HEURISTIC_H
#if defined (_MSC_VER) && _MSC_VER <= 1300
#pragma warning (disable : 4786)
#endif
#include <heuristic.h>
#include <graph.h>
namespace NS_NOMORE {

//! base class for dynamic heuristics
/*!
 * To determine the best choice point a dynamic heuristic first evaluates 
 * the heuristic value of each candidate node and then selects the node with 
 * the "best" heuristic value.
 * 
 * I.e. a dynamic heuristic reevaluates the heuristic value of nodes each time
 * a choice point is requested.
 *
 * \par Template parameter: \n
 * - HeuStrat: The policy (aka strategy) to be used to evaluate nodes and to
 * compare two heuristic values.
 * .
 * HeuStrat must provide the following types/typedefs:
 * - ValueType: The type of the heuristic value used by this heuristic (i.e. int)
 * - ComparatorType: A function-type that accepts two values of type ValueType and
 * returns true if the first value is "better" than the second.
 * - EvaluatorType: A function-type that accepts a node.
 * The EvaluatorType shall install a heuristic value of type ValueType in each
 * node for which it is called.
 *
 */
template <class HeuStrat>
class DynamicHeuristic : public Heuristic {
public:
  //! the type of the binary predicate to be used for value comparisons
  typedef typename HeuStrat::ComparatorType ComparatorType;
  
  //! the type of the unary predicate to be used for valuation of nodes.
  typedef typename HeuStrat::EvaluatorType EvaluatorType;
  
  //! the type of the heuristic value
  typedef typename HeuStrat::ValueType ValueType;
  
  //! creates a new object
  /*! 
   * \param c A binary predicate used to compare two values of type ValueType
   * \param e An unary predicate used to valuate nodes.
   */
  DynamicHeuristic(const ComparatorType& c = ComparatorType(), const EvaluatorType& e = EvaluatorType())
    : graph_(0)
    , comparator_(c)
    , evaluator_(e) {
  }
    
  void setGraph(Graph& g) {
    graph_ = &g;
  }
  
  /*!
   * For each uncolored body node of the graph previously set using setGraph,
   * assigns a heuristic value using the EvaluatorType of the used Policy-Parameter. 
   * Then selects the body node with the best heuristic value that satisfies 
   * the given constraint.
   * The Policy's ComparatorType is used for value-comparision.
   * 
   * \pre setGraph was called.
   *
   * \complexity linear in the number of uncolored nodes
   */
  BodyNode* selectBodyNode(const Constraint& c) {
    assert(graph_);
    return getBestNode(graph_->uncoloredBodyNodesBegin(), 
      graph_->uncoloredBodyNodesEnd(), c, (BodyNode*)0);
  }

  /*!
   * For each uncolored head node of the graph previously set using setGraph,
   * assigns a heuristic value using the EvaluatorType of the used Policy-Parameter. 
   * Then selects the head node with the best heuristic value that satisfies 
   * the given constraint.
   * The Policy's ComparatorType is used for value-comparision.
   *
   * \complexity linear in the number of uncolored nodes
   */
  HeadNode* selectHeadNode(const Constraint& c) {
    assert(graph_);
    return getBestNode(graph_->uncoloredHeadNodesBegin(), 
      graph_->uncoloredHeadNodesEnd(), c, (HeadNode*)0);
  }
private:
  template <class It, class NT>
  NT* getBestNode(It first, It last, const Constraint& c, NT*) {
    NT* best = 0;
    for (; first != last; ++first) {
      if (c.isSatisfiedBy(**first)) {
        evaluator_(*first);
        if (best == 0 || comparator_(*first, best))
          best = *first;
      }
    }
    return best;
  }
  
  typedef HVCompare<ValueType, ComparatorType> CMP;
  
  Graph* graph_;
  CMP comparator_;
  EvaluatorType evaluator_;


};

}
#endif
