/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         * 
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         * 
 ***************************************************************************/

#ifndef LIBNOMORE_HEURISTIC_H
#define LIBNOMORE_HEURISTIC_H
#if defined (_MSC_VER) && _MSC_VER <= 1300
#pragma warning (disable : 4786)
#endif
#include <node.h>
#include <generic_value.h>
#include <list>
#include <vector>
#include <constraint.h>

namespace NS_NOMORE {

class BodyNode;
class HeadNode;
class Graph;

//! Base class for Heuristics.
/*!
 * heuristics guide the selection of choice points. They are used together
 * with a \ref ChoiceOperator "choice-operator".
 */
class Heuristic {
public: 
  //! Constructs the heuristics.
  Heuristic();

  //! Destroys the heuristics.
  virtual ~Heuristic();
  
  //! sets the graph on which this heuristic should work.
  virtual void setGraph(Graph& g);

  //! returns the "best" body node according to this heuristic that satisfies the given constraint.
  /*!
   * \param c the constraint a potential best node must satisfy.
   * \return the "best" body node according to this heuristic or 0 if no such node exists
   */
  virtual BodyNode* selectBodyNode(const Constraint& c) = 0;
  
  
  //! returns the "best" body node according to this heuristic that satisfies the given constraint.
  /*!
   * \param c the constraint a potential best node must satisfy.
   * \return the "best" head node according to this heuristic or 0 if no such node exists
   */
  virtual HeadNode* selectHeadNode(const Constraint& c) = 0;
};

/*!
 * This heuristic selects the first uncolored node that satisfies the
 * the given constraint.
 */
class SelectFirst : public Heuristic {
public:
  SelectFirst();
  virtual void setGraph(Graph& g);
  //! returns the first uncolored node of g that satisfies c
  virtual BodyNode* selectBodyNode(const Constraint& c);
  
  //! returns the first uncolored node of g that satisfies c
  virtual HeadNode* selectHeadNode(const Constraint& c);
private:
  Graph* graph_;
};

typedef SelectFirst NoHeuristic;


/*!
 * Binary-Predicate that extracts the heuristic values of two given nodes and then compares
 * the heuristic values using the stored Compare-Predicate.
 *
 * \par template parameters \n
 * - V Type of the heuristic value installed in the nodes
 * - C Compare-Predicate that can compare two values of type V
 * .
 */
template <class V, class C>
struct HVCompare {
  HVCompare(const C& c)
    : cmp_(c) {
  }
  /*!
   * \pre n1 and n2 either store values of type V or no value.
   * \note the precondition is not checked. Comparing nodes with contained
   * values that are not of type V leads to undefined behaviour.
   */
  bool operator()(const Node* n1, const Node* n2) const {
    VT* v1 = static_cast<VT*>(n1->getValue());
    VT* v2 = static_cast<VT*>(n2->getValue());
    if (v1 && v2)
      return cmp_(v1->value(), v2->value());
    return v1 != 0;
  }
private:
  typedef ValueHolder<V> VT;
  C cmp_;
};

}

#endif
