/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         * 
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         * 
 ***************************************************************************/

#ifndef LIBNOMORE_NODE_H
#define LIBNOMORE_NODE_H

#if defined (_MSC_VER) && _MSC_VER <= 1300
#pragma warning (disable : 4786)
#endif


#include <color.h>
#include <colorerror.h>
#include <string>

namespace event {
	class ChannelManager;
}
namespace NS_NOMORE {

class Graph;
class ValueBase;

//! Base class for all nodes to be stored in nomore's body-head graph.
/*! This base node holds common information (like identifier and node color)
 * 	needed by all subclasses e.g. HeadNode or BodyNode. 
 * \see Graph
 */
class Node {
  friend class Graph;

public:
  //! Creates a new node with the given id.
  /* \param id The id for the new node.
   */
  explicit Node(long id);

  //! Destroys a node.
  virtual ~Node();

  //! Returns the current color value of the node.
  Color::ColorValue getColor() const {
    return color_;
  }

  //! Returns the unique identifier of the node.
  long getId() const {
    return id_;
  }

  //! associates a generic value with the node.
  /*! 
   * This method can be used for example to associate a heuristic value
   * of arbitrary type with a node as long as readers of that value
   * know the concrete type of it.
   */
  void installValue(ValueBase* v);

  //! returns the generic value associated with this node.
  /*!
   * callers must know the concrete type of the stored value in order 
   * to use the returned value in a meaningful way.
   */
  ValueBase* getValue() const {
    return value_;
  }

  //! Returns the color which preserves the recursive support used for the U operator.
  /*! The method returns the value of the supportedness of the node coded in the color
   *  weak_plus and plus.
   */
  virtual Color::ColorValue recursiveSupportColor() const = 0;

  //! Returns the preferred choice point color of this node.
  virtual Color::ColorValue preferredChoicePointColor() const = 0;

  //! Returns the alternative choice point color for this node.
  virtual Color::ColorValue alternativeChoicePointColor() const = 0;

protected:
  //! Colors the node.
  /*! Sets the node's color to c and calls doSetColor(oldColor) if the transition
   *  from the current color to c is valid and c differs from the current color.
   *  \note Refer to \ref NS_NOMORE::isValidColoring "isValidColoring" for information
   *  about valid colorings.
   *  \param c The new color for the node.
   *  \pre c is not Color::none
   *  \return True if the color was changed, false otherwise.
   *  \exception ColorError Throws this exception if a color error occured because of 
   *             an invalid coloring. */
  bool setColor(Color::ColorValue c);

  //! Restores the node's color to c.
  /*! Sets the color of the node to c and calls doRestoreColor(oldColor).
   *  \param c The new color of the node.
   *  \pre c is either Color::ignore, Color::none or Color::weak_plus. */
   void restoreColor(Color::ColorValue c);

  //! Called by setColor if the color was changed.
  /*! Subclasses can implement necessary update-operations here.
   * \param oldColor The old color before coloring this node. */
  virtual void doSetColor(Color::ColorValue oldColor) = 0;

  //! Called by restoreColor.
	/*! Subclasses can implement necessary update-operations here.
   * \param oldColor The old color before restoring this node. */
  virtual void doRestoreColor(Color::ColorValue oldColor) = 0;

  //! Should fire a node-specific color event.
  /*! \ref _event "More information on events." */
  virtual void fireColorEvent(event::ChannelManager& c, Color::ColorValue oldColor) = 0;

  //! Should fire a node-specific color choice point event.
  /*! \ref _event "More information on events." */
  virtual void fireColorChoicePointEvent(event::ChannelManager& c, Color::ColorValue oldColor, bool recolored) = 0;

  //! Should fire a node-specific restore event.
  /*! \ref _event "More information to events." */
  virtual void fireRestoreEvent(event::ChannelManager& c, Color::ColorValue oldColor) = 0;

private:
  //! Copying of nodes is not supported
  Node(const Node& n);
  Node& operator=(const Node& n);

  //! The color of the node. Initialized to Color::none.
  Color::ColorValue color_;

  //! The unique identifier of the node.
  long id_;
  
  ValueBase* value_;
};

//! id-based comparison of two nodes
inline bool operator==(const Node& lhs, const Node& rhs) {
  return lhs.getId() == rhs.getId();
}

//! Checks if a transition from oldColor to newColor is valid.
/*!
 * Valid colorings are:
 * - none -> X
 * - ignore	-> X
 * - weak_plus	-> plus
 * .
 * where X is any color value except Color::unknown and Color::none.
 **/
inline bool isValidColoring(Color::ColorValue oldColor, Color::ColorValue newColor)
{
	return newColor != Color::none
		&& (oldColor == Color::none || oldColor == Color::ignore
		|| (oldColor == Color::weak_plus && newColor == Color::plus));
}

//! Struct for checking the color of a node against a color mask.
struct HasColor {
	/*!
   * \param colorMask An ORed enumeration of color values
   * \see Color::ColorValue
   */
  explicit HasColor(int colorMask)
		: colorMask_(colorMask) {
	}

  //! returns true if n has a color that was specified in the color mask.
	bool operator()(Node* n) const {
		return (n->getColor() & colorMask_) != 0;
	}

private:
	int colorMask_;
};

} // NS_NOMORE

#endif
