/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         * 
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         * 
 ***************************************************************************/

#ifndef LIBNOMORE_OPERATOR_STRING_PARSER_H
#define LIBNOMORE_OPERATOR_STRING_PARSER_H
#include <string>
#include <stdexcept>
namespace NS_NOMORE {

//! A parser that parses operator name strings.
/*!
 * The parser recognizes the following grammar:
 * opStr        :- OP | seq
 * seq          :- (seqList) opt *
 * seqList      :- seq, seqOptList | optList
 * seqOptList   :- seq | seq, seqOptList | optList
 * optList      :- OP | OP, optList | OP, seq
 *
 * Examples for valid strings:
 * P                  
 * (P)*               
 * (P, B)             
 * ((P, B)*, V)       
 * (P, B, V, (I, J)*)
 *
 * Examples for invalid strings:
 * P, B       -> missing parentheses, correct: (P, B)
 * P*         -> missing parentheses, correct: (P)*
 * (P, B*)    -> missing parentheses, correct: (P, (B)*)
 * ((P)*)     -> missing sequence,    correct: (P)*
 *
 */
class OperatorStringParser {
public:
  //! Exception type thrown to signal a syntax error in the input
  class SyntaxError : public std::runtime_error {
  public:
    SyntaxError(const std::string& msg);
  };
  
  OperatorStringParser();
  virtual ~OperatorStringParser();
  
  //! parses the string str
  /*!
   * \return true if str is valid
   * \throw SyntaxError if str is not valid
   */
  bool parse(const std::string& str);
private:
  enum Token {T_EOF = '\0', OP, LPARAN = '(', RPARAN = ')', CLOSURE = '*', COMMA = ','};
  Token currentToken_;
  std::string tokenVal_;
  const char* p_;

///////////////////////////////////////////////////////////////////////////////
// action hooks for derived classes
///////////////////////////////////////////////////////////////////////////////
  //! semantic action to be performed whenever a sequence is found
  virtual void startSequence() = 0;
  //! semantic action to be performed whenever the parser finished parsing a sequence
  virtual void endSequence(bool closure) = 0;
  //! seamntic action to be performed whenever an operator name is found
  virtual void addOperator(const std::string& name) = 0;
///////////////////////////////////////////////////////////////////////////////

  void opStr();
  void seq();
  void seqList();
  void seqOptList();
  void optList();

  void match(Token t);
  Token nextToken();

  void error(const char*);
  void error(Token t);
};

}

#endif
