/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         * 
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         * 
 ***************************************************************************/

#ifndef NOMORE_BODY_LOOKAHEAD_OPERATOR_H
#define NOMORE_BODY_LOOKAHEAD_OPERATOR_H

#include <operators/lookahead.h>
#include <map>

namespace NS_NOMORE {

class BodyNode;
struct HeadNodeColored;
struct HeadNodeColoredChoicePoint;
struct BodyNodeColored;

//! Lookahead operator that works on uncolored body nodes only.
/*! 
 *  Every uncolored body node is checked with Color::plus and Color::minus. 
 *  If assigning one color results in a conflict the node is colored to
 *  the opposite color.
 * 
 *  Additionally this operator produces information that can be used as
 *  heuristic value. For each node that can be colored either Color::plus
 *  or Color::minus this operator fires an event of type BodyLookahead::EventType.
 *  The value-member of this event type contains counters that store the number
 *  of head nodes colored because of the coloring of the body node.
 *  
 *  The algorithm looks like the following code fragment:
 *  \code
 *    check-queue contains all uncolored body nodes;
 *    while(check-queue not empty) {
 *      node := check-queue.pop();
 *      if (propagation with node colored to + leads to conflict) {
 *          assign(node, -);
 *      }
 *      else if (propagation with node colored to - leads to conflict) {
 *          assign(node, +);
 *      }
 *      else { 
 *        store heuristic value for node   
 *      }
 *    }
 *  \endcode
 *
 *  Technically the operator does not check every uncolored body node but
 *  only those nodes that don't get a color as a result of propagating other nodes. 
 *  That is if assigning (and propagating) color c1 to node n1 leads to the coloring
 *  of n2 to the color c1 and n1 is supported then n2 is removed from the
 *  lookahead-set. The restriction that n1 must be supported is needed in order
 *  to get better heuristic-values for supported-driven choice-operators.
 *
 */
class BodyLookahead : public Lookahead {
public:
  //! type that stores information for one lookahead-step.
  struct ResultType {
    ResultType(long v1 = 0, long v2 = 0) {
      if (v1 > v2) {
        max_ = v1;
        min_ = v2;
      }
      else {
        max_ = v2;
        min_ = v1;
      }
    }
    bool operator>(const ResultType& other) const {
      return max_ > other.max_ || (max_ == other.max_ && min_ > other.min_);
    }
    long max_;
    long min_;
  };
  
  /*!
   * event type fired whenever an uncolored body node can be either colored
   * to Color::plus or to Color::minus, i.e. the lookahead-operator can't
   * derivate a conflict.
   *
   * Operator-Based heuristics may listen to this event in order to
   * select choices based on lookahead-results.
   */
  struct EventType {
    /*!
     * \param n the node for which the lookahead-operator could not derive a conflict.
     * \param r the lookahead-information for n
     */
    EventType(Node* n, const ResultType& r)
      : node_(n)
      , value_(r) {
    }
    Node* node_;
    ResultType value_;
  };

  //! Creates the operator.
  /*! 
   * \param grp The graph the operator should work on. 
   * \param op Operator to be used for propagation
   */ 
  BodyLookahead(Graph& grp, Operator& op);

  explicit BodyLookahead(Graph& grp);

  //! event handler for HeadNodeColored events.
  void handle(const HeadNodeColored& e);

  //! event handler for HeadNodeColoredChoicePoint events
  void handle(const HeadNodeColoredChoicePoint& e);

  //! event handler for BodyNodeColored events.
  void handle(const BodyNodeColored& e);

  //! Returns the name of the operator.
  static std::string getOpName() {
    return "BLA";
  }

protected:
  //! Checks all uncolored body nodes.
  bool checkBodyNodes();
    
  //! Resets the lookahead counter.
  void resetCounter();
  
  //! fires an event of type BodyLookahead::EventType
  void storeHeuristic();

  //! Checks the current lookahead node by calling Lookahead::checkColor.
  /*! 
   * If the color check fails (a conflict is found) the node is colored to
   * the opposite color.
   */
  bool checkCurrentNode();

private:  
  typedef std::map<long, bool> LookaheadInfoMap;
  typedef LookaheadInfoMap::iterator LookaheadInfoMapIterator;

  //! implements the body-lookahead
  virtual bool execute();

  /*! \see Lookahead::postExecute */
  virtual void postExecute();    

  void UpdateLookaheadInfoMap(LookaheadInfoMap &infoMap, long nodeId);

  //! The heuristics counter. (Only heads that have one-successors)
  long counterPlus_;
  long counterMinus_;

  LookaheadInfoMap lookaheadPlusInfos_;
  LookaheadInfoMap lookaheadMinusInfos_;

  Node* lookaheadNode_;
  bool isLookaheadNodeSupported_;
  Color::ColorValue lookaheadColorCheck_;
  Color::ColorValue alternativeColor_;
};

}

#endif
