/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         * 
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         * 
 ***************************************************************************/

#if defined (_MSC_VER) && _MSC_VER <= 1300
#pragma warning (disable : 4786)
#endif

#include <operators/hybrid_lookahead.h>
#include <body_node.h>
#include <colorerror.h>
#include <head_node.h>
#include <graph.h>
#include <iostream>

namespace NS_NOMORE {

HybridLookahead::HybridLookahead(Graph& grp, Operator& op) 
  : Lookahead(HybridLookahead::getOpName(), grp, op), counter_(0), lookaheadNode_(0), isLookaheadNodeSupported_(false) {
}

HybridLookahead::HybridLookahead(Graph& grp) 
  : Lookahead(HybridLookahead::getOpName(), grp)
  , counter_(0)
  , lookaheadNode_(0)
  , isLookaheadNodeSupported_(false) {
}

bool HybridLookahead::execute() {
  bool result = false;
  
  Lookahead::execute();
  event::ChannelManager &em = getGraph().getEventManager();
  em.getChannelFor(event::Event<HeadNodeColored>()).connect(*this);
  em.getChannelFor(event::Event<HeadNodeColoredChoicePoint>()).connect(*this);
  em.getChannelFor(event::Event<BodyNodeColored>()).connect(*this);

  while(checkBodyNodes() || checkHeadNodes())
    result = true;
    
  return result;
}

bool HybridLookahead::checkBodyNodes() {
  Graph::UncoloredBodyNodeIterator end = getGraph().uncoloredBodyNodesEnd();
  Graph::UncoloredBodyNodeIterator it = getGraph().uncoloredBodyNodesBegin();
  for(; it != end; ++it) {
    lookaheadNode_ = *it;
    assert(lookaheadNode_->getColor() == Color::none);
    isLookaheadNodeSupported_ = (*it)->isSupported();   
    if(checkCurrentNode()) {
        return true;
    }
  }
  return false;
}

bool HybridLookahead::checkHeadNodes() {
  Graph::UncoloredHeadNodeIterator end = getGraph().uncoloredHeadNodesEnd();
  Graph::UncoloredHeadNodeIterator it = getGraph().uncoloredHeadNodesBegin();

  for(; it != end; ++it) {
    lookaheadNode_ = *it;
    assert(lookaheadNode_->getColor() == Color::none);
    isLookaheadNodeSupported_ = false;
    if(checkCurrentNode()) {
      return true;
    }
  }
  return false;
}

void HybridLookahead::handle(const HeadNodeColored& e) {
  if(e.node_->oneSuccessorsBegin() != e.node_->oneSuccessorsEnd())
    ++counter_;

  if(e.node_->getColor() == Color::minus) {
    lookaheadInfos_[e.node_->getId()] = true;
  }
}

void HybridLookahead::handle(const HeadNodeColoredChoicePoint& e) {
  if(e.node_->oneSuccessorsBegin() != e.node_->oneSuccessorsEnd())
    ++counter_;
}

void HybridLookahead::handle(const BodyNodeColored& e) {
  if(e.node_->getColor() == Color::plus || e.node_->getColor() == Color::weak_plus) {
    LookaheadInfoMapIterator it = lookaheadInfos_.find(e.node_->getId());
    if(it != lookaheadInfos_.end() && !it->second) {
      it->second = isLookaheadNodeSupported_;
    } else {
      lookaheadInfos_[e.node_->getId()] = isLookaheadNodeSupported_;
    }
  }
}

void HybridLookahead::resetCounter() {
  counter_ = 0;
}

void HybridLookahead::storeHeuristic() {
  event::ChannelManager &em = getGraph().getEventManager();
  em.dispatch(EventType(lookaheadNode_, counter_));
}

void HybridLookahead::postExecute() {
  event::ChannelManager &em = getGraph().getEventManager();
  em.getChannelFor(event::Event<HeadNodeColored>()).disconnect(*this);
  em.getChannelFor(event::Event<HeadNodeColoredChoicePoint>()).disconnect(*this);  
  em.getChannelFor(event::Event<BodyNodeColored>()).disconnect(*this);
  lookaheadInfos_.clear();

  Lookahead::postExecute();
}

bool HybridLookahead::checkCurrentNode() {
  LookaheadInfoMapIterator it = lookaheadInfos_.find(lookaheadNode_->getId());
  if(it != lookaheadInfos_.end() && (it->second || !isLookaheadNodeSupported_))
    return false;
    
  try {
    resetCounter();
    checkColor(*lookaheadNode_, lookaheadNode_->preferredChoicePointColor());
    storeHeuristic();    
  } catch(const ColorError&) {
    // conflict found
    color(*lookaheadNode_, lookaheadNode_->alternativeChoicePointColor());    
    // reset the info map (all nodes must be re-checked)
    lookaheadInfos_.clear();
    // conflict, but other way work
    return true;
  }  

  // no conflict
  return false;
}


}
