#if defined (_MSC_VER) && _MSC_VER <= 1200
#define for if(0);else for
#endif
#include <cppunit/TestFixture.h>
#include <cppunit/TestAssert.h>
#include <cppunit/extensions/HelperMacros.h>
#include <string>
#include <sstream>
#include <event/channel.h>
using namespace event;
namespace {
	
	struct FakeEvent 
	{
		FakeEvent(int nr)
			: nr_(nr)
		{}
		int nr_;
	};
	std::stringstream globalEvents_g;

	struct HandlerThatConnectsOnEvent
	{
		HandlerThatConnectsOnEvent(Channel<int>* c)
			: c_(c)
			, events_(0)
		{}
		void handle(const int&)
		{
			++events_;
			c_->connect(*this);
		}
		Channel<int>* c_;
		int events_;
	};
	
	struct HandlerThatDisconnectsOnEvent
	{
		HandlerThatDisconnectsOnEvent(Channel<int>* c)
			: c_(c)
			, events_(0)
		{}
		void handle(const int&)
		{
			++events_;
			c_->disconnect(*this);
		}
		Channel<int>* c_;
		int events_;
	};
	
	struct Handler 
	{
		Handler()
			: events_(0)
		{}
		void handle(const int&)
		{
			++events_;
		}
		void handle(const FakeEvent& e)
		{
			++events_;
			globalEvents_g << e.nr_ << ";";
		}
		int events_;
	};

	struct HandlerThatPostsAnEvent
	{
		HandlerThatPostsAnEvent(Channel<FakeEvent>* c)
			: channel_(c)
			, firstTime_(true)
		{}
		void handle(const FakeEvent& e)
		{
			if (firstTime_)
			{
				globalEvents_g << e.nr_ << ";";
				firstTime_ = false;
				channel_->dispatch(FakeEvent(e.nr_ + 1));
			}
			else
				globalEvents_g << e.nr_ << ";";
		}
		Channel<FakeEvent>* channel_;
		bool firstTime_;
	};
}
class TestChannel : public CppUnit::TestFixture
{
public:
	void setUp()
	{
		channel_ = new Channel<int>();
		globalEvents_g.str("");
	}
	void tearDown()
	{
		delete channel_;
	}
	void testConnectOne()
	{
		Handler h;
		channel_->connect(h);
		CPPUNIT_ASSERT_EQUAL(1u, channel_->numHandlers());
	}
	void testConnectSeveral()
	{
		Handler h;
		channel_->connect(h);
		channel_->connect(h);
		channel_->connect(h);
		CPPUNIT_ASSERT_EQUAL(3u, channel_->numHandlers());
	}
	
	void testDisconnectOne()
	{
		Handler h;
		channel_->connect(h);
		channel_->disconnect(h);
		CPPUNIT_ASSERT_EQUAL(false, channel_->hasHandlers());
	}
	
	void testDisconnectSeveral()
	{
		Handler h1, h2, h3;
		channel_->connect(h1);
		channel_->connect(h2);
		channel_->connect(h3);
		channel_->disconnect(h1);
		channel_->disconnect(h2);
		channel_->disconnect(h3);
		CPPUNIT_ASSERT_EQUAL(false, channel_->hasHandlers());
	}
	
	void testDispatchOneReceiver()
	{
		Handler h;
		channel_->connect(h);
		channel_->dispatch(1);
		CPPUNIT_ASSERT_EQUAL(1, h.events_);
		channel_->dispatch(1);
		CPPUNIT_ASSERT_EQUAL(2, h.events_);
	}
	void testDispatchMultipleReceiver()
	{
		Handler h1, h2, h3;
		channel_->connect(h1);
		channel_->connect(h2);
		channel_->connect(h3);
		channel_->dispatch(1);
		CPPUNIT_ASSERT_EQUAL(3, h1.events_ + h2.events_ + h3.events_);
	}

	void testLifoOrder()
	{
		Channel<FakeEvent> channel;
		HandlerThatPostsAnEvent h1(&channel);
		Handler h2;
		channel.connect(h1);
		channel.connect(h2);
		channel.dispatch(FakeEvent(1));
		CPPUNIT_ASSERT_EQUAL(
			std::string("1;2;2;1;"), globalEvents_g.str()
		);
	}

	void testFifoOrder()
	{
		Channel<FakeEvent> channel;
		channel.setEventDispatchingState(EventDispatchingState::FIFO);
		HandlerThatPostsAnEvent h1(&channel);
		Handler h2;
		channel.connect(h1);
		channel.connect(h2);
		channel.dispatch(FakeEvent(1));
		CPPUNIT_ASSERT_EQUAL(
			std::string("1;1;2;2;"), globalEvents_g.str()
		);
	}

	void testHolding()
	{
		channel_->setEventDispatchingState(EventDispatchingState::HOLD);
		Handler h1, h2;
		channel_->connect(h1);
		channel_->connect(h2);
		channel_->dispatch(1);
		CPPUNIT_ASSERT_EQUAL(0, h1.events_ + h2.events_);
		channel_->setEventDispatchingState(EventDispatchingState::LIFO);
		CPPUNIT_ASSERT_EQUAL(2, h1.events_ + h2.events_);
	}

	void testDiscarding()
	{
		channel_->setEventDispatchingState(EventDispatchingState::DISCARD);
		Handler h1, h2;
		channel_->connect(h1);
		channel_->connect(h2);
		channel_->dispatch(1);
		CPPUNIT_ASSERT_EQUAL(0, h1.events_ + h2.events_);
		channel_->setEventDispatchingState(EventDispatchingState::LIFO);
		CPPUNIT_ASSERT_EQUAL(0, h1.events_ + h2.events_);
	}

	void testConnectWhileActive()
	{
		HandlerThatConnectsOnEvent h(channel_);
		Handler h2;
		channel_->connect(h);
		channel_->connect(h2);
		channel_->dispatch(1);
		CPPUNIT_ASSERT_EQUAL(1, h.events_);
		CPPUNIT_ASSERT_EQUAL(1, h2.events_);
		CPPUNIT_ASSERT_EQUAL(3u, channel_->numHandlers());
		
		h.events_ = 0;
		channel_->dispatch(1);
		CPPUNIT_ASSERT_EQUAL(2, h.events_);
	}
	
	void testDisconnectWhileActive()
	{
		HandlerThatDisconnectsOnEvent h(channel_);
		channel_->connect(h);
		channel_->connect(h);
		channel_->connect(h);
		channel_->dispatch(1);
		CPPUNIT_ASSERT_EQUAL(1, h.events_);
		h.events_ = 0;
		channel_->dispatch(1);
		CPPUNIT_ASSERT_EQUAL(0, h.events_);
	}
	
	CPPUNIT_TEST_SUITE( TestChannel );
		CPPUNIT_TEST(testConnectOne);
		CPPUNIT_TEST(testConnectSeveral);
		CPPUNIT_TEST(testDisconnectOne);
		CPPUNIT_TEST(testDisconnectSeveral);
		CPPUNIT_TEST(testDispatchOneReceiver);
		CPPUNIT_TEST(testDispatchMultipleReceiver);
		CPPUNIT_TEST(testLifoOrder);
		CPPUNIT_TEST(testFifoOrder);
		CPPUNIT_TEST(testHolding);
		CPPUNIT_TEST(testDiscarding);
		CPPUNIT_TEST(testConnectWhileActive);
		CPPUNIT_TEST(testDisconnectWhileActive);
	CPPUNIT_TEST_SUITE_END();
private:
	Channel<int>* channel_;
};

CPPUNIT_TEST_SUITE_REGISTRATION( TestChannel );



