#include <cppunit/TestFixture.h>
#include <cppunit/TestAssert.h>
#include <cppunit/extensions/HelperMacros.h>
#include <operators/hybrid_lookahead.h>
#include <operators/forward_prop.h>
#include <graph.h>
#include <head_node.h>
#include <body_node.h>
#include <generic_value.h>

using namespace NS_NOMORE;

namespace NS_NOMORE_TESTS {

class HybridLookaheadProp : public CppUnit::TestFixture {

  CPPUNIT_TEST_SUITE(HybridLookaheadProp);
  CPPUNIT_TEST(testSuccess);
  CPPUNIT_TEST(testException);
  CPPUNIT_TEST(testFailure1);
  CPPUNIT_TEST(testFailure2);
  CPPUNIT_TEST_SUITE_END();

public:
  void handle(const HybridLookahead::EventType& e) {
    e.node_->installValue(new ValueHolder<HybridLookahead::ResultType>(e.value_));
  }
  void setUp() {
    grp = new Graph();

    body1 = grp->insertBodyNode();
    body2 = grp->insertBodyNode();
    body3 = grp->insertBodyNode();
    head1 = grp->insertHeadNode(1);
    head2 = grp->insertHeadNode(2);

    det = new ForwardPropagator(*grp);
    lookahead = new HybridLookahead(*grp, *det);
    grp->getEventManager().getChannelFor(event::Event<HybridLookahead::EventType>()).connect(*this);
  }

  void tearDown() {
    delete lookahead;
    delete det;
    delete grp;
  }

  void testSuccess() {
    body1->insertSuccessor(*head1);
    body1->insertSuccessor(*head2);
    head2->insertZeroSuccessor(*body2);
    head2->insertOneSuccessor(*body3);    
    
    (*lookahead)();

    CPPUNIT_ASSERT_EQUAL( true, body1->getValue() != 0 );
    CPPUNIT_ASSERT_EQUAL( false, body2->getValue() != 0 );
    CPPUNIT_ASSERT_EQUAL( true, body3->getValue() != 0 );
    CPPUNIT_ASSERT_EQUAL( true, head1->getValue() != 0 );
    CPPUNIT_ASSERT_EQUAL( true, head2->getValue() != 0 );
    CPPUNIT_ASSERT_EQUAL( 1l, value_cast<HybridLookahead::ResultType>(body1->getValue()));
    CPPUNIT_ASSERT_EQUAL( 0l, value_cast<HybridLookahead::ResultType>(body3->getValue()));
    CPPUNIT_ASSERT_EQUAL( 0l, value_cast<HybridLookahead::ResultType>(head1->getValue()));
    CPPUNIT_ASSERT_EQUAL( 1l, value_cast<HybridLookahead::ResultType>(head2->getValue()));

    CPPUNIT_ASSERT_EQUAL( Color::none, body1->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::none, body2->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::none, body3->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::none, head1->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::none, head2->getColor() );
  }

  void testException() {
    HeadNode *head3 = grp->insertHeadNode(3);
  
    body1->insertSuccessor(*head1);
    body1->insertSuccessor(*head2);
    head2->insertZeroSuccessor(*body2);
    head2->insertOneSuccessor(*body3);    
    body3->insertSuccessor(*head3);
    head3->insertZeroSuccessor(*body1);
    
    CPPUNIT_ASSERT_THROW( (*lookahead)(), ColorError );
  }

  void testFailure1() {    
    body1->insertSuccessor(*head1);
    head1->insertZeroSuccessor(*body2);
    body2->insertSuccessor(*head2);
    head2->insertOneSuccessor(*body2);
    
    grp->color(*head2, Color::minus);
    grp->color(*body2, Color::minus);
    
    (*lookahead)();

    CPPUNIT_ASSERT_EQUAL( Color::minus, body1->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::minus, body2->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::minus, head1->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::minus, head2->getColor() );
  }

  void testFailure2() {    
    body1->insertSuccessor(*head1);
    head1->insertOneSuccessor(*body2);
    body2->insertSuccessor(*head2);
    head2->insertOneSuccessor(*body2);
    
    grp->color(*head2, Color::minus);
    grp->color(*body2, Color::minus);
    grp->color(*body1, Color::ignore);
    
    (*lookahead)();

    CPPUNIT_ASSERT_EQUAL( Color::ignore, body1->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::minus, body2->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::weak_plus, head1->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::minus, head2->getColor() );
  }
  
private:
  Graph *grp;

  Operator *det;
  HybridLookahead *lookahead;

  BodyNode *body1;
  BodyNode *body2;
  BodyNode *body3;
  HeadNode *head1;
  HeadNode *head2;
};

CPPUNIT_TEST_SUITE_REGISTRATION(HybridLookaheadProp);

} // end namespace NS_NOMORE_TESTS
