#include <cppunit/TestFixture.h>
#include <cppunit/TestAssert.h>
#include <cppunit/extensions/HelperMacros.h>
#include <object_factory.h>
#include <graph.h>
#include <heuristic.h>
#include <operator.h>
#include <choice_operator.h>
#include <operators/lookahead.h>
#include <operator_decorator_factory.h>
#include <operators/sequence.h>
#include <operators/forward_prop.h>
#include <operators/backward_prop.h>
#include <operators/unfounded_set.h>
#include <typeinfo>
using namespace NS_NOMORE;

namespace NS_NOMORE_TESTS {

class TestObjectFactory : public CppUnit::TestFixture {

  CPPUNIT_TEST_SUITE(TestObjectFactory);
  CPPUNIT_TEST(testCreateOperator);
  CPPUNIT_TEST(testCreateChoiceOperator);
  CPPUNIT_TEST(testcreateOperators);
  CPPUNIT_TEST(testcreateOperatorsMissingComponent);
  CPPUNIT_TEST(testcreateOperatorsToManyComponents);
  CPPUNIT_TEST(testInvalidCombinations);
  CPPUNIT_TEST(testDecoration);
  CPPUNIT_TEST(testCreateWithLookahead);
  CPPUNIT_TEST(testInvalidLookaheadStr);
  CPPUNIT_TEST_SUITE_END();

public:
  void setUp() {
    factory_ = new ObjectFactory();
  }
  void tearDown() {
    delete factory_;
  }
  void testCreateOperator() {
    Graph g;
    std::auto_ptr<Operator> o1(factory_->createOperator(g, ForwardPropagator::getOpName()));
    CPPUNIT_ASSERT(o1->getName() == ForwardPropagator::getOpName());

    std::auto_ptr<Operator> o2(factory_->createOperator(g, BackwardPropagator::getOpName()));
    CPPUNIT_ASSERT(o2->getName() == BackwardPropagator::getOpName());

    std::auto_ptr<Operator> o3(factory_->createOperator(g, UnfoundedSetOperator::getOpName()));
    CPPUNIT_ASSERT(o3->getName() == UnfoundedSetOperator::getOpName());

    CPPUNIT_ASSERT_THROW(factory_->createOperator(g, "Foo"), UnknownObject);
  }
  void testCreateChoiceOperator() {
    Graph g;
    std::auto_ptr<Operator> o1(factory_->createChoiceOperator(g, "C", "LexOrder"));
    CPPUNIT_ASSERT(o1->getName() == "C");

    std::auto_ptr<Operator> o2(factory_->createChoiceOperator(g,"D", "HybridLA"));
    CPPUNIT_ASSERT(o2->getName() == "D");

    CPPUNIT_ASSERT_THROW(factory_->createChoiceOperator(g, "D", "Foo"), UnknownObject);
    CPPUNIT_ASSERT_THROW(factory_->createChoiceOperator(g, "Foo", "HybridLA"), UnknownObject);
  }

  void testcreateOperators() {
    Graph g;
    ObjectFactory::Operators o = factory_->createOperators(g, "D:((P, B)*,U)*:None:HybridLA");
    CPPUNIT_ASSERT(o.choiceOp_->getName() == "D");
    CPPUNIT_ASSERT_EQUAL(std::string("((P,B)*,U)*"), o.detOp_->getName());
  }
  
  void testcreateOperatorsMissingComponent() {
    Graph g;
    CPPUNIT_ASSERT_THROW(factory_->createOperators(g, "P:None:HybridLA"), FormatError);
    CPPUNIT_ASSERT_THROW(factory_->createOperators(g, ":P:None:HybridLA"), FormatError);
    CPPUNIT_ASSERT_THROW(factory_->createOperators(g, "D::None:HybridLA"), FormatError);
    CPPUNIT_ASSERT_THROW(factory_->createOperators(g, "D:None:HybridLA"), FormatError);
    CPPUNIT_ASSERT_THROW(factory_->createOperators(g, "D:P:HybridLA"), FormatError);
  }

  void testcreateOperatorsToManyComponents() {
    Graph g;
    CPPUNIT_ASSERT_THROW(factory_->createOperators(g, "D:P:N:None:HybridLA:Foo"), FormatError);
  }

  void testInvalidCombinations() {
    Graph g;  
    /*
    CPPUNIT_ASSERT_THROW(factory_->createOperators(g, "D:P:None:HybridLA"), InvalidOperatorCombination);

    
    CPPUNIT_ASSERT_THROW(factory_->createOperators(g, "D:U:None:HybridLA"), InvalidOperatorCombination);
    */
  }

  void testDecoration() {
    Graph g;
    ObjectFactory of(new FakeDecoratorFactory);
    std::auto_ptr<Operator> o1(of.createOperator(g, "P"));
    CPPUNIT_ASSERT(typeid(FakeDecoratorFactory::NullDecorator) == typeid(*o1));

    std::auto_ptr<Operator> o2(of.createChoiceOperator(g, "D", "None"));
    CPPUNIT_ASSERT(typeid(FakeDecoratorFactory::NullDecorator) == typeid(*o2));

    ObjectFactory::Operators o = of.createOperators(g, "D:(P, B):None:None");
    
    CPPUNIT_ASSERT(typeid(Sequence) == typeid(*o.detOp_));
    Operator* s1 = o.detOp_->extractOperator("P");
    CPPUNIT_ASSERT(typeid(FakeDecoratorFactory::NullDecorator) == typeid(*s1));
    
    CPPUNIT_ASSERT(typeid(FakeDecoratorFactory::NullDecorator) == typeid(*o.choiceOp_));
  }
  void testCreateWithLookahead() {
    Graph g;
    ObjectFactory::Operators o = factory_->createOperators(g, "D:P:HLA[(P,B)]:None");
    Lookahead* hla = dynamic_cast<Lookahead*>(o.detOp_->extractOperator("HLA"));
    CPPUNIT_ASSERT(hla != 0);
    Operator* pInHla = hla->getPropagationOperator().extractOperator("P");
    Operator* pInDet = o.detOp_->extractOperator("P");
    CPPUNIT_ASSERT_EQUAL(pInHla, pInDet);
  }

  void testInvalidLookaheadStr() {
    Graph g;
    CPPUNIT_ASSERT_THROW(factory_->createOperators(g, "D:P:HLA(P,B)]:None"), FormatError);
    CPPUNIT_ASSERT_THROW(factory_->createOperators(g, "D:P:HLA[(P,B):None"), FormatError);
    CPPUNIT_ASSERT_THROW(factory_->createOperators(g, "D:P:HLA(P,B):None"), UnknownObject);
    CPPUNIT_ASSERT_THROW(factory_->createOperators(g, "D:P:Foo[(P,B)]:None"), UnknownObject);
  }
private:
  class FakeDecoratorFactory : public OperatorDecoratorFactory {
  public:
    virtual Operator* decorate(Operator* op) {return new NullDecorator(op);}
    struct NullDecorator : public Operator {
    public:
      NullDecorator(Operator* o)
        : Operator(o->getName())
        , op_(o) {
      }
      ~NullDecorator() {
        delete op_;
      }
      bool execute() {
        return (*op_)();
      }
    private:
      Operator* op_;
    };
  };
  ObjectFactory* factory_;
};

CPPUNIT_TEST_SUITE_REGISTRATION(TestObjectFactory);

}	// end namespace NS_NOMORE_TESTS
