#include <cppunit/TestFixture.h>
#include <cppunit/TestAssert.h>
#include <cppunit/extensions/HelperMacros.h>
#include <operator_string_parser.h>
#include <stack>
using namespace NS_NOMORE;

namespace NS_NOMORE_TESTS {

class TestOperatorStringParser : public CppUnit::TestFixture {

  CPPUNIT_TEST_SUITE(TestOperatorStringParser);
  CPPUNIT_TEST(testSimple);
  CPPUNIT_TEST(testContainingWhite);
  CPPUNIT_TEST(testValidCases);
  CPPUNIT_TEST(testInvalidCases);
  CPPUNIT_TEST_SUITE_END();

public:
  void testSimple() {
    FakeParser p;
    CPPUNIT_ASSERT_EQUAL(true, p.parse("A"));
    CPPUNIT_ASSERT_EQUAL(std::string("A"), p.result_);
  }
  void testContainingWhite() {
    FakeParser p;
    CPPUNIT_ASSERT_EQUAL(true, p.parse("((A,   B)  *, C) *"));
    CPPUNIT_ASSERT_EQUAL(std::string("((A,B)*,C)*"), p.result_);
  }

  void testValidCases() {
    const int nrCases = 22;
    std::string cases[nrCases] = {
      "a", "(a)", "(a,b)", "(a,b,c)", "(a,b,c,d)",
      "(a)*", "(a,b)*", "(a,b,c)*", "(a,b,c,d)*", 
      "(a,(A))", "(a,(A,b)*)", "(a,(A,b,c)*)", "((A),a)",
      "((A,b)*,a)", "((A,b,c)*,a)", "((A),(B))", "((A,b)*,(A,b))*",
      "((A,b,c)*,(A,b,c))", "((A),(B),(C))", "((A,b)*,(A,b),(B,c))*",
      "(((A,B),C,(D,C))*,D)*", "(A,(((A,B),C)*,(D,E)))*"
    };
    for (int i = 0; i != nrCases; ++i) {
      FakeParser p;
      CPPUNIT_ASSERT_EQUAL(true, p.parse(cases[i]));
      CPPUNIT_ASSERT_EQUAL(cases[i], p.result_);
    }
  }

  void testInvalidCases() {
    const int nrCases = 12;
    std::string cases[nrCases] = {
      "a,b", "((a))", "(a,b", "a,b,c)", "(a,b,c,d*",
      "(a),c", "(a,b),c*", "(a,((A)))", "((A),a",
      "((A,b,a)", "(((A,B),C,D,C))*,D)*",
      "(A,(((A,B),C)*,(D,E))*"
    };
    for (int i = 0; i != nrCases; ++i) {
      FakeParser p;
      CPPUNIT_ASSERT_EQUAL(false, p.parse(cases[i]));
    }
    
  }
private:
  class FakeParser : public OperatorStringParser {
  public:
    FakeParser() {
    }
    bool parse(const std::string& str) {
      try {
        return OperatorStringParser::parse(str);
      } catch(const OperatorStringParser::SyntaxError&) {}
      return false;
    }
    std::string result_;
  private:
    void startSequence() {
      if (!result_.empty() && *result_.rbegin() != '(' && *result_.rbegin() != ',')
        result_ += ',';
      result_ += '(';
    }
    void endSequence(bool cl)  {
      result_ += ')';
      if (cl)
        result_ += '*';
      
    }
    void addOperator(const std::string& name) {
      if (!result_.empty() && *result_.rbegin() != ',' && *result_.rbegin() != '(')
        result_ += ',';
      result_ += name;
      
    }
  };
};

CPPUNIT_TEST_SUITE_REGISTRATION(TestOperatorStringParser);

}	// end namespace NS_NOMORE_TESTS
