#include <cppunit/TestFixture.h>
#include <cppunit/TestAssert.h>
#include <cppunit/extensions/HelperMacros.h>
#include <operators/sequence.h>

using namespace NS_NOMORE;

namespace NS_NOMORE_TESTS {

class TestSequenceOp : public CppUnit::TestFixture {

  CPPUNIT_TEST_SUITE(TestSequenceOp);
  CPPUNIT_TEST(testBothFalse);
	CPPUNIT_TEST(testOneTrue);
	CPPUNIT_TEST(testBothTrue);
	CPPUNIT_TEST(testClosure);
  CPPUNIT_TEST(testName);
  CPPUNIT_TEST(testExtract);
  CPPUNIT_TEST(testOwnership);
	CPPUNIT_TEST_SUITE_END();

public:
	void setUp() {
		o1_ = new FakeOperator("F");
		o2_ = new FakeOperator("F2");
		seqOp_ = new Sequence();
    seqOp_->add(o1_);
    seqOp_->add(o2_);
	}
	void tearDown() {
		delete seqOp_;
	}
	void testBothFalse() {
		CPPUNIT_ASSERT_EQUAL(false, (*seqOp_)());
	}
	void testOneTrue() {
		o1_->result_ = true;
		CPPUNIT_ASSERT_EQUAL(true, (*seqOp_)());
		o1_->result_ = false;
		o2_->result_ = true;
		CPPUNIT_ASSERT_EQUAL(true, (*seqOp_)());
	}
	void testBothTrue() {
		o1_->result_ = true;
		o2_->result_ = true;
		CPPUNIT_ASSERT_EQUAL(true, (*seqOp_)());
	}

	void testClosure() {
		o1_->result_ = true;
		o1_->maxRepeat_ = 3;
		seqOp_->setClosure(true);
		CPPUNIT_ASSERT_EQUAL(false, (*seqOp_)());
		CPPUNIT_ASSERT_EQUAL(-1, o1_->maxRepeat_);
	}
  void testName() {
    CPPUNIT_ASSERT_EQUAL( std::string("(F,F2)"), seqOp_->getName() );

    seqOp_->setClosure(true);
    CPPUNIT_ASSERT_EQUAL( std::string("(F,F2)*"), seqOp_->getName() );

    seqOp_->add(new FakeOperator("NF"));
    CPPUNIT_ASSERT_EQUAL( std::string("(F,F2,NF)*"), seqOp_->getName() );
  }
  void testExtract() {
    CPPUNIT_ASSERT_EQUAL( (Operator*)o1_, seqOp_->extractOperator("F") );
    CPPUNIT_ASSERT_EQUAL( (Operator*)o2_, seqOp_->extractOperator("F2") );
    CPPUNIT_ASSERT_EQUAL( (Operator*)0, seqOp_->extractOperator("F22") );
  }

  void testOwnership() {
    bool f1 = false;
    bool f2 = false;
    FakeOperator o1("F1", 1, &f1);
    {
      Sequence s;
      s.add(o1);
      s.add(new FakeOperator("F2", 1, &f2));
    }
    CPPUNIT_ASSERT_EQUAL(false, f1);
    CPPUNIT_ASSERT_EQUAL(true, f2);

  }
private:
	Sequence* seqOp_;
	class FakeOperator : public Operator {
	public:
    explicit FakeOperator(const std::string& name, int maxRepeat = 1, bool* trackDeletion = 0)
			: Operator(name)
			, result_(false)
			, maxRepeat_(maxRepeat)
      , trackDeletion_(trackDeletion){
		}
    ~FakeOperator() {
      if (trackDeletion_)
        *trackDeletion_ = true;
    }
    bool execute() {
			return result_ && maxRepeat_--;
		}
    void postExecute() {
    }
		bool result_;
		int maxRepeat_;
    bool* trackDeletion_;
  } *o1_, *o2_;
};

CPPUNIT_TEST_SUITE_REGISTRATION(TestSequenceOp);

}	// end namespace NS_NOMORE_TESTS
