/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         * 
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         * 
 ***************************************************************************/

#ifndef NOMORE_OPERATOR_H
#define NOMORE_OPERATOR_H

#include <string>
#include <set>
#include <color.h>

namespace NS_NOMORE {
class Graph;

//! The base class for all nomore operators
class Operator {
public:
  //! Creates a new operator.
  Operator();
  
  //! Creates a new operator.
  /*! 
   *  \param graph The Graph the new operator should work on.
   */
  Operator(Graph& graph);
  
  //! Destroys the operator.
  virtual ~Operator();

  //! Executes this operator.
  /*!
   * \return
   *  - ColorOperation::succeeded to signal that the operator executed successfully
   *  - ColorOperation::unchanged to signal that the execution of the operator did not change anything.
   *  - ColorOperation::failed to signal an error
   */
  virtual ColorOpResult operator()() = 0;

  //! resets this operator
  /*!
   * \note the default implementation does nothing
   *
   * Concrete operators may overwrite this method for example to clear internal
   * queues.
   */
  virtual void reset();

  
  typedef std::set<std::string> OpSet;
  //! returns true if this operator is compatible with all operators in ops
  /*!
   * \param ops a set of operator names.
   * \return the default implementation returns true, meaning that this operator
   * does not depend on any operator (except from forward propagation).
   * \note override this method to enforce operator-dependencies.
   */
  virtual bool validate(const OpSet& ops) const;
  
  //! returns the name of the concrete operator type
  virtual const char* getName() const = 0;

  //! return this if name is equal to this->getName()
  virtual Operator* extractOperator(const std::string& name);
private:
  Operator(const Operator&);
  Operator& operator=(const Operator&);
};

//! base class for propagation operators like P or B.
class PropOperator : public Operator {
public:
  //! Creates a new operator.
  /*
   *  \param graph The Graph the new operator should work on.
   */
  PropOperator(Graph& grp);

  //! returns the graph this operator works on
  const Graph& getGraph() const {
    return *graph_;
  }
  
  //! returns the graph this operator works on
  Graph& getGraph() {
    return *graph_;
  }

  //! Executes this operator.
  /*!
   * This base class implements the following algorithm:
   * \code
   *  uncoloredNodesBeforeExecution = graph.countUncoloredNodes();
   *  execute();
   *  if (graph.hasConflict()) {
   *    return ColorOperation::failed;
   *  }
   *  else {
   *    if (uncoloredNodesBeforeExecution != graph.countUncoloredNodes())
   *      return ColorOperation::succeeded;
   *    else
   *      return ColorOperation::unchanged;
   *  }
   *
   * \endcode
   *
   * \note 
   *  concrete subclasses shall not override this method. Instead they should
   *  override the hook-method PropOperator::execute().
   */
  ColorOpResult operator()();
protected:
  //! hook-method for concrete propagation operators
  /*!
   * implement concrete propagation operators by overriding this method.
   */
  virtual void execute() = 0;
private:
  //! The graph to work on.
  Graph *graph_;
};

template <class T>
const char* getName(T*);

template <class T>
Operator* createDetOp(Graph&, T*);


}

#endif
