/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         * 
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         * 
 ***************************************************************************/

#ifndef NOMORE_BACKWARD_PROP_OPERATOR_H
#define NOMORE_BACKWARD_PROP_OPERATOR_H

#include <operator.h>
#include <util/queue.h>
#include <color.h>
#include <graph.h>
#include <utility>

namespace NS_NOMORE {

class HeadNode;
class BodyNode;
struct HeadNodeColored;
struct BodyNodeColored;
struct BodyNodeColoredChoicePoint;
struct HeadNodeColoredChoicePoint;
struct GraphConstructionFinished;

//! Backward inference operator for the nomore body-head graph.
/*! The backward inference or backward propagation (also called \c B-operator) operator
 * 	tries to infer a color for predecessors of a colored node.
 * \par Conditions: \n
 * Backward propagation differentiates between head- and body-nodes. \n \n
 * <b>For \ref HeadNode "head nodes" the following derivations are made:</b> \n
 *
 * - If the head node has Color::plus or Color::weak_plus and all but one of 
 * its predecessors have Color::minus, then this last uncolored predecessors 
 * is colored Color::weak_plus and marked for backward-propagation.
 *
 * - If the head node has Color::minus all of its uncolored predecessors are 
 * colored Color::minus and marked for backward-propagation.
 * .
 * <b>For \ref BodyNode "body nodes" the following derivations are made:</b> \n
 * - If the body node has Color::plus or Color::weak_plus all its uncolored zero predecessors 
 *	are colored Color::weak_plus and its uncolored one predecessors are colored Color::minus.
 *
 * - If the body nodes has Color::minus, is \ref BodyNode::isUnblocked "unblocked" and
 *  and all but one of its zero predecessors are colored Color::plus or Color::weak_plus
 *	the last zero predecessors not colored is colored to Color::minus and marked for backward-propagation.
 *
 * - If the body nodes has Color::minus, is \ref BodyNode::isWeakSupported "(weak) supported" 
 * and all but one of its one predecessors are colored Color::minus the last
 * uncolored one predecessor is colored to Color::weak_plus and marked for backward-propagation.
 *
 * \par  <b>Which nodes are backward propagated?</b> \n
 *
 *	- Choice Points are always marked for backward propagation
 *	- While the graph is in its construction phase all colored head nodes are marked for backward propagation
 *	.
 *
 *  if the operator works in jumping-mode:
 *
 *	- Whenever a body node changes its color to Color::minus all of its 
 *	Color::plus or Color::weak_plus colored successors are marked for backward propagation.
 *	
 *	- Whenever a head node changes its color to Color::plus or Color::weak_plus
 *	all of its Color::minus colored zero successors are marked for backward propagation.
 *
 *	- Whenever a head node changes its color to Color::minus all of its 
 *	Color::minus colored one successors are marked for backward propagation.
 *	.
 *
 *  if the operator works in ignore-mode:
 *
 *	- whenever a head node changes its color all uncolored predecessors that 
 *	\ref BodyNode::canBeIgnored "can be ignored" are marked and colored to
 *	Color::ignore the next time the operator is invoked.
 *	
 * \par <b>To which events does this operator listen?</b> \n
 * In order to fulfill its job this operator listens to the following events: \n
 *	- GraphConstructionFinished
 *	- BodyNodeColoredChoicePoint
 *	- HeadNodeColoredChoicePoint
 *	- HeadNodeColored
 *	- BodyNodeColored, iff operator acts in jumping-mode
 */
class BackwardPropagator : public PropOperator {
public:
  BackwardPropagator(Graph& grp);
  ~BackwardPropagator();

  //! event handler for HeadNodeColored events.
	void handle(const HeadNodeColored& e);
	
	//! event handler for BodyNodeColored events.
	void handle(const BodyNodeColored& e);
	
	//! event handler for BodyNodeColoredChoicePoint events
	void handle(const BodyNodeColoredChoicePoint& e);
	
	//! event handler for HeadNodeColoredChoicePoint events
	void handle(const HeadNodeColoredChoicePoint& e);
	
	//! event handler for GraphConstructionFinished events
	void handle(const GraphConstructionFinished& e);

  void disableJumping();
  void disableIgnore();

  //! clears the backpropagation queues.
  virtual void reset();

  const char* getName() const;

  typedef util::queue<BodyNode*> BodyQueue;
  typedef util::queue<HeadNode*> HeadQueue;

  //! swaps the operator's internal queues with the provided queues.
  void swapQueues(BodyQueue& bq, HeadQueue& hq);

  bool backpropagate(const BodyNode&);
	bool backpropagate(const HeadNode&);

private:
  //! implements backward propagation as defined in the detailed section of BackwardPropagator.
  void execute();
  
  //! in jumping-mode?
	bool jumping_;	
	//! in ignore-mode?
	bool ignore_;
	//! graph fully constructed?
	bool graphConstructed_;
	
	
  //! body nodes to be backpropagated
	BodyQueue bodyNodes_;
	
	//! head nodes to backpropagated
	HeadQueue headNodes_;
	
	//! body nodes to be colored to ignore
	BodyQueue	bodyNodesToIgnore_;

	void handleIgnoreNodes();
  template <class NT>
  void backpropagate(util::queue<NT>& q) {
    Graph& g = getGraph();
    while (!g.hasConflict() && !q.empty()) {
      NT node = q.front();
      q.pop();
      backpropagate(*node);
    }
  }
	
  inline bool hasWorkToDo() const;
};


}

#endif
