/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         * 
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         * 
 ***************************************************************************/

#if defined (_MSC_VER) && _MSC_VER <= 1300
#pragma warning (disable : 4786)
#endif

#include <head_node.h>
#include <body_node.h>
#include <graph.h>
#include <event/channel_manager.h>
#include "detail/functors.h"
#include <algorithm>
namespace NS_NOMORE {

HeadNode::HeadNode(long id, long atomId) 
  : Node(id, NodeType::head_node)
	, atomId_(atomId)
  , atomName_("")
	, unsupported_(0) {
}

HeadNode::HeadNode(long id, long atomId, const std::string& name) 
  : Node(id, NodeType::head_node)
	, atomId_(atomId)
  , atomName_(name)
	, unsupported_(0) {
}

HeadNode::~HeadNode() {
}


void HeadNode::setAtomName(const std::string& name) {
  atomName_ = name;
}

const std::string& HeadNode::getAtomName() const {
  return atomName_;
}

long HeadNode::getAtomId() const {
  return atomId_;
}

void HeadNode::predecessorColored(Color::ColorValue, 
                                  Color::ColorValue newColor) {

  // modify unsupported_ if newColor is minus
  unsupported_ -= (newColor == Color::minus);
}

void HeadNode::predecessorRestored(Color::ColorValue oldColor, 
                                   Color::ColorValue) {

  // modify unsupported_ if newColor is minus
  unsupported_ += (oldColor == Color::minus);
}

void HeadNode::insertPredecessor(BodyNode &b){
  assert(b.getColor() == Color::none);
  predecessors_.push_back(&b);
  ++unsupported_;
}

void HeadNode::insertZeroSuccessor(BodyNode &b) {
  zeroSuccessors_.push_back(&b);
  b.insertZeroPredecessor(*this);
}

void HeadNode::insertOneSuccessor(BodyNode &b) {
  oneSuccessors_.push_back(&b);
  b.insertOnePredecessor(*this);
}

void HeadNode::doSetColor(Color::ColorValue oldColor) {
  BodyNodeSet::const_iterator it = predecessors_.begin();
  BodyNodeSet::const_iterator end = predecessors_.end();
  for(; it != end; ++it) {
    (*it)->successorColored(oldColor, getColor());
  }

  end = zeroSuccessors_.end();
  for(it = zeroSuccessors_.begin(); it != end; ++it) {
    (*it)->zeroPredecessorColored(oldColor, getColor());
  }

  end = oneSuccessors_.end();
  for(it = oneSuccessors_.begin(); it != end; ++it) {
    (*it)->onePredecessorColored(oldColor, getColor());
  }
}

void HeadNode::doRestoreColor(Color::ColorValue oldColor) {
  BodyNodeSet::const_iterator it = predecessors_.begin();
  BodyNodeSet::const_iterator end = predecessors_.end();
  for(; it != end; ++it) {
    (*it)->successorRestored(oldColor, getColor());
  }

  end = zeroSuccessors_.end();
  for(it = zeroSuccessors_.begin(); it != end; ++it) {
    (*it)->zeroPredecessorRestored(oldColor, getColor());
  }

  end = oneSuccessors_.end();
  for(it = oneSuccessors_.begin(); it != end; ++it) {
    (*it)->onePredecessorRestored(oldColor, getColor());
  }
}

void HeadNode::fireColorEvent(event::ChannelManager& c, Color::ColorValue oldColor) {
	HeadNodeColored e(this, oldColor);
	c.dispatch(e);
}

void HeadNode::fireColorChoicePointEvent(event::ChannelManager& c, Color::ColorValue oldColor, bool recolored) {
  HeadNodeColoredChoicePoint e(this, oldColor, recolored);
  c.dispatch(e);
}

bool HeadNode::trivialComponent() const {
  using std::find_if;
  DETAIL::InComponent inComponent(getComponentNumber());
  return getComponentNumber() != -1
    && find_if(predecessorsBegin(), predecessorsEnd(), inComponent) == predecessorsEnd()
    && find_if(zeroSuccessorsBegin(), zeroSuccessorsEnd(), inComponent) == zeroSuccessorsEnd();
}

double HeadNode::getTightnessFactor(int gradient) const {
  double sum = 0;
  for (size_t i = 0; i < predecessors_.size(); ++i) {
    if (predecessors_[i]->getColor() != Color::minus) {
      sum += predecessors_[i]->getTightnessFactor(gradient);
    }
  }
  return sum / unsupported_;
}

namespace {


inline void prop(Graph& g, HeadNode::BodyNodeIterator first, HeadNode::BodyNodeIterator last) {
  for (; first != last; ++first) {
    if ((*first)->isWeakSupported() && (*first)->isUnblocked()){   
      g.color(**first, (*first)->recursiveSupportColor());
    }
    else if ((*first)->isBlocked() || (*first)->isUnsupported()) {
      g.color(**first, Color::minus);
    }
  }
}

}

void HeadNode::propagate(Graph& g) {
  prop(g, zeroSuccessorsBegin(), zeroSuccessorsEnd());
  prop(g, oneSuccessorsBegin(), oneSuccessorsEnd());
}

} // NS_NOMORE
