#include <cppunit/TestFixture.h>
#include <cppunit/TestAssert.h>
#include <cppunit/extensions/HelperMacros.h>
#include <operators/forward_prop.h>
#include <graph.h>
#include <head_node.h>
#include <body_node.h>
#include <strong_components.h>

using namespace NS_NOMORE;

namespace NS_NOMORE_TESTS {

class TestStrongComponents : public CppUnit::TestFixture {

  CPPUNIT_TEST_SUITE(TestStrongComponents);
  CPPUNIT_TEST(testEmpty);
  CPPUNIT_TEST(testDAG);
  CPPUNIT_TEST(testOneLoop);
  CPPUNIT_TEST(testTwoLoops);
  CPPUNIT_TEST(testFourLoops);
  CPPUNIT_TEST_SUITE_END();

public:
  void setUp() {
    grp = new Graph();
  }

  void tearDown() {
    delete grp;
  }

  void testEmpty() {
    CPPUNIT_ASSERT_EQUAL(0, identifyStrongComponents(*grp));
  }
  void testDAG() {
    HeadNode* n0 = grp->insertHeadNode(1);
    BodyNode* n1 = grp->insertBodyNode();
    BodyNode* n2 = grp->insertBodyNode();
    n1->insertSuccessor(*n0);
    n2->insertSuccessor(*n0);

    CPPUNIT_ASSERT_EQUAL(0, identifyStrongComponents(*grp));
    CPPUNIT_ASSERT_EQUAL(true, n0->trivialComponent());
    CPPUNIT_ASSERT_EQUAL(true, n1->trivialComponent());
    CPPUNIT_ASSERT_EQUAL(true, n2->trivialComponent());
  }

  void testOneLoop() {
    HeadNode* n1 = grp->insertHeadNode(1);
    BodyNode* n2 = grp->insertBodyNode();
    BodyNode* n3 = grp->insertBodyNode();
    HeadNode* n4 = grp->insertHeadNode(2);

    n1->insertZeroSuccessor(*n3);
    n2->insertSuccessor(*n1);
    n3->insertSuccessor(*n4);
    n4->insertZeroSuccessor(*n2);

    CPPUNIT_ASSERT_EQUAL(1, identifyStrongComponents(*grp));
    CPPUNIT_ASSERT_EQUAL(0, n1->getComponentNumber());
    CPPUNIT_ASSERT_EQUAL(0, n2->getComponentNumber());
    CPPUNIT_ASSERT_EQUAL(0, n3->getComponentNumber());
    CPPUNIT_ASSERT_EQUAL(0, n4->getComponentNumber());

    CPPUNIT_ASSERT_EQUAL(false, n1->trivialComponent());
    CPPUNIT_ASSERT_EQUAL(false, n2->trivialComponent());
    CPPUNIT_ASSERT_EQUAL(false, n3->trivialComponent());
    CPPUNIT_ASSERT_EQUAL(false, n4->trivialComponent());
  }

  void testTwoLoops() {
    HeadNode* n1 = grp->insertHeadNode(1);
    BodyNode* n2 = grp->insertBodyNode();
    BodyNode* n3 = grp->insertBodyNode();
    HeadNode* n4 = grp->insertHeadNode(2);
    HeadNode* n5 = grp->insertHeadNode(3);
    BodyNode* n6 = grp->insertBodyNode();

    n1->insertZeroSuccessor(*n3);
    n2->insertSuccessor(*n1);
    n3->insertSuccessor(*n4);
    n4->insertZeroSuccessor(*n2);
    n5->insertZeroSuccessor(*n6);
    n6->insertSuccessor(*n5);

    CPPUNIT_ASSERT_EQUAL(2, identifyStrongComponents(*grp));
    CPPUNIT_ASSERT_EQUAL(0, n1->getComponentNumber());
    CPPUNIT_ASSERT_EQUAL(0, n2->getComponentNumber());
    CPPUNIT_ASSERT_EQUAL(0, n3->getComponentNumber());
    CPPUNIT_ASSERT_EQUAL(0, n4->getComponentNumber());
    CPPUNIT_ASSERT_EQUAL(1, n5->getComponentNumber());
    CPPUNIT_ASSERT_EQUAL(1, n6->getComponentNumber());
    CPPUNIT_ASSERT_EQUAL(false, n1->trivialComponent());
    CPPUNIT_ASSERT_EQUAL(false, n2->trivialComponent());
    CPPUNIT_ASSERT_EQUAL(false, n3->trivialComponent());
    CPPUNIT_ASSERT_EQUAL(false, n4->trivialComponent());
    CPPUNIT_ASSERT_EQUAL(false, n5->trivialComponent());
    CPPUNIT_ASSERT_EQUAL(false, n6->trivialComponent());
  }

  void testFourLoops() {
    HeadNode* n0 = grp->insertHeadNode(1);
    BodyNode* n1 = grp->insertBodyNode();
    BodyNode* n2 = grp->insertBodyNode();
    HeadNode* n3 = grp->insertHeadNode(2);
    HeadNode* n4 = grp->insertHeadNode(3);
    BodyNode* n5 = grp->insertBodyNode();
    BodyNode* n6 = grp->insertBodyNode();
    BodyNode* n7 = grp->insertBodyNode();
    HeadNode* n8 = grp->insertHeadNode(4);
    HeadNode* n9 = grp->insertHeadNode(5);
    BodyNode* n10 = grp->insertBodyNode();
    HeadNode* n11 = grp->insertHeadNode(6);
    BodyNode* n12 = grp->insertBodyNode();
    BodyNode* n13 = grp->insertBodyNode();
    HeadNode* n14 = grp->insertHeadNode(7);
    BodyNode* n15 = grp->insertBodyNode();
    
    n0->insertZeroSuccessor(*n1);
    n0->insertZeroSuccessor(*n5);
    n0->insertZeroSuccessor(*n6);

    n2->insertSuccessor(*n0);
    n2->insertSuccessor(*n3);

    n3->insertZeroSuccessor(*n2);
    n3->insertZeroSuccessor(*n5);
    
    n4->insertZeroSuccessor(*n2);
    n4->insertZeroSuccessor(*n7);
    n4->insertZeroSuccessor(*n12);

    n5->insertSuccessor(*n8);
    
    n6->insertSuccessor(*n4);
    n6->insertSuccessor(*n11);

    n7->insertSuccessor(*n3);
    
    n8->insertZeroSuccessor(*n7);

    n9->insertZeroSuccessor(*n6);
    n9->insertZeroSuccessor(*n10);

    n10->insertSuccessor(*n9);
    n10->insertSuccessor(*n11);

    n11->insertZeroSuccessor(*n12);
    n11->insertZeroSuccessor(*n13);

    n12->insertSuccessor(*n14);

    n13->insertSuccessor(*n14);
    
    n14->insertZeroSuccessor(*n15);
    
    n15->insertSuccessor(*n11);
    
    CPPUNIT_ASSERT_EQUAL(4, identifyStrongComponents(*grp));
    
    
    Node* loop1[] = { n1 };
    Node* loop2[] = { n0, n2, n3, n4, n5, n6, n7, n8 };
    Node* loop3[] = { n9, n10 };
    Node* loop4[] = { n11, n12, n13, n14, n15 };
    checkLoop(loop1, sizeof(loop1)/sizeof(Node*));
    checkLoop(loop2, sizeof(loop2)/sizeof(Node*));
    checkLoop(loop3, sizeof(loop3)/sizeof(Node*));
    checkLoop(loop4, sizeof(loop4)/sizeof(Node*));
  }
private:
  Graph* grp;

  void checkLoop(Node** n, int size) {
    int c = n[0]->getComponentNumber();
    for (int i = 1; i != size; ++i)
      CPPUNIT_ASSERT_EQUAL(c, n[i]->getComponentNumber());
  }
};

CPPUNIT_TEST_SUITE_REGISTRATION(TestStrongComponents);

}	// end namespace NS_NOMORE_TESTS
